# Copyright (c) 2015 Arista Networks, Inc.  All rights reserved.
# Arista Networks, Inc. Confidential and Proprietary.

# CliPlugin module for Evpn commands
import BasicCli
from BgpLib import getVpwsName
import CliCommand
import ShowCommand
import CliParser
import CliMatcher
import ConfigMount
import Ethernet
import LazyMount
import Tac
from Toggles import (
   BgpCommonToggleLib,
   EvpnToggleLib,
   EvpnLibToggleLib,
)
from CliMode.Evpn import EthernetSegmentMode
import CliToken.Clear
from CliToken.RoutingBgpShowCliTokens import bgpAfterShow
from CliToken.Evpn import (
   evpnShow,
   nexthopSelfSupportNlris,
)
import CliToken.Evpn as EvpnTokens
import CliToken.RoutingBgp as bgpTokens

import CliPlugin.IntfCli as IntfCli
from CliPlugin.BgpVpnCli import (
   RouteImportMatchFailureDiscardCmd,
)
import CliPlugin.BgpVpnDomainIdCli as BgpVpnDomainIdCli
from CliPlugin import TechSupportCli
from CliPlugin.ArBgpCli import ArBgpAsyncCliCommand, doShowIpBgpSummaryAcrImpl
from CliPlugin.BgpMacVrfConfigCli import (
   getBundleName,
   getStandAloneName,
)
from CliPlugin.EvpnCliModels import (
   BgpEvpnInstances,
)
from CliPlugin.EvpnCliModels import BgpEvpnHostFlapEntry, BgpEvpnHostFlapEntries
import CliPlugin.Esid as Esid
from CliPlugin.MlagWarningCli import canShutdownMlagHook
import CliPlugin.RoutingBgpCli as RoutingBgpCli
from CliPlugin.EthIntfCli import EthIntf
from CliPlugin.LagIntfCli import EthLagIntf
from CliPlugin.RoutingBgpCli import (
   RouterBgpBaseAfEvpnMode,
   deleteRouterBgpMacVrfHook,
   RouterBgpAfEvpnModelet,
   BgpCmdBaseClass,
)
from CliPlugin.RoutingBgpInstanceCli import (
   NexthopResolutionDisabledCmd,
)
from CliPlugin.RoutingBgpShowCli import (
   ArBgpShowOutput,
)
from CliPlugin.VlanCli import vlanSetMatcher
from CliPlugin.VxlanCli import vxlanSupportedGuard
import CliPlugin.VxlanModel as VxlanModel
from Vlan import computeVlanRangeSet

vtiConfigDir = None
vtiStatusDir = None
bgpMacVrfConfigDir = None
evpnStatus = None
vxlanCtrlConfig = None
vxlanCtrlConfigBox = []

# pkgdeps: rpm Evpn-lib

#-------------------------------------------------------------------------------
# EVPN Ethernet Segment Configuration Commands
# config-if
#   evpn ethernet-segment
#       identifier <10 octet>
#       designated-forwarder election hold-time <N>
#       redundancy [all-active | single-active]
#       route-target import 12:34:45:67:89:01
#       mpls tunnel flood filter time <N>
#-------------------------------------------------------------------------------

U32_MAX_VALUE = 0xFFFFFFFF
ethernetSegmentConfigDir = None
evpnConfig = None
macDuplicationBlacklist = None
electionHoldTimeVal = \
      Tac.Value( 'Evpn::DFElectionHoldTime' )
defaultRT = Tac.Value( 'Arnet::EthAddr' )
checkEsid = Tac.Value( 'Evpn::Esid' )

def getSbdVrfName( l3VrfName ):
   return Tac.Type( "Routing::Bgp::SbdMacVrfUtil" ).getSbdMacVrfName(
      l3VrfName )

# evpn ethernet-segment only makes sense for L2 interfaces. Command should not
# be available for SVIs, Managment interfaces, peer interfaces...etc.
def evpnEthernetSegmentSupportedGuard( mode, token ):
   if isinstance( mode.intf, ( EthIntf, EthLagIntf ) ):
      return None
   return CliParser.guardNotThisPlatform

# Check whether "esi" can be applied. Currently, validate Type 0 and
# non-duplicate ESI.
def checkEsidAssignment( allEsConfig, esi, intf ):
   if esi[ : 2 ] != "00":
      return ( False, 'only Type 0 ESID is supported' )
   if esi == checkEsid.zero or esi == checkEsid.max:
      return ( False, 'reserved ESID entered' )
   for intfName, cfg in allEsConfig.intfConfig.iteritems():
      if cfg.esid.value == esi and intfName != intf:
         return ( False, 'ESID is already assigned to interface %s' % intfName )
   return ( True, '' )

# Check if there are duplicate "rt"
def checkRtAssignment( allEsConfig, rt, intf ):
   for intfName, cfg in allEsConfig.intfConfig.iteritems():
      if cfg.esImportRouteTarget == rt and intfName != intf:
         return ( True, 'Import RT is already assigned to interface %s' % intfName )
   return ( False, '' )

class EvpnIntf( IntfCli.IntfDependentBase ):
   def setDefault( self ):
      del ethernetSegmentConfigDir.intfConfig[ self.intf_.name ]

class EthernetSegmentConfigMode( EthernetSegmentMode, BasicCli.ConfigModeBase ):
   modeParseTree = CliParser.ModeParseTree()

   def __init__( self, parent, session, intf ):
      self.intf_ = intf
      self.session_ = session
      EthernetSegmentMode.__init__( self )
      BasicCli.ConfigModeBase.__init__( self, parent, session )
      self.config()

   def config( self ):
      config = ethernetSegmentConfigDir.intfConfig.get( self.intf_, None )
      if config is None:
         config = ethernetSegmentConfigDir.newIntfConfig( self.intf_ )
      return config

   def setIdentifier( self, args ):
      esi = args[ 'ESI' ]
      config = self.config()
      ( allow, reason ) = checkEsidAssignment( ethernetSegmentConfigDir, esi, 
                                               self.intf_ )
      if not allow:
         self.session_.addError( 
               'Unable to configure identifier because %s' % reason )
         return
      newEsi = Tac.Value( "Evpn::EsiOrNone" )
      newEsi.value = esi
      config.esid = newEsi
      newEsiType = Tac.Value( "Evpn::EsiType", 'type0' )
      config.esiType = newEsiType

   def noIdentifier( self, args ):
      config = self.config()
      config.esid = Tac.Value( 'Evpn::EsiOrNone' )
      config.esiType = Tac.Value( "Evpn::EsiType" )

   def setRedundancy( self, args ):
      config = self.config()
      if 'all-active' in args:
         config.multihomeMode = 'multihomeModeAllActive'
      else:
         config.multihomeMode = 'multihomeModeSingleActive'

   def noRedundancy( self, args ):
      config = self.config()
      config.multihomeMode = config.multihomeModeDefault

   def setHoldTime( self, args ):
      config = self.config()
      config.holdTime = Tac.Value( 'Evpn::DFElectionHoldTime',
                                    args[ 'HOLD_TIME' ] )
      delayTime = 0
      delayTime = args.get( 'DELAY_TIME', 0 ) / 1000.0
      config.dfElectionDelayTime = delayTime

   def noHoldTime( self, args ):
      config = self.config()
      config.holdTime = electionHoldTimeVal.defaultHoldTime
      config.dfElectionDelayTime = 0

   def setEncapFloodFilterTime( self, args ):
      config = self.config()
      time_ms = args[ 'TIME' ] / 1000.0
      config.encapBumFilterHoldTime = time_ms

   def noEncapFloodFilterTime( self, args ):
      config = self.config()
      config.encapBumFilterHoldTime = 0

   def setSharedEsiLabelIndex( self, args ):
      indexVal = args[ 'INDEX' ]
      self.config().sharedEsiLabelIndex = indexVal

   def noSharedEsiLabelIndex( self, args ):
      self.config().sharedEsiLabelIndex = 0

   def setRouteTargetImport( self, args ):
      rt = args[ 'RT' ]
      config = self.config()
      ( warn, reason ) = checkRtAssignment( ethernetSegmentConfigDir, rt, 
                                            self.intf_ )
      if warn:
         self.session_.addWarning( reason )
      config.esImportRouteTarget = rt

   def noRouteTargetImport( self, args ):
      config = self.config()
      config.esImportRouteTarget = defaultRT

#--------------------------------------------------------------------------------
# [ no | default ] evpn ethernet-segment
#--------------------------------------------------------------------------------
def gotoEthernetSegmentConfigMode( mode, args ):
   intf = mode.intf
   childMode = mode.childMode( EthernetSegmentConfigMode, intf=intf.name )
   # Do not enter evpn ethernet-segment submode when configured via interface
   # range config mode.
   if mode.session.mode_ != childMode.parent_:
      return
   mode.session_.gotoChildMode( childMode )

def deleteEthernetSegmentConfigMode( mode, args ):
   intf = mode.intf.name
   if intf in ethernetSegmentConfigDir.intfConfig:
      del ethernetSegmentConfigDir.intfConfig[ intf ]

class EvpnEthernetSegmentCmd( CliCommand.CliCommandClass ):
   syntax = 'evpn ethernet-segment'
   noOrDefaultSyntax = 'evpn ethernet-segment ...'
   data = {
      'evpn': CliCommand.Node(
         matcher=CliMatcher.KeywordMatcher( 'evpn', helpdesc='EVPN Configuration' ),
         guard=evpnEthernetSegmentSupportedGuard ),
      'ethernet-segment': 'Ethernet segment configuration',
   }
   handler = gotoEthernetSegmentConfigMode
   noOrDefaultHandler = deleteEthernetSegmentConfigMode

IntfCli.IntfConfigMode.addCommandClass( EvpnEthernetSegmentCmd )

#--------------------------------------------------------------------------------
# identifier ESI
#--------------------------------------------------------------------------------
class IdentifierEsiCmd( CliCommand.CliCommandClass ):
   syntax = 'identifier ESI'
   noOrDefaultSyntax = 'identifier ...'
   data = {
      'identifier': '10 octet ethernet segment identifier',
      'ESI': Esid.esiMatcher,
   }
   handler = EthernetSegmentConfigMode.setIdentifier
   noOrDefaultHandler = EthernetSegmentConfigMode.noIdentifier

EthernetSegmentConfigMode.addCommandClass( IdentifierEsiCmd )

#--------------------------------------------------------------------------------
# [ no | default ] redundancy ( all-active | single-active )
#--------------------------------------------------------------------------------
class RedundancyCmd( CliCommand.CliCommandClass ):
   syntax = 'redundancy ( all-active | single-active )'
   noOrDefaultSyntax = 'redundancy ...'
   data = {
      'redundancy': 'Multihoming redundancy mode configuration',
      'all-active': 'All PEs attached to an Ethernet segment are forwarding',
      'single-active': ( 'Only a single PE attached to an Ethernet segment is '
                         'forwarding' ),
   }
   hidden = not EvpnToggleLib.toggleEvpnSingleActiveToggleEnabled()
   handler = EthernetSegmentConfigMode.setRedundancy
   noOrDefaultHandler = EthernetSegmentConfigMode.noRedundancy

if EvpnToggleLib.toggleEvpnSingleActiveToggleEnabled():
   EthernetSegmentConfigMode.addCommandClass( RedundancyCmd )

#--------------------------------------------------------------------------------
# designated-forwarder election hold-time HOLD_TIME
#                               [ subsequent-hold-time DELAY_TIME ]
#--------------------------------------------------------------------------------
class HoldTimeCmd( CliCommand.CliCommandClass ):
   syntax = 'designated-forwarder election hold-time HOLD_TIME'
   if EvpnToggleLib.toggleEvpnDfElectionDelayEnabled():
      syntax += ' [ subsequent-hold-time DELAY_TIME ]'
   noOrDefaultSyntax = 'designated-forwarder election hold-time ...'
   data = {
      'designated-forwarder': ( 'Designated PE for broadcast, unknown unicast '
                                'and multicast packets' ),
      'election': 'Election time',
      'hold-time': 'Timer waiting to receive updates from other PE',
      'HOLD_TIME': CliMatcher.IntegerMatcher( electionHoldTimeVal.min,
                                              electionHoldTimeVal.max,
                                              helpdesc='Number of seconds' ),
   }
   if EvpnToggleLib.toggleEvpnDfElectionDelayEnabled():
      data[ 'subsequent-hold-time' ] = 'Time to wait before running each election'
      data[ 'DELAY_TIME' ] = CliMatcher.IntegerMatcher(
         10, 10000, helpdesc='Time in milliseconds' )
   handler = EthernetSegmentConfigMode.setHoldTime
   noOrDefaultHandler = EthernetSegmentConfigMode.noHoldTime

EthernetSegmentConfigMode.addCommandClass( HoldTimeCmd )

#--------------------------------------------------------------------------------
# mpls tunnel flood filter time TIME
#--------------------------------------------------------------------------------
class EncapFloodFilterTimeCmd( CliCommand.CliCommandClass ):
   syntax = 'mpls tunnel flood filter time TIME'
   noOrDefaultSyntax = 'mpls tunnel flood filter time ...'
   data = {
      'mpls' : 'MPLS config',
      'tunnel' : 'MPLS tunnel config',
      'flood' : 'Broadcast traffic',
      'filter' : ( 'Filter MPLS encapsulated broadcast traffic into tunnel when DF'
                   ' election is triggered' ),
      'time' : 'Time after running triggered DF election when filter is removed',
      'TIME' : CliMatcher.IntegerMatcher( 10, 10000,
                                         helpdesc='Time in milliseconds' ),
   }
   handler = EthernetSegmentConfigMode.setEncapFloodFilterTime
   noOrDefaultHandler = EthernetSegmentConfigMode.noEncapFloodFilterTime

if EvpnToggleLib.toggleEvpnEncapBumFilterToggleEnabled():
   EthernetSegmentConfigMode.addCommandClass( EncapFloodFilterTimeCmd )

#--------------------------------------------------------------------------------
# mpls shared index INDEX
#--------------------------------------------------------------------------------
class SharedEsiLabelIndexCmd( CliCommand.CliCommandClass ):
   syntax = 'mpls shared index INDEX'
   noOrDefaultSyntax = 'mpls shared index ...'
   data = {
      'mpls' : 'MPLS config',
      'shared' : 'Shared MPLS label config',
      'index' : 'MPLS label index into l2evpn shared ethernet-segment range',
      'INDEX' : CliMatcher.IntegerMatcher( 1, 1024,
                                           helpdesc='Label index value' )
   }
   handler = EthernetSegmentConfigMode.setSharedEsiLabelIndex
   noOrDefaultHandler = EthernetSegmentConfigMode.noSharedEsiLabelIndex

if EvpnLibToggleLib.toggleSharedEsiLabelEnabled():
   EthernetSegmentConfigMode.addCommandClass( SharedEsiLabelIndexCmd )

#--------------------------------------------------------------------------------
# route-target import RT
#--------------------------------------------------------------------------------
class RouteTargetImportCmd( CliCommand.CliCommandClass ):
   syntax = 'route-target import RT'
   noOrDefaultSyntax = 'route-target import ...'
   data = {
      'route-target': 'Route Target',
      'import': 'Route Import',
      'RT': CliMatcher.PatternMatcher( Ethernet.macAddrPattern,
                                       helpname='H.H.H',
                                       helpdesc='Low-order 6 bytes of ES-Import '
                                                'Route Target' ),
   }
   handler = EthernetSegmentConfigMode.setRouteTargetImport
   noOrDefaultHandler = EthernetSegmentConfigMode.noRouteTargetImport

EthernetSegmentConfigMode.addCommandClass( RouteTargetImportCmd )

#-------------------------------------------------------------------------------
# END EVPN Ethernet Segment Configuration Commands
#-------------------------------------------------------------------------------

#-------------------------------------------------------------------------------
# config-bgp-vrf-foo# evpn multicast
#-------------------------------------------------------------------------------
class EvpnOismCmd( CliCommand.CliCommandClass ):
   syntax = 'evpn multicast'
   noOrDefaultSyntax = 'evpn multicast ...'
   data = { 'evpn' : CliMatcher.KeywordMatcher( 'evpn', 'EVPN configuration' ),
            'multicast' : CliMatcher.KeywordMatcher(
               'multicast', helpdesc='Configure EVPN multicast' ) }

   @staticmethod
   def handler( mode, args ):
      config = RoutingBgpCli.configForVrf( mode.vrfName )
      config.afEvpnOismEnabled = True

   @staticmethod
   def noOrDefaultHandler( mode, args ):
      config = RoutingBgpCli.configForVrf( mode.vrfName )
      config.afEvpnOismEnabled = False

if BgpCommonToggleLib.toggleArBgpOismToggleEnabled():
   RoutingBgpCli.RouterBgpVrfMode.addCommandClass( EvpnOismCmd )

#-------------------------------------------------------------------------------
# "show bgp evpn summary [ route-type auto-discovery | mac-ip | imet |
#                                     ethernet-segment | ( ip-prefix ipv4 | ipv6 ) |
#                                     smet | spmsi |join-sync | leave-sync ]"
#-------------------------------------------------------------------------------
# Show BGP EVPN peers and optionally filter by route-type.
# When route-type is not specified, summary statistics reflect the number of routes
# received and accepted for all evpn route-types.
# When a route-type is specified, summary statistics reflect the number of routes
# received and accepted for the specified route-type.

@ArBgpShowOutput( 'doShowBgpEvpnSummary', arBgpModeOnly=True )
def doShowBgpEvpnSummary( *args, **kwargs ):
   mode = args[ 0 ]
   argsValue = kwargs.pop( 'args', None )

   nlriTypeValue = argsValue.pop( 'nlriTypeValue', None )
   if nlriTypeValue:
      nlriType = nlriTypeValue[ 'nlriType' ]
      return doShowIpBgpSummaryAcrImpl( mode, nlriType=nlriType )
   else:
      return doShowIpBgpSummaryAcrImpl( mode, nlriAfiSafi='l2vpnEvpn' )

# TODO remove these once we remove the AgentCommandRequest implementation
def tunnelEncapExtComm( tunnelTypeStr ):
   # TODO - Currently we only support "vxlan" tunnel type. Need to add a matcher
   # with other possible tokens when we decide to support more types.
   assert tunnelTypeStr == 'vxlan'
   c = ( 0x030c << 48 ) | 0x0008
   return c

def macMobilityExtComm( seqNoStr ):
   c = ( 0x0600 << 48 )
   if seqNoStr == 'sticky':
      c |= ( 0x0100 << 32 )
   else:
      c |= int( seqNoStr )
   return c

def esiLabelExtComm( esiLabel ):
   c = ( 0x0601 << 48 )
   if esiLabel[ 'esiLabelRedundancyMode' ] == 'single-active':
      c |= ( 0x0100 << 32 )
   c |= int( esiLabel[ 'esiLabelValue' ] )
   return c

def routerMacExtComm( macStr ):
   mac = Tac.Value( 'Arnet::EthAddr', stringValue=macStr )
   c = ( 0x0603 << 48 ) | ( mac.word0 << 32 ) | ( mac.word1 << 16 ) | mac.word2
   return c

@ArBgpShowOutput( 'doShowBgpEvpn', arBgpModeOnly=True )
def doShowBgpEvpn( mode, peerAddrValue=None, bgpRouteTypeValue=None,
                   commValuesAndExact=None, extCommValuesAndExact=None,
                   largeCommValuesAndExact=None,
                   nlriTypeAndPrefixValues=None, rdValue=None, vniValue=None,
                   nexthopValue=None, esiValue=None, detail=None, internal=None ):
   assert False, "This method is abstract. " + \
      "If this was reached then something went wrong with EvpnCliHelperCli loading"
   
#------------------------------------------------------------------
# "show bgp evpn instance"
# "show bgp evpn instance vlan VLANID"
# "show bgp evpn instance vlan-aware-bundle BUNDLE"
# "show bgp evpn instance vpws VPWS"
# "show bgp evpn instance sbd <SBD-VRF-NAME>"
#
# Shows the details of corresponding mac vrf.
#------------------------------------------------------------------
class ShowBgpEvpnInstanceCmd( ShowCommand.ShowCliCommandClass ):
   syntax = """show bgp evpn instance
   [ ( vlan VLANID )
   | ( vlan-aware-bundle BUNDLE )
   | ( vpws VPWS )
   | ( sbd SBD_VRF_NAME ) ]"""
   cliModel = BgpEvpnInstances
   data = {
      "bgp" : bgpAfterShow,
      "evpn" : evpnShow,
      "instance" : "EVPN instance",
      "vlan" : "Show information for VLAN-based EVPN instance",
      "VLANID" : CliMatcher.IntegerMatcher( 1, 4094,
         helpdesc="Identifier for a Virtual LAN" ),
      "vlan-aware-bundle" : "Show information for VLAN-aware bundle EVPN instance",
      "BUNDLE" : CliMatcher.QuotedStringMatcher(
         helpname="WORD",
         helpdesc="Unique name to identify VLAN-aware bundle" ),
      "vpws" : ( "Show information for EVPN virtual private wire service (VPWS) "
                 "instance" ),
      "VPWS" : CliMatcher.PatternMatcher(
         pattern=r".+",
         helpname="WORD",
         helpdesc="Virtual private wire service (VPWS) instance name" ),
      "sbd" : "Show information for SBD VLAN EVPN instance",
      "SBD_VRF_NAME" : CliMatcher.PatternMatcher(
         pattern=r".+",
         helpname="WORD",
         helpdesc="IP-VRF name" ),
   }

   @staticmethod
   @ArBgpShowOutput( "doShowBgpEvpnInstance", arBgpModeOnly=True )
   def handler( mode, args ):
      if args.get( "vlan" ):
         macVrfName = getStandAloneName( args.get( "VLANID" ) )
      elif args.get( "vlan-aware-bundle" ):
         macVrfName = getBundleName( args.get( "BUNDLE" ) )
      elif args.get( "vpws" ):
         macVrfName = getVpwsName( args.get( "VPWS" ) )
      elif args.get( "sbd" ):
         macVrfName = getSbdVrfName( args.get( "SBD_VRF_NAME" ) )
      else:
         macVrfName = None
      cmd = ArBgpAsyncCliCommand( mode, "show evpn instance" )
      if macVrfName:
         cmd.addParam( "macVrf", macVrfName )
      cmd.run()
      return BgpEvpnInstances

BasicCli.addShowCommandClass( ShowBgpEvpnInstanceCmd )

#--------------------------------------------------------------------------
# register show tech-support extended evpn
#--------------------------------------------------------------------------
def _showTechEvpnCmds():
   return [
            'show bgp evpn',
            'show bgp evpn instance',
            'show bgp evpn detail',
            'show bgp evpn summary',
          ]
TechSupportCli.registerShowTechSupportCmdCallback( '2017-11-03 12:06:06',
               _showTechEvpnCmds, extended='evpn' )

# Register cmd for 'show tech-support summary' cmd
TechSupportCli.registerShowTechSupportCmdCallback( '2020-05-14 23:49:42',
               lambda: [],
               summaryCmdCallback=lambda: [ 'show bgp evpn summary' ] )

#------------------------------------------------------------------
#  routing-bgp <asn>
#     address-family evpn
#        "[ no | default ] host-flap detection window SECONDS"
#        "[ no | default ] host-flap detection threshold MOVES"
#
#  Configures the parameters used to determine if flapping MAC
#  addresses should be blacklisted according to RFC 7432.
#  If a MAC address flaps more than MOVES times within
#  SECONDS seconds it will be blacklisted.
#------------------------------------------------------------------
class HostFlapDetectionParamsCmd( CliCommand.CliCommandClass ):
   syntax = """host-flap detection
                ( ( window    SECONDS [ threshold MOVES   ] )
                | ( threshold MOVES   [ window    SECONDS ] ) )"""
   noOrDefaultSyntax = """host-flap detection ..."""
   data = {
         'host-flap' : 'Host-flap',
         'detection' : 'Detection',
         'window' : 'Time (in seconds) to detect a MAC duplication issue',
         'SECONDS' : CliMatcher.FloatMatcher( 0, U32_MAX_VALUE,
                                              precisionString='%.25g',
                                              helpdesc='Time (in seconds) to detect '
                                                       'a MAC duplication issue' ),
         'threshold' : 'Minimum number of MAC moves that indicate '
                       'a MAC Duplication issue',
         'MOVES' : CliMatcher.IntegerMatcher( 0, U32_MAX_VALUE,
                                              helpdesc='Minimum number of MAC moves '
                                                       'that indicate a MAC '
                                                       'duplication issue' ),
   }

   @staticmethod
   def handler( mode, args ):
      timeout = args.get( 'SECONDS', evpnConfig.duplicateTimeoutDefault )
      evpnConfig.duplicateTimeout = timeout

      threshold = args.get( 'MOVES', evpnConfig.duplicateThresholdDefault )
      evpnConfig.duplicateThreshold = threshold

   @staticmethod
   def noHandler( mode, args ):
      evpnConfig.duplicateTimeout = 0
      evpnConfig.duplicateThreshold = 0

   @staticmethod
   def defaultHandler( mode, args ):
      evpnConfig.duplicateTimeout = evpnConfig.duplicateTimeoutDefault
      evpnConfig.duplicateThreshold = evpnConfig.duplicateThresholdDefault

RouterBgpBaseAfEvpnMode.addCommandClass( HostFlapDetectionParamsCmd )

# Clean-up hook for evpnConfig if bgpConfig is unconfigured.
def deleteEvpnConfigHook():
   evpnConfig.duplicateTimeout = evpnConfig.duplicateTimeoutDefault
   evpnConfig.duplicateThreshold = evpnConfig.duplicateThresholdDefault

deleteRouterBgpMacVrfHook.addExtension( deleteEvpnConfigHook )

blacklistWarning = ( 'EVPN/VXLAN are currently configured. Shutting down MLAG '
                     'in this state may lead to layer 2 loops which eventually '
                     'cause MAC blacklisting.' )

def canShutdownMlag( isPrimaryOrSecondaryMlag ):
   msg = None
   if ( isPrimaryOrSecondaryMlag and evpnStatus.encapType == 'encapVxlan' ):
      msg = blacklistWarning
   return msg

canShutdownMlagHook.addExtension( canShutdownMlag )

#-----------------------------------------------------------------------------------
# "[no|default] encapsulation vxlan layer-3 set next-hop igp-cost"
# under 'address-family evpn' mode.
# For L3 NLRI it sets IGP cost in the best path selection to underlay next hop
# IGP cost. In the future L2 related config can be accomodated here.
#-----------------------------------------------------------------------------------
class EvpnVxlanL3UseNhIgpCostCmd( BgpCmdBaseClass ):
   syntax = 'encapsulation vxlan layer-3 set next-hop igp-cost'
   noOrDefaultSyntax = syntax
   data = BgpCmdBaseClass._createSyntaxData( {
         'encapsulation' : bgpTokens.encap,
         'vxlan' : bgpTokens.vxlan,
         'layer-3' : EvpnTokens.layer3,
         'set' : bgpTokens.setForBestPathUse,
         'next-hop' : bgpTokens.setNhAttrForBestPath,
         'igp-cost' : bgpTokens.setNhIgpCost
   } )

   @staticmethod
   def _setUseNhIgpCostCommon( mode, vxlanL3UseNhIgpCost ):
      ''' Handler for 'encapsulation vxlan layer-3 set next-hop igp-cost'
      commands '''
      # config currently only supported in default vrf
      config = RoutingBgpCli.configForVrf( mode.vrfName )
      config.afEvpnVxlanL3UseNexthopIgpCost = vxlanL3UseNhIgpCost

   @staticmethod
   def _handleNormal( mode, args ):
      EvpnVxlanL3UseNhIgpCostCmd._setUseNhIgpCostCommon( mode, True )

   @staticmethod
   def _handleNoOrDefault( mode, args, noOrDefault ):
      EvpnVxlanL3UseNhIgpCostCmd._setUseNhIgpCostCommon( mode, False )

if BgpCommonToggleLib.toggleEvpnVxlanL3UseNexthopIgpCostEnabled():
   RouterBgpAfEvpnModelet.addCommandClass( EvpnVxlanL3UseNhIgpCostCmd )

#------------------------------------------------------------------
# "[ show | clear ] bgp evpn host-flap" 
#
# Shows or clears flapping MAC addresses from the blacklist.
#------------------------------------------------------------------

def showBgpEvpnHostFlapHandler( mode, args ):
   model = BgpEvpnHostFlapEntries()
   model.duplicationBlacklistedMacs = []
   for entry, time in macDuplicationBlacklist.blacklist.iteritems():
      blacklistedEntry = BgpEvpnHostFlapEntry()
      blacklistedEntry.vlan = entry.vlanId
      blacklistedEntry.macAddr = entry.macaddr
      blacklistedEntry.time = convertToUtcTime( time )
      model.duplicationBlacklistedMacs.append( blacklistedEntry )
   return model

def convertToUtcTime( timestamp ):
   timeStampInUtcTime = timestamp + Tac.utcNow() - Tac.now()
   return timeStampInUtcTime

class showBgpEvpnHostFlapCmd( ShowCommand.ShowCliCommandClass ):
   syntax = 'show bgp evpn host-flap'
   data = {
         'bgp': 'BGP information',
         'evpn' : evpnShow,
         'host-flap': 'MAC addresses blacklisted due to duplication'
         }
   cliModel = BgpEvpnHostFlapEntries
   handler = showBgpEvpnHostFlapHandler

class clearBgpEvpnHostFlapCmd( CliCommand.CliCommandClass ):
   syntax = 'clear bgp evpn host-flap'
   data = {
         'clear': CliToken.Clear.clearKwNode,
         'bgp': 'Bgp',
         'evpn': 'Evpn',
         'host-flap': 'MAC addresses blacklisted due to duplication'
         }

   @staticmethod
   def handler( mode, args ):
      evpnConfig.schedClearTime = Tac.now()

BasicCli.addShowCommandClass( showBgpEvpnHostFlapCmd ) 
BasicCli.EnableMode.addCommandClass( clearBgpEvpnHostFlapCmd )

# VPN domain id configuration under EVPN address-family.
# "domain identifier ASN:LOCAL_ADMIN"
class SetEvpnDomainIdCmd( CliCommand.CliCommandClass ):
   syntax = 'domain identifier DOMAIN_ID'
   noOrDefaultSyntax = 'domain identifier ...'
   data = { 'domain' : BgpVpnDomainIdCli.tokenDomain,
            'identifier' : BgpVpnDomainIdCli.tokenIdentifier,
            'DOMAIN_ID' : BgpVpnDomainIdCli.DomainIdExpression }
   @staticmethod
   def handler( mode, args ):
      mode.setVpnDomainId( args[ 'DOMAIN_ID' ] )

   @staticmethod
   def noOrDefaultHandler( mode, args ):
      mode.noVpnDomainId()

RouterBgpBaseAfEvpnMode.addCommandClass( SetEvpnDomainIdCmd )

#---------------------------------------------------------------------------------
# "[no|default] route import match-failure action discard"
# in "address-family evpn" mode.
#
# Enables the discarding of EVPN paths that won't be imported into any VRF (AID6625).
#---------------------------------------------------------------------------------

RouterBgpBaseAfEvpnMode.addCommandClass( RouteImportMatchFailureDiscardCmd )

#-------------------------------------------------------------------------------
# "[no|default] neighbor default next-hop-self" command,
# in "address-family evpn" mode
#-------------------------------------------------------------------------------

class EvpnRouteTypeExpression( CliCommand.CliExpression ):
   expression = 'RTYPE_KW'
   data = {
      'RTYPE_KW' : CliMatcher.EnumMatcher( { t[ 'keyword' ] : t[ 'helpdesc' ]
                     for t in nexthopSelfSupportNlris.values() } ),
   }

   @staticmethod
   def adapter( mode, args, argsList ):
      keywordToTypeMap = { nexthopSelfSupportNlris[ k ][ 'keyword' ] : k
            for k in nexthopSelfSupportNlris
      }
      types = set( [ keywordToTypeMap[ args[ 'RTYPE_KW' ] ] ] )
      args[ 'ROUTE_TYPES' ] = types

def setNhSelfEvpn( mode, vrfName, routeTypes ):
   config = RoutingBgpCli.configForVrf( vrfName )
   for rtype in routeTypes:
      if rtype in nexthopSelfSupportNlris:
         attr = 'nexthopSelfEvpnType' + str( rtype )
         setattr( config, attr, "isTrue" )

def noNhSelfEvpn( mode, vrfName, routeTypes ):
   config = RoutingBgpCli.configForVrf( vrfName )
   if config:
      for rtype in routeTypes:
         if rtype in nexthopSelfSupportNlris:
            attr = 'nexthopSelfEvpnType' + str( rtype )
            val = getattr( config, attr + 'Default' )
            setattr( config, attr, val )

class SetEvpnNeighborNextHopSelfCmd( BgpCmdBaseClass ):
   syntax = ( 'neighbor default next-hop-self received-evpn-routes '
              'route-type ROUTE_TYPES' )
   noOrDefaultSyntax = syntax
   data = {
         'neighbor' : bgpTokens.neighbor,
         'default' : bgpTokens.default,
         'next-hop-self' : bgpTokens.nextHopSelf,
         'received-evpn-routes' : EvpnTokens.receivedEvpnRoutes,
         'route-type' : EvpnTokens.routeTypeNhSelf,
         'ROUTE_TYPES' : EvpnRouteTypeExpression,
   }

   @staticmethod
   def _setNeighborNexthopSelfEvpn( mode, routeTypes ):
      setNhSelfEvpn( mode, mode.vrfName, routeTypes )

   @staticmethod
   def _noNeighborNexthopSelfEvpn( mode, routeTypes, noOrDefault ):
      noNhSelfEvpn( mode, mode.vrfName, routeTypes )

   @staticmethod
   def _handleNoOrDefault( mode, args, noOrDefault ):
      SetEvpnNeighborNextHopSelfCmd._noNeighborNexthopSelfEvpn(
         mode,
         args[ 'ROUTE_TYPES' ],
         noOrDefault )

   @staticmethod
   def _handleNormal( mode, args ):
      SetEvpnNeighborNextHopSelfCmd._setNeighborNexthopSelfEvpn(
         mode,
         args[ 'ROUTE_TYPES' ] )

if BgpCommonToggleLib.toggleArBgpAllowEvpnNexthopSelfEnabled():
   RouterBgpAfEvpnModelet.addCommandClass( SetEvpnNeighborNextHopSelfCmd )

# NOTE: This more properly belongs in the Vxlan package, but because it needs to
# mount both VCS and Evpn paths to source its data, and Vxlan can't depend
# on ArBgp because of dependency cycles, putting it here instead
#-------------------------------------------------------------------------------
# show vxlan control-plane [ import | export ] [ vlan <multi-range-rule> ]
#-------------------------------------------------------------------------------
class ShowVxlanControlPlaneCmd( ShowCommand.ShowCliCommandClass ):
   syntax = 'show vxlan control-plane [ import | export ] [ vlan VLANSET ]'
   data = {
      'vxlan': CliCommand.Node(
         matcher=CliMatcher.KeywordMatcher( 'vxlan',
            helpdesc='Configure VXLAN parameters' ),
         guard=vxlanSupportedGuard ),
      'control-plane': 'See control planes active in specific VLANs',
      'import': 'Restrict output to VLANs importing from a control plane',
      'export': 'Restrict output to VLANs exporting to a control plane',
      'vlan': 'Specify specific VLANs to show control plane information for',
      'VLANSET': vlanSetMatcher
   }
   cliModel = VxlanModel.VxlanControlPlaneConfigModel

   _DIRECTION_IMPORT = 0x1
   _DIRECTION_EXPORT = 0x2
   _DIRECTION_BOTH = _DIRECTION_IMPORT | _DIRECTION_EXPORT

   @classmethod
   def _vlanHasVniMapping( cls, vlan ):
      return any( vlan in vtiStatus.vlanToVniMap for
            vtiStatus in vtiStatusDir.vtiStatus.values() )

   @classmethod
   def _directionString( cls, direction ):
      if direction == cls._DIRECTION_IMPORT:
         return 'import'
      elif direction == cls._DIRECTION_EXPORT:
         return 'export'
      elif direction == cls._DIRECTION_BOTH:
         return 'both'
      else:
         return 'unknown'

   @classmethod
   def _addEvpnVlansToModel( cls, model, filterVlans, onlyImports, onlyExports ):
      encapType = Tac.Type( 'Evpn::EncapType' )
      # Check all vlans configured for EVPN to see if they're vlans of interest
      vlanToDirection = {}
      for macVrfConfig in bgpMacVrfConfigDir.config.values():
         # Only check vlan macvrfs
         if macVrfConfig.isVlanMacVrf():
            hasImports = bool( macVrfConfig.importRtList )
            hasExports = bool( macVrfConfig.exportRtList )

            # We only care about reporting this macvrf if it is importing or
            # exporting routes
            if not hasImports and not hasExports:
               continue

            direction = 0x0
            if hasImports:
               direction |= cls._DIRECTION_IMPORT
            if hasExports:
               direction |= cls._DIRECTION_EXPORT

            for brId in macVrfConfig.brIdToEtId:
               vlan = brId.vlanId
               # If we're filtering by specific vlans, skip vlans that we aren't
               # interested in
               if filterVlans and vlan not in filterVlans:
                  continue

               # We're only interested in vlans that have an encap type of Vxlan,
               if vlan in evpnStatus.imetLabel:
                  label = evpnStatus.imetLabel[ vlan ]
                  if label.encapType != encapType.encapVxlan:
                     continue
               else:
                  # If we can't check the encap type, just skip it
                  continue

               # We're only interested in vlans that have a vni mapping
               if not cls._vlanHasVniMapping( vlan ):
                  continue

               # If this is a vlan we haven't seen yet, remember the direction
               # If we've already seen this vlan, OR the new direction with the
               # existing direction
               # This handles the case where multiple macvrfs are importing/exporting
               # from the same vlans with different directions
               if vlan not in vlanToDirection:
                  vlanToDirection[ vlan ] = direction
               else:
                  vlanToDirection[ vlan ] |= direction

      # Update the model with the vlans of interest from EVPN
      for vlan, direction in vlanToDirection.items():
         # Do any direction filtering here, because multiple macvrfs can contribute
         # different directions to the same VLAN, and we want to filter on the
         # "combined" direction, not the direction of any individual macvrf
         if ( onlyImports and ( direction == cls._DIRECTION_EXPORT ) ) or \
               ( onlyExports and ( direction == cls._DIRECTION_IMPORT ) ):
            continue

         if vlan not in model.vlans:
            model.vlans[ vlan ] = VxlanModel.VlanControlPlaneConfigModel()

         if 'EVPN' not in model.vlans[ vlan ].controlPlanes:
            model.vlans[ vlan ].controlPlanes[ 'EVPN' ] = \
               VxlanModel.ControlPlaneConfigSourceModel()

         direction = VxlanModel.ControlPlaneConfigDirectionModel(
               direction=cls._directionString( direction ) )
         model.vlans[ vlan ].controlPlanes[ 'EVPN' ].sources[ 'configuration' ] = \
               direction

   @classmethod
   def _addVcsVlansToModel( cls, model, filterVlans, onlyImports, onlyExports ):
      def processStaticVlanDirections( directionAndSourceMap, vlanRange, direction,
            vtiStatus ):
         for vlan in computeVlanRangeSet( vlanRange ):
            # If we're filtering by specific vlans, skip vlans that we aren't
            # interested in
            if filterVlans and vlan not in filterVlans:
               continue

            # We're only interested in vlans that have a vni mapping
            if vlan not in vtiStatus.vlanToVniMap:
               continue

            # If this is a vlan we haven't seen yet, remember the direction and
            # source.  If we've already seen this vlan, OR the new direction with the
            # existing direction
            if vlan not in directionAndSourceMap:
               directionAndSourceMap[ vlan ] = { 'configuration': direction }
            elif 'configuration' not in directionAndSourceMap[ vlan ]:
               directionAndSourceMap[ vlan ][ 'configuration' ] = direction
            else:
               directionAndSourceMap[ vlan ][ 'configuration' ] |= direction

      # Combine configuration for both statically and dynamically configured vlans
      vlanToDirectionAndSource = {}

      for vtiIntfName, vtiStatus in vtiStatusDir.vtiStatus.items():
         # Skip adding any VCS entries if controller client mode is disabled on
         # this VTI
         if not vtiStatus.controllerClientMode:
            continue

         # Process vlans that have a static configuration for import or export
         vtiIntfId = Tac.Value( "Arnet::IntfId", vtiIntfName )
         vtiConfig = vtiConfigDir.vtiConfig.get( vtiIntfId )
         processStaticVlanDirections(
               vlanToDirectionAndSource,
               vtiConfig.cliVccImportVlans,
               cls._DIRECTION_IMPORT,
               vtiStatus )
         processStaticVlanDirections(
               vlanToDirectionAndSource,
               vtiConfig.cliVccExportVlans,
               cls._DIRECTION_EXPORT,
               vtiStatus )

         # Process vlans that have dynamic configurations
         for vlan, vsp in vtiStatus.vlanToVniMap.items():
            # If we're filtering by specific vlans, skip vlans that we aren't
            # interested in
            if filterVlans and vlan not in filterVlans:
               continue

            # Skip any vlans whose vni mapping source isn't in the dynamic vlan
            # source whitelist
            try:
               sourceValue = Tac.enumValue( 'Vxlan::VniSource', vsp.source )
               if ( sourceValue & vtiConfig.vccVlansSourceWhitelist ) != sourceValue:
                  continue
            except AttributeError:
               # If the source doesn't map to a dynamic enum value, we assume
               # it's a static entry from the CLI (static entries have an empty
               # source)
               continue

            # Currently, all dynamic vlans are treated as having direction "both"
            if vlan not in vlanToDirectionAndSource:
               vlanToDirectionAndSource[ vlan ] = { vsp.source: cls._DIRECTION_BOTH }
            elif vsp.source not in vlanToDirectionAndSource[ vlan ]:
               vlanToDirectionAndSource[ vlan ][ vsp.source ] = cls._DIRECTION_BOTH
            else:
               vlanToDirectionAndSource[ vlan ][ vsp.source ] |= cls._DIRECTION_BOTH

      # Update the model with the vlans of interest from the VCS configuration
      # sources
      for vlan, sourceAndDirection in vlanToDirectionAndSource.items():
         for source, direction in sourceAndDirection.items():
            # Do any direction filtering
            if ( onlyImports and ( direction == cls._DIRECTION_EXPORT ) ) or \
                  ( onlyExports and ( direction == cls._DIRECTION_IMPORT ) ):
               continue

            if vlan not in model.vlans:
               model.vlans[ vlan ] = VxlanModel.VlanControlPlaneConfigModel()

            if 'VCS' not in model.vlans[ vlan ].controlPlanes:
               model.vlans[ vlan ].controlPlanes[ 'VCS' ] = \
                     VxlanModel.ControlPlaneConfigSourceModel()

            directionModel = VxlanModel.ControlPlaneConfigDirectionModel(
                  direction=cls._directionString( direction ) )
            model.vlans[ vlan ].controlPlanes[ 'VCS' ].sources[ source ] = \
                  directionModel

   @classmethod
   def handler( cls, mode, args ):
      onlyImports = 'import' in args
      onlyExports = 'export' in args
      filterVlans = args[ 'VLANSET' ].ids if 'VLANSET' in args else None

      model = VxlanModel.VxlanControlPlaneConfigModel()

      cls._addEvpnVlansToModel( model, filterVlans, onlyImports, onlyExports )
      cls._addVcsVlansToModel( model, filterVlans, onlyImports, onlyExports )

      return model

BasicCli.addShowCommandClass( ShowVxlanControlPlaneCmd )

#---------------------------------------------------------------------------------
# "[no|default] next-hop resolution disabled" in "router-bgp-af" mode for EVPN
#---------------------------------------------------------------------------------
RouterBgpBaseAfEvpnMode.addCommandClass( NexthopResolutionDisabledCmd )

def Plugin( entityManager ):
   global ethernetSegmentConfigDir
   global vxlanCtrlConfig
   global evpnConfig
   global macDuplicationBlacklist
   global vtiConfigDir
   global vtiStatusDir
   global bgpMacVrfConfigDir
   global evpnStatus

   ethernetSegmentConfigDir = ConfigMount.mount( 
         entityManager, 'evpn/interface/config',
         'Evpn::EthernetSegmentCliConfigDir', 'w' )
   IntfCli.Intf.registerDependentClass( EvpnIntf )
   vxlanCtrlConfig = LazyMount.mount( entityManager,
                                      "vxlancontroller/config",
                                      "VxlanController::Config", "r" )
   
   # Boxing vxlanCtrlConfig since VniMatcher needs it at mod-load
   vxlanCtrlConfigBox.append( vxlanCtrlConfig )
   macDuplicationBlacklist = LazyMount.mount(
         entityManager, 'routing/bgp/macdupblacklist',
         'Routing::Bgp::MacPlugin::MacDuplicationBlacklist', 'r' )
   evpnConfig = ConfigMount.mount( entityManager, 'routing/bgp/evpnconfig',
                                   'Routing::Bgp::MacPlugin::EvpnConfig', 'w' )

   # Need read access to vtiConfigDir and vtiStatusDir for the implementation of
   # the vxlan show control-plane command
   vtiConfigDir = LazyMount.mount( entityManager,
         "interface/config/eth/vxlan", "Vxlan::VtiConfigDir", "r" )
   vtiStatusDir = LazyMount.mount( entityManager,
         "interface/status/eth/vxlan", "Vxlan::VtiStatusDir", "r" )
   # Need read access to macVrfConfigDir and evpnStatus for the implementation of
   # the vxlan show control-plane command
   bgpMacVrfConfigDir = LazyMount.mount( entityManager,
         "routing/bgp/macvrf", "Routing::Bgp::MacVrfConfigDir", "r" )
   evpnStatus = LazyMount.mount( entityManager,
         "evpn/status", "Evpn::EvpnStatus", "r" )
