#!/usr/bin/env python
# Copyright (c) 2019 Arista Networks, Inc.  All rights reserved.
# Arista Networks, Inc. Confidential and Proprietary.

from __future__ import absolute_import, division, print_function

# pylint: disable=ungrouped-imports

import Tac
import CliCommand
import CliMatcher
import Url
import CliPlugin.EventMonCli as EventMonCli
import BasicCliModes

matcherBackup = CliMatcher.KeywordMatcher( 'backup', 
      helpdesc='Backed up log files' )
matcherBuffer = CliMatcher.KeywordMatcher( 
      'buffer', helpdesc='Local buffer settings' )
matcherEventMonitor = CliMatcher.KeywordMatcher( 
      'event-monitor', helpdesc='Enable event-monitor agent' )
matcherMaxSize = CliMatcher.KeywordMatcher( 
      'max-size', helpdesc='Maximum number of stored backup logs' )
matcherMaxBufSize = CliMatcher.KeywordMatcher( 
      'max-size', helpdesc='Maximum size of the primary event monitor log' )
matcherPath = CliMatcher.KeywordMatcher( 'path', helpdesc='Path to backup logs' )
matcherTables = CliCommand.Node(
      matcher=CliMatcher.EnumMatcher( {
         'all': 'All tables',
         'route': 'Route table',
         'route6': 'Route6 table',
         'mroute': 'Mroute table',
         'lacp': 'LACP table',
         'arp': 'ARP entries',
         'neighbor': 'Neighbor table',
         'mac': 'MAC table',
         'igmpsnooping': 'IGMP Snooping table',
         'stpunstable': 'STP Unstable table',
         } ) )

BackupSizeType = Tac.Type( 'EventMon::BackupSize' )
BufferSizeType = Tac.Type( 'EventMon::BufferSize' )

#--------------------------------------------------------------------------------
# ( no | default ) event-monitor backup max-size SIZE
#--------------------------------------------------------------------------------
class EventMonitorBackupMaxSizeCmd( CliCommand.CliCommandClass ):
   syntax = 'event-monitor backup max-size SIZE'
   noOrDefaultSyntax = 'event-monitor backup max-size ...'
   data = {
      'event-monitor': matcherEventMonitor,
      'backup': matcherBackup,
      'max-size': matcherMaxSize,
      'SIZE': CliMatcher.IntegerMatcher( BackupSizeType.min, BackupSizeType.max, 
         helpdesc='Maximum number of stored backup logs' )
   }
   handler = EventMonCli.setForeverLogMaxSize
   noOrDefaultHandler = EventMonCli.setForeverLogMaxSize

BasicCliModes.GlobalConfigMode.addCommandClass( EventMonitorBackupMaxSizeCmd )

#--------------------------------------------------------------------------------
# [ no | default ] event-monitor 
#  [route|route6|mroute|lacp|arp|neighbor|mac|igmpsnooping|
#  stpunstable] backup path URL
#--------------------------------------------------------------------------------
class EventMonitorBackupPathCmd( CliCommand.CliCommandClass ):
   syntax = \
      ('event-monitor [ TABLE ] backup path URL')
   noOrDefaultSyntax = ('event-monitor [ TABLE ] backup [ path ] ...')
   data = {
      'event-monitor': matcherEventMonitor,
      'TABLE': matcherTables,
      'backup': matcherBackup,
      'path': matcherPath,
      'URL': Url.UrlMatcher(
         fsFunc=lambda fs: fs.fsType == 'flash', helpdesc='forever log URL' ), 

   }
   handler = EventMonCli.setForeverLogPath
   noOrDefaultHandler = EventMonCli.restoreForeverLogPath

BasicCliModes.GlobalConfigMode.addCommandClass( EventMonitorBackupPathCmd )

#--------------------------------------------------------------------------------
# [ no | default ] event-monitor buffer max-size SIZE
#--------------------------------------------------------------------------------
class EventMonitorBufferMaxSizeCmd( CliCommand.CliCommandClass ):
   syntax = 'event-monitor buffer max-size SIZE'
   noOrDefaultSyntax = 'event-monitor buffer max-size ...'
   data = {
      'event-monitor': matcherEventMonitor,
      'buffer': matcherBuffer,
      'max-size': matcherMaxBufSize,
      'SIZE': CliMatcher.IntegerMatcher( BufferSizeType.min, BufferSizeType.max, 
         helpdesc='Maximum size of the primary event monitor log' ),
   }
   handler = EventMonCli.setBufferSize
   noOrDefaultHandler = EventMonCli.setBufferSize

BasicCliModes.GlobalConfigMode.addCommandClass( EventMonitorBufferMaxSizeCmd )

#--------------------------------------------------------------------------------
# [ no | default ] event-monitor
#--------------------------------------------------------------------------------
class EventMonitorCmd( CliCommand.CliCommandClass ):
   syntax = 'event-monitor'
   noOrDefaultSyntax = syntax
   data = {
      'event-monitor': 'Enable event-monitor agent',
   }

   @staticmethod
   def handler( mode, args ):
      EventMonCli.config.agentEnabled = True

   @staticmethod
   def noHandler ( mode, args ):
      EventMonCli.config.agentEnabled = False

   @staticmethod
   def defaultHandler ( mode, args ):
      EventMonCli.config.agentEnabled = EventMonCli.config.defaultAgentEnabled

BasicCliModes.GlobalConfigMode.addCommandClass( EventMonitorCmd )

#--------------------------------------------------------------------------------
# [ no | default ] event-monitor 
#   [all|route|route6|mroute|lacp|arp|neighbor|mac|
#   igmpsnooping|stpunstable]
#--------------------------------------------------------------------------------
class EventMonitorTableCmd( CliCommand.CliCommandClass ):
   syntax = ('event-monitor TABLE')
   noOrDefaultSyntax = syntax
   data = {
      'event-monitor': 'Enable event-monitor agent',
      'TABLE': matcherTables,
   }

   @staticmethod
   def handler( mode, args ):
      enabledTable = args[ 'TABLE' ]
      if enabledTable == 'all':
         for table in EventMonCli.config.table.itervalues():
            table.enabled = True
      else:
         EventMonCli.config.table[enabledTable].enabled = True

   @staticmethod
   def noHandler ( mode, args ):
      disabledTable = args[ 'TABLE' ]
      if disabledTable == 'all':
         for table in EventMonCli.config.table.itervalues():
            table.enabled = False
      else:
         EventMonCli.config.table[disabledTable].enabled = False

   @staticmethod
   def defaultHandler ( mode, args ):
      enabled = Tac.newInstance( 'EventMon::TableConfig', '' ).defaultEnabled
      defaultTable = args[ 'TABLE' ]
      if defaultTable == 'all':
         for table in EventMonCli.config.table.itervalues():
            table.enabled = enabled
      else:
         EventMonCli.config.table[defaultTable].enabled = enabled

BasicCliModes.GlobalConfigMode.addCommandClass( EventMonitorTableCmd )

