#!/usr/bin/env python
# Copyright (c) 2010, 2011 Arista Networks, Inc.  All rights reserved.
# Arista Networks, Inc. Confidential and Proprietary.

# pylint: disable-msg=R0201
# pylint: disable-msg=E1103

import Tac, os, re
import LazyMount
import ConfigMount
import subprocess
import Intf.IntfRange
import sqlite3

BufferSizeType = Tac.Type( 'EventMon::BufferSize' )
BackupSizeType = Tac.Type( 'EventMon::BackupSize' )

# Module globals set up by the Plugin
config = None
status = None
epochMarker = None

#-------------------------------------------------------------------------------
# Tokens and functions used for supported sql statements
# show event-monitor {tables} {filter(s)}
#-------------------------------------------------------------------------------


# Take the filters specified at the Cli, and translate them
# into a suffix-clause to append onto our sql query
def generateSqlClauseFromFilters( mode, args ):
   groupClause = ''
   whereClause = ''
   limitClause = ''
   whereArgs = []
   if 'group-by' in args:
      groupVar = args[ 'GROUPBY' ]
      if groupVar == 'ip' or groupVar == 'ip6':
         groupVar = 'prefix'
      elif groupVar == 'interface' or groupVar == 'member':
         groupVar = 'intf'
      elif groupVar == 'mac':
         groupVar = 'ethAddr'
      elif groupVar == 'group-ip':
         groupVar = 'groupIp'
      elif groupVar == 'source-ip':
         groupVar = 'sourceIp'
      elif groupVar == 'state-machine':
         groupVar = 'statemachine'
      elif groupVar == 'port-channel':
         groupVar = 'portchannel'
      groupClause += 'group by %s' % groupVar
   if 'limit' in args:
      limitClause += 'limit %d' % args[ 'LIMIT' ]
   if 'match-time' in args:
      column = 'time'
      matchTime = args.get( 'TIME' )
      if matchTime == 'last-minute':
         whereClause = 'strftime("%%s","now") - '\
               'strftime("%%s",%s,"utc") < %d' % ( column, 60 )
         whereArgs.append(whereClause)
      elif matchTime == 'last-hour':
         whereClause = 'strftime("%%s","now") - '\
               'strftime("%%s",%s,"utc") < %d' % ( column, 60 * 60 )
         whereArgs.append(whereClause)
      elif matchTime == 'last-day':
         whereClause = 'strftime("%%s","now") - '\
               'strftime("%%s",%s,"utc") < %d' % ( column, 60 * 60 * 24 )
         whereArgs.append(whereClause)
      elif matchTime == 'last-week':
         whereClause = 'strftime("%%s","now") - '\
               'strftime("%%s",%s,"utc") < %d' % ( column, 60 * 60 * 24 * 7 )
         whereArgs.append(whereClause)
   if 'match-vlan' in args:
      whereClause = "vlan LIKE '%s'" % ( args[ 'VLANID' ][ 0 ] )
      whereArgs.append(whereClause)
   if 'match-ip' in args:
      whereClause = "prefix LIKE '%s'" % ( args[ 'IPADDRESS' ][ 0 ] )
      whereArgs.append( whereClause )
   if 'match-ipv6' in args:
      whereClause = "prefix LIKE '%s'" % ( args[ 'IPV6ADDRESS' ][ 0 ] )
      whereArgs.append( whereClause )
   if 'match-vrf' in args:
      whereClause = "vrf LIKE '%s'" % ( args[ 'VRF' ][ 0 ] )
      whereArgs.append( whereClause )
   if 'match-interface' in args:
      intfs = []
      if 'INTF1' in args:
         intfs = Intf.IntfRange.intfListFromCanonical( [ args[ 'INTF1' ][ 0 ] ],
            shortName=False,
            mode=mode )
         whereClause = ( 
            "( " + ( ' OR '.join( "intf = '%s'" % i for i in intfs ) ) + " )" )
      else:
         if 'LAG' in args:
            intfs = Intf.IntfRange.intfListFromCanonical( [ args[ 'LAG' ] ],
               shortName=False,
               mode=mode )
            whereClause = (
               "( " + ( ' OR '.join( 
                  "portchannel = '%s'" % i for i in intfs ) ) + " )" )
         elif 'ETH' in args:
            intfs = Intf.IntfRange.intfListFromCanonical( [ args[ 'ETH' ] ],
               shortName=False,
               mode=mode )
            whereClause = (
               "( " + ( ' OR '.join( "intf = '%s'" % i for i in intfs ) ) + " )" )
      whereArgs.append( whereClause )
   if 'match-mac' in args:
      whereClause = "ethAddr LIKE '%s'" % ( args[ 'MACADDRESS' ][ 0 ] )
      whereArgs.append( whereClause )
   if 'match-sm' in args:
      whereClause = "statemachine LIKE '%s'" % ( args[ 'MATCH_SM' ] )
      whereArgs.append( whereClause )
   if 'match-source-ip' in args:
      whereClause = "sourceIp LIKE '%s'" % ( args[ 'SRCADDR' ][ 0 ] )
      whereArgs.append( whereClause )
   if 'match-group-ip' in args:
      whereClause = "groupIp LIKE '%s'" % ( args[ 'GRPADDR' ][ 0 ] )
      whereArgs.append( whereClause )
   #combine the where args to form one where clause:
   if whereArgs:
      whereClause = ' WHERE ' + (' AND '.join( map( str,whereArgs ) ) )
   else:
      whereClause = ''
   return whereClause + ' ' + groupClause + ' ' + limitClause

def showTable( mode, filters, tableName ):
   if not config.table[ tableName ].enabled:
      mode.addError( "event-monitor not enabled for %s" % ( tableName ) )
      return
   
   syncBuffer( mode, [] )
   suffix = generateSqlClauseFromFilters( mode, filters )
   statement = "select * from %s %s;" % ( tableName, suffix )
   try:
      os.stat( config.dbLocation )
   except OSError:
      mode.addError( "Database does not exist. "
                     "Run 'event-monitor sync'" )
      return
   printSqlite3Statement( mode, statement )


def showTables( mode, args ):
   if not agentRunnable():
      mode.addError( "event-monitor not running. "
                     "It can be started by running 'event-monitor' " 
                     "command in config mode" )
      return

   try:
      os.stat( config.dbLocation )
   except OSError:
      mode.addError( "Database does not exist. "
                     "Restart event-monitor by running 'no event-monitor' "
                     "followed by 'event-monitor' in config mode" )
      return
   
   syncBuffer( mode, [] )

   for table in config.table.itervalues():
      tableName = table.name
      if not config.table[ tableName ].enabled:
         continue

      statement = "select * from %s;" % ( tableName)
      printSqlite3Statement( mode, statement )

#-------------------------------------------------------------------------------
# Commands used for manual interactions with buffer/database
#
# show event-monitor sqlite {sqlstatement}
# event-monitor interact 
# event-monitor sync 
# event-monitor clear
#-------------------------------------------------------------------------------
def restartEventMonAll( func ):
   def innerFunc( *args, **kw ):
      enabledTables = [ table for table in config.table.itervalues()
                        if table.enabled ]
      for table in enabledTables:
         table.enabled = False
      try:
         func( *args, **kw )
      finally:
         for table in enabledTables:
            table.enabled = True
   return innerFunc

def agentRunnable():
   return config.agentEnabled or config.agentEnabledOverride

@restartEventMonAll
def clearLogs( mode, args ):
   if not agentRunnable():
      mode.addError( "event-monitor not running" )
      return
   
   epochMarker.bufferFlushEpoch += 1
   try:
      Tac.waitFor( 
         lambda: 
            status.newBufferFlushEpoch == epochMarker.bufferFlushEpoch,
         description='buffer clear to complete',
         warnAfter=None, sleep=True, maxDelay=0.5, timeout=15 )
   except Tac.Timeout:
      mode.addError( 'Buffer clear timed out, restarting EventMon' )

def syncBuffer( mode, args ):
   if not agentRunnable():
      mode.addError( "event-monitor not running" )
      return
 
   epochMarker.syncEpoch += 1
   try:
      Tac.waitFor( 
         lambda: 
            epochMarker.syncEpoch == status.newSyncEpoch,
         description='database sync to complete',
         warnAfter=None, sleep=True, maxDelay=0.5, timeout=150 )
   except Tac.Timeout:
      mode.addError( 'Database sync timed out.'
            'event-monitor may still be syncing...' )


def prepShowEventMon( mode, args ):
   if not agentRunnable():
      mode.addError( "event-monitor not running" )
      return

   syncBuffer( mode, [] )

def showEventMon( mode, args ):
   statementContent = args[ 'QUERY' ]
   printSqlite3Statement( mode, statementContent[ 0 ] )


def printSqlite3Statement( mode, statement ):
   if not config.agentEnabled:
      return 

   try:
      conn = sqlite3.connect( config.dbLocation )
      c = conn.cursor()
      for row in c.execute( statement ):
         print( '|'.join( map( str, row ) ) )
      conn.close()
   except sqlite3.Error:
      mode.addError( "Cannot read from EventMon DB table" )


def interactSQLite( mode, args ):
   try:
      subprocess.call( [ '/usr/bin/sqlite3', config.dbLocation ], 
                       shell=False )
   except OSError:
      mode.addError( "Failed to open EventMon DB for interaction" )

@restartEventMonAll
def setBufferSize( mode, args ):
   size = args.get( 'SIZE', 32 )
   config.bufferSize = BufferSizeType( size )

def setForeverLogMaxSize( mode, args ):
   size = args.get( 'SIZE', 10 )
   config.foreverLogMaxSize = BackupSizeType( size )
   # currently all tables get configured with same backup size
   for table in config.table.itervalues():
      table.foreverLogMaxSize = config.foreverLogMaxSize

def setEventMonAgentOverride():
   agentEnabledOverride = False 
   if config.foreverLogEnabled:
      agentEnabledOverride = True
   else:
      for table in config.table.itervalues():
         if table.foreverLogOverride and table.foreverLogEnabled:
            agentEnabledOverride = True
   config.agentEnabledOverride = agentEnabledOverride

@restartEventMonAll
def disableForeverLog( mode, args ):
   assert( config.defaultForeverLogEnabled == False )
   for table in config.table.itervalues():
      table.foreverLogOverride = False
   restoreGlobalForeverLogPath( mode )

def ensureDir( f ):
   d = os.path.dirname( f )
   if not os.path.exists( d ):
      os.makedirs( d )
   
def verifyForeverLogUrl( mode, url ):
   directoryList = url.parent().listdir()
   for entry in directoryList:
      if entry.startswith( url.basename() ):
         match = re.findall( '%s.(\d+)' % url.basename(), entry ) 
         if not match:
            mode.addError( 'Conflicting file/directory at specified location' )
            return False
         expectedFilename = "%s.%d" % ( url.basename(), int( match[ 0 ] ) )
         if ( not entry == expectedFilename
               ) or os.path.isdir( expectedFilename ):
            mode.addError( 'Conflicting file/directory at specified location' )
            return False
   return True

def configTableForeverLog( tableName, url=None ):
   table = config.table[ tableName ]
   tableEnabled = table.enabled
   # disable the table while making these changes
   table.enabled = False
   if url:
      if isinstance( url, str ):
         fileName = os.path.basename( url )
         fileDir = os.path.dirname( url )         
      else:
         fileDir = url.parent().localFilename()
         fileName = url.basename()
      table.foreverLogDirectory = fileDir
      table.foreverLogFileName = fileName
      table.foreverLogEnabled = True
      # enable agent as forever logging enabled
   else:
      table.foreverLogDirectory = config.foreverLogDirectory
      table.foreverLogFileName = config.foreverLogFileName
      table.foreverLogEnabled = config.foreverLogEnabled
   # re-enable table (if it was enabled) to re-initialize
   # the QT buffers with the new changes
   table.enabled = tableEnabled

# check whether folder path is allowed or not
def checkAllowedPath( path ):
   # if no restrictions
   # allowed paths on 'file:' : '/var/tmp', '/tmp' and 'var/log' 
   if path.fs.allowedPaths_ == None:
      return True

   fName = path.localFilename()
   if any( x for x in path.fs.allowedPaths_ if fName == x ):
      return False
   return True   

# set forever log path for specified table to be the given url
def setTableForeverLogPath( mode, table, url ):
   # Make the parent directory if it doesn't exist
   ensureDir( url.localFilename() )
   if url.parent().isdir():
      if not verifyForeverLogUrl( mode, url ):
         return
      if not checkAllowedPath(url) or url == "flash:/":
         mode.addError( 'Forever log parent directory cannot be flash root' )
         return
   else:
      mode.addError( 'Invalid file or directory' )
      return
   config.table[ table ].foreverLogOverride = True
   configTableForeverLog( table, url )
   setEventMonAgentOverride()

@restartEventMonAll
def setGlobalForeverLogPath( mode, url ):
   # Make the parent directory if it doesn't exist
   ensureDir( url.localFilename() )
   if url.parent().isdir():
      if not verifyForeverLogUrl( mode, url ):
         return
      if not checkAllowedPath(url) or url == "flash:/":
         mode.addError( 'Forever log parent directory cannot be flash root' )
         return

      config.foreverLogDirectory = url.parent().localFilename()
      config.foreverLogFileName  = url.basename()
      config.foreverLogEnabled   = True
      # set all non-overriden table configs to the global values
      for tableName, table in config.table.iteritems():
         if not table.foreverLogOverride:
            configTableForeverLog( tableName )
   else:
      mode.addError( 'Invalid file or directory' )
   setEventMonAgentOverride()

def setForeverLogPath( mode, args ):
   url = args['URL']
   if 'TABLE' in args and args[ 'TABLE' ] != 'all':
      setTableForeverLogPath( mode, args[ 'TABLE' ], url )
   else:
      setGlobalForeverLogPath( mode, url )

# restore forever log path for specified table to be the same as global setting
def restoreTableForeverLogPath( mode, table ):
   config.table[ table ].foreverLogOverride = False
   configTableForeverLog( table )
   setEventMonAgentOverride()

def restoreGlobalForeverLogPath( mode ):
   config.foreverLogDirectory = config.defaultForeverLogDirectory
   config.foreverLogFileName  = config.defaultForeverLogFileName
   config.foreverLogEnabled   = config.defaultForeverLogEnabled
   # restore all non-overriden table configs to the global values
   for tableName, table in config.table.iteritems():
      if not table.foreverLogOverride:
         configTableForeverLog( tableName )
   setEventMonAgentOverride()

def restoreForeverLogPath(mode, args ):
   if 'path' not in args:
      disableForeverLog(mode, args)
   if 'TABLE' in args and args[ 'TABLE' ] !=  'all':
      restoreTableForeverLogPath(mode, args[ 'TABLE' ])
   else:
      restoreGlobalForeverLogPath( mode )

#-------------------------------------------------------------------------------
# Have the Cli Agent mount all needed state from sysdb
#-------------------------------------------------------------------------------
def Plugin( entityManager ):
   global config, status, epochMarker
   config = ConfigMount.mount( entityManager, "eventMon/config", 
                   "EventMon::Config", "w" )
   epochMarker = LazyMount.mount( entityManager, "eventMon/epochMarker", 
                   "EventMon::EpochMarker", "w" )
   status = LazyMount.mount( entityManager, "eventMon/status", 
                   "EventMon::Status", "r" )
