# Copyright (c) 2017 Arista Networks, Inc.  All rights reserved.
# Arista Networks, Inc. Confidential and Proprietary.

from __future__ import absolute_import, division, print_function

from datetime import datetime
import prettytable
import Tac
import Cell
import LazyMount
import BasicCli
import CliMatcher
import CliModel
from CliPlugin.AgentCli import agentNameNewMatcher
import CliPlugin.TechSupportCli
import CliToken.Agent
import EventHistoryUtil
import ShowCommand

eventHistoryPersistent = None
eventHistoryDefault = None
eventHistoryRestored = None
eventHistoryMultiCell = None
_asuHwStatus = None

class EventHistoryModel( CliModel.Model ):
   class LogEntryModel( CliModel.Model ):
      timestamp = CliModel.Float( help="Timestamp when the event was logged" )
      category = CliModel.Str( help="EventHistory category" )
      agentName = CliModel.Str( help="Agent name" )
      messageGroup = CliModel.Str( help="Message group" )
      message = CliModel.Str( help="Log message" )

   events = CliModel.List( valueType=LogEntryModel,
      help="List of EventHistory events", optional=True )
   filters = CliModel.Dict( keyType=str, valueType=str,
         help="Regular expression filters to get a subset of EventHistory",
         optional=True )

   def render( self ):
      print( "Output filters: Agent=\"%s\", Group=\"%s\"" % ( str( self.filters[
         "agent" ] ), str( self.filters[ "group" ] ) ) )
      print( "" )
      t = prettytable.PrettyTable()
      t.field_names = [ "Timestamp", "Category", "Agent", "Group", "Message" ]
      t.hrules = prettytable.HEADER
      t.vrules = prettytable.NONE
      t.align = "r"
      for event in self.events:
         timevalue = datetime.fromtimestamp( event.timestamp )
         formattedTime = timevalue.strftime( '%Y-%m-%d %H:%M:%S.%f' )
         t.add_row( [ formattedTime, event.category, event.agentName,
                     event.messageGroup, '"' + event.message + '"'] )
      t.align = "l"
      print( t )

def doShowEventHistory( mode, args ):
   agent = args.get( 'AGENT' )
   group = args.get( 'GROUP' )
   eventHistoryModel = EventHistoryModel()
   eventHistoryModel.filters[ "agent" ] = agent if agent else "None"
   eventHistoryModel.filters[ "group" ] = group if group else "None"
   # Since C++ expects a fully mounted directory, mounts must be forced.
   eventHistoryDirs = [
      eventHistoryPersistent.force(),
      eventHistoryDefault.force(),
      eventHistoryRestored.force(),
   ]
   for cellDir in eventHistoryMultiCell.itervalues():
      eventHistoryDirs.append( cellDir )

   eventHistoryReaderStatus = EventHistoryUtil.doReadEventHistory( eventHistoryDirs,
      agentPattern=agent, groupPattern=group )

   for logEntry in eventHistoryReaderStatus.log.itervalues():
      logEntryModel = EventHistoryModel.LogEntryModel()
      logEntryModel.timestamp = logEntry.timestamp
      logEntryModel.category = logEntry.category
      logEntryModel.agentName = logEntry.agentName
      logEntryModel.messageGroup = logEntry.messageGroup
      logEntryModel.message = logEntry.message
      eventHistoryModel.events.append( logEntryModel )
   return eventHistoryModel

#--------------------------------------------------------------------------------
# show agent [ AGENT ] event history [ group GROUP ]
#
# Command to show EventHistory events in chronological order.
#--------------------------------------------------------------------------------
class AgentAgentEventHistoryCmd( ShowCommand.ShowCliCommandClass ):
   syntax = 'show agent [ AGENT ] event history [ group GROUP ]'
   data = {
      'agent': CliToken.Agent.agentKwForShow,
      'AGENT': agentNameNewMatcher,
      'event': 'Show agent Event History',
      'history': 'Show Event History events in chronological order',
      'group': 'Filter EventHistory by Message Group',
      'GROUP': CliMatcher.PatternMatcher( pattern='.+',
         helpdesc='Regular Expression to filter EventHistory', helpname='WORD' ),
   }
   handler = doShowEventHistory
   cliModel = EventHistoryModel

BasicCli.addShowCommandClass( AgentAgentEventHistoryCmd )

#------------------------------------------------------------------------------------
# 'show tech support' registration
#
# Register 'show agent event history' as part of 'show tech support' command.
#------------------------------------------------------------------------------------
def _showTechCmds():
   return [ 'show agent event history' ]

CliPlugin.TechSupportCli.registerShowTechSupportCmdCallback(
   '2018-01-09 14:08:14', _showTechCmds, summaryCmdCallback=_showTechCmds )

#--------------------------------------------------
# Plugin method - Mount the objects we need from Sysdb
#--------------------------------------------------
def Plugin( entityManager ):
   global eventHistoryPersistent
   global eventHistoryDefault
   global eventHistoryRestored
   global eventHistoryMultiCell
   global _asuHwStatus

   eventHistoryPersistent = LazyMount.mount( entityManager, Cell.path(
      'eventhistory/persistent' ), 'Tac::Dir', 'ri' )
   eventHistoryDefault = LazyMount.mount( entityManager, Cell.path(
      'eventhistory/default' ), 'Tac::Dir', 'ri' )
   eventHistoryRestored = LazyMount.mount( entityManager, Cell.path(
      'eventhistory/restored' ), 'Tac::Dir', 'ri' )
   eventHistoryMultiCell = LazyMount.mount( entityManager, 'eventhistory',
         'Tac::Dir', 'ri')
   _asuHwStatus = LazyMount.mount( entityManager, 'asu/hardware/status',
         'Asu::AsuStatus', 'r' )
