# Copyright (c) 2011 Arista Networks, Inc.  All rights reserved.
# Arista Networks, Inc. Confidential and Proprietary.

import re
import os

import BasicCli
import BasicCliUtil
import CliCommand
import CliMatcher
from CliMode.EventMgr import EventHandlerMode, TriggerOnCountersMode, \
                             TriggerOnLoggingMode
import CliParser
import CliPlugin.HealthCli as HealthCli
from CliPlugin.MaintenanceCliLib import dynamicUnitName
from CliPlugin.MaintenanceCliLib import maintEnterStageClass, maintExitStageClass
import ConfigMount
from EventCliModel import EventHandler, EventHandlers, EventHandlerTrigger, \
   EventHandlerTriggerCounters, EventSubhandler
from EventCounters import Counters
from EventLib import actionKindCliToTacc, triggerTypeCliToTacc
from EventLib import getSubhandlerName
import LazyMount
import Tac
import Tracing

__defaultTraceHandle__ = Tracing.Handle( "EventCli" )
t0 = Tracing.trace0
t1 = Tracing.trace1
t2 = Tracing.trace2
t5 = Tracing.trace5
t8 = Tracing.trace8
t9 = Tracing.trace9

config = None
status = None
allVrfConfig = None
healthStatus = None

# intf inputs operstatus and ip default to False

# router(config)#event-handler [hName]
# router(config-handler-hName)#trigger on-boot
# router(config-handler-hName)#trigger on-intf intfName [operstatus] [ip] [ip6]
# router(config-handler-hName)#trigger on-counters
# router(config-handler-hName-counters)#poll interval [sec]
# router(config-handler-hName-counters)#condition [logical-expression]
# router(config-handler-hName)#trigger on-logging
# router(config-handler-hName-logging)#poll interval [sec]
# router(config-handler-hName-logging)#regex [regular-expression]
# router(config-handler-hName)#action bash [cmd]
# router(config-handler-hName)#delay [delayPeriod]
# router(config-handler-hName)#repeat interval [intervalPeriod]
# router(config-handler-hName)#threshold [timeWindow] count [eventCount]
# router(config-handler-hName)#exit

#-------------------------------------------------------------------------------
# The "config-event-handler" mode.
#-------------------------------------------------------------------------------
class EventHandlerConfigMode( EventHandlerMode, BasicCli.ConfigModeBase ):

   #----------------------------------------------------------------------------
   # Attributes required of every Mode class.
   #----------------------------------------------------------------------------
   name = 'Event handler configuration'
   modeParseTree = CliParser.ModeParseTree()

   #----------------------------------------------------------------------------
   # Constructs a new EventConfig and begins config-event-handler mode
   #----------------------------------------------------------------------------
   def __init__( self, parent, session, handlerName ):
      self.handlerName = handlerName
      # Set values to those of prior Config if exists, else defaults (set here)
      if self.handlerName in config.event:
         eventConfig = config.event.get( self.handlerName )
         self.copyEventHandler( self, eventConfig )
      else:
         self.triggerType = config.defaultTriggerType
         self.intfName = config.defaultIntfName
         self.operstatus = False
         self.ip = False
         self.ip6 = False
         self.actionKind = config.defaultActionKind
         self.command = config.defaultCommand
         self.delay = config.defaultDelay
         self.repeatInterval = config.defaultRepeatInterval
         self.asynchronous = config.defaultAsynchronous
         self.timeout = config.defaultTimeout
         self.thresholdCount = config.defaultThresholdCount
         self.threshold = config.defaultThreshold
         self.pollInterval = config.defaultPollInterval
         self.countersCondition = config.defaultCountersCondition
         self.hasSubhandlers = config.defaultHasSubhandlers
         self.logRegex = config.defaultLogRegex
         self.maintenanceUnitName = config.defaultMaintenanceUnitName
         self.maintenanceStage = config.defaultMaintenanceStage
         self.maintenanceOper = config.defaultMaintenanceOper
         self.maintenanceBgpPeer = config.defaultMaintenanceBgpPeer
         self.vrfName = config.defaultVrfName
         self.metricName = config.defaultMetricName
      EventHandlerMode.__init__( self, handlerName )
      BasicCli.ConfigModeBase.__init__( self, parent, session )

   def copyEventHandler( self, dst, src ):
      dst.triggerType = src.triggerType
      dst.operstatus = src.operstatus
      dst.ip = src.ip
      dst.ip6 = src.ip6
      dst.intfName = src.intfName
      dst.actionKind = src.actionKind
      dst.command = src.command
      dst.delay = src.delay
      dst.repeatInterval = src.repeatInterval
      dst.asynchronous = src.asynchronous
      dst.timeout = src.timeout
      dst.thresholdCount = src.thresholdCount
      dst.threshold = src.threshold
      dst.pollInterval = src.pollInterval
      dst.countersCondition = src.countersCondition
      dst.logRegex = src.logRegex
      dst.maintenanceUnitName = src.maintenanceUnitName
      dst.maintenanceStage = src.maintenanceStage
      dst.maintenanceOper = src.maintenanceOper
      dst.maintenanceBgpPeer = src.maintenanceBgpPeer
      dst.vrfName = src.vrfName
      dst.metricName = src.metricName
      dst.hasSubhandlers = src.hasSubhandlers

   def setTriggerType( self, args ):
      triggerType = 'on-boot' if 'on-boot' in args else 'on-startup-config'
      assert triggerType in args
      self.triggerType = triggerTypeCliToTacc[ triggerType ]
      self.intfName = config.defaultIntfName
      self.operstatus = False
      self.ip = False
      self.ip6 = False

   def setOnIntf( self, args ):
      intf = args[ 'INTF' ]
      inputs = args[ 'INPUTS' ]
      self.triggerType = 'onIntf'
      # Used to lookup the deviceName for the kernel interface
      self.intfName = intf.strWithLongTag()
      for i in intf.intfNames():
         self.intfName += ' ' + i

      for att in [ 'operstatus', 'ip', 'ip6' ]:
         value = att in inputs
         setattr( self, att, value )

   def setOnMaintenance( self, maintenanceOper, unitName, maintenanceStage,
                         peer="", vrfName="", intfName="" ):
      self.triggerType = 'onMaintenance'
      self.maintenanceUnitName = unitName
      if maintenanceOper == 'enter':
         self.maintenanceOper = maintEnterStageClass
      else:
         self.maintenanceOper = maintExitStageClass
      self.maintenanceStage = maintenanceStage
      self.maintenanceBgpPeer = peer
      self.vrfName = vrfName
      self.intfName = intfName

   def setOnMaintenanceUnit( self, args ):
      maintenanceOper = args[ 'OPERATION' ]
      maintenanceHookType = args[ 'HOOK_TYPE' ]
      maintenanceStage = args.get( 'STAGE' )
      unitName = args[ 'UNIT_NAME' ]
      maintStage = maintenanceHookType
      if maintenanceStage:
         maintStage = maintStage.lower() + '_' + maintenanceStage.lower()
      self.setOnMaintenance( maintenanceOper, unitName, maintStage )
         
   def setOnMaintenanceBgp( self, args ):
      vrfName = args.get( 'VRF', 'default' )
      maintenanceOper = args[ 'OPERATION' ]
      maintenanceHookType = args[ 'HOOK_TYPE' ]
      maintenanceStage = args.get( 'STAGE' )
      peer = args[ 'PEER' ]
      unitName = dynamicUnitName( str( peer ), vrfName )
      maintStage = maintenanceHookType
      if maintenanceStage:
         maintStage = maintStage.lower() + '_' + maintenanceStage.lower()
      self.setOnMaintenance( maintenanceOper, unitName, maintStage, peer=str(peer),
                             vrfName=vrfName)

   def setOnMaintenanceInterface( self, args ):
      maintenanceOper = args[ 'OPERATION' ]
      maintenanceHookType = args[ 'HOOK_TYPE' ]
      maintenanceStage = args.get( 'STAGE' )
      intfName = args[ 'INTF' ]
      unitName = dynamicUnitName( intfName.name )
      maintStage = maintenanceHookType
      if maintenanceStage:
         maintStage = maintStage.lower() + '_' + maintenanceStage.lower()
      self.setOnMaintenance( maintenanceOper, unitName, maintStage,
                             intfName=intfName.name )

   def setAction( self, cmd, actionKind='bash' ):
      # Should set actionKind here too
      self.actionKind = actionKindCliToTacc[ actionKind ]
      self.command = cmd
      self.deleteActionMultilineFile()

   def setDefaultAction( self ):
      self.actionKind = config.defaultActionKind
      self.command = config.defaultCommand
      self.deleteActionMultilineFile()

   def setLogAction( self ):
      self.actionKind = actionKindCliToTacc[ 'log' ]
      self.command = ""
      self.deleteActionMultilineFile()

   def createActionMultilineFile( self, script ):
      if not os.path.exists( config.multilineActionScriptDir ):
         Tac.run( [ "mkdir", config.multilineActionScriptDir ], asRoot=True )
         Tac.run( [ "chmod", "777", config.multilineActionScriptDir ], asRoot=True )
      filePath = config.multilineActionScriptDir + \
                 config.multilineActionScriptPrefix + \
                 self.handlerName
      with open( filePath, "w" ) as fd:
         fd.write( script )
         Tac.run( [ "chmod", "777", filePath ], asRoot=True )

   def deleteActionMultilineFile( self ):
      filePath = config.multilineActionScriptDir + \
                 config.multilineActionScriptPrefix + \
                 self.handlerName
      if os.path.exists( filePath ):
         os.remove( filePath )

   def setActionMultiline( self, actionKind='bash' ):
      # Should set actionKind here too
      self.actionKind = actionKindCliToTacc[ actionKind ]
      actionMultiline = BasicCliUtil.getMultiLineInput(
         self, cmd="action", prompt="Enter Multi-line Action" )
      # There are two cases when parsing the config: manually entering the command
      # or reading the startup config. The latter comes with indentation of 3 spaces
      # for every sub config mode. 6 spaces  in our case.
      # We need to preserve the original indentation of the script since it could be
      # important like in a python script
      self.command = removeStartupConfigIndention( actionMultiline, 6 )
      self.createActionMultilineFile( self.command )

   def setHealthAction( self, args ):
      # Health "actions" (a bit of a misnomer internally) are orthogonal to actions
      # that run bash commands
      metric = args[ 'METRIC' ]
      metricNames = healthStatus.metric.keys()
      self.metricName = metric
      if metric not in metricNames:
         self.addWarning( "No such metric '%s' has been registered." % metric )

   def noOrDefaultHealthAction( self, args ):
      self.metricName = config.defaultMetricName

   def setDelay( self, args ):
      self.delay = args[ 'DELAY' ]

   def setRepeatInterval( self, args ):
      self.repeatInterval = args[ 'INTERVAL' ]

   def cmdAsynchronous( self, args ):
      self.asynchronous = True

   def cmdNoAsynchronous( self, args ):
      self.asynchronous = False

   def setTimeout( self, args ):
      self.timeout = args[ 'TIMEOUT' ]

   def setThreshold( self, args ):
      self.threshold = args[ 'THRESHOLD' ]
      self.thresholdCount = args[ 'COUNT' ]

   def isNewEventHandler( self, event ):
      if( self.triggerType != event.triggerType or
          self.operstatus != event.operstatus or
          self.ip != event.ip or self.ip6 != event.ip6 or
          self.intfName != event.intfName or
          self.actionKind != event.actionKind or
          self.command != event.command or
          self.maintenanceUnitName != event.maintenanceUnitName or
          self.maintenanceStage != event.maintenanceStage or
          self.maintenanceOper != event.maintenanceOper or
          self.asynchronous != event.asynchronous or
          self.delay != event.delay or
          self.metricName != event.metricName or
          self.hasSubhandlers != event.hasSubhandlers ):
         return True
      return False

   def commitEventHandler( self ):
      # save or update eventConfig
      isNew = True
      builtinHandler = False
      if self.triggerType != 'onCounters':
         self.hasSubhandlers = False
      if self.handlerName in config.event:
         eventConfig = config.event.get( self.handlerName )

         # A built-in handler need to stay built-in after creating a new one
         if eventConfig.builtinHandler:
            builtinHandler = True

         if self.isNewEventHandler( eventConfig ) or builtinHandler:
            del config.event[ self.handlerName ]
         else:
            isNew = False
            self.copyEventHandler( eventConfig, self )

      if isNew:
         config.event.newMember( 
                                 self.handlerName,
                                 self.triggerType,
                                 self.operstatus,
                                 self.ip,
                                 self.ip6,
                                 self.intfName,
                                 self.actionKind,
                                 self.command,
                                 self.delay,
                                 self.repeatInterval,
                                 self.asynchronous,
                                 self.timeout,
                                 self.thresholdCount,
                                 self.threshold,
                                 self.pollInterval,
                                 self.countersCondition,
                                 self.logRegex,
                                 self.maintenanceUnitName,
                                 self.maintenanceOper,
                                 self.maintenanceStage,
                                 self.maintenanceBgpPeer,
                                 self.vrfName,
                                 self.metricName )
         if builtinHandler:
            config.event[ self.handlerName ].builtinHandler = True
         config.event[ self.handlerName ].hasSubhandlers = self.hasSubhandlers

   def onExit( self ):
      t0( 'starting commit')
      self.commitEventHandler()
      t0( 'finished commiting, trying to exit')
      BasicCli.ConfigModeBase.onExit( self )

   def setVmTracerVm( self, args ):
      self.triggerType = 'vmTracerVm'
      self.operstatus = False
      self.ip = False
      self.ip6 = False

   #-------------------------------------------------------------------------------
   # "[no|default] trigger on-counters command, in "event-handler" mode.
   #-------------------------------------------------------------------------------
   def gotoTriggerOnCountersMode( self, args ):
      self.triggerType = 'onCounters'
      childMode = self.childMode( EventTriggerOnCountersMode )
      self.session_.gotoChildMode( childMode )
   
   def deleteTriggerOnCountersMode( self ):
      self.triggerType = config.defaultTriggerType
      self.pollInterval = config.defaultPollInterval
      self.countersCondition = config.defaultCountersCondition
      self.hasSubhandlers = config.defaultHasSubhandlers

   #-------------------------------------------------------------------------------
   # "[no|default] trigger on-logging command, in "event-handler" mode.
   #-------------------------------------------------------------------------------
   def gotoTriggerOnLoggingMode( self, args ):
      self.triggerType = 'onLogging'
      childMode = self.childMode( EventTriggerOnLoggingMode )
      self.session_.gotoChildMode( childMode )
   
   def deleteTriggerOnLoggingMode( self ):
      self.triggerType = config.defaultTriggerType
      self.pollInterval = config.defaultLogPoll
      self.logRegex = config.defaultLogRegex

def removeStartupConfigIndention( multiLineCmd, indentLen ):
   # There are two cases when parsing the config: manually entering the command
   # or reading the startup config. The latter comes with indentation of 3 spaces
   # for every sub config mode. 6 spaces  in our case.
   # We need to preserve the original indentation of the script since it could be
   # important like in a python script
   indent = ' ' * indentLen
   cmd = ''
   for line in multiLineCmd.split( '\n' ):
      if not line.strip():
         cmd += '\n'
      elif line[ 0:indentLen ] == indent:
         cmd += line[ indentLen: ] + '\n'
      else:
         return multiLineCmd
   return cmd[ : -1 ]

class EventTriggerOnCountersMode( TriggerOnCountersMode,
                                  BasicCli.ConfigModeBase ):
   name = 'Event handler on-counters configuration'
   modeParseTree = CliParser.ModeParseTree()

   def __init__( self, parent, session ):
      TriggerOnCountersMode.__init__( self, parent.handlerName )
      BasicCli.ConfigModeBase.__init__( self, parent, session )
      self.sources = {}
      Counters.initSources( self.sources, status.counterPlugin )

   def setPollingInterval ( self, args ):
      self.parent_.pollInterval = args[ 'INTERVAL' ]

   def defaultPollingInterval ( self, args ):
      self.parent_.pollInterval = config.defaultPollInterval

   def setCondition ( self, args ):
      condition = args[ 'CONDITION' ]
      parsedExp = Counters.ParsedExpression()
      errorMessage = Counters.verifyExpression( condition, parsedExp, self.sources )
      if errorMessage:
         self.addError( errorMessage )
      else:
         self.parent_.countersCondition = condition
         # If the new condition is not wildcarded, turn off the hasSubhandlers
         # property
         if not parsedExp.hasWildcard:
            self.parent_.hasSubhandlers = False

   def defaultCondition ( self, args ):
      self.parent_.countersCondition = config.defaultCountersCondition
      # Default condition is not wildcarded, so resetting this property
      self.parent_.hasSubhandlers = config.defaultHasSubhandlers

   def setPerSourceGranularity( self, args ):
      parsedExp = Counters.ParsedExpression()
      error = Counters.verifyExpression( self.parent_.countersCondition,
                                         parsedExp, self.sources )
      if error or not parsedExp.hasWildcard:
         self.addError( "Condition must have wildcard for per-source granularity" )
      else:
         self.parent_.hasSubhandlers = True

   def defaultGranularity( self, args ):
      self.parent_.hasSubhandlers = config.defaultHasSubhandlers

class EventTriggerOnLoggingMode( TriggerOnLoggingMode,
                                  BasicCli.ConfigModeBase ):
   name = 'Event handler on-logging configuration'
   modeParseTree = CliParser.ModeParseTree()

   def __init__( self, parent, session ):
      TriggerOnLoggingMode.__init__( self, parent.handlerName )
      BasicCli.ConfigModeBase.__init__( self, parent, session )
      self.parent_.pollInterval = config.defaultLogPoll

   def setMaxPollingInterval ( self, args ):
      self.parent_.pollInterval = args[ 'INTERVAL' ]

   def defaultMaxPollingInterval ( self, args ):
      self.parent_.pollInterval = config.defaultLogPoll

   def setLogRegex ( self, args ):
      regex = args[ 'REGEX' ]
      try:
         re.compile( regex )
         self.parent_.logRegex = regex
      except re.error as e:
         self.addError( "Invalid regular expression: %s" % e )

   def defaultLogRegex ( self, args ):
      self.parent_.logRegex = config.defaultLogRegex

#---------------------------------------------------------------------
# transitions to eventHandler config mode
#---------------------------------------------------------------------
def gotoEventHandlerConfigMode( mode, args ):
   eventHandlerName = args[ 'EVENT_HANDLER' ]
   lastMode = mode.session_.modeOfLastPrompt()
   if isinstance( lastMode, EventHandlerConfigMode ) and \
          lastMode.handlerName == eventHandlerName:
      # We are trying to re-enter the current mode. Commit pending Event-handler.
      lastMode.commitEventHandler()

   childMode = mode.childMode( EventHandlerConfigMode, handlerName=eventHandlerName )
   mode.session_.gotoChildMode( childMode )

   # A built-in handler can be re-enabled just by entering the handler config mode
   if eventHandlerName in config.event:
      event = config.event[ eventHandlerName ]
      if event.builtinHandler:
         event.builtinHandlerDisabled = False


#---------------------------------------------------------------------
# Global config mode command to enter event-handler config mode
#---------------------------------------------------------------------
eventHandlerNameMatcher = CliMatcher.DynamicNameMatcher(
   lambda mode: ( eventConfig for eventConfig in config.event ),
   'Event-handler name' )

#---------------------------------------------------------------------
# Global config mode command to show event-handler status
#---------------------------------------------------------------------
def doShowEventStatus( mode, args ):
   hName = args.get( 'EVENT_HANDLER' )
   handlers = EventHandlers()
   def populateSubhandlers( handler, hStat ):
      for source in hStat.subhandlers:
         subhName = getSubhandlerName( hName, source )
         subhStat = status.event.get( subhName )
         if subhStat:
            subhandler = EventSubhandler()
            subhandler.fromTacc( subhStat )
            handler.eventSubhandlers[ source ] = subhandler

   if hName:
      hConf = config.event.get( hName )
      hStat = status.event.get( hName )
      if hConf is None or hStat is None:
         mode.addError( "Invalid event-handler name entered" )
      else:
         handler = EventHandler()
         handler.fromTacc( hConf, hStat, healthStatus )
         if hConf.hasSubhandlers:
            populateSubhandlers( handler, hStat )
         handlers.eventHandlers[ hName ] = handler
   else:
      for hName in status.event:
         hConf = config.event.get( hName )
         hStat = status.event.get( hName )
         if hConf is not None and hStat is not None:
            handler = EventHandler()
            handler.fromTacc( hConf, hStat, healthStatus )
            if hConf.hasSubhandlers:
               populateSubhandlers( handler, hStat )
            handlers.eventHandlers[ hName ] = handler
   return handlers

def doShowCounters( mode, args ):
   sourcesOnly = 'sources' in args
   counterSources = {}
   Counters.initSources( counterSources, status.counterPlugin )
   sources = counterSources.keys()
   ctrs = EventHandlerTrigger()
   if sourcesOnly:
      ctrs.eventHandlerTriggerCounters = None
      for source in sources:
         ctrs.eventHandlerTriggerCounterSources.append( source )
   else:
      ctrs.eventHandlerTriggerCounterSources = None
      for source in sources:
         counters = counterSources.get( source )
         counterList = EventHandlerTriggerCounters()
         counterList.counters =  counters
         ctrs.eventHandlerTriggerCounters[ source ] = counterList
   return ctrs

#---------------------------------------------------------------------
# Event removal
#---------------------------------------------------------------------
def destroyEventHandler( mode, args ):
   eventHandlerName = args.get( 'EVENT_HANDLER' )
   if not eventHandlerName in config.event:
      return
   event = config.event[ eventHandlerName ]
   if event.builtinHandler:
      event.builtinHandlerDisabled = True
   else:
      del config.event[ eventHandlerName ]

#-------------------------------------------------------------------------------
# action commands
#-------------------------------------------------------------------------------

actionMatcher = CliMatcher.KeywordMatcher( 'action',
      helpdesc = 'Define event-handler action' )

class actionLogCmd( CliCommand.CliCommandClass ):
   syntax = '''action log'''
   noOrDefaultSyntax = syntax
   data = { 'action': actionMatcher,
            'log': 'Log a message when the event is triggered' }
   @staticmethod
   def handler( mode, args ):
      mode.setLogAction()

   @staticmethod
   def noOrDefaultHandler( mode, args ):
      mode.setDefaultAction()

EventHandlerConfigMode.addCommandClass( actionLogCmd )

def getEventForMetric( metricName ):
   handlerList = []
   for ( handler, handlerInfo ) in config.event.items():
      if handlerInfo.metricName == metricName and \
            not handlerInfo.builtinHandlerDisabled:
         handlerList.append( handler )
   return handlerList
HealthCli.eventHandlerDataFn = getEventForMetric

def Plugin( entityManager ):
   global config, status, allVrfConfig, healthStatus
   config = ConfigMount.mount( entityManager, "sys/event/config",
                               "Event::Config", "w" )
   status = LazyMount.mount( entityManager, "sys/event/status",
                             "Event::Status", "r" )
   allVrfConfig = LazyMount.mount( entityManager, 'ip/vrf/config',
                                   'Ip::AllVrfConfig', 'r' )
   healthStatus = LazyMount.mount( entityManager, "health/status",
                                   "Health::Status", "r" )
