# Copyright (c) 2005-2011 Arista Networks, Inc.  All rights reserved.
# Arista Networks, Inc. Confidential and Proprietary.

#-------------------------------------------------------------------------------
# This module implements the Ethernet interface type, and associated commands.
# In particular, it provides:
# -  the EthIntf class
# -  the "mac-address <mac_addr>" command
# -  the "no mac-address [<mac_addr>]" command.
# -  the "[no|default] mac timestamp <mode>" command
# -  the "[no] logging event congestion-drops ( interface config level )
# -  the "[no] logging event congestion-drops interval ( global config level )
#-------------------------------------------------------------------------------
'''Commands supported on an Ethernet-like interface'''

from __future__ import absolute_import, print_function
from collections import namedtuple
import math
import os
import re
import time

import Ark
import Arnet
import BasicCli
import BasicCliUtil
import CliCommand
import CliExtensions
import CliMatcher
from CliMode.EthIntf import InterfaceDefaultsEthernetMode
import CliParser
import CliPlugin.EthIntfModel as EthIntfModel
from CliPlugin.EthIntfModel import EthInterfaceCounters, EthInterfaceStatistics
import CliPlugin.IntfCli as IntfCli
from CliPlugin.IntfCli import LoggingEventGlobalCmd, LoggingEventIntfCmd
from CliPlugin.IntfModel import ErrorCounters, HalfDuplexErrorCounters, \
      InterfaceCountersRate
import CliPlugin.IntfRangeCli as IntfRangeCli
import CliPlugin.FruCli as FruCli
import CliPlugin.MacAddr as MacAddr
import CliPlugin.ModuleIntfCli as ModuleIntfCli
import CliPlugin.PhysicalIntfRule as PhysicalIntfRule
import CliPlugin.SpeedGroupCli as SpeedGroupCli
from CliPlugin.VirtualIntfRule import IntfMatcher
import CliPlugin.XcvrEthIntfDir as XcvrEthIntfDir
import CliPlugin.XcvrCapabilitiesModel as XcvrCapabilitiesModel
import CliToken.Mac
import Cell
import ConfigMount
import Ethernet
import EthIntfLib
import Intf.IntfRange as IntfRange
from IntfRangePlugin.EthIntf import EthPhyAutoIntfType, MgmtPhyAutoIntfType, \
    UnconnectedEthPhyAutoIntfType
from IntfRangePlugin.EthIntf import MgmtAutoIntfType
import LazyMount
import PhyStatusLib
import QuickTrace
import SharedMem
import ShowCommand
import Tracing
import Tac
from TypeFuture import TacLazyType

__defaultTraceHandle__ = Tracing.Handle( "EthIntfCli" )
t0 = __defaultTraceHandle__.trace0
qv = QuickTrace.Var
qt8 = QuickTrace.trace8

ZeroEthIntfCounter = Tac.Type( 'Interface::ZeroEthIntfCounter' )

ethPhyIntfConfigSliceDir = None
ethIntfStatusDir = None
ethIntfGenerationSliceDir = None
ethIntfModeSliceDir = None
entmib = None
intfHwStatus = None
ethPhyIntfStatusDir = None
ethPhyIntfDefaultConfigDir = None
ethIntfSliceDefaultConfigDir = None
ethPhyIntfConfigReqDir = None
ethPhyDefaultCapsSliceDir = None
ethIntfCounterWriterStatusDir = None
checkpointCounterDir = None
ethIntfCounterReaderSm = None
ethPhyIntfCounterDir_ = None
intfXcvrStatusDir = None
entityMib = None
timestampingStatus = None

systemName = None
entityManager = None
bridgingHwCapabilities = None

# l2MtuValidIntfType tracks all interface types
# that support L2 mtu.
l2MtuValidIntfType = []

loopbackMode = Tac.Type( 'Interface::LoopbackMode' )
LOOPBACK_MAC = 1
LOOPBACK_PHY = 2
LOOPBACK_PHYREMOTE = 3

# Annotations for interfaces in show vlan configured port, when bridging
# config is overridden.
# NOTE: If you change these chars, you must change them in XxxCliTests.py
vlanDataLinkAnnotations = ( '*', '^', '+', '$' )
# Annotations used for explanation of quietDataLink
vlanQuietDataLinkAnnotations = ( '#', '%' )
# Annotations used for explanation of unauthorized
vlanUnauthorizedAnnotations = ( '~', '-' )
# Annotations used for explanation of recirculation
vlanRecirculationAnnotations = ( '!' )

# A canPromptForAbortHook accepts three arguments: the mode, and the intfList
# which contains the single interface, or the interfaces in the interface range,
# and the requested linkMode.
# If any of the hooks return true, the request is ignored. As these functions
# can issue prompts for user input we only call them in interactive mode.
# This hook can be called for signle interface mode or for interface range mode.
canPromptForAbortHook = CliExtensions.CliHook()

AutonegMode = Tac.Type( "Interface::AutonegMode" )
EthTypesApi = Tac.Type( "Interface::EthTypesApi" )
EthSpeed = TacLazyType( "Interface::EthSpeed" )
EthLaneCount = TacLazyType( "Interface::EthLaneCount" )

def ethPhyIntfCounterDir():
   return ethPhyIntfCounterDir_

# Method to indicate an interface type supports L2 mtu.
# intfType is a class that defines a particular interface class.
def registerL2MtuValidIntfType( intfType ):
   l2MtuValidIntfType.append( intfType )

def isVEos():
   return entityMib.root and entityMib.root.modelName == 'vEOS'

def isModular():
   # Breadth tests related to mgmtactiveintf setup entmib to a give notion of
   # modular system. So we use modularSystemGuard for modular system check,
   # although celltype alone will be sufficient on a real dut.
   return ( ( not FruCli.modularSystemGuard( None, None ) ) or
            ( Cell.cellType() == "supervisor" ) )

#-------------------------------------------------------------------------------
# A subclass of the base IntfCli.Intf class for Ethernet / LAG interfaces.
# Ethernet and Lag further subclass this class.
#-------------------------------------------------------------------------------
class EthIntf( IntfCli.Intf ):
   #----------------------------------------------------------------------------
   # Creates a new EthIntf instance of the specified name.
   #----------------------------------------------------------------------------
   def __init__( self, name, mode ):
      IntfCli.Intf.__init__( self, name, mode )
      t0( "EthIntf::__init__", name )
      self.ethIntfConfigDir = None
      self.ethIntfStatuses = ethIntfStatusDir.intfStatus
      self.intfXcvrStatusDir = intfXcvrStatusDir
      self.ethIntfConfig = None

   #----------------------------------------------------------------------------
   # Determines whether the Interface::EthIntfStatus object for this interface
   # exists.
   #----------------------------------------------------------------------------
   def lookupPhysical( self ):
      ethIntfStatus = self.status()
      return ethIntfStatus is not None

   #----------------------------------------------------------------------------
   # Returns the EthIntfConfig object for this interface.
   # EthIntfConfig is cached to prevent an update from the EntityLog thread
   # in the middle of processing a "show interfaces" command from causing an
   # inconsistent snapshot to be displayed.
   # The caching is done only if not in a config session, since the above
   # scenario cannot happen when in a config session.
   #----------------------------------------------------------------------------
   def config( self ):
      config = self.ethIntfConfigDir.get( self.name )
      if not self.mode_.session_.inConfigSession():
         self.ethIntfConfig = config
      return config

   #----------------------------------------------------------------------------
   # Returns the EthIntfStatus object for this interface.
   # This method is Tac.memoized to prevent an update from the EntityLog thread
   # in the middle of processing a "show interfaces" command from causing an
   # inconsistent snapshot to be displayed.
   #----------------------------------------------------------------------------
   @Tac.memoize
   def status( self ):
      return self.ethIntfStatuses.get( self.name )

   #----------------------------------------------------------------------------
   # Return the EthIntfXcvrStatus object for this interface
   #----------------------------------------------------------------------------
   @Tac.memoize
   def intfXcvrStatus( self ):
      return self.intfXcvrStatusDir.get( self.name )

   #----------------------------------------------------------------------------
   # Returns the EthIntfCounterDir object for this interface.
   #----------------------------------------------------------------------------
   def getIntfCounterDir( self ):
      assert self.status()
      parent = self.status().parent
      if not parent:
         return None
      return ethPhyIntfCounterDir_

   #----------------------------------------------------------------------------
   # Returns a tuple for the explanation of a datalink
   # ( Reason, Long Reason, Legend, Vlan annotation )
   #----------------------------------------------------------------------------
   def dataLinkExplanation( self, mode, ethIntfStatus ):
      reason = None
      legend = None
      longReason = None
      for ( index, ext ) in \
         enumerate( IntfCli.dataLinkExplanationHook.extensions() ):
         if ethIntfStatus:
            ( reason, longReason, legend ) = ext( ethIntfStatus.intfId, mode )
            if reason is not None:
               assert index < len( vlanDataLinkAnnotations )
               return ( reason, longReason,
                           legend, vlanDataLinkAnnotations[ index ] )
      return "Not bridged", None, None, None 

   #----------------------------------------------------------------------------
   # Returns a tuple for the explanation of a quietDataLink port
   # ( Reason, Long Reason, Legend, Vlan annotation )
   #----------------------------------------------------------------------------
   def quietDataLinkExplanation( self, mode, ethIntfStatus ):
      reason = None
      legend = None
      longReason = None
      for ( index, ext ) in \
         enumerate( IntfCli.quietDataLinkExplanationHook.extensions() ):
         if ethIntfStatus:
            ( reason, longReason, legend ) = ext( ethIntfStatus.intfId, mode )
            if reason is not None:
               assert index < len( vlanQuietDataLinkAnnotations )
               return ( reason, longReason,
                           legend, vlanQuietDataLinkAnnotations[ index ] )
      return "Not bridged", None, None, None 

   #----------------------------------------------------------------------------
   # Returns a tuple for the explanation of an unauthorized port
   # ( Reason, Long Reason, Legend, Vlan annotation )
   #----------------------------------------------------------------------------
   def unauthorizedExplanation( self, mode, ethIntfStatus ):
      reason = None
      legend = None
      longReason = None
      for ( index, ext ) in \
         enumerate( IntfCli.unauthorizedExplanationHook.extensions() ):
         if ethIntfStatus:
            ( reason, longReason, legend ) = ext( ethIntfStatus.intfId, mode )
            if reason is not None:
               assert index < len( vlanUnauthorizedAnnotations )
               return ( reason, longReason,
                           legend, vlanUnauthorizedAnnotations[ index ] )
      return "Not bridged", None, None, None 

   #----------------------------------------------------------------------------
   # Returns a tuple for the explanation of an unauthorized port
   # ( Reason, Long Reason, Legend, Vlan annotation )
   #----------------------------------------------------------------------------
   def recirculationExplanation( self, mode, ethIntfStatus ):
      reason = None
      legend = None
      longReason = None
      for ( index, ext ) in \
         enumerate( IntfCli.recirculationExplanationHook.extensions() ):
         if ethIntfStatus:
            ( reason, longReason, legend ) = ext( ethIntfStatus.intfId, mode )
            if reason is not None:
               assert index < len( vlanRecirculationAnnotations )
               return ( reason, longReason,
                           legend, vlanRecirculationAnnotations[ index ] )
      return "Not bridged", None, None, None 

   #----------------------------------------------------------------------------
   # Prints the interface membership (for e.g. if part of
   # monitor destination or a Port-Channel)
   #----------------------------------------------------------------------------
   def getIntfMembership( self , mode):
      if( ( self.status().forwardingModel != "intfForwardingModelDataLink" ) and
          ( self.status().forwardingModel != "intfForwardingModelRecirculation" ) ):
         return None
      elif self.status().forwardingModel == "intfForwardingModelDataLink":
         ( _, longExpl, _, _ ) = \
              self.dataLinkExplanation( mode, self.status() )
         return longExpl
      else:
         ( _, longExpl, _, _ ) = \
              self.recirculationExplanation( mode, self.status() )
         return longExpl

   #----------------------------------------------------------------------------
   # Outputs information about this interface in an interface type-specific
   # manner.
   #----------------------------------------------------------------------------
   def showPhysical( self, mode, intfStatusModel ):
      intfStatusModel.interfaceMembership = self.getIntfMembership( mode )
      self.showLinkTypeSpecific( intfStatusModel )
      self.showLoopbackMode( intfStatusModel )
      self.showL2Mru( intfStatusModel )
      if self.countersSupported():
         intfStatusModel.interfaceStatistics = self.getInterfaceStatistics()
         intfStatusModel.interfaceCounters = self.getInterfaceCounters()

   def bandwidth( self ):
      raise NotImplementedError

   def autonegActive( self ):
      raise NotImplementedError

   def flowcontrolRxPause( self ):
      raise NotImplementedError

   def flowcontrolTxPause( self ):
      raise NotImplementedError

   def showLoopbackMode( self, intfStatusModel ):
      raise NotImplementedError

   def getIntfStatusModel( self ):
      return EthIntfModel.EthInterfaceStatus( name=self.status().intfId )

   def ratePct( self, bps, pps ):
      """ Uses the link bandwidth to determinte the % of theoretical
      bitrate achieved including the 12 byte inter-frame gap and the
      8 byte preable.

      Returns None of bandwidth is not set"""

      maxBw = self.bandwidth() * 1000.0
      per = None
      if maxBw != 0:
         per = ( bps + 160 * pps ) * 100 / maxBw
      return per

   def rateValue( self, attr ):
      # Calculate the rate counter value by exponentially decay the
      # previously saved value and subtract it from the current rate
      # value.
      assert self.counter()
      counter = self.counter()
      ckpt = self.getLatestCounterCheckpoint()
      loadInterval = IntfCli.getActualLoadIntervalValue(
         self.config().loadInterval )

      sysdbValue = getattr( counter, attr )
      if ckpt:
         ckptValue = getattr( ckpt.rates, attr, 0 )
         ckptStatsUpdateTime = getattr( ckpt.rates, "statsUpdateTime", 0 )
      else:
         ckptValue = 0
         ckptStatsUpdateTime = 0

      # If loadInterval = 0, return rate is the rate at the most recent snapshot
      # without being decayed.
      if loadInterval == 0:
         return sysdbValue

      # If an interface is error disabled, its counters would never be updated. For
      # an error disabled interface, If clear command is issued, the last time when
      # counters are updated will be less than the last time counter are cleared.
      # So Addin a guard for this case!
      if counter.rates.statsUpdateTime < ckptStatsUpdateTime:
         return 0.0

      # If the interface is not enabled, then just return the rate counter as 0
      if not ( self.ethIntfConfig and self.ethIntfConfig.enabled ):
         return 0.0

      expFactor = - ( ( counter.rates.statsUpdateTime - ckptStatsUpdateTime ) /
                      float( loadInterval ) )
      decayedCkptValue = ckptValue * math.exp( expFactor )
      return max( 0.0, sysdbValue - decayedCkptValue )

   def getInterfaceStatistics( self ):
      if self.counter() is None:
         return None

      interfaceStatistics = EthInterfaceStatistics()
      interfaceStatistics.updateInterval = \
                     IntfCli.getActualLoadIntervalValue( self.config().loadInterval )
      interfaceStatistics.inBitsRate = self.rateValue( "inBitsRate" )
      interfaceStatistics.outBitsRate = self.rateValue( "outBitsRate" )
      interfaceStatistics.inPktsRate = self.rateValue( "inPktsRate" )
      interfaceStatistics.outPktsRate = self.rateValue( "outPktsRate" )
      return interfaceStatistics

   def updateInterfaceCountersRateModel( self ):
      if self.counter() is None:
         return None
      intfCountersRateModel = InterfaceCountersRate( _name=self.name )
      intfCountersRateModel.description = self.description()
      intfCountersRateModel.interval = int( round(
            IntfCli.getActualLoadIntervalValue( self.config().loadInterval ) ) )
      intfCountersRateModel.inBpsRate = self.rateValue( "inBitsRate" )
      intfCountersRateModel.inPpsRate = self.rateValue( "inPktsRate" )
      intfCountersRateModel.inPktsRate = \
                     self.ratePct( intfCountersRateModel.inBpsRate,
                     intfCountersRateModel.inPpsRate )
      intfCountersRateModel.outBpsRate = self.rateValue( "outBitsRate" )
      intfCountersRateModel.outPpsRate = self.rateValue( "outPktsRate" )
      intfCountersRateModel.outPktsRate = \
                     self.ratePct( intfCountersRateModel.outBpsRate,
                     intfCountersRateModel.outPpsRate )
      # lastUpdateTimestamp should not be a "normalized" delta -- it should be the
      # last update UTC value
      intfCountersRateModel.lastUpdateTimestamp = (
            Ark.switchTimeToUtc( self.counter().rates.statsUpdateTime ) )

      return intfCountersRateModel

   def getLastUpdateTimeStamp( self ):
      ckpt = self.getLatestCounterCheckpoint()
      if ckpt:
         statsUpdateTime = getattr( ckpt.rates, 'statsUpdateTime', None )
      else:
         statsUpdateTime = None
      if statsUpdateTime is not None:
         return statsUpdateTime + Tac.utcNow() - Tac.now()
      else:
         return None

   def getInterfaceCounters( self, notFoundString=None, checkpoint=None ):
      counter = self.counter()
      if counter is None:
         return None
      if checkpoint is None:
         ckpt = self.getLatestCounterCheckpoint()
      else:
         ckpt = checkpoint
      def stat( attr ):
         sysdbValue = getattr( counter.statistics, attr, None )
         if sysdbValue is None:
            ethStats = getattr( counter, 'ethStatistics', None )
            if ethStats is not None:
               sysdbValue = getattr( ethStats, attr, None )
         if sysdbValue is None:
            return notFoundString
         checkpointValue = 0
         if ckpt:
            checkpointValue = getattr( ckpt.statistics, attr, None )
            if checkpointValue is None:
               checkpointValue = getattr( ckpt.ethStatistics, attr, 0 )
         return sysdbValue - checkpointValue

      interfaceCounters = EthInterfaceCounters()
      interfaceCounters.lastClear = self.getLastUpdateTimeStamp()

      # Handle interface types with and without the
      # physical layer counters.  The counters that
      # aren"t present are returned as "None", so be
      # careful doing math with them (like collisions)!
      interfaceCounters._name = self.name
      interfaceCounters.inUcastPkts = stat( "inUcastPkts" )
      interfaceCounters.inMulticastPkts = stat( "inMulticastPkts" )
      interfaceCounters.inBroadcastPkts = stat( "inBroadcastPkts" )
      interfaceCounters.inOctets = stat( "inOctets" )
      interfaceCounters.inTotalPkts = stat( "inTotalPkts" )
      if interfaceCounters.inTotalPkts == 0:
         interfaceCounters.inTotalPkts = interfaceCounters.inUcastPkts + \
                                         interfaceCounters.inMulticastPkts + \
                                         interfaceCounters.inBroadcastPkts
      interfaceCounters.inDiscards = stat( "inDiscards" )
      interfaceCounters.totalInErrors = stat( "inErrors" )
      runtFrames = stat( "frameTooShorts" )
      if runtFrames is not None:
         inputErrors = EthInterfaceCounters.PhysicalInputErrors()
         inputErrors.runtFrames = runtFrames
         inputErrors.giantFrames = stat( "frameTooLongs" )
         inputErrors.fcsErrors = stat( "fcsErrors" )
         inputErrors.alignmentErrors = stat( "alignmentErrors" )
         inputErrors.symbolErrors = stat( "symbolErrors" )
         # Use the same function as "show interfaces flowcontrol" to eliminate
         # duplicate code that is out of sync.
         inputErrors.rxPause = self.flowcontrolRxPause( )
         interfaceCounters.inputErrorsDetail = inputErrors

      interfaceCounters.outUcastPkts = stat( "outUcastPkts" )
      interfaceCounters.outMulticastPkts = stat( "outMulticastPkts" )
      interfaceCounters.outBroadcastPkts = stat( "outBroadcastPkts" )
      interfaceCounters.outOctets = stat( "outOctets" )
      interfaceCounters.outDiscards = stat( "outDiscards" )

      interfaceCounters.totalOutErrors = stat( "outErrors" )
      interfaceCounters.counterRefreshTime = Tac.utcNow()
      collisions = stat( "singleCollisionFrames" )
      if collisions is not None:
         outputErrors = EthInterfaceCounters.PhysicalOutputErrors()
         outputErrors.collisions = collisions + stat( "multipleCollisionFrames" )
         outputErrors.lateCollisions = stat( "lateCollisions" )
         outputErrors.deferredTransmissions = stat( "deferredTransmissions" )
         # Use the same function as "show interfaces flowcontrol" to eliminate
         # duplicate code that is out of sync.
         outputErrors.txPause = self.flowcontrolTxPause( )
         interfaceCounters.outputErrorsDetail = outputErrors

      # Get the number of link status changes since last
      # 'clear counters [session]'.
      interfaceCounters.linkStatusChanges = self.status().operStatusChange.count - \
                                             getattr( ckpt, "linkStatusChanges", 0 )

      return interfaceCounters

   #----------------------------------------------------------------------------
   # Utility functions used by showPhysical().
   #----------------------------------------------------------------------------
   def addr( self ):
      return Ethernet.convertMacAddrCanonicalToDisplay( self.status().addr )

   def routedAddr( self ):
      return Ethernet.convertMacAddrCanonicalToDisplay( self.status().routedAddr )

   def hardware( self ):
      raise NotImplementedError()

   def addrStr( self ):
      raise NotImplementedError()

   def showLinkTypeSpecific( self, intfStatusModel ):
      raise NotImplementedError()

   def showL2Mru( self, intfStatusModel ):
      if self.status().l2Mru:
         intfStatusModel.l2Mru = self.status().l2Mru

   #----------------------------------------------------------------------------
   # Helper function that returns the string representation of the interface's
   # link status in one of a number of specifiable formats.
   #----------------------------------------------------------------------------
   linkStatusStringsFormat1 = { 'linkUp' : 'up',
                                'linkDown' : 'down',
                                'linkUnknown' : 'unknown' }

   linkStatusStringsFormat2 = { 'linkUp' : 'connected',
                                'linkDown' : 'notconnect',
                                'linkUnknown' : 'unknown' }

   linkStatusStrings = { 'Format1' : linkStatusStringsFormat1,
                         'Format2' : linkStatusStringsFormat2 }

   def _linkStatusStr( self, stringFormat ):
      l = self.linkStatusStrings.get( stringFormat )
      return l[ self.status().linkStatus ]

   #----------------------------------------------------------------------------
   # Utility functions used by "showInterfacesStatus()"
   #----------------------------------------------------------------------------
   def linkStatus( self ):
      if not self.config().enabled:
         # If the adminEnabled state, and enabled state are
         # different, then the interface must be errdisabled.
         # the enabledStateReason contains the status string
         if ( not self.config().adminEnabled ) and self.status().active:
            return 'disabled'
         else:
            return self.config().enabledStateReason
      return self._linkStatusStr( 'Format2' )

   #----------------------------------------------------------------------------
   # Utility function used by switchport CLI modelet
   #----------------------------------------------------------------------------
   def switchportEligible( self ):
      raise NotImplementedError( )

   #----------------------------------------------------------------------------
   # A hack to lazy mount counters
   #----------------------------------------------------------------------------
   def counter( self ):
      return IntfCli.Intf.counter( self )

   #----------------------------------------------------------------------------
   # Sets the L2 Mru value
   #----------------------------------------------------------------------------
   def setL2Mru( self, value ):
      cfg = self.config()
      # MRU range for an interface may change dynamically, thus a value cannot be
      # used now may be configured. In this case, CLI will accept the value, and
      # agent code should handle it.
      if self.lookupPhysical() and value != 0:
         status = self.status()
         if value < status.l2MruMin:
            self.mode_.addWarning( 'Min MRU for %s is %d bytes' %
                                   ( self.name, status.l2MruMin ) )
         elif value > status.l2MruMax:
            self.mode_.addWarning( 'Max MRU for %s is %d bytes' %
                                   ( self.name, status.l2MruMax ) )
      cfg.l2Mru = value

#-------------------------------------------------------------------------------
# A subclass of the EthIntf class for physical Ethernet interfaces.
#-------------------------------------------------------------------------------
class EthPhyIntf( EthIntf ):

   #----------------------------------------------------------------------------
   # Creates a new EthPhyIntf instance of the specified name.
   #----------------------------------------------------------------------------
   def __init__( self, name, mode ):
      t0( "EthPhyIntf::__init__", name )
      EthIntf.__init__( self, name, mode )

   #----------------------------------------------------------------------------
   # Returns the EthPhyDefaultCaps object for this interface
   #----------------------------------------------------------------------------
   def ethPhyDefaultCaps( self ):
      if isModular():
         sliceName = self.sliceName()
      else:
         sliceName = "FixedSystem"
      if sliceName:
         ethPhyDefaultCapsDirs = ethPhyDefaultCapsSliceDir.get( sliceName )
         if ethPhyDefaultCapsDirs:
            for ethPhyDefaultCapsDir in ethPhyDefaultCapsDirs.itervalues():
               ethPhyDefaultCaps = ethPhyDefaultCapsDir.portCaps.get( self.name )
               if ethPhyDefaultCaps:
                  return ethPhyDefaultCaps

      return None

   #----------------------------------------------------------------------------
   # Returns the EthIntfMode object for this interface.
   #----------------------------------------------------------------------------
   def ethIntfMode( self ):
      if isModular():
         sliceName = self.sliceName()
      else:
         sliceName = "FixedSystem"
      if sliceName:
         ethIntfModeDirs = ethIntfModeSliceDir.get( sliceName )
         if ethIntfModeDirs:
            ethIntfGeneration = ethIntfGenerationSliceDir.get( sliceName )
            for ethIntfModeDir in ethIntfModeDirs.itervalues():
               if ethIntfModeDir.generation == ethIntfGeneration.generation:
                  ethIntfMode = ethIntfModeDir.ethIntfMode.get( self.name )
                  if ethIntfMode:
                     return ethIntfMode

      return None

   #----------------------------------------------------------------------------
   # Returns the EthIntfConfig object for this interface.
   # Overrides the parent method
   #----------------------------------------------------------------------------
   def config( self ):
      def getEthIntfConfig():
         if self.sliceName():
            ethIntfConfigDir = ethPhyIntfConfigSliceDir.get( self.sliceName() )
            if ethIntfConfigDir:
               return ethIntfConfigDir.intfConfig.get( self.name )

         return None

      if self.mode_.session_.inConfigSession():
         return getEthIntfConfig()
      else:
         if not self.ethIntfConfig:
            self.ethIntfConfig = getEthIntfConfig()
         return self.ethIntfConfig

   @Tac.memoize
   def sliceName( self ):
      return EthIntfLib.sliceName( self.name )

   #----------------------------------------------------------------------------
   # Handles the 'no interface' command for physical Ethernet interfaces
   # For interfaces that does not exist in the hardware, the command goes
   # through and destroys the interface.
   #----------------------------------------------------------------------------
   def noInterface( self ):
      if not self.lookupPhysical():
         self.destroy()
      else:
         self.mode_.addError( "Removal of physical interfaces is not permitted" )

   #----------------------------------------------------------------------------
   # The rule for matching Ethernet interface names.  When this pattern matches,
   # it returns an instance of the EthPhyIntf class.
   #
   # This rule gets added to the Intf.rule when this class is registered with
   # the Intf class by calling Intf.addPhysicalIntfType, below.
   #----------------------------------------------------------------------------
   # pylint: disable=undefined-variable
   valueFunc_ = lambda mode, intf: EthPhyIntf( intf, mode )
   matcher = IntfMatcher()
   ethOrMgmtMatcher = matcher
   ethMatcher = PhysicalIntfRule.PhysicalIntfMatcher( 'Ethernet',
                                                      value=valueFunc_ )
   mgmtMatcher = PhysicalIntfRule.PhysicalIntfMatcher( 'Management',
                                                       value=valueFunc_ )
   matcher |= ethMatcher
   matcher |= mgmtMatcher

   def vrfSupported( self ):
      # Support VRF on Ethernet and Management interfaces
      return ( self.name.startswith( 'Ethernet' ) or
            self.name.startswith( 'Management' ) )

   def intfCreationCurrentlySupported( self, mode ):
      return True
   #----------------------------------------------------------------------------
   # Creates the Interface::EthPhyIntfConfig object for this interface if it does
   # not already exist.
   #----------------------------------------------------------------------------
   def createPhysical( self, startupConfig=False ):
      t0( "createPhysical", self.name )
      if not self.lookupPhysical( ) and not startupConfig:
         self.mode_.addWarning(
            "Interface does not exist. The configuration will not take effect "
            "until the module is inserted.")

      # It's important to create the slice dir before creating the config request.
      # This way when the Sysdb reactor for the config request fires the slice dir
      # will already exist
      ethPhyIntfConfigDir = ethPhyIntfConfigSliceDir.newEntity(
         "Interface::EthPhyIntfConfigDir",
         self.sliceName() )

      IntfCli.cliCreateIntfConfigRequest( ethPhyIntfConfigDir, self.name )

      ethPhyIntfConfigDir.intfConfig.newMember( self.name )
      t0( "added ", ethPhyIntfConfigDir[ self.name ] )
      #do we need the above? I guess if the EthIntf SysdbPlugin is not there

   #----------------------------------------------------------------------------
   # Destroys the Interface::EthPhyIntfConfig object for this interface if it already
   # exists.
   #----------------------------------------------------------------------------
   def destroyPhysical( self ):
      # This method should be called only after making sure the interface can
      # indeed be deleted. i.e. when there is no intfStatus.
      t0( "destroyPhysical", self.name )
      ethPhyIntfConfigDir = ethPhyIntfConfigSliceDir[ self.sliceName() ]
      assert not self.status()
      assert 'Sysdb' not in IntfCli.intfConfigRequestors( ethPhyIntfConfigDir,
                                                          self.name )
      IntfCli.cliDeleteIntfConfigRequest( ethPhyIntfConfigDir, self.name )
      del ethPhyIntfConfigDir[ self.name ]

   def counter( self ):
      counterDir = self.getIntfCounterDir()
      if counterDir and counterDir.intfCounterDir.get( self.name ):
         return counterDir.intfCounterDir[ self.name ][ 'current' ]
      qt8( "EthPhyIntf.counter:", qv( self.name ), "is None" )
      # Read counter from intfStatus.counter
      counter = IntfCli.Intf.counter( self )
      if not counter:
         qt8( "EthPhyIntf.counter:", qv( self.name ),
              "No counter found in status, returning default counter" )
         counter = ZeroEthIntfCounter( self.name )
      return counter

   #----------------------------------------------------------------------------
   # Returns the EthIntfCounterDir object for this interface.
   #----------------------------------------------------------------------------
   def getIntfCounterDir( self ):
      assert self.status()
      parent = self.status().parent
      if not parent:
         return None
      return ethPhyIntfCounterDir_

   #----------------------------------------------------------------------------
   # Utility functions used by showPhysical().
   #----------------------------------------------------------------------------
   def addrStr( self ):
      if self.status().forwardingModel == 'intfForwardingModelRouted':
         addr = self.routedAddr()
         if self.routedAddr() != self.addr():
            # If an interface has a different address, it is probably an
            # l2 port, so the l2 and bia are just information overload.
            # If the addresses are the same, it is probably a management
            # port, so the bia is relevant.
            return "address is %s" % addr
      else:
         addr = self.addr()
      return "address is %s (bia %s)" % ( addr, self.burnedInAddr() )

   def getIntfStatusModel( self ):
      return EthIntfModel.EthPhyInterfaceStatus( name=self.status().intfId )

   def hardware( self ):
      return "ethernet"

   def burnedInAddr( self ):
      return Ethernet.convertMacAddrCanonicalToDisplay(
         self.status().burnedInAddr )

   def showLinkTypeSpecific( self, intfStatusModel ):
      intfStatusModel.duplex = self.status().duplex
      intfStatusModel.autoNegotiate = \
                                         self.autonegStateStr( longStr=True ).lower()
      intfStatusModel._autoNegotiateActive = self.autonegActive()
      if self.showLanes():
         laneCount = self.getLaneCount()
      else:
         # If interface only supports legacy speeds, do not show lane count.
         laneCount = EthLaneCount.laneCountUnknown
      intfStatusModel.lanes = EthTypesApi.laneCountNumber( laneCount )
      if self.unidirectionalLinkSupported():
         intfStatusModel._uniLinkMode = self.showUnidirectionalLinkMode()
      else:
         intfStatusModel._uniLinkMode = 'n/a'

      # pylint: disable-msg=W0212
      return ( intfStatusModel.duplex, intfStatusModel.autoNegotiate,
               intfStatusModel._autoNegotiateActive )
      # pylint: enable-msg=W0212

   #----------------------------------------------------------------------------
   # Helper function that returns the string representation of the interface's
   # duplex in one of a number of specifiable formats.
   #----------------------------------------------------------------------------
   def autonegActive( self ):
      return self.status().autonegActive

   def operDuplex( self, stringFormat ):
      return EthIntfLib.generateOperDuplex(stringFormat,
                                           self.autonegActive(),
                                           self.status().duplex )

   def adminDuplex( self, stringFormat ):
      if self.autonegActive():
         return EthIntfLib.autonegDuplexStrings[ stringFormat ]
      ( _, _, duplex ) = \
            EthIntfLib.linkModeToSpeedLanesDuplex[ self.config().linkMode ]
      return EthIntfLib._duplexStr( duplex, stringFormat )

   #----------------------------------------------------------------------------
   # Helper function that returns the string representation of the interface's
   # speed in one of a number of specifiable formats.
   #----------------------------------------------------------------------------
   speedStringsFormat1 = { 'speed10Mbps': '10Mb/s',
                           'speed100Mbps': '100Mb/s',
                           'speed1Gbps': '1Gb/s',
                           'speed2p5Gbps': '2.5Gb/s',
                           'speed5Gbps': '5Gb/s',
                           'speed10Gbps': '10Gb/s',
                           'speed25Gbps': '25Gb/s',
                           'speed40Gbps': '40Gb/s',
                           'speed50Gbps': '50Gb/s',
                           'speed100Gbps': '100Gb/s',
                           'speed200Gbps': '200Gb/s',
                           'speed400Gbps': '400Gb/s',
                           'speedUnknown' : 'Unconfigured' }

   speedStringsFormat2 = { 'speed10Mbps': '10M',
                           'speed100Mbps': '100M',
                           'speed1Gbps': '1G',
                           'speed2p5Gbps': '2.5G',
                           'speed5Gbps': '5G',
                           'speed10Gbps': '10G',
                           'speed25Gbps': '25G',
                           'speed40Gbps': '40G',
                           'speed50Gbps': '50G',
                           'speed100Gbps': '100G',
                           'speed200Gbps': '200G',
                           'speed400Gbps': '400G',
                           'speedUnknown' : 'unconfigured' }

   speedStringsFormat3 = { 'speed10Mbps': 'a-10M',
                           'speed100Mbps': 'a-100M',
                           'speed1Gbps': 'a-1G',
                           'speed2p5Gbps': 'a-2.5G',
                           'speed5Gbps': 'a-5G',
                           'speed10Gbps': 'a-10G',
                           'speed25Gbps': 'a-25G',
                           'speed40Gbps': 'a-40G',
                           'speed50Gbps': 'a-50G',
                           'speed100Gbps': 'a-100G',
                           'speedUnknown' : 'unconfigured' }

   speedStrings = { 'Format1' : speedStringsFormat1,
                    'Format2' : speedStringsFormat2,
                    'Format3' : speedStringsFormat3 }

   autonegSpeedStrings = { 'Format1' : 'Auto-speed',
                           'Format2' : 'auto',
                           'Format3' : 'auto' }

   @staticmethod
   def _speedStr( value, stringFormat ):
      s = EthPhyIntf.speedStrings.get( stringFormat )
      return s[ value ]

   def operSpeed( self, stringFormat ):
      speed = self.status().speed
      autonegActive = self.autonegActive()
      if autonegActive and speed == EthSpeed.speedUnknown:
         return self.autonegSpeedStrings[ stringFormat ]

      laneCount = self.getLaneCount()
      ethIntfMode = self.ethIntfMode()
      if ethIntfMode and ethIntfMode.speed != EthSpeed.speedUnknown and \
         ethIntfMode.laneCount != EthLaneCount.laneCountUnknown:
         speed = ethIntfMode.speed
         laneCount = ethIntfMode.laneCount

      return EthPhyIntf._speedLanesStr( speed, laneCount, self.showLanes(),
                                        stringFormat )

   def setOperSpeedInProperties( self ):
      '''
      Method to return interface speed and lane count.
      Returns
      ------
      speed : Interfaces::EthSpeed. If the interface speed is 10G, 
              the speed will be speed10Gbps.
      laneCount : Interfaces:EthLaneCount. If the interface is installed
                  a x-lane transceiver, the laneCount will be laneCountx.
      '''
      speed = self.status().speed
      laneCount = self.getLaneCount()
      ethIntfMode = self.ethIntfMode()
      if( ethIntfMode and ethIntfMode.speed != EthSpeed.speedUnknown and 
          ethIntfMode.laneCount != EthLaneCount.laneCountUnknown ):
         speed = ethIntfMode.speed
         laneCount = ethIntfMode.laneCount
      return speed, laneCount

   def adminSpeed( self, stringFormat ):
      if self.autonegActive():
         return self.autonegSpeedStrings[ stringFormat ]
      speed, lanes, _ = \
            EthIntfLib.linkModeToSpeedLanesDuplex[ self.config().linkMode ]
      return speed, lanes

   #----------------------------------------------------------------------------
   # Helper function that returns the string representation of the interface's
   # speed and lane count.
   #----------------------------------------------------------------------------
   @staticmethod
   def _speedLanesStr( speed, lanes, showLanes, stringFormat ):
      if( speed == EthSpeed.speedUnknown or
          lanes == EthLaneCount.laneCountUnknown or
          not showLanes ):
         return EthPhyIntf._speedStr( speed, stringFormat )
      else:
         return EthIntfLib.speedLanestoSpeedLanesStr[ speed, lanes ]

   #----------------------------------------------------------------------------
   # Helper function that returns the integer value of the interface's
   # bandwidth (in kbits/sec).
   #----------------------------------------------------------------------------
   bandwidthVals = { 'speed10Mbps': 10000,
                     'speed100Mbps': 100000,
                     'speed1Gbps': 1000000,
                     'speed2p5Gbps': 2500000,
                     'speed5Gbps': 5000000,
                     'speed10Gbps': 10000000,
                     'speed25Gbps': 25000000,
                     'speed40Gbps': 40000000,
                     'speed50Gbps': 50000000,
                     'speed100Gbps': 100000000,
                     'speed200Gbps': 200000000,
                     'speed400Gbps': 400000000,
                     'speedUnknown' : 0 }

   def bandwidth( self ):
      return self.bandwidthVals[ self.status().speed ]

   #----------------------------------------------------------------------------
   # Helper function that returns a string containing active speeds in an
   # an EthLinkModeSet
   #----------------------------------------------------------------------------
   def _speedSetStr( self, speedSet, detail=False ):
      setStr = ''
      sep = ''
      if speedSet.mode10MbpsHalf or speedSet.mode10MbpsFull:
         setStr += sep + '10M'
         sep = '/'
      if speedSet.mode100MbpsHalf or speedSet.mode100MbpsFull:
         setStr += sep + '100M'
         sep = '/'
      if speedSet.mode1GbpsHalf or speedSet.mode1GbpsFull:
         setStr += sep + '1G'
         sep = '/'
      if speedSet.mode2p5GbpsFull:
         setStr += sep + '2.5G'
         sep = '/'
      if speedSet.mode5GbpsFull:
         setStr += sep + '5G'
         sep = '/'
      if speedSet.mode10GbpsFull:
         setStr += sep + '10G'
         sep = '/'
      if speedSet.mode25GbpsFull:
         setStr += sep + '25G'
         sep = '/'
      if speedSet.mode40GbpsFull:
         setStr += sep + '40G'
         sep = '/'
      if speedSet.mode50GbpsFull:
         setStr += sep + '50G'
         sep = '/'
      if speedSet.mode50GbpsFull1Lane:
         setStr += sep + '50G'
         sep = '/'
      if speedSet.mode100GbpsFull:
         setStr += sep + '100G'
         sep = '/'
      if speedSet.mode100GbpsFull2Lane:
         setStr += sep + '100G'
         sep = '/'
      if speedSet.mode200GbpsFull4Lane:
         setStr += sep + '200G'
         sep = '/'
      if speedSet.mode400GbpsFull8Lane:
         setStr += sep + '400G'
         sep = '/'
      if setStr == '':
         setStr = 'None' if detail else '-'
      return setStr

   def localSpeedSet( self, detail=False ):
      advert = self.status().autonegStatus.advertisement
      return self._speedSetStr( advert.linkModes, detail )

   def partnerSpeedSet( self, detail=False ):
      advert = self.status().autonegStatus.partnerAdvertisement
      return self._speedSetStr( advert.linkModes, detail )

   #----------------------------------------------------------------------------
   # Helper function that returns a string containing active duplexes in an
   # an EthLinkModeSet
   #----------------------------------------------------------------------------
   def _duplexSetStr( self, speedSet, detail=False ):
      setStr = ''
      sep = ''
      setStr = ''
      sep = ''
      if ( speedSet.mode10MbpsHalf or speedSet.mode100MbpsHalf or
           speedSet.mode1GbpsHalf ):
         setStr += sep + 'half'
         sep = '/'
      if ( speedSet.mode10MbpsFull or speedSet.mode100MbpsFull or
           speedSet.mode1GbpsFull or speedSet.mode10GbpsFull or
           speedSet.mode25GbpsFull or speedSet.mode40GbpsFull or
           speedSet.mode50GbpsFull or speedSet.mode50GbpsFull1Lane or
           speedSet.mode100GbpsFull or speedSet.mode100GbpsFull2Lane or
           speedSet.mode200GbpsFull4Lane or speedSet.mode400GbpsFull8Lane ):
         setStr += sep + 'full'
      if setStr == '':
         setStr = 'None' if detail else '-'
      return setStr

   def localDuplexSet( self, detail=False ):
      advert = self.status().autonegStatus.advertisement
      return self._duplexSetStr( advert.linkModes, detail )

   def partnerDuplexSet( self, detail=False ):
      advert = self.status().autonegStatus.partnerAdvertisement
      return self._duplexSetStr( advert.linkModes, detail )

   def getLaneCount( self ):
      if self.autonegActive():
         return self.status().autonegStatus.laneCount
      ethIntfMode = self.ethIntfMode()
      if ethIntfMode:
         return ethIntfMode.laneCount
      else:
         return EthLaneCount.laneCountUnknown

   #----------------------------------------------------------------------------
   # Helper function that returns a string describing advertised Pause
   # capabilities in an AutonegCapabilities object
   #----------------------------------------------------------------------------
   _anegPauseStrMap = { 'pauseUnknown' : 'None',
                        'pauseDisabled' : 'Disabled',
                        'pauseTxOnly' : 'TxOnly',
                        'pauseSymmetric' : 'Symmetric',
                        'pauseRxOrSymmetric' : 'RxOrSymmetric'
                        }

   def localAnegPause( self ):
      advert = self.status().autonegStatus.advertisement
      return self._anegPauseStrMap[ advert.pause ]

   def partnerAnegPause( self ):
      advert = self.status().autonegStatus.partnerAdvertisement
      return self._anegPauseStrMap[ advert.pause ]

   #----------------------------------------------------------------------------
   # Helper function that returns the string representation of the AutonegState
   # in the autonegStatus member of an interface
   #----------------------------------------------------------------------------
   _anegStateShortStrMap = { 'anegStateUnknown' : 'unknown',
                             'anegStateOff' : 'off',
                             'anegStateNegotiating' : 'nego',
                             'anegStateSuccessful' : 'success',
                             'anegStateFailed' : 'failed'
                             }

   _anegStateLongStrMap = { 'anegStateUnknown' : 'Unknown',
                            'anegStateOff' : 'Off',
                            'anegStateNegotiating' : 'Negotiating',
                            'anegStateSuccessful' : 'Success',
                            'anegStateFailed' : 'Failed'
                            }
   anegFecEncodingStrMap = { 'fecEncodingUnknown' : 'N/A',
                             'fecEncodingDisabled' : 'Off',
                             'fecEncodingReedSolomon544' : 'RS',
                             'fecEncodingReedSolomon' : 'RS',
                             'fecEncodingFireCode' : 'FC'
                             }

   def _autonegState( self ):
      anegStatus = self.status().autonegStatus
      state = ( anegStatus.state if self.config().enabled else 'anegStateOff' )
      return state

   def autonegStateStr( self, longStr=False ):
      if longStr:
         return self._anegStateLongStrMap[ self._autonegState() ]
      else:
         return self._anegStateShortStrMap[ self._autonegState() ]

   #----------------------------------------------------------------------------
   # Helper function to check if autonegotiation mode is in 'Consortium' mode.
   #----------------------------------------------------------------------------
   _consortiumTechTypes = [ 'techAbilityEthCon25gCr',
                            'techAbilityEthCon50gCr2',
                            'techAbilityEthCon400gCr8' ]

   def autonegConsortiumMode( self ):
      autonegStatus = self.status().autonegStatus
      techType = autonegStatus.technologyType
      mode = autonegStatus.mode
      return mode == 'anegModeClause73' and techType in self._consortiumTechTypes

   #----------------------------------------------------------------------------
   # Helper function to return a mode string for an autonegotiation mode
   #----------------------------------------------------------------------------
   _anegModeStrMap = { 'anegModeUnknown' : 'None',
                       'anegModeClause28' : 'BASE-T (IEEE Clause 28)',
                       'anegModeClause37' : '1000BASE-X (IEEE Clause 37)',
                       'anegModeClause73' : 'BASE-CR (IEEE Clause 73)',
                       'anegModeSgmii' : 'SGMII' }

   def autonegMode( self ):
      if self.autonegConsortiumMode():
         return 'Ethernet Consortium'
      else:
         mode = self.status().autonegStatus.mode
         return self._anegModeStrMap[ mode ]

   #----------------------------------------------------------------------------
   # Utility functions used by "showInterfacesStatus()"
   #----------------------------------------------------------------------------
   def getIntfDuplexStr( self ):
      return self.status().duplex

   def xcvrTypeStr( self ):
      xcvrPresence = self.intfXcvrStatus().xcvrPresence
      xcvrType = self.intfXcvrStatus().xcvrType
      if xcvrPresence == 'xcvrPresenceUnknown':
         return 'Unknown'
      elif xcvrPresence == 'xcvrNotPresent':
         return 'Not Present'
      elif xcvrPresence == 'xcvrNotApplicable':
         return "N/A"
      else:
         assert xcvrPresence == 'xcvrPresent', \
                "Unhandled xcvrPresence value '%s'" % xcvrPresence
         return xcvrType

   #----------------------------------------------------------------------------
   # Utility functions used by "showInterfacesFlowcontrol()"
   #----------------------------------------------------------------------------
   flowcontrolCfgStringsFormat1 = { 'flowControlConfigOn' : 'on',
                                    'flowControlConfigOff' : 'off',
                                    'flowControlConfigDesired' : 'desired',
                                    'flowControlConfigUnknown' : 'unknown' }

   flowcontrolCfgStrings = { 'Format1' : flowcontrolCfgStringsFormat1 }

   flowcontrolStatusStringsFormat1 = { 'flowControlStatusOn' : 'on',
                                       'flowControlStatusOff' : 'off',
                                       'flowControlStatusDisagree' : 'disagree',
                                       'flowControlStatusUnknown' : 'unknown' }

   flowcontrolStatusStrings = { 'Format1' : flowcontrolStatusStringsFormat1 }

   def flowcontrolStateStr( self, capability, dictionary, key ):
      if capability == 'flowControlNotCapable':
         return 'Unsupp.'
      return dictionary[ key ]

   def flowcontrolTxAdminState( self ):
      return self.flowcontrolStateStr( self.status().txFlowcontrolCapabilities,
                                       self.flowcontrolCfgStringsFormat1,
                                       self.config().txFlowcontrol )

   def flowcontrolTxOperState( self ):
      return self.flowcontrolStateStr( self.status().txFlowcontrolCapabilities,
                                       self.flowcontrolStatusStringsFormat1,
                                       self.status().txFlowcontrol )

   def flowcontrolRxAdminState( self ):
      return self.flowcontrolStateStr( self.status().rxFlowcontrolCapabilities,
                                       self.flowcontrolCfgStringsFormat1,
                                       self.config().rxFlowcontrol )

   def flowcontrolRxOperState( self ):
      return self.flowcontrolStateStr( self.status().rxFlowcontrolCapabilities,
                                       self.flowcontrolStatusStringsFormat1,
                                       self.status().rxFlowcontrol )

   def flowcontrolRxPause( self ):
      counter = self.counter()
      if counter is None:
         qt8( "EthPhyIntf-flowcontrolRxPause:", qv( self.name ), "is None" )
         return 0
      sysdbValue = getattr( counter.ethStatistics, 'inPauseFrames', None )
      if sysdbValue is None:
         return 0
      ckpt = self.getLatestCounterCheckpoint( )
      if ckpt:
         checkpointValue = getattr( ckpt.ethStatistics, 'inPauseFrames', 0 )
      else:
         checkpointValue = 0
      return sysdbValue - checkpointValue

   def flowcontrolTxPause( self ):
      counter = self.counter()
      if counter is None:
         qt8( "EthPhyIntf-flowcontrolTxPause:", qv( self.name ), "is None" )
         return 0
      sysdbValue = getattr( counter.ethStatistics, 'outPauseFrames', None )
      if sysdbValue is None:
         return 0
      ckpt = self.getLatestCounterCheckpoint( )
      if ckpt:
         checkpointValue = getattr( ckpt.ethStatistics, 'outPauseFrames', 0 )
      else:
         checkpointValue = 0
      return sysdbValue - checkpointValue

   #----------------------------------------------------------------------------
   # Utility functions used by "showInterfacesCapabilities()"
   #----------------------------------------------------------------------------
   def _xcvrSlot( self, card, port ):
      # Find the xcvrSlot on the given card for this interface.  We have to
      # match the tag and label rather than trying to just index by the port
      # number, because management and regular ethernet ports all live together
      # in the xcvrSlot collection of a fixed config system.
      m = re.match( r'^([^\d]+)', self.name )
      if not m:
         return None
      tag = m.group( 1 )
      for slot in card.xcvrSlot.values():
         for p in slot.port.values():
            if (p.tag == tag) and (p.label == port):
               return slot
      return None

   def modelStr( self, mode ):
      # Return a model string appropriate for the 'show interface capabilities'
      # command.  This is a bit weird.  It returns the model number of the
      # card or fixed config system where the port sits, concatenated with
      # the transceiver slot type if there is one for the port.  To get this
      # information, we rummage around in the entity mib.
      (mod, port) = self.moduleAndPort()
      if mod is None and port is None:
         return "Unknown"
      if not entmib.root:
         # On vEOS, we are not populating the entity mib at all. This
         # needs further rationalization - BUG9410
         return "Unknown"
      elif entmib.fixedSystem:
         # Fixed system.
         card = entmib.root
      else:
         # Modular system.
         cardSlot = entmib.root.cardSlot.get( mod, None )
         if not cardSlot:
            return "Unknown"
         card = cardSlot.card
      if not card:
         return "Unknown"
      modelStr = card.modelName
      xcvrSlot = self._xcvrSlot( card, port )
      if not xcvrSlot:
         return modelStr
      xcvrSlotType = xcvrSlot.slotType
      if not xcvrSlotType:
         return modelStr
      return modelStr + "-" + xcvrSlotType.upper()

   @staticmethod
   def _showLanes( ethPhyDefaultCaps ):
      if ethPhyDefaultCaps is None:
         return False
      caps = ethPhyDefaultCaps.linkModeCapabilities
      return ( caps.mode50GbpsFull1Lane or
               caps.mode100GbpsFull2Lane or
               caps.mode200GbpsFull4Lane or
               caps.mode200GbpsFull8Lane or
               caps.mode400GbpsFull8Lane )

   def showLanes( self ):
      return EthPhyIntf._showLanes( self.ethPhyDefaultCaps() )

   def speedGroupStatus( self ):
      for linecard in SpeedGroupCli.speedGroupStatusSliceDir.itervalues():
         for speedGroupStatusDir in linecard.itervalues():
            group = speedGroupStatusDir.intfSpeedGroup.get( self.name )
            if group:
               return speedGroupStatusDir.group.get( group )
      return None

   def linkModePriorityList( self ):
      if isModular():
         sliceName = self.sliceName()
      else:
         sliceName = "FixedSystem"
      pList = Tac.newInstance( "Interface::PriorityListFactory" )
      resolveHelper = Tac.newInstance( "Interface::XcvrModeResolveHelper" )
      resolveHelper.sliceDefaultConfig = \
            ethIntfSliceDefaultConfigDir.get( sliceName )
      resolveHelper.globalDefaultConfig = ethPhyIntfDefaultConfigDir 
      resolveHelper.resolveXcvrMode()
      pList.xcvrMode = resolveHelper.xcvrMode
      return pList

   # return : List of CapList
   def linkModeCapabilitiesToList( self ):
      lmCaps = []
      if self.intfXcvrStatus().xcvrPresence != 'xcvrPresent':
         return lmCaps
      # The Capabilities::ResolverSm writes phyAutonegCapabilities which AutonegSm
      # will copy into autonegCapabilities. So, the two are generally equivalent.
      # Some btests set phyAutonegCapabilities and the do not run the
      # AutonegSm, others set autonegCapabilities.
      # For that reason, we need to check both here.
      # Eventually we can remove phyAutonegCapabilities and switch
      # all usages to autonegCapabilities.  phyAutonegCapabilities defaults to
      # anegModeUnknown. After the resolver runs it will update it to the supported
      # anegMode or anegModeUnsupported.  In keeping with past behavior we will
      # consider both anegModeUnknown and anegModeUnsupported as indicating the
      # interface is not capable of autoneg and therefore not display auto in
      # the capabilities.
      autonegCapable = (
         ( self.status().phyAutonegCapabilities.mode != 'anegModeUnknown' and
           self.status().phyAutonegCapabilities.mode != 'anegModeUnsupported' ) or
         ( self.status().autonegCapabilities.mode != 'anegModeUnknown' and
           self.status().autonegCapabilities.mode != 'anegModeUnsupported' ) )
      caps = self.status().linkModeCapabilities

      # For non-hotswappable BASE-T transceivers, xcvr is always present even
      # though capabilities may not yet be available. The condition makes sure that
      # we return 'Unknown' in these cases.
      # All fixed BASE-T ports support autoneg so we can just check
      # if autonegCapable is False here and then conclude the BASE-T PHY
      # has not yet registered capabilities.
      if ( not autonegCapable and self.intfXcvrStatus().fixed and
           'BASE-T' in self.intfXcvrStatus().xcvrType ):
         lmCaps.append( EthIntfLib.CapList ( 'speedUnknown', 'laneCountUnknown',
                                             False, 'duplexUnknown', False,
                                             False ) )
         return lmCaps

      # This should be consistent with how autonegLib determines when autoneg is
      # active with a default configuration.
      anegMode = self.status().autonegCapabilities.mode
      cl28Cl37ByDefault = ( ( anegMode == AutonegMode.anegModeClause28 or
                              ( anegMode == AutonegMode.anegModeClause37 and
                                caps.exclusive1GbpsFull ) ) and
                            self.status().speedAutoAdj )
      groupStatus = self.speedGroupStatus()
      imcompatElms = None
      if groupStatus and groupStatus.setting:
         speedCompat = Tac.Type( 'Interface::SpeedCompatSetting' )
         cl28Cl37Compat = groupStatus.setting.get( speedCompat.compatibility10g )
         if cl28Cl37ByDefault and cl28Cl37Compat:
            # Default speed is shown as 'auto' if interface supports autoneg at 1g
            # and serdes speed is set to 10g ( 1g and 10g are compatible ).
            defaultMode = 'auto'
         else:
            # For SFP28 port, when serdes speed is set, determine the default
            # linkMode based on serdes speed settings and linkModeCapabilities
            if groupStatus.masterIntf:
               defaultMode = groupStatus.defaultLinkMode( caps )
            else:
               pList = self.linkModePriorityList()
               defaultMode = self.status().defaultLinkMode( pList )
         incompatMode = set( groupStatus.supportedModes ) - \
                        set( groupStatus.setting )
         imcompatElms = Tac.Value( "Interface::EthLinkModeSet" )
         for mode in incompatMode:
            imcompatElms |= groupStatus.supportedModes[ mode ]
      else:
         if cl28Cl37ByDefault:
            defaultMode = 'auto'
         else:
            pList = self.linkModePriorityList()
            defaultMode = self.status().defaultLinkMode( pList )
         # Handle SFP28 case where master interface drives serdes setting.
         if groupStatus and groupStatus.masterIntf and \
            self.name != groupStatus.masterIntf:
            if self.ethIntfStatuses.get( groupStatus.masterIntf ).speed == \
                  EthSpeed.speed25Gbps:
               imcompatElms = Tac.Value( "Interface::EthLinkModeSet",
                                         mode1GbpsFull=True,
                                         mode10GbpsFull=True )
            else:
               imcompatElms = Tac.Value( "Interface::EthLinkModeSet",
                                         mode25GbpsFull=True )

      return EthIntfLib.linkModeCapabilitiesToList( caps,
                                                    autonegCapable=autonegCapable,
                                                    defaultMode=defaultMode,
                                                    showLanes=self.showLanes(),
                                                    imcompatElms=imcompatElms )

   def autonegCapabilitiesMode( self ):
      return self.status().autonegCapabilities.mode

   # return : List of CapList
   def autonegCapabilitiesToList( self ):
      anegCapsList = EthIntfLib.linkModeCapabilitiesToList(
               self.status().autonegCapabilities.linkModes,
               showLanes=self.showLanes() )
      if not anegCapsList:
         anegCapsList = EthIntfLib.CapList( 'speedUnknown', 'laneCountUnknown',
                                            False, 'duplexUnknown', False, False )
      return anegCapsList

   # input: encoding, shall be value of errorCorrectionEncodingToLbls
   # return: List of FecCapList
   def errorCorrectionCapabilitiesToList( self, encoding ):
      if self.intfXcvrStatus().xcvrPresence != 'xcvrPresent':
         return []
      # Use a Dict to store supported link modes for given FEC. If the FEC is
      # the default, set the value to True.
      linkModes = {}
      for linkMode, fecSet in self.status().phyFecCapabilities.iteritems():
         # Don't display linkModes for FEC that are not in the linkMode caps for the
         # interface. This can happen when the Xcvr inserted does not have all
         # linkModes which support FEC available.
         if not self.status().linkModeSupported[ linkMode ]:
            continue
         for attr in EthIntfLib.tokenToFecEncodingAttrs[ encoding ]:
            if getattr( fecSet, attr ):
               # FEC is supported in this linkMode
               linkModes[ linkMode ] = [ False, attr ]
               if( self.intfXcvrStatus().mediaFecStandard.has_key( linkMode ) and
                   self.intfXcvrStatus().mediaFecStandard[ linkMode ] == attr ):
                  # Mark this FEC as default FEC for this linkMode.
                  linkModes[ linkMode ] = [ True, attr ]
                  break

      # Sort based on the order of keys in linkModeToSpeedLanesDuplex
      linkModeCap = []
      showLanes = self.showLanes()
      for linkMode in EthIntfLib.linkModeToSpeedLanesDuplex.iterkeys():
         if linkMode in linkModes:
            speed, lanes, _ = EthIntfLib.linkModeToSpeedLanesDuplex[ linkMode ]
            # fix lanes for legacy interfaces
            speed, lanes = EthIntfLib.modeStrToEthSpeedAndLaneCount(
                     EthPhyIntf._speedLanesStr( speed, lanes, showLanes,
                                                'Format2' ) )
            linkModeCap.append(
                     EthIntfLib.FecCapList( linkModes[ linkMode ][ 1 ], speed,
                                 lanes, showLanes,
                                 True if linkModes[ linkMode ][ 0 ] else False ) )

      return linkModeCap

   #----------------------------------------------------------------------------
   # Utility functions used by "showInterfacesXcvrProperties()"
   #----------------------------------------------------------------------------
   def adminSpeedStr( self ):
      return self.adminSpeed( 'Format2' )

   def adminDuplexStr( self ):
      return self.adminDuplex( 'Format2' )

   def _autonegSuffix( self ):
      if not self.status().linkStatus == 'linkUp':
         return ""

      if self.status().autonegStatus.mode == 'anegModeUnknown':
         # If auto-negotiation is not enabled and the link mode is the
         # default, the speed/duplex setting are the default for the
         # media type of the Xcvr, else they are forced by configuration
         if self.config().linkMode != 'linkModeUnknown':
            return 'forced'
         else:
            return ""
      else:
         return 'autoNegotiated'

   #----------------------------------------------------------------------------
   # Utility functions used by "showInterfacesDebounceTime()"
   #----------------------------------------------------------------------------
   def linkUpDebouncePeriod( self ):
      return self.config().linkUpDebouncePeriod

   def linkDownDebouncePeriod( self ):
      return self.config().linkDownDebouncePeriod

   #------------------------------------------------------------------------------
   # Sets the link debounce times for the interface to the specified values.
   #------------------------------------------------------------------------------
   def setLinkDebounceTimes( self, linkUpDebounceTime, linkDownDebounceTime ):
      self.config().linkUpDebouncePeriod = linkUpDebounceTime
      self.config().linkDownDebouncePeriod = linkDownDebounceTime

   #----------------------------------------------------------------------------
   # Sets the timestamp mode for an interface.
   #----------------------------------------------------------------------------
   def setTimestampMode( self, timestampMode ):
      self.config().timestampMode = timestampMode

   #----------------------------------------------------------------------------
   # When linkMode is set to auto-negotiate, verify support for the desired
   # speeds and duplex that are advertised to the link partner.
   #----------------------------------------------------------------------------
   def isAdvertisedModesSupported( self, advertisedModes ):
      if advertisedModes is not None:
         # confirm that the advertised modes are supported for autoneg
         return self.status().advertisedModesSupported( advertisedModes )
      if self.status().phyAutonegCapabilities.mode == 'anegModeClause73':
         # clause 73 needs a speed to advertise
         return False
      # if advertisedModes is not explicitly configured then we advertise what we can
      return True

   #----------------------------------------------------------------------------
   # Check if the desired linkMode is supported for the interface.
   #----------------------------------------------------------------------------
   def isLinkModeSupported( self, desiredLinkMode, advertisedModes=None ):
      supported = True
      if desiredLinkMode != 'linkModeUnknown':
         if self.lookupPhysical():
            if self.intfXcvrStatus().xcvrPresence != 'xcvrPresent':
               # we optimistically claim support after a warning label
               self.mode_.addWarning(
                  'Transceiver for interface %s is not present. '
                  'Cannot verify compatibility of speed and duplex settings.' %
                  self.shortname_ )
            else:
               if not self.status().linkModeSupported[ desiredLinkMode ]:
                  # desired linkMode is not supported
                  self.mode_.addError(
                     'Speed and duplex settings are not compatible '
                     'with transceiver for interface %s.' % self.name )
                  supported = False
               elif desiredLinkMode == 'linkModeAutoneg':
                  supported = self.isAdvertisedModesSupported( advertisedModes )
                  if not supported:
                     self.mode_.addError( 'Cannot advertise desired speeds and '
                                          'duplex settings for interface %s.'
                                          % self.name )
         else:
            # we optimistically claim support after a warning label
            self.mode_.addWarning(
               'Transceiver is not present. '
               'Cannot verify compatibility of speed and duplex settings.' )

      return supported

   #----------------------------------------------------------------------------
   # Sets the modes that are advertised when autonegotiation is turnd on.
   #----------------------------------------------------------------------------
   def setAdvertisedModes( self, advertisedModes=None ):
      cfg = self.config()
      if advertisedModes is None:
         cfg.doClearAdvertisedModes()
      else:
         cfg.advertisedModesLocal = advertisedModes
         cfg.advertisedModesConfiguredLocal = True
         cfg.advertisedModesConfiguredForSfp1000BaseT = False

   #----------------------------------------------------------------------------
   # Sets the linkMode (speed and duplex) of the interface to a specified value
   # or the default value.
   #
   # When the linkMode is set to auto-negotiate, advertisedModes can be used to
   # limit the modes that are advertised to the link partner.
   #----------------------------------------------------------------------------
   def setLinkMode( self, linkMode, advertisedModes=None ):
      if linkMode != 'linkModeAutoneg' and linkMode != 'linkModeAuto40GbpsFull' :
         # advertisedModes is only valid when the port is set to autonegotiate
         assert advertisedModes is None

      if not self.isLinkModeSupported( linkMode, advertisedModes ):
         return

      self.setAdvertisedModes( advertisedModes )
      self.config().linkModeLocal = linkMode

   #----------------------------------------------------------------------------
   # Special case to allow users to configure the link mode *only* on
   # 1000BASE-T SFPs, into one of three values:
   #
   # o autoneg, 1000/full  (Default)
   # o autoneg, 100/full
   # o force link at 100/full (No autoneg)
   #
   # This really ought to be unified with "setLinkMode" above, but
   # since these settings affect all our platforms (all PHYs and all
   # MACs), and this is going into a release with fairly short notice,
   # defer that work (and testing!) until later.
   # ----------------------------------------------------------------------------
   def setLinkModeSfp100BaseTAuto100Full( self ):
      desiredLinkMode = 'linkModeAutoneg'
      advertisedMode = Tac.Value( 'Interface::EthLinkModeSet' )
      advertisedMode.mode100MbpsFull = True
      if not self.isLinkModeSupported( desiredLinkMode, advertisedMode ):
         return

      if self.lookupPhysical() and \
         ( self.intfXcvrStatus().xcvrPresence == 'xcvrPresent' ):
         if self.intfXcvrStatus().xcvrType != '1000BASE-T':
            # Actually, this test won't catch the case where this
            # command is entered on native RJ45 1000BASE-T ports,
            # but that is a misconfig, and the failure mode is that
            # those ports will autoneg to their max capabilities.
            self.mode_.addError( 'Speed and duplex settings are not compatible '
                                 'with transceiver for interface %s.' %
                                 self.name )
            return

      # Disable auto-100/full or forced-100/full, and just go back to
      # the SFP's default capabilities -- auto-1000/full. (In fact,
      # note that we don't even verify that there is config state for
      # this interface, or an xcvr is present, etc.)
      cfg = self.config()

      elms = Tac.Value( 'Interface::EthLinkModeSet' )
      elms.mode100MbpsFull = True
      self.setAdvertisedModes( elms )
      cfg.advertisedModesConfiguredForSfp1000BaseT = True

      cfg.linkModeLocal = desiredLinkMode

   #----------------------------------------------------------------------------
   # Sets the loopback mode of the interface to a specified value or the
   # default value.
   #----------------------------------------------------------------------------
   def setNoLoopbackMode( self ):
      cfg = self.config()
      cfg.loopbackMode = 'loopbackNone'

   def setLoopbackMode( self, loopbackSources, loopbackDevice ):
      cfg = self.config()
      if loopbackSources == 'network' and loopbackDevice == 'phy':
         cfg.loopbackMode = LOOPBACK_PHYREMOTE
      elif loopbackSources == 'system' and loopbackDevice == 'mac':
         cfg.loopbackMode = LOOPBACK_MAC
      elif loopbackSources == 'system' and loopbackDevice == 'phy':
         cfg.loopbackMode = LOOPBACK_PHY

   def loopbackModeSupported( self, lbMode='any' ):
      if lbMode == 'any':
         return self.status() and \
             any( self.status().loopbackModeSupported.values() )
      else:
         return self.status() and \
             lbMode in self.status().loopbackModeSupported.keys() and \
             self.status().loopbackModeSupported[ lbMode ]

   def showLoopbackMode( self, intfStatusModel ):
      intfStatusModel.loopbackMode = self.status().loopbackMode
      return intfStatusModel.loopbackMode

   #----------------------------------------------------------------------------
   # Sets the autoneg 25g mode of the interface to a specified value or the
   # default value.
   #----------------------------------------------------------------------------
   def setAutonegStandard( self, aneg25gModes=None ):
      negotiation25gMode = Tac.Value( "Interface::Negotiation25gMode" )

      if aneg25gModes:
         # SetRule returns a tuple of names and keywords,
         # the names in this case are names of TAC attributes.
         for attr in aneg25gModes:
            setattr( negotiation25gMode, attr, True )
      else:
         # This configures all Negotiation Modes to default setting,
         # which is consortium extension and ieee standard both 'On'
         negotiation25gMode.consortium25g = True
         negotiation25gMode.ieee25g = True

      self.config().negotiation25gModeConfig = negotiation25gMode

   def autonegStandardSupported( self, aneg25gMode ):
      return getattr( self.status().aneg25gModeCaps, aneg25gMode )

   #----------------------------------------------------------------------------
   # Sets the flowcontrol of the interface to a specified value or the default
   # value.
   #----------------------------------------------------------------------------
   def setFlowcontrol( self, direction, value ):
      cfg = self.config()
      if self.lookupPhysical():
         status = self.status()
         caps  = status.autonegCapabilities
         flowControlValid = {
            ( 'send', 'flowControlConfigUnknown' ) : True,
            ( 'send', 'flowControlConfigOff' ) : True,
                  ( 'send', 'flowControlConfigOn' ) :
            status.txFlowcontrolCapabilities != 'flowControlNotCapable' ,
            ( 'send', 'flowControlConfigDesired' ) :
                  caps.txPause and
                  status.txFlowcontrolCapabilities != 'flowControlNotCapable',
            ( 'receive', 'flowControlConfigUnknown' ) : True,
            ( 'receive', 'flowControlConfigOff' ) : True,
            ( 'receive', 'flowControlConfigOn' ) :
                  status.rxFlowcontrolCapabilities != 'flowControlNotCapable' ,
            ( 'receive', 'flowControlConfigDesired' ) :
                  caps.rxPause and
                  status.rxFlowcontrolCapabilities != 'flowControlNotCapable'
            }
         if not flowControlValid[ ( direction, value ) ]:
            self.mode_.addError( 'Flow control setting is not compatible with '
                                 'interface %s.' % self.name )
            return

      if direction == 'send':
         cfg.txFlowcontrolLocal = value
      else:
         assert direction == 'receive'
         cfg.rxFlowcontrolLocal = value

   def setCongestionDropsLoggingMode( self, logMode ):
      self.config().congestionDropsLoggingMode = logMode

   #----------------------------------------------------------------------------
   # Sets the L2 Mtu value
   #----------------------------------------------------------------------------
   def setL2Mtu( self, value ):
      cfg = self.config()
      cfg.l2Mtu = value

   #----------------------------------------------------------------------------
   # Returns an unsorted list of EthPhyIntf objects representing all the existing
   # Interface::EthPhyIntfStatus objects.
   #----------------------------------------------------------------------------
   @staticmethod
   def getAllPhysical( mode ):
      intfs = []
      for name in ethPhyIntfStatusDir:
         if name.startswith( "Internal" ):
            # Skip internal interfaces
            # BUG944 - need a more general way of doing this. Should 'internal' be
            # an attribute of the IntfConfig intead?
            continue
         if name.startswith( "UnconnectedEthernet" ):
            # Skip them here, handled in UnconnectedEthPhyIntf.
            continue
         else:
            intf = EthPhyIntf( name, mode )
         if intf.lookupPhysical():
            intfs.append( intf )
      return intfs

   #----------------------------------------------------------------------------
   # Hooks/support for "clear counters".
   #----------------------------------------------------------------------------
   def clearCounters( self, sessionOnly=False ):
      ckpt = self.getCounterCheckpoint( className="Interface::EthIntfCounterBase",
                                        sessionOnly=sessionOnly )
      if ckpt is None or self.counter() is None:
         return
      ckpt.ethStatistics = self.counter().ethStatistics
      EthIntf.clearCounters( self, sessionOnly=sessionOnly )

   #----------------------------------------------------------------------------
   # Is this port eligible for "switchport" commands?
   #----------------------------------------------------------------------------
   def switchportEligible( self ):
      return self.name.startswith( "Ethernet" )

   def setDefault( self ):
      # First run through the registered hooks and confirm that we can go back
      # to the default link mode.
      if checkChangeCancelHooks( self.mode_, [ self ], 'linkModeUnknown', None ):
         return
      if checkChangeCancelOnSfp28( self.mode_, [ self ], 'linkModeUnknown', None ):
         return

      self.setFlowcontrol( 'send', 'flowControlConfigUnknown')
      self.setFlowcontrol( 'receive', 'flowControlConfigUnknown')
      self.setMacAddr( None )
      self.setL2Mtu( 0 )
      self.setTimestampMode( 'timestampModeDisabled' )
      self.setLinkMode( 'linkModeUnknown' )
      self.setLinkDebounceTimes( 0, 0 )
      self.setCongestionDropsLoggingMode( 'useGlobal' )
      self.setUnidirectionalLinkMode( 'uniLinkModeDisabled' )
      self.setL2Mtu( 0 )
      self.setL2Mru( 0 )
      self.setNoLoopbackMode()
      self.setErrorCorrectionEncoding( noOrDefault='default' )
      self.noDefaultErrorCorrectionBypass()
      self.setAutonegStandard()
      EthIntf.setDefault( self )

   #----------------------------------------------------------------------------
   # Does this port support UnidirectionalLinkMode?
   #----------------------------------------------------------------------------
   def unidirectionalLinkSupported( self ):
      return self.status() and self.status().unidirectionalLinkSupported

   #----------------------------------------------------------------------------
   # Hooks/support for "unidirectional" config commands
   #----------------------------------------------------------------------------
   def setUnidirectionalLinkMode( self, unidirectionalLinkMode ):
      self.config().unidirectionalLinkMode = unidirectionalLinkMode

   #----------------------------------------------------------------------------
   # Hooks/support for "show interfaces [<name>] unidirectional"
   #----------------------------------------------------------------------------
   uniLinkModeStrings = {
         'uniLinkModeSendOnly' : 'send-only',
         'uniLinkModeReceiveOnly' : 'receive-only',
         'uniLinkModeSendReceive' : 'send-receive',
         'uniLinkModeDisabled' : 'disabled',
   }

   def showUnidirectionalLinkMode( self ):
      uniLinkMode = self.config().unidirectionalLinkMode
      return EthPhyIntf.uniLinkModeStrings[ uniLinkMode ]

   #----------------------------------------------------------------------------
   # Support for error-correction configuration
   #----------------------------------------------------------------------------
   def errorCorrectionEncodingSupported( self, encoding, ignoreOutputErrors=True ):

      if encoding and encoding not in EthIntfLib.tokenToFecEncodingAttrs.keys():
         self.mode_.addError( 'Invalid error-correction encoding %s' % encoding )
         return False
      supported = False
      if self.lookupPhysical():
         for fecSet in self.status().phyFecCapabilities.itervalues():
            for attr in EthIntfLib.tokenToFecEncodingAttrs[ encoding ]:
               supported |= getattr( fecSet, attr )
         if not supported and not ignoreOutputErrors:
            self.mode_.addError(
               'Error-correction encoding %s is unsupported for interface %s.' %
               ( encoding, self.name ) )
         return supported
      else:
         return True

   def errorCorrectionSupported( self ):
      """If the interface supports any encoding value this will return
      True.  If no encodings at any speed or the EthPhyIntfStatus has
      not yet been created it will return False. We use this to
      suppress output in show commands on interfaces that just don't
      suppport FEC at all or we don't yet know if they support FEC."""

      if self.lookupPhysical():
         for fecSet in self.status().phyFecCapabilities.itervalues():
            for attrList in EthIntfLib.tokenToFecEncodingAttrs.values():
               for attr in attrList:
                  if getattr( fecSet, attr ):
                     return True
      return False

   def coherentErrorCorrectionSupported( self ):
      if 'coherent' in errorCorrectionShCapFns:
         return errorCorrectionShCapFns[ 'coherent' ].checkFn( self.name )
      return None

   def coherentErrorCorrectionEncodingSupp( self, encoding ):
      allPhyCoherentStatus = {}
      # Get all coherent status' at once.
      if 'Modulation' in interfaceCapabilityFns:
         allPhyCoherentStatus = interfaceCapabilityFns[ 'Modulation' ].getAll()

      if 'coherent' in errorCorrectionShCapFns:
         return errorCorrectionShCapFns[ 'coherent' ].checkFec(
                   allPhyCoherentStatus[ self.name ].fecCapabilities, encoding )
      return None

   def setErrorCorrectionEncoding( self, encoding=None, noOrDefault=False ):
      """
      1. if encoding is specified we need to go down one of the client/coherent
         paths, configure the encoding and set the other path to default.
      2. if no/default is specified go down both the paths and configure no/default.
         Note: no/default with or without encoding will exercise both paths
               with no/default
      """
      coherentEncoding = nonCoherentEncoding = encoding
      coherentNoOrDefault = nonCoherentNoOrDefault = noOrDefault
      if 'coherent' in errorCorrectionCfgFns:
         if ( encoding and
              EthIntfLib.encodingType( encoding ) == "non-coherent" ):
            coherentEncoding = None
            coherentNoOrDefault = "default"
         self.config().isCoherentFecConfigured = \
            errorCorrectionCfgFns[ 'coherent' ].configFn( self, coherentEncoding,
                                                          coherentNoOrDefault )
      if ( encoding and
           EthIntfLib.encodingType( encoding ) == "coherent" ):
         nonCoherentEncoding = None
         if noOrDefault != True:
            nonCoherentNoOrDefault = "default"
      self.setNonCoherentFec( nonCoherentEncoding, nonCoherentNoOrDefault )

   def setNonCoherentFec( self, encoding=None, noOrDefault=False ):
      isNo = noOrDefault is True
      isDefault = str( noOrDefault ).lower() == 'default'

      if encoding and not self.errorCorrectionEncodingSupported( encoding, False ):
         return

      fecEncoding = Tac.nonConst( self.config().fecEncodingConfig )

      # Get the current encoding and clear the 'none' value as we are
      # about to configure a valid encoding.
      # If no form with no encoding then that completely disables FEC
      # no and default form with an encoding just turns off that encoding.
      if encoding:
         for attr in EthIntfLib.tokenToFecEncodingAttrs[ encoding ]:
            setattr( fecEncoding,
                     attr,
                     not bool( noOrDefault ) )
         # If we enabled an encoding then fecEncodingDisabled must be False.
         if not bool( noOrDefault ):
            fecEncoding.fecEncodingDisabled = False

      # Without an encoding no and default differ.
      # No disables FEC by setting fecEncodingDisabled to True
      # Default runs FEC when appropriate based on the xcvr media
      elif isNo or isDefault:
         # This disables FEC
         fecEncoding.fecEncodingDisabled = isNo
         # Turn off each of the individual FEC types.
         for attrList in EthIntfLib.tokenToFecEncodingAttrs.itervalues():
            for attr in attrList:
               setattr( fecEncoding, attr, False )
      else:
         raise ValueError( "Invalid input parameter combination: %s and %s" %
                           encoding, noOrDefault )

      self.config().fecEncodingConfig = fecEncoding


   def errorCorrectionBypassSupported( self, bypassValue, ignoreOutputErrors=True ):
      if not self.errorCorrectionEncodingSupported( 'reed-solomon',
                                                    ignoreOutputErrors ):
         return False

      if bypassValue and bypassValue not in EthIntfLib.tokenToBypassAttr.keys():
         self.mode_.addError( "Invalid bypass value %s" % bypassValue )
         return False

      if self.lookupPhysical():
         supported = getattr( self.status().fecBypassCapabilities,
                              EthIntfLib.tokenToBypassAttr[ bypassValue ] )
         if not supported and not ignoreOutputErrors:
            self.mode_.addError(
               'Error-correction bypass is unsupported for interface %s.' %
               self.name )
         return supported
      else:
         return True

   FecBypassMode = Tac.Type( 'Interface::EthFecBypassMode' )
   def setErrorCorrectionBypass( self, bypassValue ):
      if bypassValue and not self.errorCorrectionBypassSupported( bypassValue,
                                                                  False ):
         return
      self.config().fecBypass = EthIntfLib.tokenToBypassAttr[ bypassValue ]

   def noDefaultErrorCorrectionBypass( self ):
      self.config().fecBypass = EthPhyIntf.FecBypassMode.fecBypassDisabled

   def _populateCountersErrorsModel( self, errors, attributes ):
      def stat( counter, checkpoint, attribute ):
         if counter is None:
            # Unders some circumstances (e.g. a linecard being hotswapped),
            # the counter argument may be None. In theory, the right thing
            # to do in this case would be to return notFoundString.
            # This would however cause some non-optional counter attributes
            # in the interface counters model (e.g., inOctets, inUcastPkts, etc.)
            # to be missing. Because of that we for now simply return 0.
            # Mark as BUG159812
            return 0

         sysdbValue = getattr( counter.ethStatistics, attribute, None )
         if sysdbValue is None:
            sysdbValue = getattr( counter.statistics, attribute, None )
         if sysdbValue is None:
            return 'n/a'
         if checkpoint:
            checkpointValue = getattr( checkpoint.ethStatistics, attribute, None )
            if checkpointValue is None:
               checkpointValue = getattr( checkpoint.statistics, attribute, 0 )
         else:
            checkpointValue = 0
         return sysdbValue - checkpointValue
      checkpoint = self.getLatestCounterCheckpoint()
      counter = self.counter()
      for attribute in attributes:
         setattr( errors, attribute, stat( counter, checkpoint, attribute ) )

   def getCountersErrorsModel( self ):
      errors = ErrorCounters()
      attributes = ( 'fcsErrors', 'alignmentErrors', 'symbolErrors', 'inErrors',
                     'frameTooShorts', 'frameTooLongs', 'outErrors', )
      self._populateCountersErrorsModel( errors, attributes )
      return errors

   def getCountersHalfDuplexErrorsModel( self ):
      errors = HalfDuplexErrorCounters()
      attributes = ( 'singleCollisionFrames', 'multipleCollisionFrames',
                     'lateCollisions', 'excessiveCollisions',
                     'deferredTransmissions', )
      self._populateCountersErrorsModel( errors, attributes )
      return errors

   def countersErrorsSupported( self ):
      return True

#-------------------------------------------------------------------------------
# Register the EthPhyIntf class as a type of physical interface.
# Routing protocol is not supported on management interfaces, and hence excluded
# in ruleWithRoutingSupport
#-------------------------------------------------------------------------------
IntfCli.Intf.addPhysicalIntfType( EthPhyIntf,
                               ( EthPhyAutoIntfType, MgmtAutoIntfType ),
                               withIpSupport=True,
                               withRoutingProtoSupport=True,
                               matcherWithRoutingProtoSupport=EthPhyIntf.ethMatcher )

#-------------------------------------------------------------------------------
# Register the EthPhyIntf class as a type that supports L2 mtu.
#-------------------------------------------------------------------------------
registerL2MtuValidIntfType( EthPhyIntf )

#-------------------------------------------------------------------------------
# This class represents an EthPhyDefaultCapabilities entity.
#-------------------------------------------------------------------------------
class EthPhyDefaultCaps( object ):

   def __init__( self, name, parentDir ):
      t0( "EthPhyIntf::__init__", name )
      self.name = name
      self.ethPhyDefaultCapsDir = parentDir

   def showLanes( self ):
      return EthPhyIntf._showLanes( self.capabilities() )

   def linkModeCapabilitiesToList( self ):
      return EthIntfLib.linkModeCapabilitiesToList(
                     self.capabilities().linkModeCapabilities,
                     showLanes=self.showLanes() )

   # return : List of CapList
   def autonegCapabilitiesToList( self, autonegMode ):
      anegCaps = self.capabilities().autonegCapabilities
      if autonegMode not in anegCaps:
         anegCapsList = EthIntfLib.CapList( 'speedUnknown', 'laneCountUnknown',
                                            False, 'duplexUnknown', False, False )
         return anegCapsList

      # we never resolve to anegModeUnknown
      assert anegCaps[ autonegMode ].mode != AutonegMode.anegModeUnknown

      anegLinkModes = anegCaps[ autonegMode ].linkModes
      return EthIntfLib.linkModeCapabilitiesToList( anegLinkModes,
                                                    showLanes=self.showLanes() )

   # return : List of CapList
   def anegCl73ModeCapabilitiesToList( self, negCl73Mode ):
      anegCl73Caps = []
      if AutonegMode.anegModeClause73 not in self.capabilities().autonegCapabilities:
         anegCl73Caps = EthIntfLib.CapList( 'speedUnknown', 'laneCountUnknown',
                                            False, 'duplexUnknown', False, False )
         return anegCl73Caps

      if self.showLanes():
         mode50g2Lane = "50G-2/full"
         mode100g4Lane = "100G-4/full"
      else:
         mode50g2Lane = "50G/full"
         mode100g4Lane = "100G/full"

      # we never resolve to anegModeUnknown
      assert self.capabilities().autonegCapabilities[ \
                  AutonegMode.anegModeClause73 ] != AutonegMode.anegModeUnknown

      clause73LinkModes = self.capabilities().autonegCapabilities[ \
            AutonegMode.anegModeClause73 ].linkModes

      if negCl73Mode == 'ieee':
         # Both 25G and 50G Clause 73 Auto-Negotiation are not always supported
         # by hardware, so we must check our aneg25gModeCaps before displaying
         # those speeds.
         IeeeSupport = self.capabilities().aneg25gModeCaps.ieee25g
         for mode, supported, linkModeStr in [
               ( clause73LinkModes.mode25GbpsFull, IeeeSupport, '25G/full' ),
               ( clause73LinkModes.mode40GbpsFull, True, '40G/full' ),
               ( clause73LinkModes.mode50GbpsFull1Lane, IeeeSupport, '50G-1/full' ),
               ( clause73LinkModes.mode100GbpsFull, True, mode100g4Lane ),
               ( clause73LinkModes.mode100GbpsFull2Lane, True, '100G-2/full' ),
               ( clause73LinkModes.mode200GbpsFull4Lane, True, '200G-4/full' ) ]:
            if mode and supported:
               speed, lanes = EthIntfLib.modeStrToEthSpeedAndLaneCount(
                                             linkModeStr.partition('/')[0] )
               duplex = linkModeStr.partition('/')[2]
               anegCl73Caps.append( EthIntfLib.CapList( speed, lanes,
                                                        self.showLanes(), duplex,
                                                        False, False ) )

      elif negCl73Mode == 'consortium':
         if self.capabilities().aneg25gModeCaps.consortium25g:
            for mode, linkModeStr in [
                  ( clause73LinkModes.mode25GbpsFull, '25G/full' ),
                  ( clause73LinkModes.mode50GbpsFull, mode50g2Lane ),
                  ( clause73LinkModes.mode400GbpsFull8Lane, '400G-8/full' ) ]:
               if mode:
                  speed, lanes = EthIntfLib.modeStrToEthSpeedAndLaneCount(
                                             linkModeStr.partition('/')[0] )
                  duplex = linkModeStr.partition('/')[2]
                  anegCl73Caps.append( EthIntfLib.CapList( speed, lanes,
                                                           self.showLanes(), duplex,
                                                           False, False ) )

      else:
         # If we hit this case, perhaps some new consortium Mode was added
         assert False, "Unrecognized Clause 73 Mode " + negCl73Mode

      return anegCl73Caps

   def errorCorrectionEncodings( self ):
      capableEncodingTokens = []
      for token, attrs in EthIntfLib.tokenToFecEncodingAttrs.iteritems():
         for attr in attrs:
            for fecCapabilities in self.capabilities().fecCapabilities.itervalues():
               if( ( getattr( fecCapabilities, attr ) ) and
                   ( token not in capableEncodingTokens ) ):
                  capableEncodingTokens.append( token )
      return capableEncodingTokens

   # input: encoding, shall be value of errorCorrectionEncodingToLbls
   # return: List of FecCapList
   def errorCorrectionLinkModes( self, encoding ):
      # The CapabilitiesResolver checks that the linkModes in FEC capabilities are
      # actually supported in linkModeCapabilities. Therefore we do not need
      # to cross check the two again.
      linkModeToEncodings = {}
      for linkMode, fecCapsSet in self.capabilities().fecCapabilities.iteritems():
         for attr in EthIntfLib.tokenToFecEncodingAttrs[ encoding ]:
            if getattr( fecCapsSet, attr ):
               if linkMode not in linkModeToEncodings:
                  linkModeToEncodings.update( { linkMode : attr } )

      # Sort based on the order of keys in linkModeToSpeedLanesDuplex
      linkModeCap = []
      showLanes = self.showLanes()
      for linkMode in EthIntfLib.linkModeToSpeedLanesDuplex.iterkeys():
         if linkMode in linkModeToEncodings:
            speed, lanes, _ = EthIntfLib.linkModeToSpeedLanesDuplex[ linkMode ]
            # fix lanes for legacy interfaces
            speed, lanes = EthIntfLib.modeStrToEthSpeedAndLaneCount(
                     EthPhyIntf._speedLanesStr( speed, lanes, showLanes,
                                                'Format2' ) )
            linkModeCap.append( EthIntfLib.FecCapList(
                                                linkModeToEncodings[ linkMode ],
                                                speed, lanes, showLanes,
                                                False ) )
      return linkModeCap

   @Tac.memoize
   def capabilities( self ):
      return self.ethPhyDefaultCapsDir[ self.name ]

   @staticmethod
   def getIntfCapsInfo( intfs ):
      allEthPhyDefaultCaps = {}
      for subDirs in ethPhyDefaultCapsSliceDir.itervalues():
         for ethPhyDefaultCapsDir in subDirs.itervalues():
            for intfName in ethPhyDefaultCapsDir.portCaps:
               allEthPhyDefaultCaps[ intfName ] = \
                     EthPhyDefaultCaps( intfName, ethPhyDefaultCapsDir )

      # Our return value, a list of (EthPhyIntf, EthPhyDefaultCapapbilities) tuples.
      # The return list preserves the same order as the intfs list.
      intfCapsInfos = []
      for intf in intfs:
         # There should never be a case where an EthPhyDefaultCapabilities object
         # is missing from Sysdb. It should be invariant that there is a one-to-one
         # matching of EthPhyIntfStatus to EthPhyDefaultCapabilities; however, to
         # be absolutely safe we check anyways
         if allEthPhyDefaultCaps.has_key( intf.name ):
            intfCapsInfos.append( ( intf, allEthPhyDefaultCaps[ intf.name ] ) )

      return intfCapsInfos


#-------------------------------------------------------------------------------
# A subclass of EthPhyIntf class for physical Unconnected Ethernet interfaces.
#-------------------------------------------------------------------------------

# Guard for UnconnectedEthernet when there are none on the system
def ueGuard( mode, token ):
   if UnconnectedEthPhyIntf.getAllPhysical( mode ):
      return None
   return CliParser.guardNotThisPlatform

class UnconnectedEthPhyIntf(  EthPhyIntf ):

   #----------------------------------------------------------------------------
   # Creates a new UnconnectedEthPhyIntf instance of the specified name.
   #----------------------------------------------------------------------------
   def __init__( self, name, mode ):
      t0( "UnconnectedEthPhyIntf::__init__", name )
      # Deal with the fact that short prefix of UnconnectedEthernet is Ue (not Un)
      if name.lower().startswith( 'ue' ):
         suffix = name[ len( 'ue' ): ]
         name = 'UnconnectedEthernet' + suffix
      EthPhyIntf.__init__( self, name, mode )

   def isLinkModeSupported( self, desiredLinkMode, advertisedModes=None ):
      if desiredLinkMode != 'linkModeUnknown' and self.lookupPhysical():
         if not self.status().linkModeSupported[ desiredLinkMode ]:
            # desired linkMode is not supported
            self.mode_.addError( 'Speed and duplex settings are not available on '
                                 'interface %s.' % self.name )
            return False
         elif( desiredLinkMode == 'linkModeAutoneg' and
               not self.isAdvertisedModesSupported( advertisedModes ) ):
            self.mode_.addError( 'Cannot advertise desired speeds and '
                                 'duplex settings for interface %s.'
                                 % self.name )
            return False
      return True

   def switchportEligible( self ):
      return True

   #----------------------------------------------------------------------------
   # The rule for matching Ethernet unconnected interface names.  When this pattern
   # matches, it returns an instance of the UnconnectedEthPhyIntf class.
   #
   # This rule gets added to the Intf.rule when this class is registered with
   # the Intf class by calling Intf.addPhysicalIntfType, below.
   #----------------------------------------------------------------------------
   matcher = PhysicalIntfRule.PhysicalIntfMatcher( 'UnconnectedEthernet',
                                              alternates=[ 'Ue' ],
                                              value=lambda mode, intf:
                                              UnconnectedEthPhyIntf( intf, mode ),
                                              guard=ueGuard )

   def linkModeCapabilitiesToList( self ):
      # There are no xcvrs on unconnected ports
      assert self.intfXcvrStatus().xcvrPresence != 'xcvrPresent'

      # Just return the default link mode
      caps = self.status().linkModeCapabilities
      defaultMode = self.config().defaultConfig.linkModeLocal
      return EthIntfLib.linkModeCapabilitiesToList( caps, autonegCapable=False,
                                                    defaultMode=defaultMode,
                                                    showLanes=self.showLanes() )
   def linkModeCapabilitiesStr( self ):
      # There are no xcvrs on unconnected ports
      assert self.intfXcvrStatus().xcvrPresence != 'xcvrPresent'

      # Just return the default link mode
      caps = self.status().linkModeCapabilities
      defaultMode = self.config().defaultConfig.linkModeLocal
      return EthIntfLib.linkModeCapabilitiesToStr( caps, autonegCapable=False,
                                                   defaultMode=defaultMode,
                                                   showLanes=self.showLanes() )

   #----------------------------------------------------------------------------
   # Returns an unsorted list of UnconnectedEthPhyIntf objects representing all the
   # existing Interface::EthPhyIntfStatus objects for unconnected ports.
   #----------------------------------------------------------------------------
   @staticmethod
   def getAllPhysical( mode ):
      intfs = []
      for name in ethPhyIntfStatusDir:
         if not name.startswith( "UnconnectedEthernet" ):
            continue
         intf = UnconnectedEthPhyIntf( name, mode )
         if intf.lookupPhysical():
            intfs.append( intf )
      return intfs

#-------------------------------------------------------------------------------
# Register the UnconnectedEthPhyIntf class as a type of physical interface.
#-------------------------------------------------------------------------------
IntfCli.Intf.addPhysicalIntfType( UnconnectedEthPhyIntf,
                                  UnconnectedEthPhyAutoIntfType,
                                  withIpSupport=True,
                                  withRoutingProtoSupport=True )

IntfRange.registerIntfTypeGuard( UnconnectedEthPhyAutoIntfType, ueGuard )

#-------------------------------------------------------------------------------
# Register the EthPhyIntf class as able to participate in interface ranges.
#-------------------------------------------------------------------------------

def _updateEthIntfHelpDescForModular():
   """ Updates the EthPhyAutoIntfType to have the correct help description
   for slot/port numbers. """
   EthPhyAutoIntfType.helpDesc = [ 'Slot number', 'Port number' ]

FruCli.registerModularSystemCallback( _updateEthIntfHelpDescForModular )

EthPhyAutoIntfType.registerEthPhyIntfClass( EthPhyIntf )
UnconnectedEthPhyAutoIntfType.registerEthPhyIntfClass( UnconnectedEthPhyIntf )

def _updateMgmtPhyIntfHelpDescForModular():
   """ Updates the MgmtPhyAutoIntfType to have the correct help description
   for slot/port numbers. """
   MgmtPhyAutoIntfType.helpDesc = [ 'Slot number', 'Port number' ]
FruCli.registerModularSystemCallback( _updateMgmtPhyIntfHelpDescForModular )

#-------------------------------------------------------------------------------
# Adds dataplan-specific (e.g. not management) CLI commands to the
# "config-if" mode.
#-------------------------------------------------------------------------------
class DataplaneIntfModelet( CliParser.Modelet ):
   def __init__( self, mode ):
      CliParser.Modelet.__init__( self )

   @staticmethod
   def shouldAddModeletRule( mode ):
      return ( isinstance( mode.intf, EthPhyIntf ) and
               mode.intf.switchportEligible() )

   modeletParseTree = CliParser.ModeletParseTree()

#-------------------------------------------------------------------------------
# Adds Ethernet-specific CLI commands to the "config-if" mode.
#-------------------------------------------------------------------------------
class EthIntfModelet( CliParser.Modelet ):
   def __init__( self, mode ):
      CliParser.Modelet.__init__( self )

   @staticmethod
   def shouldAddModeletRule( mode ):
      return isinstance( mode.intf, EthPhyIntf )

   modeletParseTree = CliParser.ModeletParseTree()

#--------------------------------------------------------------------------------
# [ no | default ] mac-address router MAC_ADDR
# [ no | default ] mac-address MAC_ADDR
#
# Note that "mac-address 0.0.0" will reset the MAC address to the default.
# Remarkably, this is consistent with the industry-standard.
#--------------------------------------------------------------------------------
matcherMacAddress = CliMatcher.KeywordMatcher( 'mac-address',
      helpdesc='Set interface MAC address' )
class MacAddrCmd( CliCommand.CliCommandClass ):
   syntax = 'mac-address ( MAC_ADDR | ( router MAC_ADDR ) )'
   noOrDefaultSyntax = 'mac-address [ router ] ...'
   data = {
      'mac-address' : matcherMacAddress,
      'router' : 'use MAC address for routing on interface',
      'MAC_ADDR' : MacAddr.macAddrMatcher,
   }

   @staticmethod
   def handler( mode, args ):
      macAddr = args[ 'MAC_ADDR' ]
      mode.intf.setMacAddr( macAddr, routerMacConfigured='router' in args )

   @staticmethod
   def noOrDefaultHandler( mode, args ):
      mode.intf.setMacAddr( None )

EthIntfModelet.addCommandClass( MacAddrCmd )

#--------------------------------------------------------------------------------
# [ no| default ] link-debounce time LINK_DEBOUNCE_TIME [ LINK_DOWN_DEBOUNCE_TIME ]
#
# Note that a debounce time of 0ms will disable link debouncing. Link
# debouncing is disabled by default.
#--------------------------------------------------------------------------------
class LinkDebounceCmd( CliCommand.CliCommandClass ):
   syntax = 'link-debounce time LINK_DEBOUNCE_TIME [ LINK_DOWN_DEBOUNCE_TIME ]'
   noOrDefaultSyntax = 'link-debounce ...'
   data = {
      'link-debounce' : 'Configure a link debounce timer for this interface',
      'time' : 'Specify the debounce time period (in milliseconds)',
      'LINK_DEBOUNCE_TIME' : CliMatcher.IntegerMatcher( 0, 1800000,
         helpdesc='Debounce time for both link-up and link-down in milliseconds' ),
      'LINK_DOWN_DEBOUNCE_TIME' : CliMatcher.IntegerMatcher( 0, 1800000,
         helpdesc='Debounce time for link-down only transitions in milliseconds' ),
   }

   @staticmethod
   def handler( mode, args ):
      linkDebounceTime = args[ 'LINK_DEBOUNCE_TIME' ]
      linkDownDebounceOpt = args.get( 'LINK_DOWN_DEBOUNCE_TIME' )

      # Use a seperate link down debounce time only if it has been specified
      if linkDownDebounceOpt is None:
         linkDownDebounceOpt = linkDebounceTime

      # Set the link-up and link-down debounce times for the interface
      mode.intf.setLinkDebounceTimes( linkDebounceTime, linkDownDebounceOpt )
      Tracing.trace0( "Set", mode.intf.name, "link-up debounce time to",
                      str( linkDebounceTime ), "ms and link-down debounce time to ",
                      str( linkDownDebounceOpt ), "ms" )

   @staticmethod
   def noOrDefaultHandler( mode, args ):
      # Disable debouncing by setting both link-up and
      # link-down debounce times to zero
      mode.intf.setLinkDebounceTimes( 0, 0 )
      Tracing.trace0( "No Set " + mode.intf.name + " link debounce time" )

EthIntfModelet.addCommandClass( LinkDebounceCmd )

#-------------------------------------------------------------------------------
# The "speed" commands:
#   speed { 100full }
#   speed forced { 10half | 10full | 100half | 100full | 1000half | 1000full }
#   no speed [ forced [ 10half | 10full | 100half | 100full | 1000half | 1000full ] ]
#
#   speed auto
#   no speed auto
#
# BUG96410 : Deprecate the "speed forced" command in favor of "speed <speed>"
#            commands. This will be done in stages with 100Full being the first
#            "speed <speed>" command.
#
# BUG282: allow users to optionally specify the capabilities that are
#         advertised during autoneg, i.e., implement
#
#   speed auto [ 10half, 10full, 100half, 100full, 1000half, 1000full, 10000full ]
#   no speed auto [ 10half, 10full, 100half, 100full, 1000half, 1000full, 10000full ]
#-------------------------------------------------------------------------------
matcherSpeed = CliMatcher.KeywordMatcher( 'speed',
      helpdesc='Configure autoneg and speed/duplex/flowcontrol' )
matcherAuto = CliMatcher.KeywordMatcher( 'auto',
      helpdesc='Enable autoneg for speed, duplex, and flowcontrol' )

def checkChangeCancelHooks( mode, intfList, linkMode, advertisedModes ):
   ''' Execute all the registered hooks for the link mode change and confirm that
       the change can be made. If any of the hooks return true then the user aborted
       the change and the request has to be ignored.
   '''
   if mode.session.commandConfirmation():
      for hook in canPromptForAbortHook.extensions():
         if hook( mode, intfList, linkMode, advertisedModes ):
            Tracing.trace0( "LinkMode change for " +
                            ','.join( intf.name for intf in intfList ) +
                            " cancelled by " + str( hook ) )
            # One of the registered hooks returned true so the linkmode change
            # is cancelled
            return True

   # We are good to go with the link mode change
   return False

def checkChangeCancelOnSfp28( mode, intfList, linkMode, advertisedModes ):
   promptTemplate = "Changing the speed setting on master interface {0} may cause " \
                    "slave interfaces in the speed group to flap."
   warningPrompt = ""
   if mode.session.commandConfirmation():
      masterIntfs = []
      for intf in intfList:
         groupStatus = intf.speedGroupStatus()
         if( groupStatus and
             intf.name == groupStatus.masterIntf and
             not groupStatus.setting and
             linkMode != intf.config().linkModeLocal and
             advertisedModes != intf.config().advertisedModesLocal ):
            masterIntfs.append( groupStatus.masterIntf )
      if masterIntfs:
         masterIntfString = IntfRange.intfListToCanonical( masterIntfs,
                                                           noHoleRange=True )[ 0 ]
         warningPrompt = promptTemplate.format( masterIntfString )

   if warningPrompt:
      mode.addWarning( warningPrompt )
      promptText = "Do you wish to proceed with this command? [y/N]"
      ans = BasicCliUtil.confirm( mode, promptText, answerForReturn=False )

      # See if user cancelled the command.
      if not ans:
         intfString = IntfRange.intfListToCanonical( intfList,
                                                     noHoleRange=True )[ 0 ]
         abortMsg = "Command aborted by user for " + intfString
         mode.addMessage( abortMsg )
         return True

   return False

noSpeedChangeHook = CliExtensions.CliHook()

def configureLinkMode( mode, linkMode, advertisedModes=None ):
   # First run through the registered hooks and confirm that we can go this linkMode
   if checkChangeCancelHooks( mode, [ mode.intf ], linkMode, advertisedModes ):
      return
   if checkChangeCancelOnSfp28( mode, [ mode.intf ], linkMode, advertisedModes ):
      return

   mode.intf.setLinkMode( linkMode, advertisedModes )
   Tracing.trace0( "Set " + mode.intf.name + " speed/duplex to " +
                   str( linkMode ) + " ('None' means auto)" )

def setForcedLinkMode( mode, args ):
   speedTypeDict = speedForceLinkMode if 'forced' in args else speedLinkMode
   configureLinkMode( mode, speedTypeDict[ args[ 'SPEED_TYPE' ] ] )

def setAutoLinkMode( mode, args ):
   advertisedModesOpt=speedAutoLinkMode.get( args.get( 'SPEED_TYPE' ) )
   advertisedModes = Tac.Value( 'Interface::EthLinkModeSet' )

   if not advertisedModesOpt:
      # Turn on Autoneg and advertise at all possible speeds
      configureLinkMode( mode, linkMode='linkModeAutoneg' )
   elif advertisedModesOpt == 'advertise10MbpsHalf':
      advertisedModes.mode10MbpsHalf = True
      configureLinkMode( mode, linkMode='linkModeAutoneg',
                         advertisedModes=advertisedModes )
   elif advertisedModesOpt == 'advertise10MbpsFull':
      advertisedModes.mode10MbpsFull = True
      configureLinkMode( mode, linkMode='linkModeAutoneg',
                         advertisedModes=advertisedModes )
   elif advertisedModesOpt == 'advertise100MbpsHalf':
      # Turn on Autoneg and advertise only 100MHalf
      advertisedModes.mode100MbpsHalf = True
      configureLinkMode( mode, linkMode='linkModeAutoneg',
                         advertisedModes=advertisedModes )
   elif advertisedModesOpt == 'advertise100MbpsFull':
      # Turn on Autoneg and advertise only 100MFull
      advertisedModes.mode100MbpsFull = True
      configureLinkMode( mode, linkMode='linkModeAutoneg',
                         advertisedModes=advertisedModes )
   elif advertisedModesOpt == 'advertise1GbpsFull':
      # Turn on Autoneg and advertise only 1GFull
      advertisedModes.mode1GbpsFull = True
      configureLinkMode( mode, linkMode='linkModeAutoneg',
                          advertisedModes=advertisedModes )
   elif advertisedModesOpt == 'advertise2p5GbpsFull':
      # Turn on Autoneg and advertise only 2.5GFull
      advertisedModes.mode2p5GbpsFull = True
      configureLinkMode( mode, linkMode='linkModeAutoneg',
                         advertisedModes=advertisedModes )
   elif advertisedModesOpt == 'advertise5GbpsFull':
      # Turn on Autoneg and advertise only 5GFull
      advertisedModes.mode5GbpsFull = True
      configureLinkMode( mode, linkMode='linkModeAutoneg',
                         advertisedModes=advertisedModes )
   elif advertisedModesOpt == 'advertise10GbpsFull':
      # Turn on Autoneg and advertise only 10GFull
      advertisedModes.mode10GbpsFull = True
      configureLinkMode( mode, linkMode='linkModeAutoneg',
                          advertisedModes=advertisedModes )
   elif advertisedModesOpt == 'advertise25GbpsFull':
      # Turn on Autoneg and advertise only 25GFull
      advertisedModes.mode25GbpsFull = True
      configureLinkMode( mode, linkMode='linkModeAutoneg',
                          advertisedModes=advertisedModes )
   elif advertisedModesOpt == 'advertise40GbpsFull':
      # We should to get rid of linkModeAuto40GbpsFull and use advertisedModes to
      # turn on autoneg and advertise 40GFull. BUG97475 has more details.
      advertisedModes.mode40GbpsFull = True
      configureLinkMode( mode, linkMode='linkModeAuto40GbpsFull',
                         advertisedModes=advertisedModes )
   elif advertisedModesOpt == 'advertise50GbpsFull':
      # Turn on Autoneg and advertise only 50GFull
      advertisedModes.mode50GbpsFull = True
      configureLinkMode( mode, linkMode='linkModeAutoneg',
                          advertisedModes=advertisedModes )
   elif advertisedModesOpt == 'advertise50GbpsFull1Lane':
      # Turn on Autoneg and advertise only 50G-1
      advertisedModes.mode50GbpsFull1Lane = True
      configureLinkMode( mode, linkMode='linkModeAutoneg',
                          advertisedModes=advertisedModes )
   elif advertisedModesOpt == 'advertise100GbpsFull':
      # Turn on Autoneg and advertise only 100GFull
      advertisedModes.mode100GbpsFull = True
      configureLinkMode( mode, linkMode='linkModeAutoneg',
                          advertisedModes=advertisedModes )
   elif advertisedModesOpt == 'advertise100GbpsFull2Lane':
      # Turn on Autoneg and advertise only 100G-2
      advertisedModes.mode100GbpsFull2Lane = True
      configureLinkMode( mode, linkMode='linkModeAutoneg',
                          advertisedModes=advertisedModes )
   elif advertisedModesOpt == 'advertise200GbpsFull4Lane':
      # Turn on Autoneg and advertise only 200G-4
      advertisedModes.mode200GbpsFull4Lane = True
      configureLinkMode( mode, linkMode='linkModeAutoneg',
                          advertisedModes=advertisedModes )
   elif advertisedModesOpt == 'advertise400GbpsFull8Lane':
      # Turn on Autoneg and advertise only 400G-8
      advertisedModes.mode400GbpsFull8Lane = True
      configureLinkMode( mode, linkMode='linkModeAutoneg',
                          advertisedModes=advertisedModes )
   else:
      mode.addError( "Cannot advertise the given speed and duplex" )

def noSpeed( mode, args ):
   configureLinkMode( mode, linkMode='linkModeUnknown' )
   Tracing.trace0( "Set " + mode.intf.name + " speed to default" )

#--------
# Implement the speed <speed> commands.
#--------

speedLinkMode = {
   '10half': 'linkMode10MbpsHalf',
   '10mhalf': 'linkMode10MbpsHalf',
   '10full': 'linkMode10MbpsFull',
   '10mfull': 'linkMode10MbpsFull',
   '100half': 'linkMode100MbpsHalf',
   '100mhalf': 'linkMode100MbpsHalf',
   '100full': 'linkMode100MbpsFull',
   '100mfull': 'linkMode100MbpsFull',
   '1g': 'linkModeForced1GbpsFull',
   '10g': 'linkModeForced10GbpsFull',
   '25g': 'linkModeForced25GbpsFull',
   '40g': 'linkModeForced40GbpsFull',
   '50g': 'linkModeForced50GbpsFull',
   '50g-1': 'linkModeForced50GbpsFull1Lane',
   '50g-2': 'linkModeForced50GbpsFull',
   '100g': 'linkModeForced100GbpsFull',
   '100g-2': 'linkModeForced100GbpsFull2Lane',
   '100g-4': 'linkModeForced100GbpsFull',
   '200g': 'linkModeForced200GbpsFull4Lane',
   '200g-4': 'linkModeForced200GbpsFull4Lane',
   '400g': 'linkModeForced400GbpsFull8Lane',
   '400g-8': 'linkModeForced400GbpsFull8Lane',
   'linkModeUnknown': 'linkModeUnknown',
}

speedForceLinkMode = {
   '10half': 'linkModeForced10MbpsHalf',
   '10full': 'linkModeForced10MbpsFull',
   '100half': 'linkModeForced100MbpsHalf',
   '100full': 'linkModeForced100MbpsFull',
   '1000half': 'linkModeForced1GbpsHalf',
   '1000full': 'linkModeForced1GbpsFull',
   '10000full': 'linkModeForced10GbpsFull',
   '25gfull': 'linkModeForced25GbpsFull',
   '40gfull': 'linkModeForced40GbpsFull',
   '50gfull': 'linkModeForced50GbpsFull',
   '100gfull': 'linkModeForced100GbpsFull',
   '10mhalf': 'linkModeForced10MbpsHalf',
   '10mfull': 'linkModeForced10MbpsFull',
   '100mhalf': 'linkModeForced100MbpsHalf',
   '100mfull': 'linkModeForced100MbpsFull',
   '1ghalf': 'linkModeForced1GbpsHalf',
   '1gfull': 'linkModeForced1GbpsFull',
   '10gfull': 'linkModeForced10GbpsFull',
   '25000full': 'linkModeForced25GbpsFull',
   '40000full': 'linkModeForced40GbpsFull',
   '50000full': 'linkModeForced50GbpsFull',
   '100000full': 'linkModeForced100GbpsFull',
   'linkModeUnknown': 'linkModeUnknown',
}

speedAutoLinkMode = {
   '10000full': 'advertise10GbpsFull',
   '1000full': 'advertise1GbpsFull',
   '100full': 'advertise100MbpsFull',
   '100g-2': 'advertise100GbpsFull2Lane',
   '100g-4': 'advertise100GbpsFull',
   '100gfull': 'advertise100GbpsFull',
   '100half': 'advertise100MbpsHalf',
   '10full': 'advertise10MbpsFull' ,
   '10gfull': 'advertise10GbpsFull',
   '10half': 'advertise10MbpsHalf',
   '1gfull': 'advertise1GbpsFull',
   '2.5gfull': 'advertise2p5GbpsFull',
   '200g-4': 'advertise200GbpsFull4Lane',
   '25gfull': 'advertise25GbpsFull',
   '400g-8': 'advertise400GbpsFull8Lane',
   '40gfull': 'advertise40GbpsFull',
   '50g-1': 'advertise50GbpsFull1Lane',
   '50g-2': 'advertise50GbpsFull',
   '50gfull': 'advertise50GbpsFull',
   '5gfull': 'advertise5GbpsFull',
   '10mhalf': 'advertise10MbpsHalf',
   '10mfull': 'advertise10MbpsFull',
   '100mhalf': 'advertise100MbpsHalf',
   '100mfull': 'advertise100MbpsFull',
   '25000full': 'advertise25GbpsFull',
   '40000full': 'advertise40GbpsFull',
   '50000full': 'advertise50GbpsFull',
   '100000full': 'advertise100GbpsFull',
}

class speedForceExpression( CliCommand.CliExpression ):
   expression = 'SPEED_FORCE_NODE | SPEED_FORCE_HIDDEN_NODE'
   data = {
      'SPEED_FORCE_NODE': CliCommand.Node( 
         matcher=CliMatcher.EnumMatcher( {
            '10000full': 'Disable autoneg and force 10 Gbps/full duplex operation',
            '1000half': 'Disable autoneg and force 1 Gbps/half duplex operation',
            '1000full': 'Disable autoneg and force 1 Gbps/full duplex operation',
            '100full': 'Disable autoneg and force 100 Mbps/full duplex operation',
            '100gfull': 'Disable autoneg and force 100 Gbps/full duplex operation',
            '100half': 'Disable autoneg and force 100 Mbps/half duplex operation',
            '10full': 'Disable autoneg and force 10 Mbps/full duplex operation',
            '10half': 'Disable autoneg and force 10 Mbps/half duplex operation',
            '25gfull': 'Disable autoneg and force 25 Gbps/full duplex operation',
            '40gfull': 'Disable autoneg and force 40 Gbps/full duplex operation',
            '50gfull': 'Disable autoneg and force 50 Gbps/full duplex operation',
       } ),
         alias='SPEED_TYPE' ),
      'SPEED_FORCE_HIDDEN_NODE': CliCommand.Node( 
         matcher=CliMatcher.EnumMatcher( {
            '10mhalf': 'Disable autoneg and force 10 Mbps/half duplex operation',
            '10mfull': 'Disable autoneg and force 10 Mbps/full duplex operation',
            '100mhalf': 'Disable autoneg and force 100 Mbps/half duplex operation',
            '100mfull': 'Disable autoneg and force 100 Mbps/full duplex operation',
            '1ghalf': 'Disable autoneg and force 1 Gbps/half duplex operation',
            '1gfull': 'Disable autoneg and force 1 Gbps/full duplex operation',
            '10gfull': 'Disable autoneg and force 10 Gbps/full duplex operation',
            '25000full': 'Disable autoneg and force 25 Gbps/full duplex operation',
            '40000full': 'Disable autoneg and force 40 Gbps/full duplex operation',
            '50000full': 'Disable autoneg and force 50 Gbps/full duplex operation',
            '100000full': 'Disable autoneg and force 100 Gbps/full duplex operation',
        } ),
         alias='SPEED_TYPE',
         hidden=True ),
}


class SpeedForcedCmd( CliCommand.CliCommandClass ):
   syntax = 'speed forced SPEED_TYPE'
   data = {
      'speed': matcherSpeed,
      'forced': 'Disable autoneg and force speed/duplex/flowcontrol',
      'SPEED_TYPE': speedForceExpression,
   }

   handler = setForcedLinkMode

EthIntfModelet.addCommandClass( SpeedForcedCmd )

class speedAutoExpression( CliCommand.CliExpression ):
   expression = 'SPEEDAUTO_NODE | SPEEDAUTO_HIDDEN_NODE'
   data = {
      'SPEEDAUTO_NODE': CliCommand.Node(
         matcher=CliMatcher.EnumMatcher( {
            '10000full': 'Enable autoneg for 10 Gbps/full duplex operation',
            '1000full': 'Enable autoneg for 1 Gbps/full duplex operation',
            '100full': 'Enable autoneg for 100 Mbps/full duplex operation',
            '100g-2': 'Enable autoneg for 100 Gbps/full duplex operation'
                      ' over 2 lanes',
            '100g-4': 'Enable autoneg for 100 Gbps/full duplex operation'
                      ' over 4 lanes',
            '100gfull': 'Enable autoneg for 100 Gbps/full duplex operation',
            '100half': 'Enable autoneg for 100 Mbps/half duplex operation',
            '10full': 'Enable autoneg for 10 Mbps/full duplex operation' ,
            '10gfull': 'Enable autoneg for 10 Gbps/full duplex operation',
            '10half': 'Enable autoneg for 10 Mbps/half duplex operation',
            '1gfull': 'Enable autoneg for 1 Gbps/full duplex operation',
            '2.5gfull': 'Enable autoneg for 2.5 Gbps/full duplex operation',
            '200g-4': 'Enable autoneg for 200 Gbps/full duplex operation'
                      ' over 4 lanes',
            '25gfull': 'Enable autoneg for 25 Gbps/full duplex operation',
            '400g-8': 'Enable autoneg for 400 Gbps/full duplex operation'
                      ' over 8 lanes',
            '40gfull': 'Enable autoneg for 40 Gbps/full duplex operation',
            '50g-1': 'Enable autoneg for 50 Gbps/full duplex operation'
                     ' over 1 lane',
            '50g-2': 'Enable autoneg for 50 Gbps/full duplex operation'
                     ' over 2 lanes',
            '50gfull': 'Enable autoneg for 50 Gbps/full duplex operation',
            '5gfull': 'Enable autoneg for 5 Gbps/full duplex operation',
       } ),
         alias='SPEED_TYPE' ),
      'SPEEDAUTO_HIDDEN_NODE': CliCommand.Node(
         matcher=CliMatcher.EnumMatcher( {
            '10mhalf': 'Enable autoneg for 10 Mbps/half duplex operation',
            '10mfull': 'Enable autoneg for 10 Mbps/full duplex operation',
            '100mhalf': 'Enable autoneg for 100 Mbps/half duplex operation',
            '100mfull': 'Enable autoneg for 100 Mbps/full duplex operation',
            '25000full': 'Enable autoneg for 25 Gbps/full duplex operation',
            '40000full': 'Enable autoneg for 40 Gbps/full duplex operation',
            '50000full': 'Enable autoneg for 50 Gbps/full duplex operation',
            '100000full': 'Enable autoneg for 100 Gbps/full duplex operation',
        } ),
         alias='SPEED_TYPE',
         hidden=True ),
}

class SpeedAutoCmd( CliCommand.CliCommandClass ):
   syntax = 'speed auto [ SPEED_TYPE ]'
   data = {
      'speed': matcherSpeed,
      'auto': matcherAuto,
      'SPEED_TYPE': speedAutoExpression,
   }

   handler = setAutoLinkMode

EthIntfModelet.addCommandClass( SpeedAutoCmd )

class speedExpression( CliCommand.CliExpression ):
   expression = 'SPEED_NODE | SPEED_HIDDEN_NODE'
   data = {
      'SPEED_NODE': CliCommand.Node(
         matcher=CliMatcher.EnumMatcher( {
            '10half': 'Disable autoneg and force 10 Mbps/half duplex operation',
            '10full': 'Disable autoneg and force 10 Mbps/full duplex operation',
            '100half': 'Disable autoneg and force 100 Mbps/half duplex operation',
            '100full': 'Disable autoneg and force 100 Mbps/full duplex operation',
            '1g':  'Disable autoneg and force 1 Gbps/full duplex operation'
                   ' over 1 lane',
            '10g': 'Disable autoneg and force 10 Gbps/full duplex operation' 
                   ' over 1 lane',
            '25g': 'Disable autoneg and force 25 Gbps/full duplex operation'
                   ' over 1 lane',
            '40g': 'Disable autoneg and force 40 Gbps/full duplex operation'
                   ' over 4 lanes',
            '50g': 'Disable autoneg and force 50 Gbps/full duplex operation'
                   ' over 2 lanes',
            '50g-1': 'Disable autoneg and force 50 Gbps/full duplex operation'
                   ' over 1 lane',
            '50g-2': 'Disable autoneg and force 50 Gbps/full duplex operation'
                   ' over 2 lane',
            '100g': 'Disable autoneg and force 100 Gbps/full duplex operation'
                    ' over 4 or 10 lanes',
            '100g-2': 'Disable autoneg and force 100 Gbps/full duplex operation'
                   ' over 2 lanes',
            '100g-4': 'Disable autoneg and force 100 Gbps/full duplex operation'
                   ' over 4 lanes',
            '200g': 'Disable autoneg and force 200 Gbps/full duplex operation'
                   ' over 4 lanes',
            '200g-4': 'Disable autoneg and force 200 Gbps/full duplex operation'
                   ' over 4 lanes',
            '400g': 'Disable autoneg and force 400 Gbps/full duplex operation'
                   ' over 8 lanes',
            '400g-8': 'Disable autoneg and force 400 Gbps/full duplex operation'
                   ' over 8 lanes',
       } ),
         alias='SPEED_TYPE' ),
      'SPEED_HIDDEN_NODE': CliCommand.Node(
         matcher=CliMatcher.EnumMatcher( {
            '10mhalf': 'Disable autoneg and force 10 Mbps/half duplex operation',
            '10mfull': 'Disable autoneg and force 10 Mbps/full duplex operation',
            '100mhalf': 'Disable autoneg and force 100 Mbps/half duplex operation',
            '100mfull': 'Disable autoneg and force 100 Mbps/full duplex operation',
        } ),
         alias='SPEED_TYPE',
         hidden=True ),
}

class SpeedCmd( CliCommand.CliCommandClass ):
   syntax = 'speed SPEED_TYPE'
   noOrDefaultSyntax = 'speed ...'
   data = {
      'speed': matcherSpeed,
      'SPEED_TYPE': speedExpression
   }

   handler = setForcedLinkMode
   noOrDefaultHandler = noSpeed

EthIntfModelet.addCommandClass( SpeedCmd )

#--------------------------------------------------------------------------------
# speed sfp-1000baset auto 100full
#
# Special case to allow the user to configure whether to put a
# 1000BASE-T SFP into "auto-100/full" mode. Putting the SFP into
# "forced-100/full" mode is done using the "speed forced 100full"
# command.
#
# I would really just like to implement this as part of implementing
# BUG282 (allow specifying the set of capabilities to advertise during
# autoneg), but this is a fairly big project, because it affects the
# platform code (drivers) for all our PHYs and MACs/switch chips.
#--------------------------------------------------------------------------------
class SpeedSfp1000BasetAuto100FullCmd( CliCommand.CliCommandClass ):
   syntax = 'speed sfp-1000baset auto 100full'
   data = {
      'speed' : matcherSpeed,
      'sfp-1000baset' : 'Configure autoneg and speed/duplex on 1000BASE-T SFP',
      'auto' : matcherAuto,
      '100full' : ( 'Advertise 100 Mbps/full duplex as the only capability during '
                    'autoneg' ),
   }

   @staticmethod
   def handler( mode, args ):
      mode.intf.setLinkModeSfp100BaseTAuto100Full()

EthIntfModelet.addCommandClass( SpeedSfp1000BasetAuto100FullCmd )

#--------------------------------------------------------------------------------
# [ no | default ] mac timestamp ( before-fcs | replace-fcs | header )
#--------------------------------------------------------------------------------
def timestampFooterSupportedGuard( mode, token ):
   if intfHwStatus.timestampFooterSupported:
      return None
   return CliParser.guardNotThisPlatform

def timestampHeaderSupportedGuard( mode, token ):
   if intfHwStatus.timestampHeaderSupported:
      return None
   return CliParser.guardNotThisPlatform

class MacTimestampCmd( CliCommand.CliCommandClass ):
   syntax = 'mac timestamp ( before-fcs | replace-fcs | header )'
   noOrDefaultSyntax = 'mac timestamp ...'
   data = {
      'mac' : CliToken.Mac.macMatcherForConfigIf,
      'timestamp' : 'Configure interface timestamp properties',
      'before-fcs' : CliCommand.guardedKeyword( 'before-fcs',
         helpdesc='insert timestamp before fcs field',
         guard=timestampFooterSupportedGuard ),
      'replace-fcs' : CliCommand.guardedKeyword( 'replace-fcs',
         helpdesc='replace fcs field with timestamp',
         guard=timestampFooterSupportedGuard ),
      'header' : CliCommand.guardedKeyword( 'header',
         helpdesc='insert timestamp in Ethernet header',
         guard=timestampHeaderSupportedGuard ),
   }

   @staticmethod
   def handler( mode, args ):
      timestampMode = None
      if 'before-fcs' in args:
         timestampMode = 'timestampModeBeforeFcs'
      elif 'replace-fcs' in args:
         timestampMode = 'timestampModeReplaceFcs'
      else:
         timestampMode = 'timestampModeHeader'

      if timestampMode == 'timestampModeHeader':
         mode.addWarning(
            "Time-stamping is only available on tool interfaces in "
            "Tap Aggregation mode" )
      mode.intf.setTimestampMode( timestampMode )

   @staticmethod
   def noOrDefaultHandler( mode, args ):
      mode.intf.setTimestampMode( 'timestampModeDisabled' )

DataplaneIntfModelet.addCommandClass( MacTimestampCmd )

#--------------------------------------------------------------------------------
# [ no | default ] flowcontrol DIRECTION VALUE
#--------------------------------------------------------------------------------
# A canSetFlowcontrolHook accepts three arguments: the CLI mode, the
# name of the physical interface, and the requested flowcontrol
# value. If any of the hooks return false, the request is ignored.
canSetFlowcontrolHook = CliExtensions.CliHook()

class FlowcontrolCmd( CliCommand.CliCommandClass ):
   syntax = 'flowcontrol DIRECTION VALUE'
   noOrDefaultSyntax = 'flowcontrol DIRECTION ...'
   data = {
      'flowcontrol' : 'Configure flow operation',
      'DIRECTION' : CliMatcher.EnumMatcher( {
         'send' : 'Configure transmit flow operation',
         'receive' : 'Configure receiving flow operation',
      } ),
      'VALUE' : CliMatcher.EnumMatcher( {
         'on' : 'Require flow-control capable link partner',
         'off' : 'Forbid flow-control capable link partner',
         'desired' : 'Allow but do not require flow-control capable link partner',
      } ),
   }

   @staticmethod
   def handler( mode, args ):
      direction = args[ 'DIRECTION' ]
      flowcontrolVal = 'flowControlConfig' + args[ 'VALUE' ].capitalize()
      for hook in canSetFlowcontrolHook.extensions():
         if not hook( mode, mode.intf.name, direction, flowcontrolVal ):
            Tracing.trace0( "Flowcontrol for " + mode.intf.name +
                            " cancelled by " + str(hook) )
            return
      mode.intf.setFlowcontrol( direction, flowcontrolVal )
      Tracing.trace0( "Set " + mode.intf.name + " " + direction +
                      " flowcontrol to " + flowcontrolVal )

   @staticmethod
   def noOrDefaultHandler( mode, args ):
      direction = args[ 'DIRECTION' ]
      mode.intf.setFlowcontrol( direction, 'flowControlConfigUnknown' )
      Tracing.trace0( "Set " + mode.intf.name + " " + direction +
                      " flowcontrol to default." )

EthIntfModelet.addCommandClass( FlowcontrolCmd )

#-------------------------------------------------------------------------------
# The "[no|default] logging event congestion-drops interval <val>" command,
# in "config" mode.
#-------------------------------------------------------------------------------
LogDirectionType = Tac.Type( "Interface::CongestionDropsLogDirection" )
def congestionDropsGuard( mode, token ):
   if intfHwStatus.dropsLogDirectionSupport == \
          LogDirectionType.directionNone:
      return CliParser.guardNotThisPlatform
   else:
      return None

congestionDropsKw = CliCommand.guardedKeyword(
   'congestion-drops',
   'Drops due to congestion',
   congestionDropsGuard )

def enableGlobalCongestionDropsLogging( mode, logInterval ):
   assert logInterval != 0
   ethPhyIntfDefaultConfigDir.congestionDropsLogInterval = logInterval

def disableGlobalCongestionDropsLogging( mode ):
   ethPhyIntfDefaultConfigDir.congestionDropsLogInterval = 0

class LoggingEventCongestionDropsGlobal( LoggingEventGlobalCmd ):
   syntax = 'logging event congestion-drops interval INTERVAL'
   noOrDefaultSyntax = 'logging event congestion-drops ...'
   data = {
      'congestion-drops' : congestionDropsKw,
      'interval' : 'Logging interval',
      'INTERVAL' : CliMatcher.IntegerMatcher( 1, 65535,
                                             helpdesc='Logging interval in seconds' )
      }
   @staticmethod
   def handler( mode, args ):
      interval = args.get( 'INTERVAL', 0 )
      ethPhyIntfDefaultConfigDir.congestionDropsLogInterval = interval

   noOrDefaultHandler = handler

BasicCli.GlobalConfigMode.addCommandClass( LoggingEventCongestionDropsGlobal )

#-------------------------------------------------------------------------------
#   logging event congestion-drops [ use-global ]
#   no|default logging event congestion-drops
#-------------------------------------------------------------------------------
class LoggingEventCongestionDropsIntf( LoggingEventIntfCmd ):
   syntax = "logging event congestion-drops"
   data = {
      'congestion-drops' : congestionDropsKw,
   }

   @classmethod
   def _enableHandler( cls, mode, args ):
      mode.intf.setCongestionDropsLoggingMode( 'on' )

   @classmethod
   def _disableHandler( cls, mode, args ):
      mode.intf.setCongestionDropsLoggingMode( 'off' )

   @classmethod
   def _useGlobalHandler( cls, mode, args ):
      mode.intf.setCongestionDropsLoggingMode( 'useGlobal' )

EthIntfModelet.addCommandClass( LoggingEventCongestionDropsIntf )

#--------------------------------------------------------------------------------
# [ no | default ] unidirectional LINK_MODE
#--------------------------------------------------------------------------------
# A canSetUnidirectionalLinkModeHook accepts two arguments:
# CLI mode and intfList( could be a single or bunch of interfaces ).
canSetUnidirectionalLinkModeHook = CliExtensions.CliHook()

def uniLinkSupportedGuard( mode, token ):
   if mode.intf.unidirectionalLinkSupported():
      return None
   return CliParser.guardNotThisPlatform

LINK_MODE_ENUM = {
   'send-only' : 'uniLinkModeSendOnly',
   'receive-only' : 'uniLinkModeReceiveOnly',
   'send-receive' : 'uniLinkModeSendReceive'
}

class UnidirectionalCmd( CliCommand.CliCommandClass ):
   syntax = 'unidirectional LINK_MODE'
   noOrDefaultSyntax = 'unidirectional [ LINK_MODE ]'
   data = {
      'unidirectional' : CliCommand.guardedKeyword( 'unidirectional',
         helpdesc='Configure unidirectional link mode',
         guard=uniLinkSupportedGuard ),
      'LINK_MODE' : CliMatcher.EnumMatcher( {
         'send-only' : 'Configure intf send only mode',
         'receive-only' : 'Configure intf receive only mode',
         'send-receive' : 'Configure intf send/recv mode',
      } ),
   }

   @staticmethod
   def handler( mode, args ):
      if 'LINK_MODE' in args:
         unidirectionalLinkMode = LINK_MODE_ENUM[ args[ 'LINK_MODE' ] ]
      else:
         unidirectionalLinkMode = None

      for i in canSetUnidirectionalLinkModeHook.extensions():
         if not mode.maybeIntfRangeCliHook( i, 'canSetUnidirectionalLinkModeHook' ):
            return

      if CliCommand.isNoOrDefaultCmd( args ):
         if not CliCommand.isDefaultCmd( args ) or unidirectionalLinkMode is None:
            unidirectionalLinkMode = 'uniLinkModeDisabled'
      # Set the UnidirectionalLinkMode for this intf
      mode.intf.setUnidirectionalLinkMode( unidirectionalLinkMode )

   noOrDefaultHandler = handler

EthIntfModelet.addCommandClass( UnidirectionalCmd )

class ShowIntfUnidirectional( IntfCli.ShowIntfCommand ):
   syntax = 'show interfaces unidirectional'
   data = dict( unidirectional='Show unidirectional link mode' )
   moduleAtEnd = True

   @staticmethod
   def handler( mode, args ):
      intf = args.get( 'INTF' )
      mod = args.get( 'MOD' )
      intfs = getAllIntfs( mode, intf, mod, EthPhyIntf )
      if not intfs:
         return
      hdrFmt = '%-10.10s  %-12.10s %-12.12s'
      print( hdrFmt % ( 'Port', 'Status', 'UniLinkMode' ) )
      pFormat = '%-10.10s  %-12.10s %-12.12s'
      print( pFormat % ( '-' * 10, '-' * 10, '-' * 12 ) )

      for intf in intfs:
         if intf.unidirectionalLinkSupported():
            print( pFormat % ( intf.shortname,
                               intf.linkStatus(),
                               intf.showUnidirectionalLinkMode() ) )

BasicCli.addShowCommandClass( ShowIntfUnidirectional )

#--------------------------------------------------------------------------------
# [ no | default ] l2 mtu MTU
#--------------------------------------------------------------------------------
def supportsL2Mtu( mode, token ):
   if bridgingHwCapabilities.l2MtuSupported:
      return None
   return CliParser.guardNotThisPlatform

def rangeL2MtuFn( mode ):
   return bridgingHwCapabilities.l2MtuMin, bridgingHwCapabilities.l2MtuMax 

class L2MtuCmd( CliCommand.CliCommandClass ):
   syntax = 'l2 mtu MTU'
   noOrDefaultSyntax = 'l2 mtu ...'
   data = {
      'l2' : 'Configure L2',
      'mtu' : CliCommand.guardedKeyword( 'mtu',
         helpdesc='Set L2 maximum transmission unit in bytes', guard=supportsL2Mtu ),
      'MTU' : CliMatcher.DynamicIntegerMatcher( rangeL2MtuFn,
         helpdesc='L2 MTU (bytes)' ),
   }

   @staticmethod
   def handler( mode, args ):
      mode.intf.setL2Mtu( args[ 'MTU' ] )

   @staticmethod
   def noOrDefaultHandler( mode, args ):
      mode.intf.setL2Mtu( 0 )

EthIntfModelet.addCommandClass( L2MtuCmd )

#--------------------------------------------------------------------------------
# [ no | default ] l2 mru MRU
#--------------------------------------------------------------------------------
def supportsL2Mru( mode, token ):
   if bridgingHwCapabilities.l2MruSupported:
      return None
   return CliParser.guardNotThisPlatform

def rangeL2MruFn( mode ):
   return bridgingHwCapabilities.l2MruMin, bridgingHwCapabilities.l2MruMax

class L2MruCmd( CliCommand.CliCommandClass ):
   syntax = 'l2 mru MRU'
   noOrDefaultSyntax = 'l2 mru ...'
   data = {
      'l2' : 'Configure L2',
      'mru' : CliCommand.guardedKeyword( 'mru',
         helpdesc='Set L2 maximum receive unit in bytes', guard=supportsL2Mru ),
      'MRU' : CliMatcher.DynamicIntegerMatcher( rangeL2MruFn,
         helpdesc='L2 MRU (bytes)' ),
   }

   @staticmethod
   def handler( mode, args ):
      mode.intf.setL2Mru( args[ 'MRU' ] )

   @staticmethod
   def noOrDefaultHandler( mode, args ):
      mode.intf.setL2Mru( 0 )

EthIntfModelet.addCommandClass( L2MruCmd )

#-------------------------------------------------------------------------------
# Associate the DataplaneIntfConfigModelet with the "config-if" mode.
#-------------------------------------------------------------------------------
IntfCli.IntfConfigMode.addModelet( DataplaneIntfModelet )

#-------------------------------------------------------------------------------
# Associate the EthIntfConfigModelet with the "config-if" mode.
#-------------------------------------------------------------------------------
IntfCli.IntfConfigMode.addModelet( EthIntfModelet )

#-------------------------------------------------------------------------------
# The "show flowcontrol [ interface <name> ]" command, in enable mode.
#   show flowcontrol [ interface <name> ]
#   show flowcontrol module <modnum>
#-------------------------------------------------------------------------------
def getAllIntfs( mode, intf, mod, clazz, exposeInactive=False,
                 exposeUnconnected=False ):
   intfs = IntfCli.Intf.getAll( mode, intf, mod, clazz,
                                exposeInactive=exposeInactive,
                                exposeUnconnected=exposeUnconnected )
   if intfs:
      intfs = [ i for i in intfs if i.lookup() ] # See BUG9124
   return intfs

def showInterfacesFlowcontrol( mode, args ):
   intf = args.get( 'INTF' )
   mod = args.get( 'MOD' )
   intfs = getAllIntfs( mode, intf, mod, EthPhyIntf )
   intfsFlowControl = EthIntfModel.InterfacesFlowControl()

   def _convertFcState( state ):
      retState = "unsupported" if state == "Unsupp." else state
      return retState

   if intfs:
      for x in intfs:
         flowControl = EthIntfModel.FlowControlData()
         flowControl.txAdminState = _convertFcState( x.flowcontrolTxAdminState() )
         flowControl.txOperState = _convertFcState( x.flowcontrolTxOperState() )
         flowControl.rxAdminState = _convertFcState( x.flowcontrolRxAdminState() )
         flowControl.rxOperState = _convertFcState( x.flowcontrolRxOperState() )
         flowControl.rxPause = x.flowcontrolRxPause()
         flowControl.txPause = x.flowcontrolTxPause()
         intfsFlowControl.interfaceFlowControls[ x.name ] = flowControl
   return intfsFlowControl

class ShowFlowcontrol( ShowCommand.ShowCliCommandClass ):
   syntax = "show flowcontrol [ ( interface INTF ) | MOD ]"
   data = dict( flowcontrol='Details on flow control',
                interface='Show flowcontrol information for interfaces',
                INTF=IntfCli.Intf.rangeMatcher,
                MOD=ModuleIntfCli.ModuleExpressionFactory() )
   cliModel = EthIntfModel.InterfacesFlowControl
   handler = showInterfacesFlowcontrol

BasicCli.addShowCommandClass( ShowFlowcontrol )

#-------------------------------------------------------------------------------
# The "show interfaces [<name>] flow-control" command, in enable mode.
#   show interfaces [<name>] flow-control
#   show interfaces module <modnum> flow-control
#   show interfaces flow-control module <modnum>
#  legacy:
#   show interfaces [<name>] flowcontrol
#   show interfaces module <modnum> flowcontrol
#   show interfaces flowcontrol module <modnum>
#-------------------------------------------------------------------------------
flowControlDeprecated = CliCommand.Node(
   CliMatcher.KeywordMatcher( 'flowcontrol',
                              helpdesc='Details on flow control' ),
   deprecatedByCmd='show interface flow-control' )

class ShowIntfFlowcontrol( IntfCli.ShowIntfCommand ):
   syntax = 'show interfaces flow-control | flowcontrol'
   data = { 'flow-control' : 'Show interface flow-control information',
            'flowcontrol' : flowControlDeprecated }
   handler = showInterfacesFlowcontrol
   moduleAtEnd = True
   cliModel = EthIntfModel.InterfacesFlowControl

BasicCli.addShowCommandClass( ShowIntfFlowcontrol )

#-------------------------------------------------------------------------------
# The "show interfaces [<name>] debounce" command, in enable mode.
#   show interfaces [<name>] debounce
#   show interfaces debounce module <modnum>
#   show interfaces module <modnum> debounce
#-------------------------------------------------------------------------------
def showInterfacesDebounceTime( mode, args ):
   intf = args.get( 'INTF' )
   mod = args.get( 'MOD' )
   result = EthIntfModel.InterfacesDebounceTime()

   intfs = getAllIntfs( mode, intf, mod, EthPhyIntf )
   if not intfs:
      return result

   for x in intfs:
      debounceInfo = EthIntfModel.InterfaceDebounceTime()
      debounceInfo.linkUpTime = x.linkUpDebouncePeriod()
      debounceInfo.linkDownTime = x.linkDownDebouncePeriod()

      result.interfaces[ x.name ] = debounceInfo

   return result

class ShowIntfDebounce( IntfCli.ShowIntfCommand ):
   syntax = 'show interfaces debounce'
   data = dict( debounce='Show interface link-debounce information' )
   handler = showInterfacesDebounceTime
   moduleAtEnd = True
   cliModel = EthIntfModel.InterfacesDebounceTime

BasicCli.addShowCommandClass( ShowIntfDebounce )

#-------------------------------------------------------------------------------
# The "show interfaces [<name>] l2 mtu" command
#   show interfaces [<name>] l2 mtu
#   show interfaces l2 mtu
#-------------------------------------------------------------------------------
def showInterfacesL2Mtu( mode, args ):
   intf = args.get( 'INTF' )
   mod = args.get( 'MOD' )
   # Retrieve all interfaces that support L2 mtu
   intfs = getAllIntfs( mode, intf, mod, l2MtuValidIntfType )
   intfL2Mtu = EthIntfModel.InterfacesL2Mtu()
   if not intfs:
      return intfL2Mtu

   for intf in intfs:
      l2Mtu = EthIntfModel.L2Mtu()
      l2Mtu.l2MtuCfg = intf.config().l2Mtu
      l2Mtu.l2MtuStatus = intf.status().l2Mtu
      intfL2Mtu.interfaceL2Mtus[ intf.name ] = l2Mtu
   return intfL2Mtu

class ShowIntfL2Mtu( IntfCli.ShowIntfCommand ):
   syntax = 'show interfaces l2 mtu'
   data = dict( l2='Show interface l2 information',
                mtu='Show interface l2 MTU information' )
   handler = showInterfacesL2Mtu
   moduleAtEnd = True
   cliModel = EthIntfModel.InterfacesL2Mtu

BasicCli.addShowCommandClass( ShowIntfL2Mtu )

#-------------------------------------------------------------------------------
# The "show interfaces [<name>] counters bins" command.
#   show interfaces [<name>] counters bins
#   show interfaces module <modnum> counters bins
#-------------------------------------------------------------------------------
def showInterfacesBinsCounters( mode, args ):
   intf = args.get( 'INTF' )
   mod = args.get( 'MOD' )

   def stat( attr, counter, ckpt ):
      sysdbValue = getattr( counter.ethStatistics, attr, 0 ) or 0

      if ckpt:
         checkpointValue = getattr( ckpt.ethStatistics, attr, 0 ) or 0
      else:
         checkpointValue = 0
      return sysdbValue - checkpointValue

   def getCountersValues( x, direction ):

      counter = x.counter()

      if getattr( counter, direction + '64OctetFrames', None ) is None:
         return None
         # Use the absence of value for 64OctetFrames as an indication
         # that this interface does not support bins

      ckpt = x.getLatestCounterCheckpoint()

      binsCounters  = EthIntfModel.InterfaceBinsCounters()

      binsCounters.frames64Octet = stat( direction + '64OctetFrames', counter, ckpt )
      binsCounters.frames65To127Octet = stat( direction + '65To127OctetFrames',
                              counter, ckpt )
      binsCounters.frames128To255Octet = stat( direction + '128To255OctetFrames',
                              counter, ckpt )
      binsCounters.frames256To511Octet = stat( direction + '256To511OctetFrames',
                              counter, ckpt )
      binsCounters.frames512To1023Octet = stat( direction + '512To1023OctetFrames',
                              counter, ckpt )
      binsCounters.frames1024To1522Octet = stat( direction +'1024To1522OctetFrames',
                              counter, ckpt )
      binsCounters.frames1523ToMaxOctet = stat( direction + '1523ToMaxOctetFrames',
                              counter, ckpt )
      return binsCounters

   result = EthIntfModel.InterfacesBinsCounters()

   intfs = getAllIntfs( mode, intf, mod, EthPhyIntf )
   if not intfs:
      return result

   for x in intfs:

      inOutBinsCounters = EthIntfModel.InterfaceInOutBinsCounters()

      inOutBinsCounters.inBinsCounters = getCountersValues( x, 'in' )

      inOutBinsCounters.outBinsCounters = getCountersValues( x, 'out' )

      result.interfaces[ x.name ] = inOutBinsCounters

   return result

class ShowIntfCountersBins( IntfCli.ShowIntfCommand ):
   syntax = 'show interfaces counters bins'
   data = dict( counters=IntfCli.countersKw,
                bins='Packet length bin counters' )
   handler = showInterfacesBinsCounters
   moduleAtEnd = True
   cliModel = EthIntfModel.InterfacesBinsCounters

BasicCli.addShowCommandClass( ShowIntfCountersBins )

#-------------------------------------------------------------------------------
# The "show interfaces hardware" commands in enable mode:
#     show interfaces [<name>] hardware
#     show interfaces hardware module <modnum>
#     show interfaces module <modnum> hardware
#  legacy:
#     show interfaces [<name>] capabilities
#     show interfaces capabilities module <modnum>
#     show interfaces module <modnum> capabilities
#
# The "default" token, in enable mode, shows capabilities of a port regardless
# of any installed transceiver. The output captures the shared capabilities of
# every asic/phy leading up to the front panel.
#     show interfaces [<name>] hardware default
#     show interfaces hardware default module <modnum>
#     show interfaces module <modnum> hardware default
#  legacy:
#     show interfaces [<name>] capabilities default
#     show interfaces capabilities default module <modnum>
#     show interfaces module <modnum> capabilities default
#-------------------------------------------------------------------------------
# Pre-compute the format string so that things are properly indented
showIntCapabilitiesLbls = [ 'Model:', 'Type:', 'Speed/Duplex:', 'Flowcontrol:' ]

phyModulationLbls = [ 'Modulation:' ]

errorCorrectionEncodingToLbls = { 'reed-solomon' : '  Reed-Solomon:',
                                  'fire-code' : '  Fire-code:   ' }
errorCorrectionLbls = errorCorrectionEncodingToLbls.values()
# Size the output to accomodate fec values even if they don't apply to some
# interfaces.
maxShowIntCapLabelLen = max( map( len, showIntCapabilitiesLbls +
                                  errorCorrectionLbls + phyModulationLbls ) )
# This label sits on a line by itself so not included in the above max calc.
errorCorrectionHeading = 'Error Correction:'

showIntCapFmt = "  %-" + ( "%d" % maxShowIntCapLabelLen ) + "s %s"

def showInterfacesCapabilities( mode, args ):
   intf = args.get( 'INTF' )
   mod = args.get( 'MOD' )
   intfCaps = EthIntfModel.InterfacesCapabilities()
   intfs = getAllIntfs( mode, intf, mod, EthPhyIntf )
   if not intfs:
      return intfCaps

   allPhyCoherentStatus = {}
   # Get all coherent status' at once.
   if 'Modulation' in interfaceCapabilityFns:
      allPhyCoherentStatus = interfaceCapabilityFns[ 'Modulation' ].getAll()

   # Get all default capabilities at once.
   capsInfos = EthPhyDefaultCaps.getIntfCapsInfo( intfs )

   for x in intfs:
      icdd = EthIntfModel.InterfaceCapabilitiesData()

      # Model:
      icdd.modelIs( x.modelStr( mode ) )

      # Type:
      xcvrPresent = x.intfXcvrStatus().xcvrPresence == 'xcvrPresent'
      icdd.typeIs( xcvrPresent, x.xcvrTypeStr() )

      # Speed/Duplex:
      icdd.speedLanesDuplexIs( x.linkModeCapabilitiesToList() )

      groupStatus = x.speedGroupStatus()
      if groupStatus:
         icdd.speedGroupIs( groupStatus.name, groupStatus.memberIntf,
                            groupStatus.masterIntf )

      # Flowcontrol:
      icdd.flowControlIs( x.status().rxFlowcontrolCapabilities,
                          x.status().autonegCapabilities.rxPause,
                          rx=True )
      icdd.flowControlIs( x.status().txFlowcontrolCapabilities,
                          x.status().autonegCapabilities.txPause,
                          rx=False )

      if 'Modulation' in interfaceCapabilityFns:
         modStr = interfaceCapabilityFns[ 'Modulation' ].getMods(
                  allPhyCoherentStatus, x.name,
                  phyModulationLbls[ 0 ], sysCap=False )
         icdd.modulationIs( modStr )

      # Error Correction:
      # Client FEC
      if xcvrPresent and x.errorCorrectionSupported():
         for encoding, _ in errorCorrectionEncodingToLbls.iteritems():
            if x.errorCorrectionEncodingSupported( encoding ):
               caps = x.errorCorrectionCapabilitiesToList( encoding )
               icdd.errorCorrectionIs( caps )
      # Coherent FEC
      if x.coherentErrorCorrectionSupported():
         if 'coherent' in errorCorrectionShCapFns:
            fecDict = \
               errorCorrectionShCapFns[ 'coherent' ].getFecs( allPhyCoherentStatus,
                                                              x,
                                                              False )
         icdd.coherentErrorCorrectionIs( fecDict )

      # Autoneg info is stored in CAPI model but will not be printed by 'show 
      # interface et* capabilities'
      for ( epif, epdc ) in capsInfos:
         if x.name == epif.name:
            autonegMode = epif.autonegCapabilitiesMode()
            autonegCaps = epif.autonegCapabilitiesToList()
            # Clause 28
            if autonegMode == 'anegModeClause28':
               icdd.clause28Is( autonegCaps )

            # Clause 37
            if autonegMode == 'anegModeClause37':
               icdd.clause37Is( autonegCaps )
      
            # Clause 73 has both IEEE and 25G Consortium standards
            if autonegMode == 'anegModeClause73':
               icdd.clause73Is(
                  set( autonegCaps ).intersection(
                       epdc.anegCl73ModeCapabilitiesToList( 'ieee' ) ),
                  set( autonegCaps ).intersection(
                       epdc.anegCl73ModeCapabilitiesToList( 'consortium' ) ) )

      intfCaps.interfaces[ x.name ] = icdd
   return intfCaps


capabilitiesDeprecated = CliCommand.Node(
   CliMatcher.KeywordMatcher( 'capabilities',
                              helpdesc='Show interface capabilities information' ),
   deprecatedByCmd='show interface hardware' )

hardwareKw = CliMatcher.KeywordMatcher(
   'hardware',
   helpdesc='Show interface hardware capabilities information' )

class ShowIntfHardware( IntfCli.ShowIntfCommand ):
   syntax = 'show interfaces capabilities | hardware'
   data = dict( capabilities=capabilitiesDeprecated,
                hardware=hardwareKw )
   handler = showInterfacesCapabilities
   moduleAtEnd = True
   cliModel= EthIntfModel.InterfacesCapabilities

BasicCli.addShowCommandClass( ShowIntfHardware )

def showInterfacesCapabilitiesDefault( mode, args ):
   intf = args.get( 'INTF' )
   mod = args.get( 'MOD' )
   capsDefaults = EthIntfModel.InterfacesCapabilitiesDefault()
   intfs = getAllIntfs( mode, intf, mod, EthPhyIntf )
   if not intfs:
      return capsDefaults

   capsInfos = EthPhyDefaultCaps.getIntfCapsInfo( intfs )

   allPhyCoherentStatus = {}
   # Get all coherent status' at once.
   if 'Modulation' in interfaceCapabilityFns:
      allPhyCoherentStatus = interfaceCapabilityFns[ 'Modulation' ].getAll()

   for epis, epdc in capsInfos:
      icdd = EthIntfModel.InterfaceCapabilitiesDefaultData()
      
      # Model:
      icdd.modelIs( epis.modelStr( mode ) )

      # Type:
      icdd.typeIs( epis.intfXcvrStatus().xcvrPresence == 'xcvrPresent',
                   epis.xcvrTypeStr() )

      # Speed/Duplex:
      icdd.speedLanesDuplexIs( epdc.linkModeCapabilitiesToList() )
      
      # Speed Group:
      groupStatus = epis.speedGroupStatus()
      if groupStatus:
         icdd.speedGroupIs( groupStatus.name, groupStatus.memberIntf,
                            groupStatus.masterIntf )

      # Flowcontrol:
      icdd.flowControlIs( epis.status().rxFlowcontrolCapabilities,
                          epis.status().autonegCapabilities.rxPause,
                          rx=True )
      icdd.flowControlIs( epis.status().txFlowcontrolCapabilities,
                          epis.status().autonegCapabilities.txPause,
                          rx=False )

      # Autoneg labels
      # Clause 28
      icdd.clause28Is( epdc.autonegCapabilitiesToList( 'anegModeClause28' ) )

      # Clause 37
      icdd.clause37Is( epdc.autonegCapabilitiesToList( 'anegModeClause37' ) )
      
      # Clause 73 has both IEEE and 25G Consortium standards
      icdd.clause73Is( epdc.anegCl73ModeCapabilitiesToList( 'ieee' ),
                       epdc.anegCl73ModeCapabilitiesToList( 'consortium' ) )

      if 'Modulation' in interfaceCapabilityFns:
         modStr = interfaceCapabilityFns[ 'Modulation' ].getMods(
                  allPhyCoherentStatus, epis.name,
                  phyModulationLbls[ 0 ], sysCap=True )
         icdd.modulationIs( modStr )

      # Error Correction:
      # Client FEC
      for encoding in epdc.errorCorrectionEncodings():
         fecLinkModes = epdc.errorCorrectionLinkModes( encoding )
         icdd.errorCorrectionIs( fecLinkModes )
      # Coherent FEC
      if epis.coherentErrorCorrectionSupported():
         if 'coherent' in errorCorrectionShCapFns:
            fecDict = \
               errorCorrectionShCapFns[ 'coherent' ].getFecs( allPhyCoherentStatus,
                                                              epis,
                                                              True )
         icdd.coherentErrorCorrectionIs( fecDict )

      capsDefaults.interfaces[ epis.name ] = icdd
   return capsDefaults

class ShowIntfHardwareDefault( IntfCli.ShowIntfCommand ):
   syntax = 'show interfaces capabilities | hardware default'
   data = dict( capabilities=capabilitiesDeprecated,
                hardware=hardwareKw,
                default='Show default port capabilities' )
   handler = showInterfacesCapabilitiesDefault
   moduleAtEnd = True
   cliModel = EthIntfModel.InterfacesCapabilitiesDefault

BasicCli.addShowCommandClass( ShowIntfHardwareDefault )

#-------------------------------------------------------------------------------
# The "show interfaces [<name>] transceiver properties" command, in enable mode.
#   show interfaces [<name>] transceiver properties
#   show interfaces module <modnum> transceiver properties
#-------------------------------------------------------------------------------
def getIntfCapabilitiesDir( intfName ):
   isModularSys = Cell.cellType() == 'supervisor'
   isManagement = Tac.Type( 'Arnet::MgmtIntfId' ).isMgmtIntfId( intfName )
   if isModularSys:
      if isManagement:
         return None
      else:
         sliceName = EthIntfLib.sliceName( intfName )
   else:
      sliceName = 'FixedSystem'
   capabilitiesDir = LazyMount.mount( entityManager,
      'interface/archer/status/eth/capabilities/input/slice/%s/xcvr' % sliceName,
      'Interface::Capabilities::InputCapabilitiesDir', 'r' )
   return capabilitiesDir 

# return: List of linkModeCapabilities of the transceiver 
def getLinkModeCapabilitiesCaps( intfName, inputDir ):
   linkModeCapsList = None
   lanes = False 
   if inputDir and intfName in inputDir.intfCapabilities:
      intfCapabilities = inputDir.intfCapabilities[ intfName ]
      lanes = EthPhyIntf._showLanes( intfCapabilities )
      linkModeCaps = intfCapabilities.linkModeCapabilities
      linkModeCapsList = EthIntfLib.linkModeCapabilitiesToList( linkModeCaps,
                         showLanes=lanes )
      if not linkModeCapsList:
         linkModeCapsList = 'none'
   return linkModeCapsList

def getSwappableXcvr( intfName, inputDir ):
   isSwappableXcvr = False
   if inputDir and intfName in inputDir.intfCapabilities:
      intfCapabilities = inputDir.intfCapabilities[ intfName ]
      isSwappableXcvr = intfCapabilities.isSwappableXcvr
   return isSwappableXcvr

def showInterfacesXcvrProperties( mode, args ):
   interface = args.get( 'INTF' )
   mod = args.get( 'MOD' )
   model = XcvrCapabilitiesModel.TransceiverPropertiesBase()
   intfs = getAllIntfs( mode, interface, mod, EthPhyIntf )
   if not intfs:
      return model
   
   for intf in intfs:
      tp = XcvrCapabilitiesModel.TransceiverProperties()
      tp.mediaType = intf.xcvrTypeStr()
      xcvrPresent = intf.intfXcvrStatus().xcvrPresence == 'xcvrPresent' 
      tp.transceiverPresenceIs( xcvrPresent )
      stringFormat = 'Format2'
      # Passing the info of speed capabilities when the transceiver is installed.
      # If there is no transceiver installed, no speed capabilities in the output
      # of the command.
      if xcvrPresent:
         intfCapabilitiesDir = getIntfCapabilitiesDir( intf.name )
         linkModeCaps = getLinkModeCapabilitiesCaps( intf.name, intfCapabilitiesDir )
         tp.xcvrCapListIs( linkModeCaps )
         tp.isSwappableXcvrIs( getSwappableXcvr( intf.name, intfCapabilitiesDir ) )
      else:
         tp.xcvrCapListIs( None )
      if intf.autonegActive():
         # If the transceiver auto-negotiation is active, the administrative speed
         # will be 'auto' or 'Auto-speed', the laneCount will be 0.
         tp.adminSpeedIs( intf.adminSpeedStr(), 0, intf.adminDuplexStr(), 
                          intf.showLanes(), intf.autonegActive() )
         if intf.status().speed == EthSpeed.speedUnknown:
            # If the transceiver auto negotiation is active and speed is unknown,
            # the operational speed will be 'auto' or 'Auto-speed', the laneCount
            # will be 0.
            tp.operSpeedIs( intf.operSpeed( stringFormat ), 0, 
                            intf.operDuplex( stringFormat ), intf.showLanes(), 
                            intf.autonegActive(), intf._autonegSuffix() )
         else:
            operSpeed, operLanes = intf.setOperSpeedInProperties()
            tp.operSpeedIs( operSpeed, operLanes, intf.operDuplex( stringFormat ), 
                            intf.showLanes(), intf.autonegActive(), 
                            intf._autonegSuffix() )
      else: 
         adminSpeed, adminLanes = intf.adminSpeed( stringFormat )
         tp.adminSpeedIs( adminSpeed, adminLanes, intf.adminDuplexStr(), 
                          intf.showLanes(), intf.autonegActive() )
         operSpeed, operLanes = intf.setOperSpeedInProperties()
         tp.operSpeedIs( operSpeed, operLanes, intf.operDuplex( stringFormat ), 
                         intf.showLanes(), intf.autonegActive(), 
                         intf._autonegSuffix() )
      model.interfaces[ intf.name ] = tp 
   return model

class ShowIntfXcvrProperties( IntfCli.ShowIntfCommand ):
   syntax = 'show interfaces transceiver properties'
   data = dict( transceiver='Details on transceivers',
                properties='Show interface transceiver properties' )
   handler = showInterfacesXcvrProperties
   cliModel = XcvrCapabilitiesModel.TransceiverPropertiesBase 

BasicCli.addShowCommandClass( ShowIntfXcvrProperties )

#-------------------------------------------------------------------------------
# The "show interfaces [<name>] negotiation [detail]" command, in enable mode.
#   show interfaces [<name>] negotiation [detail]
#   show interfaces module <modnum> negotiation [detail]
#-------------------------------------------------------------------------------

def showInterfacesNegotiationDetails( intfs ):

   lineFmt = '  %-18.18s %-52.52s'
   negFmt = '%-17.17s %-10.10s %-20.20s'
   resolvedFmt = '%-17.17s %-10.10s %-25.25s'

   for i in intfs:

      state = i.autonegStateStr( longStr=True )
      mode = i.autonegMode()

      localSpeed = i.localSpeedSet( detail=True )
      localDuplex = i.localDuplexSet( detail=True )
      localPause = i.localAnegPause()
      localAdStr = negFmt % ( localSpeed, localDuplex, localPause )

      partnerSpeed = i.partnerSpeedSet( detail=True )
      partnerDuplex = i.partnerDuplexSet( detail=True )
      partnerPause = i.partnerAnegPause()
      partnerAdStr = negFmt % ( partnerSpeed, partnerDuplex, partnerPause )

      advert = i.status().autonegStatus.advertisement
      active = advert.linkModes.anyCapsSet()      
      resolvedSpeed = i.operSpeed( 'Format1' ) if active else 'None'
      resolvedDuplex = i.operDuplex( 'Format2' ) if active else 'None'
      if advert.pause != 'pauseUnknown':
         resolvedPause = 'Rx=%s,Tx=%s' % ( i.flowcontrolRxOperState(),
                                           i.flowcontrolTxOperState() )
      else:
         resolvedPause = 'None'
      resolvedStr = resolvedFmt % ( resolvedSpeed, resolvedDuplex, resolvedPause )

      print( i.name, ":\n" )
      print( '%-25.25s %-40.40s' % ( 'Auto-Negotiation Mode', mode  ) )
      print( '%-25.25s %-40.40s' % ( 'Auto-Negotiation Status', state ) )
      print()

      # Only display further details if autoneg is active
      autonegStatus = i.status().autonegStatus
      if ( i.config().enabled and
           autonegStatus.state not in [ 'anegStateUnknown', 'anegStateOff' ] ):
         negHdr = negFmt % ( 'Speed', 'Duplex', 'Pause' )
         print( lineFmt % ( 'Advertisements', negHdr ) )

         negHdr = negFmt % ( '-' * 17, '-' * 10, '-' * 20 )
         print( lineFmt % ( ' ', negHdr ) )

         print( lineFmt % ( ' ' * 4 + 'Local', localAdStr ) )
         print( lineFmt % ( ' ' * 4 + 'Link Partner', partnerAdStr ) )
         print()

         print( lineFmt % ( 'Resolution', resolvedStr ) )
         print()

      print()

def showInterfacesNegotiation( mode, args ):
   intf = args.get( 'INTF' )
   mod = args.get( 'MOD' )
   intfs = getAllIntfs( mode, intf, mod, EthPhyIntf )
   if not intfs:
      return

   if 'detail' in args:
      showInterfacesNegotiationDetails( intfs )
      return

   hdrFmt  = '%-10.10s  %-7.7s  %-11.11s  %-20.20s'
   print( hdrFmt % ( 'Port', 'Autoneg', ' ', 'Negotiated Settings' ) )

   fmt = '%-10.10s  %-7.7s  %-5.5s  %-5.5s  %-6.6s  %-8.8s  %-8.8s  %-4.4s'
   print( fmt % ( ' ', 'Status', 'Speed', 'Lanes', 'Duplex',
                 'Rx Pause', 'Tx Pause', 'FEC' ) )
   print( fmt % ( '-' * 10, '-' * 7, '-' * 5, '-' * 5, '-' * 6,
                 '-' * 8, '-' * 8, '-' * 4 ) )

   for i in intfs:

      s = i.status()
      name = i.shortname
      anegStatus = s.autonegStatus
      state = i.autonegStateStr()

      if state == 'off' or state == 'unknown':
         speed = '-'
         duplex = '-'
         lanes = '-'
         rxFlowControl = '-'
         txFlowControl = '-'
         fec = '-'
      else:
         advert = anegStatus.advertisement
         active = advert.linkModes.anyCapsSet()

         speed = i.operSpeed( 'Format2' ) if active else '-'
         duplex = i.operDuplex( 'Format2' ) if active else '-'
         laneCount = EthTypesApi.laneCountNumber( anegStatus.laneCount )
         laneCount = laneCount if laneCount else 'auto'
         lanes = laneCount if active else '-'

         if advert.pause != 'pauseUnknown':
            rxFlowControl = i.flowcontrolRxOperState()
            txFlowControl = i.flowcontrolTxOperState()
         else:
            rxFlowControl = txFlowControl = '-'

         fec = i.anegFecEncodingStrMap[ anegStatus.fecEncoding ] if active else '-'

      print( fmt % ( name, state, speed, lanes,
                    duplex, rxFlowControl, txFlowControl, fec ) )

class ShowIntfNegotiation( IntfCli.ShowIntfCommand ):
   syntax = 'show interfaces negotiation [ detail ]'
   data = dict( negotiation='Show interface Auto-Negotiation status',
                detail='Show detailed interface Auto-Negotiation status' )
   handler = showInterfacesNegotiation

BasicCli.addShowCommandClass( ShowIntfNegotiation )

# ------------------------------------------------------------------------------
# configure; interface defaults; ethernet
# ------------------------------------------------------------------------------
class InterfaceDefaultsEthernetConfigMode( InterfaceDefaultsEthernetMode,
                                           BasicCli.ConfigModeBase ):
   name = "Interface Defaults Ethernet Configuration"
   modeParseTree = CliParser.ModeParseTree()

   def __init__( self, parent, session ):
      InterfaceDefaultsEthernetMode.__init__( self )
      BasicCli.ConfigModeBase.__init__( self, parent, session )

#--------------------------------------------------------------------------------
# ethernet
#--------------------------------------------------------------------------------
class EthernetCmd( CliCommand.CliCommandClass ):
   syntax = 'ethernet'
   data = {
      'ethernet' : 'Set default values for new interfaces of type ethernet',
   }

   @staticmethod
   def handler( mode, args ):
      newSubMode = mode.childMode( InterfaceDefaultsEthernetConfigMode )
      mode.session_.gotoChildMode( newSubMode )

IntfCli.InterfaceDefaultsConfigMode.addCommandClass( EthernetCmd )

# ------------------------------------------------------------------------------
# configure; interface defaults; ethernet; [no] shutdown
# ------------------------------------------------------
# This is used to set what the admin state of a just instanciated ethernet interface
# should be. This is a first-day-of-service setting, and for modulars it also affects
# the inserting of a new linecard. The default admin state has historically been 'up'
# but some customers want interfaces to say "silent" until they are configured.
# The only way to change this default behavior is by shipping the EOS switch with
# a factory preset file at /mnt/flash/kickstart-config which contains this command.
# A so preset switch will have inert interfaces on first powerup.
# The kickstart-config file is only read if there is no startup-config and ZeroTouch
# is disabled. If ZeroTouch is enabled, it will get the correct config so no need to
# worry about default configured interfaces causing havoc (and note that EOS agents
# will not run until that zeroTouch config has been stored into Sysdb).
# There are security reasons to want inert interfaces on first power up, but also
# considerations like not sending the wrong frequency/color into an optical muxer.
# ------------------------------------------------------------------------------
IntfEnabledState = Tac.Type( 'Interface::IntfEnabledState' )

def defaultShutdown( mode, args ):
   if not CliCommand.isNoOrDefaultCmd( args ):
      targetShutdownState = IntfEnabledState.shutdown
   else:
      targetShutdownState = IntfEnabledState.enabled
   if IntfCli.globalIntfConfig.defaultEthernetShutdown != targetShutdownState:
      # since defaults will change soon, we have to explicitely set the actual admin
      # status of interfaces that were still 'unset' and relied on the default, else
      # the 'show run' output might say 'down' while "show interf status" says 'up'.
      # Note the sub-interfaces are not in the ethPhyIntfConfigSliceDir (and those
      # are not subject to this default settings, only physical interfaces are).
      # This feature will actually forcefully shut any interface that was still not
      # explicitely configured before, so warm the user and give him a way out.
      # Note that disabling the default shutdown feature will not affect any intfs
      # since while it is enabled both shut & unshut states are in running-config.
      def getListOfAffectedInterfaces():
         aIntfs = []
         for _, sliceVal in ethPhyIntfConfigSliceDir.iteritems():
            for intfId, intf in sliceVal.intfConfig.iteritems():
               if intfId.startswith( "Ethernet" ): # exclude management eth
                  if ( intf.adminEnabledStateLocal ==
                       IntfEnabledState.unknownEnabledState ):
                     aIntfs.append( intf )
         return aIntfs
      def proceedWithCommand( aIntfs ):
         mode.addWarning(
            "This command will change the administrative status of every Ethernet "
            "interface that has not yet been specifically configured.\n"
            "This might include the interface over which this session runs!\n"
            "Here is the list of affected interfaces:\n%s\n" % aIntfs )
         prompt = 'Do you wish to proceed with this command?'
         return BasicCliUtil.getChoice( mode, prompt, [ 'yes', 'no' ], 'yes' )
      aIntfs = getListOfAffectedInterfaces()
      answer = 'yes' # in case there are no affected interfaces
      if aIntfs:
         intfNames = " ".join( sorted( [ IntfCli.Intf.getShortname( i.intfId )
                                         for i in aIntfs ] ) )
         answer = proceedWithCommand( intfNames ).lower()
      if not ( answer == 'yes' or answer == 'ye' or answer == 'y' ):
         # cohab tests return an empty answer so take those as a 'yes'
         if not ( os.environ.get( "A4_CHROOT" ) and answer == "" ):
            print( "Command aborted" )
            return
      for intf in aIntfs:
         if intf.adminEnabledStateLocal == IntfEnabledState.unknownEnabledState:
            intf.adminEnabledStateLocal = targetShutdownState
      # now do the actual set to change the default admin status of new interfaces
      IntfCli.globalIntfConfig.defaultEthernetShutdown = targetShutdownState

#--------------------------------------------------------------------------------
# [ no | default ] shutdown
#--------------------------------------------------------------------------------
class ShutdownCmd( CliCommand.CliCommandClass ):
   syntax = 'shutdown'
   noOrDefaultSyntax = syntax
   data = {
      'shutdown' : "Set default admin state of new eth interfaces to 'disabled'",
   }

   handler = defaultShutdown
   noOrDefaultHandler = defaultShutdown

InterfaceDefaultsEthernetConfigMode.addCommandClass( ShutdownCmd )

def mountCounters( ):
   global ethIntfCounterReaderSm, ethPhyIntfCounterDir_
   mg = entityManager.mountGroup()
   mg.mount( "interface/counter/eth/intf", "Interface::AllIntfCounterDir", "w" )
   mg.close( callback=None )

   smashEm = SharedMem.entityManager( sysdbEm=entityManager )
   ethIntfCounterReaderSm = Tac.newInstance( "Interface::EthIntfCounterReaderSm",
                                             smashEm, ethIntfCounterWriterStatusDir )
   ethIntfCounterReaderSm.enableLegacyShmemManSupport()
   ethIntfCounterReaderSm.checkpointCounterDir = checkpointCounterDir
   ethIntfCounterReaderSm.handleInitialized()
   ethPhyIntfCounterDir_ = \
         ethIntfCounterReaderSm.legacyShmemManCounterAccessorInternal

#-------------------------------------------------------------------------------
# Have the Cli Agent mount all needed state from sysdb
#-------------------------------------------------------------------------------
def Plugin( em ):
   global entityManager
   global ethPhyIntfConfigSliceDir
   global ethPhyIntfDefaultConfigDir
   global ethIntfSliceDefaultConfigDir
   global ethIntfStatusDir, ethPhyIntfStatusDir, entmib, intfHwStatus
   global ethIntfGenerationSliceDir, ethIntfModeSliceDir
   global ethIntfCounterWriterStatusDir, checkpointCounterDir
   global ethPhyDefaultCapsSliceDir
   global intfXcvrStatusDir
   global bridgingHwCapabilities
   global entityMib
   global timestampingStatus

   entityManager = em
   mg = entityManager.mountGroup()
   ethIntfCounterWriterStatusDir = mg.mount(
      "interface/ethIntfCounter/writerStatus", "Tac::Dir", "ri" )
   # We use a cell specific path to store the Cli clear counter snapshot because the
   # current snapshot is stored in Smash which is not synchronized across
   # supervisors. Thus, after a switchover we don't want to use obsolete values (see
   # BUG217544). This needs to be revisited if we ever use synchronization of Smash
   # tables between supervisors for the current snapshot.
   checkpointCounterDir = mg.mount( Cell.path( "interface/cli/counters/eth" ),
                                    "Interface::AllEthIntfCounterBaseDir", "wf" )
   mg.close( mountCounters )

   ethPhyIntfDefaultConfigDir = ConfigMount.mount(
                         entityManager,
                         "interface/config/eth/phy/globalDefault",
                         "Interface::EthPhyIntfGlobalDefaultConfigDir", "w" )

   ethPhyIntfConfigSliceDir = ConfigMount.mount( entityManager,
                                        "interface/config/eth/phy/slice",
                                        "Tac::Dir", "wi" )
   ethIntfStatusDir = LazyMount.mount( entityManager,
                                "interface/status/eth/intf",
                                "Interface::EthIntfStatusDir", "r" )

   ethIntfSliceDefaultConfigDir = LazyMount.mount( entityManager,
                                "interface/archer/config/eth/phy/sliceDefault",
                                "Interface::EthIntfSliceDefaultConfigDir", "r" )

   mg2 = entityManager.mountGroup()
   redStatus = entityManager.lookup( Cell.path( 'redundancy/status' ) )
   ethPhyIntfStatusDir = mg2.mount( "interface/status/eth/phy/all",
                                    "Interface::AllEthPhyIntfStatusDir", "r" )
   timestampingStatus = mg2.mount( "interface/status/timestamping",
                                   "Timestamping::Status", "w" )

   def _createTimestampingStatusSm():
      if 'NO_TIMESTAMPING_SM' not in os.environ:
         entityManager.timestampingSm = Tac.newInstance(
            'Timestamping::StatusAggregateSm', redStatus, ethPhyIntfStatusDir,
            timestampingStatus )
   mg2.close( _createTimestampingStatusSm )

   ethIntfGenerationSliceDir = LazyMount.mount( entityManager,
                                "interface/archer/config/eth/phy/generation/slice",
                                "Tac::Dir", "ri" )

   ethIntfModeSliceDir = LazyMount.mount( entityManager,
                                "interface/archer/status/eth/phy/mode/slice",
                                "Tac::Dir", "ri" )

   # The only reason this mount is here is to ensure that
   # CliSessionMgr will have it when certain CliSavePlugin files
   # request this path.
   _ethPhyIntfConfigDir = LazyMount.mount( entityManager,
                                           "interface/config/eth/phy/all",
                                           "Interface::AllEthPhyIntfConfigDir",
                                           "r" )

   intfXcvrStatusDir = XcvrEthIntfDir.ethIntfXcvrStatusDir( entityManager )

   intfHwStatus = LazyMount.mount( entityManager, "interface/hwstatus",
                                   "Interface::HwStatus", "r" )
   entmib = LazyMount.mount( entityManager, "hardware/entmib",
                             "EntityMib::Status", "r" )

   ethPhyDefaultCapsSliceDir = LazyMount.mount( entityManager,
                        "interface/archer/status/eth/phy/capabilities/default/slice",
                        "Tac::Dir", "ri" )

   bridgingHwCapabilities = LazyMount.mount( entityManager,
                                             "bridging/hwcapabilities",
                                             "Bridging::HwCapabilities", "r" )

   entityMib = LazyMount.mount(
      entityManager,
      'hardware/entmib',
      'EntityMib::Status', 'r' )

   global systemName
   systemName = entityManager.sysname()

#--------------------------------------------------------------------
#
#   show interface [interface name] mac [detail]
#
# Here's the idea: MAC status information is stored in EthPhyIntfStatus.
# We will only look at those interfaces that have physical ports associated
# with them. We will also make an explicit assumption that for every physical
# port, the MAC display information in the ethPhyIntfStatus object is
# valid.
#
# If a platform wishes to provide hook to display additional information,
# they can do so. That will be displayed as part of the "detail" option.
#
# There is one monkey wrench in this scheme: Unlike PHYs, where PHY type
# can differentiate who is the "owner" to display additional info, there
# is nothing of that sort at the interface level. When the CLI Plugins are
# loaded, multiple registrations can happen for specific display.
# The scheme to loop through the registered functions is not great,
# but will do for now.
#--------------------------------------------------------------------

def _maybeCallShowIntMacRegisteredFunctions( table, *args ):

   for entry in table.values():
      checker = entry[ 'checkFunc' ]
      if not checker or checker( systemName ):
         entry[ 'func' ]( *args )


def _showInterfaceMacDetailForIntf( intfMacData ):

   hwPhyStatus = intfMacData.phyIntfStatus
   ethStatus = intfMacData.ethStatus
   fmt = '  %-27.27s %-16.16s %8.8s %22.22s'
   fmt2 = '  %-27.27s %s'

   linkStatus = ethStatus.linkStatus if ethStatus.active else "inactive"

   print( '' )
   print( ethStatus.intfId )
   print( fmt % ( '', 'Current State', 'Changes', 'Last Change' ) )

   phyState = None
   phyStateChanges = None
   lastPhyStateChange = None

   if isinstance( hwPhyStatus, Tac.Type( 'Hardware::PhyStatus::PhyStatusGen3' ) ):
      phyState = hwPhyStatus.phyState.current
      phyStateChanges = hwPhyStatus.phyState.changes
      lastPhyStateChange = Ark.timestampToStr( hwPhyStatus.phyState.lastChange )
   elif hasattr( hwPhyStatus, 'phyState' ):
      phyState = hwPhyStatus.phyState
      phyStateChanges = hwPhyStatus.phyStateChanges
      lastPhyStateChange = Ark.timestampToStr( hwPhyStatus.lastPhyStateChange )
   else:
      # copy interface state attributes to phyState if phyState not present
      phyState = linkStatus
      phyStateChanges = ethStatus.operStatusChange.count
      lastPhyStateChange = Ark.timestampToStr( ethStatus.operStatusChange.time )

   print( fmt % ( 'PHY State', phyState, phyStateChanges, lastPhyStateChange ) )

   print( fmt % ( 'Interface State',
                 linkStatus,
                 ethStatus.operStatusChange.count,
                 Ark.timestampToStr( ethStatus.operStatusChange.time ) ) )

   print( fmt % ( 'MAC Rx Local Fault',
                 ethStatus.macRxLocalFault,
                 ethStatus.macRxLocalFaultChanges,
                 Ark.timestampToStr( ethStatus.lastMacRxLocalFaultChange ) ) )

   print( fmt % ( 'MAC Rx Remote Fault',
                 ethStatus.macRxRemoteFault,
                 ethStatus.macRxRemoteFaultChanges,
                 Ark.timestampToStr( ethStatus.lastMacRxRemoteFaultChange ) ) )

   # If there is additional status, it is stored as a set of strings. Let's
   # just dump them for the user. This is sorted based on the key.

   sortedTuples = sorted( ethStatus.macLayerAdditionalStatus.items(),
                          key=lambda(key,value) : key )

   for (key, value) in sortedTuples:
      print( fmt2 % ( key, value ) )



def _getInterfaceMacStatusSummaryForIntf( intfMacData ):

   output = []

   hwPhyStatus = intfMacData.phyIntfStatus
   ethStatus = intfMacData.ethStatus

   phyState = None
   lastPhyStateChange = None

   if isinstance( hwPhyStatus, Tac.Type( 'Hardware::PhyStatus::PhyStatusGen3' ) ):
      phyState = hwPhyStatus.phyState.current
      lastPhyStateChange = hwPhyStatus.phyState.lastChange
   elif hasattr( hwPhyStatus, 'phyState' ):
      phyState = hwPhyStatus.phyState
      lastPhyStateChange = hwPhyStatus.lastPhyStateChange
   else:
      # copy interface state attributes to phyState if phyState not present
      phyState = ethStatus.linkStatus if ethStatus.active else "inactive"
      lastPhyStateChange = ethStatus.operStatusChange.time

   adminEnabled = intfMacData.ethConfig.adminEnabled
   lastChangeTime = 0.0
   fault = '-'

   output.append( ethStatus.intfId )
   output.append( 'Up' if adminEnabled else 'Down' )
   linkStatus = ethStatus.linkStatus if ethStatus.active else "inactive"
   output.append( linkStatus )
   output.append( phyState )

   # Local and/or Remote fault reception could have been reported.
   # But we will make sure that the reported last change is
   # whatever the latest among the 4, including PHY state change
   # and Interface change.

   # If the PHY state is anything other than link up/down, what's
   # the point of looking at MAC status anyway?

   if phyState == 'linkDown' or phyState == 'linkUp':
      fault = ''
      lastChangeTime = max( lastPhyStateChange,
                            ethStatus.operStatusChange.time,
                            ethStatus.lastMacRxLocalFaultChange,
                            ethStatus.lastMacRxRemoteFaultChange,
                            lastChangeTime )

      if ethStatus.macRxLocalFault:
         fault += 'L'
      if ethStatus.macRxRemoteFault:
         fault += 'R'

   output.append( fault )
   output.append( Ark.timestampToStr( lastChangeTime ) )

   return output


def _showInterfacesMacDetail( hwPhyStatusTuples ):

   print( 'Current System Time: '+ str( time.ctime( Tac.utcNow() ) ) )

   for t in hwPhyStatusTuples:
      _showInterfaceMacDetailForIntf( t[ 1 ] )


def _showInterfacesMacSummary( hwPhyStatusTuples ):

   print( '''Key:
   L  = Rx Local Fault Reported
   R  = Rx Remote Fault Reported
   Last Change: Time when most recent fault status changed
   ''' )

   formatStr = '%-18.18s %-6.6s %-8.8s %-15.15s %-5.5s %16.16s'
   print( formatStr % ( '', 'Config', 'Oper', '', 'MAC', 'Last' ) )
   print( formatStr % ( 'Interface', 'State', 'State', 'PHY State',
                        'Fault', 'Change' ) )
   print( formatStr % ( '-' * 18, '-' * 6, '-' * 8, '-' * 15, '-' * 5, '-' * 16 ) )

   for t in hwPhyStatusTuples:
      output = _getInterfaceMacStatusSummaryForIntf( t[ 1 ] )
      print( formatStr % tuple( output ) )


def _getPrintableInterfaceList( interfaceNames ):
   ShowIntMacData = namedtuple( 'ShowIntMacData',
                                'phyIntfStatus ethConfig ethStatus' )
   # mount phy topology
   phyTopology = PhyStatusLib.mountPhyTopology( entityManager )
   # Look in the phy topology for phy config/status objects first.
   phyStatusList = {}
   for intfName in interfaceNames:
      phyData = PhyStatusLib.phyConfigStatusAtPos(
         entityManager.root(), phyTopology, intfName,
         Tac.Type( "PhyEee::PhyPosition" ).asicPhy, firstLaneOnly=True )
      if phyData:
         assert len( phyData ) == 1
         phyStatusList[ intfName ] = phyData[ 0 ][ 1 ]

   # Grab other information necessary for show int mac to work. If all information
   # is not present, don't return data on that interface.
   interfaceList = {}
   for intfName, phyIntfStatus in phyStatusList.iteritems():
      ethStatus = ethPhyIntfStatusDir.intfStatus[ phyIntfStatus.intfId ]
      sliceName = EthIntfLib.sliceName( intfName )
      sliceDir = ethPhyIntfConfigSliceDir.get( sliceName )
      if sliceDir:
         interfaceList[ intfName ] = ShowIntMacData(
            phyIntfStatus, sliceDir.intfConfig[ intfName ], ethStatus )

   # Let's sort it and send a list of tuples back
   printableInterfaceList = sorted( interfaceList.items(),
                                    key=lambda k: Arnet.intfNameKey( k[ 0 ] ) )
   return printableInterfaceList


def showInterfacesMacCommandHandler( mode, detail, interfaceNames ):

   printableInterfaceList = _getPrintableInterfaceList( interfaceNames )

   if not printableInterfaceList:
      return

   if detail:
      _showInterfacesMacDetail( printableInterfaceList )
   else:
      _showInterfacesMacSummary( printableInterfaceList )


def showInterfacesMac( mode, args ):
   intf = args.get( 'INTF' )
   mod = args.get( 'MOD' )
   detail = 'detail' in args

   intfs = IntfCli.Intf.getAll( mode, intf, mod )
   if not intfs:
      return

   interfaceNames = Arnet.sortIntf( x.name for x in intfs )
   _maybeCallShowIntMacRegisteredFunctions( showIntMacDisplayHandlers_,
                                            mode, detail,
                                            interfaceNames )

class ShowIntfMac( IntfCli.ShowIntfCommand ):
   syntax = 'show interfaces mac [ detail ]'
   data = dict( mac='Display MAC status',
                detail='Display information in detail' )
   handler = showInterfacesMac

BasicCli.addShowCommandClass( ShowIntfMac )

showIntMacDisplayHandlers_ = {}

def registerShowIntMacDisplayFunction( name,
                                       displayFunction,
                                       checkFunction=None ):

   '''
   A check function, if provided, will be called with the sysname as parameter.
   If that function returns true then the actual display function is called.
   This allows callers to pass in a check function that can determine at run time
   (based on the actual HW etc.) whether or not to run their display function.
   '''

   assert not showIntMacDisplayHandlers_.has_key( name )
   entry = { 'func' : displayFunction, 'checkFunc' : checkFunction }
   showIntMacDisplayHandlers_[ name ] = entry

capsTuple = namedtuple( 'capabilityFn', 'getAll printStr getMods' )
interfaceCapabilityFns = {}
def registerInterfaceCapabilities( capability, getAllFn, printStrFn, getModsFn ):
   ''' A getAllFn can be called to gather all the information from
   Sysdb at once.'''
   interfaceCapabilityFns[ capability ] = capsTuple( getAllFn, printStrFn,
                                                     getModsFn )

fecTuple = namedtuple( 'errorCorrectionCfgFn', 'configFn' )
errorCorrectionCfgFns = {}
def registerErrorCorrectionCfgFn( errorCorrection, configFn ):
   # Configure the corresponding error-correction depending on the
   # interface type
   errorCorrectionCfgFns[ errorCorrection ] = fecTuple( configFn )

fecShowCapTuple = namedtuple( 'errorCorrectionShCapFn',
                              'checkFn checkFec printStr getFecs' )
errorCorrectionShCapFns = {}
def registerErrorCorrectionShCapFn( errorCorrection, checkFn, checkFec, printStr,
                                    getFecs ):
   # Print the corresponding capabilities for
   # "show interfaces capabilities [ default ]"
   errorCorrectionShCapFns[ errorCorrection ] = fecShowCapTuple( checkFn, checkFec,
                                                                 printStr, getFecs )

fecShowErrTuple = namedtuple( 'errorCorrectionShErrFn', 'showErr' )
errorCorrectionShErrFns = {}
def registerErrorCorrectionShErrFn( errorCorrection, showErr ):
   # Set the corresponding error-correction attrs for
   # "show interfaces error-correction"
   errorCorrectionShErrFns[ errorCorrection ] = fecShowErrTuple( showErr )

# Modelet for IntfRangeConfigMode
class EthIntfRangeModelet( CliParser.Modelet ):
   modeletParseTree = CliParser.ModeletParseTree()

   def __init__( self, mode ):
      CliParser.Modelet.__init__( self )

      if not isinstance( mode.individualIntfModes_[0].intf, EthPhyIntf ):
         Tracing.trace0( "Ethernet config commands not supported on %s" %
               mode.longModeKey )

# action for the cli "speed forced" in the "config-if-range" mode
def _setIntfRangeLinkMode( mode, args ):
   # First run through the registered hooks and confirm that we can go this linkMode
   speedTypeDict = speedForceLinkMode if 'forced' in args else speedLinkMode
   linkMode = speedTypeDict[ args.get( 'SPEED_TYPE', 'linkModeUnknown' ) ]
   if checkChangeCancelHooks( mode, [ m.intf for m in mode.individualIntfModes_ ],
                              linkMode, None ):
      return
   if checkChangeCancelOnSfp28( mode,
                                [ m.intf for m in mode.individualIntfModes_ ],
                                linkMode,
                                None ):
      return

   for m in mode.individualIntfModes_:
      m.intf.setLinkMode( linkMode )
      Tracing.trace0( "Set " + m.intf.name + " speed/duplex to " +
                   str( linkMode ) + " ('None' means auto)" )

def _setIntfRangeAutoLinkMode( mode, args ):
   linkMode = 'linkModeAutoneg'
   advertisedModesOpt=speedAutoLinkMode.get( args.get( 'SPEED_TYPE' ) )

   advertisedModes = Tac.Value( 'Interface::EthLinkModeSet' )
   if not advertisedModesOpt:
      advertisedModes = None
   elif advertisedModesOpt == 'advertise10MbpsHalf':
      advertisedModes.mode10MbpsHalf = True
   elif advertisedModesOpt == 'advertise10MbpsFull':
      advertisedModes.mode10MbpsFull = True
   elif advertisedModesOpt == 'advertise100MbpsHalf':
      advertisedModes.mode100MbpsHalf = True
   elif advertisedModesOpt == 'advertise100MbpsFull':
      advertisedModes.mode100MbpsFull = True
   elif advertisedModesOpt == 'advertise1GbpsFull':
      advertisedModes.mode1GbpsFull = True
   elif advertisedModesOpt == 'advertise2p5GbpsFull':
      advertisedModes.mode2p5GbpsFull = True
   elif advertisedModesOpt == 'advertise5GbpsFull':
      advertisedModes.mode5GbpsFull = True
   elif advertisedModesOpt == 'advertise10GbpsFull':
      advertisedModes.mode10GbpsFull = True
   elif advertisedModesOpt == 'advertise25GbpsFull':
      advertisedModes.mode25GbpsFull = True
   elif advertisedModesOpt == 'advertise40GbpsFull':
      # As of 11/20/2015, we still need to use the special 40G link mode
      # for FocalPoint (perhaps others as well).  See BUG97475
      linkMode = 'linkModeAuto40GbpsFull'
      advertisedModes.mode40GbpsFull = True
   elif advertisedModesOpt == 'advertise50GbpsFull':
      advertisedModes.mode50GbpsFull = True
   elif advertisedModesOpt == 'advertise50GbpsFull1Lane':
      advertisedModes.mode50GbpsFull1Lane = True
   elif advertisedModesOpt == 'advertise100GbpsFull':
      advertisedModes.mode100GbpsFull = True
   elif advertisedModesOpt == 'advertise100GbpsFull2Lane':
      advertisedModes.mode100GbpsFull2Lane = True
   elif advertisedModesOpt == 'advertise200GbpsFull4Lane':
      advertisedModes.mode200GbpsFull4Lane = True
   elif advertisedModesOpt == 'advertise400GbpsFull8Lane':
      advertisedModes.mode400GbpsFull8Lane = True

   if mode.session.commandConfirmation():
      for hook in canPromptForAbortHook.extensions():
         if hook( mode, [ m.intf for m in mode.individualIntfModes_ ], linkMode,
                  advertisedModes ):
            Tracing.trace0( "LinkMode change for " + mode.longModeKey +
                            " cancelled by " + str( hook ) )
            return

   for m in mode.individualIntfModes_:
      m.intf.setLinkMode( linkMode, advertisedModes )
      Tracing.trace0( "Set " + m.intf.name + " speed/duplex to " +
                   str( linkMode ) + " ('None' means auto)" )

def noSpeedIntfRange( mode, args ):
   _setIntfRangeLinkMode( mode, args )
   Tracing.trace0( "Set " + mode.longModeKey + " speed to default" )

class SpeedForcedRangeCmd( CliCommand.CliCommandClass ):
   syntax = 'speed forced SPEED_TYPE'
   data = {
      'speed' : matcherSpeed,
      'forced' : 'Disable autoneg and force speed/duplex/flowcontrol',
      'SPEED_TYPE': speedForceExpression,
   }

   handler = _setIntfRangeLinkMode

EthIntfRangeModelet.addCommandClass( SpeedForcedRangeCmd )


class SpeedAutoRangeCmd( CliCommand.CliCommandClass ):
   syntax = 'speed auto [ SPEED_TYPE ]'
   data = {
      'speed' : matcherSpeed,
      'auto' : matcherAuto,
      'SPEED_TYPE' : speedAutoExpression,
   }

   handler = _setIntfRangeAutoLinkMode

EthIntfRangeModelet.addCommandClass( SpeedAutoRangeCmd )


class SpeedRangeCmd( CliCommand.CliCommandClass ):
   syntax = 'speed SPEED_TYPE'
   noOrDefaultSyntax = 'speed ...'
   data = {
      'speed' : matcherSpeed,
      'SPEED_TYPE': speedExpression
   }

   handler = _setIntfRangeLinkMode
   noOrDefaultHandler = noSpeedIntfRange

EthIntfRangeModelet.addCommandClass( SpeedRangeCmd )

#-------------------------------------------------------------------------------
# Associate the EthIntfRangeModelet with the "config-if-range" mode.
#-------------------------------------------------------------------------------
IntfRangeCli.IntfRangeConfigMode.addModelet( EthIntfRangeModelet )

#-------------------------------------------------------------------------------
# The "[no|default] transceiver qsfp default-mode <mode>" command, in "config" mode.
#-------------------------------------------------------------------------------
xcvrShowKw = CliMatcher.KeywordMatcher( 'transceiver',
                                        helpdesc='Show transceiver information' )

xcvrKw = CliMatcher.KeywordMatcher( 'transceiver',
                                    helpdesc='configure transceiver settings' )

qsfpKw = CliMatcher.KeywordMatcher( 'qsfp', helpdesc='QSFP transceiver' )

class XcvrQsfpDefaultMode( CliCommand.CliCommandClass ):
   syntax = 'transceiver qsfp default-mode 4x10G'
   noOrDefaultSyntax = 'transceiver qsfp default-mode ...'
   data = {
      'transceiver' : xcvrKw,
      'qsfp' : qsfpKw,
      'default-mode' : 'default mode',
      '4x10G': '4-by-10G mode',
   }

   @staticmethod
   def handler( mode, args ):
      qsfpConfig = ethPhyIntfDefaultConfigDir.xcvrModeConfig[ 'qsfp' ]
      if qsfpConfig.isSestoFixedSystem():
         qsfpConfig.globalMode = Tac.Type( 'Interface::XcvrMode' ).noXcvrMode
      else:
         qsfpConfig.globalMode = Tac.Type( 'Interface::XcvrMode' ).xcvrMode4x10G

   @staticmethod
   def noOrDefaultHandler( mode, args ):
      qsfpConfig = ethPhyIntfDefaultConfigDir.xcvrModeConfig[ 'qsfp' ]
      qsfpConfig.globalMode = Tac.Type( 'Interface::XcvrMode' ).noXcvrMode

BasicCli.GlobalConfigMode.addCommandClass( XcvrQsfpDefaultMode )

#-------------------------------------------------------------------------------
# Forward Error Correction (FEC) commands:
#-------------------------------------------------------------------------------

#-------------------------------------------------------------------------------
# "[no|default] error-correction encoding clause-91|clause-74"
# The no and default forms have different actions for this command.
#-------------------------------------------------------------------------------
# Allow the coherent keywords to show up if coherentErrorCorrection is
# supported on that interface.
def coherentFecTokenGuard( mode, token ):
   if mode.intf.coherentErrorCorrectionSupported():
      return None
   return CliParser.guardNotThisPlatform

# Allow 'reed-solomon' keyword to show up on a coherent interface if the interface
# has 'g709' capability else reject all the client FECs. If the client FECs are
# configured on a empty DCO slots we need to allow them.
def nonCoherentFecTokenGuard( mode, token ):
   if mode.intf.coherentErrorCorrectionSupported():
      if ( ( token == 'reed-solomon' and
             mode.intf.coherentErrorCorrectionEncodingSupp( 'fecEncodingG709' ) ) or
             mode.intf.errorCorrectionSupported() ):
         return None
      return CliParser.guardNotThisPlatform
   return None

nodeReedSolomon = CliCommand.guardedKeyword( 'reed-solomon',
      helpdesc='Configure Reed-Solomon (RS-FEC) forward error correction',
      guard=nonCoherentFecTokenGuard )
nodeSoftDecision = CliCommand.guardedKeyword( 'soft-decision',
      helpdesc='Configure soft decision based forward error correction',
      guard=coherentFecTokenGuard )
matcherG709 = CliMatcher.KeywordMatcher( 'g709',
      helpdesc='Configure Reed-Solomon G.709 forward error correction',
      value=lambda mode,match: 'coherentFecEncodingCfgG709' )
nodeG709ReedSolomon = CliCommand.Node( matcher=matcherG709,
      guard=coherentFecTokenGuard )
matcherOverhead = CliMatcher.KeywordMatcher( 'overhead',
      helpdesc='Configure the overhead for forward error correction' )
matcherPercent = CliMatcher.KeywordMatcher( 'percent',
      helpdesc='Percent' )
matcherBch = CliMatcher.KeywordMatcher( 'bch',
      helpdesc='Configure forward error correction with outer BCH code',
      value=lambda mode,match: 'coherentFecEncodingCfgSd25Bch' )
matcherOverhead7 = CliMatcher.KeywordMatcher( '7',
      helpdesc='Configure 7% overhead for forward error correction',
      value=lambda mode, match: 'coherentFecEncodingCfgHd7' )
matcherOverhead15 = CliMatcher.KeywordMatcher( '15',
      helpdesc='Configure 15% overhead for forward error correction',
      value=lambda mode, match: 'coherentFecEncodingCfgSd15' )
matcherOverhead20 = CliMatcher.KeywordMatcher( '20',
      helpdesc='Configure 20% overhead for forward error correction',
      value=lambda mode, match: 'coherentFecEncodingCfgSd20' )
matcherOverhead25 = CliMatcher.KeywordMatcher( '25',
      helpdesc='Configure 25% overhead for forward error correction',
      value=lambda mode, match: 'coherentFecEncodingCfgSd25' )

def setErrorCorrectionEncoding( mode, args ):
   noOrDefault = 'default' if CliCommand.isDefaultCmd( args ) \
         else CliCommand.isNoCmd( args )
   encodingValue = ( args.get( 'fire-code' ) or
                  args.get( 'g709' ) or
                  args.get( 'reed-solomon' ) or
                  args.get( '7' ) or
                  args.get( 'bch' ) or
                  args.get( '25' ) or
                  args.get( '20' ) or
                  args.get( '15' ) )
   mode.intf.setErrorCorrectionEncoding( encodingValue, noOrDefault )

#--------------------------------------------------------------------------------
# [ no | default ] error-correction encoding ( fire-code | reed-solomon [ g709 ] |
# ( soft-decision overhead ( 15 | 20 ) percent ) | 
# soft-decision overhead 25 percent [ bch ] |
# ( staircase overhead 7 percent ) )
#--------------------------------------------------------------------------------
class ErrorCorrectionEncodingCmd( CliCommand.CliCommandClass ):
   syntax = '''error-correction encoding 
            ( ( fire-code ) 
            | ( reed-solomon [ g709 ] ) 
            | ( soft-decision overhead ( 15 | 20 ) percent )  
            | ( soft-decision overhead 25 percent [ bch ] ) 
            | ( staircase overhead 7 percent ) )'''
   noOrDefaultSyntax = '''error-correction encoding 
            [ ( fire-code ) 
            | ( reed-solomon [ g709 ] ) 
            | ( soft-decision overhead ( 15 | 20 ) percent ) 
            | ( soft-decision overhead 25 percent [ bch ] ) 
            | ( staircase overhead 7 percent ) ]'''
   data = {
      'error-correction' : 'Configure forward error correction options',
      'encoding' : 'Configure forward error encoding',
      'fire-code' : CliCommand.guardedKeyword( 'fire-code',
         helpdesc='Configure Fire code (BASE-R) forward error correction',
         guard=nonCoherentFecTokenGuard ),
      'reed-solomon' : CliCommand.guardedKeyword( 'reed-solomon', 
         helpdesc='Configure Reed-Solomon (RS-FEC) forward error correction',
         guard=nonCoherentFecTokenGuard ),
      'g709' : nodeG709ReedSolomon,
      '7' : matcherOverhead7,
      'soft-decision' : nodeSoftDecision,
      'overhead' : matcherOverhead,
      '15' : matcherOverhead15,
      '20' : matcherOverhead20,
      'percent' : matcherPercent,
      '25' : matcherOverhead25,
      'bch' : matcherBch,
      'staircase' : CliCommand.guardedKeyword( 'staircase',
         helpdesc='Configure a staircase based forward error correction',
         guard=coherentFecTokenGuard ),
   }
   
   handler = setErrorCorrectionEncoding
   noOrDefaultHandler = handler

EthIntfModelet.addCommandClass( ErrorCorrectionEncodingCmd )

#--------------------------------------------------------------------------------
# [ no | default ] error-correction reed-solomon bypass BYPASS_VALUE
#--------------------------------------------------------------------------------
class ErrorCorrectionReedSolomonBypassCmd( CliCommand.CliCommandClass ):
   syntax = 'error-correction reed-solomon bypass BYPASS_VALUE'
   noOrDefaultSyntax = 'error-correction reed-solomon bypass ...'
   data = {
      'error-correction' : 'Configure forward error correction options',
      'reed-solomon' : nodeReedSolomon,
      'bypass' : 'Configure error correction bypass options',
      'BYPASS_VALUE' : CliMatcher.EnumMatcher( {
         'correction' : 'Disable correction of correctable errors',
         'indication' : 'Disable per frame indication of uncorrectable errors.',
      } )
   }
   hidden = True

   @staticmethod
   def handler( mode, args ):
      mode.intf.setErrorCorrectionBypass( args[ 'BYPASS_VALUE' ] )

   @staticmethod
   def noOrDefaultHandler( mode, args ):
      mode.intf.noDefaultErrorCorrectionBypass()

EthIntfModelet.addCommandClass( ErrorCorrectionReedSolomonBypassCmd )

#-------------------------------------------------------------------------------
# "show interfaces error-correction" command.
#-------------------------------------------------------------------------------

# Order is important. Coherent FEC data needs to be filled in before
# client FEC data. This allows coherent to be set first which sets
# all the client FEC attributes to false. Having the client FEC code
# run after coherent FEC code allows re-writing the client FEC
# attributes if required. If not, they're correctly set to false.
def showErrorCorrection( mode, args ):
   intf = args.get( 'INTF' )
   mod = args.get( 'MOD' )
   interfacesErrCorr = EthIntfModel.InterfacesErrorCorrectionStatus()
   intfs = IntfCli.Intf.getAll( mode, intf, mod, intfType=EthPhyIntf )
   if not intfs:
      return interfacesErrCorr

   FecEncodingStat = Tac.Type( 'Interface::EthFecEncoding' )
   fecStatusString = { FecEncodingStat.fecEncodingUnknown : 'unknown',
                       FecEncodingStat.fecEncodingDisabled : 'disabled',
                       FecEncodingStat.fecEncodingReedSolomon544 : 'reedSolomon',
                       FecEncodingStat.fecEncodingReedSolomon : 'reedSolomon',
                       FecEncodingStat.fecEncodingFireCode : 'fireCode' }

   fecEncodingConfigAttrXlate = {
      'fecEncodingDisabled' : 'fecEncodingConfigDisabled',
      'fecEncodingReedSolomon544' : 'fecEncodingConfigReedSolomon544Enabled',
      'fecEncodingReedSolomon' : 'fecEncodingConfigReedSolomonEnabled',
      'fecEncodingFireCode' : 'fecEncodingConfigFireCodeEnabled'
      }

   for x in intfs:
      if x.errorCorrectionSupported() or x.coherentErrorCorrectionSupported():
         errCorrStatus = EthIntfModel.InterfaceErrorCorrectionStatus()
         if x.coherentErrorCorrectionSupported():
            if "coherent" in errorCorrectionShErrFns:
               errCorrStatus = errorCorrectionShErrFns[ 'coherent' ].showErr(
                               x, errCorrStatus )
               interfacesErrCorr._hasCoherentIntf = True

         if x.errorCorrectionSupported():
            errCorrStatus._nonCoherentIntf = True # pylint:disable-msg=W0212
            errCorrStatus.errCorrEncodingStatus = fecStatusString[
                  x.status().fecStatus ]
            for tacAttr, modelAttr in fecEncodingConfigAttrXlate.items():
               tacVal = getattr( x.config().fecEncodingConfig, tacAttr )
               setattr( errCorrStatus, modelAttr, tacVal )

            # An encoding is available if fecEncodingDisabled is False,
            # the phy claims support for the encoding at the current speed,
            # and the configuration allows it. If the speed is unknown, we
            # are more lenient and use the logical or of the caps at all
            # speeds.
            if x.status().speed == EthSpeed.speedUnknown:
               # All caps will be initially False.
               fecCaps = Tac.Value( 'Interface::EthFecCapabilitySet' )
               for phyCaps in x.status().phyFecCapabilities.itervalues():
                  for attr in phyCaps.attributes:
                     if attr in [ 'stringValue', ]:
                        continue
                     val = getattr( fecCaps, attr ) or getattr( phyCaps, attr )
                     setattr( fecCaps, attr, val )
            elif x.status().autonegActive:
               fecCaps = None
               for i, j in EthIntfLib.autonegToforcedMapping.iteritems():
                  if getattr( x.status().autonegStatus.advertisement.linkModes, i ):
                     fecCaps = x.status().phyFecCapabilities.get( j )
                     break
            else:
               try:
                  fecCaps = \
                          x.status().phyFecCapabilities[ x.status().linkModeStatus ]
               except KeyError:
                  fecCaps = None

            fecEncodingAvailXlate = { 'fecEncodingReedSolomon544' :
                                         'fecEncodingReedSolomon544Available',
                                      'fecEncodingReedSolomon' :
                                         'fecEncodingReedSolomonAvailable',
                                      'fecEncodingFireCode' :
                                         'fecEncodingFireCodeAvailable' }

            for tacAttr, modelAttr in fecEncodingAvailXlate.items():
               setattr( errCorrStatus, modelAttr, False )
               if fecCaps is not None and getattr( fecCaps, tacAttr ):
                  setattr( errCorrStatus, modelAttr, True )
         interfacesErrCorr.interfaceErrCorrStatuses[ x.name ] = errCorrStatus
   return interfacesErrCorr

class ShowIntfErrorCorrection( IntfCli.ShowIntfCommand ):
   syntax = 'show interfaces error-correction'
   data = { 'error-correction' : 'Show interface error correction information' }
   handler = showErrorCorrection
   moduleAtEnd = True
   cliModel = EthIntfModel.InterfacesErrorCorrectionStatus

BasicCli.addShowCommandClass( ShowIntfErrorCorrection )

#--------------------------------------------------------------------------------
# Add loopback CLI commands to the "config-if" mode
# traffic-loopback source system device phy
# [ no | default ] traffic-loopback source ( ( system device ( phy | mac ) ) |
#                                            ( network device phy ) )
#--------------------------------------------------------------------------------
def loopbackSystemMacSupportedGuard( mode, token ):
   if mode.intf.loopbackModeSupported( lbMode=loopbackMode.loopbackMac ):
      return None
   return CliParser.guardNotThisPlatform

def loopbackSystemPhySupportedGuard( mode, token ):
   if mode.intf.loopbackModeSupported( lbMode=loopbackMode.loopbackPhy ):
      return None
   return CliParser.guardNotThisPlatform

def loopbackNetworkPhySupportedGuard( mode, token ):
   if mode.intf.loopbackModeSupported( lbMode=loopbackMode.loopbackPhyRemote ):
      return None
   return CliParser.guardNotThisPlatform

def loopbackModeSupportedGuard( mode, token ):
   if mode.intf.loopbackModeSupported():
      return None
   return CliParser.guardNotThisPlatform

class LoopbackModeCmd( CliCommand.CliCommandClass ):
   syntax = ( 'traffic-loopback source ( ( system device ( phy | mac ) ) | '
                                        '( network device phy ) )' )
   noOrDefaultSyntax = 'traffic-loopback ...'
   data = {
      'traffic-loopback' : CliCommand.guardedKeyword( 'traffic-loopback',
         helpdesc='Configure loopback',
         guard=loopbackModeSupportedGuard ),
      'source' : 'Configure traffic source',
      'system' : 'Traffic from local host',
      'device' : 'Configure loopback device',
      'network' : CliCommand.guardedKeyword( 'network',
         helpdesc='traffic from peer host',
         guard=loopbackNetworkPhySupportedGuard ),
      'phy' : CliCommand.guardedKeyword( 'phy',
         helpdesc='implement loopback in PHY layer',
         guard=loopbackSystemPhySupportedGuard ),
      'mac' : CliCommand.guardedKeyword( 'mac',
         helpdesc='implement loopback in MAC layer',
         guard=loopbackSystemMacSupportedGuard ),
   }

   @staticmethod
   def handler( mode, args ):
      if 'system' in args:
         if 'phy' in args:
            mode.intf.setLoopbackMode( 'system', 'phy' )
         else:
            mode.intf.setLoopbackMode( 'system', 'mac' )
      else:
         mode.intf.setLoopbackMode( 'network', 'phy' )

   @staticmethod
   def noOrDefaultHandler( mode, args ):
      mode.intf.setNoLoopbackMode()

EthIntfModelet.addCommandClass( LoopbackModeCmd )

#--------------------------------------------------------------------------------
# The CLI command to configure autoneg 25g/50g consortium operation
# [ no | default ] phy media 25gbase-cr negotiation standard { consortium | ieee }
#--------------------------------------------------------------------------------
def autonegConsortiumSupportedGuard( mode, token ):
   if mode.intf.autonegStandardSupported( 'consortium25g' ):
      return None
   return CliParser.guardNotThisPlatform

def autonegIeeeSupportedGuard( mode, token ):
   if mode.intf.autonegStandardSupported( 'ieee25g' ):
      return None
   return CliParser.guardNotThisPlatform

class PhyMedia25GbaseCrNegotiationStandardCmd( CliCommand.CliCommandClass ):
   syntax = 'phy media 25gbase-cr negotiation standard { consortium | ieee }'
   noOrDefaultSyntax = 'phy media 25gbase-cr negotiation standard ...'
   data = {
      'phy' : 'Configure phy-specific parameters',
      'media' : 'Configure media specific options',
      '25gbase-cr' : 'Options for 25gBase-cr media',
      'negotiation' : 'Configure auto-negotiation options',
      'standard' : 'Configure auto-negotiation standards',
      'consortium' : CliCommand.guardedKeyword( 'consortium',
         helpdesc='Configure 25G/50G ethernet consortium operation',
         guard=autonegConsortiumSupportedGuard, maxMatches=1 ),
      'ieee' : CliCommand.guardedKeyword( 'ieee',
         helpdesc='Configure IEEE 802.3 Clause 73 standard operation',
         guard=autonegIeeeSupportedGuard, maxMatches=1 ),
   }

   @staticmethod
   def handler( mode, args ):
      aneg25gModes = []
      if 'consortium' in args:
         aneg25gModes.append( 'consortium25g' )
      if 'ieee' in args:
         aneg25gModes.append( 'ieee25g' )
      mode.intf.setAutonegStandard( aneg25gModes )

   @staticmethod
   def noOrDefaultHandler( mode, args ):
      mode.intf.setAutonegStandard( None )

EthIntfModelet.addCommandClass( PhyMedia25GbaseCrNegotiationStandardCmd )
