# Copyright (c) 2019 Arista Networks, Inc.  All rights reserved.
# Arista Networks, Inc. Confidential and Proprietary.

from CliModel import Dict, Int, Model
from TableOutput import createTable, Format

class HandlerCounter( Model ):
   '''Model for routing handler counter.'''
   ignored = Int( help="Number of ignored packets" )
   routed = Int( help="Number of routed packets" )

class HandlerCounters( Model ):
   '''Model for show platform etba counters'''
   ebraCounters = Dict( keyType=str,
                        valueType=HandlerCounter, optional=True,
                        help="Maps EbraTestBridge routing handler names "
                             "to respective counters" )
   ebraIgnored = Int( help="Number of packets that were ignored by "
                           "all EbraTestBridge routing handlers" )
   ebraRouted = Int( help="Number of packets that were routed by at least "
                          "one EbraTestBridge routing handler" )
   fwdIntfCounters = Dict( keyType=str,
                           valueType=HandlerCounter, optional=True,
                           help="Maps FwdIntfDevice routing handler names "
                                "to respective counters" )
   fwdIntfIgnored = Int( help="Number of packets that were ignored by "
                              "all FwdIntfDevice routing handlers" )
   fwdIntfRouted = Int( help="Number of packets that were routed by "
                             "a FwdIntfDevice routing handler" )

   def render( self ):
      ignoredMsg = 'Ignored by all handlers: '
      fwdIntfRoutingHandlerStr = 'FwdIntfDevice:RoutingHandler' 
      ebraRoutingHandlerStr = 'EbraTestBridge:RoutingHandler' 

      formats = {}
      formats[ fwdIntfRoutingHandlerStr ] = Format( justify='left' )
      formats[ ebraRoutingHandlerStr ] = Format( justify='left' )
      formats[ 'Ignored' ] = Format( justify='left' )
      formats[ 'Routed' ] = Format( justify='left' )
      for columnFormat in formats.values():
         columnFormat.noPadLeftIs( True )
         columnFormat.padLimitIs( True )

      if self.fwdIntfCounters:
         fwdIntfCounterTable = createTable( ( fwdIntfRoutingHandlerStr,
                                              'Ignored', 'Routed' ),
                                            tableWidth=100 )
         fwdIntfCounterTable.formatColumns( formats[ fwdIntfRoutingHandlerStr ],
                                            formats[ 'Ignored' ],
                                            formats[ 'Routed' ], )
         # Displayed in the order the routing handlers are called because as soon as
         # one successully routes a packet, others aren't tried for FwdIntfDevice.
         for name, counter in self.fwdIntfCounters.items():
            fwdIntfCounterTable.newRow( name, counter.ignored, counter.routed )
         print fwdIntfCounterTable.output()
      else:
         print( fwdIntfRoutingHandlerStr + '\n' +
               '-' * len( fwdIntfRoutingHandlerStr ) )

      print ignoredMsg + str( self.fwdIntfIgnored ) 
      print 'Routed by a handler: ' + str( self.fwdIntfRouted ) + '\n\n'

      if self.ebraCounters:
         ebraCounterTable = createTable( ( ebraRoutingHandlerStr,
                                           'Ignored', 'Routed' ),
                                         tableWidth=100 )
         ebraCounterTable.formatColumns( formats[ ebraRoutingHandlerStr ],
                                         formats[ 'Ignored' ],
                                         formats[ 'Routed' ], )
         for name, counter in sorted( self.ebraCounters.items() ):
            ebraCounterTable.newRow( name, counter.ignored, counter.routed )
         print ebraCounterTable.output()
      else:
         print ebraRoutingHandlerStr + '\n' + '-' * len( ebraRoutingHandlerStr )

      print ignoredMsg + str( self.ebraIgnored )
      print 'Routed by at least one handler: ' + str( self.ebraRouted )
