#!/usr/bin/env python
# Copyright (c) 2016 Arista Networks, Inc.  All rights reserved.
# Arista Networks, Inc. Confidential and Proprietary.

import ConfigMount
import LazyMount, CliParser
import BasicCli, IntfCli, EthIntfCli
import Tac, Tracing
import CliMatcher
import CliToken
from CliToken.Monitor import monitorMatcher
import CliCommand
import ShowCommand
from ShowCommand import ShowCliCommandClass
from CliMode.Eoam import EoamProfileModeBase, LinkMonitoringModeBase
from CliMode.Eoam import EoamConfigModeBase
from EoamTypes import tacEoamAction, tacEoamInterval, tacIntervalUnit, tacErrorType
from EoamTypes import errorTypeToEnumMap, actionToEnumMap, intervalUnitToEnumMap
from Intf.IntfRange import intfListToCanonical
__defaultTraceHandle__ = Tracing.Handle( 'EoamCli' )
t0 = Tracing.trace0

eoamConfig = None
eoamStatus = None
eoamHwStatus = None

class EoamIntfJanitor( IntfCli.IntfDependentBase ):
   def setDefault( self ):
      t0( "EoamIntfJanitor: interface", self.intf_.name, "going away" )
      if eoamConfig.intfToProfileName.get( self.intf_.name ):
         del eoamConfig.intfToProfileName[ self.intf_.name ]

class EoamProfileModeContext( object ):
   def __init__( self, mode, profileName ):
      self.mode = mode
      self.profile_ = None
      self.profileName_ = profileName
      self.currentEntry_ = None
      self.previousEntry_ = None # used for config rollback
      eoamProfileConfig = eoamConfig.eoamProfile
      if profileName in eoamProfileConfig:
         self.profile_ = eoamProfileConfig[ profileName ]
         self.profileName_ = profileName
         prevEoamProfile =  Tac.newInstance( 'Eoam::EoamProfile',
                                             self.profileName_ )
         copyEoamProfile( prevEoamProfile, self.profile_ )
         self.previousEntry_ = prevEoamProfile

   def copyEditEntry( self ):
      newEoamProfile =  Tac.newInstance( 'Eoam::EoamProfile', self.profileName_ )
      copyEoamProfile( newEoamProfile, self.profile_ )
      self.currentEntry_ = newEoamProfile
      return newEoamProfile

   def newEditEntry( self ):
      newEoamProfile = Tac.newInstance( 'Eoam::EoamProfile', self.profileName_ )
      self.currentEntry_ = newEoamProfile

   def profileName( self ):
      return self.profileName_

   def currentEntry( self ):
      return self.currentEntry_

   def eoamProfile( self ):
      return self.profile_

   def commit( self ):
      # Commit current profile
      eoamProfileConfig = eoamConfig.eoamProfile
      intfToProfileName = eoamConfig.intfToProfileName
      if self.profile_ is None:
         self.profile_ = eoamProfileConfig.newMember( self.profileName_ )
      self.profile_ = eoamProfileConfig.get( self.profileName_ )
      if not self.profile_:
         self.profile_ = Tac.newInstance( 'Eoam::EoamProfile',
                                          self.profileName_ )

      if self.currentEntry_:
         self.profile_.recoveryTimeoutSecs = self.currentEntry_.recoveryTimeoutSecs
         for errorType in self.currentEntry_.linkMonitorConfig:
            if errorType not in self.profile_.linkMonitorConfig:
               self.profile_.linkMonitorConfig.newMember( errorType )
            self.profile_.linkMonitorConfig[ errorType ].threshold = \
                   self.currentEntry_.linkMonitorConfig[ errorType ].threshold
            self.profile_.linkMonitorConfig[ errorType ].action = \
                   self.currentEntry_.linkMonitorConfig[ errorType ].action
            self.profile_.linkMonitorConfig[ errorType ].period = \
                   self.currentEntry_.linkMonitorConfig[ errorType ].period

         for errorType in self.profile_.linkMonitorConfig:
            if errorType not in self.currentEntry_.linkMonitorConfig:
               del self.profile_.linkMonitorConfig[ errorType ]
         if not self.profile_.linkMonitorConfig and \
             self.profile_.recoveryTimeoutSecs == 0:
            self.profile_ = None
      else:
         self.profile_ = None

      for intf in intfToProfileName:
         if intfToProfileName[ intf ] == self.profileName_:
            applyProfile( self.mode, self.profileName_, intf )

class eoamModelet( CliParser.Modelet ):
   modeletParseTree = CliParser.ModeletParseTree()
   def __init__( self, mode ):
      CliParser.Modelet.__init__( self )

   @staticmethod
   def shouldAddModeletRule( mode ):
      return ( mode.intf.name.startswith( 'Ethernet' ) and
               not mode.intf.isSubIntf() )

IntfCli.IntfConfigMode.addModelet( eoamModelet )

def removeProfile( mode, profileName, intfName ):
   intfToProfileName = eoamConfig.intfToProfileName
   # return when non-existent profile is removed
   if intfName not in intfToProfileName:
      return
   if mode and profileName and intfToProfileName[ intfName ] != profileName:
      mode.addWarning( '%s not attached to interface %s' % ( profileName,
                                                             intfName ) )
      return
   del intfToProfileName[ intfName ]

def applyProfile( mode, profileName=None, interfaceName=None, noOrDefaultKw=False ):
   intfList = []
   intfName = None
   prevProfile = None
   eoamProfileConfig = eoamConfig.eoamProfile
   intfToProfileName = eoamConfig.intfToProfileName
   applicableModes = ( IntfCli.IntfConfigMode, )
   if mode:
      if isinstance( mode, IntfCli.IntfConfigMode ):
         intfList = [ mode.intf ]
      elif interfaceName:
         intfList = [ interfaceName ]
   else:
      intfList = [ interfaceName ]
   if noOrDefaultKw:
      for intf in intfList:
         if hasattr( intf, 'name' ):
            intfName = intf.name
         else:
            intfName = intf
         removeProfile( mode, profileName, intfName )
      return
   for intf in intfList:
      if hasattr( intf, 'name' ):
         intfName = intf.name
      else:
         intfName = intf
      if mode and isinstance( mode, applicableModes ) and \
              intfName in intfToProfileName and \
              intfToProfileName[ intfName ] == profileName :
         mode.addWarning( "%s already applied on %s" % ( profileName,
                                                         intfName ) )
         continue
      for eoamProfileName in eoamProfileConfig:
         if intfName in intfToProfileName and \
               intfToProfileName[ intfName ] == eoamProfileName:
            prevProfile = eoamProfileName
      if prevProfile != profileName:
         removeProfile( mode, None, intfName )
      profile = eoamProfileConfig.get( profileName, None )
      if profile and profile.linkMonitorConfig:
         ethIntfStatus = EthIntfCli.ethIntfStatusDir.intfStatus.get( intfName )
         for errorType in profile.linkMonitorConfig:
            if ethIntfStatus and ethIntfStatus.speed == 'speed1Gbps' and \
               profile.linkMonitorConfig[ errorType ].action == \
               tacEoamAction.actionLinkfault:
               if mode and isinstance( mode, applicableModes ):
                  mode.addWarning( "LFS is not supported on 1G interfaces" )
                  continue
      intfToProfileName[ intfName ] = profileName

def copyEoamProfile( newProfile, oldProfile ):
   # Copy old to new
   if oldProfile:
      newProfile.recoveryTimeoutSecs = oldProfile.recoveryTimeoutSecs
      for errorType in oldProfile.linkMonitorConfig:
         newProfile.linkMonitorConfig.newMember( errorType )
         newProfile.linkMonitorConfig[ errorType ].threshold = \
             oldProfile.linkMonitorConfig[ errorType ].threshold
         newProfile.linkMonitorConfig[ errorType ].action = \
             oldProfile.linkMonitorConfig[ errorType ].action
         newProfile.linkMonitorConfig[ errorType ].period = \
             oldProfile.linkMonitorConfig[ errorType ].period

class EoamConfigMode( EoamConfigModeBase, BasicCli.ConfigModeBase ):
   name = "Eoam Configuration Mode"
   modeParseTree = CliParser.ModeParseTree()
   showActiveCmdRegistered_ = True

   def __init__( self, parent, session ):
      EoamConfigModeBase.__init__( self )
      BasicCli.ConfigModeBase.__init__( self, parent, session )

   def onExit( self ):
      BasicCli.ConfigModeBase.onExit( self )

   def abort( self ):
      #pylint: disable-msg=W0201
      self.session_.gotoParentMode()

class EoamProfileMode( EoamProfileModeBase, BasicCli.ConfigModeBase ):
   name = "Eoam Profile Configuration"
   modeParseTree = CliParser.ModeParseTree()
   showActiveCmdRegistered_ = True

   def __init__( self, parent, session, context ):
      assert isinstance( parent, EoamConfigMode )
      self.eoamProfileModeContext = context
      self.profileName_ = context.profileName()
      param = ( self.profileName_ )
      EoamProfileModeBase.__init__( self, param )
      BasicCli.ConfigModeBase.__init__( self, parent, session )

   def onExit( self ):
      self.commitContext()
      BasicCli.ConfigModeBase.onExit( self )

   def abort( self ):
      #pylint: disable-msg=W0201
      self.currentEntry_ = None
      self.eoamProfileModeContext = None
      self.session_.gotoParentMode()

   def commitContext( self ):
      if self.eoamProfileModeContext is None:
         t0( 'commitContext has no context' )
         return

      context = self.eoamProfileModeContext
      self.eoamProfileModeContext = None
      context.commit()

#-------------------------------------------------------------------------------
# link-monitoring config mode. A new instance of this mode is created when the
# user enters "link-error" inside config-eoam-profile mode.
#-------------------------------------------------------------------------------
class LinkMonitoringMode( LinkMonitoringModeBase, BasicCli.ConfigModeBase ):
   #----------------------------------------------------------------------------
   # Attributes required of every Mode class.
   #----------------------------------------------------------------------------
   name = 'Link monitoring mode'
   modeParseTree = CliParser.ModeParseTree()
   showActiveCmdRegistered_ = True

   #----------------------------------------------------------------------------
   # Constructs a new Link Monitoring instance.
   #----------------------------------------------------------------------------
   def __init__( self, parent, session ):
      assert isinstance( parent, EoamProfileMode )
      LinkMonitoringModeBase.__init__( self, parent.profileName_ )
      BasicCli.ConfigModeBase.__init__( self, parent, session )

def gotoEoamConfigMode( mode ):
   childMode = mode.childMode( EoamConfigMode )
   mode.session_.gotoChildMode( childMode )

def gotoEoamProfileMode( mode, args ):
   profileName = args[ "PROFILE_NAME" ]
   eoamProfileConfig = eoamConfig.eoamProfile
   context = EoamProfileModeContext( mode, profileName )
   if profileName in eoamProfileConfig:
      context.copyEditEntry()
   else:
      context.newEditEntry()

   mode.eoamProfileModeContext = context
   childMode = mode.childMode( EoamProfileMode, context=context )
   mode.session_.gotoChildMode( childMode )

def eoamProfileNameRule( mode ):
   eoamProfileConfig = eoamConfig.eoamProfile
   suggestedCompletions = eoamProfileConfig.keys()
   return sorted( suggestedCompletions )

def gotoLinkMonitoringMode( mode, args ):
   childMode = mode.childMode( LinkMonitoringMode )
   mode.session_.gotoChildMode( childMode )

def deleteEoamProfile( mode, args ):
   eoamProfileConfig = eoamConfig.eoamProfile
   profileName = args[ "PROFILE_NAME" ]
   if profileName not in eoamProfileConfig:
      return
   del eoamProfileConfig[ profileName ]

def isDefaultLinkMonitorConfig( profile, errorType ):
   config = profile.linkMonitorConfig[ errorType ]
   return ( config.threshold == 0 and config.action == tacEoamAction.actionLog and
            config.period.val == tacEoamInterval.val )

def doConfigureLinkMonitoringAction( mode, args ):
   mode = mode.parent_
   noOrDefaultKw = CliCommand.isNoOrDefaultCmd( args )
   errorType = args[ 'ERROR_TYPE' ]
   if 'log' in args:
      action = 'log'
   elif 'errdisable' in args:
      action = 'errdisable'
   elif 'linkfault' in args:
      action = 'linkfault'
   else:
      action = None
   profile = mode.eoamProfileModeContext.currentEntry_

   errorType = errorTypeToEnumMap[ errorType ]
   if action:
      action = actionToEnumMap[ action ]
   if noOrDefaultKw:
      if profile and errorType in profile.linkMonitorConfig:
         profile.linkMonitorConfig[ errorType ].action = tacEoamAction.actionLog
         if isDefaultLinkMonitorConfig( profile, errorType ):
            del profile.linkMonitorConfig[ errorType ]
   else:
      if not profile and action != tacEoamAction.actionLog:
         profile = Tac.newInstance( 'Eoam::EoamProfile',
                                    mode.eoamProfileModeContext.profileName_ )
      if errorType and errorType not in profile.linkMonitorConfig and \
         action != tacEoamAction.actionLog:
         profile.linkMonitorConfig.newMember( errorType )

      if profile and errorType in profile.linkMonitorConfig:
         profile.linkMonitorConfig[ errorType ].action = action
         if isDefaultLinkMonitorConfig( profile, errorType ):
            del profile.linkMonitorConfig[ errorType ]

def doConfigureLinkMonitoringThreshold( mode, args ):
   mode = mode.parent_
   noOrDefaultKw = CliCommand.isNoOrDefaultCmd( args )
   errorType = args[ 'ERROR_TYPE' ]
   errorType = errorTypeToEnumMap[ errorType ]
   threshold = args.get( 'THRESHOLD_VALUE' )
   profile = mode.eoamProfileModeContext.currentEntry_
   if noOrDefaultKw:
      if profile and errorType in profile.linkMonitorConfig:
         profile.linkMonitorConfig[ errorType ].threshold = 0
         if isDefaultLinkMonitorConfig( profile, errorType ):
            del profile.linkMonitorConfig[ errorType ]
   else:
      if not profile and threshold:
         profile = Tac.newInstance( 'Eoam::EoamProfile',
                                    mode.eoamProfileModeContext.profileName_ )
      if errorType not in profile.linkMonitorConfig and threshold:
         profile.linkMonitorConfig.newMember( errorType )
      if profile and errorType in profile.linkMonitorConfig:
         profile.linkMonitorConfig[ errorType ].threshold = threshold
         if isDefaultLinkMonitorConfig( profile, errorType ):
            del profile.linkMonitorConfig[ errorType ]

def doConfigureLinkMonitoringPeriod( mode, args, noOrDefaultKw=None, errorType=None,
                                     period=None, periodUnit='secs' ):
   mode = mode.parent_
   noOrDefaultKw = CliCommand.isNoOrDefaultCmd( args )
   errorType = args[ 'ERROR_TYPE' ]
   errorType = errorTypeToEnumMap[ errorType ]
   profile = mode.eoamProfileModeContext.currentEntry_
   if 'secs' in args:
      period = args[ 'PERIOD_SECS' ]
      periodUnit = 'secs'
   elif 'frames' in args:
      period = args[ 'PERIOD_FRAMES' ]
      periodUnit = 'frames'
   else:
      period = None
      periodUnit = None
   if periodUnit:
      periodUnit = intervalUnitToEnumMap[ periodUnit ]
   if noOrDefaultKw:
      if profile and errorType in profile.linkMonitorConfig:
         profile.linkMonitorConfig[ errorType ].period = tacEoamInterval
         if isDefaultLinkMonitorConfig( profile, errorType ):
            del profile.linkMonitorConfig[ errorType ]
   else:
      if not profile and period:
         profile = Tac.newInstance( 'Eoam::EoamProfile',
                                    mode.eoamProfileModeContext.profileName_ )
      eoamInterval = Tac.newInstance( "Eoam::Interval", period,
                                      periodUnit )
      if errorType not in profile.linkMonitorConfig and period:
         profile.linkMonitorConfig.newMember( errorType )
      if profile and errorType in profile.linkMonitorConfig:
         profile.linkMonitorConfig[ errorType ].period = eoamInterval
         if isDefaultLinkMonitorConfig( profile, errorType ):
            del profile.linkMonitorConfig[ errorType ]

def setRecoveryTime( mode, args ):
   mode = mode.parent_
   recoveryTime = args.get( 'TIMEOUT' )
   noOrDefaultKw = CliCommand.isNoOrDefaultCmd( args )
   profile = mode.eoamProfileModeContext.currentEntry_
   if noOrDefaultKw:
      if profile:
         profile.recoveryTimeoutSecs = 0
   else:
      if not profile:
         profile = ( "EoamProfile", )
      profile.recoveryTimeoutSecs = recoveryTime

def thresholdRangeFn( mode ):
   return ( 1, 100 )

def periodFramesRangeFn( mode ):
   return ( 1, 4000 * 1000000 )

def periodSecsRangeFn( mode ):
   return ( 2, 200 )

def timeoutRangeFn( mode ):
   return ( 20, 200 )

def eoamGuard( mode, token ):
   if not eoamHwStatus.eoamSupported:
      return CliParser.guardNotThisPlatform
   return None

def linkErrorGuard( mode, token ):
   if not eoamHwStatus.lfsSupported:
      return CliParser.guardNotThisPlatform
   return None

def linkFaultActionGuard( mode, token ):
   if not eoamHwStatus.lfaultActionSupported:
      return CliParser.guardNotThisPlatform
   return None

matcherEthernet = CliMatcher.KeywordMatcher( 'ethernet',
                     helpdesc='Ethernet protocol configuration' )
nodeOam = CliCommand.Node( matcher=CliMatcher.KeywordMatcher(
   'oam', helpdesc='Operation, Administration and Management' ),
                           guard=eoamGuard )
matcherProfile = CliMatcher.KeywordMatcher( 'profile',
                     helpdesc='EOAM Profile' )

def _showEoamActive( mode ):
   mode.showActive()

def defLinkMonitoring( mode, args ):
   profile = mode.eoamProfileModeContext.currentEntry_
   if profile:
      for key in profile.linkMonitorConfig:
         del profile.linkMonitorConfig[ key ]
      profile.recoveryTimeoutSecs = 0

def defEoamConfig( mode ):
   eoamProfileConfig = eoamConfig.eoamProfile
   eoamProfileConfig.clear()

#------------------------------------------------------------------------
# [ no/default ] monitor ethernet oam in GlobalConfig mode
#------------------------------------------------------------------------

class MonitorEthernetOamCmd( CliCommand.CliCommandClass ):
   syntax = 'monitor ethernet oam'
   noOrDefaultSyntax = syntax
   data = {
      'monitor' :  monitorMatcher,
      'ethernet' : matcherEthernet,
      'oam' : nodeOam,
   }

   @staticmethod
   def handler( mode, args ):
      childMode = mode.childMode( EoamConfigMode )
      mode.session_.gotoChildMode( childMode )

   @staticmethod
   def noOrDefaultHandler( mode, args ):
      eoamProfileConfig = eoamConfig.eoamProfile
      eoamProfileConfig.clear()

BasicCli.GlobalConfigMode.addCommandClass( MonitorEthernetOamCmd )

class ShowActive( ShowCliCommandClass ):
   syntax = 'show active'
   data = { 'active': 'Display the active Eoam Profile configuration' }
   handler = _showEoamActive

#------------------------------------------------------------------------
# show active in EoamConfig mode
#------------------------------------------------------------------------
EoamConfigMode.addShowCommandClass( ShowActive )

#------------------------------------------------------------------------
# profile <profileName> in EoamConfig mode
#------------------------------------------------------------------------

profileNameMatcher = CliMatcher.DynamicNameMatcher(
   lambda mode: eoamConfig.eoamProfile,
   "Eoam Profile Name",
   pattern=r'(?!summary$)[A-Za-z0-9_:{}\[\]-]+' )

class EoamProfileCmd( CliCommand.CliCommandClass ):
   syntax = 'profile PROFILE_NAME'
   noOrDefaultSyntax = syntax
   data = {
      'profile' : matcherProfile,
      'PROFILE_NAME' : profileNameMatcher
   }

   handler = gotoEoamProfileMode
   noOrDefaultHandler = deleteEoamProfile

EoamConfigMode.addCommandClass( EoamProfileCmd )

#------------------------------------------------------------------------
# show active in EoamProfile mode
#------------------------------------------------------------------------
EoamProfileMode.addShowCommandClass( ShowActive )

#------------------------------------------------------------------------
# link-error in EoamProfile mode
#------------------------------------------------------------------------
class LinkErrorCmd( CliCommand.CliCommandClass ):
   syntax = 'link-error'
   noOrDefaultSyntax = syntax
   data = {
      'link-error' : CliCommand.guardedKeyword( 'link-error',
                                       helpdesc='enter link-monitoring mode',
                                       guard=linkErrorGuard ),
   }
   handler = gotoLinkMonitoringMode
   noOrDefaultHandler = defLinkMonitoring

EoamProfileMode.addCommandClass( LinkErrorCmd )

#------------------------------------------------------------------------
# show active in LinkMonitoringMode mode
#------------------------------------------------------------------------
LinkMonitoringMode.addShowCommandClass( ShowActive )

#---------------------------------------------------------------------------
# monitor ethernet oam profile <profileName>
#---------------------------------------------------------------------------
class MonitorEthernetOamProfileCmd( CliCommand.CliCommandClass ):
   syntax = 'monitor ethernet oam profile PROFILE_NAME'
   noOrDefaultSyntax = syntax

   data = {
      'monitor' : 'Configure monitoring',
      'ethernet' : matcherEthernet,
      'oam' : nodeOam,
      'profile' : matcherProfile,
      'PROFILE_NAME' : profileNameMatcher,
      }

   @staticmethod
   def handler( mode, args ):
      noOrDefaultKw = CliCommand.isNoOrDefaultCmd( args )
      applyProfile( mode, profileName=args[ 'PROFILE_NAME' ],
                    noOrDefaultKw=noOrDefaultKw )

   noOrDefaultHandler = handler

eoamModelet.addCommandClass( MonitorEthernetOamProfileCmd )

#------------------------------------------------------------------------
# recovery-time <timeout> in Link Monitoring mode
#------------------------------------------------------------------------
class RecoveryTimeoutCommand( CliCommand.CliCommandClass ):
   syntax = 'recovery-time TIMEOUT'
   noOrDefaultSyntax = 'recovery-time ...'

   data = {
      'recovery-time' : 'Timeout value for recovery',
      'TIMEOUT' : CliMatcher.IntegerMatcher( 20, 200,
                                             helpdesc="Recovery timeout for LFS" ),
   }

   handler = setRecoveryTime
   noOrDefaultHandler = setRecoveryTime

LinkMonitoringMode.addCommandClass( RecoveryTimeoutCommand )

#--------------------------------------------------------------------------
# [ no/default ] <fcs|symbol> threshold value in Link Monitoring mode
#--------------------------------------------------------------------------
class LinkErrorThresholdCommand( CliCommand.CliCommandClass ):
   syntax = 'ERROR_TYPE threshold THRESHOLD_VALUE'
   noOrDefaultSyntax = 'ERROR_TYPE threshold ...'

   data = {
      'ERROR_TYPE' : CliMatcher.EnumMatcher( {
         'fcs' : 'Fcs Errors',
         'symbol' : 'Symbol Errors'
      } ),
      'threshold' : "Link monitoring threshold",
      'THRESHOLD_VALUE' : CliMatcher.IntegerMatcher( 1, 100,
                                       helpdesc="Threshold value in num of errors" ),
   }

   handler = doConfigureLinkMonitoringThreshold
   noOrDefaultHandler = doConfigureLinkMonitoringThreshold

LinkMonitoringMode.addCommandClass( LinkErrorThresholdCommand )

#---------------------------------------------------------------------------------
# [ no/default] <fcs|symbol> action <value> in Link Monitoring mode
#---------------------------------------------------------------------------------
class LinkErrorActionCommand( CliCommand.CliCommandClass ):
   syntax = 'ERROR_TYPE action ( log | errdisable | linkfault )'
   noOrDefaultSyntax = 'ERROR_TYPE action ...'

   data = {
      'ERROR_TYPE' : CliMatcher.EnumMatcher( {
         'fcs' : 'Fcs Errors',
         'symbol' : 'Symbol Errors'
      } ),
      'action' : "Link monitoring action",
      'log' : 'Generate syslog',
      'errdisable' : "Errdisable port",
      'linkfault' : CliCommand.guardedKeyword( 'linkfault',
         helpdesc='Generate link fault', guard=linkFaultActionGuard ),
   }

   handler = doConfigureLinkMonitoringAction
   noOrDefaultHandler = doConfigureLinkMonitoringAction

LinkMonitoringMode.addCommandClass( LinkErrorActionCommand )

#------------------------------------------------------------------------------------
# [ no/default] <fcs|symbol> period <val> <secs | frames >
#------------------------------------------------------------------------------------
class LinkErrorPeriodCommand( CliCommand.CliCommandClass ):
   syntax = 'ERROR_TYPE period ( ( PERIOD_SECS secs ) | ( PERIOD_FRAMES frames ) )'
   noOrDefaultSyntax = 'ERROR_TYPE period ...'

   data = {
      'ERROR_TYPE' : CliMatcher.EnumMatcher( {
         'fcs' : 'Fcs Errors',
         'symbol' : 'Symbol Errors'
      } ),
      'period' : "Link monitoring period",
      'PERIOD_SECS' : CliMatcher.IntegerMatcher( 2, 200,
                                 helpdesc="Link monitoring period in seconds" ),
      'PERIOD_FRAMES' : CliMatcher.IntegerMatcher( 1, 4000 * 1000000,
                                   helpdesc="Link monitoring period in frames" ),
      'secs' : CliMatcher.KeywordMatcher( 'secs',
                                          helpdesc='Monitor errors per N secs' ),
      'frames' : CliMatcher.KeywordMatcher( 'frames',
                                            helpdesc='Monitor errors per N frames' ),
   }

   handler = doConfigureLinkMonitoringPeriod
   noOrDefaultHandler = doConfigureLinkMonitoringPeriod

LinkMonitoringMode.addCommandClass( LinkErrorPeriodCommand )

def showProfile( eoamProfile, operEoamProfile, profileName ):
   print 'Ethernet OAM Profile : %s' % profileName

   if not eoamProfile:
      return

   attr = '%-10s : '
   tab1 = ' ' * 3
   tab2 = ' ' * 12

   for errorType in eoamProfile.linkMonitorConfig:
      threshold = eoamProfile.linkMonitorConfig[ errorType ].threshold
      action = eoamProfile.linkMonitorConfig[ errorType ].action
      periodVal = eoamProfile.linkMonitorConfig[ errorType ].period.val
      periodUnit = eoamProfile.linkMonitorConfig[ errorType ].period.unit
      error = 'fcs'

      if errorType == tacErrorType.errorFcs:
         error = 'fcs'
      elif errorType == tacErrorType.errorSymbol:
         error = 'symbol'

      print ( tab1 + 'Error Type : %s' ) % ( error )
      print ( tab2 + attr + '%5s %7s' ) % ( 'Threshold', threshold, 'frames' )

      actionToString = { tacEoamAction.actionNone : 'none',
                         tacEoamAction.actionLog : 'log',
                         tacEoamAction.actionErrdisable : 'errdisable',
                         tacEoamAction.actionLinkfault : 'linkfault' }

      configActionType = actionToString[ action ]

      operActionType = None
      if operEoamProfile:
         operAction = operEoamProfile.linkMonitorConfig[ errorType ].action
         operActionType = actionToString[ operAction ]
      if operActionType and ( configActionType != operActionType ):
         print ( tab2 + attr + ' %13s' + ' ( Configured : %s ) ' ) % \
            ( 'Action', operActionType, configActionType )
      else:
         print ( tab2 + attr + ' %13s' ) % ( 'Action', configActionType )

      if periodUnit == tacIntervalUnit.intervalSecs:
         intervalType = 'seconds'
      elif periodUnit == tacIntervalUnit.intervalFrames:
         intervalType = 'frames'
      print ( tab2 + attr + '%5s %7s' ) % ( 'Period', periodVal, intervalType )

   recoveryTimeout = eoamProfile.recoveryTimeoutSecs
   print tab1 + 'Recovery Timeout : %s' % recoveryTimeout

def showEoamProfile( mode, args ):
   profileName = args.get( "PROFILE_NAME" )
   eoamProfileColl = eoamConfig.eoamProfile
   profileNames = eoamProfileColl.keys()
   if profileName is None:
      for pName, eoamProfile in eoamConfig.eoamProfile.iteritems():
         operEoamProfile = eoamStatus.operEoamProfile[ pName ]
         showProfile( eoamProfile, operEoamProfile, pName )
   elif profileName in profileNames:
      eoamProfile = eoamProfileColl[ profileName ]
      operEoamProfile = eoamStatus.operEoamProfile.get( profileName, None )
      showProfile( eoamProfile, operEoamProfile, profileName )
   else:
      showProfile( None, None, profileName )

def showEoamProfileSummary( mode, args ):
   profileName = args.get( "PROFILE_NAME" )
   eoamProfileColl = eoamConfig.eoamProfile
   profileNames = eoamProfileColl.keys()
   intfToProfileName = eoamConfig.intfToProfileName
   eoamProfileNames = [ ]
   if profileName:
      if profileName in profileNames:
         eoamProfileNames = [ profileName ]
   else:
      for intf in intfToProfileName:
         eoamProfileNames.append( intfToProfileName.get( intf ) )

   eoamProfileNames = list( set( eoamProfileNames ) )

   for key in eoamProfileNames:
      print 'Eoam Profile : %s' % key
      intfs = [ ]
      for intf in intfToProfileName:
         if intfToProfileName[ intf ] == key:
            intfs.append( intf )

      intfList = intfListToCanonical( intfs )
      print ( ' '*3 + 'Configured on: ' ),

      if intfList:
         print intfList[ 0 ]
      else:
         print

#------------------------------------------------------------------------------------
# show eoam profile [ <name> ]
#------------------------------------------------------------------------------------
class ShowEoamProfile( ShowCommand.ShowCliCommandClass ):
   syntax = 'show monitor ethernet oam profile [ PROFILE_NAME ]'
   data = {
      'monitor' : CliToken.Monitor.monitorMatcherForShow,
      'ethernet' : matcherEthernet,
      'oam' : nodeOam,
      'profile' : matcherProfile,
      'PROFILE_NAME' : profileNameMatcher,
      }
   handler = showEoamProfile
   privileged = True

BasicCli.addShowCommandClass( ShowEoamProfile )

#------------------------------------------------------------------------------------
# show eoam profile [ <name> ] summary
#------------------------------------------------------------------------------------
class ShowEoamProfileSummary( ShowCommand.ShowCliCommandClass ):
   syntax = 'show monitor ethernet oam profile [ PROFILE_NAME ] summary'
   data = {
      'monitor' : CliToken.Monitor.monitorMatcherForShow,
      'ethernet' : matcherEthernet,
      'oam' : nodeOam,
      'profile' : matcherProfile,
      'PROFILE_NAME' : profileNameMatcher,
      'summary' : 'Eoam profile summary',
      }
   handler = showEoamProfileSummary
   privileged = True

BasicCli.addShowCommandClass( ShowEoamProfileSummary )

def Plugin( entityManager ):
   global eoamConfig, eoamStatus, eoamHwStatus
   eoamConfig = ConfigMount.mount( entityManager, "eoam/config", "Eoam::Config",
                                   "w" )
   eoamStatus = LazyMount.mount( entityManager, "eoam/status", "Eoam::Status", "r" )
   eoamHwStatus = LazyMount.mount( entityManager, "eoam/hardware/status/global",
                                   "Eoam::HwStatus", "r" )
   IntfCli.Intf.registerDependentClass( EoamIntfJanitor )
