# Copyright (c) 2007-2011 Arista Networks, Inc.  All rights reserved.
# Arista Networks, Inc. Confidential and Proprietary.

import CliParser
import ConfigMount
import EntityMib
import EnvironmentModels
import LazyMount
import Tac
import Tracing
from ArPyUtils import naturalOrderKey

t0 = Tracing.trace0

DEFAULT_TEMPERATURE_POLL_INTERVAL = 5

temperatureConfig = None
temperatureStatus = None
archerCellTempStatus = None
archerSystemTempStatus = None
powerStatusDir = None
coolingConfig = None
archerCoolingStatus = None
thermostatConfig = None
thermostatStatus = None
entityMib = None

#--------------------------------------------------
# 'show system environment temperature'
#
# [legacy]
# 'show environment temperature'
#--------------------------------------------------

def tempSanitize( temperature ):
   # We cannot set model attributes to -infinity
   return temperature if temperature != float( "-INFINITY" ) else None

def _populateModuleTempSensorInfo( modelSlot, tempSensors, tempConfig,
                                   archerCellStatus, archerSystemStatus,
                                   detail,
                                   isXcvrDomTempSensor=False ):

   tempSensors.sort( key=lambda ts: ts.relPos )
   for tempSensor in tempSensors:
      name = EntityMib.componentName( tempSensor )
      ths = thermostatStatus.tempSensorPidStatus.get( name )
      tsc = tempConfig.tempSensor.get( name )
      tss = archerSystemStatus.get( name )
      if not tss:
         for cell in archerCellStatus.itervalues():
            tss = cell.get( name )
            if tss:
               break

      if not tsc or not tss:
         # sensor has been removed
         continue

      sensor = EnvironmentModels.TempSensor( _renderSensorDetail=bool( detail ),
                                             _isXcvrDomTempSensor=bool(
                                                isXcvrDomTempSensor ) )

      sensor.relPos = tempSensor.label
      sensor.name = name
      sensor.description = tsc.description
      if tss.hwStatus != 'disabled':
         sensor.currentTemperature = tempSanitize( tss.temperature )
         sensor.maxTemperature = tempSanitize( tss.maxTemperature )
      sensor.overheatThreshold = tempSanitize( tsc.overheatThreshold )
      sensor.criticalThreshold = tempSanitize( tsc.criticalThreshold )
      sensor.targetTemperature = tempSanitize( tsc.targetTemperature )

      #Tac.now() is a monotonic counter, we need to convert the value into utc
      #converting a given TS into utc
      #TS in utc = Tac.utcNow() - ( Tac.noww() - (TS in monotonic time ) )
      utcConversionFactor = Tac.utcNow() - Tac.now()

      # There is a short period (a few seconds ) of time when the thermostat agent
      # has not added the new temp sensor to the tempSensorPidStatus list yet.
      if ths is not None:
         sensor.setPointTemperature = tempSanitize( ths.setPointTemperature )
         sensor.isPidDriver = bool( ths.pidDriver )
         sensor.pidDriverCount = ths.pidDriverCount

         if ths.lastPidDriverTime != 0:
            sensor.lastPidDriverTime = ( utcConversionFactor \
                                         + ths.lastPidDriverTime )

      sensor.hwStatus = tss.hwStatus
      if tss.maxTemperatureTime != 0:
         sensor.maxTemperatureLastChange = utcConversionFactor \
                                                      + tss.maxTemperatureTime
      sensor.inAlertState = bool( tss.alertRaised )
      sensor.alertCount = tss.alertRaisedCount

      if tss.lastAlertRaisedTime != 0:
         sensor.alertLastChange = ( utcConversionFactor + tss.lastAlertRaisedTime )

      modelSlot.tempSensors.append( sensor )

def _powerLoss( slot ):
   supply = slot.powerSupply
   psName = 'PowerSupply%s' % supply.label
   supplyStatus = powerStatusDir.get( psName )
   return supplyStatus.state == 'powerLoss'

def doShowTemperature( mode, args ):
   module = args.get( 'MODULE' )
   detail = 'detail' in args
   # module is a named tuple with slot, tag and label as its members
   entityMibRoot = entityMib.root
   model = EnvironmentModels.SystemTemperature( _renderDetail=detail,
         _renderModule=bool( module ) )
   model.systemStatus = thermostatStatus.temperatureAlarmLevel
   model.shutdownOnOverheat = thermostatConfig.shutdownOnOverheat

   def _populatePsSensors( model ):
      powerSupplySlots = entityMibRoot.powerSupplySlot.values()
      powerSupplySlots.sort( key=lambda slot: slot.relPos )

      for slot in powerSupplySlots:
         supply = slot.powerSupply
         if supply:
            modelPsSlot = EnvironmentModels.FruSlot()
            tempSensors = [ sensor for sensor in supply.sensor.values()
                            if sensor.tag == "TempSensor" ]
            if tempSensors:
               # Power supply is inserted
               modelPsSlot.entPhysicalClass = slot.tag
               modelPsSlot.relPos = slot.label
               _populateModuleTempSensorInfo( modelPsSlot, tempSensors,
                                              temperatureConfig,
                                              archerCellTempStatus,
                                              archerSystemTempStatus,
                                              detail=detail )
               model.powerSupplySlots.append( modelPsSlot )

   if entityMibRoot is None or entityMibRoot.initStatus != "ok":
      if mode.session_ and \
            mode.session_.entityManager_.redundancyStatus().mode == 'active':
         raise CliParser.InformationalShowCmdError(
               "System is not yet initialized." )

   if module:
      tempSensors = []
      modelCardSlot = EnvironmentModels.FruSlot()
      if module.tag == 'PowerSupply':
         if module.powerSupply:
            modelCardSlot.entPhysicalClass = module.tag
            modelCardSlot.relPos = module.label
            tempSensors = [ sensor for sensor in module.powerSupply.sensor.values() \
                               if sensor.tag == "TempSensor" ]
      elif module.slot.card:
         modelCardSlot.entPhysicalClass = module.slot.tag
         modelCardSlot.relPos = module.slot.label
         tempSensors = [ sensor for sensor in module.slot.card.sensor.values()
                         if sensor.tag == "TempSensor" ]
      if tempSensors:
         _populateModuleTempSensorInfo( modelCardSlot, tempSensors,
                                        temperatureConfig, archerCellTempStatus,
                                        archerSystemTempStatus, detail=detail )
      else:
         raise CliParser.InformationalShowCmdError( "Invalid module" )
      model.cardSlots.append(modelCardSlot)

   elif entityMibRoot.tacType.fullTypeName == "EntityMib::Chassis":
      cardSlots = entityMibRoot.cardSlot.values()
      cardSlots.sort( key=lambda slot: slot.relPos )
      for slot in cardSlots:
         if slot.card:
            # Card is inserted
            modelCardSlot = EnvironmentModels.FruSlot()
            modelCardSlot.entPhysicalClass = slot.tag
            modelCardSlot.relPos = slot.label
            tempSensors = [ sensor for sensor in slot.card.sensor.values()
                            if sensor.tag == "TempSensor" ]
            if tempSensors:
               _populateModuleTempSensorInfo( modelCardSlot, tempSensors,
                                              temperatureConfig,
                                              archerCellTempStatus,
                                              archerSystemTempStatus,
                                              detail=detail )
            model.cardSlots.append(modelCardSlot)
      _populatePsSensors( model )
   elif entityMibRoot.tacType.fullTypeName == "EntityMib::FixedSystem":
      tempSensors = [ sensor for sensor in entityMibRoot.sensor.values()
                      if sensor.tag == "TempSensor" ]
      #We can pass model here instead of a CardSlot since model also
      #has an optional TempSensors List (for fixed systems)
      _populateModuleTempSensorInfo( model, tempSensors, temperatureConfig,
                                     archerCellTempStatus, archerSystemTempStatus,
                                     detail=detail )
      _populatePsSensors( model )

   return model

#--------------------------------------------------
# 'show system environment temperature transceiver'
#
# [legacy]
# 'show environment temperature transceiver'
#--------------------------------------------------

def doShowTemperatureXcvr( mode, args ):
   module = args.get( 'MODULE' )
   detail = 'detail' in args
   # module is a named tuple with slot, tag and label as its members
   entityMibRoot = entityMib.root
   model = EnvironmentModels.SystemXcvrTemperature( _renderDetail=detail,
                                                    _renderModule=bool( module ) )

   if entityMibRoot is None or entityMibRoot.initStatus != "ok":
      if mode.session_ and \
            mode.session_.entityManager_.redundancyStatus().mode == 'active':
         raise CliParser.InformationalShowCmdError(
               "System is not yet initialized." )

   def _getXcvrDomTempSensor( xcvrSlots ):
      if xcvrSlots == None:
         return []
      xcvrSlots = xcvrSlots.values()
      xcvrSlots.sort( key=lambda slot: slot.relPos )
      tempSensors = []
      for slot in xcvrSlots:
         if slot.xcvr != None:
            sensors = slot.xcvr.sensor
            for sensor in sensors.values():
               if sensor.tag == "DomTemperatureSensor":
                  tempSensors.append( sensor )
      return tempSensors

   if module:
      modelCardSlot = EnvironmentModels.FruSlot()
      if module.tag == 'PowerSupply':
         pass
      elif module.slot.card:
         modelCardSlot.entPhysicalClass = module.slot.tag
         modelCardSlot.relPos = module.slot.label
         xcvrDomTempSensors = _getXcvrDomTempSensor( module.slot.card.xcvrSlot )
         _populateModuleTempSensorInfo( modelCardSlot,
                                        xcvrDomTempSensors, temperatureConfig,
                                        archerCellTempStatus,
                                        archerSystemTempStatus, detail=detail,
                                        isXcvrDomTempSensor=True )
         model.cardSlots.append( modelCardSlot )
   elif entityMibRoot.tacType.fullTypeName == "EntityMib::Chassis":
      cardSlots = entityMibRoot.cardSlot.values()
      cardSlots.sort( key=lambda slot: slot.relPos )
      for slot in cardSlots:
         if slot.card:
            # Card is inserted
            modelCardSlot = EnvironmentModels.FruSlot()
            if slot.tag.lower().find( "linecard" ) == -1:
               continue
            modelCardSlot.entPhysicalClass = slot.tag
            modelCardSlot.relPos = slot.label
            xcvrDomTempSensors = _getXcvrDomTempSensor( slot.card.xcvrSlot )
            _populateModuleTempSensorInfo( modelCardSlot, xcvrDomTempSensors,
                                           temperatureConfig,
                                           archerCellTempStatus,
                                           archerSystemTempStatus,
                                           detail=detail,
                                           isXcvrDomTempSensor=True )
            model.cardSlots.append( modelCardSlot )
   elif entityMibRoot.tacType.fullTypeName == "EntityMib::FixedSystem":
      xcvrDomTempSensors = _getXcvrDomTempSensor( entityMibRoot.xcvrSlot )
      _populateModuleTempSensorInfo( model, xcvrDomTempSensors,
                                     temperatureConfig, archerCellTempStatus,
                                     archerSystemTempStatus, detail=detail,
                                     isXcvrDomTempSensor=True )

   return model

#--------------------------------------------------
# 'show system environment cooling'
#
# [legacy]
# 'show environment cooling'
#--------------------------------------------------

def _populateFanTrayInfo( coolingModel, fanTraySlots, powerSupplySlots, coolConfig,
      archerCoolStatus, thermoStatus ):
   def labelNaturalOrderKey( item ):
      return naturalOrderKey( item.label )

   def _populateFanTrayInfoHelper( modelSlot, slot, labelPrefix="" ):
      supplyStatus = None
      if hasattr( slot, "fanTray" ):
         fanTray = slot.fanTray
      elif hasattr( slot, "powerSupply" ):
         fanTray = slot.powerSupply
         if fanTray:
            psName = 'PowerSupply%s' % fanTray.label
            supplyStatus = powerStatusDir.get( psName )
      else:
         assert 0, "Invalid parameter passed to _printFanTrayInfoHelper"
      if not fanTray or fanTray.initStatus != "ok":
         modelSlot.label = labelPrefix + EntityMib.formatEntityMibName(
            EntityMib.componentName( slot ) )
         modelSlot.status = "notInserted"
      else:
         fans = fanTray.fan.values()
         fans.sort( key=labelNaturalOrderKey )
         modelSlot.label = labelPrefix + EntityMib.formatEntityMibName(
            EntityMib.componentName( fanTray ) )
         _populateFanAggregatedInfo( modelSlot, fans, coolConfig, thermoStatus,
                                     archerCoolStatus, supplyStatus )
         for fan in fans:
            modelFan = EnvironmentModels.Fan()
            _populateFanDetailedInfo( modelFan, fan, coolConfig,
                                      thermoStatus, archerCoolStatus,
                                      supplyStatus )
            modelFan.label = labelPrefix + EntityMib.formatEntityMibName(
               EntityMib.componentName( fan ) )
            modelSlot.fans.append( modelFan )

   # populate the info for all fan tray slots
   fanTraySlots.sort( key=labelNaturalOrderKey )
   for fanTraySlot in fanTraySlots:
      modelFanCollection = EnvironmentModels.FanCollection()
      _populateFanTrayInfoHelper( modelFanCollection, fanTraySlot )
      coolingModel.fanTraySlots.append( modelFanCollection )

   # populate info for all power supply slots if they are managed
   powerSupplySlots.sort( key=labelNaturalOrderKey )
   for powerSupplySlot in powerSupplySlots:
      if _powerSupplyFansAreManaged( powerSupplySlot, coolConfig ):
         modelPsSlot = EnvironmentModels.FanCollection()
         _populateFanTrayInfoHelper( modelPsSlot, powerSupplySlot,
                                             labelPrefix="PowerSupply" )
         coolingModel.powerSupplySlots.append( modelPsSlot )

def _powerSupplyFansAreManaged( powerSupplySlot, coolConfig ):
   if not powerSupplySlot.powerSupply:
      # Not inserted. Assume fans are managed
      return True
   for fan in powerSupplySlot.powerSupply.fan.values():
      name = EntityMib.componentName( fan )
      fanConfig = coolConfig.fan.get( name )
      if fanConfig.managed:
         # If 1 fan is managed, assume they're all managed
         return True
   return False

# get the fan's status and speed
def _fanStatusStr( fanConfig, fanStatus, supplyStatus=None ):
   if not fanStatus or not fanConfig or fanStatus.hwStatus == "unknownHwStatus":
      return "unknownHwStatus"
   elif not fanConfig.properties:
      return "unsupported"
   elif fanStatus.hwStatus == "failed":
      return "failed"
   elif ( fanConfig.offOnPowerLoss and
          supplyStatus and supplyStatus.state == "powerLoss" ):
      return "powerLoss"
   else:
      return "ok"

def _populateFanAggregatedInfo( modelSlot, fans, coolConfig,
                       thermoStatus, archerCoolStatus, supplyStatus=None ):

   # Get the fan speed from one of the fans
   name = EntityMib.componentName( fans[ 0 ] )
   fanConfig = coolConfig.fan.get( name )
   fanStatus = archerCoolStatus.get( name )
   if fanConfig:
      if not fanConfig.readOnly:
         modelSlot.speed = int( thermoStatus.fanConfig.get( name ).speed )
      if ( fanConfig.offOnPowerLoss and
           supplyStatus and supplyStatus.state == "powerLoss" ):
         modelSlot.speed = None

   fanStatusStrs = []
   for fan in fans:
      name = EntityMib.componentName( fan )
      fanConfig = coolConfig.fan.get( name )
      fanStatus = archerCoolStatus.get( name )
      fanStatusStrs.append( _fanStatusStr( fanConfig, fanStatus, supplyStatus ) )

   sev = { "failed" : 0,
           "unsupported" : 1,
           "unknownHwStatus" : 2,
           "powerLoss" : 3,
           "ok" : 4 }
   fanStatusStrs.sort( key=lambda s: sev[ s ] )
   modelSlot.status = fanStatusStrs[0]

def _populateFanDetailedInfo( modelFan, fan, coolConfig,
                       thermoStatus, archerCoolStatus, supplyStatus=None ):
   name = EntityMib.componentName( fan )
   fanConfig = coolConfig.fan.get( name )
   fanStatus = archerCoolStatus.get( name )

   if fanConfig and not fanConfig.readOnly:
      modelFan.configuredSpeed = int( thermoStatus.fanConfig.get( name ).speed )
      if fanConfig.properties:
         modelFan.maxSpeed = int( fanConfig.properties.maxRpm )
   if fanStatus:
      modelFan.actualSpeed = int( fanStatus.speed )

   modelFan.status = _fanStatusStr( fanConfig, fanStatus, supplyStatus )
   if modelFan.status == "ok" and fanStatus:
      # Saved TS was in local time, convert that to UTC
      deltaTime = Tac.now() - fanStatus.lastHwStatusChangeTime
      modelFan.uptime = Tac.utcNow() - deltaTime
      modelFan.speedStable = fanStatus.speedStable
      if modelFan.speedStable:
         deltaTime = Tac.now() - fanStatus.lastSpeedStableChangeTime
         modelFan.lastSpeedStableChangeTime = Tac.utcNow() - deltaTime
      modelFan.speedHwOverride = fanStatus.fanSpeedOverriddenByHardware

   if ( fanConfig.offOnPowerLoss and
        supplyStatus and supplyStatus.state == "powerLoss" ):
      modelFan.configuredSpeed = None
      modelFan.actualSpeed = None

def doShowCooling( mode, args ):
   entityMibRoot = entityMib.root
   model = EnvironmentModels.SystemCooling( _renderFanDetail='detail' in args )
   if entityMibRoot is None or entityMibRoot.initStatus != "ok":
      if mode.session_ and \
            mode.session_.entityManager_.redundancyStatus().mode == 'standby':
         raise CliParser.InformationalShowCmdError(
               "This command only works on the active supervisor." )
      else: 
         raise CliParser.InformationalShowCmdError(
               "System is not yet initialized." )

   model.systemStatus = thermostatStatus.coolingAlarmLevel
   model.ambientTemperature = tempSanitize( thermostatStatus.inletTemperature )
   model.airflowDirection = thermostatStatus.airflowDirection
   model.shutdownOnInsufficientFans = \
         thermostatConfig.shutdownOnInsufficientFans
   model.coolingMode = thermostatConfig.mode
   model.overrideFanSpeed = int( thermostatConfig.fanSpeed )
   model.minFanSpeed = int( thermostatConfig.userMinFanSpeed )

   fanTraySlots = entityMibRoot.fanTraySlot.values()
   powerSupplySlots = entityMibRoot.powerSupplySlot.values()
   _populateFanTrayInfo( model, fanTraySlots, powerSupplySlots, coolingConfig,
         archerCoolingStatus, thermostatStatus )
   return model

#--------------------------------------------------
# 'show system environment all'
#
# [legacy]
# 'show environment all'
#--------------------------------------------------

def doShowEnvironmentAll( mode, args ):
   cmds = [ "show system environment temperature",
            "show system environment cooling",
            "show system environment power" ]
   for cmd in cmds:
      try:
         mode.session_.runCmd( cmd, aaa=False )
         print
      except CliParser.InvalidInputError:
         pass

#--------------------------------------------------
# 'environment fan-speed override <fan speed>'
# 'environment fan-speed minimum <fan speed>'
# 'environment fan-speed auto'
# '[no|default] environment fan-speed'
#--------------------------------------------------

def overrideFanSpeed( mode, args ):
   print "===================================================================="
   print "WARNING: Overriding the system fan speed is unsupported and should only "
   print "be done under the direction of an Arista Networks engineer."
   print "You can risk damaging hardware by setting the fan speed too low"
   print "and doing so without direction from Arista Networks can be grounds"
   print "for voiding your warranty."
   print "To set the fan speed back to automatic mode, use the"
   print "'environment fan-speed auto' command"
   print "===================================================================="
   fanSpeed = args.get( 'FANSPEED' )
   thermostatConfig.fanSpeed = fanSpeed
   thermostatConfig.mode = 'manual'

def setMinFanSpeed( mode, args ):
   fanSpeed = args.get( 'FANSPEED' )
   thermostatConfig.userMinFanSpeed = fanSpeed

def noMinFanSpeed( mode, args ):
   thermostatConfig.userMinFanSpeed = 0

def autoFanSpeed( mode, args ):
   thermostatConfig.mode = 'automatic'
   thermostatConfig.fanSpeed = 0

#--------------------------------------------------
# '[no|default] environment overheat action (shutdown|ignore)'
# '[no|default] environment insufficient-fans action (shutdown|ignore)'
#--------------------------------------------------

warning = """====================================================================
WARNING: Overriding the system shutdown behavior when the system
%s is unsupported and should only be done under
the direction of an Arista Networks engineer. You risk damaging
hardware by not shutting down the system in this situation, and doing
so without direction from Arista Networks can be grounds for voiding
your warranty. If in any circumstances the system loses all the fans,
it will shut down to prevent accidental burnout.
To re-enable the shutdown-on-overheat behavior, use the
'environment %s action shutdown' command.
===================================================================="""

def setShutdownOnOverheat( mode, args ):
   # the default is that we shutdown on insufficient fans
   action = args.get( 'ACTION', 'shutdown' )
   if action == 'ignore':
      mode.addWarning( warning % ( "is overheating", "overheat" ) )
   thermostatConfig.shutdownOnOverheat = ( action == 'shutdown' )

def setShutdownOnInsufficientFans( mode, args ):
   # the default is that we shutdown on insufficient fans
   action = args.get( 'ACTION', 'shutdown' )
   if action == 'ignore':
      mode.addWarning( warning % ( "has insufficient fans inserted",
                                   "insufficient-fans" ) )
   thermostatConfig.shutdownOnInsufficientFans = ( action == 'shutdown' )

#------------------------------------------------------
# Register env show commands into "show tech-support".
#------------------------------------------------------
import CliPlugin.TechSupportCli              # pylint: disable-msg=F0401

def _showTechCmds():
   cmds = [ 'show system env cooling',
            'show system env temperature',
            'show system env temperature transceiver' ]
   return cmds

# Timestamps are made up to maintain historical order within show tech-support
CliPlugin.TechSupportCli.registerShowTechSupportCmdCallback(
      '2010-01-01 00:15:00',
      _showTechCmds,
      summaryCmdCallback=lambda: [
         "show system env temperature transceiver",
         "show system environment all",
         ] )

#--------------------------------------------------------------------------------
# '[ no | default ] environment temperature invalid action ignore'
# '[ no | default ] environment temperature transceiver third-party invalid
#  action ignore'
#--------------------------------------------------------------------------------

def setIgnoreTempTrue( mode, args ):
   thermostatConfig.ignoreInvalidTemp = True

def setIgnoreTempFalse( mode, args ):
   thermostatConfig.ignoreInvalidTemp = False

def setIgnoreThirdPartyXcvrTrue( mode, args ):
   thermostatConfig.ignoreInvalidThirdPartyXcvrTemp = True

def setIgnoreThirdPartyXcvrFalse( mode, args ):
   thermostatConfig.ignoreInvalidThirdPartyXcvrTemp = False

#--------------------------------------------------------------------------------
# [ no | default ] environment temperature poll-interval
#--------------------------------------------------------------------------------
def setTemperaturePollInterval( mode, args ):
   interval = args.get( 'INTERVAL', DEFAULT_TEMPERATURE_POLL_INTERVAL )
   if 'milliseconds' in args:
      interval /= 1000.0
   thermostatConfig.pollInterval = interval

#--------------------------------------------------
# Plugin method - Mount the objects we need from Sysdb
#--------------------------------------------------
def Plugin( entityManager ):
   global temperatureConfig, coolingConfig
   global archerCellTempStatus, archerSystemTempStatus
   global archerCoolingStatus, powerStatusDir
   global thermostatConfig, thermostatStatus, entityMib
   temperatureConfig = LazyMount.mount( entityManager,
                                        "environment/temperature/config",
                                        "Environment::Temperature::Config", "r" )
   archerCellTempStatus = LazyMount.mount( 
      entityManager, "environment/archer/temperature/status/cell", "Tac::Dir", "ri" )
   archerSystemTempStatus = LazyMount.mount( 
      entityManager, "environment/archer/temperature/status/system",
      "Tac::Dir", "ri" )
   coolingConfig = LazyMount.mount( entityManager, "environment/cooling/config",
                                    "Environment::Cooling::Config", "r" )
   archerCoolingStatus = LazyMount.mount(
      entityManager, "environment/archer/cooling/status", "Tac::Dir", "ri" )
   thermostatConfig = ConfigMount.mount( entityManager,
                                         "environment/thermostat/config",
                                         "Environment::Thermostat::Config", "w" )
   thermostatStatus = LazyMount.mount( entityManager,
                                       "environment/thermostat/status",
                                       "Environment::Thermostat::Status", "r" )
   powerStatusDir = LazyMount.mount( entityManager,
                                     'environment/archer/power/status/powerSupply',
                                     'Tac::Dir', 'ri' )
   entityMib = LazyMount.mount( entityManager, "hardware/entmib",
                                "EntityMib::Status", "r" )
