# Copyright (c) 2009, 2010 Arista Networks, Inc.  All rights reserved.
# Arista Networks, Inc. Confidential and Proprietary.

import BasicCli
import CliCommand
import CliMatcher
import CliParser
import ConfigMount
import DesCrypt
from CliMode.Email import EmailMode
from CliPlugin.VrfCli import VrfExprFactory, DEFAULT_VRF

emailConfig = None

matcherAuth = CliMatcher.KeywordMatcher( 'auth',
      helpdesc='Email account authentication' )

# --------------------------------------------------------------------------
# The email mode
# --------------------------------------------------------------------------
class EmailConfigMode( EmailMode, BasicCli.ConfigModeBase ):
   name = "Email configuration"
   modeParseTree = CliParser.ModeParseTree()
   
   def __init__( self, parent, session ):
      EmailMode.__init__( self, param = " " )
      BasicCli.ConfigModeBase.__init__( self, parent, session )

class GotoEmailModeCmd( CliCommand.CliCommandClass ):
   syntax = 'email'
   data = { 'email': 'Configure email client' }

   @staticmethod
   def handler( mode, args ):
      childMode = mode.childMode( EmailConfigMode )
      mode.session_.gotoChildMode( childMode )

BasicCli.GlobalConfigMode.addCommandClass( GotoEmailModeCmd )

# --------------------------------------------------------------------------
# The "[no|default] server [ vrf VRF ] SERVER" command, in email mode.
# --------------------------------------------------------------------------
def setServer( mode, args ):
   server = args[ 'SERVER' ]
   hostAndPort = server.split( ':', 1 )
   try:
      port = int( hostAndPort[ 1 ] )
      if not 0 < port < 65536:
         raise ValueError # Go to the error-and-stop.
   except ValueError:
      mode.addErrorAndStop( "Invalid port number" )
   except IndexError: # Wasn't specified; default to 25.
      port = 25

   emailConfig.host = hostAndPort[ 0 ]
   emailConfig.port = port
   emailConfig.vrfName = args.get( 'VRF', DEFAULT_VRF )

def noServer( mode, args ):
   emailConfig.host = ''
   emailConfig.port = 25
   emailConfig.vrfName = DEFAULT_VRF

class ServerCmd( CliCommand.CliCommandClass ):
   syntax = 'server [ VRF ] SERVER'
   noOrDefaultSyntax = 'server ...'
   data = {
      'server': 'Email relay',
      'VRF': VrfExprFactory( helpdesc='Use a specific VRF' ),
      'SERVER': CliMatcher.PatternMatcher( pattern='[^\s:]+(?::\d+)?',
                                           partialPattern='[^\s:]*(?::\d*)?',
                                           helpdesc='E-Mail server / relay',
                                           helpname='HOST:PORT' ),
   }

   handler = setServer
   noOrDefaultHandler = noServer

EmailConfigMode.addCommandClass( ServerCmd )

# --------------------------------------------------------------------------
# The "[no|default] from-user EMAIL_ADDRESS" command, in email mode.
# --------------------------------------------------------------------------

def setFromUser( mode, args ):
   fromUser = args[ 'EMAIL_ADDRESS' ]
   if '@' not in fromUser:
      mode.addError( 'Email address must be fully qualified including @domain' )
      return
   emailConfig.fromUser = fromUser

def noFromUser( mode, args ):
   emailConfig.fromUser = ''

class FromUserCmd( CliCommand.CliCommandClass ):
   syntax = 'from-user EMAIL_ADDRESS'
   noOrDefaultSyntax = 'from-user ...'
   data = {
      'from-user': 'Send email from this user',
      'EMAIL_ADDRESS': CliMatcher.PatternMatcher( pattern='\S+', 
                                    helpdesc='E-Mail address', helpname='WORD' ),
   }
   
   handler = setFromUser
   noOrDefaultHandler = noFromUser

EmailConfigMode.addCommandClass( FromUserCmd )

# --------------------------------------------------------------------------
# The "[no|default] auth username USERNAME" command, in email mode.
# --------------------------------------------------------------------------
def setUser( mode, args ):
   user = args[ 'USERNAME' ]
   emailConfig.user = user

def noUser( mode, args ):
   emailConfig.user = ''

class AuthUsernameCmd( CliCommand.CliCommandClass ):
   syntax = 'auth username USERNAME'
   noOrDefaultSyntax = 'auth username ...'
   data = {
      'auth': 'Email account authentication',
      'username': 'Email login username',
      'USERNAME': CliMatcher.PatternMatcher( pattern='\S+', 
                           helpdesc='Account name string', helpname='WORD' ),
   }
   handler = setUser
   noOrDefaultHandler = noUser

EmailConfigMode.addCommandClass( AuthUsernameCmd )

# --------------------------------------------------------------------------
# The "[no] auth password [0 | 7] PASSWORD" command, in email mode.
# --------------------------------------------------------------------------
def setPassword( mode, args ):
   password = args[ 'PASSWORD' ]
   if '7' in args:
      try:
         password = DesCrypt.decrypt( emailConfig.name + '_cryptkey_email',
                                      password )
      except:
         mode.addError( "Invalid encrypted keystring" )
         return
   
   emailConfig.password = password

def noPassword( mode, args ):
   emailConfig.password = ''

class AuthPasswordCmd( CliCommand.CliCommandClass ):
   syntax = 'auth password [ 0 | 7 ] PASSWORD'
   noOrDefaultSyntax = 'auth password ...'
   data = {
      'auth': matcherAuth,
      'password': 'Email login password',
      '0': CliMatcher.KeywordMatcher( '0', helpdesc='No encryption' ),
      '7': CliMatcher.KeywordMatcher( '7', helpdesc='Symmetric encryption' ),
      'PASSWORD': CliCommand.Node(
                  matcher=CliMatcher.PatternMatcher( pattern='\S+',
                  helpdesc='Password', helpname='WORD' ), sensitive=True ),
   }
   handler = setPassword
   noOrDefaultHandler = noPassword

EmailConfigMode.addCommandClass( AuthPasswordCmd )

# --------------------------------------------------------------------------
# The "[no|default] tls" command, in email mode.
# --------------------------------------------------------------------------

class TlsCmd( CliCommand.CliCommandClass ):
   syntax = 'tls'
   noOrDefaultSyntax = syntax
   data = {
      'tls': 'Require TLS',
   }

   @staticmethod
   def handler ( mode, args ):
      emailConfig.useTls = True
   
   @staticmethod
   def noOrDefaultHandler ( mode, args ):
      emailConfig.useTls = False

EmailConfigMode.addCommandClass( TlsCmd )

# --------------------------------------------------------------------------
def Plugin( entityManager ):
   global emailConfig
   emailConfig = ConfigMount.mount( entityManager, "sys/email/config",
                                    "System::EmailConfig", "w" )
