# Copyright (c) 2005-2011 Arista Networks, Inc.  All rights reserved.
# Arista Networks, Inc. Confidential and Proprietary.

import CliParser, IntfCli, EthIntfCli, PhysicalIntfRule
import CliCommand
import CliMatcher
from CliPlugin.EbraEthIntfCliModel import(
   InterfacesStatus, interfaceStatusHook, Encapsulation )
from CliPlugin.VirtualIntfRule import IntfMatcher
import LazyMount
import SubIntfCli
import BasicCli, Tac, ConfigMount
import CliToken.Switch
from TypeFuture import TacLazyType
from VlanCli import SwitchportModelet
from VlanCli import getSwitchIntfConfigEvenIfDisabled

ethIntfStatusDir = None
bridgingHwCapabilities = None
bridgingConfig = None
cliConfig = None

EncapConfigMode = TacLazyType( "Interface::SubIntfDot1qEncapConfigMode" )

switchPortMatcher = IntfMatcher()
switchPortMatcher |= PhysicalIntfRule.PhysicalIntfMatcher(
   'Ethernet',
   value=lambda mode, intf: EthIntfCli.EthPhyIntf( intf, mode ) )

switchPortIntfTypes = []
switchPortIntfTypes.append( EthIntfCli.EthPhyAutoIntfType )
switchPortIntfTypes.append( EthIntfCli.UnconnectedEthPhyAutoIntfType )

#-------------------------------------------------------------------------------
# The "show interfaces [<name>] status" command, in enable mode.
#  show interfaces [<name>] status
#  show interfaces status module <modnum>
#  show interfaces module <modnum> status
#-------------------------------------------------------------------------------
# returns a tuple. The first parameter is the mode (routed, inactive, bridged
# or trunk). The second parameter returns the VLAN id if applicable otherwise None
def _generateBridgedOrRoutedVlanInfo( intf ):
   switchIntfConfig = bridgingConfig.switchIntfConfig.get( intf.name ) 
   if( not switchIntfConfig or switchIntfConfig.switchportMode == 'routed' ): 
      return ( 'routed', None )
   elif switchIntfConfig.switchportMode == 'access' or \
        switchIntfConfig.switchportMode == 'dot1qTunnel':
      if switchIntfConfig.nativeVlan == 0:
         return ( 'inactive', None )
      else:
         return ( 'bridged', switchIntfConfig.nativeVlan )
   elif switchIntfConfig.switchportMode in ( 'trunk', 'tap', 'tool', 'tapTool' ):
      return ( switchIntfConfig.switchportMode, None )
   else:
      assert False, "unexpected switchportMode"

def getVlanInfo( intf, mode ):
   # If intf isn't bridged then vlan isn't relevant.  Overwrite that field
   # with explanation for why not bridged.
   eis = ethIntfStatusDir.get( intf.name )
   vlanInfo = InterfacesStatus.InterfaceStatusEntry.VlanInformation()
   if intf.isSubIntf():
      subIntfStatus = intf.status()
      if subIntfStatus.isL2SubIntf():
         if subIntfStatus.forwardingVlanId > 0:
            vlanInfo.vlanId = subIntfStatus.forwardingVlanId
            vlanInfo.interfaceMode = "encap"
         else:
            vlanInfo.interfaceMode = "inactive"
      else:
         subIntfConfig = intf.config()
         # We populate encap outer tag into vlanId for single tag L3 subintf
         # to be backwards compatible.
         if not subIntfConfig.dot1qEncap.innerTag:
            vlanInfo.vlanId = subIntfConfig.dot1qEncap.outerTag
         vlanInfo.interfaceMode = "encap"
   elif eis:
      if eis.forwardingModel == "intfForwardingModelRecirculation":
         ( vlanInfo.vlanExplanation , _, _, _ ) = \
               intf.recirculationExplanation( mode, intf.status() )
      elif eis.forwardingModel == "intfForwardingModelDataLink":
         ( vlanInfo.vlanExplanation , _, _, _ ) = \
               intf.dataLinkExplanation( mode, intf.status() )
      elif eis.forwardingModel == "intfForwardingModelQuietDataLink":
         ( vlanInfo.vlanExplanation , _, _, _ ) = \
               intf.quietDataLinkExplanation( mode, intf.status() )
      elif eis.forwardingModel == "intfForwardingModelUnauthorized":
         ( vlanInfo.vlanExplanation , _, _, _ ) = \
               intf.unauthorizedExplanation( mode, intf.status() )
      else:
         ( vlanInfo.interfaceMode, 
           vlanInfo.vlanId ) = _generateBridgedOrRoutedVlanInfo( intf )

   vlanInfo.interfaceForwardingModel = intf.forwardingModel()
   return vlanInfo

def getEncapsulation( intf ):
   encapsulation = None
   if intf.isSubIntf():
      subIntfConfig = intf.config()
      if subIntfConfig.dot1qEncap.outerTag:
         encapsulation = Encapsulation()
         encapsulation.encapsulationType = "dot1q"
         if ( subIntfConfig.dot1qEncapConfigMode ==
              EncapConfigMode.subIntfDot1qEncapConfigLegacy ):
            tags = [ subIntfConfig.dot1qEncap.outerTag ]
            if subIntfConfig.dot1qEncap.innerTag:
               tags.append( subIntfConfig.dot1qEncap.innerTag )
            encapsulation.dot1qVlanTags = tags
         elif ( subIntfConfig.dot1qEncapConfigMode ==
              EncapConfigMode.subIntfDot1qEncapConfigFlexEncap ):
            # Flex Encap information cannot be filled into 'show interface status'
            # due to limited space in the existing encapsulation column.
            # Instead, a * will be printed in the encapsulation column, along
            # with a warning message, directing the operator to use the alternate
            # show command.
            # pylint: disable=protected-access
            encapsulation._incompleteInformation = True
   return encapsulation

showIntfStatusKw = CliMatcher.KeywordMatcher( 'status',
                                      helpdesc='Details on the state of interfaces' )
filterData = { t : CliCommand.singleKeyword( t,
                                             'Include %s interfaces' % t )
               for t in ( 'connected', 'notconnect', 'disabled', 'inactive' ) }
filterData[ 'sub-interfaces' ] = CliCommand.singleKeyword( 'sub-interfaces',
                                                           'Include sub interfaces' )
statusData = dict( status=showIntfStatusKw )
statusData.update( filterData )

def showIntfStatusIncompleteEncapWarningMsg():
   return ( "* Incomplete encapsulation information. "
            "Use 'show interface encapsulation vlan' instead" )

def showInterfacesStatus( mode, args ):
   intf = args.get( 'INTF' )
   mod = args.get( 'MOD' )
   filters = set( args ) & set( filterData )
   if 'sub-interfaces' in filters:
      filters.remove( 'sub-interfaces' )
      subIntfs = 'sub-interfaces'
   else:
      subIntfs = None
   # We need to set exposeInactive to True for Intf.getAll calls, when
   # trying to show only inactive interfaces, otherwise inactive
   # interfaces will be hidden.
   exposeInactive = 'inactive' in filters
   ret = InterfacesStatus()

   intfs = []
   if intf:
      if '.' in intf.__str__():
         intfs = IntfCli.Intf.getAll( mode, intf, mod, SubIntfCli.SubIntf,
                                      exposeInactive=exposeInactive )
      else:
         if subIntfs:
            mode.addError( 'Invalid option: sub-interfaces' )
            return
         intfs = IntfCli.Intf.getAll( mode, intf, mod, EthIntfCli.EthIntf,
                                      exposeInactive=exposeInactive )
   else:
      intfs = IntfCli.Intf.getAll( mode, intf, mod, SubIntfCli.SubIntf,
                                   exposeInactive=exposeInactive )
      if not subIntfs:
         intfs.extend( IntfCli.Intf.getAll( mode, intf, mod, EthIntfCli.EthIntf,
                                            exposeInactive=exposeInactive ) )

   if not intfs:
      return ret
   intfs = [ i for i in intfs if i.lookup() ] # See BUG9124
   if not intfs:
      return ret

   showIncompleteEncapWarning = False
   for x in intfs:
      if filters and x.linkStatus() not in filters:
         continue
      try:
         encapsulation = getEncapsulation( x )
         interfaceStatusEntry = InterfacesStatus.InterfaceStatusEntry(
            description=x.description(),
            linkStatus=x.linkStatus(),
            vlanInformation=getVlanInfo( x, mode ),
            encapsulation=encapsulation,
            bandwidth=x.bandwidth() * 1000,
            autoNegotigateActive=x.autonegActive(),
            autoNegotiateActive=x.autonegActive(),
            duplex=x.getIntfDuplexStr(),
            interfaceType=x.xcvrTypeStr(),
            lineProtocolStatus=x.getStatus()[ 0 ] )
         for hook in interfaceStatusHook.extensions():
            hook( x.name, interfaceStatusEntry )
         ret.interfaceStatuses[ x.name ] = interfaceStatusEntry
         # pylint: disable=protected-access
         if encapsulation._incompleteInformation:
            showIncompleteEncapWarning = True
      except AttributeError:
         # Some interfaces have been deleted during the process
         continue
   if showIncompleteEncapWarning:
      warning = showIntfStatusIncompleteEncapWarningMsg()
      mode.addWarning( warning )
   return ret

class ShowIntfStatus( IntfCli.ShowIntfCommand ):
   syntax = 'show interfaces status [ { %s } ]' % ( ' | '.join( filterData ) )
   data = statusData
   cliModel = InterfacesStatus
   handler = showInterfacesStatus
   moduleAtEnd = True

BasicCli.addShowCommandClass( ShowIntfStatus )

#-------------------------------------------------------------------------------
# The "[no|default] switchport source-interface tx" command, in "config-if" mode.
#
# The full syntax of this command is:
#
#   switchport source-interface tx
#   {no|default} switchport source-interface tx
#-------------------------------------------------------------------------------

def sourceportFilterSupportedGuard( mode, token ):
   if bridgingHwCapabilities.sourceportFilterSupported:
      return None
   return CliParser.guardNotThisPlatform


def disableSourceportFilterMode( mode, args ):
   sic = getSwitchIntfConfigEvenIfDisabled( mode )
   sic.sourceportFilterMode = 'sourceportFilterDisabled'

def defaultSourceportFilterMode( mode, args ):
   sic = getSwitchIntfConfigEvenIfDisabled( mode )
   sic.sourceportFilterMode = 'sourceportFilterEnabled'

class SourceportFilterModeCmd( CliCommand.CliCommandClass ):
   syntax = 'switchport source-interface tx'
   noOrDefaultSyntax = syntax
   data = {
         'switchport' : 'Set switching mode characteristics',
         'source-interface' : CliCommand.guardedKeyword( 'source-interface', 
                                 helpdesc='Configure source interface properties',
                                 guard=sourceportFilterSupportedGuard ),
         'tx' : CliCommand.guardedKeyword( 'tx',
                   helpdesc='Allow bridged packets to go out on the '
                            'source interface',
                   guard=sourceportFilterSupportedGuard )
         }
   handler = disableSourceportFilterMode
   noOrDefaultHandler = defaultSourceportFilterMode

SwitchportModelet.addCommandClass( SourceportFilterModeCmd )

#-------------------------------------------------------------------------------
# The "[no|default] forwarding l3 source-interface drop" command
#-------------------------------------------------------------------------------
def l3SourceportDropSupportedGuard( mode, token ):
   if not bridgingHwCapabilities.l3SourceportDropSupported:
      return CliParser.guardNotThisPlatform
   # FIXME: remove this guard if we support it on LAG and all
   # other switchports
   if not isinstance( mode.intf, EthIntfCli.EthPhyIntf ):
      return "Not supported for this interface"
   return None

class ForwardingL3SourceIntfDrop( CliCommand.CliCommandClass ):
   syntax = "forwarding l3 source-interface drop"
   noOrDefaultSyntax = syntax
   data = {
      "forwarding" : CliToken.Switch.matcherForwarding,
      "l3" : 'Configure L3 forwarding features',
      "source-interface" : CliCommand.guardedKeyword(
         'source-interface',
         'Configure source interface properties',
         l3SourceportDropSupportedGuard ),
      "drop" : 'Drop packets'
   }

   @staticmethod
   def handler( mode, args ):
      sic = getSwitchIntfConfigEvenIfDisabled( mode )
      sic.l3SourceportDrop = True

   @staticmethod
   def noOrDefaultHandler( mode, args ):
      sic = getSwitchIntfConfigEvenIfDisabled( mode )
      sic.l3SourceportDrop = False

SwitchportModelet.addCommandClass( ForwardingL3SourceIntfDrop )

#-------------------------------------------------------------------------------
# Have the Cli Agent mount all needed state from sysdb
#-------------------------------------------------------------------------------
def Plugin( entityManager ):
   global ethIntfStatusDir, bridgingConfig
   global bridgingHwCapabilities, cliConfig
   ethIntfStatusDir = LazyMount.mount( entityManager, "interface/status/eth/intf",
                                       "Interface::EthIntfStatusDir", "r" )
   bridgingConfig = LazyMount.mount( entityManager, "bridging/config",
                                     "Bridging::Config", "r" )

   bridgingHwCapabilities = LazyMount.mount( entityManager,
                                             "bridging/hwcapabilities",
                                             "Bridging::HwCapabilities", "r" )
   cliConfig = ConfigMount.mount( entityManager, "bridging/input/config/cli", 
                                  "Bridging::Input::CliConfig", "w" )
