# Copyright (c) 2017 Arista Networks, Inc.  All rights reserved.
# Arista Networks, Inc. Confidential and Proprietary.

from CliModel import Str, Model, Enum, Int
from CliModel import GeneratorDict, GeneratorList
from ArnetModel import IpGenericPrefix
from CliPlugin.DynamicPrefixListHelper import getDplConfigParams

DPL_STATE_MAP = {
      1 : 'inactive',
      2 : 'active',
      3 : 'matchRouteMapNotConfigured',
      4 : 'notApplicableToRoutingInstance',
}

DPL_STATE_RENDER_MAP = {
      'active' : 'Active',
      'inactive' : 'Inactive',
      'matchRouteMapNotConfigured' : 'Match route map not configured',
      'notApplicableToRoutingInstance' : 'Not applicable to routing instance'
}

def printDynamicPrefixListLegendInformation():
   print( '* - Not all contributing routes are listed' )

class DynamicPrefixListContributingRoutes( Model ):
   prefix = IpGenericPrefix( help='Route Prefix' )

   def processData( self, data ):
      return data

   def render( self ):
      print( '       %s' % ( self.prefix ) )

class DynamicPrefixListModel( Model ):
   matchRouteMap = Str( help='Match route map name', optional=True )
   ipv4PrefixList = Str( help='IPv4 prefix list name', optional=True )
   ipv6PrefixList = Str( help='IPv6 prefix list name', optional=True )
   state = Enum( values=DPL_STATE_MAP.values(),
                 help='State of the dynamic prefix list' )
   numContributingRoutes = Int( help='Number of contributing routes', optional=True )
   contributingRoutes = GeneratorList( 
                           valueType=DynamicPrefixListContributingRoutes,
                           help='Capped list of contributing routes (a sample of '\
                                'contributing routes), current cap is 10.',
                           optional=True )
   
   def getKey( self, data ):
      assert 'name' in data
      return data[ 'name' ]

   def processData( self, data ):
      dplState = data.pop( 'dpl_state', None )
      if dplState is not None:
         self.state = DPL_STATE_MAP[ dplState ]
      if 'ipv6PrefixList' in data:
         # Remove '.ipv6.' from prefix list name
         self.ipv6PrefixList = data.pop( 'ipv6PrefixList' )[ 6: ]

      # if state == notApplicableToRoutingInstance, that means the
      # rib instance does not have information about the dpl. So
      # populate the available fields from the config directly
      dplName = data.get( 'name', None )
      if self.state == 'notApplicableToRoutingInstance':
         matchMap, ipv4PrefixList, ipv6PrefixList = getDplConfigParams( dplName )
         if matchMap:
            self.matchRouteMap = matchMap
         if ipv4PrefixList:
            self.ipv4PrefixList = ipv4PrefixList
         if ipv6PrefixList:
            self.ipv6PrefixList = ipv6PrefixList

      return data

   def renderEntry( self, dynamicPrefixListName ):
      print( '  Dynamic prefix list : %s' % ( dynamicPrefixListName ) )
      if self.matchRouteMap:
         print( '    Match route map: %s' % ( self.matchRouteMap ) )
      if self.ipv4PrefixList:
         print( '    IPv4 prefix list: %s' % ( self.ipv4PrefixList ) )
      if self.ipv6PrefixList:
         print( '    IPv6 prefix list: %s' % ( self.ipv6PrefixList ) )
      print( '    State: %s' % ( DPL_STATE_RENDER_MAP[ self.state ] ) )
      if self.state == 'active':
         notAllListed = '*' if self.numContributingRoutes > 10 else ''
         print( '    %sContributing routes: %d' % ( notAllListed, 
                                                    self.numContributingRoutes ) )
         for route in self.contributingRoutes:
            route.render()

class DynamicPrefixLists( Model ):
   _vrf = Str( help="Vrf Name" )
   dynamicPrefixLists = GeneratorDict( valueType=DynamicPrefixListModel,
      help='Table of dynamic prefix lists by name' )

   def processData( self, data ):
      self._vrf = data.pop( 'vrf-name' )

   def render( self ):
      print( 'VRF: %s' % ( self._vrf ) )
      for dynamicPrefixListName, dynamicPrefixList in self.dynamicPrefixLists:
         dynamicPrefixList.renderEntry( dynamicPrefixListName )
