#!/usr/bin/env python
# Copyright (c) 2017 Arista Networks, Inc.  All rights reserved.
# Arista Networks, Inc. Confidential and Proprietary.

import Tracing, Tac
import ConfigMount
import CliParser, BasicCli
import LazyMount
import DynamicPrefixListModel
import itertools

import CliCommand
import CliMatcher
import CliPlugin.TechSupportCli
from CliPlugin.DynamicPrefixListHelper import dynPfxListNameMatcher
from CliPlugin.IraServiceCli import getEffectiveProtocolModel
from CliPlugin.RouteMapCli import (
   CommonTokens,
   RouteMapMode,
)
from CliPlugin.VrfCli import (
      VrfExecCmdDec,
      generateVrfCliModel,
      getAllPlusReservedVrfNames,
)
from CliMode.DynamicPrefixList import DynamicPrefixListMode
from IpLibTypes import ProtocolAgentModelType as ProtoAgentModel
from RouteMapLib import (
   isMatchAttrEnabled,
   getErrorMessageCannotDeleteBecauseUsedBy
)
import ShowCommand

traceHandle = Tracing.Handle( 'DynamicPrefixList' )
t5 = traceHandle.trace5 # Info

dynPfxListConfigDir = None # Dynamic Prefix List
allVrfConfig = None
routeMapConfig = None

#----------------------------------------------------------------------------------
# Config Handler Functions.
#----------------------------------------------------------------------------------

# Delete Dynamic Prefix List.
def deleteDynPfxList( dynPfx ):
   del dynPfxListConfigDir.dynamicPrefixList[ dynPfx ]

# Get Dynamic Prefix List.
def getDynPfxList( dynPfx ):
   return dynPfxListConfigDir.dynamicPrefixList.get( dynPfx )

def getOrCreateDynPfxList ( dynPfx ):
   dynPfxList = getDynPfxList( dynPfx )
   if not dynPfxList:
      dynPfxList = dynPfxListConfigDir.dynamicPrefixList.newMember( dynPfx )

   return dynPfxList

#----------------------------------------------------------------------------------
# dynamic-prefix mode
#----------------------------------------------------------------------------------
class DynamicPrefixListConfigMode( DynamicPrefixListMode, BasicCli.ConfigModeBase ):
   name = 'Dynamic Prefix Configuration'
   modeParseTree = CliParser.ModeParseTree()
   #-------------------------------------------------------------------------------
   # Construct a new Dynamic-Prefix Config Mode
   #-------------------------------------------------------------------------------
   def __init__( self, parent, session, dynPfx ):
      self.dynPfx = dynPfx
      DynamicPrefixListMode.__init__( self, self.dynPfx )
      BasicCli.ConfigModeBase.__init__( self, parent, session )
      self.dynPfxEntry = Tac.newInstance(
         "Routing::DynamicPrefixList::DynamicPrefixListEntry", dynPfx )
      existingEntry = getDynPfxList( dynPfx )
      self.copyEntry( self.dynPfxEntry, existingEntry )
      self.abort_ = False

   def copyEntry( self, dstEntry, srcEntry ):
      if not dstEntry or not srcEntry:
         return
      dstEntry.matchMap = srcEntry.matchMap
      dstEntry.ipv4PrefixList = srcEntry.ipv4PrefixList
      dstEntry.ipv6PrefixList = srcEntry.ipv6PrefixList

   def commitContext( self ):
      if self.abort_:
         return
      entry = getOrCreateDynPfxList( self.dynPfx )
      self.copyEntry( entry, self.dynPfxEntry )

      # Increment version.
      entry.version += 1
      t5( 'Commiting changes' )

   def onExit( self ):
      self.commitContext()
      BasicCli.ConfigModeBase.onExit( self )

#----------------------------------------------------------------------------------
# (config)# [ no ] dynamic prefix-list <P1>
#----------------------------------------------------------------------------------
def routeMapsMatchingOnDynPrefixList( dynamicPrefixListName ):
   """ @brief Given a dynamic prefix-list name, it returns a set of all route-maps
              in the running-config that "use" (i.e., in a match clause)
              that dynamic prefix-list.
   """
   RouteMapMatchOption = Tac.Type( 'Routing::RouteMap::MatchOption' )
   routeMapsDep = set()
   for mapName, routeMap in routeMapConfig.routeMap.iteritems():
      for mapSeqno, mapEntry in routeMap.mapEntry.iteritems():
         matchRule = mapEntry.matchRule.get(
            RouteMapMatchOption.matchIpAddrDynamicPrefixList,
            mapEntry.matchRule.get(
               RouteMapMatchOption.matchIpv6AddrDynamicPrefixList ) )
         if matchRule:
            if matchRule.strValue == dynamicPrefixListName:
               routeMapsDep.add( ( mapName, mapSeqno ) )
   return routeMapsDep

def blockDynamiPrefixListDeleteIfInUse( mode, dynPrefixListName ):
   """ @brief Checks if a 'dynamic prefix-list' can be safely deleted without
              breaking any dependency with a route-map.
              Moreover, it prints an error message on the mode (passed as argument)
              in case of dependency.
       @return 'True' if the dynamic prefix-list should not be deleted.
               'False' in the nominal case (no errors).
   """
   if routeMapConfig.routeMapPolicyReferenceUnconfiguredError:
      routeMapsDep = routeMapsMatchingOnDynPrefixList( dynPrefixListName )
      if routeMapsDep:
         mode.addError( getErrorMessageCannotDeleteBecauseUsedBy(
            'dynamic prefix-list',
            dynPrefixListName,
            routeMapsDep ) )
         return True
   return False

class DynamicPrefixListConfig( CliCommand.CliCommandClass ):
   syntax = 'dynamic prefix-list NAME'
   noOrDefaultSyntax = syntax
   data = {
      'dynamic': 'Configure dynamic prefix-list',
      'prefix-list': 'Prefix list',
      'NAME': dynPfxListNameMatcher,
   }

   @staticmethod
   def handler( mode, args ):
      name = args[ 'NAME' ]
      childMode = mode.childMode( DynamicPrefixListConfigMode, dynPfx=name )
      mode.session_.gotoChildMode( childMode )

   @staticmethod
   def noOrDefaultHandler( mode, args ):
      name = args[ 'NAME' ]
      t5( 'Cleaning config for dynamic prefix-list name', name )
      if blockDynamiPrefixListDeleteIfInUse( mode, name ):
         return
      deleteDynPfxList( name )

BasicCli.GlobalConfigMode.addCommandClass( DynamicPrefixListConfig )

#----------------------------------------------------------------------------------
# (config-dyn-pfx-P1)# [ no ] match-map <route-map-name>
#----------------------------------------------------------------------------------
class DynamicPrefixListConfigModeMatchMap( CliCommand.CliCommandClass ):
   syntax = 'match-map MAPNAME'
   noOrDefaultSyntax = syntax
   data = {
      'match-map': 'Route-Map to Match',
      'MAPNAME': CommonTokens.routeMapName,
   }

   @staticmethod
   def handler( mode, args ):
      name = args[ 'MAPNAME' ]
      t5( 'Setting MatchMap name', name )
      mode.dynPfxEntry.matchMap = name

   @staticmethod
   def noOrDefaultHandler( mode, args ):
      name = args[ 'MAPNAME' ]
      if mode.dynPfxEntry.matchMap == name:
         t5( 'Removing MatchMap name', name )
         mode.dynPfxEntry.matchMap = ''

DynamicPrefixListConfigMode.addCommandClass( DynamicPrefixListConfigModeMatchMap )

#----------------------------------------------------------------------------------
# (config-dyn-pfx-P1)# [ no ] prefix-list ( ipv4 | ipv6 ) <prefix-list-name>
#----------------------------------------------------------------------------------
class DynamicPrefixListConfigModePrefixList( CliCommand.CliCommandClass ):
   syntax = 'prefix-list ( ipv4 | ipv6 ) LISTNAME'
   noOrDefaultSyntax = syntax
   data = {
      'prefix-list': 'Prefix list',
      'ipv4': 'IPv4 specific information',
      'ipv6': 'IPv6 specific information',
      'LISTNAME': dynPfxListNameMatcher,
   }

   @staticmethod
   def handler( mode, args ):
      name = args[ 'LISTNAME' ]
      if args.get( 'ipv4' ):
         t5( 'Setting IPv4 PrefixList name', name )
         mode.dynPfxEntry.ipv4PrefixList = name
      elif args.get( 'ipv6' ):
         t5( 'Setting IPv6 PrefixList name', name )
         mode.dynPfxEntry.ipv6PrefixList = name

   @staticmethod
   def noOrDefaultHandler( mode, args ):
      name = args[ 'LISTNAME' ]
      if args.get( 'ipv4' ):
         if mode.dynPfxEntry.ipv4PrefixList == name:
            t5( 'Removing IPv4 PrefixList', name )
            mode.dynPfxEntry.ipv4PrefixList = ''
      elif args.get( 'ipv6' ):
         if mode.dynPfxEntry.ipv6PrefixList == name:
            t5( 'Removing IPv6 PrefixList', name )
            mode.dynPfxEntry.ipv6PrefixList = ''

DynamicPrefixListConfigMode.addCommandClass( DynamicPrefixListConfigModePrefixList )

#----------------------------------------------------------------------------------
# (config-dyn-pfx-P1)# abort
#----------------------------------------------------------------------------------
class DynamicPrefixListConfigModeAbort( CliCommand.CliCommandClass ):
   syntax = 'abort'
   data = {
      'abort': 'Exit without committing changes',
   }

   @staticmethod
   def handler( mode, args ):
      t5( 'Aborting changes' )
      mode.abort_ = True
      mode.session_.gotoParentMode()

DynamicPrefixListConfigMode.addCommandClass( DynamicPrefixListConfigModeAbort )

#-------------------------------------------------------------------------------
# Show commands
#-------------------------------------------------------------------------------

routeMapShowCmdDict = {}

allVrfWithNameMatcher = CliMatcher.DynamicNameMatcher(
   getAllPlusReservedVrfNames, helpdesc='VRF name' )

def getVrfsFunc():
   return allVrfConfig.vrf.members()

dynamicPrefixListVrfModel = \
                  generateVrfCliModel( DynamicPrefixListModel.DynamicPrefixLists,
                                       "Dynamic prefix list information for VRFs" )

@VrfExecCmdDec( getVrfsFunc=getVrfsFunc, cliModel=dynamicPrefixListVrfModel )
def showPrefixList( mode, vrfName=None, prefixListName=None ):

   def dynamicPrefixListFromConfig():
      for dynamicPrefixListName in dynPfxListConfigDir.dynamicPrefixList:
         if not prefixListName or prefixListName == dynamicPrefixListName:
            dynamicPrefixListInst = DynamicPrefixListModel.DynamicPrefixListModel()
            dynamicPrefixListInst.state = 'notApplicableToRoutingInstance'
            psuedoData = { 'name' : dynamicPrefixListName }
            dynamicPrefixListInst.processData( psuedoData )
            yield dynamicPrefixListName, dynamicPrefixListInst

   if getEffectiveProtocolModel( mode ) == ProtoAgentModel.multiAgent:
      showDynPfxListCb = routeMapShowCmdDict[ 'doArBgpShowDynPfxList' ]
   else:
      showDynPfxListCb = routeMapShowCmdDict[ 'showDynPfxListCb' ]

   if mode.session_.outputFormat_ == "text":
      DynamicPrefixListModel.printDynamicPrefixListLegendInformation()

   args = {}
   args[ 'vrfName' ] = vrfName
   if prefixListName:
      args[ 'prefix-list' ] = prefixListName

   try:
      result = showDynPfxListCb( mode, DynamicPrefixListModel.DynamicPrefixLists,
                                 args )
      return result
   except routeMapShowCmdDict.get( 'EmptyResponseException' ):
      inst = DynamicPrefixListModel.DynamicPrefixLists()
      if prefixListName:
         inst.dynamicPrefixLists = itertools.chain( inst.dynamicPrefixLists,
                                                    dynamicPrefixListFromConfig() )
      return inst

class ShowDynamicPrefixListCmd( ShowCommand.ShowCliCommandClass ):
   syntax = 'show dynamic prefix-list [ LISTNAME ] [ vrf VRFNAME ]'
   data = {
      'dynamic': 'Show dynamic policy',
      'prefix-list': 'Prefix list',
      'LISTNAME': dynPfxListNameMatcher,
      'vrf': 'VRF name',
      'VRFNAME': allVrfWithNameMatcher,
   }
   cliModel = dynamicPrefixListVrfModel

   @staticmethod
   def handler( mode, args ):
      prefixListName = args.get( 'LISTNAME' )
      vrfName = args.get( 'VRFNAME' )
      return showPrefixList( mode, vrfName=vrfName, prefixListName=prefixListName )

BasicCli.addShowCommandClass( ShowDynamicPrefixListCmd )

#-------------------------------------------------------------------------------
# Register dynamic prefix-list Show Commands Into "Show tech-Support"
#-------------------------------------------------------------------------------

def dynamicPrefixListTechSupportCmds():
   Cmds = [ "show dynamic prefix-List vrf all" ]
   return Cmds

CliPlugin.TechSupportCli.registerShowTechSupportCmdCallback(
   '2018-07-02 20:28:13',
   dynamicPrefixListTechSupportCmds )

def Plugin( entityManager ):
   global dynPfxListConfigDir
   global allVrfConfig
   global routeMapConfig

   allVrfConfig = LazyMount.mount( entityManager, "ip/vrf/config",
                                   "Ip::AllVrfConfig", "r" )
   dynPfxListConfigDir = ConfigMount.mount(
      entityManager, 'routing/dynPfxList/config',
      'Routing::DynamicPrefixList::Config', 'w' )
   routeMapConfig = LazyMount.mount( entityManager,
                                     "routing/routemap/config",
                                     "Routing::RouteMap::Config",
                                     "r" )

# Add matches for dynamic prefix-list

# match ip address dynamic prefix-list
class MatchIpAddrDynamicPrefixListCmd( CliCommand.CliCommandClass ):
   syntax = 'match ip address dynamic prefix-list LISTNAME'
   noOrDefaultSyntax = syntax
   data = {
      'match': 'Route map match rule',
      'ip': 'Match IP specific information',
      'address': 'Match ip address',
      'dynamic': 'Configure dynamic prefix-list',
      'prefix-list': 'Prefix list',
      'LISTNAME': dynPfxListNameMatcher,
   }

   @staticmethod
   def handler( mode, args ):
      no = CliCommand.isNoOrDefaultCmd( args )
      mode.setMatchValue( no, 'matchIpAddrDynamicPrefixList', args[ 'LISTNAME' ] )

   noOrDefaultHandler = handler

if isMatchAttrEnabled( 'matchIpAddrDynamicPrefixList' ):
   RouteMapMode.addCommandClass( MatchIpAddrDynamicPrefixListCmd )

# match ipv6 address dynamic prefix-list
class MatchIpv6AddrDynamicPrefixListCmd( CliCommand.CliCommandClass ):
   syntax = 'match ipv6 address dynamic prefix-list LISTNAME'
   noOrDefaultSyntax = syntax
   data = {
      'match': 'Route map match rule',
      'ipv6': 'Match IPv6 specific information',
      'address': 'Match ip address',
      'dynamic': 'Configure dynamic prefix-list',
      'prefix-list': 'Prefix list',
      'LISTNAME': dynPfxListNameMatcher,
   }

   @staticmethod
   def handler( mode, args ):
      no = CliCommand.isNoOrDefaultCmd( args )
      mode.setMatchValue( no, 'matchIpv6AddrDynamicPrefixList', args[ 'LISTNAME' ] )

   noOrDefaultHandler = handler

if isMatchAttrEnabled( 'matchIpv6AddrDynamicPrefixList' ):
   RouteMapMode.addCommandClass( MatchIpv6AddrDynamicPrefixListCmd )
