# Copyright (c) 2020 Arista Networks, Inc.  All rights reserved.
# Arista Networks, Inc. Confidential and Proprietary.

import Tac
import TacSigint
import CliCommand

import IntfCli
from VlanCli import switchportEligible
from VlanCli import switchportMatcher
from VlanCli import vlanIdMatcher
from VlanCli import SwitchportModelet
from VlanCli import ConfigMount

dynVlanDir = None
dynamicCliAccessVlanConfig = None

#-------------------------------------------------------------------------------
# The "[no] switchport access vlan dynamic" command, in "config-if" mode.
#
# This command sets the effective access vlan of the interface *without changing
# running-config*; it's supposed to be used when a temporary effect is wanted.
# When disabled, the regular access vlan is restored.
#
# The full syntax of this command is:
#
#   switchport access vlan dynamic <vlan_id>
#   {no|default} switchport access vlan dynamic ...
#-------------------------------------------------------------------------------

def setDynamicCliAccessVlan( mode, args ):
   vlanId = args.get( 'VLANID' )
   assert switchportEligible( mode )
   dynamicCliAccessVlanConfig.sourceDetail = 'dynamic cli'
   dynamicCliAccessVlanConfig.switchIntfConfigPriority = 101
   # ^ Regular CLI config has pririty = 100 and we want to override it
   # Assign port:
   sic = dynamicCliAccessVlanConfig.switchIntfConfig.newMember(
      mode.intf.name, 'access' )
   if not sic:
      return
   sic.source = 'dynamic cli'
   sic.accessVlan = vlanId.id

def noDynamicCliAccessVlan( mode, args ):
   if mode.intf.name not in dynamicCliAccessVlanConfig.switchIntfConfig:
      return
   # We don't care about the vlan argument, just remove the assignment
   # of the current intf
   del dynamicCliAccessVlanConfig.switchIntfConfig[ mode.intf.name ]

class DynamicCliAccessVlan( CliCommand.CliCommandClass ):
   syntax = 'switchport access vlan dynamic VLANID'
   noOrDefaultSyntax = 'switchport access vlan dynamic ...'
   data = {
      'switchport': switchportMatcher,
      'access': 'Set access mode characteristics of the interface',
      'vlan': 'Set VLAN when interface is in access mode',
      'dynamic': 'Dynamic (non-persistent) access VLAN setting',
      'VLANID': vlanIdMatcher,
      }

   handler = setDynamicCliAccessVlan
   noOrDefaultHandler = noDynamicCliAccessVlan

SwitchportModelet.addCommandClass( DynamicCliAccessVlan )

# Support for 'default interface'
class SwitchIntfConfigCleaner( IntfCli.IntfDependentBase ):
   """This class is responsible for removing entries from the cliConfig's
   switchIntfConfig collection when the interfaces associated with those
   switchIntfConfigs are removed."""

   def setDefault( self ):
      # Tacc collection - no key check required.
      del dynamicCliAccessVlanConfig.switchIntfConfig[ self.intf_.name ]

def Plugin( entityManager ):
   global dynamicCliAccessVlanConfig
   dynamicCliAccessVlanConfig = ConfigMount.mount(
      entityManager, "bridging/input/config/dynamicCliAccessVlan",
      "Bridging::Input::Config", "w" )
   priority = 10 # arbitrarily chosen priority.
   IntfCli.Intf.registerDependentClass( SwitchIntfConfigCleaner, priority )
