# Copyright (c) 2017 Arista Networks, Inc.  All rights reserved.
# Arista Networks, Inc. Confidential and Proprietary.

from __future__ import absolute_import, division, print_function

import CliCommand
import CliMatcher
import ConfigMount
import Tac
import Tracing

from CliPlugin.DynamicPrefixListHelper import dynPfxListNameMatcher
from CliPlugin.RouterGeneralCli import RouterGeneralVrfMode
from CliPlugin.RouterGeneralCli import routerGeneralVrfCleanupHook

traceHandle = Tracing.Handle( 'DynPolicyRoutesCli' )
t5 = traceHandle.trace5 # Info

dynRouteConfig = None
prefDefault = Tac.Type( "Routing::DynPolicyRoutes::Constants" ).preferenceDefault
#-------------------------------------------------------------------------------
# Config handler functions 
#-------------------------------------------------------------------------------
def getDynRtConfig( vrfName ):
   return dynRouteConfig.vrfConfig.get( vrfName, None )

def getOrCreateDynRtConfig( vrfName ):
   vrfConfig = getDynRtConfig( vrfName )

   if not vrfConfig:
      vrfConfig = dynRouteConfig.vrfConfig.newMember( vrfName )

   return  vrfConfig

def deleteDynRtConfig( vrfName ):
   del dynRouteConfig.vrfConfig[ vrfName ]

def deleteAllDynRtConfig():
   dynRouteConfig.vrfConfig.clear()

def getDynRtPrefixListConfig( vrfConfig, dynPlName ):
   return vrfConfig.dynamicRoute.get( dynPlName, None )

def getOrCreateDynRtPrefixListConfig( vrfConfig, dynPlName ):
   dynPlConfig = getDynRtPrefixListConfig( vrfConfig, dynPlName )
   if not dynPlConfig:
      dynPlConfig = vrfConfig.dynamicRoute.newMember( dynPlName )

   return dynPlConfig

def deleteDynRtPrefixListConfig( vrfName, dynPlName ):
   vrfConfig = getDynRtConfig( vrfName )
   if vrfConfig:
      del vrfConfig.dynamicRoute[ dynPlName ]
      if not len( vrfConfig.dynamicRoute ):
         del dynRouteConfig.vrfConfig[ vrfName ]


#-------------------------------------------------------------------------------
# Construct 'install drop [ preference PREFERENCE ]' expression
#-------------------------------------------------------------------------------
class InstallOptsExpression( CliCommand.CliExpression ):
   expression = 'install drop [ preference PREFERENCE ]'
   data = {
      'install': 'Install route in FIB',
      'drop': 'Install as drop route',
      'preference': 'Set admin distance for route',
      'PREFERENCE': CliMatcher.IntegerMatcher( 1, 255, 
         helpdesc='Admin distance value' )
   }

#-------------------------------------------------------------------------------
# [ no|default ] routes dynamic prefix-list PREFIXLIST 
#                [ install drop [ preference PREFERENCE ] ]
# in "config-router-general-vrf" mode 
#-------------------------------------------------------------------------------
class RoutesDynamicPrefixListCmd( CliCommand.CliCommandClass ):
   syntax = 'routes dynamic prefix-list PREFIXLIST [ INSTALLOPTS ]'
   noOrDefaultSyntax = 'routes dynamic prefix-list PREFIXLIST ...'
   data = {
      'routes': 'Import dynamic routes',
      'dynamic': 'Import dynamic routes',
      'prefix-list': 'Prefix list', 
      'PREFIXLIST': dynPfxListNameMatcher,
      'INSTALLOPTS': InstallOptsExpression,
   }

   @staticmethod
   def handler( mode, args ):
      dynamicPrefixListName = args[ 'PREFIXLIST' ]
      installOpts = args.get( 'install', '')
      installOpts += ' drop' if 'drop' in args else ''
      installOpts += ' preference' if 'preference' in args else ''
      installOpts += ( ' ' + str( 
         args.get( 'PREFERENCE', '' ) ) ) if 'PREFERENCE' in args else ''
      t5( "Setting prefix-list for vrf", mode.vrfName, ":", dynamicPrefixListName,
            "(", installOpts, ")"  )
      vrfConfig = getOrCreateDynRtConfig( mode.vrfName )
      dynPlConfig = getOrCreateDynRtPrefixListConfig( vrfConfig, 
         dynamicPrefixListName )
   
      # Create a dynamic-prefix 
      if 'install' in args:
         dynPlConfig.install = True
         dynPlConfig.drop = 'drop' in args
         dynPlConfig.preference = args.get( 'PREFERENCE', prefDefault )
      else:
         # If install options weren't specified reset them
         dynPlConfig.install = False
         dynPlConfig.drop = False
         dynPlConfig.preference = prefDefault
   
   @staticmethod
   def noOrDefaultHandler( mode, args ):
      dynamicPrefixListName = args.get( 'PREFIXLIST' )
      t5( "Deleting prefix-list in vrf ", mode.vrfName, ": ", dynamicPrefixListName )
      deleteDynRtPrefixListConfig( mode.vrfName, dynamicPrefixListName )

RouterGeneralVrfMode.addCommandClass( RoutesDynamicPrefixListCmd )

t5( 'Registered commands - in RouterGeneralVrfMode.addCommandClass' )

# Register callback handler for cleaning 'router general' vrf sub-modes
def noOrDefaultRouterGeneralVrfMode( vrfName ):
   deleteDynRtConfig( vrfName )

routerGeneralVrfCleanupHook.addExtension( noOrDefaultRouterGeneralVrfMode )

def Plugin( entityManager ):
   global dynRouteConfig

   dynRouteConfig = ConfigMount.mount( entityManager, 
                                       'routing/general/config/dynPolicyRoutes',
                                       'Routing::DynPolicyRoutes::Config', 'w' )
