#!/usr/bin/env python
# Copyright (c) 2019 Arista Networks, Inc.  All rights reserved.
# Arista Networks, Inc. Confidential and Proprietary.

import BasicCli
import CliCommand
import CliMatcher
import CliParser
import CliToken.Clear
import DpsCliModel
import IpAddrMatcher
import LazyMount
import re
import ShowCommand
import SmashLazyMount
import Tac
import TechSupportCli

PathType = Tac.Type( 'Dps::PathType' )
UnsecuredPath = PathType.unsecuredPath
SecuredPath = PathType.securedPath
PathState = Tac.Type( 'Dps::PathState' )
UnresolvedPath = PathState.unresolved
ResolvedPath = PathState.resolved
ItsSessKey = Tac.Type( 'Its::ItsSessKey' )
VrfAppKey = Tac.Type( 'Dps::VrfAppKey' )

dpsCountersSmash = None
dpsCountersSnapShot = None
dpsHwCapability = None
dpsInput = None
dpsPathStatusDir = None
dpsPeerStatus = None
dpsVrfStatus = None
itsSessStatus = None

def dpsSupported():
   if dpsHwCapability and dpsHwCapability.supported:
      return True
   return False

def pathSelectionSupportedGuard( mode, token ):
   if dpsSupported():
      return None
   return CliParser.guardNotThisPlatform

def getPathIndex( pathName ):
   p = r'^[pP]ath([0-9]*)'
   m = re.search( p, pathName )
   if m is None:
      return None
   return int( m.group( 1 ) )

DpsLbProfileCounterEntry = Tac.Type( 'Dps::DpsLbProfileCounterEntry' )
DpsPathCounterEntry = Tac.Type( 'Dps::DpsPathCounterEntry' )

pathSelectionMatcher = CliMatcher.KeywordMatcher( 'path-selection',
                          helpdesc='Show dynamic path selection information' )
pathSelectionMatcherForClear = CliMatcher.KeywordMatcher( 'path-selection',
                               helpdesc='Clear dynamic path selection information' )
pathSelectionNode = CliCommand.Node( matcher=pathSelectionMatcher,
                                     guard=pathSelectionSupportedGuard )
pathSelectionNodeForClear = CliCommand.Node( matcher=pathSelectionMatcherForClear,
                                     guard=pathSelectionSupportedGuard )
clearMatcher = CliToken.Clear.clearKwNode

peerIpv4AddrMatcher = IpAddrMatcher.IpAddrMatcher( helpdesc='Peer address' )
vrfMatcher = CliMatcher.PatternMatcher( pattern='[a-zA-Z0-9_-]+',
                                        helpdesc='Match the VRF',
                                        helpname='WORD' )
appProfileMatcher = CliMatcher.PatternMatcher( pattern='[a-zA-Z0-9_-]+',
                                               helpname='WORD',
                                               helpdesc='Application profile' )
dstIpv4AddrMatcher = IpAddrMatcher.IpAddrMatcher( helpdesc='Destination address' )
srcIpv4AddrMatcher = IpAddrMatcher.IpAddrMatcher( helpdesc='Source address' )
peerIpv4AddrMatcher = IpAddrMatcher.IpAddrMatcher( helpdesc='Peer address' )
trafficClassMatcher = CliMatcher.IntegerMatcher( 0, 7,
                                             helpdesc='Internal class of traffic' )
pathGroupMatcher = CliMatcher.PatternMatcher( pattern='[a-zA-Z0-9_-]+',
                                              helpname='WORD',
                                              helpdesc='Name of the path group' )
pathNameMatcher = CliMatcher.PatternMatcher( pattern='[a-zA-Z0-9_-]+',
                                             helpname='WORD',
                                             helpdesc='Name of the path' )
peerNameMatcher = CliMatcher.PatternMatcher( pattern='[a-zA-Z0-9_-]+',
                                             helpname='WORD',
                                             helpdesc='Name of the peer' )

def getDpsGroups():
   groups = {}
   for pathStatus in dpsPathStatusDir.pathStatus.values():
      for pathEntry in pathStatus.pathStatusEntry.values():
         # mapping from path index to group name
         groups[ pathEntry.pathInfo.pathIndex ] = pathEntry.pgName
   return groups

def getDpsCounters( mode, args ):
   lbProfileDict = {}
   groups = getDpsGroups()

   vrf = args.get( 'VRF' )
   peerIp = args.get( 'PEER' )
   appProfile = args.get( 'APPPROF' )

   for lbKey, profile in dpsCountersSmash.lbProfileCounter.items():
      lbId = lbKey.lbId
      lbProfile = DpsCliModel.DpsLbProfiles()

      profileKeyInfo = profile.lbKeyInfo
      vrfAppKey = VrfAppKey( profileKeyInfo.vrfId, profileKeyInfo.appId )
      if vrfAppKey in dpsVrfStatus.vrfApp:
         lbProfile.vrf = dpsVrfStatus.vrfApp[ vrfAppKey ].vrfName
         lbProfile.appProfile = dpsVrfStatus.vrfApp[ vrfAppKey ].appName
      else:
         lbProfile.vrf = "vrf" + str( profileKeyInfo.vrfId )
         lbProfile.appProfile = "app" + str( profileKeyInfo.appId )
      lbProfile.peerIp = profileKeyInfo.vtepIp
      lbProfile.peerName = ''
      if profileKeyInfo.vtepIp in dpsPeerStatus.peerStatusEntry:
         peerStatus = dpsPeerStatus.peerStatusEntry[ profileKeyInfo.vtepIp ]
         lbProfile.peerName = peerStatus.peerName

      if ( ( vrf and vrf != lbProfile.vrf ) or
           ( peerIp and peerIp != str( lbProfile.peerIp ) ) or
           ( appProfile and appProfile != lbProfile.appProfile ) ):
         continue

      outBytesSnapShot = 0
      outPktsSnapShot = 0
      lbCounterSnapShot = dpsCountersSnapShot.lbProfileCounter.get( lbKey )
      if lbCounterSnapShot and lbCounterSnapShot.timestamp > profile.timestamp:
         outBytesSnapShot = lbCounterSnapShot.outBytes
         outPktsSnapShot = lbCounterSnapShot.outPkts
      lbProfile.outBytes = profile.outBytes - outBytesSnapShot
      lbProfile.outPkts = profile.outPkts - outPktsSnapShot
      lbProfile.flows = profile.flows
      lbProfile.throughput = profile.throughput
      lbProfile.pathGroups = {}

      for pathId, path in profile.pathCounter.items():
         if path.pathId == 0:
            continue
         groupName = groups.get( path.pathId, '' )
         if groupName not in lbProfile.pathGroups.keys():
            pG = DpsCliModel.DpsLbProfiles.PathGroups()
            pG.paths = {}
            lbProfile.pathGroups[ groupName ] = pG
         p = DpsCliModel.DpsLbProfiles.PathGroups.Paths()
         outBytesSnapShot = 0
         outPktsSnapShot = 0
         if lbCounterSnapShot:
            pCounterSnapShot = lbCounterSnapShot.pathCounter[ pathId ]
         else:
            pCounterSnapShot = None
         if pCounterSnapShot and pCounterSnapShot.timestamp > path.timestamp:
            outBytesSnapShot = pCounterSnapShot.outBytes
            outPktsSnapShot = pCounterSnapShot.outPkts
         p.outBytes = path.outBytes - outBytesSnapShot
         p.outPkts = path.outPkts - outPktsSnapShot
         p.flows = path.flows
         p.throughput = path.throughput
         lbProfile.pathGroups[ groupName ].paths[ pathId ] = p

      lbProfileDict[ lbId ] = lbProfile
   return lbProfileDict

#-----------------------------------------------------------------------------------
# show path-selection load-balance counters [ detail ]
#    [ vrf VRF ] [ peer PEER ] [ application-profile APPPROF ]
#-----------------------------------------------------------------------------------
class ShowPathSelectionLbCounters( ShowCommand.ShowCliCommandClass ):
   syntax = '''show path-selection load-balance counters [ detail ]
               [ vrf VRF ] [ peer PEER ] [ application-profile APPPROF ]'''
   data = {
             'path-selection': pathSelectionNode,
             'load-balance' : 'Show load-balance information',
             'counters' : 'Show path-selection counters',
             'detail': 'Show path-selection counters in detail',
             'vrf': 'Show path-selection counters filtered by vrf name',
             'VRF': vrfMatcher,
             'peer': 'Show path-selection counters filtered by peer IP',
             'PEER': peerIpv4AddrMatcher,
             'application-profile':
                   'Show path-selection counters filtered by application-profile',
             'APPPROF': appProfileMatcher,
          }
   cliModel = DpsCliModel.DpsLbCounters

   @staticmethod
   def handler( mode, args ):
      lbCounters = DpsCliModel.DpsLbCounters()
      lbCounters.lbProfiles = getDpsCounters( mode, args )
      lbCounters._detail = 'detail' in args # pylint: disable=protected-access
      return lbCounters

BasicCli.addShowCommandClass( ShowPathSelectionLbCounters )

#-----------------------------------------------------------------------------------
# show path-selection application counters
#    [ vrf VRF ] [ peer PEER ] [ application-profile APPPROF ]
#-----------------------------------------------------------------------------------
class ShowPathSelectionAppCounters( ShowCommand.ShowCliCommandClass ):
   syntax = '''show path-selection application counters
               [ vrf VRF ] [ peer PEER ] [ application-profile APPPROF ]'''
   data = {
             'path-selection': pathSelectionNode,
             'application' : 'Show application information',
             'counters' : 'Show path-selection counters',
             'vrf': 'Show path-selection counters filtered by vrf name',
             'VRF': vrfMatcher,
             'peer': 'Show path-selection counters filtered by peer IP',
             'PEER': peerIpv4AddrMatcher,
             'application-profile':
                   'Show path-selection counters filtered by application-profile',
             'APPPROF': appProfileMatcher,
          }
   cliModel = DpsCliModel.DpsAppCounters

   @staticmethod
   def handler( mode, args ):
      appCounters = DpsCliModel.DpsAppCounters()
      appCounters.lbProfiles = getDpsCounters( mode, args )
      return appCounters

BasicCli.addShowCommandClass( ShowPathSelectionAppCounters )

def getDpsPaths( mode, args ):
   model = DpsCliModel.DpsPaths()
   itsCharacteristics = itsSessStatus.itsSessCharacData

   filtPeer = args.get( 'PEERIP' )
   filtPeerName = args.get( 'PEERNAME' )
   filtGroup = args.get( 'GROUPNAME' )
   filtPath = args.get( 'PATHNAME' )
   filtSrc = args.get( 'SRCIP' )
   filtDst = args.get( 'DSTIP' )
   filtTc = args.get( 'TC' )

   for _, pathStatus in dpsPathStatusDir.pathStatus.items():
      for pathKey, pathEntry in pathStatus.pathStatusEntry.items():
         pathId = pathEntry.pathInfo.pathIndex
         peerIp = pathEntry.peerIp
         peerName = ''
         if peerIp in dpsPeerStatus.peerStatusEntry:
            peerName = dpsPeerStatus.peerStatusEntry[ peerIp ].peerName
         groupName = pathEntry.pgName
         pathName = "path" + str( pathId )
         srcIp = pathKey.srcAddr.stringValue
         dstIp = pathKey.dstAddr.stringValue

         # Check filters
         if ( ( filtPeer and filtPeer != peerIp.stringValue ) or
              ( filtPeerName and filtPeerName != peerName ) or
              ( filtGroup and filtGroup != groupName ) or
              ( filtPath and getPathIndex( filtPath ) != pathId ) or
              ( filtSrc and filtSrc != srcIp ) or
              ( filtDst and filtDst != dstIp ) ):
            continue

         pathState = ""
         if pathEntry.programmed:
            if pathEntry.pathInfo.pathType == SecuredPath:
               pathState = "ipsecEstablished"
            else:
               pathState = "resolved"
         elif pathEntry.pathInfo.pathState == ResolvedPath:
            pathState = "arpPending"
         else:
            if ( pathEntry.pathInfo.via is not None ) and \
               ( pathEntry.pathInfo.pathType == SecuredPath ):
               pathState = "ipsecPending"
            else:
               pathState = "routePending"

         if peerIp in model.dpsPeers:
            peer = model.dpsPeers[ peerIp ]
         else:
            peer = DpsCliModel.DpsPeer()
            peer.peerName = peerName
            model.dpsPeers[ peerIp ] = peer

         if groupName in peer.dpsGroups:
            group = peer.dpsGroups[ groupName ]
         else:
            group = DpsCliModel.DpsGroup()
            peer.dpsGroups[ groupName ] = group

         if pathName in group.dpsPaths:
            path = group.dpsPaths[ pathName ]
         else:
            path = DpsCliModel.DpsPath()
            path.source = srcIp
            path.destination = dstIp
            path.state = pathState
            group.dpsPaths[ pathName ] = path

         # Read session status from ITS smash table
         for tc in range( 0, 8 ):
            if filtTc and filtTc != tc:
               continue

            itsSessKey = ItsSessKey( pathKey, pathId, tc )
            session = None
            if itsSessKey in itsCharacteristics:
               itsSess = itsCharacteristics[ itsSessKey ]
               session = DpsCliModel.DpsSession()
               if itsSess.active:
                  session.active = True
                  session.seconds = int( round( itsSess.activeTime / 1e9 ) )
               else:
                  session.active = False
                  session.seconds = 0
            elif tc == 0:
               # Always show tc 0
               session = DpsCliModel.DpsSession()
               session.active = False
               session.seconds = 0
            if session:
               path.dpsSessions[ tc ] = session

   return model

#-----------------------------------------------------------------------------------
# show path-selection paths
# [ peer PEER ] [ peer-name PEERNAME ]
# [ path-group GROUPNAME ] [ path-name PATHNAME ]
# [ source SRCIP ] [ destination DSTIP ] [ traffic-class TC ]
#-----------------------------------------------------------------------------------
class ShowPathSelectionPaths( ShowCommand.ShowCliCommandClass ):
   syntax = '''show path-selection paths
               [ peer PEERIP ] [ peer-name PEERNAME ]
               [ path-group GROUPNAME ] [ path-name PATHNAME ]
               [ source SRCIP ] [ destination DSTIP ] [ traffic-class TC ]'''
   data = {
             'path-selection': pathSelectionNode,
             'paths': 'Show path status information',
             'peer': 'Show path status filtered by peer IP',
             'PEERIP': peerIpv4AddrMatcher,
             'peer-name': 'Show path status filtered by peer name',
             'PEERNAME': peerNameMatcher,
             'path-group': 'Show path status filtered by path group name',
             'GROUPNAME': pathGroupMatcher,
             'path-name': 'Show path status filtered by path name',
             'PATHNAME': pathNameMatcher,
             'source': 'Show path status filtered by source IP',
             'SRCIP': srcIpv4AddrMatcher,
             'destination': 'Show path status filtered by destination IP',
             'DSTIP': dstIpv4AddrMatcher,
             'traffic-class': 'Show path status filtered by traffic class',
             'TC': trafficClassMatcher,
          }
   cliModel = DpsCliModel.DpsPaths
   handler = getDpsPaths

BasicCli.addShowCommandClass( ShowPathSelectionPaths )

#-----------------------------------------------------------------------------------
# clear path-selection counters
#-----------------------------------------------------------------------------------
class ClearPathSelectionCounters( CliCommand.CliCommandClass ):
   syntax = 'clear path-selection counters'
   data = {
             'clear': clearMatcher,
             'path-selection': pathSelectionNodeForClear,
             'counters': 'Clear path-selection counters',
          }

   @staticmethod
   def handler( mode, args ):
      for lbKey, profile in dpsCountersSmash.lbProfileCounter.items():
         lbCounterSnapShot = DpsLbProfileCounterEntry( lbKey )
         lbCounterSnapShot.outBytes = profile.outBytes
         lbCounterSnapShot.outPkts = profile.outPkts
         lbCounterSnapShot.timestamp = Tac.now()
         for pathId, path in profile.pathCounter.items():
            pCounterSnapShot = DpsPathCounterEntry( pathId )
            pCounterSnapShot.outBytes = path.outBytes
            pCounterSnapShot.outPkts = path.outPkts
            pCounterSnapShot.timestamp = Tac.now()
            lbCounterSnapShot.pathCounter[ pathId ] = pCounterSnapShot
         dpsCountersSnapShot.addLbProfileCounter( lbCounterSnapShot )
      # clear stale entries
      for lbKey in dpsCountersSnapShot.lbProfileCounter:
         if lbKey not in dpsCountersSmash.lbProfileCounter:
            del dpsCountersSnapShot.lbProfileCounter[ lbKey ]

BasicCli.EnableMode.addCommandClass( ClearPathSelectionCounters )

def _DpsShowTechCmds():
   if dpsSupported():
      return [ 'show path-selection load-balance counters',
               'show path-selection load-balance counters detail',
               'show path-selection application counters',
               'show path-selection paths',
             ]
   return []
timeStamp = '2019-07-09 13:58:09'
TechSupportCli.registerShowTechSupportCmdCallback( timeStamp,
                                                    _DpsShowTechCmds )
TechSupportCli.registerShowTechSupportCmdCallback( timeStamp,
                                                    _DpsShowTechCmds,
                                                    extended='sfe' )

def Plugin( em ):
   global dpsCountersSmash
   global dpsCountersSnapShot
   global dpsHwCapability
   global dpsInput
   global dpsPathStatusDir
   global dpsPeerStatus
   global dpsVrfStatus
   global itsSessStatus
   dpsCountersSmash = SmashLazyMount.mount( em, 'dps/counters',
                                            'DpsSmash::DpsCounters',
                                            SmashLazyMount.mountInfo( 'reader' ) )
   dpsCountersSnapShot = LazyMount.mount( em, 'dps/cli/counters/snapshot',
                                          'Dps::DpsCountersSnapShot', 'wr' )
   dpsHwCapability = LazyMount.mount( em, 'dps/hwCapability',
                                      'Dps::HwCapability', 'r' )
   dpsInput = LazyMount.mount( em, 'dps/input/cli',
                                      'Dps::DpsCliConfig', 'r' )
   dpsPathStatusDir = LazyMount.mount( em, 'dps/path/status',
                                      'Dps::PathStatusDir', 'r' )
   dpsPeerStatus = LazyMount.mount( em, 'dps/peer/status',
                                    'Dps::PeerStatus', 'r' )
   dpsVrfStatus = LazyMount.mount( em, 'dps/vrf/status',
                                   'Dps::VrfStatus', 'r' )
   itsSessStatus = SmashLazyMount.mount( em, 'its/sessStatus',
                                     'ItsSmash::ItsSessStatus',
                                     SmashLazyMount.mountInfo( 'reader' ) )
