#!/usr/bin/env python
# Copyright (c) 2019 Arista Networks, Inc.  All rights reserved.
# Arista Networks, Inc. Confidential and Proprietary.

from Arnet import IpGenAddr
from ArnetModel import IpGenericAddress
from CliModel import Bool
from CliModel import Dict
from CliModel import Enum
from CliModel import Float
from CliModel import Int
from CliModel import Model
from CliModel import Str
import datetime
import TableOutput

fl = TableOutput.Format( justify='left' )
fl.padLimitIs( True )

pathStateToStrMap = {
   'ipsecEstablished' : 'IPSec established',
   'resolved' : 'resolved',
   'arpPending' : 'ARP pending',
   'ipsecPending' : 'IPSec pending',
   'routePending' : 'route pending',
}

class DpsLbProfiles( Model ):
   appProfile = Str( help='Name of the application profile' )
   vrf = Str( help='Name of the VRF' )
   peerIp = IpGenericAddress( help='Peer IP address of the ITS session' )
   peerName = Str( help='Name of the peer' )

   # Aggregated counters
   outBytes = Int( help='Number of out bytes for the application' )
   outPkts = Int( help='Number of out packets for the application' )
   throughput = Float( help='Throughput in Mbps for the application' )
   flows = Int( help='Number of flows for the application' )

   class PathGroups( Model ):
      # Aggregated counters
      # outBytes = Int( help='Number of bytes going via the path group' )
      # outPkts = Int( help='Number of packets going via the path group' )
      # throughput = Int( help='Throughput in Mbps of the path group' )
      # flows = Int( help='Number of flows going through the path group' )

      class Paths( Model ):
         outBytes = Int( help='Number of bytes going via the path' )
         outPkts = Int( help='Number of packets going via the path' )
         throughput = Float( help='Throughput in Mbps of the path' )
         flows = Int( help='Number of flows going through the path' )

      # key is pathId
      paths = Dict( keyType=int,
                    valueType=Paths,
                    help="A mapping of path index to its counters",
                    optional=True )

   # key is groupName
   pathGroups = Dict( keyType=str,
                      valueType=PathGroups,
                      help="A mapping of group name to its counters",
                      optional=True )

class DpsLbCounters( Model ):
   # key is lbId
   lbProfiles = Dict( keyType=int,
                      valueType=DpsLbProfiles,
                      help="A mapping of load-balance profile index" \
                      " to its path selection counters" )
   _detail = Bool( help='Show detail information' )

   def render( self ):
      if self._detail:
         # show path-selection load-balance counters detail
         header = [ 'App Profile', 'Vrf', 'Peer', 'Path Group', 'Path',
                    'Flows', 'Throughput(Mbps)', 'Out Bytes', 'Out Packets' ]
      else:
         # show path-selection load-balance counters
         header = [ 'App Profile', 'Vrf', 'Peer', 'Path Group', 'Path',
                    'Flows', 'Throughput(Mbps)' ]
      table = TableOutput.createTable( header, tableWidth=160 )
      table.formatColumns( *( fl for _ in header ) )

      for lbProfile in self.lbProfiles.values():
         peer = lbProfile.peerIp.stringValue + ( ' (' + lbProfile.peerName + ')' \
                          if lbProfile.peerName else '' )
         for groupName, group in sorted( lbProfile.pathGroups.items() ):
            for pathId, path in sorted( group.paths.items() ):

               if self._detail:
                  table.newRow( lbProfile.appProfile, lbProfile.vrf, peer,
                                groupName, "path" + str ( pathId ), path.flows,
                                path.throughput, path.outBytes, path.outPkts )
               else:
                  table.newRow( lbProfile.appProfile, lbProfile.vrf, peer,
                                groupName, "path" + str ( pathId ), path.flows,
                                path.throughput )
      print table.output()

class DpsAppCounters( Model ):
   # key is lbId
   lbProfiles = Dict( keyType=int,
                      valueType=DpsLbProfiles,
                      help="A mapping of load-balance profile index" \
                      " to its application counters" )

   def render( self ):
      # show path-selection application counters
      header = [ 'App Profile', 'Vrf', 'Peer', 'Throughput(Mbps)',
                 'Out Bytes', 'Out Packets' ]
      table = TableOutput.createTable( header, tableWidth=120 )
      table.formatColumns( *( fl for _ in header ) )
      for lbProfile in self.lbProfiles.values():
         peer = lbProfile.peerIp.stringValue + ( ' (' + lbProfile.peerName + ')' \
                                     if lbProfile.peerName else '' )
         table.newRow( lbProfile.appProfile, lbProfile.vrf, peer,
                       lbProfile.throughput, lbProfile.outBytes, lbProfile.outPkts )
      print table.output()

class DpsSession( Model ):
   active = Bool ( help='The telemetry session state is active' )
   seconds = Int( help='Number of seconds the telemetry session has'
                  ' been in this state' )

class DpsPath( Model ):
   source = Str( help='Source IP address of the path group' )
   destination = Str( help='Destination IP address of the path group' )
   state = Enum( values=( 'ipsecEstablished', 'resolved', 'arpPending',
                          'ipsecPending', 'routePending' ),
                 help='Routing state of this path' )
   dpsSessions = Dict( keyType=int,
                       valueType=DpsSession,
                       help="A mapping of traffic class to DPS session" )

class DpsGroup( Model ):
   dpsPaths = Dict( keyType=str,
                    valueType=DpsPath,
                    help='A mapping of path name to its state' )


class DpsPeer( Model ):
   peerName = Str( help='Peer name' )
   dpsGroups = Dict( keyType=str,
                     valueType=DpsGroup,
                     help='A mapping of group name to its paths' )

class DpsPaths( Model ):
   dpsPeers = Dict( keyType=IpGenericAddress,
                    valueType=DpsPeer,
                    help='A mapping of peer IP to peer path groups' )

   def render( self ):
      header = [ "Peer", "Path Group", "Source",
                 "Destination", "Path Name", "TC",
                 "Route State", "Telemetry State" ]
      table = TableOutput.createTable( header, tableWidth=160 )
      table.formatColumns( *( fl for _ in header ) )

      for peerIp in sorted( self.dpsPeers,
                            key=lambda peerIp: IpGenAddr( peerIp ).sortKey ):
         peer = self.dpsPeers[ peerIp ]
         for groupName, group in sorted( peer.dpsGroups.items() ):
            for pathName, path in sorted( group.dpsPaths.items() ):
               for tc, sess in sorted( path.dpsSessions.items() ):
                  peerName = peerIp + ( ' (' + peer.peerName + ')' \
                                        if peer.peerName != '' else '' )
                  sessState = 'inactive'
                  if sess.active:
                     sessState = 'active (' + str( datetime.timedelta( \
                                        seconds=sess.seconds ) ) + ')'
                  pathState = pathStateToStrMap[ path.state ]

                  table.newRow( peerName, groupName, path.source,
                                path.destination, pathName, tc,
                                pathState, sessState )
      print table.output()
