# Copyright (c) 2019 Arista Networks, Inc.  All rights reserved.
# Arista Networks, Inc. Confidential and Proprietary.

import weakref
import Tac
from ClassificationAppProfileId import updateAppProfileId

class DpsPathGroupContext( object ):
   def __init__( self, config, status, peerStatus, pathGroupName ):
      self.config = config
      self.status = status
      self.peerStatus = peerStatus
      self.pathGroupName_ = pathGroupName
      self.mode = None
      self.pgCfg = None
      self.routerIp = None
      self.routerIpCfg = None

   def modeIs( self, mode ):
      self.mode = weakref.ref( mode ) if mode else None

   def pathGroupName( self ):
      return self.pathGroupName_

   def addOrRemovePathGroup( self, pgName, add=True ):
      pathGroups = self.config.pathGroupConfig
      pgCfg = pathGroups.get( pgName )
      self.pathGroupName_ = pgName
      if add:
         if not pgCfg:
            pgCfg = self.config.newPathGroupConfig( pgName )
      else:
         del self.config.pathGroupConfig[ pgName ]
         pgCfg = None
      self.pgCfg = pgCfg

   def copyEditPathGroup( self ):
      pass

   def currentPgCfg( self ):
      return self.pgCfg

   def addOrRemoveRouterIp( self, routerIp, add=True ):
      remoteViaConfig = getattr( self.pgCfg, "remoteViaConfig" )
      router = Tac.Value( "Arnet::IpGenAddr", routerIp )
      if add:
         remoteViaConfig.newMember( router )
         self.routerIpCfg = remoteViaConfig[ router ]
      else:

         del remoteViaConfig[ router ]
      self.routerIp = routerIp

   def currentRouterIp( self ):
      return self.routerIp

   def addOrRemoveDynamicPeer( self, add=True ):
      self.pgCfg.remoteDynamic = ( add )

   def addOrRemoveIntf( self, intfName, add=True ):
      intf = Tac.Value( "Arnet::IntfId", intfName )
      if add:
         self.pgCfg.localIntf[ intf ] = True
      else:
         del self.pgCfg.localIntf[ intf ]

   def addOrRemoveLocalIp( self, ip, add=True ):
      ipEntry = Tac.Value( "Arnet::IpGenAddr", ip )
      if add:
         self.pgCfg.localIp[ ipEntry ] = True
      else:
         del self.pgCfg.localIp[ ipEntry ]

   def addOrRemovePathGroupIpsec( self, profileName, add=True ):
      if add:
         self.pgCfg.ipsecProfile = profileName
      else:
         self.pgCfg.ipsecProfile = ""

   def setMss( self, mss ):
      self.pgCfg.mssEgress = mss

   def addOrRemoveRouterVia( self, viaIp, add=True ):
      remoteVia = getattr( self.routerIpCfg, "remoteVia" )
      via = Tac.Value( "Arnet::IpGenAddr", viaIp )
      if add:
         remoteVia[ via ] = True
      else:
         del remoteVia[ via ]

   def checkPeerName( self, peerName ):
      peerStatusEntry = self.peerStatus.peerStatusEntry
      router = Tac.Value( "Arnet::IpGenAddr", self.routerIp )
      if ( router not in peerStatusEntry ) or \
         ( peerStatusEntry[ router ].peerName == peerName ) or \
         ( not peerStatusEntry[ router ].groupName ) or \
         ( len( peerStatusEntry[ router ].groupName ) == 1 and \
           self.pathGroupName_ in peerStatusEntry[ router ].groupName ):
         return None
      else:
         return ( router.stringValue, peerStatusEntry[ router ].peerName )

   def addOrRemovePeerName( self, peerName, add=True ):
      if add:
         self.routerIpCfg.peerName = peerName
      else:
         self.routerIpCfg.peerName = ""

class DpsPolicyContext( object ):
   def __init__( self, config, status, policyName, appProfileIdMap ):
      self.config = config
      self.status = status
      self.policyName_ = policyName
      self.mode = None
      self.policyCfg = None
      self.ruleKey = None
      self.ruleKeyCfg = None
      self.appProfile = None
      self.defaultRuleCfgd = None
      self.appProfileIdMap = appProfileIdMap
      self.appProfileId = None

   def modeIs( self, mode ):
      self.mode = weakref.ref( mode ) if mode else None

   def policyName( self ):
      return self.policyName_

   def addOrRemovePolicy( self, policyName, add=True ):
      policies = self.config.policyConfig
      policyCfg = policies.get( policyName )
      if add:
         if not policyCfg:
            policyCfg = self.config.newPolicyConfig( policyName )
      else:
         del self.config.policyConfig[ policyName ]
         policyCfg = None
      self.policyCfg = policyCfg

   def copyEditPolicyCfg( self ):
      pass

   def currentPolicyCfg( self ):
      return self.policyCfg

   def currentRuleKey( self ):
      return self.ruleKey

   def currentAppProfile( self ):
      return self.appProfile

   def currentDefaultRuleCfgd( self ):
      return self.defaultRuleCfgd

   def addOrRemoveRuleKey( self, ruleKey, appProfile, add=True ):
      appProfilePolicyRuleList = getattr( self.policyCfg,
            "appProfilePolicyRuleList" )
      if add:
         appProfilePolicyRuleList.newMember( ruleKey )
         appProfilePolicyRuleList[ ruleKey ].appProfileName = appProfile
         self.appProfileId = updateAppProfileId( self.appProfileIdMap, appProfile )
         appProfilePolicyRuleList[ ruleKey ].appProfileId = self.appProfileId
         self.ruleKey = ruleKey
         self.appProfile = appProfile
         self.ruleKeyCfg = appProfilePolicyRuleList[ ruleKey ]
         self.defaultRuleCfgd = False
      else:
         del appProfilePolicyRuleList[ ruleKey ]
         self.ruleKey = None
         self.appProfile = None
         self.ruleKeyCfg = None
      self.defaultRuleCfgd = False

   def addOrRemoveDefaultRule( self, add=True ):
      if add:
         self.policyCfg.defaultRuleCfgd = True
         self.defaultRuleCfgd = True
      else:
         self.policyCfg.defaultRuleCfgd = False
         self.policyCfg.defaultLbGrpName = ""
         self.defaultRuleCfgd = False

   def setLbGrpName( self, lbGrpName ):
      if self.defaultRuleCfgd:
         self.policyCfg.defaultLbGrpName = lbGrpName
      else:
         self.ruleKeyCfg.lbGrpName = lbGrpName

class DpsLoadBalanceProfileContext( object ):
   def __init__( self, config, status, profileName ):
      self.config = config
      self.status = status
      self.profileName_ = profileName
      self.mode = None
      self.profile = None

   def modeIs( self, mode ):
      self.mode = weakref.ref( mode ) if mode else None

   def profileName( self ):
      return self.profileName_

   def addProfile( self, profileName ):
      profile = self.config.loadBalanceProfile.get( profileName )
      if not profile:
         profile = self.config.newLoadBalanceProfile( profileName )
      self.profile = profile

   def delProfile( self, profileName ):
      profile = self.config.loadBalanceProfile.get( profileName )
      if profile:
         del self.config.loadBalanceProfile[ profileName ]
      self.profile = None

   def addPathGroup( self, pathGroupName, priority ):
      profile = self.currentProfile()
      if profile:
         profile.pathGroupPriority[ pathGroupName ] = priority

   def delPathGroup( self, pathGroupName ):
      profile = self.currentProfile()
      if profile:
         if pathGroupName in profile.pathGroupPriority:
            del profile.pathGroupPriority[ pathGroupName ]

   def setLatency( self, latency ):
      profile = self.currentProfile()
      profile.latency = latency

   def setJitter( self, jitter ):
      profile = self.currentProfile()
      profile.jitter = jitter

   def setLossRate( self, lossRate ):
      profile = self.currentProfile()
      profile.lossRate = lossRate

   def currentProfile( self ):
      return self.profile

class DpsVrfConfigContext( object ):
   def __init__( self, config, status, vrfName ):
      self.config = config
      self.status = status
      self.vrfName_ = vrfName
      self.mode = None
      self.vrfConfig = None

   def modeIs( self, mode ):
      self.mode = weakref.ref( mode ) if mode else None

   def vrfName( self ):
      return self.vrfName_

   def currentVrfCfg( self ):
      return self.vrfConfig

   def addVrfConfig( self, vrfName ):
      vrfCfg = self.config.vrfConfig.get( vrfName )
      if not vrfCfg:
         vrfCfg = self.config.newVrfConfig( vrfName )
      self.vrfConfig = vrfCfg

   def delVrfConfig( self, vrfName ):
      vrfCfg = self.config.vrfConfig.get( vrfName )
      if vrfCfg:
         del self.config.vrfConfig[ vrfName ]
      self.vrfConfig = None

   def setPolicy( self, policyName ):
      self.vrfConfig.policyName = policyName

class DpsEncapConfigContext( object ):
   def __init__( self, config, status ):
      self.config = config
      self.status = status
      self.mode = None

   def modeIs( self, mode ):
      self.mode = weakref.ref( mode ) if mode else None

   def currentUdpPortCfg( self ):
      return self.config.udpPortConfig()

   def setUdpPort( self, portNum ):
      self.config.udpPortConfig = portNum
