# Copyright (c) 2019 Arista Networks, Inc.  All rights reserved.
# Arista Networks, Inc. Confidential and Proprietary.

import Tac
import BasicCli
import CliCommand
import CliMatcher
import CliParser
import ConfigMount
import EthIntfCli
import IpAddrMatcher
import LazyMount
from CliMode.Dps import ( RouterPathSelectionConfigMode,
                          DpsPathGroupConfigMode,
                          DpsPathGroupRemoteRouterConfigMode,
                          DpsPolicyConfigMode,
                          DpsPolicyRuleKeyConfigMode,
                          DpsPolicyDefaultRuleConfigMode,
                          DpsLoadBalanceProfileConfigMode,
                          DpsVrfConfigMode,
                          DEFAULT_PRIORITY,
                          DEFAULT_JITTER, JITTER_SCALE,
                          DEFAULT_LATENCY, LATENCY_SCALE,
                          DEFAULT_LOSSRATE, LOSS_RATE_SCALE, LOSS_RATE_ADJUSTMENT,
                          DEFAULT_UDP_PORT )
from CliToken.Router import routerMatcherForConfig
from DpsCliLib import ( DpsPathGroupContext,
                        DpsPolicyContext,
                        DpsLoadBalanceProfileContext,
                        DpsVrfConfigContext,
                        DpsEncapConfigContext )
from DpsShowCli import pathSelectionSupportedGuard

#from IraVrfCli import getAllVrfNames

MAX_RULES = 255

dpsCliConfig = None
dpsStatus = None
entityManager = None
appProfileIdMap = None
ipsecConfig = None
peerStatus = None

tokenRuleKey = CliMatcher.IntegerMatcher( 1, MAX_RULES, helpdesc='rule key' )

tokenMilliSecond = CliMatcher.IntegerMatcher( 0, 10000, helpdesc='millisecond' )

tokenPercentage = CliMatcher.FloatMatcher( 0, 100,
                                  helpdesc='percentage',
                                  precisionString='%.2f' )

pathSelectionMatcher = CliMatcher.KeywordMatcher( 'path-selection',
                                     helpdesc='Configure dynamic path selection' )
localMatcherForPathGroup = CliMatcher.KeywordMatcher( 'local',
                                 helpdesc='Configure local info for DPS path group' )
pathSelectionNode = CliCommand.Node( matcher=pathSelectionMatcher,
                                     guard=pathSelectionSupportedGuard )

pathGroupConfigNameMatcher = CliMatcher.DynamicNameMatcher(
      lambda mode: [] if dpsCliConfig is None else dpsCliConfig.pathGroupConfig,
      helpdesc='name of the path group',
      helpname='WORD',
      priority=CliParser.PRIO_LOW )

policyConfigNameMatcher = CliMatcher.DynamicNameMatcher(
      lambda mode: [] if dpsCliConfig is None else dpsCliConfig.policyConfig,
      helpdesc='name of the Dps policy',
      helpname='WORD',
      priority=CliParser.PRIO_LOW )

loadBalanceProfileNameMatcher = CliMatcher.DynamicNameMatcher(
      lambda mode: [] if dpsCliConfig is None else dpsCliConfig.loadBalanceProfile,
      helpdesc='name of the load-balance policy',
      helpname='WORD',
      priority=CliParser.PRIO_LOW )

ipsecProfileNameMatcher = CliMatcher.DynamicNameMatcher(
      lambda mode: [] if ipsecConfig is None else ipsecConfig.ipsecProfile,
      helpdesc='name of the Dps pathGroup ipsec profile',
      helpname='WORD',
      priority=CliParser.PRIO_LOW )

vrfConfigNameMatcher = CliMatcher.DynamicNameMatcher(
      lambda mode: [] if dpsCliConfig is None else dpsCliConfig.vrfConfig,
      helpdesc='name of the vrf',
      helpname='WORD',
      priority=CliParser.PRIO_LOW )

nameToIdNameMatcher = CliMatcher.DynamicNameMatcher(
      lambda mode: [] if appProfileIdMap is None else appProfileIdMap.nameToId,
      helpdesc='name of the application profile',
      helpname='WORD',
      priority=CliParser.PRIO_LOW )

siteNameMatcher = CliMatcher.PatternMatcher( pattern='[a-zA-Z0-9_-]+',
                                             helpname='WORD',
                                             helpdesc='name of the site' )


def getPathGroupNames():
   return dpsCliConfig.pathGroupConfig.iterKeys()

class RouterPathSelectionConfigCmd( CliCommand.CliCommandClass ):
   syntax = '''router path-selection'''
   noOrDefaultSyntax = syntax
   data = {
         'router': routerMatcherForConfig,
         'path-selection': pathSelectionNode,
   }

   @staticmethod
   def handler( mode, args ):
      dpsCliConfig.dpsConfigured = True
      childMode = mode.childMode( RouterPathSelectionConfigMode )
      mode.session_.gotoChildMode( childMode )

   @staticmethod
   def noOrDefaultHandler( mode, args ):
      dpsCliConfig.udpPortConfig = DEFAULT_UDP_PORT
      dpsCliConfig.pathGroupConfig.clear()
      dpsCliConfig.loadBalanceProfile.clear()
      dpsCliConfig.policyConfig.clear()
      dpsCliConfig.vrfConfig.clear()

class DpsPathGroupConfigCmd( CliCommand.CliCommandClass ):
   syntax = '''path-group <path-group-name>'''
   noOrDefaultSyntax = syntax
   data = {
         'path-group': 'Configure path group',
         '<path-group-name>': pathGroupConfigNameMatcher
   }

   @staticmethod
   def handler( mode, args ):
      # DpsPathGroupConfigCmd.registerNoHandler(
      #  lambda: self.noDpsPathGroupHandler( mode ) )
      name = args.get( '<path-group-name>' )
      context = DpsPathGroupContext( dpsCliConfig, dpsStatus, peerStatus, name )

      context.addOrRemovePathGroup( name, True )
      childMode = mode.childMode( DpsPathGroupConfigMode, context=context )
      mode.session_.gotoChildMode( childMode )

   @staticmethod
   def noOrDefaultHandler( mode, args ):
      name = args.get( '<path-group-name>' )
      context = DpsPathGroupContext( dpsCliConfig, dpsStatus, peerStatus, name )

      context.addOrRemovePathGroup( name, False )

class DpsPathGroupIpsecConfigCmd( CliCommand.CliCommandClass ):
   syntax = '''ipsec profile <profile>'''
   noOrDefaultSyntax = syntax

   data = {
         'ipsec': 'Configure ipsec for DPS Path Group',
         'profile': 'Configure ipsec profile for DPS Path Group',
         '<profile>': ipsecProfileNameMatcher
   }

   @staticmethod
   def handler( mode, args ):
      profile = args[ '<profile>' ]
      mode.context.addOrRemovePathGroupIpsec( profile, True )

   @staticmethod
   def noOrDefaultHandler( mode, args ):
      profile = args[ '<profile>' ]
      mode.context.addOrRemovePathGroupIpsec( profile, False )

class DpsIntfConfigCmd( CliCommand.CliCommandClass ):
   syntax = '''local interface INTF'''
   noOrDefaultSyntax = syntax

   data = {
         'local': localMatcherForPathGroup,
         'interface': 'interface',
         'INTF': EthIntfCli.EthPhyIntf.ethMatcher,
   }

   @staticmethod
   def handler( mode, args ):
      intf = args[ 'INTF' ]
      mode.context.addOrRemoveIntf( intf.name, True )

   @staticmethod
   def noOrDefaultHandler( mode, args ):
      intf = args[ 'INTF' ]
      mode.context.addOrRemoveIntf( intf.name, False )

class DpsPathGroupIpConfigCmd( CliCommand.CliCommandClass ):
   syntax = '''local ip IPADDR'''
   noOrDefaultSyntax = syntax

   data = {
         'local': localMatcherForPathGroup,
         'ip': 'ip',
         'IPADDR': IpAddrMatcher.ipAddrMatcher,
   }

   @staticmethod
   def handler( mode, args ):
      localIp = args[ 'IPADDR' ]
      mode.context.addOrRemoveLocalIp( localIp, True )

   @staticmethod
   def noOrDefaultHandler( mode, args ):
      localIp = args[ 'IPADDR' ]
      mode.context.addOrRemoveLocalIp( localIp, False )


class DpsPathGroupRemoteRouterCmd( CliCommand.CliCommandClass ):
   syntax = '''peer static router-ip IPADDR'''
   noOrDefaultSyntax = syntax
   data = {
         'peer': 'Configure Peer Router Info for DPS Path Group',
         'static': 'Static peer router',
         'router-ip': 'Peer router-IP',
         'IPADDR': IpAddrMatcher.ipAddrMatcher,
   }

   @staticmethod
   def handler( mode, args ):
      routerIp = args[ 'IPADDR' ]
      mode.context.addOrRemoveRouterIp( routerIp, True )
      childMode = mode.childMode( DpsPathGroupRemoteRouterConfigMode,
                                  context=mode.context )
      mode.session_.gotoChildMode( childMode )

   @staticmethod
   def noOrDefaultHandler( mode, args ):
      routerIp = args[ 'IPADDR' ]
      mode.context.addOrRemoveRouterIp( routerIp, False )

class DpsPathGroupPeerDynamicCmd( CliCommand.CliCommandClass ):
   syntax = '''peer dynamic'''
   noOrDefaultSyntax = syntax
   data = {
         'peer': 'Configure Peer Router Info for DPS Path Group',
         'dynamic': 'Peer discovered dynamically throught BGP',
   }

   @staticmethod
   def handler( mode, args ):
      mode.context.addOrRemoveDynamicPeer( True )

   @staticmethod
   def noOrDefaultHandler( mode, args ):
      mode.context.addOrRemoveDynamicPeer( False )

class DpsPathGroupPeerName( CliCommand.CliCommandClass ):
   syntax = '''name NAME'''
   noOrDefaultSyntax = '''name ...'''
   data = {
      'name': 'Configure Peer Router Name',
      'NAME': siteNameMatcher,
   }

   @staticmethod
   def handler( mode, args ):
      peerName = args[ 'NAME' ]
      curName = mode.context.checkPeerName( peerName )
      if curName:
         warning = "Peer %s has been configured with name %s." % \
                   ( curName[ 0 ], curName[ 1 ] )
         mode.addWarning( warning )
      else:
         mode.context.addOrRemovePeerName( peerName, True )

   @staticmethod
   def noOrDefaultHandler( mode, args ):
      mode.context.addOrRemovePeerName( '', False )

class DpsPathGroupRemoteViaCmd( CliCommand.CliCommandClass ):
   syntax = '''ipv4 address IPADDR'''
   noOrDefaultSyntax = syntax
   data = {
      'ipv4': 'Configure Peer Router Via for DPS Path Group',
      'address': 'Static ipv4 address',
      'IPADDR': IpAddrMatcher.ipAddrMatcher,
   }

   @staticmethod
   def handler( mode, args ):
      viaIp = args[ 'IPADDR' ]
      mode.context.addOrRemoveRouterVia( viaIp, True )

   @staticmethod
   def noOrDefaultHandler( mode, args ):
      viaIp = args[ 'IPADDR' ]
      mode.context.addOrRemoveRouterVia( viaIp, False )

class DpsPathGroupTcpMssCeilingCmd( CliCommand.CliCommandClass ):
   syntax = '''tcp mss ceiling
               ipv4 MSSv4 egress'''
   noOrDefaultSyntax = '''tcp mss ceiling ...'''
   data = {
      'tcp': 'TCP',
      'mss': 'Maximum segment size',
      'ceiling': 'Set maximum limit',
      'ipv4': 'Internet Protocol version 4',
      'MSSv4': CliMatcher.IntegerMatcher( 64, 65515,
                                          helpdesc='Segment size' ),
      'egress': 'Enforce on packets forwarded to the network',
   }

   @staticmethod
   def handler( mode, args ):
      ipv4TcpMss = args.get( 'MSSv4', 0 )
      mode.context.setMss( ipv4TcpMss )

   @staticmethod
   def noOrDefaultHandler( mode, args ):
      mode.context.setMss( 0 )

class DpsPolicyConfigCmd( CliCommand.CliCommandClass ):
   syntax = '''policy <policy-name>'''
   noOrDefaultSyntax = syntax
   data = {
         'policy': 'Configure Dps policy',
         '<policy-name>': policyConfigNameMatcher,
   }

   @staticmethod
   def handler( mode, args ):
      name = args.get( '<policy-name>' )
      context = DpsPolicyContext( dpsCliConfig, dpsStatus, name, appProfileIdMap )

      context.addOrRemovePolicy( name, True )
      childMode = mode.childMode( DpsPolicyConfigMode, context=context )
      mode.session_.gotoChildMode( childMode )

   @staticmethod
   def noOrDefaultHandler( mode, args ):
      name = args.get( '<policy-name>' )
      context = DpsPolicyContext( dpsCliConfig, dpsStatus, name, appProfileIdMap )
      context.addOrRemovePolicy( name, False )

class DpsPolicyRuleKeyConfigCmd( CliCommand.CliCommandClass ):
   syntax = '''<rule-key> application-profile <app-profile>'''
   noOrDefaultSyntax = syntax
   data = {
         '<rule-key>': tokenRuleKey,
         'application-profile': 'application-profile',
         '<app-profile>': nameToIdNameMatcher
   }

   @staticmethod
   def handler( mode, args ):
      ruleKey = args.get( '<rule-key>' )
      appProfile = args.get( '<app-profile>' )
      mode.context.addOrRemoveRuleKey( ruleKey, appProfile, True )
      mode.rulesChanged = True
      childMode = mode.childMode( DpsPolicyRuleKeyConfigMode,
                                  context=mode.context )
      mode.session_.gotoChildMode( childMode )

   @staticmethod
   def noOrDefaultHandler( mode, args ):
      ruleKey = args.get( '<rule-key>' )
      appProfile = args.get( '<app-profile>' )
      mode.context.addOrRemoveRuleKey( ruleKey, appProfile, False )
      mode.rulesChanged = True

class DpsPolicyDefaultRuleConfigCmd( CliCommand.CliCommandClass ):
   syntax = '''default-match'''
   noOrDefaultSyntax = syntax
   data = {
         'default-match': 'default matching rule',
   }

   @staticmethod
   def handler( mode, args ):
      mode.context.addOrRemoveDefaultRule( True )
      childMode = mode.childMode( DpsPolicyDefaultRuleConfigMode,
                                  context=mode.context )
      mode.session_.gotoChildMode( childMode )
      mode.rulesChanged = True

   @staticmethod
   def noOrDefaultHandler( mode, args ):
      mode.context.addOrRemoveDefaultRule( False )
      mode.rulesChanged = True

class DpsPolicyRuleLbCmd( CliCommand.CliCommandClass ):
   syntax = '''load-balance <load-balance-grp>'''
   noOrDefaultSyntax = '''load-balance ...'''
   data = {
         'load-balance': 'attach load-balance group',
         '<load-balance-grp>': loadBalanceProfileNameMatcher,
   }

   @staticmethod
   def handler( mode, args ):
      lbGrpName = args.get( '<load-balance-grp>' )
      mode.context.setLbGrpName( lbGrpName )

   @staticmethod
   def noOrDefaultHandler( mode, args ):
      mode.context.setLbGrpName( "" )

class DpsLoadBalanceProfileConfigCmd( CliCommand.CliCommandClass ):
   syntax = '''load-balance policy <profile-name>'''
   noOrDefaultSyntax = syntax
   data = {
         'load-balance': 'Configure load balance parameters',
         'policy': 'Configure load balance policy parameters',
         '<profile-name>': loadBalanceProfileNameMatcher,
   }

   @staticmethod
   def handler( mode, args ):
      name = args.get( '<profile-name>' )
      context = DpsLoadBalanceProfileContext( dpsCliConfig, dpsStatus, name )
      context.addProfile( name )
      childMode = mode.childMode( DpsLoadBalanceProfileConfigMode, context=context )
      mode.session_.gotoChildMode( childMode )

   @staticmethod
   def noOrDefaultHandler( mode, args ):
      name = args.get( '<profile-name>' )
      context = DpsLoadBalanceProfileContext( dpsCliConfig, dpsStatus, name )
      context.delProfile( name )

class DpsLbpLatencyConfigCmd( CliCommand.CliCommandClass ):
   syntax = '''latency MILLISECONDS'''
   noOrDefaultSyntax = 'latency ...'

   data = {
         'latency': 'Configure one way delay requirement'
                    ' for this load balance policy',
         'MILLISECONDS': tokenMilliSecond,
   }

   @staticmethod
   def handler( mode, args ):
      timeArg = args.get( 'MILLISECONDS' )
      mode.context.setLatency( int( timeArg ) * LATENCY_SCALE )

   @staticmethod
   def noOrDefaultHandler( mode, args ):
      mode.context.setLatency( DEFAULT_LATENCY )

class DpsLbpJitterConfigCmd( CliCommand.CliCommandClass ):
   syntax = '''jitter MILLISECONDS'''
   noOrDefaultSyntax = 'jitter ...'

   data = {
         'jitter': 'Configure jitter requirement for this load balance policy',
         'MILLISECONDS': tokenMilliSecond,
   }

   @staticmethod
   def handler( mode, args ):
      timeArg = args.get( 'MILLISECONDS' )
      mode.context.setJitter( int( timeArg ) * JITTER_SCALE )

   @staticmethod
   def noOrDefaultHandler( mode, args ):
      mode.context.setJitter( DEFAULT_JITTER )

class DpsLbpLossRateConfigCmd( CliCommand.CliCommandClass ):
   syntax = '''loss-rate <percentage>'''
   noOrDefaultSyntax = 'loss-rate ...'

   data = {
         'loss-rate': 'Configure loss rate requirement for this load balance policy',
         '<percentage>': tokenPercentage,
   }

   @staticmethod
   def handler( mode, args ):
      percentArg = args.get( '<percentage>' )
      loss = int( float( percentArg ) * LOSS_RATE_SCALE )
      loss += LOSS_RATE_ADJUSTMENT
      mode.context.setLossRate( loss )

   @staticmethod
   def noOrDefaultHandler( mode, args ):
      mode.context.setLossRate( DEFAULT_LOSSRATE )

class DpsLbpPathGroupConfigCmd( CliCommand.CliCommandClass ):
   syntax = '''path-group <name> [ priority <priority> ]'''
   noOrDefaultSyntax = '''path-group <name> ...'''

   data = {
      'path-group': 'Configure the path-group to use with this load balance policy',
      '<name>': pathGroupConfigNameMatcher,
      'priority': 'Configure priority for this path group',
      '<priority>': CliMatcher.PatternMatcher( pattern='[0-9]+',
                                                helpname='number',
                                                helpdesc='priority of this group' )
   }

   @staticmethod
   def handler( mode, args ):
      pathGroup = args.get( '<name>' )
      priority = DEFAULT_PRIORITY
      if '<priority>' in args.keys():
         priority = int( args.get( '<priority>' ) )
      mode.context.addPathGroup( pathGroup, priority )

   @staticmethod
   def noOrDefaultHandler( mode, args ):
      pathGroup = args.get( '<name>' )
      mode.context.delPathGroup( pathGroup )

class DpsVrfConfigCmd( CliCommand.CliCommandClass ):
   syntax = '''vrf <vrf-name>'''
   noOrDefaultSyntax = syntax
   data = {
         'vrf': 'Configure path selection policy for a vrf',
         '<vrf-name>': vrfConfigNameMatcher,
   }

   @staticmethod
   def handler( mode, args ):
      name = args.get( '<vrf-name>' )
      context = DpsVrfConfigContext( dpsCliConfig, dpsStatus, name )

      context.addVrfConfig( name )
      childMode = mode.childMode( DpsVrfConfigMode, context=context )
      mode.session_.gotoChildMode( childMode )

   @staticmethod
   def noOrDefaultHandler( mode, args ):
      name = args.get( '<vrf-name>' )
      context = DpsVrfConfigContext( dpsCliConfig, dpsStatus, name )

      context.delVrfConfig( name )
# XXX: The cpu keyword may not be used in the future.
# a design change may let the customer use the same policy for both data
# path and cpu path.
class DpsVrfPathSelectionPolicyConfigCmd( CliCommand.CliCommandClass ):
   syntax = '''path-selection-policy <policy-name>'''
   noOrDefaultSyntax = 'path-selection-policy ...'

   data = {
      'path-selection-policy': 'Configure policy name to use for this vrf',
      '<policy-name>': policyConfigNameMatcher,
   }

   @staticmethod
   def handler( mode, args ):
      policyName = args.get( '<policy-name>' )
      mode.context.setPolicy( policyName )

   @staticmethod
   def noOrDefaultHandler( mode, args ):
      mode.context.setPolicy( "" )

class DpsEncapConfigCmd( CliCommand.CliCommandClass ):
   syntax = '''encapsulation path-telemetry udp port PORT'''
   noOrDefaultSyntax = '''encapsulation path-telemetry udp port ...'''
   data = {
      'encapsulation': 'Configure encapsulation parameters',
      'path-telemetry': 'Configure encapsulation path-telemetry parameters',
      'udp': 'Configure UDP parameters',
      'port': 'Configure UDP port number',
      'PORT': CliMatcher.IntegerMatcher( 1, 65536, helpdesc='Destination UDP port' ),
   }

   @staticmethod
   def handler( mode, args ):
      portNum = args.get( 'PORT', DEFAULT_UDP_PORT )
      context = DpsEncapConfigContext( dpsCliConfig, dpsStatus )
      context.setUdpPort( portNum )

   noOrDefaultHandler = handler

# router path-selection
BasicCli.GlobalConfigMode.addCommandClass( RouterPathSelectionConfigCmd )

# path-group sub-commands
RouterPathSelectionConfigMode.addCommandClass( DpsPathGroupConfigCmd )
DpsPathGroupConfigMode.addCommandClass( DpsIntfConfigCmd )
DpsPathGroupConfigMode.addCommandClass( DpsPathGroupIpConfigCmd )
DpsPathGroupConfigMode.addCommandClass( DpsPathGroupTcpMssCeilingCmd )
DpsPathGroupConfigMode.addCommandClass( DpsPathGroupIpsecConfigCmd )
DpsPathGroupConfigMode.addCommandClass( DpsPathGroupRemoteRouterCmd )
DpsPathGroupConfigMode.addCommandClass( DpsPathGroupPeerDynamicCmd )
DpsPathGroupRemoteRouterConfigMode.addCommandClass( DpsPathGroupRemoteViaCmd )
DpsPathGroupRemoteRouterConfigMode.addCommandClass( DpsPathGroupPeerName )

# policy sub-commands
RouterPathSelectionConfigMode.addCommandClass( DpsPolicyConfigCmd )
DpsPolicyConfigMode.addCommandClass( DpsPolicyRuleKeyConfigCmd )
DpsPolicyConfigMode.addCommandClass( DpsPolicyDefaultRuleConfigCmd )
DpsPolicyRuleKeyConfigMode.addCommandClass( DpsPolicyRuleLbCmd )
DpsPolicyDefaultRuleConfigMode.addCommandClass( DpsPolicyRuleLbCmd )

# load-balance
RouterPathSelectionConfigMode.addCommandClass( DpsLoadBalanceProfileConfigCmd )
DpsLoadBalanceProfileConfigMode.addCommandClass( DpsLbpLatencyConfigCmd )
DpsLoadBalanceProfileConfigMode.addCommandClass( DpsLbpJitterConfigCmd )
DpsLoadBalanceProfileConfigMode.addCommandClass( DpsLbpLossRateConfigCmd )
DpsLoadBalanceProfileConfigMode.addCommandClass( DpsLbpPathGroupConfigCmd )

# Vrf
RouterPathSelectionConfigMode.addCommandClass( DpsVrfConfigCmd )
DpsVrfConfigMode.addCommandClass( DpsVrfPathSelectionPolicyConfigCmd )

# UDP port
RouterPathSelectionConfigMode.addCommandClass( DpsEncapConfigCmd )

def Plugin( em ):
   global dpsCliConfig, dpsStatus, appProfileIdMap, peerStatus
   global entityManager
   global ipsecConfig
   entityManager = em
   mountGroup = entityManager.mountGroup()
   dpsStatus = mountGroup.mount(
      'dps/status',
      'Tac::Dir', 'ri' )
   mountGroup.close( callback=None, blocking=False )

   dpsCliConfig = ConfigMount.mount( entityManager,
        'dps/input/cli',
        'Dps::DpsCliConfig', 'wi' )
   appProfileIdMap = ConfigMount.mount(
      entityManager, 'classification/app-profile-id',
      'Classification::AppProfileIdMap', 'w' )
   ipsecConfig = LazyMount.mount(
      entityManager, 'ipsec/ike/config',
      'Ipsec::Ike::Config', 'r' )
   peerStatus = LazyMount.mount(
      entityManager, 'dps/peer/status',
      'Dps::PeerStatus', 'r' )
