#!/usr/bin/env python
# Copyright (c) 2016 Arista Networks, Inc.  All rights reserved.
# Arista Networks, Inc. Confidential and Proprietary.

from CliModel import Dict, Model, Enum, Str
import CliPlugin.FruModel as FruModel
import re

def pickNth( o, idx ):
   return o[idx] if isinstance( o, tuple ) else o

states = [
   "disabled",
   "initializing",
   "failed",
   "active",
   ( "freeRunning", 'free-running' )
]
statesJson = [ pickNth( s, 0 ) for s in states ]
jsonToCli = dict( [ ( pickNth( s, 0 ), pickNth( s, 1 ) ) for s in states ] )

roleMap = {
   'pllTimeSync' : 'timeSync',
   'pllTimeSyncStandalone' : 'timeSyncFreeRun',
   'pllPhy' : 'phy',
   'pllServo' : 'servo',
   'pllSystem' : 'system',
}
stateMap = {
   'pllStateUnknown' : 'disabled',
   'pllStateHwNotPresent' : 'disabled',
   'pllStateInitializing' : 'initializing',
   'pllStateFailed' : 'failed',
   'pllStateLocking' : 'initializing',
   'pllStateFreeRunning' : 'freeRunning',
   'pllStateClkInput' : 'active',
}

clockNameRe = re.compile( "^([A-Za-z]+)([0-9]*)/([0-9]+)$" )
moduleOrder = {
   'Supervisor' : 0,
   'Fabric' : 1,
   'Linecard' : 2
}

def parseClockName( clockName ):
   m = clockNameRe.match( clockName )
   if m:
      module = m.group( 1 )
      module = moduleOrder.get( module, module )
      return ( module, int( m.group( 2 ) or "0" ), int( m.group( 3 ) ) )
   return clockName

class ClockModel( Model ):
   model = Str( help="PLL manufacturer and model" )
   description = Str( help="PLL description" )
   role = Enum( help="PLL function", values=( "timeSync", "phy", "servo",
                                              "timeSyncFreeRun", "system" ) )
   state = Enum( values=statesJson,
                 help="Current state of the PLL" )

class PlatformModel( Model ):
   clocks = Dict( keyType=str, valueType=ClockModel,
                  help="Clock name to clock information" )
   def render( self ):
      fmt = "%-20s %-30s %-25s %15s %15s"
      print fmt % ( "Name", "Model", "Description", "State", "Role" )
      FruModel.printDashes( fmt )
      for clockName in sorted( self.clocks.keys(), key=parseClockName ):
         clockInfo = self.clocks[ clockName ]
         print fmt % ( clockName, clockInfo.model, clockInfo.description,
                       jsonToCli[ clockInfo.state ], clockInfo.role )

