#!/usr/bin/env python
# Copyright (c) 2016 Arista Networks, Inc.  All rights reserved.
# Arista Networks, Inc. Confidential and Proprietary.

import BasicCli
import CliCommand
import CliMatcher
import CliParser
import CliPlugin.FruCli as FruCli
import CliPlugin.TechSupportCli as TechSupportCli
import CliToken.Reset
import CliToken.Hardware
from DpllModel import ClockModel, PlatformModel, roleMap, stateMap, parseClockName
import LazyMount
import ShowCommand
import Tac

allPllConfig = None
allPllStatus = None
resetConfig = None

#--------------------------------------------------------------------------------
# show hardware pll [ module MODULE ]
#--------------------------------------------------------------------------------
class HardwarePllCmd( ShowCommand.ShowCliCommandClass ):
   syntax = 'show hardware pll [ module MODULE ]'
   data = {
      'hardware' : CliToken.Hardware.hardwareMatcherForShow,
      'pll' : 'Show PLL status',
      'module' : CliCommand.guardedKeyword( 'module',
         helpdesc='Show PLL status for a specific module',
         guard=FruCli.modularSystemGuard ),
      'MODULE' : FruCli.SlotExpressionFactory(),
   }
   cliModel = PlatformModel

   @staticmethod
   def handler( mode, args ):
      slotName = args.get( 'MODULE' )
      clocks = {}
      for pllModel in allPllConfig:
         thisPllTypeConfig = allPllConfig.entity[ pllModel ]
         thisPllTypeStatus = allPllStatus.entity[ pllModel ]
         for pll in thisPllTypeConfig.pllConfig.values():
            thisSlotName = pll.name.split( '/' )[ 0 ]
            if ( slotName is not None and
                 slotName.tag + slotName.label != thisSlotName ):
               continue
            pllStatus = thisPllTypeStatus.pll.get( pll.name, None )
            if pllStatus and pllStatus.genId != pll.genId:
               # status is stale but config exists
               # display 'pllStateUnknown' as state
               pllStatus = None
            pllm = ClockModel( model='%s %s' % ( pll.mfgName, pll.modelName ),
                               description=pll.description, role=roleMap[ pll.role ],
                               state=stateMap[ pllStatus.state if pllStatus
                                               else 'pllStateUnknown' ] )
            clocks[ pll.name ] = pllm
      return PlatformModel( clocks=clocks )

BasicCli.addShowCommandClass( HardwarePllCmd )

#--------------------------------------------------------------------------------
# reset hardware pll CLOCK_NAME
#--------------------------------------------------------------------------------
def clockNames():
   res = set()
   for pllModel in allPllConfig:
      thisPllType = allPllConfig.entity[ pllModel ]
      for pll in thisPllType.pllConfig.values():
         res.add( pll.name )
   return sorted( res, key=parseClockName )

def doUpdatePlls():
   for pllModel in allPllConfig:
      if pllModel not in resetConfig:
         resetConfig.newEntity( 'Hardware::Dpll::ResetConfigDir', pllModel )

      thisPllType = allPllConfig.entity[ pllModel ]
      for pllName in resetConfig.entity[ pllModel ].pll:
         if pllName not in thisPllType.pllConfig:
            del resetConfig.pll[ pllName ]
      for pllName in thisPllType.pllConfig:
         if pllName not in resetConfig.entity[ pllModel ].pll:
            resetConfig.entity[ pllModel ].newPll( pllName )

def resetHardwarePllGuard( mode, token ):
   pllName = token
   doUpdatePlls()
   for pllModel in allPllConfig:
      pllType = allPllConfig.entity[ pllModel ]
      if pllName in pllType.pllConfig:
         pllConfig = pllType.pllConfig[ pllName ]
         if not pllConfig.hwResetGpoName:
            return CliParser.guardNotThisPlatform
         else:
            return None
   return None

class ResetHardwarePllClocknameCmd( CliCommand.CliCommandClass ):
   syntax = 'reset hardware pll CLOCK_NAME'
   data = {
      'reset' : CliToken.Reset.resetMatcher,
      'hardware' : CliToken.Hardware.hardwareMatcherForReset,
      'pll' : 'Reset a PLL',
      'CLOCK_NAME' : CliCommand.Node(
         matcher=CliMatcher.DynamicNameMatcher(
            lambda mode: clockNames(),
            helpdesc='PLL Name', pattern='[A-Za-z0-9]+/[0-9]+' ),
         guard=resetHardwarePllGuard )
   }

   @staticmethod
   def handler( mode, args ):
      clockName = args[ 'CLOCK_NAME' ]
      doUpdatePlls()
      for pllModel in resetConfig:
         thisPllType = resetConfig.entity[ pllModel ]
         if clockName in thisPllType.pll:
            thisPllType.pll[ clockName ].resetRequests += 1
            return
      mode.addError( 'PLL not found: %s' % clockName )

BasicCli.EnableMode.addCommandClass( ResetHardwarePllClocknameCmd )

def _showTechCmds():
   return [ 'show hardware pll', ]

TechSupportCli.registerShowTechSupportCmdCallback( '2018-03-26 12:18:56',
                                                   _showTechCmds )

def Plugin( entityManager ):
   global allPllConfig, allPllStatus, resetConfig
   allPllConfig = LazyMount.mount( entityManager,
      'hardware/pllClockMux/config',
      'Tac::Dir',
      'ri' )
   allPllStatus = LazyMount.mount( entityManager,
      'hardware/pllClockMux/status',
      'Tac::Dir',
      'ri' )
   resetConfig = LazyMount.mount( entityManager,
      'hardware/pllClockMux/resetConfig',
      'Tac::Dir',
      'w' )
   # Note: I'm relying on FruCli initialising itself, so calling prepareSlotRule
   # here is in fact redundant/harmful
   # FruCli.prepareSlotRule( entityManager )

