# Copyright (c) 2017 Arista Networks, Inc.  All rights reserved.
# Arista Networks, Inc. Confidential and Proprietary.

from __future__ import absolute_import, division, print_function

import itertools
import AclLib
import AclCliLib
import BasicCli
import Cell
import CliCommand
import CliMatcher
import CliExtensions
import CliParser
import CliPlugin.AclCli as AclCli
import CliPlugin.AclCliModel as AclCliModel
import CliPlugin.VlanIntfCli as VlanIntfCli
import CliPlugin.IntfCli as IntfCli
import CliPlugin.EthIntfCli as EthIntfCli
import CliPlugin.LagCli as LagCli
import CliPlugin.LagIntfCli as LagIntfCli
import CliToken.Ip
import CliToken.Mac
import CliToken.Ipv6
import CliToken.Hardware
import ConfigMount
import LazyMount
import ShowCommand
import Tac

AclFeature = Tac.Type( 'Acl::AclFeature' )

cliConfig = None
paramConfig = None
config = None
intfConfigCli = None
intfConfigSecureMonitor = None
hwConfig = None
status = None
statusDp = None

def isAclHwStatusOk( mode, portName, newLagName, oldLagName ):
   if cliConfig is not None and not mode.session_.inConfigSession():
      # no need to wait unless there is ACL configured
      wait = False
      # which interface's ACL was in effect
      oldIntfName = oldLagName if oldLagName else portName
      for v in itertools.chain( intfConfigCli.config.itervalues(),
                                intfConfigSecureMonitor.config.itervalues() ):
         for d in v.intf.itervalues():
            if d.intf.get( newLagName ) != d.intf.get( oldIntfName ):
               # new ACL is different from old ACL
               wait = True
               break
      if wait:
         err = tryWaitForHwStatus( mode )
         if err == 'Operation timed out':
            return ( False, err )
         elif err == 'Keyboard Interrupt':
            return ( False, err )
         elif err:
            return ( False, 'Insufficent hardware resources to program ACL' )
   # We allow the configuration, but warn if the physical interface does not
   # have the same ACL configuration as the new LAG.
   if newLagName:
      # added to a new Lag
      for c in itertools.chain( intfConfigCli.config.itervalues(),
                                intfConfigSecureMonitor.config.itervalues() ):
         for d in c.intf.itervalues():
            portAcl = d.intf.get( portName )
            lagAcl = d.intf.get( newLagName )
            if portAcl and portAcl != lagAcl:
               mode.addWarning( 'ACL configuration of %s is overriden by %s '
                                'until it leaves the channel group.'
                                % ( portName, newLagName ) )
               break
   return ( True, '' )

LagCli.registerLagConfigValidCallback( isAclHwStatusOk )

def waitForAllHwStatus( timestamp ):
   for inst in statusDp.itervalues():
      if inst.waitForHwStatus and inst.dpAclUpdateTime < timestamp:
         return False
   return True

def tryWaitForHwStatus( mode ):
   if mode.session_.startupConfig():
      # no agent running, don't wait
      return ''
   timestamp = Tac.now( )
   hwConfig.dpAclUpdateRequestTime = timestamp
   try:
      Tac.waitFor( lambda: waitForAllHwStatus( timestamp ), 
                   warnAfter=0.5, sleep=True,
                   maxDelay=0.25, timeout=AclCliLib.hwWaitTimeout,
                   description='Acl configuration to be applied' )
      return AclCli._hwStatusErr() # pylint: disable-msg=protected-access
   except Tac.Timeout:
      return 'Operation timed out'
   except KeyboardInterrupt:
      return 'Keyboard Interrupt'

#---------------------------------------------------------------------
# hardware access-list update default-result permit
#---------------------------------------------------------------------
def dpPermitDuringAclUpdate( mode, token ):
   if status.dpPermitDuringAclUpdate:
      return None
   return CliParser.guardNotThisPlatform

def gotoHardwareAclUpdate( mode, args ):
   paramConfig.permitDuringAclUpdateConfig \
      = AclLib.ActionDuringAclUpdate.actionDuringAclUpdatePermit

def gotoNoHardwareAclUpdate( mode, args ):
   paramConfig.permitDuringAclUpdateConfig \
      = AclLib.ActionDuringAclUpdate.actionDuringAclUpdateDeny

def gotoDefaultHardwareAclUpdate( mode, args ):
   paramConfig.permitDuringAclUpdateConfig \
      = AclLib.ActionDuringAclUpdate.actionDuringAclUpdateUnset

#-----------------------------------------------------------------------------------
# hardware access-list ipv6 { security-acl | qos-policy } key-width narrow in
# no hardware access-list ipv6 { security-acl | qos-policy } key-width in
# 
# hardware access-list ipv4-ipv6 { mirroring-policy } key-width narrow in
# no hardware access-list ipv4-ipv6 { mirroring-policy } key-width in
#-----------------------------------------------------------------------------------
ingressSecurityAcIpv6KeyWidthModeCanPromptForAbortHook = CliExtensions.CliHook()
ingressQosAcIpv6KeyWidthModeCanPromptForAbortHook = CliExtensions.CliHook()
ingressMirroringAcKeyWidthModeCanPromptForAbortHook = CliExtensions.CliHook()

def dpIp6KeyWidthModeSupported( mode, token ):
   if status.dpIpv6SecurityAclKeyWidthModeSupported or \
         status.dpIpv6QosAclKeyWidthModeSupported:
      return None
   return CliParser.guardNotThisPlatform

def dpIp6SecurityAclKeyWidthModeSupported( mode, token ):
   if status.dpIpv6SecurityAclKeyWidthModeSupported:
      return None
   return CliParser.guardNotThisPlatform

def dpIp6QosAclKeyWidthModeSupported( mode, token ):
   if status.dpIpv6QosAclKeyWidthModeSupported:
      return None
   return CliParser.guardNotThisPlatform

def dpMirroringAclKeyWidthModeSupported( mode, token ):
   if status.dpMirrorAclKeyWidthModeSupported:
      return None
   return CliParser.guardNotThisPlatform

def gotoHardwareInIpv6SecurityAclKeyWidthMode( mode, 
      sharingMode=AclLib.TcamBankSharingMode.bankSharingModeNone ):
   if paramConfig.ingressIpv6SecurityAclKeyWidthMode == sharingMode:
      return
   if mode.session.commandConfirmation():
      for hook in \
            ingressSecurityAcIpv6KeyWidthModeCanPromptForAbortHook.extensions():
         if hook( mode ):
            return
   paramConfig.ingressIpv6SecurityAclKeyWidthMode = sharingMode

def gotoHardwareInIpv6QosAclKeyWidthMode( mode, 
      sharingMode=AclLib.TcamBankSharingMode.bankSharingModeNone ):
   if paramConfig.ingressIpv6QosAclKeyWidthMode == sharingMode:
      return
   if mode.session.commandConfirmation():
      for hook in \
            ingressQosAcIpv6KeyWidthModeCanPromptForAbortHook.extensions():
         if hook( mode ):
            return
   paramConfig.ingressIpv6QosAclKeyWidthMode = sharingMode

def gotoHardwareInMirroringAclKeyWidthMode( mode, 
      sharingMode=AclLib.TcamBankSharingMode.bankSharingModeNone ):
   if paramConfig.ingressMirrorAclKeyWidthMode == sharingMode:
      return
   if mode.session.commandConfirmation():
      for hook in \
            ingressMirroringAcKeyWidthModeCanPromptForAbortHook.extensions():
         if hook( mode ):
            return
   paramConfig.ingressMirrorAclKeyWidthMode = sharingMode

def gotoHardwareInMirroringAclKeyWidthModeNarrow( mode, args ):
   gotoHardwareInMirroringAclKeyWidthMode( mode,
         sharingMode=AclLib.TcamBankSharingMode.bankSharingModeNarrow )

def gotoHardwareInMirroringAclKeyWidthModeNone( mode, args ):
   gotoHardwareInMirroringAclKeyWidthMode( mode,
         sharingMode=AclLib.TcamBankSharingMode.bankSharingModeNone )

#---------------------------------------------------------------------------------
# hardware access-list resource sharing ipv4-ipv6 { security-acl | 
#                                                   mirroring-policy | 
#                                                   qos-policy } in
# no hardware access-list resource sharing ipv4-ipv6 { security-acl | 
#                                                      mirroring-policy | 
#                                                      qos-policy } in
#---------------------------------------------------------------------------------
ingressSecurityAclBankSharingCanPromptForAbortHook = CliExtensions.CliHook()
ingressMirroringAclBankSharingCanPromptForAbortHook = CliExtensions.CliHook()
ingressQosAclBankSharingCanPromptForAbortHook = CliExtensions.CliHook()

def dpIpv4Ip6BankSharingSupported( mode, token ):
   if status.dpIpv4Ip6SecurityAclBankSharingSupported or \
         status.dpIpv4Ip6MirroringAclBankSharingSupported or \
         status.dpIpv4Ip6QosAclBankSharingSupported:
      return None
   return CliParser.guardNotThisPlatform

def dpIpv4Ip6SecurityAclBankSharingSupported( mode, token ):
   if status.dpIpv4Ip6SecurityAclBankSharingSupported:
      return None
   return CliParser.guardNotThisPlatform

def dpIpv4Ip6MirroringAclBankSharingSupported( mode, token ):
   if status.dpIpv4Ip6MirroringAclBankSharingSupported:
      return None
   return CliParser.guardNotThisPlatform

def dpIpv4Ip6QosAclBankSharingSupported( mode, token ):
   if status.dpIpv4Ip6QosAclBankSharingSupported:
      return None
   return CliParser.guardNotThisPlatform

def gotoHardwareInIpv4Ipv6SecurityAclSharing( mode, support ):
   if paramConfig.ingressIpv4Ipv6PaclSharingSupported == support:
      return
   if mode.session.commandConfirmation():
      for hook in ingressSecurityAclBankSharingCanPromptForAbortHook.extensions():
         if hook( mode ):
            return
   paramConfig.ingressIpv4Ipv6PaclSharingSupported = support

def gotoHardwareInIpv4Ipv6MirroringAclSharing( mode, support ):
   if paramConfig.ingressIpv4Ipv6MirrorAclSharingSupported == support:
      return
   if mode.session.commandConfirmation():
      for hook in ingressMirroringAclBankSharingCanPromptForAbortHook.extensions():
         if hook( mode ):
            return
   paramConfig.ingressIpv4Ipv6MirrorAclSharingSupported = support 

def gotoHardwareInIpv4Ipv6QosAclSharing( mode, support ):
   if paramConfig.ingressIpv4Ipv6QosAclSharingSupported == support:
      return
   if mode.session.commandConfirmation():
      for hook in ingressQosAclBankSharingCanPromptForAbortHook.extensions():
         if hook( mode ):
            return
   paramConfig.ingressIpv4Ipv6QosAclSharingSupported = support

#---------------------------------------------------------------------
# hardware access-list resource sharing vlan [ ip | ipv6 ] [ in | out ]
#---------------------------------------------------------------------
def dpIngressRaclSharingSupported( mode, token ):
   if status.dpIngressRaclSharingSupported:
      return None
   return CliParser.guardNotThisPlatform

def dpEgressRaclSharingSupported( mode, token ):
   if status.dpEgressRaclSharingSupported:
      return None
   return CliParser.guardNotThisPlatform

def dpAclSharingSupported( mode, token ):
   if ( status.dpIngressRaclSharingSupported or
        status.dpEgressRaclSharingSupported ):
      return None
   return CliParser.guardNotThisPlatform

ingressRaclSharingCanPromptForAbortHook = CliExtensions.CliHook()
egressIpRaclSharingCanPromptForAbortHook = CliExtensions.CliHook()
egressIp6RaclSharingCanPromptForAbortHook = CliExtensions.CliHook()

def gotoHardwareAclSharing( mode, raclSharing=True ):
   if paramConfig.ingressRaclSharing == raclSharing:
      return
   if mode.session.commandConfirmation():
      for hook in ingressRaclSharingCanPromptForAbortHook.extensions():
         if hook( mode ):
            return

   paramConfig.ingressRaclSharing = raclSharing

def gotoHardwareInAclSharing( mode, args ):
   gotoHardwareAclSharing( mode )

def gotoNoHardwareInAclSharing( mode, args ):
   gotoHardwareAclSharing( mode, raclSharing=False )

def gotoHardwareOutIp6AclSharing( mode, raclSharing=True ):
   if paramConfig.egressIp6RaclSharing == raclSharing:
      return
   # No-op. Retaining the CLI to preserve downward compaitibility of config.
   mode.addWarning( 'This command has no effect. The shared mode is the only ' \
                       'available mode.' )

def gotoHardwareIpv6OutAclSharing( mode, args ):
   gotoHardwareOutIp6AclSharing( mode )

def gotoNoHardwareIpv6OutAclSharing( mode, args ):
   gotoHardwareOutIp6AclSharing( mode, raclSharing=False )

def gotoHardwareOutIpv4AclSharing( mode, raclSharing=True ):
   if paramConfig.egressIpRaclSharing == raclSharing:
      return
   if mode.session.commandConfirmation():
      for hook in egressIpRaclSharingCanPromptForAbortHook.extensions():
         if hook( mode ):
            return
   paramConfig.egressIpRaclSharing = raclSharing

def addURPFWarning( mode ):
   mode.addWarning( '!Unicast RPF will not work when egress acls '
                    'sharing mode is on' )

def gotoHardwareIpv4OutAclSharing( mode, args ):
   addURPFWarning( mode )
   gotoHardwareOutIpv4AclSharing( mode )

def gotoNoHardwareIpv4OutAclSharing( mode, args ):
   gotoHardwareOutIpv4AclSharing( mode, raclSharing=False )

#-----------------------------------------------------------------------------------
#hardware access-list ipv4 egress resource sharing routed-interfaces
#-----------------------------------------------------------------------------------
def dpEgressRoutedInterfaceAclSharingSupported( mode, token ):
   if status.dpEgressRoutedInterfaceAclSharingSupported:
      return None
   return CliParser.guardNotThisPlatform

egressRoutedInterfaceAclSharingCanPromptForAbortHook = CliExtensions.CliHook()

def _changeEgressRoutedInterfaceAclSharing( mode, egressRoutedInterfaceAclSharing ):
   if paramConfig.egressRoutedInterfaceAclSharing == egressRoutedInterfaceAclSharing:
      return

   if mode.session.commandConfirmation():
      for hook in egressRoutedInterfaceAclSharingCanPromptForAbortHook.extensions():
         if hook( mode ):
            return
   paramConfig.egressRoutedInterfaceAclSharing = egressRoutedInterfaceAclSharing

def changeEgressRoutedInterfaceAclSharing( mode, args ):
   addURPFWarning( mode )
   _changeEgressRoutedInterfaceAclSharing( mode,
                                           egressRoutedInterfaceAclSharing=True )

def changeNoEgressRoutedInterfaceAclSharing( mode, args ):
   _changeEgressRoutedInterfaceAclSharing( mode,
                                           egressRoutedInterfaceAclSharing=False )

#-----------------------------------------------------------------------------------
#[ no ]hardware access-list ipv6 implicit-permit icmpv6 ( all | neighbor-discovery )
#-----------------------------------------------------------------------------------
def allIcmpTypesInAclSupportedGuard( mode, token ):
   if status.dpConfigurableImplicitIcmp6Rules:
      return None
   return CliParser.guardNotThisPlatform

def ipv6AclImcp6NeighborDiscoveryTypes( mode, args ):
   permitIcmp6TypesConfig = ( AclLib.Icmp6RuleType.all if 'all' in args 
         else AclLib.Icmp6RuleType.neighborDiscovery )
   paramConfig.permitIcmp6TypesConfig = permitIcmp6TypesConfig

def noIpv6AclImcp6NeighborDiscoveryTypes( mode, args ):
   paramConfig.permitIcmp6TypesConfig = AclLib.Icmp6RuleType.all

#---------------------------------------------------------------------
# hardware access-list router-acl exclude mlag peer-link
#---------------------------------------------------------------------
def dpRouterAclExcludeMlagPeerLink( mode, token ):
   if status.dpRouterAclExcludeMlagPeerLink:
      return None
   else:
      return CliParser.guardNotThisPlatform

def excludeMlagPeerLink( mode, args ):
   paramConfig.excludeMlagPeerLink = True

def noExcludeMlagPeerLink( mode, args ):
   paramConfig.excludeMlagPeerLink = False

#----------------------------------------------------------------------------
# 'deep-inspection payload l2 skip <index value>' command, in 'config' mode
# 'deep-inspection payload l4 skip <index value>' command, in 'config' mode
#----------------------------------------------------------------------------
def deepInspMaxSkipL2Fn_( mode ):
   return ( 0, status.dpAclDeepInspMaxL2Skip )

def deepInspMaxSkipL4Fn_( mode ):
   return ( 0, status.dpAclDeepInspMaxL4Skip )

def setDeepInspL2Offset( mode, args ):
   paramConfig.deepInspSkipL2 = args[ 'SKIP' ]

def noSetDeepInspL2Offset( mode, args ):
   paramConfig.deepInspSkipL2 = 0

def setDeepInspL4Offset( mode, args ):
   paramConfig.deepInspSkipL4 = args[ 'SKIP' ]

def noSetDeepInspL4Offset( mode, args ):
   paramConfig.deepInspSkipL4 = 0

def showDeepInspL2L4Offset( mode, args ):
   skipvalue = AclCliModel.DeepSkipVal()
   skipvalue.l2SkipValue = paramConfig.deepInspSkipL2
   skipvalue.l4SkipValue = paramConfig.deepInspSkipL4
   return skipvalue

class ShowDeepInspectionSummary( ShowCommand.ShowCliCommandClass ):
   syntax = 'show deep-inspection summary'
   data = {
            'deep-inspection': CliCommand.Node(
               matcher=CliMatcher.KeywordMatcher( 'deep-inspection',
                  helpdesc='Show deep inspection parameters' ),
               guard=AclCli.dpAclDIParsingSupported ),
            'summary': 'Deep inspection summary for L2 and L4'
         }
   cliModel = AclCliModel.DeepSkipVal
   handler = showDeepInspL2L4Offset

BasicCli.addShowCommandClass( ShowDeepInspectionSummary )

def dpMacIngressSupportedGuard( mode, token ):
   if mode.intf.hardware() == 'tunnel' and \
                                 not status.dpTunnelIntfIngressMacAclSupported:
      return CliParser.guardNotThisPlatform
   if status.dpMacAclSupported:
      return None
   return CliParser.guardNotThisPlatform

def dpMacEgressSupportedGuard( mode, token ):
   if mode.intf.hardware() == 'tunnel' and \
                                 not status.dpTunnelIntfEgressMacAclSupported:
      return CliParser.guardNotThisPlatform
   if status.dpMacEgressAclSupported: 
      return None
   return CliParser.guardNotThisPlatform

def dpIpEgressSupportedGuard( mode, token ):
   if isinstance( mode.intf, VlanIntfCli.VlanIntf ):
      if status.dpRouterEgressAclSupported:
         return None
   elif mode.intf.hardware() == 'tunnel' and \
                                 not status.dpTunnelIntfEgressIpAclSupported:
      return CliParser.guardNotThisPlatform
   elif status.dpIpEgressAclSupported: 
      return None
   return CliParser.guardNotThisPlatform

#----------------------------------------------------------------
# ip/ipv6/mac access-group (Eth, Port-Channel, SVI, SubInterface)
#----------------------------------------------------------------
def dpIpIngressSupportedGuard( mode, token ):
   if isinstance( mode.intf, VlanIntfCli.VlanIntf ) and \
                                 not status.dpRouterAclSupported:
      return CliParser.guardNotThisPlatform
   elif mode.intf.hardware() == 'tunnel' and \
                                 not status.dpTunnelIntfIngressIpAclSupported:
      return CliParser.guardNotThisPlatform
   elif mode.intf.isSubIntf() and not status.dpSubIntfAclSupported:
      return CliParser.guardNotThisPlatform
   if status.dpAclSupported:
      return None
   return CliParser.guardNotThisPlatform

def dpIp6IngressSupportedGuard( mode, token ):
   if isinstance( mode.intf, VlanIntfCli.VlanIntf ) and \
                                 not status.dpRouterAclSupported:
      return CliParser.guardNotThisPlatform
   elif mode.intf.hardware() == 'tunnel' and \
                                 not status.dpTunnelIntfIngressIpv6AclSupported:
      return CliParser.guardNotThisPlatform
   elif mode.intf.isSubIntf() and not status.dpSubIntfAclSupported:
      return CliParser.guardNotThisPlatform
   if status.dpIp6AclSupported: 
      return None
   return CliParser.guardNotThisPlatform

def dpMacAclSupportedGuard( mode, token ):
   if isinstance( mode.intf, VlanIntfCli.VlanIntf ) and \
                                 not status.dpRouterAclSupported:
      return CliParser.guardNotThisPlatform
   elif mode.intf.hardware() == 'tunnel' and \
             not ( status.dpTunnelIntfIngressMacAclSupported and \
             status.dpTunnelIntfEgressMacAclSupported ):
      return CliParser.guardNotThisPlatform
   elif mode.intf.isSubIntf() and not status.dpMacSubIntfAclSupported:
      return CliParser.guardNotThisPlatform
   if status.dpMacAclSupported or status.dpMacEgressAclSupported:
      return None
   return CliParser.guardNotThisPlatform

def dpIngressOrEgressAclSupportedGuard( mode, token ):
   if not dpIpIngressSupportedGuard( mode, token ):
      return None
   elif not dpIpEgressSupportedGuard( mode, token ):
      return None
   return CliParser.guardNotThisPlatform

def dpIp6IngressOrEgressAclSupportedGuard( mode, token ):
   if not dpIp6IngressSupportedGuard( mode, token ):
      return None
   elif not dpIp6EgressSupportedGuard( mode, token ):
      return None
   return CliParser.guardNotThisPlatform

def dpIp6EgressSupportedGuard( mode, token ):
   if isinstance( mode.intf, VlanIntfCli.VlanIntf ):
      if status.dpIp6RouterEgressAclSupported:
         return None
   elif mode.intf.hardware() == 'tunnel' and \
        not status.dpTunnelIntfEgressIpv6AclSupported:
      return CliParser.guardNotThisPlatform
   elif status.dpIp6EgressAclSupported:
      return None
   return CliParser.guardNotThisPlatform

def checkSecureMonitor( mode, intf, aclType, direction ):
   # Check that one cannot remove ACL from interface with different secure-monitor
   if mode.session.secureMonitor():
      # We could be in config session so check the sysdb object
      checkIntfConfig = intfConfigCli.sysdbObj()
   else:
      checkIntfConfig = intfConfigSecureMonitor

   if intf in checkIntfConfig.config[ aclType ].intf[ direction ].intf:
      mode.addError( str( CliParser.guardNotPermitted ) )
      raise CliParser.AlreadyHandledError()
   return

def _configurePortAcl( mode, args ):
   ''' This supports the initial implementation of the CLI Config Sessions
   feature (config-replace only, no config sessions). _configurePortAcl is
   factored into two parts. The first part updates the config, but does not
   wait for agent responses. Agents will not see the config changes if we are
   doing a config-replace. The second part is executed after the session is
   committed.

   This code must be refactored when the full CLI Config Sessions feature is
   enabled. '''

   aclName = args[ 'ACL_NAME' ]
   aclType = None
   if 'mac' in args:
      aclType = 'mac'
   elif 'ip' in args:
      aclType = 'ip'
   else:
      assert 'ipv6' in args
      aclType = 'ipv6'
   direction = 'in' if 'in' in args else 'out'
   assert direction in args

   # make sure we cannot override existing ACL with mismatching secure-monitor
   checkSecureMonitor( mode, mode.intf.name, aclType, direction )

   aclConfig = cliConfig.config[ aclType ].acl.get( aclName )
   if aclConfig and mode.session.secureMonitor() != aclConfig.secureMonitor:
      errMsg = str( CliParser.guardNotPermitted )
      mode.addError( AclCliLib.intfAclConfigError( aclName, aclType,
                                                   mode.intf.name,
                                                   errMsg ) )
      return

   # Get the hardware status before removing the Acl.
   origHwStatusOK = AclCli._hwStatusOK() # pylint: disable-msg=protected-access

   # Since attach() api directly writes to Sysdb (without using a scratchpad),
   # get an instance of the api sm and call attach() on it.
   aclApi = AclCli.aclApiForAcl( mode, aclName, aclType, apiType='attach' )
   aclApi.attach( mode.intf.name, direction )
   err = aclApi.apiState
   if err not in ( 'succeeded', 'waitingForHw', 'errHw' ):
      errMsg = ','.join( aclApi.errString.values() )
      mode.addError( AclCliLib.intfAclConfigError( aclName, aclType,
                                                   mode.intf.name,
                                                   errMsg ) )
      return

   if not aclApi.dpAclOk():
      errMsg = ','.join( aclApi.errString.values() )
      AclCli.printUnsupportedRulesWarning( mode, aclName, aclType, errMsg )

   if not AclCli._hwStatusPresent(): # pylint: disable-msg=protected-access
      mode.addWarning( AclCliLib.noHwWarningMsg )
   if mode.session.inConfigSession():
      AclCli.registerAclSessionOnCommitHandler( mode, aclApi, False )
   else:
      if origHwStatusOK:
         # Throw error and Revert only if the hardware status before
         # removing the Acl was in a sane state.
         _configurePortAclCheckHwStatus( mode, aclName, aclType, aclApi )

def _configurePortAclCheckHwStatus( mode, aclName, aclType, aclApi ):
   ''' The second part waits for agents/constrainers to respond,
   does reverts and generates error messages if needed. '''

   ret, errStr = AclCli.tryWaitForApiStatus( aclApi )
   if ret != 'succeeded':
      mode.addError( AclCliLib.intfAclConfigError( aclName, aclType,
                                                   mode.intf.name,
                                                   ','.join( errStr.values() ) ) )
      aclApi.revert()
      AclCli.tryWaitForApiStatus( aclApi )
   else:
      AclCli.maybePrintUnsupportedRulesWarning( mode, aclName, aclType, statusDp,
                                         feature=AclFeature.featureUnknown )

def _configureNoPortAcl( mode, args ):
   ''' This supports the initial implementation of the CLI Config Sessions
   feature (config-replace only, no config sessions). _configureNoPortAcl
   is factored into two parts. The first part updates the config, but does
   not wait for agent responses. Agents will not see the config changes if
   we are doing a config-replace. The second part is executed after the
   session is committed.
      
   This code must be refactored when the full CLI Config Sessions feature is
   enabled. '''
   aclName = args.get( 'ACL_NAME', '' )
   aclType = None
   if 'mac' in args:
      aclType = 'mac'
   elif 'ip' in args:
      aclType = 'ip'
   else:
      assert 'ipv6' in args
      aclType = 'ipv6'
   direction = 'in' if 'in' in args else 'out'
   assert direction in args

   checkSecureMonitor( mode, mode.intf.name, aclType, direction )

   # Get the hardware status before removing the Acl.
   origHwStatusOK = AclCli._hwStatusOK() # pylint: disable-msg=protected-access

   # Since detachAcl() api directly writes to Sysdb (without using a scratchpad),
   # get an instance of the api sm and call detachAcl() on it.
   aclApi = AclCli.aclApiForAcl( mode, aclName, aclType, apiType='detach' )
   origAclName = aclApi.attachedAcl( mode.intf.name, direction )
   ret = aclApi.detach( mode.intf.name, direction )

   if ret not in ( 'succeeded', 'waitingForHw', 'errHw' ):
      if ret == 'errNotConfigured':
         AclCli.printNotConfiguredError( mode, aclType, aclName, mode.intf.name )
      else:
         # should there be other errors?
         errMsg = ','.join( aclApi.errString.values() )
         mode.addError( AclCliLib.intfAclConfigError( aclName, aclType,
                                                      mode.intf.name,
                                                      errMsg ) )
      return

   if not AclCli._hwStatusPresent(): # pylint: disable-msg=protected-access
      mode.addWarning( AclCliLib.noHwWarningMsg )
   if mode.session.inConfigSession():
      AclCli.registerAclSessionOnCommitHandler( mode, aclApi, False )
   else:
      if origHwStatusOK:
         # Throw error and Revert only if the hardware status before
         # removing the Acl was in a sane state.
         _configureNoPortAclCheckHwStatus(
            mode, aclType, aclName, aclApi, origAclName )

def _configureNoPortAclCheckHwStatus( mode, aclType, aclName, aclApi, 
                                      origAclName ):
   ''' The second part waits for agents/constrainers to respond,
   does reverts and generates error messages if needed. '''

   ret, errStr = AclCli.tryWaitForApiStatus( aclApi )
   if ret != 'succeeded':
      mode.addError(
         AclCliLib.intfAclDeleteError( origAclName, aclType,
                                       mode.intf.name,
                                       ','.join( errStr.values() ) ) )
      aclApi.revert()
      AclCli.tryWaitForApiStatus( aclApi )

class DpAclEthIntfModelet( CliParser.Modelet ):
   modeletParseTree = CliParser.ModeletParseTree()
   def __init__( self, mode ):
      CliParser.Modelet.__init__( self )

   @staticmethod
   def shouldAddModeletRule( mode ):
      return ( ( isinstance( mode.intf, EthIntfCli.EthPhyIntf ) and
                 mode.intf.name.startswith( 'Et' ) ) )

class DpAclEthLagIntfModelet( CliParser.Modelet ):
   modeletParseTree = CliParser.ModeletParseTree()
   def __init__( self, mode ):
      CliParser.Modelet.__init__( self )

   @staticmethod
   def shouldAddModeletRule( mode ):
      return isinstance( mode.intf, LagIntfCli.EthLagIntf )

class DpAclVlanIntfModelet( CliParser.Modelet ):
   modeletParseTree = CliParser.ModeletParseTree()
   def __init__( self, mode ):
      CliParser.Modelet.__init__( self )

   @staticmethod
   def shouldAddModeletRule( mode ):
      return isinstance( mode.intf, VlanIntfCli.VlanIntf )

class DpAclTunnelIntfModelet( CliParser.Modelet ):
   modeletParseTree = CliParser.ModeletParseTree()
   def __init__( self, mode ):
      CliParser.Modelet.__init__( self )

   @staticmethod
   def shouldAddModeletRule( mode ):
      return mode.intf.hardware() == 'tunnel'

class DpAclSubIntfModelet( CliParser.Modelet ):
   modeletParseTree = CliParser.ModeletParseTree()
   def __init__( self, mode ):
      CliParser.Modelet.__init__( self )

   @staticmethod
   def shouldAddModeletRule( mode ):
      return mode.intf.isSubIntf()

tokenDpAccessGroup = CliCommand.Node(
      matcher=CliMatcher.KeywordMatcher( 'access-group',
         helpdesc='Configure access control list' ),
      guard=dpIngressOrEgressAclSupportedGuard )
tokenDpIp6AccessGroup = CliCommand.Node(
      matcher=CliMatcher.KeywordMatcher( 'access-group',
         helpdesc='Configure IPv6 access control list' ),
      guard=dpIp6IngressOrEgressAclSupportedGuard )

for modelet in ( DpAclEthIntfModelet, DpAclEthLagIntfModelet, 
                 DpAclSubIntfModelet, DpAclVlanIntfModelet,
                 DpAclTunnelIntfModelet ):
   accessGroupMatcherIp = ( AclCli.accessGroupKwMatcher 
         if modelet == DpAclEthIntfModelet else tokenDpAccessGroup )
   class IpAccessGroup( CliCommand.CliCommandClass ):
      syntax = 'ip access-group ACL_NAME ( in | out )'
      noOrDefaultSyntax = 'ip access-group [ ACL_NAME ] ( in | out )'
      data = {
               'ip': CliToken.Ip.ipMatcherForConfigIf,
               'access-group': accessGroupMatcherIp,
               'ACL_NAME': AclCli.ipAclNameMatcher,
               'in': CliCommand.Node(
                  matcher=CliMatcher.KeywordMatcher( 'in',
                     helpdesc='Inbound packets' ),
                  guard=dpIpIngressSupportedGuard ),
               'out': CliCommand.Node(
                  matcher=CliMatcher.KeywordMatcher( 'out',
                     helpdesc='Outbound packets' ),
                  guard=dpIpEgressSupportedGuard ),
            }
      handler = _configurePortAcl
      noOrDefaultHandler = _configureNoPortAcl

   modelet.addCommandClass( IpAccessGroup )

   accessGroupMatcherIp6 = ( AclCli.accessGroupKwMatcher 
         if modelet == DpAclEthIntfModelet else tokenDpIp6AccessGroup )

   class Ip6AccessGroup( CliCommand.CliCommandClass ):
      syntax = 'ipv6 access-group ACL_NAME ( in | out )'
      noOrDefaultSyntax = 'ipv6 access-group [ ACL_NAME ] ( in | out )'
      data = {
               'ipv6': CliToken.Ipv6.ipv6MatcherForConfigIf,
               'access-group': accessGroupMatcherIp6,
               'ACL_NAME': AclCli.ip6AclNameMatcher,
               'in': CliCommand.Node(
                  matcher=CliMatcher.KeywordMatcher( 'in',
                     helpdesc='Inbound packets' ),
                  guard=dpIp6IngressSupportedGuard ),
               'out': CliCommand.Node(
                  matcher=CliMatcher.KeywordMatcher( 'out',
                     helpdesc='Outbound IPv6 packets' ),
                  guard=dpIp6EgressSupportedGuard ),
            }
      handler = _configurePortAcl
      noOrDefaultHandler = _configureNoPortAcl

   modelet.addCommandClass( Ip6AccessGroup )

# L2-only commands
for modelet in ( DpAclEthIntfModelet, DpAclEthLagIntfModelet, DpAclSubIntfModelet ):
   # mac access-group ... in/out, currently no support for vlan interfaces
   class MacAccessGroup( CliCommand.CliCommandClass ):
      syntax = 'mac access-group ACL_NAME ( in | out )'
      noOrDefaultSyntax = 'mac access-group [ ACL_NAME ] ( in | out )'
      data = {
               'mac': CliToken.Mac.macMatcherForConfigIf,
               'access-group': CliCommand.Node(
                  matcher=CliMatcher.KeywordMatcher( 'access-group',
                     helpdesc='Configure MAC access control list' ),
                  guard=dpMacAclSupportedGuard ),
               'ACL_NAME': AclCli.macAclNameMatcher,
               'in': CliCommand.Node(
                  matcher=CliMatcher.KeywordMatcher( 'in',
                     helpdesc='Inbound packets' ),
                  guard=dpMacIngressSupportedGuard ),
               'out': CliCommand.Node(
                  matcher=CliMatcher.KeywordMatcher( 'out',
                     helpdesc='Outbound packets' ),
                  guard=dpMacEgressSupportedGuard ),
            }
      handler = _configurePortAcl
      noOrDefaultHandler = _configureNoPortAcl

   modelet.addCommandClass( MacAccessGroup )

for i in ( DpAclEthIntfModelet, DpAclEthLagIntfModelet, DpAclVlanIntfModelet, 
           DpAclSubIntfModelet, DpAclTunnelIntfModelet ):
   IntfCli.IntfConfigMode.addModelet( i )

# Provide the lagToPortMap function for AclCli due to dependency
# as Acl cannot depend on Lag.
__lagConfig__ = None

def _lagToPortMap():
   lagToPort = {}
   for intf, lagInfo in __lagConfig__.phyIntf.iteritems():
      if lagInfo.lag:
         if lagToPort.get( lagInfo.lag.intfId ):
            lagToPort[ lagInfo.lag.intfId ].append( intf )
         else:
            lagToPort[ lagInfo.lag.intfId ] = [ intf ]
   return lagToPort

def Plugin( entityManager ):
   global cliConfig, paramConfig, intfConfigCli, intfConfigSecureMonitor
   global hwConfig, status, statusDp
   global __lagConfig__
   cliConfig = ConfigMount.mount( entityManager, 'acl/config/cli',
                                  'Acl::Input::Config', 'w' )
   paramConfig = ConfigMount.mount( entityManager, 'acl/paramconfig',
                             'Acl::ParamConfig', 'w' )
   intfConfigCli = ConfigMount.mount( entityManager, 'acl/intf/config/cli',
                                      'Acl::IntfConfig', 'w' )
   intfConfigSecureMonitor = LazyMount.mount( entityManager,
                                              'acl/intf/config/input/secure-monitor',
                                              'Acl::IntfConfig', 'w' )
   hwConfig = LazyMount.mount( entityManager, 'acl/hwconfig/cli',
                               'Acl::HwConfig', 'w' )
   status = LazyMount.mount( entityManager,
                             'acl/status/all', 'Acl::Status', 'r' )
   statusDp = LazyMount.mount( entityManager, 
                               'cell/%d/acl/status/dp' % Cell.cellId(),
                               'Tac::Dir', 'ri' )
   __lagConfig__ = LazyMount.mount( entityManager, 'lag/input/config/cli',
                                    'Lag::Input::Config', 'r' )
   AclCli.__lagToPortMapFunc__ = _lagToPortMap
