#!/usr/bin/env python
# Copyright (c) 2013 Arista Networks, Inc.  All rights reserved.
# Arista Networks, Inc. Confidential and Proprietary.

import Tac
import Arnet
from CliMode.Intf import IntfMode
from CliModel import Enum
from CliModel import UncheckedModel
from CliModel import Dict
from CliModel import Float
from CliModel import Int
from CliModel import Bool
from CliModel import Str
from CliModel import Model
from ArnetModel import MacAddress
from collections import OrderedDict
from IntfModels import Interface
from Ethernet import convertMacAddrToDisplay
from TableOutput import createTable, Format
import Toggles.Dot1xToggleLib as Dot1xToggle
from Dot1xLib import portControlEnum2Str, portStatusEnum2Str, hostModeEnum2Str, \
      authStageDict, cliStageShorten

#--------------------------------------------------------------------------------
# EAPI Models
#--------------------------------------------------------------------------------   
class Dot1xInterfaceInformation( UncheckedModel ):
   portControl = Enum( values=( "controlled", "forceAuth", 
                                "forceUnauth" ),
                            help="Controlled -- enable 802.1X for the interface. "
                            "ForceAuth -- disable 802.1X and put the "
                            "interface into authorized state. "
                            "ForceUnauth -- disable 802.1X and put the "
                            "interface into unauthorized state. " )
   hostMode = Enum( values=( "singleHost", "multiHost", "authAllHost" ),
                        help="singleHost -- Allow only one supplicant on an "
                        "authorized port. "
                        "multiHost -- Allow multiple clients on an "
                        "authorized port. " 
                        "authAllHost -- Allow multiple dot1x-authenticated clients "
                        "on an authorized port. " )
   quietPeriod = Float( help="Waiting period in seconds "
                             "after a failed authentication. " )
   txPeriod = Float( help="Waiting period in seconds "
                          "before retrying a message to supplicant. " )
   maxReauthReq = Int( help="Maximum number of reauthentication attempts" )
   reauthTimeoutIgnore = Bool( help="Retain current port auth status on "
                               "reauth server or supplicant timeouts "\
                               "or new reauth requests" ) 
   authFailVlan = Int( help="Authentication failure VLAN" )
   unauthorizedAccessVlanEgress = Bool( help="Give unauthorized access port egress "
                                             "membership of access VLAN " ) 
   unauthorizedNativeVlanEgress = Bool( help="Give unauthorized trunk port egress "
                                             "membership of native VLAN " ) 
   eapolDisabled = Bool( help="EAPOL is disabled" )
   mbaEnabled = Bool( help="MAC-based authentication is enabled" )
   mbaTimeout = \
         Float( help="Timeout for starting MAC based authentication in seconds" )
   mbaFallback = Bool( help="MBA fallback for EAPOL authentication failure" )
   mbaHostMode = Bool( help="MAC based authentication host mode is enabled" )
   
   def render( self ):
      # pylint: disable-msg=no-member
      print '--------------------------------------------'
      print 'Port control: %s' % portControlEnum2Str[ self.portControl ]
      print 'EAPOL: %s' % ( 'disabled' if self.eapolDisabled else 'enabled' )
      print 'Host mode: %s' % hostModeEnum2Str[ self.hostMode ]
      print 'MAC-based authentication: %s' % ( 'enabled' if self.mbaEnabled
                                                else 'disabled' )
      mbaHostModeStr = 'MAC-based authentication host mode: '
      if self.mbaHostMode:
         mbaHostModeStr += hostModeEnum2Str[ self.hostMode ]
      else:
         mbaHostModeStr += 'Unconfigured'
      print mbaHostModeStr
      print 'Quiet period: %d seconds' % self.quietPeriod
      print 'TX period: %d seconds' % self.txPeriod
      print 'Maximum reauth requests: %d' % self.maxReauthReq
      print 'Ignore reauth timeout: %s' % ( 'Yes' if self.reauthTimeoutIgnore 
                                              else 'No' )
      print 'Auth failure VLAN: %s' % ( 'Unconfigured' if self.authFailVlan == 0 
                                          else self.authFailVlan )
      print 'Unauthorized access VLAN egress: %s' % \
            ( 'Yes' if self.unauthorizedAccessVlanEgress else 'No' )
      print 'Unauthorized native VLAN egress: %s' % \
            ( 'Yes' if self.unauthorizedNativeVlanEgress else 'No' )
      if Dot1xToggle.toggleDot1xEAPOLPriorityEnabled():
         fallbackStr = 'EAPOL authentication failure fallback: '
         if self.mbaFallback:
            fallbackStr += 'MBA, timeout %d seconds' % self.mbaTimeout
         else:
            fallbackStr += 'Unconfigured'
         print fallbackStr

      # pylint: disable-msg=W0105
      '''
      we don't support guest vlan and restricted vlan now
      print 'AuthFailMaxAttempts\t: %d' % self.authFailMaxAttempt
      if self.guestVlan:
         print 'GuestVlan\t\t: %d' % self.guestVlan
      if self.authFailVlan:
         print 'AuthFailVlan\t\t: %d' % self.authFailVlan
      '''
class Dot1xInterfaceRangeInformation( Model ):
   interfaces = Dict( help="A mapping between interfaces and 802.1X "
                           "interface information", keyType=Interface,
                           valueType=Dot1xInterfaceInformation )

   def render( self ):
      if not self.interfaces:
         return

      for key in Arnet.sortIntf( self.interfaces ):
         print 'Dot1X Information for %s' % key
         self.interfaces[ key ].render()
         print

class Dot1xInterfaceDetails( Dot1xInterfaceInformation ):
   portAuthorizationState = Enum( values=( "blocked", "authorized", 
                                           "guestVlan", "restrictedVlan" ),
                                    help="Blocked -- port is unauthorized. "
                                    "Authorized -- port is authorized. "
                                    "GuestVlan -- port is authorized through "
                                    "guest VLAN"
                                    "RestrictedVlan -- port is authorized through "
                                    "restricted VLAN",
                                    optional=True )
   supplicantMacs = Dict( help="Mapping of supplicant MAC address to"
                              " its reauthentication period in seconds",
                              keyType=MacAddress, valueType=int )

   def render( self ):
      Dot1xInterfaceInformation.render( self )
      if self.portAuthorizationState:
         print '\nDot1X Authenticator Client\n'
         print 'Port status: %s' % \
                  portStatusEnum2Str[ self.portAuthorizationState ]
         if self.supplicantMacs:
            print 'Supplicant MAC\tReauth Period (in seconds)'
            print '--------------\t--------------------------'
            for mac, reauthPeriod in self.supplicantMacs.iteritems():
               print '%s\t%d' % \
                  ( convertMacAddrToDisplay( mac ), reauthPeriod )
                  
class Dot1xInterfaceRangeDetails( Model ):
   interfaces = Dict( help="A mapping between interfaces and 802.1X "
                           "interface detailed information", keyType=Interface,
                           valueType=Dot1xInterfaceDetails )

   def render( self ):
      if not self.interfaces:
         return

      for key in Arnet.sortIntf( self.interfaces ):
         print 'Dot1X Information for %s' % key
         self.interfaces[ key ].render()
         print
                 
class Dot1xInterfaceStats( UncheckedModel ):
   rxStart = Int( help="EAPOL Start frames received" )
   rxLogoff = Int( help="EAPOL Logoff frames received" )
   rxRespId = Int( help="EAPOL Resp/Id frames received" )
   rxResp = Int( help="EAP Response frames received" )
   rxInvalid = Int( help="Invalid EAPOL frames received" )
   rxTotal = Int( "EAPOL frames received" )
   txReqId = Int( help="EAP Initial Request frames transmitted" )
   txReq = Int( help="EAP Request frames transmitted" )
   txTotal = Int( help="EAPOL frames transmitted" )
   rxVersion = Int( help="Last EAPOL frame version" )
   lastRxSrcMac = MacAddress( help="Last EAPOL frame source" )

   def render( self ):
      # pylint: disable-msg=no-member
      print '-------------------------------------------------'
      print 'RX start = %d\t RX logoff = %d\t RX response ID = %d' % \
                  ( self.rxStart, self.rxLogoff, self.rxRespId )
      print 'RX response = %d\t RX invalid = %d\t RX total = %d' % \
                  ( self.rxResp, self.rxInvalid, self.rxTotal )
      print 'TX request ID = %d\t TX request = %d\t TX total = %d' % \
                  ( self.txReqId, self.txReq, self.txTotal )
      print 'RX version = %d\t Last RX src MAC = %s' % \
                  ( self.rxVersion, 
                        convertMacAddrToDisplay( self.lastRxSrcMac.stringValue ) )
   
class Dot1xInterfaceRangeStats( Model ):
   interfaces = Dict( help="A mapping between interfaces and 802.1X "
                           "interface statistics", keyType=Interface,
                           valueType=Dot1xInterfaceStats )

   def render( self ):
      if not self.interfaces:
         return

      for key in Arnet.sortIntf( self.interfaces ):
         print 'Dot1X Authenticator Port Statistics for %s' % key
         self.interfaces[ key ].render()
         print

class Dot1xAllInformation( UncheckedModel ):
   systemAuthControl = Bool( help="System Authentication Control is enabled" )
   lldpBypass = Bool( help="Dot1X LLDP Bypass is enabled" )
   dynAuth = Bool( help="Dot1X Dynamic Authorization is enabled" )
   version = Int( help="Dot1X protocol version" )
   interfaces = Dict( keyType=Interface, valueType=Dot1xInterfaceInformation,
                          help="A mapping of interface to its Dot1X information" )

   def render( self ):
      print 'System authentication control: %s' % ( 'Enabled' \
                        if self.systemAuthControl else 'Disabled' )
      print 'Dot1X LLDP bypass: %s' % ( 'Enabled' if self.lldpBypass \
                                                    else 'Disabled' )
      print 'Dot1X dynamic authorization: %s' % ( 'Enabled' if self.dynAuth \
                                                              else 'Disabled' )
      print 'Dot1X protocol version: %d\n' % self.version
      for key in Arnet.sortIntf( self.interfaces ):
         print 'Dot1X Information for %s' % key
         self.interfaces[ key ].render()
         print

class Dot1xAllDetails( UncheckedModel ):
   systemAuthControl = Bool( help="System Authentication Control is enabled" )
   lldpBypass = Bool( help="Dot1X LLDP Bypass is enabled" )
   dynAuth = Bool( help="Dot1X Dynamic Authorization is enabled" )
   version = Int( help="Dot1X protocol version" )
   interfaces = Dict( keyType=Interface, valueType=Dot1xInterfaceDetails,
                          help="A mapping of interface to its "
                          "Dot1X detailed information" )

   def render( self ):
      print 'System authentication control: %s' % ( 'Enabled' \
                        if self.systemAuthControl else 'Disabled' )
      print 'Dot1X LLDP bypass: %s' % ( 'Enabled' if self.lldpBypass \
                                                    else 'Disabled' )
      print 'Dot1X dynamic authorization: %s' % ( 'Enabled' if self.dynAuth \
                                                              else 'Disabled' )
      print 'Dot1X protocol version: %d\n' % self.version
      for key in Arnet.sortIntf( self.interfaces ):
         print 'Dot1X Information for %s' % key
         self.interfaces[ key ].render()
         print

class Dot1xAllSummary( Dot1xAllDetails ):
   def render( self ):
      print '%-24s%-24s%s' % ( 'Interface', 'Client', 'Status' )
      print '-------------------------------------------------------------'
      for key in Arnet.sortIntf( self.interfaces ):
         details = self.interfaces[ key ]
         if details.portAuthorizationState:
            for mac in details.supplicantMacs:
               client = convertMacAddrToDisplay( mac ) if mac else 'None'
               print '%-24s%-24s%s' % ( key, client,
                       portStatusEnum2Str[ details.portAuthorizationState ] )

class Dot1xSupplicant( Model ):
   identity = Str( help="EAP identity in use on this interface" )
   eapMethod = Enum( values=( "unset", "fast" ),
                     help="unset -- EAP method is currently unset."
                     "fast -- EAP method in use is EAP FAST." )
   status = Enum( values=( "unused", "down", "connecting", "failed", "success" ),
                  help="unused -- 802.1X supplicant is uninitialized"
                  "down -- 802.1X supplicant is down."
                  "connecting -- 802.1X supplicant is connecting."
                  "failed -- 802.1X supplicant has failed."
                  "success -- 802.1X supplicant has succeeded." )
   supplicantMac = MacAddress( help="802.1X supplicant MAC address for "
                                    "the interface" )
   authenticatorMac = MacAddress( help="802.1X authenticator MAC address for "
                                       "the interface" )

   def fromTacc( self, supplicantIntfStatus, wpaSupplicantIntfStatus ):
      self.identity = supplicantIntfStatus.identity
      self.eapMethod = supplicantIntfStatus.eapMethod
      self.status = wpaSupplicantIntfStatus.connectionStatus
      self.supplicantMac = wpaSupplicantIntfStatus.eapKey.myMacAddr
      self.authenticatorMac = wpaSupplicantIntfStatus.eapKey.peerMacAddr

   def render( self ):
      indentation = ' ' * 4
      print indentation + "Identity: %s" % self.identity
      print indentation + "EAP method: %s" % self.eapMethod
      print indentation + "Status: %s" % self.status
      print indentation + "Supplicant MAC: %s" % self.supplicantMac
      print indentation + "Authenticator MAC: %s" % self.authenticatorMac

class Dot1xSupplicants( Model ):
   interfaces = Dict( help="A mapping between interfaces and 802.1X supplicant "
                           "interface information", keyType=Interface,
                           valueType=Dot1xSupplicant )

   def render( self ):
      if not self.interfaces:
         return

      for key in Arnet.sortIntf( self.interfaces ):
         print
         print "Interface: %s" % key
         self.interfaces[ key ].render()

class Dot1xHost( Model ):
   supplicantMac = MacAddress( help="supplicant MAC address" )
   authMethod = Str( help="Authentication method used by the supplicant" )
   authStage = Str( help="Supplicant state" )
   vlanId = Str( help='VLAN ID' )
   cachedAuthAtLinkDown = Bool( help="Cache supplicant on link down" )
   reauthTimeoutSeen = Bool( help="Reauth timeout seen" )

   def fromTacc( self, supplicantStatus ):
      self.supplicantMac = supplicantStatus.mac
      self.cachedAuthAtLinkDown = supplicantStatus.cachedAuthAtLinkDown
      self.reauthTimeoutSeen = supplicantStatus.reauthTimeoutSeen
      self.authMethod = 'EAPOL'
      if supplicantStatus.mbaSupplicant:
         self.authMethod = 'MAC-BASED-AUTH'
      self.vlanId = supplicantStatus.vlan
      self.authStage = authStageDict[ supplicantStatus.authStage ]

   def getAuthStageDetails( self ):
      macString = convertMacAddrToDisplay( self.supplicantMac.stringValue )
      if self.cachedAuthAtLinkDown:
         authStage = "DISCONNECTED-CACHED"
      elif self.reauthTimeoutSeen:
         authStage = "SUCCESS-CACHED"
      else:
         authStage = self.authStage

      for state in cliStageShorten:
         authStage = authStage.replace( state, cliStageShorten[ state ] )

      return ( macString, self.authMethod, authStage, self.vlanId )

class Dot1xHosts( Model ):
   supplicants = Dict( help="A mapping betweeen MAC addresses and Dot1X or MBA "
                            "supplicant information", keyType=MacAddress,
                            valueType=Dot1xHost )

   def populate( self, dot1xHostsTable, intf ):
      if not self.supplicants:
         return

      for supplicant in self.supplicants:
         ( macString, authMethod, authStage, vlanId ) = \
               self.supplicants[ supplicant ].getAuthStageDetails()
         dot1xHostsTable.newRow( intf, macString, authMethod, authStage, vlanId )

   def render( self ):
      if not self.supplicants:
         return

      widths = OrderedDict( [
         ( 'Supplicant MAC', 24 ),
         ( 'Auth Method', 20 ),
         ( 'State', 24 ),
         ( 'VLAN ID', 8 ) ] )

      dot1xHostTable = createTable( widths )
      columns = []
      for column in widths:
         formatColumn = Format( justify='left', minWidth=widths[ column ],
               maxWidth=widths[ column ] )
         formatColumn.noPadLeftIs( True )
         formatColumn.padLimitIs( True )
         formatColumn.noTrailingSpaceIs( True )
         columns.append( formatColumn )
      dot1xHostTable.formatColumns( *columns )

      for supplicant in self.supplicants:
         self.supplicants[ supplicant ].populate( dot1xHostTable )

      print dot1xHostTable.output()


class Dot1xAllHosts( Model ):
   intfSupplicantsDict = Dict( help="A mapping between the interfaces and all the "
                                    "MBA or Dot1X supplicants for that interface",
                               keyType=Interface, valueType=Dot1xHosts )
   def render( self ):
      if not self.intfSupplicantsDict:
         return

      widths = OrderedDict( [
         ( 'Port', 10 ),
         ( 'Supplicant MAC', 18 ),
         ( 'Auth Method', 20 ),
         ( 'State', 24 ),
         ( 'VLAN ID', 8 ) ] )

      dot1xHostsTable = createTable( widths )
      columns = []
      for column in widths:
         formatColumn = Format( justify='left', minWidth=widths[ column ],
               maxWidth=widths[ column ] )
         formatColumn.noPadLeftIs( True )
         formatColumn.padLimitIs( True )
         formatColumn.noTrailingSpaceIs( True )
         columns.append( formatColumn )
      dot1xHostsTable.formatColumns( *columns )

      for key in Arnet.sortIntf( self.intfSupplicantsDict ):
         self.intfSupplicantsDict[ key ].populate( dot1xHostsTable,
               IntfMode.getShortname( key ) )
      print dot1xHostsTable.output()

class Dot1xBlockedMacEntry( Model ):
   interfaces = Dict( help="A mapping between interface and VLAN",
                      keyType=Interface, valueType=int )
   def renderEntry( self, table, macAddr ):
      first = True
      for intf in Arnet.sortIntf( self.interfaces ):
         if first:
            first = False
            table.startRow()
            table.newRow( convertMacAddrToDisplay( macAddr ), intf,
                          self.interfaces[ intf ] )
         else:
            table.newRow( "", intf, self.interfaces[ intf ] )

class Dot1xBlockedMacTable( Model ):
   macAddresses = Dict( help="A mapping between blocked MAC address and list of "
                        "intferace and VLAN pair on which the MAC is blocked"
                        " through dynamic authorization",
                        keyType=MacAddress, valueType=Dot1xBlockedMacEntry )

   def render( self ):
      headings = ( "MAC Address", "Interface", "VLAN ID" )
      table = createTable( headings )
      for macAddr in sorted( self.macAddresses ):
         self.macAddresses[ macAddr ].renderEntry( table, macAddr )
      print table.output()

class Dot1xVlanAssignmentGroupNames( Model ):
   vlanGroupNameDict = Dict( help="A mapping of group name to that group's VLANs",
                         keyType=str, valueType=str )
   def render( self ):
      if not self.vlanGroupNameDict:
         return
      fields = [ 'VLAN Assignment Group', 'VLANs' ]
      fmt = "%-33s%s"
      print fmt % ( fields[ 0 ], fields[ 1 ] )
      print fmt % ( len( fields[ 0 ] ) * "-", len( fields[ 1 ] ) * "-" )
      for key in sorted( self.vlanGroupNameDict ):
         print fmt % ( key, self.vlanGroupNameDict[ key ] )
