# Copyright (c) 2009-2013 Arista Networks, Inc.  All rights reserved.
# Arista Networks, Inc. Confidential and Proprietary.
import BasicCliModes
import CliParser
import BasicCli
import LazyMount
import CliMatcher
import CliCommand
import ShowCommand
import Tac
import IntfCli
from IntfCli import Intf
from Intf.IntfRange import IntfRangeMatcher
from IntfRangePlugin.EthIntf import EthPhyAutoIntfType
from EthIntfCli import EthIntfModelet, EthPhyIntf
from SubIntfCli import SubIntfModelet
from BridgingCli import bridgingCheckStaticMacHook
from VlanCli import vlanIdMatcher, vlanSetMatcher
import CliToken.Clear
import Dot1xModel
from Dot1xLib import portControlStr2Enum, Dot1xConsts, hostModeStr2Enum
from Dot1xSupplicantCli import profileExpression, setDot1xSupplicant
from Dot1xSupplicantCli import noDot1xSupplicant, delSuppProfiles, matcherSupplicant
from Dot1xModeCli import Dot1xMode
import ConfigMount
import Tracing
import TechSupportCli
import Toggles.Dot1xToggleLib as Dot1xToggle
import Url
import UrlPlugin.NetworkUrl as _
import MacAddr
from TypeFuture import TacLazyType
from MultiRangeRule import multiRangeToCanonicalString
from AclCli import AclNameMatcher

t0 = Tracing.trace0
t8 = Tracing.trace8

config = None
cliConfig = None
configReq = None
status = None
identityDot1xStatus = None
dot1xInputHostTable = None
mergedHostTable = None
hwstatus = None
subIntfCapabilityDir = None
ipStatus = None
blockedMacTable = None

# Tac Type
EthIntfId = TacLazyType( "Arnet::EthIntfId" )
IntfId = TacLazyType( "Arnet::IntfId" )
PrivateTcpPorts = TacLazyType( "Arnet::PrivateTcpPorts" )

# Other global variables
possibleMgmtIntfs = [ "Management0",
                      "Management1",
                      "Management1/1", "Management1/2",
                      "Management2/1", "Management2/2" ]
webAuthHttpPort = PrivateTcpPorts.dot1xWebHttpPort
webAuthHttpsPort = PrivateTcpPorts.dot1xWebHttpsPort

def makeOrGetDot1xIntfConfig( intfName ):
   """
   Make a Dot1xIntfConfig and Dot1xConfigReq
   via the idempotent new operator.
   Returns the Dot1xIntfConfig
   """
   dot1xIntfConfig = config.newDot1xIntfConfig( intfName )
   configReq.newDot1xIntfConfigReq( intfName )
   return dot1xIntfConfig

#------------------------------------------------------------
# [no|default] dot1x system-auth-control
#------------------------------------------------------------
def setDot1xSysAuthControl( mode, args ):
   config.dot1xEnabled = True

def noDot1xSysAuthControl( mode, args ):
   config.dot1xEnabled = False

def dot1xSupportedGuard( mode, token ):
   if hwstatus.dot1xSupported:
      return None
   return CliParser.guardNotThisPlatform

def dot1xMbaSupportedGuard( mode, token ):
   if hwstatus.dot1xMbaSupported:
      return None
   return CliParser.guardNotThisPlatform

def dot1xOnSubIntfSupportedGuard( mode, token ):
   if not EthIntfId.isEthIntfId( mode.intf.name ):
      return None
   for sliceStatus in subIntfCapabilityDir.values():
      parentIntfName = EthIntfId.parentIntfIdFromSubIntf( mode.intf.name )
      if sliceStatus and parentIntfName in sliceStatus.dot1xCapable:
         return None

   return CliParser.guardNotThisPlatform 

matcherDot1x = CliMatcher.KeywordMatcher( 'dot1x',
               helpdesc='IEEE 802.1X port authentication' )
nodeDot1x = CliCommand.Node( matcherDot1x, guard=dot1xSupportedGuard )

class Dot1XSystemAuthControlCmd( CliCommand.CliCommandClass ):
   syntax = 'dot1x system-auth-control'
   noOrDefaultSyntax = 'dot1x system-auth-control ...'
   data = {
      'dot1x': nodeDot1x,
      'system-auth-control': 'Enable or disable SysAuthControl',
   }
   handler = setDot1xSysAuthControl
   noOrDefaultHandler = noDot1xSysAuthControl

BasicCliModes.GlobalConfigMode.addCommandClass( Dot1XSystemAuthControlCmd )

#------------------------------------------------------------
# [no|default] dot1x dynamic-authorization
#------------------------------------------------------------
def setDot1xDynamicAuthorization( mode, args ):
   config.dot1xDynAuthEnabled = True

def noDot1xDynamicAuthorization( mode, args ):
   config.dot1xDynAuthEnabled = False

class Dot1XDynamicAuthorizationCmd( CliCommand.CliCommandClass ):
   syntax = 'dot1x dynamic-authorization'
   noOrDefaultSyntax = 'dot1x dynamic-authorization'
   data = {
      'dot1x': nodeDot1x,
      'dynamic-authorization': 'Enable dynamic authorization',
   }
   handler = setDot1xDynamicAuthorization
   noOrDefaultHandler = noDot1xDynamicAuthorization

BasicCliModes.GlobalConfigMode.addCommandClass( Dot1XDynamicAuthorizationCmd )

#------------------------------------------------------------
# [no|default] radius av-pair service-type
#------------------------------------------------------------

def setDot1xRadiusServiceTypeAttribute( mode, args ):
   config.serviceType = True

def noDot1xRadiusServiceTypeAttribute( mode, args ):
   config.serviceType = False

class Dot1XRadiusServiceTypeCmd( CliCommand.CliCommandClass ):
   syntax = 'radius av-pair service-type'
   noOrDefaultSyntax = 'radius av-pair service-type'
   data = {
      'radius': 'Keyword to set RADIUS parameters',
      'av-pair': 'Attribute-Value Pair',
      'service-type': 'service-type av-pair',
   }
   handler = setDot1xRadiusServiceTypeAttribute
   noOrDefaultHandler = noDot1xRadiusServiceTypeAttribute

Dot1xMode.addCommandClass( Dot1XRadiusServiceTypeCmd )

#------------------------------------------------------------
# [no|default] radius av-pair framed-mtu <length>
#------------------------------------------------------------

matcherFramedMtu = CliMatcher.IntegerMatcher( Dot1xConsts.minFramedMtu, 
                                              Dot1xConsts.maxFramedMtu,
      helpdesc='RADIUS framed MTU size in bytes' )

def setDot1xRadiusFramedMtuAttribute( mode, args ):
   config.framedMtu = args[ 'LENGTH' ]

def noDot1xRadiusFramedMtuAttribute( mode, args ):
   config.framedMtu = config.defaultFramedMtu

class Dot1XRadiusFramedMtuCmd( CliCommand.CliCommandClass ):
   syntax = 'radius av-pair framed-mtu LENGTH'
   noOrDefaultSyntax = 'radius av-pair framed-mtu ...'
   data = {
      'radius': 'Keyword to set RADIUS parameters',
      'av-pair': 'Attribute-Value Pair',
      'framed-mtu': 'framed-mtu size av-pair',
      'LENGTH': matcherFramedMtu,
   }
   handler = setDot1xRadiusFramedMtuAttribute
   noOrDefaultHandler = noDot1xRadiusFramedMtuAttribute

Dot1xMode.addCommandClass( Dot1XRadiusFramedMtuCmd )

#------------------------------------------------------------
# [no|default] vlan assignment group <name> members <vlan_list>
#------------------------------------------------------------

groupNameMatcher = CliMatcher.PatternMatcher( pattern='.+',
                      helpname='WORD',
                      helpdesc='The ASCII name for the VLAN group',
                      value=lambda mode, match: match[ :32 ] )

def setDot1xVlanGroup( mode, args ):
   entry = Tac.Value( "Dot1x::VlanGroup", args[ 'GROUPNAME' ] )
   vlanIdSet = args[ 'VLAN_SET' ]
   for vlanId in vlanIdSet.ids:
      entry.vlanId.add( vlanId )
   config.vlanGroup.addMember( entry )

def noDot1xVlanGroup( mode, args ):
   del config.vlanGroup[ args[ 'GROUPNAME' ] ]
    
class Dot1XVlanGroupCmd( CliCommand.CliCommandClass ):
   syntax = 'vlan assignment group GROUPNAME members VLAN_SET'
   noOrDefaultSyntax = 'vlan assignment group GROUPNAME ...'
   data = {
         'vlan': 'VLAN group',
         'assignment': 'Assign VLAN to group',
         'group': 'Group name',
         'GROUPNAME': groupNameMatcher,
         'members': 'Members of group',
         'VLAN_SET': vlanSetMatcher,
   }
   handler = setDot1xVlanGroup
   noOrDefaultHandler = noDot1xVlanGroup

if Dot1xToggle.toggleDot1xRandomVlanAssignmentEnabled():
   Dot1xMode.addCommandClass( Dot1XVlanGroupCmd )

#------------------------------------------------------------
# [no|default] mac-based-auth radius av-pair <attribute-name> 
#                  delimiter [none|colon|hyphen|period] [uppercase|lowercase]
#------------------------------------------------------------

# XXX Remove the CLI guard once BUG182344 is fixed.
nodeMBA = CliCommand.Node( CliMatcher.KeywordMatcher( 'mac-based-auth',
                  helpdesc='Dot1x Mac Based Auth' ),
                  guard=dot1xMbaSupportedGuard )

def setDot1xMbaAttribute( mode, args ):
   delimiterMap = {
         'none': '',
         'colon': ':',
         'hyphen': '-',
         'period': '.'
   }
   config.mbaUserName = Tac.Value( "Dot1x::MbaUserName",
                                    userNameDelim = \
                                          delimiterMap[ args[ 'DELIMITER' ] ],
                                    userNameIsUpperCase = \
                                          ( args[ 'CASE' ] == 'uppercase' ),
                                    userNameGroupSize = \
                                          4 if args[ 'DELIMITER' ] == \
                                             'period' else 2 )

def noDot1xMbaAttribute( mode, args ):
   config.mbaUserName = Tac.Value( "Dot1x::MbaUserName" )

class Dot1XMbaAttributeFormatCmd( CliCommand.CliCommandClass ):
   syntax = 'mac-based-auth radius av-pair user-name delimiter DELIMITER CASE'
   noOrDefaultSyntax = 'mac-based-auth radius av-pair user-name ...'
   data = {
      'mac-based-auth': nodeMBA,
      'radius': 'Keyword to set RADIUS parameters',
      'av-pair': 'Attribute-Value Pair',
      'user-name': 'RADIUS user-name attribute',
      'delimiter': 'Delimiter to use',
      'DELIMITER': CliMatcher.EnumMatcher( {
         'none': 'No delimiter in MAC address string',
         'colon': 'Set colon as delimiter in MAC address string',
         'hyphen': 'Set hyphen as delimiter in MAC address string',
         'period': 'Set period as delimiter in MAC address string',
      } ),
      'CASE': CliMatcher.EnumMatcher( {
         'uppercase': 'MAC address string in uppercase',
         'lowercase': 'MAC address string in lowercase',
      } ),
   }
   handler = setDot1xMbaAttribute
   noOrDefaultHandler = noDot1xMbaAttribute

Dot1xMode.addCommandClass( Dot1XMbaAttributeFormatCmd )

#------------------------------------------------------------
# (dot1x)# [no|default] captive-portal [ url <URL>]
#------------------------------------------------------------

def setCaptivePortal( mode, args ):
   urlValue = args.get( 'URL', "" )
   config.captivePortal = Tac.Value( "Dot1x::CaptivePortal", 
                                     enabled=True, 
                                     url=urlValue and urlValue.url,
                                     cpHostName=urlValue and urlValue.urlhostname )

def noCaptivePortal( mode, args ):
   config.captivePortal = Tac.Value( "Dot1x::CaptivePortal" )

class Dot1xCaptivePortalCmd( CliCommand.CliCommandClass ):
   syntax = 'captive-portal [ url URL ]'
   noOrDefaultSyntax = 'captive-portal ...'

   data = {
      'captive-portal' : 'Configure captive portal parameters',
      'url' : 'Configure captive portal URL',
      'URL' :  Url.UrlMatcher( lambda fs: fs.scheme in [ 'http:', 'https:' ], 
         helpdesc="URL of form http[s]://<hostname>[:<port>]" )
   }
   handler = setCaptivePortal
   noOrDefaultHandler = noCaptivePortal
      
if Dot1xToggle.toggleDot1xWebAuthEnabled():
   Dot1xMode.addCommandClass( Dot1xCaptivePortalCmd )

#------------------------------------------------------------------
# (dot1x)# [no|default] captive-portal access-list ipv4 <acl-name>
#------------------------------------------------------------------

aclNameMatcher = AclNameMatcher()

def setCaptivePortalACL( mode, args ):
   aclName = args.get( 'ACL_NAME', "" )
   config.captivePortalIpv4Acl = aclName

class Dot1xCaptivePortalACLCmd( CliCommand.CliCommandClass ):
   syntax = 'captive-portal access-list ipv4 ACL_NAME'
   noOrDefaultSyntax = 'captive-portal access-list ipv4 ...'

   data = {
         'captive-portal': 'Configure captive portal parameters',
         'access-list': 'Configure access control list',
         'ipv4': 'Type of access control list',
         'ACL_NAME': CliCommand.Node( matcher=aclNameMatcher )
   }
   handler = setCaptivePortalACL
   noOrDefaultHandler = setCaptivePortalACL

if Dot1xToggle.toggleDot1xAclWebAuthEnabled():
   Dot1xMode.addCommandClass( Dot1xCaptivePortalACLCmd )

#------------------------------------------------------------
# (dot1x)# [no|default] address tracking ipv4
#------------------------------------------------------------

def setIpTracking( mode, args ):
   no = CliCommand.isNoOrDefaultCmd( args )
   if no:
      config.ipTrackingEnabled = False
   else:
      config.ipTrackingEnabled = True
   
class Dot1xIpTrackingCmd( CliCommand.CliCommandClass ):
   syntax = 'address tracking ipv4'
   noOrDefaultSyntax = syntax

   data = {
      'address' : 'Enable IP tracking',
      'tracking' : 'Enable IP tracking',
      'ipv4' : 'Enable tracking for IPv4 addresses'
   }
   handler = setIpTracking
   noOrDefaultHandler = handler
      
if Dot1xToggle.toggleDot1xIpTrackingEnabled():
   Dot1xMode.addCommandClass( Dot1xIpTrackingCmd )

#------------------------------------------------------------
# [no|default] dot1x protocol lldp bypass
#------------------------------------------------------------
def setLldpBypass( mode, args ):
   config.lldpBypass = True

def noLldpBypass( mode, args ):
   config.lldpBypass = False

class Dot1XProtocolLldpBypassCmd( CliCommand.CliCommandClass ):
   syntax = 'dot1x protocol lldp bypass'
   noOrDefaultSyntax = 'dot1x protocol lldp bypass'
   data = {
      'dot1x': nodeDot1x,
      'protocol': 'Set protocol processing',
      'lldp': 'LLDP frame processing',
      'bypass': 'Transmit/Receive without protection',
   }
   handler = setLldpBypass
   noOrDefaultHandler = noLldpBypass

BasicCliModes.GlobalConfigMode.addCommandClass( Dot1XProtocolLldpBypassCmd )

#------------------------------------------------------------
# (dot1x)# [no|default] aaa unresponsive action apply cached-results else traffic
#                                                             allow [ vlan <vlanID> ]
# (dot1x)# [no|default] aaa unresponsive action traffic allow [ vlan <vlanID> ]
# (dot1x)# [no|default] aaa unresponsive action apply cached-results
#------------------------------------------------------------
matcherDot1xAaa = CliMatcher.KeywordMatcher(
   'aaa', helpdesc='Configure AAA parameters' )
matcherDot1xAaaUnresp = CliMatcher.KeywordMatcher(
   'unresponsive', helpdesc='Configure AAA timeout options' )
matcherDot1xAaaUnrespAction = CliMatcher.KeywordMatcher(
   'action', helpdesc='Set action for supplicant when AAA times out' )
matcherDot1xAaaUnrespActionTraffic = CliMatcher.KeywordMatcher(
   'traffic', helpdesc='Set action for supplicant traffic when AAA times out' )
matcherDot1xAaaUnrespActionTrafficAllow = CliMatcher.KeywordMatcher(
   'allow', helpdesc='Allow traffic when AAA times out' )
matcherDot1xAaaUnrespActionApply = CliMatcher.KeywordMatcher(
   'apply', helpdesc='Apply an action when AAA times out' )
matcherDot1xAaaUnrespCachedResults = CliMatcher.KeywordMatcher(
   'cached-results', helpdesc='Use results from a previous AAA response' )
matcherDot1xElse = CliMatcher.KeywordMatcher(
   'else', helpdesc='Apply alternate action if primary action fails' )

def setAaaUnresponsiveTrafficAllow( mode, args ):
   config.aaaTimeoutTrafficAllow = 'allow' in args
   vlan = args.get( 'VLAN', 0 )
   config.aaaTimeoutTrafficAllowVlan = Tac.Value( 'Bridging::VlanIdOrNone',
                                                  vlan and vlan.id )
   config.aaaTimeoutApplyCachedResults = 'cached-results' in args

def noAaaUnresponsiveTrafficAllow( mode, args ):
   config.aaaTimeoutTrafficAllow = False
   config.aaaTimeoutTrafficAllowVlan = Tac.Value( 'Bridging::VlanIdOrNone', 0 )
   config.aaaTimeoutApplyCachedResults = False

class Dot1XAaaUnresponsiveTrafficActionAllowCmd( CliCommand.CliCommandClass ):
   if Dot1xToggle.toggleDot1xCachedAuthEnabled():
      syntax = '''aaa unresponsive action
      ( ( apply cached-results [ else traffic allow [ vlan VLAN ] ] )
      | ( traffic allow [ vlan VLAN ] ) )'''
   else:
      syntax = 'aaa unresponsive action traffic allow [ vlan VLAN ]'
   noOrDefaultSyntax = 'aaa unresponsive action ...'
   data = {
      'aaa': matcherDot1xAaa,
      'unresponsive': matcherDot1xAaaUnresp,
      'action': matcherDot1xAaaUnrespAction,
      'apply' : matcherDot1xAaaUnrespActionApply,
      'cached-results' : matcherDot1xAaaUnrespCachedResults,
      'else' : matcherDot1xElse,
      'traffic': matcherDot1xAaaUnrespActionTraffic,
      'allow': matcherDot1xAaaUnrespActionTrafficAllow,
      'vlan': 'Allow traffic in VLAN when AAA times out',
      'VLAN': vlanIdMatcher,
   }
   handler = setAaaUnresponsiveTrafficAllow
   noOrDefaultHandler = noAaaUnresponsiveTrafficAllow

if Dot1xToggle.toggleDot1xAaaUnresponsiveVlanEnabled():
   Dot1xMode.addCommandClass( Dot1XAaaUnresponsiveTrafficActionAllowCmd )

#------------------------------------------------------------
# (dot1x)# [no|default] aaa unresponsive phone action apply cached-results else
#                                                                       traffic allow
# (dot1x)# [no|default] aaa unresponsive phone action traffic allow
# (dot1x)# [no|default] aaa unresponsive phone action apply cached-results
#------------------------------------------------------------
def setAaaUnresponsivePhoneAllow( mode, args ):
   config.aaaTimeoutPhoneAllow = 'allow' in args
   if 'cached-results' in args:
      config.aaaTimeoutPhoneApplyCachedResults = 'aaaTimeoutApplyCached'
   else:
      config.aaaTimeoutPhoneApplyCachedResults = 'aaaTimeoutDoNotApplyCached'

def noAaaUnresponsivePhoneAllow( mode, args ):
   config.aaaTimeoutPhoneAllow = False
   config.aaaTimeoutPhoneApplyCachedResults = 'aaaTimeoutCacheInheritRule'

class Dot1XAaaUnresponsivePhoneActionAllowCmd( CliCommand.CliCommandClass ):
   if Dot1xToggle.toggleDot1xCachedAuthEnabled():
      syntax = '''aaa unresponsive phone action
      ( ( apply cached-results [ else traffic allow ] ) | ( traffic allow ) )'''
   else:
      syntax = 'aaa unresponsive phone action traffic allow'
   noOrDefaultSyntax = 'aaa unresponsive phone action ...'
   data = {
      'aaa': matcherDot1xAaa,
      'unresponsive': matcherDot1xAaaUnresp,
      'phone': 'Configure AAA timeout options for phone',
      'action': matcherDot1xAaaUnrespAction,
      'apply' : matcherDot1xAaaUnrespActionApply,
      'cached-results' : matcherDot1xAaaUnrespCachedResults,
      'else' : matcherDot1xElse,
      'traffic': matcherDot1xAaaUnrespActionTraffic,
      'allow': 'Allow phone traffic in phone VLAN when AAA times out',
   }
   handler = setAaaUnresponsivePhoneAllow
   noOrDefaultHandler = noAaaUnresponsivePhoneAllow

if Dot1xToggle.toggleDot1xDefaultPhoneVlanEnabled():
   Dot1xMode.addCommandClass( Dot1XAaaUnresponsivePhoneActionAllowCmd )

#--------------------------------------------------------------------------
# (dot1x)# [no|default] aaa accounting update interval [ INTERVAL ] seconds
#--------------------------------------------------------------------------
def setDot1xAccountingUpdateInterval( mode, args ):
   config.accountingUpdateInterval = args[ 'INTERVAL' ]

def noDot1xAccountingUpdateInterval( mode, args ):
   config.accountingUpdateInterval = config.defaultAccountingUpdateInterval

matcherIntervalPeriod = CliMatcher.IntegerMatcher( 5, 65535, 
      helpdesc='Interval period in seconds' )

class Dot1XAccountingUpdateIntervalCmd( CliCommand.CliCommandClass ):
   syntax = 'aaa accounting update interval INTERVAL seconds'
   noOrDefaultSyntax = 'aaa accounting update interval ...'
   data = {
      'aaa': matcherDot1xAaa,
      'accounting': 'Configure accounting',
      'update' : 'Configure update messages',
      'interval': 'Configure interval',
      'INTERVAL' : matcherIntervalPeriod,
      'seconds' : 'Unit in seconds'
   }
   handler = setDot1xAccountingUpdateInterval
   noOrDefaultHandler = noDot1xAccountingUpdateInterval

Dot1xMode.addCommandClass( Dot1XAccountingUpdateIntervalCmd )

#------------------------------------------------------------
# [no|default] dot1x pae authenticator
#------------------------------------------------------------
def setDot1xPaeAuthenticator( mode, args ):
   # Disable supplicant configuration on the interface in case it was configured 
   # earlier.
   noDot1xPaeSupplicant( mode, args )
   
   # Configure the interface to be the authenticator.
   dot1xIntfConfig = makeOrGetDot1xIntfConfig( mode.intf.name )
   dot1xIntfConfig.dot1xEnabled = True

def noDot1xPaeAuthenticator( mode, args ):
   dot1xIntfConfig = config.dot1xIntfConfig.get( mode.intf.name )
   if dot1xIntfConfig:
      dot1xIntfConfig.dot1xEnabled = False

matcherIntfDot1x = CliMatcher.KeywordMatcher( 'dot1x',
                                       helpdesc='IEEE 802.1X port authentication' )
nodeIntfDot1x = CliCommand.Node( matcherIntfDot1x, guard=dot1xSupportedGuard )
nodeSubIntfDot1x = CliCommand.Node( matcherIntfDot1x,
                                    guard=dot1xOnSubIntfSupportedGuard )
matcherPae = CliMatcher.KeywordMatcher( 'pae',
         helpdesc='Set 802.1X interface Port Access Entity type' )

class Dot1XPaeAuthenticatorCmd( CliCommand.CliCommandClass ):
   syntax = 'dot1x pae authenticator'
   noOrDefaultSyntax = 'dot1x pae authenticator ...'
   handler = setDot1xPaeAuthenticator
   noOrDefaultHandler = noDot1xPaeAuthenticator

class Dot1XPaeAuthenticatorOnIntfCmd( Dot1XPaeAuthenticatorCmd ):
   data = {
      'dot1x': nodeIntfDot1x,
      'pae': matcherPae,
      'authenticator': 'Set Port Access Entity type as Authenticator',
   }

EthIntfModelet.addCommandClass( Dot1XPaeAuthenticatorOnIntfCmd )

class Dot1XPaeAuthenticatorOnSubIntfCmd( Dot1XPaeAuthenticatorCmd ):
   data = {
      'dot1x': nodeSubIntfDot1x,
      'pae': matcherPae,
      'authenticator': 'Set Port Access Entity type as Authenticator',
   }

SubIntfModelet.addCommandClass( Dot1XPaeAuthenticatorOnSubIntfCmd )
#------------------------------------------------------------
# [no|default] dot1x pae supplicant <profileName>
#------------------------------------------------------------
def setDot1xPaeSupplicant( mode, args ):
   profileName = args[ 'PROFILE' ]
   # Disable authenticator on the interface if it was configured earlier.
   noDot1xPaeAuthenticator( mode, args )
   # Configure the interface to be the supplicant.
   setDot1xSupplicant( mode, profileName )

def noDot1xPaeSupplicant( mode, args ):
   noDot1xSupplicant( mode )

class Dot1XPaeSupplicantCmd( CliCommand.CliCommandClass ):
   syntax = 'dot1x pae supplicant PROFILE'
   noOrDefaultSyntax = 'dot1x pae supplicant ...'
   handler = setDot1xPaeSupplicant
   noOrDefaultHandler = noDot1xPaeSupplicant

class Dot1XPaeSupplicantOnIntfCmd( Dot1XPaeSupplicantCmd ):
   data = {
      'dot1x': nodeIntfDot1x,
      'pae': matcherPae,
      'supplicant': matcherSupplicant,
      'PROFILE': profileExpression,
   }

EthIntfModelet.addCommandClass( Dot1XPaeSupplicantOnIntfCmd )

class Dot1XPaeSupplicantOnSubIntfCmd( Dot1XPaeSupplicantCmd ):
   data = {
      'dot1x': nodeSubIntfDot1x,
      'pae': matcherPae,
      'supplicant': matcherSupplicant,
      'PROFILE': profileExpression,
   }

SubIntfModelet.addCommandClass( Dot1XPaeSupplicantOnSubIntfCmd )
#------------------------------------------------------------
# [no|default dot1x 
#------------------------------------------------------------
def gotoDot1xMode( mode, args ):
   childMode = mode.childMode( Dot1xMode )
   mode.session_.gotoChildMode( childMode )

def noDot1xMode( mode, args ):
   delSuppProfiles()

class Dot1XCmd( CliCommand.CliCommandClass ):
   syntax = 'dot1x'
   noOrDefaultSyntax = 'dot1x'
   data = {
      'dot1x': nodeDot1x,
   }
   handler = gotoDot1xMode
   noOrDefaultHandler = noDot1xMode

BasicCliModes.GlobalConfigMode.addCommandClass( Dot1XCmd )

#------------------------------------------------------------
# [no|default] dot1x re-authenticate
#------------------------------------------------------------
def setReauthenticateIntf( mode, args ):
   intf = args[ 'IPINTF' ]
   dot1xIntfConfigReq = configReq.dot1xIntfConfigReq.get( intf.name )
   if dot1xIntfConfigReq:
      dot1xIntfConfigReq.manualReauthRequestTime = Tac.now()

class Dot1XReAuthenticateIpintfCmd( CliCommand.CliCommandClass ):
   syntax = 'dot1x re-authenticate IPINTF'
   data = {
      'dot1x': nodeDot1x,
      're-authenticate': 'Manually re-authenticate 802.1X supplicant',
      'IPINTF': Intf.matcherWithIpSupport,
   }
   handler = setReauthenticateIntf

BasicCliModes.EnableMode.addCommandClass( Dot1XReAuthenticateIpintfCmd )

#---------------------------------------------------------------------------
# [no|default] dot1x port-control auto|force-authorized|force-unauthorized
#---------------------------------------------------------------------------
def setPortControl( mode, args ):
   portControl = args[ 'PORT_STATE' ]
   dot1xIntfConfig = makeOrGetDot1xIntfConfig( mode.intf.name )
   dot1xIntfConfigReq = configReq.dot1xIntfConfigReq.get( mode.intf.name )
   assert dot1xIntfConfigReq, \
          "dot1xIntfConfig and Req should have been made together"
   dot1xIntfConfigReq.lastPortCtrlSetting = dot1xIntfConfig.portCtrlSetting
   dot1xIntfConfig.portCtrlSetting = portControlStr2Enum[ portControl ]

def noPortControl( mode, args ):
   dot1xIntfConfig = config.dot1xIntfConfig.get( mode.intf.name )
   if dot1xIntfConfig:
      dot1xIntfConfigReq = configReq.dot1xIntfConfigReq.get( mode.intf.name )
      dot1xIntfConfigReq.lastPortCtrlSetting = dot1xIntfConfig.portCtrlSetting
      dot1xIntfConfig.portCtrlSetting = 'forceAuth'

matcherPortControl = CliMatcher.KeywordMatcher( 'port-control',
                              helpdesc='Set port control state' )
matcherPortState = CliMatcher.EnumMatcher( {
   'auto': 'Set port state to automatic',
   'force-authorized': 'Set port state to authorized',
   'force-unauthorized': 'Set port state to unauthorized',
} )

class Dot1XPortControlCmd( CliCommand.CliCommandClass ):
   syntax = 'dot1x port-control PORT_STATE'
   noOrDefaultSyntax = 'dot1x port-control ...'
   handler = setPortControl
   noOrDefaultHandler = noPortControl

class Dot1XPortControlOnIntfCmd( Dot1XPortControlCmd ):
   data = {
      'dot1x': nodeIntfDot1x,
      'port-control': matcherPortControl,
      'PORT_STATE': matcherPortState,
   }

EthIntfModelet.addCommandClass( Dot1XPortControlOnIntfCmd )

class Dot1XPortControlOnSubIntfCmd( Dot1XPortControlCmd ):
   data = {
      'dot1x': nodeSubIntfDot1x,
      'port-control': matcherPortControl,
      'PORT_STATE': matcherPortState,
   }

SubIntfModelet.addCommandClass( Dot1XPortControlOnSubIntfCmd )
#-----------------------------------------------------------
# [no|default] dot1x timeout quiet-period|tx-period
#-----------------------------------------------------------
def noQuietPeriod( mode ):
   dot1xIntfConfig = config.dot1xIntfConfig.get( mode.intf.name )
   if dot1xIntfConfig:
      dot1xIntfConfig.quietPeriod = dot1xIntfConfig.defaultQuietPeriod

def noTxPeriod( mode ):
   dot1xIntfConfig = config.dot1xIntfConfig.get( mode.intf.name )
   if dot1xIntfConfig:
      dot1xIntfConfig.txPeriod = dot1xIntfConfig.defaultTxPeriod

matcherTimeout = CliMatcher.KeywordMatcher( 'timeout', 
      helpdesc='Configure timeout setting' )
matcherTimeoutPeriod = CliMatcher.IntegerMatcher( 1, 65535,
      helpdesc='Timeout period in seconds' )
# helpname alters the default help text for the new range
matcherReauthPeriod = CliMatcher.IntegerMatcher( 1, 65535,
      helpname='<60-65535>',
      helpdesc='Timeout period in seconds' )

class Dot1XTimeoutCmd( CliCommand.CliCommandClass ):
   syntax = 'dot1x timeout PERIOD TIMEOUT'
   noOrDefaultSyntax = 'dot1x timeout PERIOD ...'
   data = {
      'timeout': matcherTimeout,
      'PERIOD': CliMatcher.EnumMatcher( {
         'quiet-period': 'Timeout for quiet period after a failed authentication',
         'tx-period': 'Timeout for supplicant retries',
      } ),
      'TIMEOUT': matcherTimeoutPeriod,
   }
   
   @staticmethod
   def handler( mode, args ):
      attr = 'quietPeriod' if args[ 'PERIOD' ] == 'quiet-period' else 'txPeriod'
      dot1xIntfConfig = makeOrGetDot1xIntfConfig( mode.intf.name )
      setattr( dot1xIntfConfig, attr, args[ 'TIMEOUT' ] )

   @staticmethod
   def noOrDefaultHandler( mode, args ):
      if 'quiet-period' in args[ 'PERIOD' ]:
         noQuietPeriod( mode )
      else:
         noTxPeriod( mode )

class Dot1XTimeoutCmdOnIntf( Dot1XTimeoutCmd ):
   data = Dot1XTimeoutCmd.data.copy()
   data[ 'dot1x' ] = nodeIntfDot1x

EthIntfModelet.addCommandClass( Dot1XTimeoutCmdOnIntf )

class Dot1XTimeoutCmdOnSubIntf( Dot1XTimeoutCmd ):
   data = Dot1XTimeoutCmd.data.copy()
   data[ 'dot1x' ] = nodeSubIntfDot1x

SubIntfModelet.addCommandClass( Dot1XTimeoutCmdOnSubIntf )
#-----------------------------------------------------------------
# [no|default] dot1x reauthentication
# [no|default] dot1x timeout reauth-period [<60-65535>|server]
#-----------------------------------------------------------------
def setReauth( mode, args ):
   dot1xIntfConfig = makeOrGetDot1xIntfConfig( mode.intf.name )
   if not dot1xIntfConfig.reauth:
      dot1xIntfConfig.reauth = True

def noReauth( mode, args ):
   dot1xIntfConfig = config.dot1xIntfConfig.get( mode.intf.name )
   if dot1xIntfConfig:
      dot1xIntfConfig.reauth = False

class Dot1XReauthenticationCmd( CliCommand.CliCommandClass ):
   syntax = 'dot1x reauthentication'
   noOrDefaultSyntax = 'dot1x reauthentication ...'
   data = {
      'reauthentication': 'Enable or disable reauthentication',
   }
   handler = setReauth
   noOrDefaultHandler = noReauth

class Dot1XReauthenticationCmdOnIntf( Dot1XReauthenticationCmd ):
   data = Dot1XReauthenticationCmd.data.copy()
   data[ 'dot1x' ] = nodeIntfDot1x

EthIntfModelet.addCommandClass( Dot1XReauthenticationCmdOnIntf )

class Dot1XReauthenticationCmdOnSubIntf( Dot1XReauthenticationCmd ):
   data = Dot1XReauthenticationCmd.data.copy()
   data[ 'dot1x' ] = nodeSubIntfDot1x

SubIntfModelet.addCommandClass( Dot1XReauthenticationCmdOnSubIntf )

class Dot1XReauthPeriodCmd( CliCommand.CliCommandClass ):
   syntax = 'dot1x timeout reauth-period ( TIMEOUT | server )'
   noOrDefaultSyntax = 'dot1x timeout reauth-period ...'
   data = {
      'timeout': matcherTimeout,
      'reauth-period': 'Timeout for periodically re-authenticate supplicant',
      'TIMEOUT': matcherReauthPeriod,
      'server': 'Obtain re-authentication timeout value from the server',
   }

   @staticmethod
   def handler( mode, args ):
      dot1xIntfConfig = makeOrGetDot1xIntfConfig( mode.intf.name )
      if 'server' in args:
         dot1xIntfConfig.reauthOpts = 'useAuthSrvrSettings'
      else:
         # the help text for period has been changed to suggest only 60 seconds or
         # above is supported.  However for upgrade purposes, we will detect smaller
         # values and use a minimum 60 second value.  If interactive, the warning
         # will be printed.
         period = args[ 'TIMEOUT' ]
         if period < 60:
            mode.addWarning(
               "A reauth period of less than 60 seconds is no longer supported" )
            period = 60
         dot1xIntfConfig.reauthPeriod = period
         dot1xIntfConfig.reauthOpts = 'useSetTimeout'

   @staticmethod
   def noOrDefaultHandler( mode, args ):
      dot1xIntfConfig = config.dot1xIntfConfig.get( mode.intf.name )
      if dot1xIntfConfig:
         dot1xIntfConfig.reauthPeriod = dot1xIntfConfig.defaultReauthPeriod
         dot1xIntfConfig.reauthOpts = 'useSetTimeout'

class Dot1XReauthPeriodCmdOnIntf( Dot1XReauthPeriodCmd ):
   data = Dot1XReauthPeriodCmd.data.copy()
   data[ 'dot1x' ] = nodeIntfDot1x

EthIntfModelet.addCommandClass( Dot1XReauthPeriodCmdOnIntf )

class Dot1XReauthPeriodCmdOnSubIntf( Dot1XReauthPeriodCmd ):
   data = Dot1XReauthPeriodCmd.data.copy()
   data[ 'dot1x' ] = nodeSubIntfDot1x

SubIntfModelet.addCommandClass( Dot1XReauthPeriodCmdOnSubIntf )

#------------------------------------------------------------ 
# [no|default] dot1x timeout reauth-timeout-ignore always
#------------------------------------------------------------ 
def setDot1xReauthTimeoutIgnore( mode, args ):
   dot1xIntfConfig = makeOrGetDot1xIntfConfig( mode.intf.name ) 
   dot1xIntfConfig.reauthTimeoutIgnore = True 

def noDot1xReauthTimeoutIgnore( mode, args ):
   dot1xIntfConfig = config.dot1xIntfConfig.get( mode.intf.name ) 
   if dot1xIntfConfig:
      dot1xIntfConfig.reauthTimeoutIgnore = False 

matcherTimeout = CliMatcher.KeywordMatcher( 'timeout',
         helpdesc='Configure timeout setting' )
matcherReauthTimeoutIgnore = CliMatcher.KeywordMatcher( 'reauth-timeout-ignore',
         helpdesc='Retain current port auth status on reauth server timeouts' )
matcherAlways = CliMatcher.KeywordMatcher( 'always',
         helpdesc='Retain port auth status indefinitely on reauth server timeouts' )

class Dot1XReauthTimeoutIgnoreCmd( CliCommand.CliCommandClass ):
   syntax = 'dot1x timeout reauth-timeout-ignore always'
   noOrDefaultSyntax = 'dot1x timeout reauth-timeout-ignore ...'
   handler = setDot1xReauthTimeoutIgnore
   noOrDefaultHandler = noDot1xReauthTimeoutIgnore

class Dot1XReauthTimeoutIgnoreOnIntfCmd( Dot1XReauthTimeoutIgnoreCmd ):
   data = {
      'dot1x': nodeIntfDot1x,
      'timeout': matcherTimeout,
      'reauth-timeout-ignore': matcherReauthTimeoutIgnore,
      'always': matcherAlways,
   }

EthIntfModelet.addCommandClass( Dot1XReauthTimeoutIgnoreOnIntfCmd )

class Dot1XReauthTimeoutIgnoreOnSubIntfCmd( Dot1XReauthTimeoutIgnoreCmd ):
   data = {
      'dot1x': nodeSubIntfDot1x,
      'timeout': matcherTimeout,
      'reauth-timeout-ignore': matcherReauthTimeoutIgnore,
      'always': matcherAlways,
   }

SubIntfModelet.addCommandClass( Dot1XReauthTimeoutIgnoreOnSubIntfCmd )
#-----------------------------------------------------------------
# [ no | default ] dot1x reauthorization request limit
#
# legacy:
# [ no | default ] dot1x max-reauth-req
#-----------------------------------------------------------------
def setMaxReauthReq( mode, args ):
   req = args[ 'LIMIT' ]
   dot1xIntfConfig = makeOrGetDot1xIntfConfig( mode.intf.name )
   dot1xIntfConfig.maxReauthReq = req

def noMaxReauthReq( mode, args ):
   dot1xIntfConfig = config.dot1xIntfConfig.get( mode.intf.name )
   if dot1xIntfConfig:
      dot1xIntfConfig.maxReauthReq = dot1xIntfConfig.defaultMaxReauthReq

# [ no | default ] dot1x max-reauth-req - deprecated 

matcherReqLimit = CliMatcher.IntegerMatcher( 1, 10,
      helpdesc='Max number of reauth request allowed' )

class Dot1XMaxReauthReqCmd( CliCommand.CliCommandClass ):
   syntax = 'dot1x max-reauth-req LIMIT'
   noOrDefaultSyntax = 'dot1x max-reauth-req ...'
   data = {
      'max-reauth-req': CliCommand.Node( 
         CliMatcher.KeywordMatcher( 'max-reauth-req',
         helpdesc='Maximum number of reauthentication attempts' ),
         deprecatedByCmd='dot1x reauthorization request limit' ),
      'LIMIT': matcherReqLimit,
   }
   handler = setMaxReauthReq
   noOrDefaultHandler = noMaxReauthReq
   
class Dot1XMaxReauthReqCmdOnIntf( Dot1XMaxReauthReqCmd ):
   data = Dot1XMaxReauthReqCmd.data.copy()
   data[ 'dot1x' ] = nodeIntfDot1x
   
EthIntfModelet.addCommandClass( Dot1XMaxReauthReqCmdOnIntf )

class Dot1XMaxReauthReqCmdOnSubIntf( Dot1XMaxReauthReqCmd ):
   data = Dot1XMaxReauthReqCmd.data.copy()
   data[ 'dot1x' ] = nodeSubIntfDot1x

SubIntfModelet.addCommandClass( Dot1XMaxReauthReqCmdOnSubIntf )

# [ no | default ] dot1x reauthorization request limit

class Dot1XReauthReqLimitCmd( CliCommand.CliCommandClass ):
   syntax = 'dot1x reauthorization request limit LIMIT'
   noOrDefaultSyntax = 'dot1x reauthorization request limit ...'
   data = {
      'reauthorization': 'Maximum number of reauthentication attempts',
      'request': 'Maximum number of reauthentication attempts',
      'limit': 'Maximum number of reauthentication attempts',
      'LIMIT': matcherReqLimit,
   }
   handler = setMaxReauthReq
   noOrDefaultHandler = noMaxReauthReq

class Dot1XReauthReqLimitCmdOnIntf( Dot1XReauthReqLimitCmd ):
   data = Dot1XReauthReqLimitCmd.data.copy()
   data[ 'dot1x' ] = nodeIntfDot1x

EthIntfModelet.addCommandClass( Dot1XReauthReqLimitCmdOnIntf )

class Dot1XReauthReqLimitCmdOnSubIntf( Dot1XReauthReqLimitCmd ):
   data = Dot1XReauthReqLimitCmd.data.copy()
   data[ 'dot1x' ] = nodeSubIntfDot1x

SubIntfModelet.addCommandClass( Dot1XReauthReqLimitCmdOnSubIntf )

#------------------------------------------------------------------------------
# [no|default] dot1x host-mode ( single-host | ( multi-host [authenticated] ) )
#------------------------------------------------------------------------------
def singleHostModeGuard( mode, token ):
   if hwstatus.singleHostModeSupported:
      return None
   return CliParser.guardNotThisPlatform

matcherHostMode = CliMatcher.KeywordMatcher( 'host-mode',
                  helpdesc='Set the Host mode for authentication on this interface' )
matcherSingleHost = CliMatcher.KeywordMatcher( 'single-host',
                                               helpdesc='Single host mode' )
nodeSingleHost = CliCommand.Node( matcherSingleHost, guard=singleHostModeGuard )
matcherMultiHost = CliMatcher.KeywordMatcher( 'multi-host',
                                              helpdesc='Multiple host mode' )
matcherAuthenticated = CliMatcher.KeywordMatcher( 'authenticated',
                        helpdesc='Require each host to authenticate individually' )
nodeAuthenticated = CliCommand.Node( matcherAuthenticated,
                                     guard=singleHostModeGuard )

class Dot1XHostModeCmd( CliCommand.CliCommandClass ):
   syntax = 'dot1x host-mode ( single-host | ( multi-host [ authenticated ] ) )'
   noOrDefaultSyntax = 'dot1x host-mode ...'
   data = {
      'dot1x' : nodeIntfDot1x,
      'host-mode' : matcherHostMode,
      'single-host' : nodeSingleHost,
      'multi-host' : matcherMultiHost,
      'authenticated' : nodeAuthenticated,
   }

   @staticmethod
   def handler( mode, args ):
      dot1xIntfConfig = config.newDot1xIntfConfig( mode.intf.name )
      if dot1xIntfConfig:
         if 'single-host' in args:
            dot1xIntfConfig.hostMode = hostModeStr2Enum[ 'single-host' ]
         elif 'multi-host' and 'authenticated' in args:
            dot1xIntfConfig.hostMode = hostModeStr2Enum[ 'multi-host authenticated' ]
         elif 'multi-host' in args:
            dot1xIntfConfig.hostMode = hostModeStr2Enum[ 'multi-host' ]

   @staticmethod
   def noOrDefaultHandler( mode, args ):
      dot1xIntfConfig = config.dot1xIntfConfig.get( mode.intf.name )
      if dot1xIntfConfig:
         dot1xIntfConfig.hostMode = hostModeStr2Enum[ 'multi-host' ]

EthIntfModelet.addCommandClass( Dot1XHostModeCmd )

#-----------------------------------------------------------------
# [no|default] dot1x eapol disabled 
#-----------------------------------------------------------------
def setEapolDisabled( mode, args ):
   dot1xIntfConfig = makeOrGetDot1xIntfConfig( mode.intf.name )
   dot1xIntfConfig.eapolDisabled = True

def noEapolDisabled( mode, args ):
   dot1xIntfConfig = config.dot1xIntfConfig.get( mode.intf.name )
   if dot1xIntfConfig:
      dot1xIntfConfig.eapolDisabled = dot1xIntfConfig.eapolDisabledDefault

class Dot1xEapolDisabledCmd( CliCommand.CliCommandClass ):
   syntax = 'dot1x eapol disabled'
   noOrDefaultSyntax = syntax
   data = {
      'dot1x' : nodeIntfDot1x, 
      'eapol' : 'Configure Dot1x EAPOL attribute',
      'disabled' : 'Disable EAPOL processing for this interface'
   }
   handler = setEapolDisabled
   noOrDefaultHandler = noEapolDisabled

EthIntfModelet.addCommandClass( Dot1xEapolDisabledCmd )

#-----------------------------------------------------------------
# [no|default] dot1x eapol authentication failure fallback mba  [ timeout <value> ]
#-----------------------------------------------------------------
def setEapolAuthFailFallback( mode, args ):
   dot1xIntfConfig = makeOrGetDot1xIntfConfig( mode.intf.name )
   dot1xIntfConfig.mbaFallback = True
   dot1xIntfConfig.mbaTimeout = args.get( 'TIMEOUT', 
         Dot1xConsts.mbaTimeoutDefault )

def noEapolAuthFailFallback( mode, args ):
   dot1xIntfConfig = config.dot1xIntfConfig.get( mode.intf.name )
   if dot1xIntfConfig:
      dot1xIntfConfig.mbaFallback = False
      dot1xIntfConfig.mbaTimeout = Dot1xConsts.mbaTimeoutDefault

class Dot1XEapolAuthFailFallbackCmd( CliCommand.CliCommandClass ):
   syntax = 'dot1x eapol authentication failure fallback mba [ timeout TIMEOUT ]'
   noOrDefaultSyntax = 'dot1x eapol authentication failure fallback mba ...'
   data = {
      'dot1x' : nodeIntfDot1x, 
      'eapol' : 'Configure Dot1x EAPOL attribute',
      'authentication' : 'Eapol authentication',
      'failure' : 'Set authentication failure action',
      'fallback' : 'Fallback to auth method',
      'mba' : 'Use MAC authentication method for this interface',
      'timeout' : 'Timeout for starting MAC authentication',
      'TIMEOUT' : CliMatcher.IntegerMatcher( 60, 65535,
         helpdesc='Timeout period in seconds' ),
   }
   handler = setEapolAuthFailFallback
   noOrDefaultHandler = noEapolAuthFailFallback

if Dot1xToggle.toggleDot1xEAPOLPriorityEnabled():
   EthIntfModelet.addCommandClass( Dot1XEapolAuthFailFallbackCmd )

#-----------------------------------------------------------------
# [no|default] dot1x mac based authentication [ host-mode common ]
#
# legacy:
# [no|default] dot1x mac authentication bypass
#-----------------------------------------------------------------
def setMacBasedAuth( mode, args, deprecatedCmd=False ):
   dot1xIntfConfig = makeOrGetDot1xIntfConfig( mode.intf.name )
   switchIntfConfig = cliConfig.switchIntfConfig.get( mode.intf.name )
   if switchIntfConfig and not switchIntfConfig.enabled:
      mode.addWarning(
            "MAC based authentication is only supported on switchport" )
   dot1xIntfConfig.mbaEnabled = True
   dot1xIntfConfig.mbaHostMode = 'common' in args

def noMacBasedAuth( mode, args ):
   dot1xIntfConfig = config.dot1xIntfConfig.get( mode.intf.name )
   if dot1xIntfConfig:
      dot1xIntfConfig.mbaHostMode = False
      dot1xIntfConfig.mbaEnabled = False

# XXX Remove the CLI guard once BUG182344 is fixed.
nodeMac = CliCommand.Node( CliMatcher.KeywordMatcher( 'mac',
                  helpdesc='Configure Dot1x MAC attributes' ),
                  guard=dot1xMbaSupportedGuard )

# [no|default] dot1x mac authentication bypass - deprecated

nodeAuth = CliCommand.Node( CliMatcher.KeywordMatcher( 'authentication',
               helpdesc='Configure MAC authentication method for this interface' ),
                  deprecatedByCmd='dot1x mac based authentication' )

class Dot1XMacAuthBypassCmd( CliCommand.CliCommandClass ):
   syntax = 'dot1x mac authentication bypass'
   noOrDefaultSyntax = syntax
   data = {
      'dot1x': nodeIntfDot1x,
      'mac': nodeMac,
      'authentication': nodeAuth,
      'bypass': 'Configure MAC authentication bypass on this interface',
   }
   handler = setMacBasedAuth
   noOrDefaultHandler = noMacBasedAuth

EthIntfModelet.addCommandClass( Dot1XMacAuthBypassCmd )

# [no|default] dot1x mac based authentication
class Dot1XMacBasedAuthCmd( CliCommand.CliCommandClass ):
   syntax = 'dot1x mac based authentication [ host-mode common ]'
   noOrDefaultSyntax = 'dot1x mac based authentication ...'
   data = {
      'dot1x': nodeIntfDot1x,
      'mac': nodeMac,
      'based': 'Configure MAC attributes to be used for this interface',
      'authentication': 'Configure MAC authentication method for this interface',
      'host-mode': 'Include MAC authentication method hosts',
      'common': 'Dot1x host-mode',
   }
   handler = setMacBasedAuth
   noOrDefaultHandler = noMacBasedAuth

EthIntfModelet.addCommandClass( Dot1XMacBasedAuthCmd )

#-------------------------------------------------------------------------------
# [no|default] dot1x authentication failure action traffic [drop|allow vlan <v>]
#-------------------------------------------------------------------------------
def setAuthFailVlan( mode, vlanId ):
   dot1xIntfConfig = config.newDot1xIntfConfig( mode.intf.name )
   dot1xIntfConfig.authFailVlan = Tac.Value( 'Bridging::VlanIdOrNone', vlanId.id )
   dot1xIntfConfig.afDropConfigured = False

def noAuthFailVlan( mode, vlanId=None ):
   dot1xIntfConfig = config.dot1xIntfConfig.get( mode.intf.name )
   if dot1xIntfConfig:
      if not vlanId:
         # Clear afVlan
         dot1xIntfConfig.authFailVlan = Tac.Value( 'Bridging::VlanIdOrNone', 0 )
         dot1xIntfConfig.afDropConfigured = False
      elif vlanId.id and ( vlanId.id == dot1xIntfConfig.authFailVlan ):
         # Clear afVlan if the provided vlanId matches the configured afVlanId
         dot1xIntfConfig.authFailVlan = Tac.Value( 'Bridging::VlanIdOrNone', 0 )
         dot1xIntfConfig.afDropConfigured = False

def dropAuthFail( mode ):
   dot1xIntfConfig = config.newDot1xIntfConfig( mode.intf.name )
   noAuthFailVlan( mode )                    # Clear AFVLAN
   dot1xIntfConfig.afDropConfigured = True   # Set flag to indicate drop-cmd config

def noDropAuthFail( mode ):
   dot1xIntfConfig = config.dot1xIntfConfig.get( mode.intf.name )
   if dot1xIntfConfig:
      dot1xIntfConfig.afDropConfigured = False

class Dot1XAuthFailureActionTrafficCmd( CliCommand.CliCommandClass ):
   syntax = ( 'dot1x authentication failure action traffic '
               '( drop | ( allow vlan VLAN ) )' )
   noOrDefaultSyntax = ( 'dot1x authentication failure action traffic ' 
                           '( drop | ( allow ( vlan | ( vlan VLAN ) ) ) )' )
   data = {
      'dot1x': nodeIntfDot1x,
      'authentication': 'Set dot1x authentication failure action',
      'failure': 'Set dot1x authentication failure action',
      'action': 'Set dot1x authentication failure action',
      'traffic': 'Set dot1x authentication failure action',
      'drop': 'Drop all unauthenticated traffic',
      'allow': 'Allow unauthenticated traffic in authentication failure VLAN',
      'vlan': 'Set authentication failure VLAN',
      'VLAN': vlanIdMatcher,
   }
   
   @staticmethod
   def handler( mode, args ):
      if 'allow' in args:
         setAuthFailVlan( mode, args.get( 'VLAN' ) )
      elif 'drop' in args:
         dropAuthFail( mode )

   @staticmethod
   def noOrDefaultHandler( mode, args ):
      if 'allow' in args:
         noAuthFailVlan( mode, args.get( 'VLAN' ) )
      elif 'drop' in args:
         noDropAuthFail( mode )

EthIntfModelet.addCommandClass( Dot1XAuthFailureActionTrafficCmd )

#-------------------------------------------------------------------------------
# [no|default] eapol unresponsive action traffic allow vlan <v>]
#-------------------------------------------------------------------------------
class Dot1xEapolUnresponsiveActionTrafficCmd( CliCommand.CliCommandClass ):
   _baseSyntax = 'eapol unresponsive action traffic allow'
   syntax = _baseSyntax + ' vlan VLAN'
   noOrDefaultSyntax = _baseSyntax + '[ vlan VLAN ]'
   
   data = {
      'eapol' : 'Configure Dot1x EAPOL attribute',
      'unresponsive': 'Configure EAPOL unresponsive hosts',
      'action': 'Action in case of EAPOL unresponsive hosts',
      'traffic': 'Traffic action in case of EAPOL unresponsive hosts',
      'allow': 'Allow traffic in case of EAPOL unresponsive hosts',
      'vlan': 'Vlan to allow traffic in case of EAPOL unresponsive hosts',
      'VLAN': vlanIdMatcher,
   }

   @staticmethod
   def handler( mode, args ):
      setGuestVlan( mode, args[ 'VLAN' ] )

   @staticmethod
   def noOrDefaultHandler( mode, args ):
      vlan = args.get( 'VLAN' )
      if vlan is None or vlan.id == config.guestVlan:
         setGuestVlan( mode, 0 )

def setGuestVlan( mode, vlan ):
   config.guestVlan = Tac.Value( 'Bridging::VlanIdOrNone',
                                  vlan and vlan.id )

if Dot1xToggle.toggleDot1xGuestVlanEnabled():
   Dot1xMode.addCommandClass( Dot1xEapolUnresponsiveActionTrafficCmd )

#--------------------------------------------------------------------------- 
# [no|default] dot1x unauthorized [ access | native ] vlan membership egress
#--------------------------------------------------------------------------- 
class UnauthorizedVlanMembershipCmd( CliCommand.CliCommandClass ):
   syntax = 'dot1x unauthorized ( access | native ) vlan membership egress'
   noOrDefaultSyntax = 'dot1x unauthorized ( access | native ) vlan membership ...'
   data = {
      'dot1x' : nodeIntfDot1x,
      'unauthorized' : 'Configuration applicable when port is unauthorized ',
      'access' : 'Applied to access VLAN of access port',
      'native' : 'Applied to native VLAN of phone trunk port',
      'vlan' : 'VLAN configuration ',
      'membership' : 'VLAN membership configuration ',
      'egress' : 'Grant VLAN membership in egress direction '
   }

   @staticmethod
   def handler( mode, args ):
      dot1xIntfConfig = makeOrGetDot1xIntfConfig( mode.intf.name ) 
      if 'access' in args:
         dot1xIntfConfig.unauthorizedAccessVlanEgress = True 
      elif 'native' in args:
         dot1xIntfConfig.unauthorizedNativeVlanEgress = True 

   @staticmethod
   def noOrDefaultHandler( mode, args ):
      dot1xIntfConfig = config.dot1xIntfConfig.get( mode.intf.name ) 
      if dot1xIntfConfig:
         if 'access' in args:
            dot1xIntfConfig.unauthorizedAccessVlanEgress = False 
         elif 'native' in args:
            dot1xIntfConfig.unauthorizedNativeVlanEgress = False 

EthIntfModelet.addCommandClass( UnauthorizedVlanMembershipCmd )
#-----------------------------------------------------------------
# [no|default] dot1x guest-vlan
#-----------------------------------------------------------------
#def setGuestVlan( mode, vlan ):
#   dot1xIntfConfig = config.newDot1xIntfConfig( mode.intf.name )
#   dot1xIntfConfig.guestVlan = vlan
#
#def noGuestVlan( mode ):
#   dot1xIntfConfig = config.dot1xIntfConfig.get( mode.intf.name )
#   if dot1xIntfConfig:
#      dot1xIntfConfig.guestVlan = 0
#
#tokenGuestVlan = CliParser.KeywordRule( 'guest-vlan',
#               helpdesc='Configure a Guest VLAN' )
#vlanIdRule = CliParser.RangeRule( 1, 4094, name='vlan', helpdesc='Enter a VLAN id' )
# we don't support guest vlan and restricted now
#EthIntfModelet.addCommand(
#      ( tokenIntfDot1x, tokenGuestVlan, vlanIdRule, setGuestVlan ) )
#EthIntfModelet.addCommand(
#      ( BasicCli.noOrDefault, tokenIntfDot1x, tokenGuestVlan,
#         BasicCli.trailingGarbage, noGuestVlan ) )

#-----------------------------------------------------------------
# [no|default] dot1x auth-fail vlan
#-----------------------------------------------------------------
#def setAuthFailVlan( mode, vlan ):
#   dot1xIntfConfig = config.newDot1xIntfConfig( mode.intf.name )
#   dot1xIntfConfig.authFailVlan = vlan
#
#def noAuthFailVlan( mode ):
#   dot1xIntfConfig = config.dot1xIntfConfig.get( mode.intf.name )
#   if dot1xIntfConfig:
#      dot1xIntfConfig.authFailVlan = 0
#
#tokenAuthFail = CliParser.KeywordRule( 'auth-fail',
#               helpdesc='Configure authentication failure behavior' )
#tokenVlan = CliParser.KeywordRule( 'vlan',
#               helpdesc='Configure an authentication failure VLAN' )
#EthIntfModelet.addCommand(
#      ( tokenIntfDot1x, tokenAuthFail, tokenVlan, vlanIdRule, setAuthFailVlan ) )
#EthIntfModelet.addCommand(
#      ( BasicCli.noOrDefault, tokenIntfDot1x, tokenAuthFail, tokenVlan,
#         BasicCli.trailingGarbage, noAuthFailVlan ) )

#-----------------------------------------------------------------
# [no|default] dot1x auth-fail max-attempts
#-----------------------------------------------------------------
#def setAuthFailMaxAttempt( mode, attempt ):
#   dot1xIntfConfig = config.newDot1xIntfConfig( mode.intf.name )
#   dot1xIntfConfig.authFailMaxAttempt = attempt
#
#def noAuthFailMaxAttempt( mode ):
#   dot1xIntfConfig = config.dot1xIntfConfig.get( mode.intf.name )
#   if dot1xIntfConfig:
#      dot1xIntfConfig.authFailMaxAttempt = dot1xIntfConfig.defaultAuthFailMaxAttempt
#
#tokenMaxAttempt = CliParser.KeywordRule( 'max-attempts',
#               helpdesc='Maximum number of authentication attempts' )
#maxAttempRule = CliParser.RangeRule( 1, 10, name='attempt',
#                     helpdesc='Enter a value between 1 and 10' )
#EthIntfModelet.addCommand(
#      ( tokenIntfDot1x, tokenAuthFail, tokenMaxAttempt,
#         maxAttempRule, setAuthFailMaxAttempt ) )
#EthIntfModelet.addCommand(
#      ( BasicCli.noOrDefault, tokenIntfDot1x, tokenAuthFail, tokenMaxAttempt,
#         BasicCli.trailingGarbage, noAuthFailMaxAttempt ) )

#-----------------------------------------------------------------
# show dot1x interface [details|statistics]
#-----------------------------------------------------------------
def setDot1xIntfInfo( info, dot1xIntfConfig, dot1xIntfStatus ):
   info.portControl = dot1xIntfConfig.portCtrlSetting
   info.hostMode = dot1xIntfConfig.hostMode
   info.quietPeriod = dot1xIntfConfig.quietPeriod
   info.txPeriod = dot1xIntfConfig.txPeriod
   info.reauthTimeoutIgnore = dot1xIntfConfig.reauthTimeoutIgnore
   info.unauthorizedAccessVlanEgress = dot1xIntfConfig.unauthorizedAccessVlanEgress
   info.unauthorizedNativeVlanEgress = dot1xIntfConfig.unauthorizedNativeVlanEgress
   info.mbaEnabled = dot1xIntfConfig.mbaEnabled
   if dot1xIntfStatus:
      info.authFailVlan = dot1xIntfStatus.authFailVlan
      info.eapolDisabled = dot1xIntfStatus.eapolDisabled
      info.mbaHostMode = dot1xIntfStatus.mbaHostModeConfigured
   else:
      info.authFailVlan = dot1xIntfConfig.authFailVlan
      info.eapolDisabled = False
      info.mbaHostMode = dot1xIntfConfig.mbaHostMode
   info.maxReauthReq = dot1xIntfConfig.maxReauthReq
   info.mbaFallback = dot1xIntfConfig.mbaFallback
   info.mbaTimeout = dot1xIntfConfig.mbaTimeout
   #info.authFailMaxAttempt = dot1xIntfConfig.authFailMaxAttempt
   #info.guestVlan = dot1xIntfConfig.guestVlan
   return info

def showIntfRangeInfo( mode, intfList ):
   info = Dot1xModel.Dot1xInterfaceRangeInformation()
   for intfName in intfList.intfNames():
      intf = intfList.type().getCliIntf( mode, intfName )
      intfInfo = showIntfInfo( mode, intf )
      if intfInfo:
         info.interfaces[ intfName ] = intfInfo
   return info

def showIntfInfo( mode, intf ):
   dot1xIntfConfig = config.dot1xIntfConfig.get( intf.name )
   dot1xIntfStatus = status.dot1xIntfStatus.get( intf.name )
   if dot1xIntfConfig and dot1xIntfConfig.dot1xEnabled:
      info = Dot1xModel.Dot1xInterfaceInformation()
      info = setDot1xIntfInfo( info, dot1xIntfConfig, dot1xIntfStatus )
      return info
   else:
      return None

def showIntfRangeDetails( mode, intfList ):
   info = Dot1xModel.Dot1xInterfaceRangeDetails()
   for intfName in intfList.intfNames():
      intf = intfList.type().getCliIntf( mode, intfName )
      intfInfo = showIntfDetails( mode, intf )
      if intfInfo:
         info.interfaces[ intfName ] = intfInfo
   return info

def showIntfDetails( mode, intf ):
   dot1xIntfConfig = config.dot1xIntfConfig.get( intf.name )
   dot1xIntfStatus = status.dot1xIntfStatus.get( intf.name )
   if dot1xIntfConfig and dot1xIntfConfig.dot1xEnabled:
      details = Dot1xModel.Dot1xInterfaceDetails()
      details = setDot1xIntfInfo( details, 
                                  dot1xIntfConfig, dot1xIntfStatus )
      dot1xIntfStatus = status.dot1xIntfStatus.get( intf.name )
      if dot1xIntfStatus:
         details.portAuthorizationState = dot1xIntfStatus.portAuthState
         # Iterate over list of supplicants and display mac of
         # all the authenticated supplicants
         authStagesToShow = { 'successfulAuth', 'webAuthStart' }
         for supplicant in dot1xIntfStatus.supplicant.values():
            if supplicant.authStage in authStagesToShow:
               details.supplicantMacs[ supplicant.mac ] = \
                               supplicant.sessionTimeout
      return details
   else:
      return None

def showIntfRangeStats( mode, intfList ):
   info = Dot1xModel.Dot1xInterfaceRangeStats()
   for intfName in intfList.intfNames():
      intf = intfList.type().getCliIntf( mode, intfName )
      intfStats = showIntfStats( mode, intf )
      if intfStats:
         info.interfaces[ intfName ] = intfStats
   return info

def showIntfStats( mode, intf ):
   dot1xIntfStatus = status.dot1xIntfStatus.get( intf.name )
   if dot1xIntfStatus:
      stats = Dot1xModel.Dot1xInterfaceStats()
      stats.rxStart = dot1xIntfStatus.stats.rxStart
      stats.rxLogoff = dot1xIntfStatus.stats.rxLogoff
      stats.rxRespId = dot1xIntfStatus.stats.rxRespId
      stats.rxResp = dot1xIntfStatus.stats.rxResp
      stats.rxInvalid = dot1xIntfStatus.stats.rxInvalid
      stats.rxTotal = stats.rxStart + stats.rxLogoff + stats.rxRespId \
                        + stats.rxResp + stats.rxInvalid
      stats.txReqId = dot1xIntfStatus.stats.txReqId
      stats.txReq = dot1xIntfStatus.stats.txReq
      stats.txTotal = stats.txReqId + stats.txReq
      stats.rxVersion = dot1xIntfStatus.stats.rxVersion
      stats.lastRxSrcMac = dot1xIntfStatus.stats.lastRxSrcMac
      return stats
   else:
      return None

matcherDetails = CliMatcher.KeywordMatcher( 'details', 
                        helpdesc='Show 802.1X port authentication detail' )
matcherStatistics = CliMatcher.KeywordMatcher( 'statistics', 
                        helpdesc='Show 802.1X port authentication statistics' )
matcherInterface = CliMatcher.KeywordMatcher( 'interface', 
                        helpdesc='Choose an interface' )

# show dot1x interface IPINTF
class Dot1XInterfaceIpintfCmd( ShowCommand.ShowCliCommandClass ):
   syntax = 'show dot1x interface INTFS'
   data = {
      'dot1x': nodeDot1x,
      'interface': matcherInterface,
      'INTFS': Intf.rangeMatcher,
   }
   
   @staticmethod
   def handler( mode, args ):
      return showIntfRangeInfo( mode, args[ 'INTFS' ] )
   cliModel = Dot1xModel.Dot1xInterfaceRangeInformation

BasicCli.addShowCommandClass( Dot1XInterfaceIpintfCmd )

# show dot1x interface IPINTF details
class Dot1XInterfaceIpintfDetailsCmd( ShowCommand.ShowCliCommandClass ):
   syntax = 'show dot1x interface INTFS details'
   data = {
      'dot1x': nodeDot1x,
      'interface': matcherInterface,
      'INTFS': Intf.rangeMatcher,
      'details': matcherDetails,
   }
   
   @staticmethod
   def handler( mode, args ):
      return showIntfRangeDetails( mode, args[ 'INTFS' ] )
   cliModel = Dot1xModel.Dot1xInterfaceRangeDetails

BasicCli.addShowCommandClass( Dot1XInterfaceIpintfDetailsCmd )

# show dot1x interface IPINTF statistics
class Dot1XInterfaceIpintfStatisticsCmd( ShowCommand.ShowCliCommandClass ):
   syntax = 'show dot1x interface INTFS statistics'
   data = {
      'dot1x': nodeDot1x,
      'interface': matcherInterface,
      'INTFS': Intf.rangeMatcher,
      'statistics': matcherStatistics,
   }
   
   @staticmethod
   def handler( mode, args ):
      return showIntfRangeStats( mode, args[ 'INTFS' ] )
   cliModel = Dot1xModel.Dot1xInterfaceRangeStats

BasicCli.addShowCommandClass( Dot1XInterfaceIpintfStatisticsCmd )

#-----------------------------------------------------------------
# show dot1x all [brief|details|statistics]
# legacy:
# show dot1x all [summary|details|statistics]
#-----------------------------------------------------------------
def showAllInfo( mode, args ):
   allInfo = Dot1xModel.Dot1xAllInformation()
   allInfo.systemAuthControl = config.dot1xEnabled
   allInfo.lldpBypass = identityDot1xStatus.lldpBypass
   allInfo.dynAuth = status.dot1xDynAuthEnabled
   allInfo.version = Dot1xConsts.version2004
   for intfName in config.dot1xIntfConfig.keys():
      intf = EthPhyIntf( intfName, mode )
      info = showIntfInfo( mode, intf )
      if info:
         allInfo.interfaces[ intfName ] = info
   return allInfo

def showAllSummary( mode, args ):
   allSummary = Dot1xModel.Dot1xAllSummary()
   allSummary.systemAuthControl = config.dot1xEnabled
   allSummary.lldpBypass = identityDot1xStatus.lldpBypass
   allSummary.dynAuth = status.dot1xDynAuthEnabled
   allSummary.version = Dot1xConsts.version2004
   for intfName in config.dot1xIntfConfig.keys():
      intf = EthPhyIntf( intfName, mode )
      detail = showIntfDetails( mode, intf )
      if detail:
         allSummary.interfaces[ intfName ] = detail
   return allSummary

def showAllDetails( mode, args ):
   allDetails = Dot1xModel.Dot1xAllDetails()
   allDetails.systemAuthControl = config.dot1xEnabled
   allDetails.lldpBypass = identityDot1xStatus.lldpBypass
   allDetails.dynAuth = status.dot1xDynAuthEnabled
   allDetails.version = Dot1xConsts.version2004
   for intfName in config.dot1xIntfConfig.keys():
      intf = EthPhyIntf( intfName, mode )
      detail = showIntfDetails( mode, intf )
      if detail:
         allDetails.interfaces[ intfName ] = detail
   return allDetails

def showAllStats( mode, args ):
   allStats = Dot1xModel.Dot1xInterfaceRangeStats()
   for intfName in status.dot1xIntfStatus.keys():
      intf = EthPhyIntf( intfName, mode )
      allStats.interfaces[ intfName ] = showIntfStats( mode, intf )
   return allStats

matcherAll = CliMatcher.KeywordMatcher( 'all', 
      helpdesc='Show 802.1X port authentication information for all interfaces' )

# show dot1x all
class Dot1XAllCmd( ShowCommand.ShowCliCommandClass ):
   syntax = 'show dot1x all'
   data = {
      'dot1x': nodeDot1x,
      'all': matcherAll,
   }
   handler = showAllInfo
   cliModel = Dot1xModel.Dot1xAllInformation

BasicCli.addShowCommandClass( Dot1XAllCmd )

# show dot1x all brief
class Dot1XAllBriefCmd( ShowCommand.ShowCliCommandClass ):
   syntax = 'show dot1x all brief'
   data = {
      'dot1x': nodeDot1x,
      'all': matcherAll,
      'brief': 'Show 802.1X port authentication brief',
   }
   handler = showAllSummary
   cliModel = Dot1xModel.Dot1xAllSummary

BasicCli.addShowCommandClass( Dot1XAllBriefCmd )

# show dot1x all summary - Deprecated Command
class Dot1XAllSummaryCmd( ShowCommand.ShowCliCommandClass ):
   syntax = 'show dot1x all summary'
   data = {
      'dot1x': nodeDot1x,
      'all': matcherAll,
      'summary': CliCommand.Node( CliMatcher.KeywordMatcher( 'summary', 
                  helpdesc='Show 802.1X port authentication summary' ),
                  deprecatedByCmd='show dot1x all brief' ),
   }
   handler = showAllSummary
   cliModel = Dot1xModel.Dot1xAllSummary

BasicCli.addShowCommandClass( Dot1XAllSummaryCmd )

# show dot1x all details
class Dot1XAllDetailsCmd( ShowCommand.ShowCliCommandClass ):
   syntax = 'show dot1x all details'
   data = {
      'dot1x': nodeDot1x,
      'all': matcherAll,
      'details': matcherDetails,
   }
   handler = showAllDetails
   cliModel = Dot1xModel.Dot1xAllDetails

BasicCli.addShowCommandClass( Dot1XAllDetailsCmd )

# show dot1x all statistics
class Dot1XAllStatisticsCmd( ShowCommand.ShowCliCommandClass ):
   syntax = 'show dot1x all statistics'
   data = {
      'dot1x': nodeDot1x,
      'all': matcherAll,
      'statistics': matcherStatistics,
   }
   handler = showAllStats
   cliModel = Dot1xModel.Dot1xInterfaceRangeStats

BasicCli.addShowCommandClass( Dot1XAllStatisticsCmd )
#-----------------------------------------------------------------
# show dot1x hosts [ <intfName> ]
#-----------------------------------------------------------------
def showHost( supplicantStatus ):
   host = Dot1xModel.Dot1xHost()
   host.fromTacc( supplicantStatus )
   return host

def showHosts( intfName ):
   hosts = Dot1xModel.Dot1xHosts()
   dot1xIntfStatus = status.dot1xIntfStatus[ intfName ]
   for supplicantMac in dot1xIntfStatus.supplicant:
      supplicant = dot1xIntfStatus.supplicant[ supplicantMac ]
      hosts.supplicants[ supplicantMac ] = showHost( supplicant )
   return hosts

def showAllHosts( mode, args ):
   intfList = args.get( 'INTFS' )
   allHosts = Dot1xModel.Dot1xAllHosts()
   if intfList:
      for ifName in intfList.intfNames():
         intf = intfList.type().getCliIntf( mode, ifName )
         if not intf.lookup() or not intf.status().deviceName:
            mode.addError( "%s is not operational" % ifName )
            continue
         if ifName in status.dot1xIntfStatus:
            allHosts.intfSupplicantsDict[ ifName ] = showHosts( ifName )
   else:
      for intfName in status.dot1xIntfStatus.keys():
         allHosts.intfSupplicantsDict[ intfName ] = showHosts( intfName )

   return allHosts

intfRangeMatcher = IntfRangeMatcher( explicitIntfTypes=( EthPhyAutoIntfType, ) )

class Dot1XHostsCmd( ShowCommand.ShowCliCommandClass ):
   syntax = 'show dot1x hosts [ INTFS ]'
   data = {
      'dot1x': nodeDot1x,
      'hosts': 'Show information for all the supplicants',
      'INTFS': intfRangeMatcher,
   }
   handler = showAllHosts
   cliModel = Dot1xModel.Dot1xAllHosts

BasicCli.addShowCommandClass( Dot1XHostsCmd )

#-----------------------------------------------------------------
# show dot1x hosts blocked
#-----------------------------------------------------------------
def showBlockedMacEntry():
   entries = {}
   for macAddr in blockedMacTable.blockedMacEntry:
      entry = Dot1xModel.Dot1xBlockedMacEntry()
      macEntry = blockedMacTable.blockedMacEntry[ macAddr ]
      entry.interfaces.update( macEntry.intfVlan )
      entries[ macAddr ] = entry
   return entries

def showBlockedMacTable( mode, args ):
   macTable = Dot1xModel.Dot1xBlockedMacTable()
   macTable.macAddresses.update( showBlockedMacEntry() )
   return macTable

class Dot1XBlockedMacTableCmd( ShowCommand.ShowCliCommandClass ):
   syntax = 'show dot1x hosts blocked'
   data = {
       'dot1x': nodeDot1x,
       'hosts': 'Show information for all the hosts',
       'blocked': 'Show MAC addresses blocked via dynamic authorization',
   }
   handler = showBlockedMacTable
   cliModel = Dot1xModel.Dot1xBlockedMacTable

BasicCli.addShowCommandClass( Dot1XBlockedMacTableCmd )

#-----------------------------------------------------------------
# show dot1x vlan assignment group [ <groupName> ]
#-----------------------------------------------------------------
def getGroupNames( mode ):
   return config.vlanGroup

matcherGroupName = CliMatcher.DynamicNameMatcher( getGroupNames,
                                                  'Assigned group name' )

def showDot1xVlanAssignmentGroupNames( mode, args ):
   groupName = args.get( 'GROUP' )
   vlanAssignedGroup = Dot1xModel.Dot1xVlanAssignmentGroupNames()
   groups = []
   if groupName:
      if groupName in config.vlanGroup:
         groups = [ groupName ]
   else:
      groups = config.vlanGroup

   for group in groups:
      vlans = config.vlanGroup[ group ].vlanId
      canonicalString = multiRangeToCanonicalString( vlans )
      vlanAssignedGroup.vlanGroupNameDict[ group ] = canonicalString
   return vlanAssignedGroup

class Dot1XVlanAssignmentGroupCmd( ShowCommand.ShowCliCommandClass ):
   syntax = 'show dot1x vlan assignment group [ GROUP ]'
   data = {
      'dot1x': nodeDot1x,
      'vlan': 'VLAN group',
      'assignment' : 'Assigned VLAN to group',
      'group' : 'Group name',
      'GROUP' : matcherGroupName,
   }
   handler = showDot1xVlanAssignmentGroupNames
   cliModel = Dot1xModel.Dot1xVlanAssignmentGroupNames

BasicCli.addShowCommandClass( Dot1XVlanAssignmentGroupCmd )

#-----------------------------------------------------------------
# clear dot1x interface
#-----------------------------------------------------------------
def clearIntfStats( mode, args ):
   intf = args[ 'IPINTF' ]
   dot1xIntfConfigReq = configReq.dot1xIntfConfigReq.get( intf.name )
   if dot1xIntfConfigReq:
      dot1xIntfConfigReq.clearStatsRequestTime = Tac.now()

matcherStatistics = CliMatcher.KeywordMatcher( 'statistics', 
               helpdesc='Clear 802.1X port authentication statistics' )
matcherHost = CliMatcher.KeywordMatcher( 'host',
               helpdesc='Clear host commands' )
matcherBlocked = CliMatcher.KeywordMatcher( 'blocked',
               helpdesc='Clear blocked hosts' )
matcherHostInterface = CliMatcher.KeywordMatcher( 'interface',
               helpdesc='Clear hosts on interface' )
matcherHostMac = CliMatcher.KeywordMatcher( 'mac',
               helpdesc='Clear hosts based on mac' )

tokenClearNode = CliToken.Clear.clearKwNode

class ClearDot1XStatisticsInterfaceIpintfCmd( CliCommand.CliCommandClass ):
   syntax = 'clear dot1x statistics interface IPINTF'
   data = {
      'clear': tokenClearNode,
      'dot1x': nodeDot1x,
      'statistics': matcherStatistics,
      'interface': 'Choose an interface',
      'IPINTF': Intf.matcherWithIpSupport,
   }
   handler = clearIntfStats

BasicCliModes.EnableMode.addCommandClass( ClearDot1XStatisticsInterfaceIpintfCmd )

#-----------------------------------------------------------------
# clear dot1x all
#-----------------------------------------------------------------
def clearAllStats( mode, args ):
   for dot1xIntfConfigReq in configReq.dot1xIntfConfigReq.values():
      dot1xIntfConfigReq.clearStatsRequestTime = Tac.now()

class ClearDot1XStatisticsAllCmd( CliCommand.CliCommandClass ):
   syntax = 'clear dot1x statistics all'
   data = {
      'clear': tokenClearNode,
      'dot1x': nodeDot1x,
      'statistics': matcherStatistics,
      'all': 'Clear 802.1X port authentication information for all interfaces',
   }
   handler = clearAllStats

BasicCliModes.EnableMode.addCommandClass( ClearDot1XStatisticsAllCmd )

def dot1xUnauthorizedExplanation( intfName, mode ):
   dot1xIntfStatus = status.dot1xIntfStatus.get( intfName )
   if dot1xIntfStatus and dot1xIntfStatus.portAuthState == 'blocked':
      return ( 'N/A', 'Unauthorized interface', 'dot1x configuration' )
   return ( None, None, None )

IntfCli.unauthorizedExplanationHook.addExtension( dot1xUnauthorizedExplanation )

def dot1xCheckStaticMac( mode, macAddr, vlanId, intfsOrDrop=None ):
   hostEntry = dot1xInputHostTable.hostEntry.get( macAddr )
   if hostEntry:
      hostEntry = mergedHostTable.hostEntry.get( macAddr )
      if hostEntry and vlanId.id == hostEntry.vlanId:
         mode.addError( 'MAC address already in use by dot1x' )
         return False
   return True

bridgingCheckStaticMacHook.addExtension( dot1xCheckStaticMac )

#-------------------------------------------------------------------------------
# Cleanup per-interface configuration
#-------------------------------------------------------------------------------
class IntfDot1xConfigCleaner( IntfCli.IntfDependentBase ):
   """This class is responsible for removing per-interface Dot1x config
   when the interface is deleted."""
   def setDefault( self ):
      del config.dot1xIntfConfig[ self.intf_.name ]

#-------------------------------------------------------------------------------
# clear dot1x host mac <mac-address>
#-------------------------------------------------------------------------------
def clearHost( mode, args ):
   hostMac = args.get( 'MAC' )
   configReq.hostToBeCleared = hostMac
   configReq.hostToBeClearedTime = Tac.now()

class ClearDot1XHostMac( CliCommand.CliCommandClass ):
   syntax = 'clear dot1x host mac MAC'
   data = {
      'clear': tokenClearNode,
      'dot1x': nodeDot1x,
      'host': matcherHost,
      'mac': matcherHostMac,
      'MAC' : MacAddr.macAddrMatcher,
   }
   handler = clearHost

BasicCliModes.EnableMode.addCommandClass( ClearDot1XHostMac )

#-------------------------------------------------------------------------------
# clear dot1x host all
#-------------------------------------------------------------------------------
def clearAllHost( mode, args ):
   configReq.allHostClearTime = Tac.now()

class ClearDot1XAllHost( CliCommand.CliCommandClass ):
   syntax = 'clear dot1x host all'
   data = {
      'clear': tokenClearNode,
      'dot1x': nodeDot1x,
      'host': matcherHost,
      'all': matcherAll,
   }
   handler = clearAllHost

BasicCliModes.EnableMode.addCommandClass( ClearDot1XAllHost )

#-------------------------------------------------------------------------------
# clear dot1x host interface <interface-name>
#-------------------------------------------------------------------------------
def clearInterfaceHost( mode, args ):
   intf = args.get( 'ETHINTF' )
   intfId = IntfId( intf.name )
   configReq.intfToBeCleared = intfId
   configReq.intfToBeClearedTime = Tac.now()

class ClearDot1XInterfaceHost( CliCommand.CliCommandClass ):
   syntax = 'clear dot1x host interface ETHINTF'
   data = {
      'clear': tokenClearNode,
      'dot1x': nodeDot1x,
      'host': matcherHost,
      'interface': matcherHostInterface,
      'ETHINTF': EthPhyIntf.ethMatcher,
   }
   handler = clearInterfaceHost

BasicCliModes.EnableMode.addCommandClass( ClearDot1XInterfaceHost )

#-------------------------------------------------------------------------------
# clear dot1x host blocked mac <mac-address>
#-------------------------------------------------------------------------------
def clearHostBlocked( mode, args ):
   hostMac = args.get( 'MAC' )
   configReq.hostToBeCleared = hostMac
   configReq.hostBlockedToBeClearedTime = Tac.now()

class ClearDot1XHostBlockedMac( CliCommand.CliCommandClass ):
   syntax = 'clear dot1x host blocked mac MAC'
   data = {
      'clear': tokenClearNode,
      'dot1x': nodeDot1x,
      'host': matcherHost,
      'blocked': matcherBlocked,
      'mac': matcherHostMac,
      'MAC': MacAddr.macAddrMatcher,
   }
   handler = clearHostBlocked

BasicCliModes.EnableMode.addCommandClass( ClearDot1XHostBlockedMac )

#-------------------------------------------------------------------------------
# clear dot1x host blocked interface <interface-name>
#-------------------------------------------------------------------------------
def clearInterfaceHostBlocked( mode, args ):
   intf = args.get( 'ETHINTF' )
   intfId = IntfId( intf.name )
   configReq.intfToBeCleared = intfId
   configReq.intfHostBlockedToBeClearedTime = Tac.now()

class ClearDot1XHostBlockedInterface( CliCommand.CliCommandClass ):
   syntax = 'clear dot1x host blocked interface ETHINTF'
   data = {
      'clear': tokenClearNode,
      'dot1x': nodeDot1x,
      'host': matcherHost,
      'blocked': matcherBlocked,
      'interface': matcherHostInterface,
      'ETHINTF': EthPhyIntf.ethMatcher,
   }
   handler = clearInterfaceHostBlocked

BasicCliModes.EnableMode.addCommandClass( ClearDot1XHostBlockedInterface )

#-------------------------------------------------------------------------------
# clear dot1x host blocked all
#-------------------------------------------------------------------------------
def clearAllHostBlocked( mode, args ):
   configReq.allHostBlockedClearTime = Tac.now()

class ClearDot1XHostBlockedAll( CliCommand.CliCommandClass ):
   syntax = 'clear dot1x host blocked all'
   data = {
      'clear': tokenClearNode,
      'dot1x': nodeDot1x,
      'host': matcherHost,
      'blocked': matcherBlocked,
      'all': matcherAll,
   }
   handler = clearAllHostBlocked

BasicCliModes.EnableMode.addCommandClass( ClearDot1XHostBlockedAll )

#-------------------------------------------------------------------------------
# Register to show tech-support
#-------------------------------------------------------------------------------
def _showTechDot1xCmds():
   cmds = []
   if status and status.dot1xEnabled:
      cmds += [ "show dot1x hosts",
                "show dot1x all details",
                "show dot1x all statistics",
                "show dot1x hosts blocked" ]
   return cmds
   
TechSupportCli.registerShowTechSupportCmdCallback(
   '2019-07-17 13:48:02', _showTechDot1xCmds )

#-------------------------------------------------------------------------------
# Have the Cli Agent mount all needed state from sysdb
#-------------------------------------------------------------------------------
def Plugin( entityManager ):
   global config, configReq, status, identityDot1xStatus, dot1xInputHostTable
   global hwstatus, cliConfig, subIntfCapabilityDir, mergedHostTable
   global ipStatus, blockedMacTable

   config = ConfigMount.mount( entityManager, "dot1x/config",
                             "Dot1x::Config", "w" )
   configReq = LazyMount.mount( entityManager, "dot1x/configReq",
                             "Dot1x::ConfigReq", "w" )
   status = LazyMount.mount( entityManager, "dot1x/status",
                             "Dot1x::Status", "r" )
   identityDot1xStatus = LazyMount.mount( entityManager, "identity/dot1x/status",
                                          "Identity::Dot1x::Status", "r" )
   dot1xInputHostTable = LazyMount.mount( entityManager,
                                          "bridging/input/hostTable/Dot1x",
                                          "Dot1x::HostTable", "r" )
   mergedHostTable = LazyMount.mount( entityManager, "dot1x/hostTable",
                                      "Dot1x::HostTable", "r" )
   hwstatus = LazyMount.mount( entityManager, "dot1x/hwstatus",
                             "Dot1x::HwStatus", "r" )
   cliConfig = LazyMount.mount( entityManager, "bridging/input/config/cli",
                                "Bridging::Input::CliConfig", "r" )
   subIntfCapabilityDir = LazyMount.mount( entityManager, 
                                     'hardware/status/dot1x/slice',
                                     'Tac::Dir', 'ri' )
   ipStatus = LazyMount.mount( entityManager, "ip/status",
                               "Ip::Status", "r" )
   blockedMacTable = LazyMount.mount( entityManager, "dot1x/blockedMacTable",
                                      "Dot1x::BlockedMacTable", "r" )
   # register interface-deletion handler
   IntfCli.Intf.registerDependentClass( IntfDot1xConfigCleaner, priority=10 )

