# Copyright (c) 2013 Arista Networks, Inc.  All rights reserved.
# Arista Networks, Inc. Confidential and Proprietary.

import Tac
import CliParser, LazyMount
import ConfigMount
import IntfCli
import CliPlugin.EthIntfCli as EthIntfCli
import CliPlugin.LagIntfCli as LagIntfCli
import CliCommand
import CliMatcher
from CliToken.Dhcp import dhcpMatcherForConfig
from CliToken.Ip import ipMatcherForConfigIf

# Globals written by the Plugin function at the end of this file
dhcpSnoopingConfig = None
dhcpSnoopingCounterConfig = None
dhcpSnoopingStatus = None
dhcpSnoopingHwStatusDir = None
platformHardwareSliceDir = None
bridgingConfig = None

def dhcpSnoopingSupportedGuard( mode, token ):
   for sliceId, hwStatus in dhcpSnoopingHwStatusDir.iteritems():
      sliceInfo = platformHardwareSliceDir.sliceInfo.get( sliceId )
      if sliceInfo and sliceInfo.generationId == hwStatus.genId and \
         hwStatus.dhcpSnoopingSupported:
         return None
   return CliParser.guardNotThisPlatform

class DhcpSnoopingIntfModelet( CliParser.Modelet ):
   modeletParseTree = CliParser.ModeletParseTree()
   def __init__( self, mode ):
      CliParser.Modelet.__init__( self )

   @staticmethod
   def shouldAddModeletRule( mode ):
      return isinstance( mode.intf, LagIntfCli.EthLagIntf ) or \
             ( isinstance( mode.intf, EthIntfCli.EthPhyIntf ) and \
               mode.intf.name.startswith( "Et" ) )

IntfCli.IntfConfigMode.addModelet( DhcpSnoopingIntfModelet )

intfModelet = DhcpSnoopingIntfModelet

#-------------------------------------------------------------------------------
# The DhcpSnoopingIntf class is used to remove any interface specific configs
#-------------------------------------------------------------------------------
class DhcpSnoopingIntf( IntfCli.IntfDependentBase ):
   def setDefault( self ):
      if self.intf_.name in dhcpSnoopingConfig.userDefinedCircuitId:
         del dhcpSnoopingConfig.userDefinedCircuitId[ self.intf_.name ]

#-------------------------------------------------------------------------------
# "[no|default] ip dhcp snooping interface information option circuit-id 
# type <circuit-id type> value <circuit-id value>" interface command.
#-------------------------------------------------------------------------------
def setInfoCircuitIdIntf( mode, args ):
   circuitIdType = args.get( 'ID', 'none' ) 
   newId = args[ 'NEWID' ]
   # circuitIdVal is unique across interface
   for intfName in dhcpSnoopingConfig.userDefinedCircuitId:
      if dhcpSnoopingConfig.userDefinedCircuitId[ intfName ].circuitIdVal == newId \
         and intfName != mode.intf.name:
         mode.addError( "Circuit ID %s is already set on another interface: %s" %
               ( newId, intfName ) )
         return

   intfName = mode.intf.name
   if intfName not in dhcpSnoopingConfig.userDefinedCircuitId:
      dhcpSnoopingConfig.userDefinedCircuitId.newMember( intfName )
   if circuitIdType == 'none':
      dhcpSnoopingConfig.userDefinedCircuitId[ intfName ].circuitIdTypeValid = False
   else:
      dhcpSnoopingConfig.userDefinedCircuitId[ intfName ].circuitIdTypeValid = True
      dhcpSnoopingConfig.userDefinedCircuitId[ intfName ].circuitIdType = \
                                                               int( circuitIdType )
   dhcpSnoopingConfig.userDefinedCircuitId[ intfName ].circuitIdVal = newId

def resetInfoCircuitIdIntf( mode, args ):
   intfName = mode.intf.name
   if intfName in dhcpSnoopingConfig.userDefinedCircuitId:
      del dhcpSnoopingConfig.userDefinedCircuitId[ intfName ]

class IpDhcpSnoopingInterfaceInformationNewidCmd( CliCommand.CliCommandClass ):
   syntax = ( 'ip dhcp snooping interface information option circuit-id type'
              '( none | ID ) value NEWID' )
   noOrDefaultSyntax = 'ip dhcp snooping interface information option circuit-id ...'
   data = {
      'ip' : ipMatcherForConfigIf,
      'dhcp' : dhcpMatcherForConfig,
      'snooping' : CliCommand.guardedKeyword( 'snooping',
         helpdesc='DHCP Snooping configuration',
         guard=dhcpSnoopingSupportedGuard ),
      'interface' : 'Dhcp snooping interface commands',
      'information' : 'Option-82 information',
      'option' : 'Configure Information Option',
      'circuit-id' : 'Circuit ID suboption in Option-82',
      'type' : 'Type encoded in the Circuit ID sub-option',
      'none' : 'Do not include type in the Circuit ID sub-option',
      'ID' : CliMatcher.IntegerMatcher( 0, 255,
         helpdesc='Specify the type in the Circuit ID sub-option' ),
      'value' : 'User defined value Circuit ID subopton in Option-82',
      'NEWID' : CliMatcher.PatternMatcher( pattern='[0-9a-zA-Z._:/\\-]{1,15}',
         helpdesc='Up to 15 alphanumeric characters describing this interface',
         helpname='WORD' ),
   }

   handler = setInfoCircuitIdIntf
   noOrDefaultHandler = resetInfoCircuitIdIntf

DhcpSnoopingIntfModelet.addCommandClass( IpDhcpSnoopingInterfaceInformationNewidCmd )

#-------------------------------------------------------------------------------
# Have the Cli Agent mount all needed state from sysdb
#-------------------------------------------------------------------------------
def Plugin( entityManager ):
   global dhcpSnoopingConfig, dhcpSnoopingStatus,  dhcpSnoopingHwStatusDir
   global bridgingConfig, dhcpSnoopingCounterConfig, platformHardwareSliceDir

   dhcpSnoopingConfig = ConfigMount.mount( entityManager,
                                           "bridging/dhcpsnooping/config",
                                           "Bridging::DhcpSnooping::Config", "w" )
   dhcpSnoopingCounterConfig = LazyMount.mount(
         entityManager, "bridging/dhcpsnooping/counterConfig",
         "Bridging::DhcpSnooping::CounterConfig", "w" )
   dhcpSnoopingStatus = LazyMount.mount( entityManager,
                                         "bridging/dhcpsnooping/status",
                                         "Bridging::DhcpSnooping::Status", "r" )
   dhcpSnoopingHwStatusDir = LazyMount.mount( entityManager,
                                             "bridging/dhcpsnooping/hardware/status",
                                             "Tac::Dir", "ri" )
   bridgingConfig = LazyMount.mount( entityManager, "bridging/config",
                                     "Bridging::Config", "r" )
   platformHardwareSliceDir = LazyMount.mount( entityManager,
                                               "platform/hardware/slice",
                                               "Hardware::PlatformSliceDir", "r" )
   IntfCli.Intf.registerDependentClass( DhcpSnoopingIntf )
