#!/usr/bin/env python
# Copyright (c) 2019 Arista Networks, Inc.  All rights reserved.
# Arista Networks, Inc. Confidential and Proprietary.

# pkgdeps: library DhcpServer

from __future__ import absolute_import, division, print_function
import Arnet
import BasicCli
import CliMatcher
import CliPlugin.IpAddrMatcher as IpAddrMatcher
import CliPlugin.Ip6AddrMatcher as Ip6AddrMatcher
from CliPlugin.DhcpServerCliModels import DhcpServerIpv4VendorOptionModel
from CliPlugin.DhcpServerCliModels import DhcpServerSubOptionModel
from CliPlugin.DhcpServerCliModels import DhcpServerIpv4SubnetModel
from CliPlugin.DhcpServerCliModels import DhcpServerIpv6SubnetModel
from CliPlugin.DhcpServerCliModels import DhcpServerIpv4Range
from CliPlugin.DhcpServerCliModels import DhcpServerIpv6Range
from CliPlugin.DhcpServerCliModels import DhcpServerShowLeasesModel
from CliPlugin.DhcpServerCliModels import DhcpServerShowVrfLeasesModel
from CliPlugin.DhcpServerCliModels import DhcpServerIpv4ShowModel
from CliPlugin.DhcpServerCliModels import DhcpServerIpv6ShowModel
from CliPlugin.DhcpServerCliModels import DhcpServerShowModel
from CliPlugin.DhcpServerCliModels import DhcpServerShowVrfModel
from CliPlugin.DhcpServerCliModels import DhcpServerInterfaceStatusShowModel
from CliPlugin.DhcpServerCliModels import DhcpServerIpv4ReservationsMacAddressModel
from CliPlugin.DhcpServerCliModels import Ipv4Leases, Ipv6Leases
from collections import namedtuple
from EosDhcpServerLib import unknownSubnets
from EosDhcpServerLib import featureOption43Status
from EosDhcpServerLib import KeaDhcpLeaseData
from EosDhcpServerLib import getSubOptionData
from EosDhcpServerLib import vendorSubOptionType
from EosDhcpServerLib import featureStaticIPv4AddressStatus
from IpLibConsts import DEFAULT_VRF
import LazyMount
import ShowCommand
import Tac

# Global Variables
dhcpServerConfig = None
dhcpServerStatus = None
ipAddrZero = Tac.Value( "Arnet::IpAddr" ).ipAddrZero

ConfigData = namedtuple( 'ConfigData', [
   'leaseInfo',
   'serverDisabledDefault',
   'serverDisabled',
   'interfaceDisabled',
   'interfaces',
   'leaseTime',
   'dnsServers',
   'dnsName',
   'tftpServerOption66',
   'tftpServerOption150',
   'tftpBootFile',
   'vendorOption' ] )

afIpv4Help = "Details related to Ipv4"
afIpv6Help = "Details related to Ipv6"
addressFamilyMatcher = CliMatcher.EnumMatcher( {
   'ipv4': afIpv4Help,
   'ipv6': afIpv6Help
} )

def _genPrefix( af, subnet ):
   genPrefix = Arnet.IpGenPrefix( str( subnet ) )
   return genPrefix

def _getReservationsMacAddr( subnetData ):
   reservationsMacAddr = {}
   for macAddr, reservationMacAddr in subnetData.reservationsMacAddr.iteritems():
      reservationModel = DhcpServerIpv4ReservationsMacAddressModel()
      reservationModel.macAddress = macAddr
      if reservationMacAddr.ipAddr != "0.0.0.0":
         reservationModel.ipv4Address = reservationMacAddr.ipAddr
      reservationsMacAddr[ macAddr ] = reservationModel

   return reservationsMacAddr or None

def _getAllOverlappingSubnet( subnetOverlapped ):
   """
   Get all the overlapping subnets for the given subnet

   @Input
      subnetOverlapped (DhcpServer::SubnetOverlapped): subnet to be inspected

   @Output
      allOverlappingSubnets list(Arnet::IpGenPrefix): list of all overlapping subnets
      or
      None
   """

   if not subnetOverlapped:
      return None

   allOverlappingSubnets = subnetOverlapped.overlappingSubnet.keys()
   return allOverlappingSubnets

def _getSubnet( af, leaseInfo ):
   subnets = []
   ipv4 = af == 'ipv4'
   subnetConfig = dhcpServerConfig.subnetConfigIpv4 if ipv4 else \
                  dhcpServerConfig.subnetConfigIpv6
   # Because the sysdb config and kea config can differ for some (small) amount
   # of time, need to union the subnets
   allSubnets = set( subnetConfig.keys() )
   allSubnets = allSubnets | set( leaseInfo.subnetData.keys() )
   subnetModelFunc = DhcpServerIpv4SubnetModel if ipv4 else \
                     DhcpServerIpv6SubnetModel
   for subnet in sorted( allSubnets ):
      subnetGenPref = _genPrefix( af, subnet )
      subnetModel = subnetModelFunc()
      subnetModel.subnet = Tac.const( subnetGenPref )
      subnetData = subnetConfig.get( subnet )
      subnetModel.name = subnetData.subnetName if subnetData else ''
      if subnetData:
         subnetModel.dnsServers = subnetData.dnsServers.values()
         if not subnetModel.dnsServers:
            # use the DNS servers from global config instead
            dnsServers = dhcpServerConfig.dnsServersIpv4.values() if ipv4 else \
                         dhcpServerConfig.dnsServersIpv6.values()
            subnetModel.dnsServers = dnsServers

         if subnetData.leaseTime:
            subnetModel.leaseDuration = subnetData.leaseTime

         if ipv4:
            subnetModel.defaultGateway = subnetData.defaultGateway
            disabledMsg = dhcpServerStatus.ipv4SubnetsDisabled.get( subnet )
            subnetOverlapped = dhcpServerStatus.subnetOverlapped.get( subnetGenPref )

            if subnetData.tftpServerOption66:
               subnetModel.tftpServerOption66 = subnetData.tftpServerOption66

            if subnetData.tftpServerOption150:
               subnetModel.tftpServerOption150 = \
                                       subnetData.tftpServerOption150.values()
            else:
               # tftpServerOption150 is an empty list by default.
               # Do this so that it won't show up in CAPI model if not set.
               subnetModel.tftpServerOption150 = None

            if subnetData.tftpBootFileName:
               subnetModel.tftpBootFile = subnetData.tftpBootFileName

            # reservations mac-address
            if featureStaticIPv4AddressStatus():
               reservationsMacAddr = _getReservationsMacAddr( subnetData )
               subnetModel.reservationsMacAddress = reservationsMacAddr
            else:
               # Do this so that it won't show in the CAPI model
               subnetModel.reservationsMacAddress = None

         # IPv6
         else:
            disabledMsg = dhcpServerStatus.ipv6SubnetsDisabled.get( subnet )
            subnetOverlapped = (
                           dhcpServerStatus.subnet6Overlapped.get( subnetGenPref ) )

            # Currently, there is no message if direct is active
            directActiveDetail = dhcpServerStatus.subnetBroadcastStatus.get( subnet )
            if not directActiveDetail:
               subnetModel.directActive = True
            else:
               subnetModel.directActive = False
               subnetModel.directActiveDetail = directActiveDetail
            # If the subnet is active, then we will always reply to relayed requests
            subnetModel.relayActive = True

         # IPv4 and IPv6 shared attributes
         if disabledMsg:
            subnetModel.disabledMessage = disabledMsg

         # overlapping subnets
         subnetModel.overlappingSubnets = (
                                _getAllOverlappingSubnet( subnetOverlapped ) )

         ranges = []
         rangeModelFunc = DhcpServerIpv4Range if ipv4 else \
                          DhcpServerIpv6Range
         for r in subnetData.ranges:
            rangeModel = rangeModelFunc()
            rangeModel.startRangeAddress = r.start
            rangeModel.endRangeAddress = r.end
            ranges.append( rangeModel )
         subnetModel.ranges = ranges

      # Unknown subnets
      elif ipv4:
         subnetModel.defaultGateway = ipAddrZero
         # tftpServerOption150 is an empty list by default.
         # Do this so that it won't show up in CAPI model if not set.
         subnetModel.tftpServerOption150 = None
         subnetModel.reservationsMacAddress = None
         subnetModel.overlappingSubnets = None
      else:
         subnetModel.directActive = False
         subnetModel.relayActive = False
         subnetModel.overlappingSubnets = None

      if subnet == unknownSubnets[ int( af[ -1 ] ) ]:
         subnetModel.disabledMessage = 'Unknown'

      subnetLeaseData = leaseInfo.subnet( subnet )
      subnetModel.activeLeases = 0
      if subnetLeaseData:
         subnetModel.activeLeases = subnetLeaseData.numActiveLeases()

      subnets.append( subnetModel )
   return subnets

def _getVendorOption():
   vendorOptions = {}
   vendorOption = dhcpServerConfig.vendorOptionIpv4
   for vendorId in vendorOption.keys():
      subOptions = vendorOption[ vendorId ].subOptionConfig
      subOptionModels = []
      for subOption in subOptions.values():
         subOptionModel = DhcpServerSubOptionModel()
         subOptionModel.optionCode = subOption.code
         subOptionModel.optionType = subOption.type
         data = getSubOptionData( subOption, raw=True )
         if subOption.type == vendorSubOptionType.string:
            # remove double quotes, so that the model matches the TACC object
            subOptionModel.optionDataString = data[ 1 : -1 ]
            # dataIpAddress is an empty list by default.
            # Do this so that it won't show up in CAPI model if not set
            subOptionModel.optionDataIpAddresses = None
         elif subOption.type == vendorSubOptionType.ipAddress:
            subOptionModel.optionDataIpAddresses = data
         else:
            assert False, 'Unknown sub-option type'
         subOptionModels.append( subOptionModel )

      vendorOptionModel = DhcpServerIpv4VendorOptionModel()
      vendorOptionModel.subOptions = subOptionModels
      vendorOptions[ vendorId ] = vendorOptionModel

   return vendorOptions

def getIpv4ConfigData():
   leaseInfo = KeaDhcpLeaseData( dhcpServerStatus.ipv4IdToSubnet )
   serverDisabledDefault = dhcpServerStatus.ipv4ServerDisabledDefault
   serverDisabled = dhcpServerStatus.ipv4ServerDisabled
   interfaceDisabled = dhcpServerStatus.interfaceIpv4Disabled
   interfaces = dhcpServerConfig.interfacesIpv4
   leaseTime = dhcpServerConfig.leaseTimeIpv4
   dnsServers = dhcpServerConfig.dnsServersIpv4
   dnsName = dhcpServerConfig.domainNameIpv4
   tftpServerOption66 = dhcpServerConfig.tftpServerOption66Ipv4
   tftpServerOption150 = dhcpServerConfig.tftpServerOption150Ipv4.values()
   tftpBootFile = dhcpServerConfig.tftpBootFileNameIpv4
   vendorOption = dhcpServerConfig.vendorOptionIpv4

   return ConfigData( leaseInfo, serverDisabledDefault, serverDisabled,
                      interfaceDisabled, interfaces, leaseTime, dnsServers,
                      dnsName, tftpServerOption66, tftpServerOption150,
                      tftpBootFile, vendorOption )

def getIpv6ConfigData():
   leaseInfo = KeaDhcpLeaseData( dhcpServerStatus.ipv6IdToSubnet, ipVersion=6 )
   serverDisabledDefault = dhcpServerStatus.ipv6ServerDisabledDefault
   serverDisabled = dhcpServerStatus.ipv6ServerDisabled
   interfaceDisabled = dhcpServerStatus.interfaceIpv6Disabled
   interfaces = dhcpServerConfig.interfacesIpv6
   leaseTime = dhcpServerConfig.leaseTimeIpv6
   dnsServers = dhcpServerConfig.dnsServersIpv6
   dnsName = dhcpServerConfig.domainNameIpv6
   tftpServerOption66 = None
   tftpServerOption150 = None
   tftpBootFile = None
   vendorOption = None

   return ConfigData( leaseInfo, serverDisabledDefault, serverDisabled,
                      interfaceDisabled, interfaces, leaseTime, dnsServers,
                      dnsName, tftpServerOption66, tftpServerOption150,
                      tftpBootFile, vendorOption )

def _afModel( af ):
   ipv4 = af == 'ipv4'

   # Determine the correct data for corresponding address family
   if ipv4:
      model = DhcpServerIpv4ShowModel()
      data = getIpv4ConfigData()

      statusDefault = data.serverDisabled == data.serverDisabledDefault
      model.ipv4ServerActive = not data.serverDisabled
      if not model.ipv4ServerActive and not statusDefault:
         model.ipv4DisabledMessage = data.serverDisabled
      if dhcpServerConfig.debugLogPath:
         model.debugLog = True
      model.ipv4ActiveLeases = data.leaseInfo.totalActiveLeases()
      model.ipv4DnsServers = data.dnsServers.values()
      model.ipv4DomainName = data.dnsName
      model.ipv4LeaseDuration = data.leaseTime

      if data.tftpServerOption66:
         model.ipv4TftpServerOption66 = data.tftpServerOption66

      # ipv4TftpServerOption150 is an empty list by default.
      # Do this so that it won't show up in CAPI model if not set.
      model.ipv4TftpServerOption150 = data.tftpServerOption150 or None

      if data.tftpBootFile:
         model.ipv4TftpBootFile = data.tftpBootFile

      for intf in data.interfaces:
         intfModel = DhcpServerInterfaceStatusShowModel()
         dhcpIntfStatus = data.interfaceDisabled.get( intf )
         if not dhcpIntfStatus:
            intfModel.active = True
         else:
            intfModel.active = False
            intfModel.detail = dhcpIntfStatus
         model.ipv4Interfaces[ intf ] = intfModel

      # Vendor Option
      if featureOption43Status():
         model.ipv4VendorOptions = _getVendorOption() or None
      else:
         # Do this so that it won't show in the CAPI model
         model.ipv4VendorOptions = None
      # subnet
      subnets = _getSubnet( af, data.leaseInfo )
      model.ipv4Subnets = { subnet.subnet: subnet for subnet in subnets }
   else:
      model = DhcpServerIpv6ShowModel()
      data = getIpv6ConfigData()

      statusDefault = data.serverDisabled == data.serverDisabledDefault
      model.ipv6ServerActive = not data.serverDisabled
      if not model.ipv6ServerActive and not statusDefault:
         model.ipv6DisabledMessage = data.serverDisabled
      if dhcpServerConfig.debugLogPath:
         model.debugLog = True
      model.ipv6ActiveLeases = data.leaseInfo.totalActiveLeases()
      model.ipv6DnsServers = data.dnsServers.values()
      model.ipv6DomainName = data.dnsName
      model.ipv6LeaseDuration = data.leaseTime

      for intf in data.interfaces:
         intfModel = DhcpServerInterfaceStatusShowModel()
         dhcpIntfStatus = data.interfaceDisabled.get( intf )
         if not dhcpIntfStatus:
            intfModel.active = True
         else:
            intfModel.active = False
            intfModel.detail = dhcpIntfStatus
         model.ipv6Interfaces[ intf ] = intfModel

      # subnet
      subnets = _getSubnet( af, data.leaseInfo )
      model.ipv6Subnets = { subnet.subnet: subnet for subnet in subnets }

   return model

# show commands
class DhcpServerShow( ShowCommand.ShowCliCommandClass ):
   syntax = '''show dhcp server [AF]'''
   data = {
      'dhcp': 'Show DHCP server and client',
      'server': 'Show DHCP server',
      'AF': addressFamilyMatcher,
   }
   cliModel = DhcpServerShowVrfModel

   @staticmethod
   def handler( mode, args ):
      af = args.pop( 'AF', None )

      model = DhcpServerShowModel()
      if af == 'ipv4':
         model.ipv4Server = _afModel( af )
         model.ipv6Server = DhcpServerIpv6ShowModel()
      elif af == 'ipv6':
         model.ipv4Server = DhcpServerIpv4ShowModel()
         model.ipv6Server = _afModel( af )
      else:
         model.ipv4Server = _afModel( 'ipv4' )
         model.ipv6Server = _afModel( 'ipv6' )
      model.addressFamily = af

      # Only supported in default vrf for now
      vrfModel = DhcpServerShowVrfModel()
      vrfModel.vrfs[ DEFAULT_VRF ] = model
      return vrfModel

class DhcpServerShowLeases( ShowCommand.ShowCliCommandClass ):
   syntax = '''show dhcp server
               ( ( ipv4 leases [ NAME | SUBNET ] )
               | ( ipv6 leases [ NAME | SUBNET6 ] )
               | ( leases [ NAME | SUBNET | SUBNET6 ] ) )
            '''

   data = {
      'dhcp': 'Show DHCP server and client',
      'server': 'Show DHCP server',
      'ipv4': afIpv4Help,
      'ipv6': afIpv6Help,
      'leases': 'Show active leases',
      'NAME': CliMatcher.PatternMatcher( pattern='[a-zA-z0-9_-]+', helpname='NAME',
                                         helpdesc='Subnet name' ),
      'SUBNET': IpAddrMatcher.IpPrefixMatcher( 'IPv4 subnet' ),
      'SUBNET6': Ip6AddrMatcher.Ip6PrefixMatcher( 'IPv6 subnet' )
   }
   cliModel = DhcpServerShowVrfLeasesModel

   @staticmethod
   def adapter( mode, args, argsList ):
      subnet = args.get( 'SUBNET' )
      subnet6 = args.pop( 'SUBNET6', None )

      af = 'ipv4' if args.get( 'ipv4' ) else args.get( 'ipv6' )
      subnet, subnetAF = ( subnet, 'ipv4' ) if subnet else ( subnet6, 'ipv6' )

      # the AF should match the subnetAF if specified
      args[ 'SUBNET' ] = subnet
      args[ 'AF' ] = subnetAF if subnet else af

      filterTypes = [ 'NAME', 'SUBNET' ]
      for f in filterTypes:
         if args.get( f ):
            args[ 'FILTER' ] = f

   @staticmethod
   def handler( mode, args ):
      afModels = {
         'ipv4': Ipv4Leases,
         'ipv6': Ipv6Leases
      }
      v4LeaseInfo = v6LeaseInfo = leaseInfo = None

      # Check if address family is specified
      af = args.pop( 'AF', None )

      def generateLeases( _leaseInfo, af ):
         leases = []
         if not _leaseInfo:
            return leases
         for _, subnetData in sorted( _leaseInfo.iteritems() ):
            for leaseIp in sorted( subnetData.leases.keys() ):
               leaseData = subnetData.lease( leaseIp )
               leaseModel = afModels.get( af )()
               leaseModel.ipAddress = leaseIp
               leaseModel.endLeaseTime = float( leaseData.endTime() )
               leaseModel.lastTransaction = float( leaseData.cltt() )
               mac = leaseData.mac()
               leaseModel.macAddress = mac if mac else '0000.0000.0000'
               leases.append( leaseModel )
         return leases

      filterType = args.get( 'FILTER' )
      filterWith = args.get( filterType )

      # returns true if a lease is desired
      def toAddLeaseFilter( lease ):
         # get af based on lease information
         ipv4 = lease.get( 'iaid' ) is None
         if ipv4:
            subnetConfigs = dhcpServerConfig.subnetConfigIpv4
            idToSubnet = dhcpServerStatus.ipv4IdToSubnet
         else:
            subnetConfigs = dhcpServerConfig.subnetConfigIpv6
            idToSubnet = dhcpServerStatus.ipv6IdToSubnet

         toAdd = True
         subnetId = lease[ "subnet-id" ]
         prefix = idToSubnet.get( subnetId )
         if filterType == "NAME" and prefix:
            subnetConfig = subnetConfigs.get( prefix )
            toAdd = subnetConfig.subnetName == filterWith

         elif filterType == "SUBNET" and prefix:
            filterPrefix = Arnet.Prefix( filterWith ) if ipv4 else \
                           Arnet.Ip6Prefix( filterWith )
            toAdd = prefix == filterPrefix

         return toAdd

      model = DhcpServerShowLeasesModel()
      filterFn = toAddLeaseFilter if filterType else None
      if af is None:
         # show both address family
         v4LeaseInfo = KeaDhcpLeaseData( dhcpServerStatus.ipv4IdToSubnet,
                                         filterFunction=filterFn )

         v6LeaseInfo = KeaDhcpLeaseData( dhcpServerStatus.ipv6IdToSubnet,
                                         ipVersion=6,
                                         filterFunction=filterFn )

         v4Leases = generateLeases( v4LeaseInfo.subnetData, 'ipv4' )
         v6Leases = generateLeases( v6LeaseInfo.subnetData, 'ipv6' )
         model.ipv4ActiveLeases = v4Leases
         model.ipv6ActiveLeases = v6Leases
      else:
         ipVersion = int( af[ -1 ] )
         idToSubnet = dhcpServerStatus.ipv4IdToSubnet if af == 'ipv4' else \
                      dhcpServerStatus.ipv6IdToSubnet

         leaseInfo = KeaDhcpLeaseData( idToSubnet, ipVersion=ipVersion,
                                       filterFunction=filterFn ).subnetData

         leases = generateLeases( leaseInfo, af )
         if af == 'ipv4':
            model.ipv4ActiveLeases = leases
         else:
            model.ipv6ActiveLeases = leases

      vrfModel = DhcpServerShowVrfLeasesModel()
      # Only supported in default vrf for now
      vrfModel.vrfs[ DEFAULT_VRF ] = model
      return vrfModel

# Register Commands
BasicCli.addShowCommandClass( DhcpServerShow )
BasicCli.addShowCommandClass( DhcpServerShowLeases )

def Plugin( entityManager ):
   global dhcpServerConfig
   global dhcpServerStatus
   dhcpServerConfig = LazyMount.mount( entityManager, 'dhcpServer/config',
                                         'DhcpServer::Config', 'r' )
   dhcpServerStatus = LazyMount.mount( entityManager, 'dhcpServer/status',
                                       'DhcpServer::Status', 'r' )
