#!/usr/bin/env python
# Copyright (c) 2019 Arista Networks, Inc.  All rights reserved.
# Arista Networks, Inc. Confidential and Proprietary.

from __future__ import absolute_import, division, print_function

from ArnetModel import Ip4Address, Ip6Address
from ArnetModel import IpGenericPrefix
from ArnetModel import MacAddress
from CliModel import Bool
from CliModel import Dict
from CliModel import Enum
from CliModel import Float
from CliModel import Int
from CliModel import List
from CliModel import Model
from CliModel import Str
from CliModel import Submodel
from CliPlugin.IntfModel import Interface
from EosDhcpServerLib import convertLeaseSeconds
from EosDhcpServerLib import defaultVendorId
from EosDhcpServerLib import tftpServerOptions
from EosDhcpServerLib import vendorSubOptionType
import time
import TableOutput
import Tac

def dhcpTimestampStr( seconds ):
   return time.strftime( "%Y/%m/%d %H:%M:%S UTC", time.gmtime( seconds ) )

class DhcpServerIpv4Range( Model ):
   startRangeAddress = Ip4Address( help="Start address" )
   endRangeAddress = Ip4Address( help="End address" )

   def render( self ):
      print( "Range: {} to {}".format( self.startRangeAddress,
                                       self.endRangeAddress ) )

class DhcpServerIpv6Range( Model ):
   startRangeAddress = Ip6Address( help="Start address" )
   endRangeAddress = Ip6Address( help="End address" )

   def render( self ):
      print( "Range: {} to {}".format( self.startRangeAddress,
                                       self.endRangeAddress ) )

# vendor-option
class DhcpServerSubOptionModel( Model ):
   optionCode = Int( help="Sub-option code" )
   optionType = Enum( values=( vendorSubOptionType.string,
                      vendorSubOptionType.ipAddress ),
                      help="Sub-option type" )
   optionDataIpAddresses = List( optional=True, valueType=Ip4Address,
                           help="IPv4 addresses for sub-option type 'ipv4-address'" )
   optionDataString = Str( optional=True,
                           help="String for sub-option type 'string'" )

class DhcpServerIpv4VendorOptionModel( Model ):
   subOptions = List( valueType=DhcpServerSubOptionModel,
                      help="Sub-options configured" )

   def render( self ):
      headings = ( "Sub-options", "Data" )
      fmt_center = TableOutput.Format( justify="center" )
      table = TableOutput.createTable( headings )
      table.formatColumns( fmt_center, fmt_center )
      for subOption in self.subOptions:
         if subOption.optionType == vendorSubOptionType.string:
            # add quotes to display
            data = '"{}"'.format( subOption.optionDataString )
         elif subOption.optionType == vendorSubOptionType.ipAddress:
            data = ', '.join( subOption.optionDataIpAddresses )
         else:
            assert False, 'Unknown sub-option type'

         table.newRow( subOption.optionCode, data )

      print( table.output() )

# reservations mac-address
class DhcpServerIpv4ReservationsMacAddressModel( Model ):
   macAddress = MacAddress( help="Mac address of client" )
   ipv4Address = Ip4Address( optional=True,
                             help="Reserved IPv4 address" )

   def render( self ):
      # pylint: disable=no-member
      print( "Mac address: {}".format( self.macAddress.displayString ) )
      if self.ipv4Address:
         print( "IPv4 address: {}".format( str( self.ipv4Address ) ) )


class DhcpServerSubnetBaseModel( Model ):
   subnet = IpGenericPrefix( help="Subnet" )
   activeLeases = Int( help="Number of active leases for this subnet" )
   name = Str( help='Subnet name' )
   leaseDuration = Float( optional=True,
                          help="Duration of leases (seconds)" )
   overlappingSubnets = List( valueType=IpGenericPrefix, optional=True,
                              help="Overlapping subnets" )

   def doAfSpecificRender( self ):
      pass

   def _render( self, ipVersion ):
      subnetMsg = "Subnet: {}".format( self.subnet )
      if self.disabledMessage:
         subnetMsg = subnetMsg + " ({})".format( self.disabledMessage )
      print( subnetMsg )
      print( 'Subnet name: {}'.format( self.name ) )
      for r in self.ranges:
         r.render()
      print( "DNS server(s): {}".format( " ".join( list( map(
         str, self.dnsServers ) ) ) ) )

      if self.leaseDuration:
         print( 'Lease duration: {} days {} hours {} minutes'.format(
            *convertLeaseSeconds( self.leaseDuration ) ) )

      self.doAfSpecificRender()

      # Disabled Reason(s)
      if self.disabledMessage and self.overlappingSubnets:
         print( "Disabled reason(s):" )

         # Overlapping subnet(s)
         overlappedList = list( map( str, self.overlappingSubnets ) )
         overlappedStr = " ".join( sorted( overlappedList ) )
         print( "Overlapping subnets: {}".format( overlappedStr ) )

class DhcpServerIpv4SubnetModel( DhcpServerSubnetBaseModel ):
   ranges = List( valueType=DhcpServerIpv4Range,
                  help="Ranges configured" )
   disabledMessage = Str( optional=True,
                          help="Why IPv4 subnet is disabled if configured" )
   dnsServers = List( valueType=Ip4Address,
                      help="DNS servers" )
   defaultGateway = Ip4Address( help="Default gateway address" )
   tftpServerOption66 = Str( optional=True, help="TFTP server option 66" )
   tftpServerOption150 = List( optional=True, valueType=Ip4Address,
                               help="TFTP server option 150" )
   tftpBootFile = Str( optional=True, help="TFTP boot file" )
   reservationsMacAddress = Dict( optional=True,
                                keyType=MacAddress,
                                valueType=DhcpServerIpv4ReservationsMacAddressModel,
                                help="Dictionary of mac address reservations"
                                      " configured, keyed by thier mac address" )

   def render( self ):
      self._render( 4 )

   def doAfSpecificRender( self ):
      print( 'Default gateway address: {}'.format( self.defaultGateway ) )

      serverOptions = tftpServerOptions( self.tftpServerOption66,
                                         self.tftpServerOption150 )
      if serverOptions:
         print( serverOptions )

      if self.tftpBootFile:
         print( "TFTP boot file: {}".format( self.tftpBootFile ) )

      print( "Active leases: {}".format( self.activeLeases ) )

      # reservations mac-address
      if self.reservationsMacAddress:
         print( "Reservations:" )
         for macAddr in sorted( self.reservationsMacAddress ):
            self.reservationsMacAddress[ macAddr ].render()
            print()

class DhcpServerIpv6SubnetModel( DhcpServerSubnetBaseModel ):
   ranges = List( valueType=DhcpServerIpv6Range,
                  help="Ranges configured" )
   disabledMessage = Str( optional=True,
                          help="Why IPv6 subnet is disabled if configured" )
   dnsServers = List( valueType=Ip6Address,
                      help="DNS servers" )
   directActive = Bool( help="IPv6 server responds to direct requests" )
   directActiveDetail = Str( optional=True, help="Details of direct status" )
   relayActive = Bool( help="IPv6 server responds to relay requests" )
   relayActiveDetail = Str( optional=True, help="Details of relay status" )

   def render( self ):
      self._render( 6 )

   def doAfSpecificRender( self ):
      directStr = "Active" if self.directActive else "Inactive"
      if self.directActiveDetail:
         directStr = "{} ({})".format( directStr, self.directActiveDetail )

      relayStr = "Active" if self.relayActive else "Inactive"
      if self.relayActiveDetail:
         relayStr = "{} ({})".format( relayStr, self.relayActiveDetail )

      print( "Direct: {}".format( directStr ) )
      print( "Relay: {}".format( relayStr ) )
      print( "Active leases: {}".format( self.activeLeases ) )

class DhcpServerInterfaceStatusShowModel( Model ):
   active = Bool( help="This interface is active" )
   detail = Str( optional=True,
                 help="Details of interface activity" )

class DhcpServerShowBaseModel( Model ):
   debugLog = Bool( optional=True, help="Debug log is enabled" )

   def _render( self, ipVersion, serverActive, activeLeases, interfaces,
                disabledMessage, subnets, dnsServers, domainName, leaseDuration,
                tftpOptions=None, vendorOptions=None ):
      msg = 'IPv{} DHCP server is {}'.format(
         ipVersion,
         'active' if serverActive else 'inactive' )
      if disabledMessage:
         msg = msg + ' ({})'.format( disabledMessage )
      print( msg )
      if self.debugLog:
         print( "Debug log is enabled" )
      print( 'DNS server(s): {}'.format( ' '.join( list( map(
         str, dnsServers ) ) ) ) )
      print( 'DNS domain name: {}'.format( domainName ) )
      print( 'Lease duration: {} days {} hours {} minutes'.format(
         *convertLeaseSeconds( leaseDuration ) ) )

      if tftpOptions:
         tftpServerOption66 = tftpOptions.get( 'serverOption66' )
         tftpServerOption150 = tftpOptions.get( 'serverOption150' )
         serverOptions = tftpServerOptions( tftpServerOption66,
                                            tftpServerOption150 )
         if serverOptions:
            print( serverOptions )

         tftpBootFile = tftpOptions.get( 'file' )
         if tftpBootFile:
            print( "TFTP file: {}".format( tftpBootFile ) )

      print( 'Active leases: {}'.format( activeLeases ) )
      print( "IPv{} DHCP interface status:".format( ipVersion ) )

      headings = ( "Interface", "Status" )
      table = TableOutput.createTable( headings )
      fmt_left = TableOutput.Format( justify="left" )
      table.formatColumns( fmt_left, fmt_left )
      for intf, status in sorted( interfaces.items() ):
         statusMsg = "Active" if status.active else "Inactive"
         detail = ""
         if status.detail:
            detail = " ({})".format( status.detail )
         rowMsg = "{}{}".format( statusMsg, detail )
         table.newRow( intf, rowMsg )

      print( table.output() )

      # Vendor Options
      if vendorOptions:
         print( "Vendor information:" )
         defaultSubOptions = vendorOptions.pop( defaultVendorId, None )
         if defaultSubOptions:
            # print default vendor-option first
            print( "Vendor ID: {}".format( defaultVendorId ) )
            defaultSubOptions.render()

         for vendorId in sorted( vendorOptions ):
            print( "Vendor ID: {}".format( vendorId ) )
            vendorOptions[ vendorId ].render()

      for _, subnet in sorted( subnets.iteritems() ):
         print()
         subnet.render()
      print()

class DhcpServerIpv4ShowModel( DhcpServerShowBaseModel ):
   ipv4ServerActive = Bool( default=False,
                            help="IPv4 DHCP server is active" )
   ipv4ActiveLeases = Int( default=0,
                           help="Number of active leases" )
   ipv4Interfaces = Dict( keyType=Interface,
                          valueType=DhcpServerInterfaceStatusShowModel,
                          help="A mapping of IPv4 DHCP interfaces to their status" )
   ipv4DisabledMessage = Str( optional=True,
                              help="Why IPv4 DHCP server is disabled if configured" )
   ipv4Subnets = Dict( keyType=IpGenericPrefix,
                       valueType=DhcpServerIpv4SubnetModel,
                       help="Dictionary of IPv4 subnets configured" )
   ipv4DnsServers = List( valueType=Ip4Address,
                          help="DNS servers" )
   ipv4DomainName = Str( default='', help='DNS domain name' )
   ipv4LeaseDuration = Float( default=7200.0,
                              help="Duration of leases (seconds)" )
   ipv4TftpServerOption66 = Str( optional=True, help="TFTP Server option 66" )
   ipv4TftpServerOption150 = List( valueType=Ip4Address, optional=True,
                                  help="TFTP Server option 150" )
   ipv4TftpBootFile = Str( optional=True, help="TFTP boot file" )

   ipv4VendorOptions = Dict( optional=True,
                             keyType=str,
                             valueType=DhcpServerIpv4VendorOptionModel,
                             help="Dictionary of IPv4 vendor-options configured, \
                                   mapping vendor-option to sub-options" )

   def render( self ):
      tftpOptions = {}
      if self.ipv4TftpServerOption66:
         tftpOptions[ 'serverOption66' ] = self.ipv4TftpServerOption66

      if self.ipv4TftpServerOption150:
         tftpOptions[ 'serverOption150' ] = self.ipv4TftpServerOption150

      if self.ipv4TftpBootFile:
         tftpOptions[ 'file' ] = self.ipv4TftpBootFile

      self._render( 4, self.ipv4ServerActive, self.ipv4ActiveLeases,
                    self.ipv4Interfaces, self.ipv4DisabledMessage, self.ipv4Subnets,
                    self.ipv4DnsServers, self.ipv4DomainName,
                    self.ipv4LeaseDuration, tftpOptions, self.ipv4VendorOptions )

class DhcpServerIpv6ShowModel( DhcpServerShowBaseModel ):
   ipv6ServerActive = Bool( default=False,
                            help="IPv6 DHCP server is active" )
   ipv6ActiveLeases = Int( default=0,
                           help="Number of active leases" )
   ipv6Interfaces = Dict( keyType=Interface,
                          valueType=DhcpServerInterfaceStatusShowModel,
                          help="A mapping of IPv4 DHCP interfaces to their status" )
   ipv6DisabledMessage = Str( optional=True,
                          help="Why IPv6 DHCP server is disabled if configured" )
   ipv6Subnets = Dict( keyType=IpGenericPrefix,
                       valueType=DhcpServerIpv6SubnetModel,
                       help="Dictionary of IPv6 subnets configured" )
   ipv6DnsServers = List( valueType=Ip6Address,
                      help="DNS servers" )
   ipv6DomainName = Str( default='', help='DNS domain name' )
   ipv6LeaseDuration = Float( default=7200.0,
                              help="Duration of leases (seconds)" )

   def render( self ):
      self._render( 6, self.ipv6ServerActive, self.ipv6ActiveLeases,
                    self.ipv6Interfaces, self.ipv6DisabledMessage, self.ipv6Subnets,
                    self.ipv6DnsServers, self.ipv6DomainName,
                    self.ipv6LeaseDuration )

class DhcpServerShowModel( Model ):
   ipv4Server = Submodel( valueType=DhcpServerIpv4ShowModel,
                          help="IPv4 DHCP server information" )
   ipv6Server = Submodel( valueType=DhcpServerIpv6ShowModel,
                          help="IPv6 DHCP server information" )
   addressFamily = Str( optional=True, help='IP version of the server called' )

   def render( self ):
      servers = { 'ipv4': self.ipv4Server,
                  'ipv6': self.ipv6Server }
      if self.addressFamily:
         servers[ self.addressFamily ].render()
      else:
         self.ipv4Server.render()
         self.ipv6Server.render()

class DhcpServerShowVrfModel( Model ):
   vrfs = Dict( keyType=str, valueType=DhcpServerShowModel,
                help="DHCP server information for all vrfs" )

   def render( self ):
      for model in self.vrfs.values():
         model.render()

class BaseLeases( Model ):
   endLeaseTime = Float( help="End time of lease" )
   lastTransaction = Float( help="Last transaction time with client" )
   macAddress = MacAddress( help="MAC address of client" )

   def render( self ):
      print( str( self.ipAddress ) )
      print( "End: {}".format(
         dhcpTimestampStr( self.endLeaseTime ) ) )
      print( "Last transaction: {}".format(
         dhcpTimestampStr( self.lastTransaction ) ) )
      # pylint: disable-msg=no-member
      print( "MAC address: {}".format( self.macAddress.displayString ) )

class Ipv4Leases( BaseLeases ):
   ipAddress = Ip4Address( help="IP address" )

class Ipv6Leases( BaseLeases ):
   ipAddress = Ip6Address( help="Ipv6 Address" )

class DhcpServerShowLeasesModel( Model ):
   ipv4ActiveLeases = List( valueType=Ipv4Leases,
                            help="Active IPv4 Leases" )
   ipv6ActiveLeases = List( valueType=Ipv6Leases,
                            help="Active IPv6 Leases" )

   def render( self ):
      numLeases = len( self.ipv4ActiveLeases )
      numLeases6 = len( self.ipv6ActiveLeases )
      for i, lease in enumerate( self.ipv4ActiveLeases ):
         lease.render()
         if i != numLeases:
            print()
      for i, lease in enumerate( self.ipv6ActiveLeases ):
         lease.render()
         if i != numLeases6:
            print()

class DhcpServerShowVrfLeasesModel( Model ):
   vrfs = Dict( keyType=str, valueType=DhcpServerShowLeasesModel,
                help="DHCP server lease information for all vrfs" )

   def render( self ):
      for model in self.vrfs.values():
         model.render()
