#!/usr/bin/env python
# Copyright (c) 2019 Arista Networks, Inc.  All rights reserved.
# Arista Networks, Inc. Confidential and Proprietary.

# pkgdeps: library DhcpServer

from __future__ import absolute_import, division, print_function
import Arnet
import BasicCli
import CliCommand
import CliMatcher
import CliParser
import CliPlugin.IpAddrMatcher as IpAddrMatcher
import CliPlugin.Ip6AddrMatcher as Ip6AddrMatcher
import CliPlugin.IntfCli
import CliPlugin.MacAddr as MacAddr
from CliPlugin.IraIpRouteCliLib import isValidPrefixWithError # BUG368920
from CliPlugin.IraIp6RouteCliLib import isValidIpv6PrefixWithError
from CliPlugin.IraIpIntfCli import DhcpIntfConfigModelet # BUG368914
from CliMode.DhcpServer import DhcpServerBaseMode, DhcpServerSubnetBaseMode
from CliMode.DhcpServer import DhcpServerReservationsBaseMode
from CliMode.DhcpServer import DhcpServerReservationsMacAddressBaseMode
from CliMode.DhcpServer import DhcpServerVendorOptionBaseMode
import CliToken
import ConfigMount
from HostnameCli import validateHostname
import LazyMount
from EosDhcpServerLib import convertDaysHoursMinutes
from EosDhcpServerLib import featureOption43Status
from EosDhcpServerLib import vendorSubOptionType
from EosDhcpServerLib import featureStaticIPv4AddressStatus
import Tac
from Url import UrlMatcher
import FileCliUtil

dhcpServerConfig = None
dhcpServerStatus = None

# set maximum lease time to be a reasonably big number
# that won't cause overflow in Kea
maxLeaseTime = 0x7FFFFFFF

dnsHelp = "DHCP DNS configuration"
dnsServerHelp = 'Set the DNS server(s) for %s DHCP clients'
dnsServerIpv4Help = 'Set the DNS server(s) for IPv4 DHCP clients'
dnsServerIpv6Help = 'Set the DNS server(s) for IPv6 DHCP clients'
dhcpHelp = 'DHCP server, client and relay interface configuration'

# maximum size of a string that kea can send in an option is 253 bytes
maxKeaStringSize = 253

# Adapters
def tftpBootFileAdapter( mode, args, argsList ):
   bootFile = args.get( 'FILENAME', '' )
   stringSize = len( bootFile.encode( 'utf-8' ) )
   if stringSize > maxKeaStringSize:
      mode.addError( 'Invalid boot file name of size {}'.format( stringSize ) )

def IP_ADDRAdapter( mode, args, argsList ):
   ipAddrDefault = "0.0.0.0"
   args[ 'IP_ADDR' ] = Arnet.IpAddr( args.get( 'IP_ADDR', ipAddrDefault ) )

class DhcpServerMode( DhcpServerBaseMode, BasicCli.ConfigModeBase ):
   name = "DHCP Server"
   modeParseTree = CliParser.ModeParseTree()

   def __init__( self, parent, session ):
      DhcpServerBaseMode.__init__( self )
      BasicCli.ConfigModeBase.__init__( self, parent, session )

class EnterDhcpServerMode( CliCommand.CliCommandClass ):
   syntax = '''dhcp server'''
   noOrDefaultSyntax = '''dhcp server'''

   data = {
      'dhcp': 'DHCP configuration',
      'server': 'DHCP server configuration',
   }

   @staticmethod
   def handler( mode, args ):
      dhcpServerConfig.dhcpServerMode = True
      childMode = mode.childMode( DhcpServerMode )
      mode.session_.gotoChildMode( childMode )

   @staticmethod
   def noOrDefaultHandler( mode, args ):
      dhcpServerConfig.dhcpServerMode = dhcpServerConfig.dhcpServerModeDefault
      dhcpServerConfig.disabled = dhcpServerConfig.disabledDefault

      dhcpServerConfig.leaseTimeIpv4 = dhcpServerConfig.leaseTimeIpv4Default
      dhcpServerConfig.domainNameIpv4 = ''
      dhcpServerConfig.dnsServersIpv4.clear()
      dhcpServerConfig.subnetConfigIpv4.clear()
      dhcpServerConfig.debugLogPath = dhcpServerConfig.debugLogPathDefault
      dhcpServerConfig.debugLogUrl = dhcpServerConfig.debugLogPathDefault

      dhcpServerConfig.tftpServerOption66Ipv4 = dhcpServerConfig.tftpServerDefault
      dhcpServerConfig.tftpBootFileNameIpv4 = \
                                          dhcpServerConfig.tftpBootFileNameDefault
      dhcpServerConfig.tftpServerOption150Ipv4.clear()

      dhcpServerConfig.leaseTimeIpv6 = dhcpServerConfig.leaseTimeIpv6Default
      dhcpServerConfig.domainNameIpv6 = ''
      dhcpServerConfig.dnsServersIpv6.clear()
      dhcpServerConfig.subnetConfigIpv6.clear()
      dhcpServerConfig.vendorOptionIpv4.clear()

class DhcpServerVendorOptionV4Mode( DhcpServerVendorOptionBaseMode,
                                    BasicCli.ConfigModeBase ):
   name = "DHCP Server IPv4 Vendor Option"
   modeParseTree = CliParser.ModeParseTree()

   def __init__( self, parent, session, vendorId ):
      param = ( vendorId, 'ipv4' )
      DhcpServerVendorOptionBaseMode.__init__( self, param )
      BasicCli.ConfigModeBase.__init__( self, parent, session )
      self.vendorOption = dhcpServerConfig.vendorOptionIpv4.newMember( vendorId )

class EnterDhcpServerVendorOptionMode( CliCommand.CliCommandClass ):
   syntax = '''vendor-option ipv4 ( default | VENDOR_ID )'''
   noOrDefaultSyntax = syntax

   data = {
      'vendor-option': 'Configure vendor specific option',
      'ipv4': 'Set the vendor specific option for IPv4 DHCP clients',
      'VENDOR_ID': CliMatcher.StringMatcher( helpname='VENDOR_ID',
                                             helpdesc='Vendor identifier name' ),
      'default': 'Set as default vendor specific option',
   }

   @staticmethod
   def adapter( mode, args, argsList ):
      # IPv4
      args[ 'af' ] = 'ipv4'
      args[ 'vendorMode' ] = DhcpServerVendorOptionV4Mode
      args[ 'VENDOR_ID' ] = args.get( 'VENDOR_ID', 'default' )

   @staticmethod
   def handler( mode, args ):
      vendorId = args[ 'VENDOR_ID' ]
      vendorMode = args[ 'vendorMode' ]
      childMode = mode.childMode( vendorMode, vendorId=vendorId )
      mode.session_.gotoChildMode( childMode )

   @staticmethod
   def noOrDefaultHandler( mode, args ):
      vendorId = args[ 'VENDOR_ID' ]
      if args[ 'af' ] == 'ipv4':
         del dhcpServerConfig.vendorOptionIpv4[ vendorId ]

class DhcpServerVendorOptionSubOption( CliCommand.CliCommandClass ):
   syntax = '''sub-option NUMBER type
               ( string data STRING )        |
               ( ipv4-address data IP_ADDR )   |
               ( array ipv4-address data { IP_ADDR } )
            '''
   noOrDefaultSyntax = '''sub-option NUMBER'''
   data = {
         'sub-option': 'Configure a sub-option',
         'NUMBER': CliMatcher.IntegerMatcher( 1, 254, helpdesc="Sub-option code" ),
         'type': 'Configure sub-option type',
         'array': 'Configure type as array',
         'string': 'Configure type as string',
         'ipv4-address': 'Configure type as ipv4-address',
         'data': 'Configure sub-option data',
         'STRING': CliMatcher.QuotedStringMatcher( requireQuote=True,
                                                   helpdesc="A quoted string" ),
         'IP_ADDR': CliCommand.Node( matcher=IpAddrMatcher.ipAddrMatcher )
         }

   @staticmethod
   def adapter( mode, args, argsList ):
      # Save ipv4 address as list
      ipAddresses = args.get( 'IP_ADDR' )
      if ipAddresses and not isinstance( ipAddresses, list ):
         ipAddresses = [ ipAddresses ]
         args[ 'IP_ADDR' ] = ipAddresses

   @staticmethod
   def handler( mode, args ):
      code = args[ 'NUMBER' ]
      subOption = mode.vendorOption.subOptionConfig.newMember( code )
      if 'string' in args:
         subOption.type = vendorSubOptionType.string
         subOption.dataString = args[ 'STRING' ]

      else:
         subOption.type = vendorSubOptionType.ipAddress
         subOption.isArray = 'array' in args
         ipAddresses = args[ 'IP_ADDR' ]
         maxIpAddr = max( len( subOption.dataIpAddress ), len( ipAddresses ) )
         for i in range( maxIpAddr ):
            try:
               subOption.dataIpAddress[ i ] = ipAddresses[ i ]
            except IndexError:
               del subOption.dataIpAddress[ i ]

   @staticmethod
   def noOrDefaultHandler( mode, args ):
      code = args[ 'NUMBER' ]
      del mode.vendorOption.subOptionConfig[ code ]

class DhcpServerDisabled( CliCommand.CliCommandClass ):
   syntax = '''disabled'''
   noOrDefaultSyntax = '''disabled'''

   data = {
      'disabled': 'Disable DHCP server'
   }

   @staticmethod
   def handler( mode, args ):
      dhcpServerConfig.disabled = True

   @staticmethod
   def noOrDefaultHandler( mode, args ):
      dhcpServerConfig.disabled = dhcpServerConfig.disabledDefault

class DhcpServerLeaseTimeExpression( CliCommand.CliExpression ):
   expression = 'DAYS days HOURS hours MINUTES minutes'
   data = {
      'DAYS': CliMatcher.IntegerMatcher( 0, 2000, helpdesc='Days' ),
      'days': 'days',
      'HOURS': CliMatcher.IntegerMatcher( 0, 23, helpdesc='Hours' ),
      'hours': 'hours',
      'MINUTES': CliMatcher.IntegerMatcher( 0, 59, helpdesc='Minutes' ),
      'minutes': 'minutes',
   }

   @staticmethod
   def adapter( mode, args, argsList ):
      # don't parse any information if noOrDefaultHandler is called
      if CliCommand.isNoOrDefaultCmd( args ):
         return

      # Save Lease Time
      days = args.pop( 'DAYS' )
      hours = args.pop( 'HOURS' )
      minutes = args.pop( 'MINUTES' )
      timeSeconds = convertDaysHoursMinutes( days, hours, minutes )
      if 0 < timeSeconds < maxLeaseTime:
         args[ 'TIME' ] = timeSeconds

      # invalid lease time
      elif timeSeconds == 0:
         mode.addError( "Cannot set a 0 duration lease" )

      else:
         mode.addError( "Invalid lease time {} days {} hours {} minutes".format(
                        days, hours, minutes ) )

class DhcpServerLeaseTime( CliCommand.CliCommandClass ):
   syntax = '''lease time ( ipv4 | ipv6 ) TIME'''
   noOrDefaultSyntax = '''lease time ( ipv4 | ipv6 ) ...'''

   data = {
      'lease': 'DHCP leases configuration',
      'time': 'Set the duration of the lease',
      'ipv4': 'Set the duration for an IPv4 lease',
      'ipv6': 'Set the duration for an IPv6 lease',
      'TIME': DhcpServerLeaseTimeExpression
   }

   @staticmethod
   def handler( mode, args ):
      if mode.session.hasError():
         return

      leaseTime = args[ 'TIME' ]
      ipv4 = 'ipv4' in args
      if ipv4:
         dhcpServerConfig.leaseTimeIpv4 = leaseTime
      else:
         dhcpServerConfig.leaseTimeIpv6 = leaseTime

   @staticmethod
   def noOrDefaultHandler( mode, args ):
      ipv4 = 'ipv4' in args
      if ipv4:
         dhcpServerConfig.leaseTimeIpv4 = dhcpServerConfig.leaseTimeIpv4Default
      else:
         dhcpServerConfig.leaseTimeIpv6 = dhcpServerConfig.leaseTimeIpv6Default

class DhcpServerSubnetLeaseTime( CliCommand.CliCommandClass ):
   syntax = '''lease time TIME'''
   noOrDefaultSyntax = '''lease time ...'''

   data = {
      'lease': 'DHCP Subnet leases configuration',
      'time': 'Set the duration of the lease',
      'TIME': DhcpServerLeaseTimeExpression
   }

   @staticmethod
   def handler( mode, args ):
      if mode.session.hasError():
         return

      leaseTime = args[ 'TIME' ]
      mode.subnetConfig.leaseTime = leaseTime

   @staticmethod
   def noOrDefaultHandler( mode, args ):
      mode.subnetConfig.leaseTime = 0.0

class DhcpServerDnsDomainName( CliCommand.CliCommandClass ):
   syntax = '''dns domain name ( ipv4 | ipv6 ) NAME'''
   noOrDefaultSyntax = '''dns domain name (ipv4 | ipv6) ...'''

   data = {
      'dns': dnsHelp,
      'domain': 'DHCP DNS domain configuration',
      'name': 'Set the domain name',
      'ipv4': 'Set the domain name for IPv4 DHCP clients',
      'ipv6': 'Set the domain name for IPv6 DHCP clients',
      'NAME': CliMatcher.StringMatcher( r'\S+',
                                        helpname='NAME',
                                        helpdesc='domain name' )
   }

   @staticmethod
   def handler( mode, args ):
      domain = args[ 'NAME' ]
      if not validateHostname( domain ):
         mode.addError( "Invalid domain name {}".format( domain ) )
         return
      ipv4 = 'ipv4' in args
      if ipv4:
         dhcpServerConfig.domainNameIpv4 = domain
      else:
         dhcpServerConfig.domainNameIpv6 = domain

   @staticmethod
   def noOrDefaultHandler( mode, args ):
      ipv4 = 'ipv4' in args
      if ipv4:
         dhcpServerConfig.domainNameIpv4 = ""
      else:
         dhcpServerConfig.domainNameIpv6 = ""

_maxDnsServers = 2

class DhcpServerDnsServer( CliCommand.CliCommandClass ):
   syntax = '''dns server ( ipv4 { SERVER } ) | ( ipv6 { SERVER6 } )'''
   noOrDefaultSyntax = '''dns server (ipv4 | ipv6) ...'''

   data = {
      'dns': dnsHelp,
      'server': "Set the DNS server(s)",
      'ipv4': dnsServerIpv4Help,
      'ipv6': dnsServerIpv6Help,
      'SERVER': CliCommand.Node(
         matcher=IpAddrMatcher.ipAddrMatcher,
         maxMatches=_maxDnsServers ),
      'SERVER6': CliCommand.Node(
         matcher=Ip6AddrMatcher.ip6AddrMatcher,
         maxMatches=_maxDnsServers )
   }

   @staticmethod
   def handler( mode, args ):
      ipv4 = 'ipv4' in args
      if ipv4:
         servers = args[ 'SERVER' ]
         for i in range( _maxDnsServers ):
            try:
               dhcpServerConfig.dnsServersIpv4[ i ] = Arnet.IpAddr( servers[ i ] )
            except IndexError:
               del dhcpServerConfig.dnsServersIpv4[ i ]
      else:
         servers = args[ 'SERVER6' ]
         for i in range( _maxDnsServers ):
            try:
               dhcpServerConfig.dnsServersIpv6[ i ] = Arnet.Ip6Addr( servers[ i ] )
            except IndexError:
               del dhcpServerConfig.dnsServersIpv6[ i ]

   @staticmethod
   def noOrDefaultHandler( mode, args ):
      ipv4 = 'ipv4' in args
      if ipv4:
         for i in range( _maxDnsServers ):
            del dhcpServerConfig.dnsServersIpv4[ i ]
      else:
         for i in range( _maxDnsServers ):
            del dhcpServerConfig.dnsServersIpv6[ i ]

def fsFunc( fs ):
   return fs.scheme == 'file:' and fs.supportsWrite()

class DhcpServerDebugLog( CliCommand.CliCommandClass ):
   syntax = '''debug log FILE'''
   noOrDefaultSyntax = '''debug log ...'''

   data = {
      'debug': "DHCP server debugging configuration",
      'log': "DHCP server debug log configuration",
      'FILE': UrlMatcher( fsFunc, "Location for debug log" ),
   }

   @staticmethod
   def handler( mode, args ):
      debugFilePath = args[ 'FILE' ]
      FileCliUtil.checkUrl( debugFilePath )
      if debugFilePath.isdir():
         mode.addError( "Cannot specify a directory" )
         return

      dhcpServerConfig.debugLogPath = debugFilePath.localFilename()
      # localFilename: /tmp/localfile
      # url: file:/tmp/localfile
      # Save the url passed in because when testing a save plugin with which uses
      # Url.filenameToUrl(), it doesn't convert properly back to file:/
      dhcpServerConfig.debugLogUrl = debugFilePath.url

   @staticmethod
   def noOrDefaultHandler( mode, args ):
      dhcpServerConfig.debugLogPath = dhcpServerConfig.debugLogPathDefault
      dhcpServerConfig.debugLogUrl = dhcpServerConfig.debugLogPathDefault

def generateTftpServerExpression( option=None ):
   class DhcpServerTftpServerExpression( CliCommand.CliExpression ):
      expression = 'tftp server'
      data = {
         'tftp': 'TFTP option',
         'server': 'Set TFTP server',
      }
      if option:
         expression += ' option {}'.format( option )
         data[ 'option' ] = "DHCP option configuration"
         data[ option ] = "DHCP TFTP option {}".format( option )

   return DhcpServerTftpServerExpression

class DhcpServerTftpOption66( CliCommand.CliCommandClass ):
   syntax = '''TFTP_EXPR ipv4 NAME'''
   noOrDefaultSyntax = '''TFTP_EXPR ipv4 ...'''

   data = {
      'TFTP_EXPR': generateTftpServerExpression( option="66" ),
      'ipv4': 'Set the TFTP option for IPv4 DHCP clients',
      'NAME': CliMatcher.PatternMatcher( pattern=r'[a-zA-Z0-9\._-]+',
                                             helpname='SERVER',
                                             helpdesc='TFTP server name' ),
   }

   @staticmethod
   def handler( mode, args ):
      dhcpServerConfig.tftpServerOption66Ipv4 = args.get(
            'NAME', dhcpServerConfig.tftpServerDefault )

   noOrDefaultHandler = handler

class DhcpServerTftpOption150( CliCommand.CliCommandClass ):
   syntax = '''TFTP_EXPR ipv4 { SERVER_ADDR }'''
   noOrDefaultSyntax = '''TFTP_EXPR ipv4 ...'''

   data = {
      'TFTP_EXPR': generateTftpServerExpression( option="150" ),
      'ipv4': 'Set the TFTP option for IPv4 DHCP clients',
      'SERVER_ADDR': CliCommand.Node( matcher=IpAddrMatcher.ipAddrMatcher ),
   }

   @staticmethod
   def handler( mode, args ):
      servers = args.get( 'SERVER_ADDR' )
      maxServerCounts = max(
         len( dhcpServerConfig.tftpServerOption150Ipv4 ), len( servers ) )
      for i in range( maxServerCounts ):
         try:
            dhcpServerConfig.tftpServerOption150Ipv4[ i ] = servers[ i ]
         except IndexError:
            del dhcpServerConfig.tftpServerOption150Ipv4[ i ]

   @staticmethod
   def noOrDefaultHandler( mode, args ):
      dhcpServerConfig.tftpServerOption150Ipv4.clear()

class DhcpServerTftpBootFile( CliCommand.CliCommandClass ):
   syntax = '''TFTP_EXPR file ipv4 FILENAME'''
   noOrDefaultSyntax = '''TFTP_EXPR file ipv4 ...'''

   data = {
      'TFTP_EXPR': generateTftpServerExpression(),
      'ipv4': 'Set the TFTP option for IPv4 DHCP clients',
      'file': 'Set the boot file name',
      'FILENAME': CliMatcher.StringMatcher( helpname='FILENAME',
                                            helpdesc='TFTP boot file name' )
   }

   adapter = tftpBootFileAdapter

   @staticmethod
   def handler( mode, args ):
      if mode.session.hasError():
         return

      dhcpServerConfig.tftpBootFileNameIpv4 = args.get(
         'FILENAME', dhcpServerConfig.tftpBootFileNameDefault )

   noOrDefaultHandler = handler

class DhcpServerSubnetV4Mode( DhcpServerSubnetBaseMode, BasicCli.ConfigModeBase ):
   name = "DHCP Server IPv4 Subnet"
   modeParseTree = CliParser.ModeParseTree()

   def __init__( self, parent, session, param ):
      prefix = Arnet.Prefix( param )
      param = ( param, 'ipv4' )
      DhcpServerSubnetBaseMode.__init__( self, param )
      BasicCli.ConfigModeBase.__init__( self, parent, session )
      self.subnetConfig = dhcpServerConfig.subnetConfigIpv4.newMember( prefix )

class DhcpServerSubnetV6Mode( DhcpServerSubnetBaseMode, BasicCli.ConfigModeBase ):
   name = "DHCP Server IPv6 Subnet"
   modeParseTree = CliParser.ModeParseTree()

   def __init__( self, parent, session, param ):
      prefix = Arnet.Ip6Prefix( param )
      param = ( param, 'ipv6' )
      DhcpServerSubnetBaseMode.__init__( self, param )
      BasicCli.ConfigModeBase.__init__( self, parent, session )
      self.subnetConfig = dhcpServerConfig.subnetConfigIpv6.newMember( prefix )

class EnterDhcpServerSubnetMode( CliCommand.CliCommandClass ):
   syntax = '''subnet ( SUBNET | SUBNET6 )'''
   noOrDefaultSyntax = syntax

   data = {
      'subnet': 'Configure a subnet',
      'SUBNET': IpAddrMatcher.IpPrefixMatcher( 'IPv4 subnet to configure ' ),
      'SUBNET6': Ip6AddrMatcher.Ip6PrefixMatcher( 'IPv6 subnet to configure ' )
   }

   @staticmethod
   def adapter( mode, args, argsList ):
      '''
      Save the necessary attributes according to specific address family
      to 'args' to pass to handler() and noOrDefaultHandler().
      '''
      subnet = args.pop( 'SUBNET', None )
      args[ 'SUBNET' ] = subnet if subnet else args.pop( 'SUBNET6' )
      args[ 'prefixCheck' ] = isValidPrefixWithError if subnet else \
                              isValidIpv6PrefixWithError
      args[ 'subnetMode' ] = DhcpServerSubnetV4Mode if subnet else \
                            DhcpServerSubnetV6Mode
      args[ 'af' ] = 'ipv4' if subnet else 'ipv6'

   @staticmethod
   def handler( mode, args ):
      prefix = args[ 'SUBNET' ]
      if not args[ 'prefixCheck' ]( mode, prefix ):
         return
      childMode = mode.childMode( args[ 'subnetMode' ], param=prefix )
      mode.session_.gotoChildMode( childMode )

   @staticmethod
   def noOrDefaultHandler( mode, args ):
      prefix = args[ 'SUBNET' ]
      if not args[ 'prefixCheck' ]( mode, prefix ):
         return
      if args[ 'af' ] == 'ipv4':
         prefix = Arnet.Prefix( prefix )
         del dhcpServerConfig.subnetConfigIpv4[ prefix ]
      else:
         prefix = Arnet.Ip6Prefix( prefix )
         del dhcpServerConfig.subnetConfigIpv6[ prefix ]

def generateDnsServerExpression( af ):
   class DhcpServerDnsServerExpression( CliCommand.CliExpression ):
      expression = 'dns server'
      data = {
         'dns': dnsHelp,
         'server': dnsServerHelp % af
      }
   return DhcpServerDnsServerExpression

class DhcpServerSubnetDnsServerBase( CliCommand.CliCommandClass ):
   syntax = '''dns server { SERVER }'''
   noOrDefaultSyntax = '''dns server ...'''

   @staticmethod
   def handler( mode, args ):
      servers = args[ 'SERVER' ]
      ipv4 = isinstance( mode, DhcpServerSubnetV4Mode )
      ipAddrFn = Arnet.IpAddr if ipv4 else Arnet.Ip6Addr
      for i in range( _maxDnsServers ):
         try:
            mode.subnetConfig.dnsServers[ i ] = ipAddrFn( servers[ i ] )
         except IndexError:
            del mode.subnetConfig.dnsServers[ i ]

   @staticmethod
   def noOrDefaultHandler( mode, args ):
      for i in range( _maxDnsServers ):
         del mode.subnetConfig.dnsServers[ i ]

class DhcpServerSubnetV4DnsServer( DhcpServerSubnetDnsServerBase ):
   data = {
      'SERVER_EXPR': generateDnsServerExpression( 'IPv4' ),
      'SERVER': CliCommand.Node(
         matcher=IpAddrMatcher.ipAddrMatcher, maxMatches=_maxDnsServers ),
   }

class DhcpServerSubnetV6DnsServer( DhcpServerSubnetDnsServerBase ):
   data = {
      'SERVER_EXPR': generateDnsServerExpression( 'IPv6' ),
      'SERVER': CliCommand.Node(
         matcher=Ip6AddrMatcher.ip6AddrMatcher, maxMatches=_maxDnsServers ),
   }

class DhcpServerSubnetV4DefaultGateway( CliCommand.CliCommandClass ):
   syntax = '''default-gateway IP_ADDR'''
   noOrDefaultSyntax = '''default-gateway ...'''

   data = {
      'default-gateway': "Configure the default gateway sent to DHCP clients",
      'IP_ADDR': IpAddrMatcher.IpAddrMatcher(
         'Address of default gateway' )
   }

   adapter = IP_ADDRAdapter

   @staticmethod
   def handler( mode, args ):
      mode.subnetConfig.defaultGateway = args[ 'IP_ADDR' ]

   noOrDefaultHandler = handler

class DhcpServerSubnetRangeBase( CliCommand.CliCommandClass ):
   syntax = '''range START END'''
   noOrDefaultSyntax = syntax

   @staticmethod
   def handler( mode, args ):
      ipv4 = isinstance( mode, DhcpServerSubnetV4Mode )
      ipAddrFn = Arnet.IpAddr if ipv4 else Arnet.Ip6Addr
      subnetRange = "DhcpServer::Range" if ipv4 else "DhcpServer::Range6"

      start = ipAddrFn( args[ 'START' ] )
      end = ipAddrFn( args[ 'END' ] )

      if start > end:
         mode.addError( 'Start range cannot be larger than end of range' )
         return

      subnetRange = Tac.Value( subnetRange, start, end )
      mode.subnetConfig.ranges[ subnetRange ] = True

   @staticmethod
   def noOrDefaultHandler( mode, args ):
      ipv4 = isinstance( mode, DhcpServerSubnetV4Mode )
      ipAddrFn = Arnet.IpAddr if ipv4 else Arnet.Ip6Addr
      subnetRange = "DhcpServer::Range" if ipv4 else "DhcpServer::Range6"

      start = ipAddrFn( args[ 'START' ] )
      end = ipAddrFn( args[ 'END' ] )

      if start > end:
         mode.addError( 'Start range cannot be larger than end of range' )
         return

      subnetRange = Tac.Value( subnetRange, start, end )
      del mode.subnetConfig.ranges[ subnetRange ]

class DhcpServerSubnetV4Range( DhcpServerSubnetRangeBase ):
   data = {
      'range': 'Configure range of addresses',
      'START': IpAddrMatcher.IpAddrMatcher(
         'Start of subnet range IPv4 (inclusive)' ),
      'END': IpAddrMatcher.IpAddrMatcher( 'End of subnet range IPv4 (inclusive)' ),
   }

class DhcpServerSubnetV6Range( DhcpServerSubnetRangeBase ):
   data = {
      'range': 'Configure range of addresses',
      'START': Ip6AddrMatcher.Ip6AddrMatcher(
         'Start of subnet range IPv6 (inclusive)' ),
      'END': Ip6AddrMatcher.Ip6AddrMatcher( 'End of subnet range IPv6 (inclusive)' ),
   }

class DhcpServerSubnetName( CliCommand.CliCommandClass ):
   syntax = '''name NAME'''
   noOrDefaultSyntax = '''name ...'''

   data = {
      'name': 'Configure name of subnet',
      'NAME': CliMatcher.StringMatcher( helpname='NAME',
                                        helpdesc='Name of subnet' )
   }

   @staticmethod
   def handler( mode, args ):
      name = args[ 'NAME' ]
      mode.subnetConfig.subnetName = name

   @staticmethod
   def noOrDefaultHandler( mode, args ):
      mode.subnetConfig.subnetName = ""

class DhcpServerSubnetTftpOption66( CliCommand.CliCommandClass ):
   syntax = '''TFTP_EXPR NAME'''
   noOrDefaultSyntax = '''TFTP_EXPR ...'''

   data = {
      'TFTP_EXPR': generateTftpServerExpression( option="66" ),
      'NAME': CliMatcher.PatternMatcher( pattern=r'[a-zA-Z0-9\._-]+',
                                             helpname='SERVER',
                                             helpdesc='TFTP server name' ),
   }

   @staticmethod
   def handler( mode, args ):
      mode.subnetConfig.tftpServerOption66 = args.get(
            'NAME', mode.subnetConfig.tftpServerDefault )

   noOrDefaultHandler = handler

class DhcpServerSubnetTftpOption150( CliCommand.CliCommandClass ):
   syntax = '''TFTP_EXPR { SERVER_ADDR }'''
   noOrDefaultSyntax = '''TFTP_EXPR ...'''

   data = {
      'TFTP_EXPR': generateTftpServerExpression( option="150" ),
      'SERVER_ADDR': CliCommand.Node( matcher=IpAddrMatcher.ipAddrMatcher ),
   }

   @staticmethod
   def handler( mode, args ):
      servers = args.get( 'SERVER_ADDR' )
      maxServerCounts = max(
         len( mode.subnetConfig.tftpServerOption150 ), len( servers ) )
      for i in range( maxServerCounts ):
         try:
            mode.subnetConfig.tftpServerOption150[ i ] = servers[ i ]
         except IndexError:
            del mode.subnetConfig.tftpServerOption150[ i ]

   @staticmethod
   def noOrDefaultHandler( mode, args ):
      mode.subnetConfig.tftpServerOption150.clear()

class DhcpServerSubnetTftpBootFile( CliCommand.CliCommandClass ):
   syntax = '''TFTP_EXPR file FILENAME'''
   noOrDefaultSyntax = '''TFTP_EXPR file ...'''

   data = {
      'TFTP_EXPR': generateTftpServerExpression(),
      'file': 'Set the boot file name',
      'FILENAME': CliMatcher.StringMatcher( helpname='FILENAME',
                                            helpdesc='TFTP boot file name' ),
   }

   adapter = tftpBootFileAdapter

   @staticmethod
   def handler( mode, args ):
      if mode.session.hasError():
         return

      mode.subnetConfig.tftpBootFileName = args.get(
            'FILENAME', mode.subnetConfig.tftpBootFileNameDefault )

   noOrDefaultHandler = handler

class InterfaceEnterDhcpServerEnable( CliCommand.CliCommandClass ):
   syntax = '''dhcp server ( ipv4 | ipv6 )'''
   noOrDefaultSyntax = syntax

   data = {
      'dhcp': dhcpHelp,
      'server': 'DHCP server interface configuration',
      'ipv4': 'Enable IPv4 DHCP server',
      'ipv6': 'Enable IPv6 DHCP server',
   }

   @staticmethod
   def handler( mode, args ):
      ipv4 = 'ipv4' in args
      interfaces = dhcpServerConfig.interfacesIpv4 if ipv4 else \
                   dhcpServerConfig.interfacesIpv6
      interfaces[ mode.intf.name ] = True

   @staticmethod
   def noOrDefaultHandler( mode, args ):
      ipv4 = 'ipv4' in args
      interfaces = dhcpServerConfig.interfacesIpv4 if ipv4 else \
                   dhcpServerConfig.interfacesIpv6
      del interfaces[ mode.intf.name ]

# When running "default interface <interface>", we need to deleteconfig
class DhcpServerIntf( CliPlugin.IntfCli.IntfDependentBase ):
   def setDefault( self ):
      intfName = self.intf_.name
      del dhcpServerConfig.interfacesIpv4[ intfName ]
      del dhcpServerConfig.interfacesIpv6[ intfName ]

class DhcpServerClearLeases( CliCommand.CliCommandClass ):
   syntax = '''clear dhcp server ( ipv4 | ipv6 ) leases'''

   data = {
      'clear': CliToken.Clear.clearKwNode,
      'dhcp': 'Clear DHCP information',
      'server': 'Clear DHCP server information',
      'ipv4': 'Clear IPv4 DHCP server information',
      'ipv6': 'Clear IPv6 DHCP server information',
      'leases': 'Clear DHCP server leases',
   }

   @staticmethod
   def handler( mode, args ):
      if 'ipv4' in args:
         dhcpServerConfig.clearIdIpv4 = max( dhcpServerStatus.lastClearIdIpv4,
                                             dhcpServerConfig.clearIdIpv4 ) + 1
      else:
         dhcpServerConfig.clearIdIpv6 = max( dhcpServerStatus.lastClearIdIpv6,
                                             dhcpServerConfig.clearIdIpv6 ) + 1

# reservations mode
class DhcpServerReservationsV4Mode( DhcpServerReservationsBaseMode,
                                    BasicCli.ConfigModeBase ):
   name = "DHCP Server IPv4 Reservations"
   modeParseTree = CliParser.ModeParseTree()

   def __init__( self, parent, session, subnetConfig ):
      param = ( 'ipv4' )
      DhcpServerReservationsBaseMode.__init__( self, param )
      BasicCli.ConfigModeBase.__init__( self, parent, session )
      self.subnetConfig = subnetConfig

class EnterDhcpServerReservationsMode( CliCommand.CliCommandClass ):
   syntax = '''reservations'''
   noOrDefaultSyntax = syntax

   data = {
         'reservations': 'DHCP Client IPv4 reservations configuration'
         }

   @staticmethod
   def handler( mode, args ):
      # only available for ipv4
      childMode = mode.childMode( DhcpServerReservationsV4Mode,
                                  subnetConfig=mode.subnetConfig )
      mode.session_.gotoChildMode( childMode )

   @staticmethod
   def noOrDefaultHandler( mode, args ):
      mode.subnetConfig.reservationsMacAddr.clear()

# reservations mac-address mode
class DhcpServerReservationsMacAddressV4Mode(
                     DhcpServerReservationsMacAddressBaseMode,
                     BasicCli.ConfigModeBase ):
   name = "DHCP Server IPv4 Mac Address Reservations"
   modeParseTree = CliParser.ModeParseTree()

   def __init__( self, parent, session, subnetConfig, param ):
      macAddr = Arnet.EthAddr( param )
      param = ( macAddr.displayString, 'ipv4' )
      DhcpServerReservationsMacAddressBaseMode.__init__( self, param )
      BasicCli.ConfigModeBase.__init__( self, parent, session )
      self.subnetConfig = subnetConfig
      self.macAddrConfig = subnetConfig.reservationsMacAddr.\
                                        newMember( macAddr.displayString )

class EnterDhcpServerReservationsMacAddressMode( CliCommand.CliCommandClass ):
   syntax = '''mac-address MAC_ADDR'''
   noOrDefaultSyntax = syntax

   data = {
         'mac-address': "Reserve using the client's mac-address",
         'MAC_ADDR': MacAddr.macAddrMatcher,
         }

   @staticmethod
   def handler( mode, args ):
      macAddr = args[ 'MAC_ADDR' ]
      childMode = mode.childMode( DhcpServerReservationsMacAddressV4Mode,
                                  subnetConfig=mode.subnetConfig,
                                  param=macAddr )
      mode.session_.gotoChildMode( childMode )

   @staticmethod
   def noOrDefaultHandler( mode, args ):
      macAddr = args[ 'MAC_ADDR' ]
      del mode.subnetConfig.reservationsMacAddr[ macAddr ]

class DhcpServerReservationsMacAddressIpAddress( CliCommand.CliCommandClass ):
   syntax = '''ipv4-address IP_ADDR'''
   noOrDefaultSyntax = '''ipv4-address ...'''

   data = {
         'ipv4-address': 'Reserve an IPv4 address',
         'IP_ADDR': IpAddrMatcher.IpAddrMatcher( 'IPv4 address' ),
         }

   adapter = IP_ADDRAdapter

   @staticmethod
   def handler( mode, args ):
      ipAddrDefault = Arnet.IpAddr( '0.0.0.0' )
      ipAddr = args[ 'IP_ADDR' ]
      prefix = mode.subnetConfig.subnetId
      # verify the ipv4-address is from the current subnet
      if ipAddr != ipAddrDefault and not prefix.contains( ipAddr ):
         errorString = "ipv4-address must be a valid IPv4 address from subnet {}"
         mode.addError( errorString.format( prefix ) )
         return

      mode.macAddrConfig.ipAddr = ipAddr

   noOrDefaultHandler = handler

BasicCli.GlobalConfigMode.addCommandClass( EnterDhcpServerMode )
DhcpServerMode.addCommandClass( DhcpServerDisabled )
DhcpServerMode.addCommandClass( DhcpServerLeaseTime )
DhcpServerMode.addCommandClass( DhcpServerDnsDomainName )
DhcpServerMode.addCommandClass( DhcpServerDnsServer )
DhcpServerMode.addCommandClass( DhcpServerDebugLog )
DhcpServerMode.addCommandClass( DhcpServerTftpOption66 )
DhcpServerMode.addCommandClass( DhcpServerTftpOption150 )
DhcpServerMode.addCommandClass( DhcpServerTftpBootFile )
DhcpServerMode.addCommandClass( EnterDhcpServerSubnetMode )

if featureOption43Status():
   DhcpServerMode.addCommandClass( EnterDhcpServerVendorOptionMode )
   DhcpServerVendorOptionV4Mode.addCommandClass( DhcpServerVendorOptionSubOption )

DhcpServerSubnetV4Mode.addCommandClass( DhcpServerSubnetV4DnsServer )
DhcpServerSubnetV4Mode.addCommandClass( DhcpServerSubnetV4DefaultGateway )
DhcpServerSubnetV4Mode.addCommandClass( DhcpServerSubnetV4Range )
DhcpServerSubnetV4Mode.addCommandClass( DhcpServerSubnetName )
DhcpServerSubnetV4Mode.addCommandClass( DhcpServerSubnetTftpOption66 )
DhcpServerSubnetV4Mode.addCommandClass( DhcpServerSubnetTftpOption150 )
DhcpServerSubnetV4Mode.addCommandClass( DhcpServerSubnetTftpBootFile )
DhcpServerSubnetV4Mode.addCommandClass( DhcpServerSubnetLeaseTime )

if featureStaticIPv4AddressStatus():
   DhcpServerSubnetV4Mode.addCommandClass( EnterDhcpServerReservationsMode )
   DhcpServerReservationsV4Mode.addCommandClass(
                        EnterDhcpServerReservationsMacAddressMode )
   DhcpServerReservationsMacAddressV4Mode.addCommandClass(
                        DhcpServerReservationsMacAddressIpAddress )

DhcpServerSubnetV6Mode.addCommandClass( DhcpServerSubnetV6DnsServer )
DhcpServerSubnetV6Mode.addCommandClass( DhcpServerSubnetV6Range )
DhcpServerSubnetV6Mode.addCommandClass( DhcpServerSubnetName )
DhcpServerSubnetV6Mode.addCommandClass( DhcpServerSubnetLeaseTime )
DhcpIntfConfigModelet.addCommandClass( InterfaceEnterDhcpServerEnable )

BasicCli.EnableMode.addCommandClass( DhcpServerClearLeases )

def Plugin( entityManager ):
   global dhcpServerConfig
   global dhcpServerStatus
   dhcpServerConfig = ConfigMount.mount( entityManager, 'dhcpServer/config',
                                         'DhcpServer::Config', 'w' )
   dhcpServerStatus = LazyMount.mount( entityManager, 'dhcpServer/status',
                                       'DhcpServer::Status', 'r' )
   CliPlugin.IntfCli.Intf.registerDependentClass( DhcpServerIntf )
