#!/usr/bin/env python
# Copyright (c) 2019 Arista Networks, Inc.  All rights reserved.
# Arista Networks, Inc. Confidential and Proprietary.

from __future__ import absolute_import, division, print_function

import Ark
import Arnet
import BasicCli
import CliCommand
import CliMatcher
import ConfigMount
import LazyMount
import ShowCommand
import Tac
import CliPlugin.AclCli as AclCli
import CliPlugin.DhcpSnoopingCli as DhcpSnoopingCli
import CliPlugin.Dhcpv6SnoopingCli as Dhcpv6SnoopingCli
from CliPlugin.AclCliModel import AllAclList
from CliPlugin.DhcpRelayCommonModel import (
      ConfiguredInterface, Helper, IpDhcpRelay, DefaultServersShowModel,
)
from CliPlugin.DhcpRelayCounterModel import (
   DhcpRelayCounterModel, Counters, GlobalCounters, InterfaceCounter,
)
from CliPlugin.DhcpRelaySnoopingModel import (
   CircuitIdModel, DhcpSnoopingModel, Dhcp6SnoopingModel, DhcpSnoopingHardwareModel,
   Dhcp6SnoopingHardwareModel, VlanList, DhcpSnoopingCounterModel, 
   Dhcp6SnoopingCounterModel, VlanCounters, DhcpSnoopingDebugCounterModel, 
   SnoopingRelayCounters, Dhcp6SnoopingDebugCounterModel
)
from CliPlugin.DhcpRelayHelperCli import (
   drMlagStatus, drStatus, drCounterConfig, drConfig,
)
from CliToken.Dhcp import dhcpMatcherForShow
from CliToken.Ip import ipMatcherForShow
from CliToken.Ipv6 import ipv6MatcherForShow
import DhcpRelayVrfCliLib
import Toggles.DhcpRelayToggleLib as DhcpRelayToggleLib

matcherCounters = CliMatcher.KeywordMatcher( 
      'counters', helpdesc='Counters' )
matcherDebug = CliMatcher.KeywordMatcher( 
      'debug', helpdesc='Internal counters for debugging' )
matcherRelay = CliMatcher.KeywordMatcher( 
      'relay', helpdesc='DHCP Relay' )
matcherSnoopingV6 = CliMatcher.KeywordMatcher( 
      'snooping', helpdesc='DHCPv6 Snooping information' )
matcherSnooping = CliMatcher.KeywordMatcher(
      'snooping', helpdesc='DHCP Snooping information' )
nodeDetail = CliCommand.guardedKeyword(
      'detail', helpdesc='Access list detail',
      guard=AclCli.countersPerChipEnabledGuard )
nodeSnoopingV6 = CliCommand.guardedKeyword( 
      'snooping', helpdesc='DHCPv6 Snooping information', 
      guard=Dhcpv6SnoopingCli.dhcp6SnoopingSupportedGuard )
nodeSnooping = CliCommand.guardedKeyword( 
      'snooping', helpdesc='DHCP Snooping information', 
      guard=DhcpSnoopingCli.dhcpSnoopingSupportedGuard )
addressFamilyExprForShow = CliMatcher.EnumMatcher( {
   'ip' : ipMatcherForShow.helpdesc_,
   'ipv6' : ipv6MatcherForShow.helpdesc_,
} )

aclCpConfig = None
aclStatus = None
aclCheckpoint = None
dhcpSnoopingConfig = None
dhcpSnoopingCounterConfig = None
dhcpSnoopingHwStatusDir = None
dhcpSnoopingStatus = None
bridgingConfig = None
platformHardwareSliceDir = None
dhcp6SnoopingConfig = None
dhcp6SnoopingHwStatusDir = None
dhcp6SnoopingStatus = None
dhcp6SnoopingCounterConfig = None

#--------------------------------------------------------------------------------
# show ( ip | ipv6 ) dhcp relay counters
#--------------------------------------------------------------------------------
def updateDhcpRelayCounters( mode ):
   drCounterConfig().counterUpdateRequestTime = Tac.now()
   def countersUpdated():
      return ( drStatus().counterUpdateTime >=
               drCounterConfig().counterUpdateRequestTime )
   try:
      Tac.waitFor( countersUpdated, description='Counter update',
                   maxDelay=0.1, sleep=True, timeout=5 )
   except Tac.Timeout:
      mode.addWarning( 'Displaying stale counters' )

def showDhcpRelayCounters( mode, args ):
   status = drStatus()
   counterIntfs = drStatus().counterIntfStatus
   result = DhcpRelayCounterModel()
   emptyIntf = Tac.newInstance( "Arnet::IntfId", "" ).stringValue

   if status.runControl is True:
      # Agent is running; Request agent to update counters.
      updateDhcpRelayCounters( mode )

   result.globalCounters = GlobalCounters()
   result.globalCounters.allRequests = Counters()
   result.globalCounters.allRequests.received = status.totalRequestsReceived
   result.globalCounters.allRequests.forwarded = status.totalRequestsForwarded
   result.globalCounters.allRequests.dropped = status.totalRequestsDropped
   result.globalCounters.allResponses = Counters()
   result.globalCounters.allResponses.received = status.totalRepliesReceived
   result.globalCounters.allResponses.forwarded = status.totalRepliesForwarded
   result.globalCounters.allResponses.dropped = status.totalRepliesDropped
   result.globalCounters.lastResetTime = \
         Ark.switchTimeToUtc( status.lastResetAllTime )

   def populateInterfaceCounter( counterIntfStatus ):
      request = Counters()
      reply = Counters()
      request.received = counterIntfStatus.requestsReceived
      request.forwarded = counterIntfStatus.requestsForwarded
      request.dropped = counterIntfStatus.requestsDropped
      reply.received = counterIntfStatus.repliesReceived
      reply.forwarded = counterIntfStatus.repliesForwarded
      reply.dropped = counterIntfStatus.repliesDropped
      intfCounter = InterfaceCounter()
      intfCounter.requests = request
      intfCounter.replies = reply
      intfCounter.lastResetTime = \
            Ark.switchTimeToUtc( counterIntfStatus.lastResetTime )
      return intfCounter

   if counterIntfs:
      result.interfaceCounters = {}
      for intfname in counterIntfs:
         if intfname != emptyIntf:
            counterIntfStatus = counterIntfs[ intfname ]
            result.interfaceCounters[ intfname ] = \
                  populateInterfaceCounter( counterIntfStatus )

   return result

class DhcpRelayCountersCmd( ShowCommand.ShowCliCommandClass ):
   syntax = 'show ADDR_FAMILY dhcp relay counters'
   data = {
      'ADDR_FAMILY' : addressFamilyExprForShow,
      'dhcp' : dhcpMatcherForShow,
      'relay' : matcherRelay,
      'counters' : matcherCounters,
   }

   handler = showDhcpRelayCounters
   cliModel = DhcpRelayCounterModel

BasicCli.addShowCommandClass( DhcpRelayCountersCmd )

def getAllV4Helpers( v4Helpers ):
   allV4Helpers = []
   for x in v4Helpers:
      helper = Helper()
      helper.serverAddr = Arnet.IpGenAddr( x.ip.v4Addr )
      helper.vrf = x.vrfName
      if x.srcIp.stringValue != '0.0.0.0':
         helper.srcAddr = x.srcIp.stringValue
      if x.srcIntf != '':
         helper.srcIntf = x.srcIntf 
      allV4Helpers.append( helper )
   return allV4Helpers

def getAllV6Helpers( v6Helpers, intfConfig ):
   allV6Helpers = []
   for x in v6Helpers:
      helper = Helper()
      helper.serverAddr = Arnet.IpGenAddr( x.ip.v6Addr.stringValue )
      helper.vrf = x.vrfName
      helper.linkAddr = intfConfig.serverIp6[ x ]
      if x.srcIp.stringValue != '::':
         helper.srcAddr = x.srcIp.stringValue
      if x.srcIntf != '':
         helper.srcIntf = x.srcIntf 
      allV6Helpers.append( helper )
   return allV6Helpers

def getAllHostnameHelpers( hostnameHelpers ):
   allHostnameHelpers = []
   for x in hostnameHelpers:
      helper = Helper()
      helper.serverHostname = x.hostname
      helper.vrf = x.vrfName
      if x.srcIp.stringValue != '0.0.0.0':
         helper.srcAddr = x.srcIp.stringValue
      if x.srcIntf != '':
         helper.srcIntf = x.srcIntf 
      allHostnameHelpers.append( helper )
   return allHostnameHelpers

#--------------------------------------------------------------------------------
# show ( ip | ipv6 ) helper-address
#--------------------------------------------------------------------------------
def showIpDhcpRelay( mode, args ):
   config = drConfig()
   infoOpt = config.circuitIdOptOn
   llAddrOpt = config.clientLinkLayerAddrOptOn
   remoteIdEncodingFormat = config.remoteIdEncodingFormat
   result = IpDhcpRelay()
   drGlobalModeConfig = config.dhcpRelayGlobalModeConfig
   # DhcpRelay runnability only depends on alwaysOn and
   # exsitence of config.intfConfig
   # pylint: disable=too-many-nested-blocks
   if not ( config.alwaysOn or config.intfConfig or
            ( drGlobalModeConfig and drStatus().intfStatus ) ):
      result.activeState = False
   else:
      result.activeState = True
      result.alwaysOn = config.alwaysOn
      result.option82 = infoOpt
      result.linkLayerAddrOpt = llAddrOpt
      result.remoteIdEncodingFormat = remoteIdEncodingFormat
      result.smartRelay = config.smartRelayGlobal
      result.allSubnetsV6 = config.allSubnetsV6Global
      result.tunnelReqDisable = drStatus().tunnelReqDisable
      if drGlobalModeConfig:
         if ( drGlobalModeConfig.serverIpGlobal or
              drGlobalModeConfig.serverHostnameGlobal ):

            defaultServersShowModel = DefaultServersShowModel()
            v4Helpers = sorted( drGlobalModeConfig.serverIpGlobal,
                                cmp=DhcpRelayVrfCliLib.compareIpAddrVrf )
            hostnameHelpers = sorted( drGlobalModeConfig.serverHostnameGlobal,
                                      cmp=DhcpRelayVrfCliLib.compareHostnameVrf )
            defaultServersShowModel.helpersV4 += getAllV4Helpers( v4Helpers )
            defaultServersShowModel.helpersV4 += ( 
                                    getAllHostnameHelpers( hostnameHelpers ) )
            result.defaultServers = defaultServersShowModel
         else:
            # This is needed so that it does not show in the JSON if not set
            result.defaultServers = None
      
      result.configuredInterfaces = {}
      # Looping through status.intfStatus in order to differentiate
      # between interfaces configured by dhcp relay mode vs interface mode
      for intfId in Arnet.sortIntf( drStatus().intfStatus ):
         intfConfig = config.intfConfig.get( intfId )
         drIntfStatus = drStatus().intfStatus[ intfId ]
         intfUsingGlobalModeServersV4 = drIntfStatus.usingGlobalModeServersV4
         if intfConfig:
            # Has interface mode configurations
            # BUT it might still have dhcp relay global mode configurations
            cfgdIntf = ConfiguredInterface()
            if infoOpt or ( intfConfig.circuitId != intfConfig.name ):
               cfgdIntf.circuitId = intfConfig.circuitId
            if intfConfig.smartRelay == "srDefault":
               cfgdIntf.smartRelay = config.smartRelayGlobal
            elif intfConfig.smartRelay == "srOn":
               cfgdIntf.smartRelay = True
            elif intfConfig.smartRelay == "srOff":
               cfgdIntf.smartRelay = False
            if intfConfig.allSubnetsV6 == "srDefault":
               cfgdIntf.allSubnetsV6 = config.allSubnetsV6Global
            elif intfConfig.allSubnetsV6 == "srOn":
               cfgdIntf.allSubnetsV6 = True
            elif intfConfig.allSubnetsV6 == "srOff":
               cfgdIntf.allSubnetsV6 = False
            cfgdIntf.helpers = []
            
            #IPv4
            v4Helpers = []
            hostnameHelpers = []
            if intfConfig.serverIp or intfConfig.serverHostname:
               v4Helpers = sorted( intfConfig.serverIp, \
                                      cmp=DhcpRelayVrfCliLib.compareIpAddrVrf )
               hostnameHelpers = sorted( intfConfig.serverHostname, \
                                       cmp=DhcpRelayVrfCliLib.compareHostnameVrf )
               assert v4Helpers + hostnameHelpers
            # IPv6
            v6Helpers = []
            if intfConfig.serverIp6:
               v6Helpers = sorted( intfConfig.serverIp6, \
                                      cmp=DhcpRelayVrfCliLib.compareIpv6AddrVrf )
               assert v6Helpers
            
            cfgdIntf.helpers += getAllV4Helpers( v4Helpers )
            cfgdIntf.helpers += getAllV6Helpers( v6Helpers, intfConfig )
            cfgdIntf.helpers += getAllHostnameHelpers( hostnameHelpers )
            if intfUsingGlobalModeServersV4:
               # This is needed so that it does not show in the JSON if not set
               cfgdIntf.usingGlobalModeServersV4 = True

            
            cfgdIntf.disabledV4 = intfConfig.disabledV4
            result.configuredInterfaces[ intfId ] = cfgdIntf
            if intfConfig.disabledV4 or v6Helpers:
               # no need to check for warnings
               continue

            result.configuredInterfaces[ intfId ] = cfgdIntf
            # check if the interface does not have any server configured
            if not ( intfUsingGlobalModeServersV4 or
                     ( v4Helpers or hostnameHelpers ) or
                     v6Helpers ):
               warning = "No DHCP server configured for interface {}"
               mode.addWarning( warning.format( intfId ) )

         else:
            # This intf only has dhcp relay global mode configurations
            if DhcpRelayToggleLib.toggleDhcpRelayGlobalModeServerCommandEnabled():
               if not ( drGlobalModeConfig and intfUsingGlobalModeServersV4 ):
                  continue
               cfgdIntf = ConfiguredInterface()
               cfgdIntf.usingGlobalModeServersV4 = intfUsingGlobalModeServersV4
               result.configuredInterfaces[ intfId ] = cfgdIntf

   return result

class HelperAddressCmd( ShowCommand.ShowCliCommandClass ):
   syntax = 'show ADDR_FAMILY helper-address'
   data = {
      'ADDR_FAMILY' : addressFamilyExprForShow,
      'helper-address' : 'DHCP server status and configuration',
   }

   handler = showIpDhcpRelay
   cliModel = IpDhcpRelay

BasicCli.addShowCommandClass( HelperAddressCmd )

#--------------------------------------------------------------------------------
# show ip dhcp relay
#--------------------------------------------------------------------------------
class IpDhcpRelayCmd( ShowCommand.ShowCliCommandClass ):
   syntax = 'show ip dhcp relay'
   data = {
      'ip' : ipMatcherForShow,
      'dhcp' : dhcpMatcherForShow,
      'relay' : matcherRelay,
   }

   handler = showIpDhcpRelay
   cliModel = IpDhcpRelay

BasicCli.addShowCommandClass( IpDhcpRelayCmd )

#--------------------------------------------------------------------------------
# show ( ip | ipv6 ) dhcp relay access-list [ ACLNAME ]
#--------------------------------------------------------------------------------
def showAcl( mode, args ):
   aclType = args[ 'ADDR_FAMILY' ]
   name = args.get( '<aclNameExpr>' )
   return AclCli.showServiceAcl( mode,
                                 aclCpConfig,
                                 aclStatus,
                                 aclCheckpoint,
                                 aclType,
                                 name,
                                 serviceName='dhcpRelay' )

class IpDhcpRelayAccessListCmd( ShowCommand.ShowCliCommandClass ):
   syntax = 'show ADDR_FAMILY dhcp relay access-list [ ACLNAME ]'
   data = {
      'ADDR_FAMILY' : addressFamilyExprForShow,
      'dhcp' : dhcpMatcherForShow,
      'relay' : matcherRelay,
      'access-list' : AclCli.accessListMatcher,
      'ACLNAME' : AclCli.ipOrIpv6AclNameExpression,
   }

   handler = showAcl
   cliModel = AllAclList

BasicCli.addShowCommandClass( IpDhcpRelayAccessListCmd )

#--------------------------------------------------------------------------------
# show ip dhcp snooping
#--------------------------------------------------------------------------------
def showDhcpSnooping( mode, args ):
   snoop = DhcpSnoopingModel()
   snoop.enabled = dhcpSnoopingConfig.enabled
   snoop.operational = dhcpSnoopingStatus.enabled
   snoop.enabledVlans = dhcpSnoopingConfig.vlan.keys()
   snoop.operationalVlans = dhcpSnoopingStatus.vlanStatus.keys()
   snoop.option82Enabled = dhcpSnoopingConfig.informationOption
   if DhcpRelayToggleLib.toggleDhcpSnoopingBridgingV4Enabled():
      snoop.bridgingEnabled = dhcpSnoopingConfig.bridging

   if dhcpSnoopingConfig.informationOption:
      snoop.circuitIdEnabled = dhcpSnoopingConfig.circuitIdTypeValid
      if dhcpSnoopingConfig.circuitIdTypeValid:
         snoop.circuitIdType = dhcpSnoopingConfig.circuitIdType
      snoop.circuitIdFormat = dhcpSnoopingConfig.circuitIdFormatString

      for intfName in dhcpSnoopingConfig.userDefinedCircuitId:
         circuitIdModel = CircuitIdModel()
         userConfigIntf = dhcpSnoopingConfig.userDefinedCircuitId[ intfName ]
         circuitIdModel.name = intfName
         circuitIdModel.valid = userConfigIntf.circuitIdTypeValid
         circuitIdModel.value = userConfigIntf.circuitIdVal
         if userConfigIntf.circuitIdTypeValid:
            circuitIdModel.circuitType = userConfigIntf.circuitIdType
         snoop.circuitIds[ intfName ] = circuitIdModel
      snoop.bridgeMac = bridgingConfig.bridgeMacAddr      
   return snoop

class IpDhcpSnoopingCmd( ShowCommand.ShowCliCommandClass ):
   syntax = 'show ip dhcp snooping'
   data = {
      'ip' : ipMatcherForShow,
      'dhcp' : dhcpMatcherForShow,
      'snooping' : nodeSnooping,
   }

   cliModel = DhcpSnoopingModel
   handler = showDhcpSnooping

BasicCli.addShowCommandClass( IpDhcpSnoopingCmd )

#--------------------------------------------------------------------------------
# show ip dhcp snooping counters
#--------------------------------------------------------------------------------
def updateCounters( mode ):
   dhcpSnoopingCounterConfig.counterUpdateRequestTime = Tac.now()
   def countersUpdated():
      return ( dhcpSnoopingStatus.counterUpdateTime >=
               dhcpSnoopingCounterConfig.counterUpdateRequestTime )
   try:
      Tac.waitFor( countersUpdated, description="counter update",
                   maxDelay=0.1, sleep=True, timeout=30.0 )
   except Tac.Timeout:
      mode.addWarning( "Displaying stale counters" )

def showDhcpSnoopingCounters( mode, args ):
   counterModel = DhcpSnoopingCounterModel()
   counterModel.enabled = dhcpSnoopingConfig.enabled
   if counterModel.enabled:
      updateCounters( mode )
      for vlan in dhcpSnoopingStatus.vlanStatus:
         vlanStatus = dhcpSnoopingStatus.vlanStatus[ vlan ]
         vlanCounters = VlanCounters()
         vlanCounters.vlan = vlan
         vlanCounters.requestsReceived = vlanStatus.requestsReceived
         vlanCounters.requestsForwarded = vlanStatus.requestsForwarded
         vlanCounters.requestsDropped = (
               vlanStatus.requestsReceived - vlanStatus.requestsForwarded )
         vlanCounters.repliesReceived = vlanStatus.repliesReceived
         vlanCounters.repliesForwarded = vlanStatus.repliesForwarded
         vlanCounters.repliesDropped = (
               vlanStatus.repliesReceived - vlanStatus.repliesForwarded )
         vlanCounters.resetTime = Ark.switchTimeToUtc( vlanStatus.lastResetTime )
         counterModel.vlanCounters[ vlan ] = vlanCounters
   return counterModel
   
class IpDhcpSnoopingCountersCmd( ShowCommand.ShowCliCommandClass ):
   syntax = 'show ip dhcp snooping counters'
   data = {
      'ip' : ipMatcherForShow,
      'dhcp' : dhcpMatcherForShow,
      'snooping' : nodeSnooping,
      'counters' : matcherCounters,
   }

   cliModel = DhcpSnoopingCounterModel
   handler = showDhcpSnoopingCounters

BasicCli.addShowCommandClass( IpDhcpSnoopingCountersCmd )

#--------------------------------------------------------------------------------
# show ip dhcp snooping counters debug
#--------------------------------------------------------------------------------
def showDhcpSnoopingDebugCounters( mode, args ):
   debugCounterModel = DhcpSnoopingDebugCounterModel()
   debugCounterModel.enabled = dhcpSnoopingConfig.enabled

   if dhcpSnoopingConfig.enabled:
      updateCounters( mode )
      counters = dhcpSnoopingStatus.debugCounters
      if not counters:
         return debugCounterModel

      receivedCounters = SnoopingRelayCounters()
      receivedCounters.snoopingToRelay = counters.snoopingToRelayReceived
      receivedCounters.relayToSnooping = counters.relayToSnoopingReceived
      debugCounterModel.receivedCounters = receivedCounters

      forwardedCounters = SnoopingRelayCounters()
      forwardedCounters.snoopingToRelay = counters.snoopingToRelayForwarded
      forwardedCounters.relayToSnooping = counters.relayToSnoopingForwarded
      debugCounterModel.forwardedCounters = forwardedCounters

      vlanIdErrCounters = SnoopingRelayCounters()
      vlanIdErrCounters.snoopingToRelay = counters.snoopingToRelayDropVlanIdErr
      vlanIdErrCounters.relayToSnooping = counters.relayToSnoopingDropVlanIdErr
      debugCounterModel.vlanIdErrCounters = vlanIdErrCounters

      parseErrCounters = SnoopingRelayCounters()
      parseErrCounters.snoopingToRelay = counters.snoopingToRelayDropParseErr
      parseErrCounters.relayToSnooping = counters.relayToSnoopingDropParseErr
      debugCounterModel.parseErrCounters = parseErrCounters

      dhcpOpErrCounters = SnoopingRelayCounters()
      dhcpOpErrCounters.snoopingToRelay = counters.snoopingToRelayDropDhcpOpErr
      dhcpOpErrCounters.relayToSnooping = counters.relayToSnoopingDropDhcpOpErr
      debugCounterModel.dhcpOpErrCounters = dhcpOpErrCounters

      infoOptErrCounters = SnoopingRelayCounters()
      infoOptErrCounters.snoopingToRelay = counters.snoopingToRelayDropInfoOptErr
      infoOptErrCounters.relayToSnooping = counters.relayToSnoopingDropInfoOptErr
      debugCounterModel.infoOptErrCounters = infoOptErrCounters

      disabledErrCounters = SnoopingRelayCounters()
      disabledErrCounters.snoopingToRelay = counters.snoopingToRelayDropDisabled
      disabledErrCounters.relayToSnooping = counters.relayToSnoopingDropDisabled
      debugCounterModel.disabledErrCounters = disabledErrCounters

      debugCounterModel.resetTime = Ark.switchTimeToUtc( counters.lastResetTime ) 
   
   return debugCounterModel

class IpDhcpSnoopingCountersDebugCmd( ShowCommand.ShowCliCommandClass ):
   syntax = 'show ip dhcp snooping counters debug'
   data = {
      'ip' : ipMatcherForShow,
      'dhcp' : dhcpMatcherForShow,
      'snooping' : nodeSnooping,
      'counters' : matcherCounters,
      'debug' : matcherDebug,
   }

   cliModel = DhcpSnoopingDebugCounterModel
   handler = showDhcpSnoopingDebugCounters

BasicCli.addShowCommandClass( IpDhcpSnoopingCountersDebugCmd )

#--------------------------------------------------------------------------------
# show ip dhcp snooping hardware
#--------------------------------------------------------------------------------
def showDhcpSnoopingHardware( mode, args ):
   hardwareModel = DhcpSnoopingHardwareModel()

   for sliceId, hwStatus in dhcpSnoopingHwStatusDir.iteritems():
      sliceInfo = platformHardwareSliceDir.sliceInfo.get( sliceId )
      if not sliceInfo or sliceInfo and sliceInfo.generationId != hwStatus.genId:
         continue

      if hwStatus.hwDhcpSnoopingEnabled:
         hardwareModel.enabled = True
      vlanList = VlanList()
      vlanList.vlans = hwStatus.snoopingVlan.keys()
      hardwareModel.vlansPerSlice[ sliceId ] = vlanList
      for vlan in hwStatus.snoopingVlan.keys():
         if vlan not in hardwareModel.vlans:
            hardwareModel.vlans.append( vlan )
   if hardwareModel.enabled is None:
      hardwareModel.enabled = False
   return hardwareModel

class IpDhcpSnoopingHardwareCmd( ShowCommand.ShowCliCommandClass ):
   syntax = 'show ip dhcp snooping hardware'
   data = {
      'ip' : ipMatcherForShow,
      'dhcp' : dhcpMatcherForShow,
      'snooping' : nodeSnooping,
      'hardware' : 'DHCP Snooping hardware status',
   }

   cliModel = DhcpSnoopingHardwareModel
   handler = showDhcpSnoopingHardware

BasicCli.addShowCommandClass( IpDhcpSnoopingHardwareCmd )

#--------------------------------------------------------------------------------
# show ipv6 dhcp relay installed routes
#--------------------------------------------------------------------------------
def showInstalledRoutes( mode, args ):
   fields = [ 'Prefix', 'Client Address', 'Interface', \
              'Remaining Lifetime(seconds)' ]
   print( "%-40s%-40s%-20s%-20s" % ( fields[ 0 ], fields[ 1 ],
                                     fields[ 2 ], fields[ 3 ] ) )
   print( "%-40s%-40s%-20s%-20s" % ( len( fields[ 0 ] ) * "-",
          len( fields[ 1 ] ) * "-", len( fields[ 2 ] ) * "-",
          len( fields[ 3 ] ) * "-" ) )
   
   prefixesPrinted = {}
   
   currTime = Tac.utcNow()
   def printRoutes( intfStatus ):
      for assignedAddr in intfStatus.prefixBinding:
         # Do not display the same prefix twice. This we do for the
         # cases where MLAG Split brain gets restored. In that case the
         # same prefix binding would be present in both dhcpRelayStatus
         # and mlagDhcpRelayStatus. We display the one present in dhcpRelayStatus
         # only.
         pb = intfStatus.prefixBinding.get( assignedAddr )
         if assignedAddr in prefixesPrinted or not pb:
            continue
         addrAttrs = pb.addrAttrs
         expiryTime = addrAttrs.allotmentTime + addrAttrs.validLifetime
         remainingTime = expiryTime - currTime
         prefixesPrinted[ assignedAddr ] = True
         print ( "%-40s%-40s%-20s%-20d" % ( assignedAddr, addrAttrs.clientAddr,
                 intfStatus.intfId, remainingTime ) )

   intfs = drStatus().intfStatus
   mlagIntfs = drMlagStatus().intfStatus
   for intfname in Arnet.sortIntf( intfs ):
      intfStatus = intfs[ intfname ]
      printRoutes( intfStatus )

   for intfname in Arnet.sortIntf( mlagIntfs ):
      intfStatus = mlagIntfs[ intfname ]
      printRoutes( intfStatus )

class Ipv6DhcpRelayInstalledRoutesCmd( ShowCommand.ShowCliCommandClass ):
   syntax = 'show ipv6 dhcp relay installed routes'
   data = {
      'ipv6' : ipv6MatcherForShow,
      'dhcp' : dhcpMatcherForShow,
      'relay' : matcherRelay,
      'installed' : 'Installed routes for DHCP server assigned addresses',
      'routes' : 'Install routes for DHCP server assigned addresses',
   }

   handler = showInstalledRoutes

BasicCli.addShowCommandClass( Ipv6DhcpRelayInstalledRoutesCmd )

#--------------------------------------------------------------------------------
# show ipv6 dhcp snooping
#--------------------------------------------------------------------------------
def showDhcp6Snooping( mode, args ):
   snoop = Dhcp6SnoopingModel()
   snoop.enabled = dhcp6SnoopingConfig.enabled
   snoop.operational = dhcp6SnoopingStatus.enabled
   snoop.enabledVlans = dhcp6SnoopingConfig.vlan.keys()
   snoop.operationalVlans = dhcp6SnoopingStatus.vlanStatus.keys()
   snoop.remoteIdOption = dhcp6SnoopingConfig.remoteIdOption
   return snoop

class Ipv6DhcpSnoopingCmd( ShowCommand.ShowCliCommandClass ):
   syntax = 'show ipv6 dhcp snooping'
   data = {
      'ipv6' : ipv6MatcherForShow,
      'dhcp' : dhcpMatcherForShow,
      'snooping' : nodeSnoopingV6,
   }

   cliModel = Dhcp6SnoopingModel
   handler = showDhcp6Snooping

BasicCli.addShowCommandClass( Ipv6DhcpSnoopingCmd )

#--------------------------------------------------------------------------------
# show ipv6 dhcp snooping counters
#--------------------------------------------------------------------------------
def updateCountersV6( mode ):
   dhcp6SnoopingCounterConfig.counterUpdateRequestTime = Tac.now()
   def countersUpdated():
      return ( dhcp6SnoopingStatus.counterUpdateTime >=
               dhcp6SnoopingCounterConfig.counterUpdateRequestTime )
   try:
      Tac.waitFor( countersUpdated, description="counter update",
                   maxDelay=0.1, sleep=True, timeout=30.0 )
   except Tac.Timeout:
      mode.addWarning( "Displaying stale counters" )

def showDhcp6SnoopingCounters( mode, args ):
   counterModel = Dhcp6SnoopingCounterModel()
   counterModel.enabled = dhcp6SnoopingConfig.enabled
   if counterModel.enabled:
      updateCountersV6( mode )
      for vlan in dhcp6SnoopingStatus.vlanStatus:
         vlanStatus = dhcp6SnoopingStatus.vlanStatus[ vlan ]
         vlanCounters = VlanCounters()
         vlanCounters.vlan = vlan
         vlanCounters.requestsReceived = vlanStatus.requestsReceived
         vlanCounters.requestsForwarded = vlanStatus.requestsForwarded
         vlanCounters.requestsDropped = (
               vlanStatus.requestsReceived - vlanStatus.requestsForwarded )
         vlanCounters.repliesReceived = vlanStatus.repliesReceived
         vlanCounters.repliesForwarded = vlanStatus.repliesForwarded
         vlanCounters.repliesDropped = (
               vlanStatus.repliesReceived - vlanStatus.repliesForwarded )
         vlanCounters.resetTime = Ark.switchTimeToUtc( vlanStatus.lastResetTime )
         counterModel.vlanCounters[ vlan ] = vlanCounters
   return counterModel
 
class Ipv6DhcpSnoopingCountersCmd( ShowCommand.ShowCliCommandClass ):
   syntax = 'show ipv6 dhcp snooping counters'
   data = {
      'ipv6' : ipv6MatcherForShow,
      'dhcp' : dhcpMatcherForShow,
      'snooping' : nodeSnoopingV6,
      'counters' : matcherCounters,
   }

   cliModel = Dhcp6SnoopingCounterModel
   handler = showDhcp6SnoopingCounters

BasicCli.addShowCommandClass( Ipv6DhcpSnoopingCountersCmd )

#--------------------------------------------------------------------------------
# show ipv6 dhcp snooping counters debug
#--------------------------------------------------------------------------------
def showDhcp6SnoopingDebugCounters( mode, args ):
   debugCounterModel = Dhcp6SnoopingDebugCounterModel()
   debugCounterModel.enabled = dhcp6SnoopingConfig.enabled

   if dhcp6SnoopingConfig.enabled:
      updateCountersV6( mode )
      counters = dhcp6SnoopingStatus.debugCounters
      if not counters:
         return debugCounterModel

      receivedCounters = SnoopingRelayCounters()
      receivedCounters.snoopingToRelay = counters.snoopingToRelayReceived
      receivedCounters.relayToSnooping = counters.relayToSnoopingReceived
      debugCounterModel.receivedCounters = receivedCounters

      forwardedCounters = SnoopingRelayCounters()
      forwardedCounters.snoopingToRelay = counters.snoopingToRelayForwarded
      forwardedCounters.relayToSnooping = counters.relayToSnoopingForwarded
      debugCounterModel.forwardedCounters = forwardedCounters

      vlanIdErrCounters = SnoopingRelayCounters()
      vlanIdErrCounters.snoopingToRelay = counters.snoopingToRelayDropVlanIdErr
      vlanIdErrCounters.relayToSnooping = counters.relayToSnoopingDropVlanIdErr
      debugCounterModel.vlanIdErrCounters = vlanIdErrCounters

      parseErrCounters = SnoopingRelayCounters()
      parseErrCounters.snoopingToRelay = counters.snoopingToRelayDropParseErr
      parseErrCounters.relayToSnooping = counters.relayToSnoopingDropParseErr
      debugCounterModel.parseErrCounters = parseErrCounters

      dhcpOpErrCounters = SnoopingRelayCounters()
      dhcpOpErrCounters.snoopingToRelay = counters.snoopingToRelayDropDhcpOpErr
      dhcpOpErrCounters.relayToSnooping = counters.relayToSnoopingDropDhcpOpErr
      debugCounterModel.dhcpOpErrCounters = dhcpOpErrCounters

      remoteIdErrCounters = SnoopingRelayCounters()
      remoteIdErrCounters.snoopingToRelay = counters.snoopingToRelayDropInfoOptErr
      remoteIdErrCounters.relayToSnooping = counters.relayToSnoopingDropInfoOptErr
      debugCounterModel.remoteIdErrCounters = remoteIdErrCounters

      disabledErrCounters = SnoopingRelayCounters()
      disabledErrCounters.snoopingToRelay = counters.snoopingToRelayDropDisabled
      disabledErrCounters.relayToSnooping = counters.relayToSnoopingDropDisabled
      debugCounterModel.disabledErrCounters = disabledErrCounters

      debugCounterModel.resetTime = Ark.switchTimeToUtc( counters.lastResetTime ) 

   return debugCounterModel

class Ipv6DhcpSnoopingCountersDebugCmd( ShowCommand.ShowCliCommandClass ):
   syntax = 'show ipv6 dhcp snooping counters debug'
   data = {
      'ipv6' : ipv6MatcherForShow,
      'dhcp' : dhcpMatcherForShow,
      'snooping' : nodeSnoopingV6,
      'counters' : matcherCounters,
      'debug' : matcherDebug,
   }

   cliModel = Dhcp6SnoopingDebugCounterModel
   handler = showDhcp6SnoopingDebugCounters

BasicCli.addShowCommandClass( Ipv6DhcpSnoopingCountersDebugCmd )

#--------------------------------------------------------------------------------
# show ipv6 dhcp snooping hardware
#--------------------------------------------------------------------------------
def showDhcp6SnoopingHardware( mode, args ):
   hardwareModel = Dhcp6SnoopingHardwareModel()

   for sliceId, hwStatus in dhcp6SnoopingHwStatusDir.iteritems():
      sliceInfo = platformHardwareSliceDir.sliceInfo.get( sliceId )
      if not sliceInfo or sliceInfo and sliceInfo.generationId != hwStatus.genId:
         continue

      if hwStatus.hwDhcpSnoopingEnabled:
         hardwareModel.enabled = True
      vlanList = VlanList()
      vlanList.vlans = hwStatus.snoopingVlan.keys()
      hardwareModel.vlansPerSlice[ sliceId ] = vlanList
      for vlan in hwStatus.snoopingVlan.keys():
         if vlan not in hardwareModel.vlans:
            hardwareModel.vlans.append( vlan )
   if hardwareModel.enabled is None:
      hardwareModel.enabled = False
   return hardwareModel

class Ipv6DhcpSnoopingHardwareCmd( ShowCommand.ShowCliCommandClass ):
   syntax = 'show ipv6 dhcp snooping hardware'
   data = {
      'ipv6' : ipv6MatcherForShow,
      'dhcp' : dhcpMatcherForShow,
      'snooping' : nodeSnoopingV6,
      'hardware' : 'DHCPv6 Snooping hardware status',
   }

   cliModel = Dhcp6SnoopingHardwareModel
   handler = showDhcp6SnoopingHardware

BasicCli.addShowCommandClass( Ipv6DhcpSnoopingHardwareCmd )

def Plugin( entityManager ):
   global bridgingConfig
   global dhcpSnoopingConfig
   global dhcpSnoopingCounterConfig
   global dhcpSnoopingStatus
   global dhcpSnoopingHwStatusDir
   global platformHardwareSliceDir
   global dhcp6SnoopingConfig
   global dhcp6SnoopingCounterConfig
   global dhcp6SnoopingStatus
   global dhcp6SnoopingHwStatusDir
   global aclCpConfig
   global aclStatus
   global aclCheckpoint

   bridgingConfig = LazyMount.mount( entityManager, "bridging/config",
                                     "Bridging::Config", "r" )
   dhcpSnoopingConfig = ConfigMount.mount( entityManager,
                                           "bridging/dhcpsnooping/config",
                                           "Bridging::DhcpSnooping::Config", "w" )
   dhcpSnoopingCounterConfig = LazyMount.mount(
         entityManager, "bridging/dhcpsnooping/counterConfig",
         "Bridging::DhcpSnooping::CounterConfig", "w" )
   dhcpSnoopingStatus = LazyMount.mount( entityManager,
                                         "bridging/dhcpsnooping/status",
                                         "Bridging::DhcpSnooping::Status", "r" )
   dhcpSnoopingHwStatusDir = LazyMount.mount(
         entityManager, "bridging/dhcpsnooping/hardware/status",
         "Tac::Dir", "ri" )
   dhcp6SnoopingCounterConfig = LazyMount.mount(
         entityManager, "bridging/dhcpsnooping/dhcp6CounterConfig",
         "Bridging::DhcpSnooping::CounterConfig", "w" )
   dhcp6SnoopingConfig = ConfigMount.mount( entityManager,
                                            "bridging/dhcpsnooping/dhcp6Config",
                                            "Bridging::DhcpSnooping::Dhcp6Config",
                                            "w" )
   dhcp6SnoopingStatus = LazyMount.mount( entityManager,
                                          "bridging/dhcpsnooping/dhcp6Status",
                                          "Bridging::DhcpSnooping::Status", "r" )
   dhcp6SnoopingHwStatusDir = LazyMount.mount(
         entityManager, "bridging/dhcpsnooping/hardware/dhcp6Status",
         "Tac::Dir", "ri" )
   platformHardwareSliceDir = LazyMount.mount( entityManager,
                                               "platform/hardware/slice",
                                               "Hardware::PlatformSliceDir", "r" )
   aclCpConfig = ConfigMount.mount( entityManager, "acl/cpconfig/cli",
                                    "Acl::Input::CpConfig", "w" )
   aclStatus = LazyMount.mount( entityManager, "acl/status/all",
                                "Acl::Status", "r" )
   aclCheckpoint = LazyMount.mount( entityManager, "acl/checkpoint",
                                    "Acl::CheckpointStatus", "w" )
