# Copyright (c) 2011 Arista Networks, Inc.  All rights reserved.
# Arista Networks, Inc. Confidential and Proprietary.

#-------------------------------------------------------------------------------
# This module implements IP DHCP Relay configuration.
# It contains interface-specific commands.
#-------------------------------------------------------------------------------

# Unused arguments 'mode' and 'entityManager'
# pylint: disable-msg=W0613

import CliCommand
import CliMatcher, IntfCli, LazyMount, Tac
import ConfigMount
import IraIpIntfCli, HostnameCli
import Arnet
import socket
import CliPlugin.Ip6AddrMatcher as Ip6AddrMatcher
import CliPlugin.IpAddrMatcher as IpAddrMatcher
from CliPlugin.VrfCli import VrfExprFactory, DEFAULT_VRF
from CliToken.Dhcp import dhcpMatcherForConfig
from CliToken.Ip import ipMatcherForConfigIf
from CliToken.Ipv6 import ipv6MatcherForConfigIf
import Toggles.DhcpRelayToggleLib as DhcpRelayToggleLib

# true if call given from forwarder CLI, used to bypass competing relay check
fromForwarder = False

dhcpRelayConfigDir = None
dhcpRelayCounterConfigDir = None
dhcpFwdConfig = None
dhcpRelayStatusDir = None
aclConfig = None
aclCpConfig = None
aclStatus = None
aclCheckpoint = None
mlagDhcpRelayStatusDir = None

# don't allow syntax from native relay and forwarder to be active at the same time
# in order to switch syntaxes, config must be reset to defaults using currently
# used syntax
def configConflict( mode ):
   if fromForwarder:
      return False
   if dhcpFwdConfig.serverIp or \
          dhcpFwdConfig.serverIntf or \
          dhcpFwdConfig.intfConfig or \
          dhcpFwdConfig.logVerbose:
      mode.addWarning(
         "To configure dhcp helper addresses first remove incompatible configuration"
         " of the form 'ip[v6] dhcp relay [ client | server ] ...'" )
      return True
   return False

def drConfig():
   return dhcpRelayConfigDir

def drCounterConfig():
   return dhcpRelayCounterConfigDir

def drStatus():
   return dhcpRelayStatusDir

def drMlagStatus():
   return mlagDhcpRelayStatusDir

def _drIntfConfig( mode ):
   intfName = mode.intf.name
   intfConfig = drConfig().intfConfig.get( intfName )
   return intfConfig

def _getOrCreateDrIntfConfig( mode ):
   intfName = mode.intf.name
   intfConfig = _drIntfConfig( mode )
   if intfConfig is None:
      intfConfig = drConfig().intfConfig.newMember( intfName )
   return intfConfig

def _deleteDrIntfConfig( mode ):
   intfName = mode.intf.name
   if _drIntfConfig( mode ) is not None:
      del drConfig().intfConfig[ intfName ]

def _deleteDrIntfConfigIfAllAttributeHaveDefaults( mode ):
   intfConfig = _drIntfConfig( mode )
   if intfConfig is None:
      return

   if ( not intfConfig.serverIp and
        not intfConfig.serverIp6 and
        not intfConfig.serverHostname and
        not intfConfig.ccapCores and
        not intfConfig.disabledV4 and 
        intfConfig.circuitId == intfConfig.name and
        intfConfig.route6Install is False and
        intfConfig.smartRelay == 'srDefault' and
        intfConfig.allSubnetsV6 == 'srDefault' ):
      _deleteDrIntfConfig( mode )

def vrfHostnamePair( vrf, hostname ):
   return Tac.Value( "Ip::Helper::DhcpRelay::VrfHostnamePair", vrf,
                     hostname )

def vrfHostnameSrcIntfSrcIp( vrf, hostname, srcIntf, srcIp ):
   intfId = Tac.Value( 'Arnet::IntfId', '' )
   if srcIntf:
      intfId = Tac.Value( 'Arnet::IntfId', srcIntf.name )
   return Tac.Value( "Ip::Helper::DhcpRelay::VrfHostnameSrcIntfSrcIp", vrf,
                     hostname, intfId, Arnet.IpGenAddr( str( srcIp ) ) )

def vrfIpSrcIntfSrcIp( vrf, ip, srcIntf, srcIp ):
   intfId = Tac.Value( 'Arnet::IntfId', '' )
   if srcIntf:
      intfId = Tac.Value( 'Arnet::IntfId', srcIntf.name )
   return Tac.Value( "Ip::Helper::DhcpRelay::VrfIpSrcIntfSrcIp", vrf,
                     Arnet.IpGenAddr( str( ip ) ), intfId,
                     Arnet.IpGenAddr( str( srcIp ) ) )

# ip helper addresses are not exactly a routing protocol, but they apply to
# the same set of interfaces (e.g. not management or loopback)
modelet = IraIpIntfCli.RoutingProtocolIntfConfigModelet

matcherHelper = CliMatcher.KeywordMatcher(
   'helper-address', helpdesc='Specify DHCP Server' )
matcherRelay = CliMatcher.KeywordMatcher(
   'relay', helpdesc='DHCP Relay' )
matcherDest = CliMatcher.KeywordMatcher(
   'destination', helpdesc='DHCP Relay destination address' )

#-------------------------------------------------------------------------------
# "[no|default] ip helper-address" interface command.
#-------------------------------------------------------------------------------
def noDhcpRelay( mode, args ):
   # disallow removing helper-addresses in native relay in order to maintain
   # consistency between forwarder and relay configs
   if configConflict( mode ):
      return
   intfConfig = _drIntfConfig ( mode )
   if intfConfig:
      intfConfig.serverIp.clear()
      intfConfig.serverHostname.clear()
      _deleteDrIntfConfigIfAllAttributeHaveDefaults( mode )

class IpHelperAddressCmd( CliCommand.CliCommandClass ):
   noOrDefaultSyntax = 'ip helper-address'
   data = {
      'ip' : ipMatcherForConfigIf,
      'helper-address' : matcherHelper,
   }

   noOrDefaultHandler = noDhcpRelay

modelet.addCommandClass( IpHelperAddressCmd )

#-------------------------------------------------------------------------------
# The "[no|default] ip helper-address <ip address/hostname> vrf <vrfName>
# [ source-interface <interface> | source-address <ip address> ]"
# interface command.
#-------------------------------------------------------------------------------
def setIntfDhcpRelay( mode, addr, no=None, vrfName=None, srcIntfIp=None ):
   # This could be converted to ( mode, args ) form, but wasn't because this
   # function is also called from OldDhcpRelayCli (via the callNativeRelay function)
   if configConflict( mode ):
      return
   ipIntfConfig = ipConfig.ipIntfConfig.get( mode.intf.name )
   if not vrfName:
      vrfName = DEFAULT_VRF if ipIntfConfig is None or ipIntfConfig.vrf == "" \
          else ipIntfConfig.vrf
   ipAddr = '0.0.0.0' # INADDR_ANY
   srcIp = '0.0.0.0' # INADDR_ANY
   srcIntf = ''
   vrfHostname = ''

   if no:
      intfConfig = _drIntfConfig( mode )
      if not intfConfig:
         return
   else:
      intfConfig = _getOrCreateDrIntfConfig( mode )

   if srcIntfIp:
      try:
         srcIp = Arnet.IpAddress( srcIntfIp )
      except TypeError:
         srcIntf = srcIntfIp

   # Remove existing VrfIpSrcIntfSrcIp entry where only src intf or src ip changed
   try:
      ipAddr = Arnet.IpAddress( addr )
      vrfIp = vrfIpSrcIntfSrcIp( vrfName, ipAddr, srcIntf, srcIp )
      for key in intfConfig.serverIp:
         if key.vrfName == vrfIp.vrfName and \
            key.ip.v4Addr == vrfIp.ip.v4Addr:
            del intfConfig.serverIp[ key ]
   except ValueError:
      # pylint: disable-msg=E1103
      vrfHostname = vrfHostnameSrcIntfSrcIp( vrfName, addr, srcIntf, srcIp )
      for key in intfConfig.serverHostname:
         if key.vrfName == vrfHostname.vrfName and \
            key.hostname == vrfHostname.hostname:
            del intfConfig.serverHostname[ key ]

   if no:
      _deleteDrIntfConfigIfAllAttributeHaveDefaults( mode )
   else:
      if vrfHostname:
         intfConfig.serverHostname[ vrfHostname ] = True
      else:
         intfConfig.serverIp[ vrfIp ] = True 

vrfExprFactory = VrfExprFactory(
      helpdesc='Specify VRF of the DHCP server' )
matcherSourceAddr = CliMatcher.KeywordMatcher(
      'source-address',
      helpdesc=( 'Specify the source address to communicate '
                 'with the DHCP server' ) )

class IpHelperHostnameVrfSourceCmd( CliCommand.CliCommandClass ):
   syntax = ( "ip helper-address HOST_OR_ADDR [ VRF ] "
              "[ ( source-interface INTF ) | ( source-address ADDR ) ]" )
   noOrDefaultSyntax = syntax

   data = {
      'ip' : ipMatcherForConfigIf,
      'helper-address' : matcherHelper,
      'HOST_OR_ADDR' : HostnameCli.IpAddrOrHostnameMatcher( ipv6=False,
         helpdesc='Hostname or IP address of DHCP server' ),
      'VRF' : vrfExprFactory,
      'source-interface' : ( 'Specify the source interface to communicate '
                             'with the DHCP server' ),
      'INTF' : IntfCli.Intf.matcherWithIpSupport,
      'source-address' : matcherSourceAddr,
      'ADDR' : IpAddrMatcher.ipAddrMatcher,
   }

   @staticmethod
   def handler( mode, args ):
      intf = args.get( 'INTF' ) or args.get( 'ADDR' )
      setIntfDhcpRelay( mode, args[ 'HOST_OR_ADDR' ], False, 
            vrfName=args.get( 'VRF' ), srcIntfIp=intf )

   @staticmethod
   def noOrDefaultHandler( mode, args ):
      intf = args.get( 'INTF' ) or args.get( 'ADDR' )
      setIntfDhcpRelay( mode, args[ 'HOST_OR_ADDR' ], True,
            vrfName=args.get( 'VRF' ), srcIntfIp=intf )

modelet.addCommandClass( IpHelperHostnameVrfSourceCmd )

#-------------------------------------------------------------------------------
# "[no|default] ipv6 helper-address" interface command.
#-------------------------------------------------------------------------------

def noDhcpRelayv6( mode, args ):
   # disallow removing helper-addresses in native relay in order to maintain
   # consistency between forwarder and relay configs
   if configConflict( mode ):
      return
   intfConfig = _drIntfConfig ( mode )
   if intfConfig:
      intfConfig.serverIp6.clear()
      _deleteDrIntfConfigIfAllAttributeHaveDefaults( mode )

class Ipv6DhcpRelayDestCmd( CliCommand.CliCommandClass ):
   noOrDefaultSyntax = 'ipv6 dhcp relay destination'
   data = {
      'ipv6' : ipv6MatcherForConfigIf,
      'dhcp' : dhcpMatcherForConfig,
      'relay' : matcherRelay,
      'destination' : matcherDest,
   }
   
   noOrDefaultHandler = noDhcpRelayv6

class Ipv6HelperAddressCmd( CliCommand.CliCommandClass ):
   noOrDefaultSyntax = 'ipv6 helper-address'
   data = {
      'ipv6' : ipv6MatcherForConfigIf,
      'helper-address' : matcherHelper,
   }

   noOrDefaultHandler = noDhcpRelayv6

modelet.addCommandClass( Ipv6DhcpRelayDestCmd )
modelet.addCommandClass( Ipv6HelperAddressCmd )

#-------------------------------------------------------------------------------
# The "[no|default] ip dhcp smart-relay' interface command.
#-------------------------------------------------------------------------------

def setIntfSmartRelay( mode, args ):
   default = CliCommand.isDefaultCmd( args )
   no = CliCommand.isNoCmd( args )
   if configConflict( mode ):
      return
   if default:
      intfConfig = _drIntfConfig( mode )
   else:
      intfConfig = _getOrCreateDrIntfConfig( mode )
   if intfConfig is None:
      return
   if default:
      # The global config will be overriden to enable smart relay
      intfConfig.smartRelay = 'srDefault'
   elif no:
      # smart relay is disabled regardless of the global config
      intfConfig.smartRelay = 'srOff'
   else:
      intfConfig.smartRelay = 'srOn'
   _deleteDrIntfConfigIfAllAttributeHaveDefaults( mode )
   
class IpDhcpSmartRelayCmd( CliCommand.CliCommandClass ):
   syntax = 'ip dhcp smart-relay'
   noOrDefaultSyntax = syntax

   data = {
      'ip' : ipMatcherForConfigIf,
      'dhcp': dhcpMatcherForConfig,
      'smart-relay' : CliCommand.Node( matcher=CliMatcher.KeywordMatcher(
         'smart-relay',
         helpdesc='DHCP Smart Relay Agent Forwarding on this interface' ),
         deprecatedByCmd='ip dhcp relay all-subnets default' ),
   }

   handler = setIntfSmartRelay
   noOrDefaultHandler = handler

class IpDhcpRelaySubnetsCmd( CliCommand.CliCommandClass ):
   syntax = 'ip dhcp relay all-subnets'
   noOrDefaultSyntax = syntax

   data = {
      'ip' : ipMatcherForConfigIf,
      'dhcp' : dhcpMatcherForConfig,
      'relay' : matcherRelay,
      'all-subnets' : 'DHCP Relay Forwarding on all the subnets',
   }

   handler = setIntfSmartRelay
   noOrDefaultHandler = handler

modelet.addCommandClass( IpDhcpSmartRelayCmd )
modelet.addCommandClass( IpDhcpRelaySubnetsCmd )

#-------------------------------------------------------------------------------
# The "[no|default] ipv6 dhcp realy all-subnets' interface command.
#-------------------------------------------------------------------------------

def setIntfAllSubnetsRelayV6( mode, args ):
   default = CliCommand.isDefaultCmd( args )
   no = CliCommand.isNoCmd( args )
   if configConflict( mode ):
      return
   if default:
      intfConfig = _drIntfConfig( mode )
   else:
      intfConfig = _getOrCreateDrIntfConfig( mode )
   if intfConfig is None:
      return
   if default:
      # The global config will be overriden to enable smart relay
      intfConfig.allSubnetsV6 = 'srDefault'
   elif no:
      # smart relay is disabled regardless of the global config
      intfConfig.allSubnetsV6 = 'srOff'
   else:
      intfConfig.allSubnetsV6 = 'srOn'
   _deleteDrIntfConfigIfAllAttributeHaveDefaults( mode )

class Ipv6DhcpRelaySubnetsCmd( CliCommand.CliCommandClass ):
   syntax = 'ipv6 dhcp relay all-subnets'
   noOrDefaultSyntax = syntax

   data = {
      'ipv6' : ipv6MatcherForConfigIf,
      'dhcp' : dhcpMatcherForConfig,
      'relay' : matcherRelay,
      'all-subnets' : 'DHCPv6 relay forwarding on all the subnets',
   }

   handler = setIntfAllSubnetsRelayV6
   noOrDefaultHandler = handler

modelet.addCommandClass( Ipv6DhcpRelaySubnetsCmd )

#-------------------------------------------------------------------------------
# The "[no|default] ipv6 helper-address <ipv6 address> vrf <vrf name > 
# [ source-interface <interface> | source-address <ipv6 address> ]
# [ link-address <ipv6 address> ]"
# interface command.
#-------------------------------------------------------------------------------

def setIntfDhcp6Relay( mode, args ):
   addrv6 = args[ 'IP6ADDR' ]
   no = CliCommand.isNoOrDefaultCmd( args ) 
   vrfName = args.get( 'VRF' )
   srcIntfIp6 = args.get( 'ADDR' ) or args.get( 'INTF' )
   linkAddr = args.get( 'LINKADDR' )
   if configConflict( mode ):
      return
   srcIp = '::'
   srcIntf= ''
   linkIp = Arnet.IpAddress( 0, addrFamily=socket.AF_INET6 )
   if srcIntfIp6:
      try:
         srcIp = Arnet.IpAddress( srcIntfIp6, addrFamily=socket.AF_INET6 )
      except TypeError:
         srcIntf = srcIntfIp6
   if linkAddr:
      linkIp = Arnet.IpAddress( linkAddr, addrFamily=socket.AF_INET6 )

   ipIntfConfig = ipConfig.ipIntfConfig.get( mode.intf.name )
   if not vrfName:
      vrfName = DEFAULT_VRF if ipIntfConfig is None or ipIntfConfig.vrf == "" \
          else ipIntfConfig.vrf
   if no:
      intfConfig = _drIntfConfig( mode )
      if intfConfig:
         if srcIp != '::' or srcIntf != '':
            del intfConfig.serverIp6[ vrfIpSrcIntfSrcIp( vrfName, addrv6, srcIntf,
                                                         srcIp ) ]
         else:
            for key in intfConfig.serverIp6:
               if key.ip.v6Addr == addrv6 and \
                  key.vrfName == vrfName:
                  del intfConfig.serverIp6[ key ]
         _deleteDrIntfConfigIfAllAttributeHaveDefaults( mode )
   else:
      intfConfig = _getOrCreateDrIntfConfig( mode )
      # Remove existing entry where only src ip or src intf is changed.
      ip6Addr = Arnet.Ip6Addr( addrv6 )
      if ip6Addr:
         for key in intfConfig.serverIp6:
            if key.ip.v6Addr == ip6Addr and \
               key.vrfName == vrfName:
               del intfConfig.serverIp6[ key ]

      if intfConfig:
         intfConfig.serverIp6[ vrfIpSrcIntfSrcIp( vrfName, addrv6, srcIntf,
                                                  srcIp ) ] = linkIp

def setIntfDhcp6RelayRtInstall( mode, args ):
   no = CliCommand.isNoOrDefaultCmd( args )
   if configConflict( mode ):
      return
   if no:
      intfConfig = _drIntfConfig( mode )
      if intfConfig:
         intfConfig.route6Install = False
         _deleteDrIntfConfigIfAllAttributeHaveDefaults( mode )
   else:
      intfConfig = _getOrCreateDrIntfConfig( mode )
      if intfConfig:
         intfConfig.route6Install = True 

ip6AddrMatcher = Ip6AddrMatcher.Ip6AddrMatcher( "DHCP Server's IPv6 address" )
matcherLocalIntf = CliMatcher.KeywordMatcher(
      'local-interface',
      helpdesc='Specify the local interface to communicate with the DHCP server' )
matcherLinkAddr = CliMatcher.KeywordMatcher(
      'link-address',
      helpdesc='Override the default link address specified in the relayed DHCP '
               'packet' )

class Ipv6HelperHostnameVrfSourceCmd( CliCommand.CliCommandClass ):
   syntax = ( "ipv6 helper-address IP6ADDR [ VRF ] "
              "[ ( local-interface INTF ) | ( source-address ADDR ) ] "
              "[ link-address LINKADDR ]" )
   noOrDefaultSyntax = syntax

   data = {
      'ipv6' : ipv6MatcherForConfigIf,
      'helper-address' : matcherHelper,
      'IP6ADDR' : ip6AddrMatcher,
      'VRF' : vrfExprFactory,
      'local-interface' : matcherLocalIntf,
      'INTF' : IntfCli.Intf.matcherWithIpSupport,
      'source-address' : matcherSourceAddr,
      'ADDR' : Ip6AddrMatcher.ip6AddrMatcher,
      'link-address' : matcherLinkAddr,
      'LINKADDR' : Ip6AddrMatcher.ip6AddrMatcher,
   }

   handler = setIntfDhcp6Relay
   noOrDefaultHandler = handler

class Ipv6DhcpRelayDestinationVrfSourceCmd( CliCommand.CliCommandClass ):
   syntax = ( "ipv6 dhcp relay destination IP6ADDR [ VRF ] "
              "[ ( local-interface INTF ) | ( source-address ADDR ) ] "
              "[ link-address LINKADDR ]" )
   noOrDefaultSyntax = syntax

   data = {
      'ipv6' : ipv6MatcherForConfigIf,
      'dhcp' : dhcpMatcherForConfig,
      'relay' : matcherRelay,
      'destination' : matcherDest, 
      'IP6ADDR' : ip6AddrMatcher,
      'VRF' : vrfExprFactory,
      'local-interface' : matcherLocalIntf,
      'INTF' : IntfCli.Intf.matcherWithIpSupport,
      'source-address' : matcherSourceAddr,
      'ADDR' : Ip6AddrMatcher.ip6AddrMatcher,
      'link-address' : matcherLinkAddr,
      'LINKADDR' : Ip6AddrMatcher.ip6AddrMatcher,
   }

   handler = setIntfDhcp6Relay
   noOrDefaultHandler = handler

modelet.addCommandClass( Ipv6DhcpRelayDestinationVrfSourceCmd )
modelet.addCommandClass( Ipv6HelperHostnameVrfSourceCmd )

#-------------------------------------------------------------------------------
# The "[no|default] ipv6 dhcp relay install routes" interface command.
#-------------------------------------------------------------------------------

class Ipv6DhcpRelayInstallRoutesCmd( CliCommand.CliCommandClass ):
   syntax = 'ipv6 dhcp relay install routes'
   noOrDefaultSyntax = syntax

   data = {
      'ipv6' : ipv6MatcherForConfigIf,
      'dhcp' : dhcpMatcherForConfig,
      'relay' : matcherRelay,
      'install' : 'Install routes for DHCP server assigned addresses',
      'routes' : 'Install routes for DHCP server assigned addresses',
   }

   handler = setIntfDhcp6RelayRtInstall
   noOrDefaultHandler = handler

modelet.addCommandClass( Ipv6DhcpRelayInstallRoutesCmd )

#-------------------------------------------------------------------------------
# "[no|default] ipv6 dhcp relay add vendor-option ccap-core <ipv6 address>" 
# interface command
#-------------------------------------------------------------------------------
def setDhcp6CcapCores( mode, args ):
   corev6 = args[ 'CORE' ]
   if configConflict( mode ):
      return
   intfConfig = _getOrCreateDrIntfConfig( mode )
   if intfConfig:
      intfConfig.ccapCores[ corev6 ] = True

def noDhcp6CcapCores( mode, args ):
   if configConflict( mode ):
      return
   if 'CORE' in args:
      corev6 = args[ 'CORE' ]
      intfConfig = _drIntfConfig( mode )
      if intfConfig:
         del intfConfig.ccapCores[ corev6 ]
   else:
      intfConfig = _drIntfConfig( mode )
      if intfConfig:
         intfConfig.ccapCores.clear()
   _deleteDrIntfConfigIfAllAttributeHaveDefaults( mode )

class Ipv6DhcpRelayAddVendorCcapCore( CliCommand.CliCommandClass ):
   syntax = 'ipv6 dhcp relay add vendor-option ccap-core CORE'
   noOrDefaultSyntax = 'ipv6 dhcp relay add vendor-option ccap-core [ CORE ]'

   data = {
      'ipv6' : ipv6MatcherForConfigIf,
      'dhcp' : dhcpMatcherForConfig,
      'relay' : matcherRelay,
      'add' : 'Add sub-options under DHCP Options',
      'vendor-option' : 'Vendor Specific Information Option (17)',
      'ccap-core' : 'Sub-option 61 containing CCAP Core Ipv6 addresses',
      'CORE' : Ip6AddrMatcher.ip6AddrMatcher,
   }

   handler = setDhcp6CcapCores
   noOrDefaultHandler = noDhcp6CcapCores

modelet.addCommandClass( Ipv6DhcpRelayAddVendorCcapCore )

#-------------------------------------------------------------------------------
# The "ip dhcp relay information option circuit-id <circuit-id>" interface command.
#-------------------------------------------------------------------------------

circuitIdValueMatcher = CliMatcher.PatternMatcher(
   pattern=r'[0-9a-zA-Z._:/\-]{1,15}',
   helpdesc='Up to 15 alphanumeric characters describing this interface',
   helpname='Circuit ID'
)

def setIntfCircuitId( mode, args ):
   newId = args[ 'ID' ]
   if configConflict( mode ):
      return
   if newId == mode.intf.name:
      # no need to create the config just to set a default value
      intfConfig = _drIntfConfig( mode )
   else:
      intfConfig = _getOrCreateDrIntfConfig( mode )
   if intfConfig:
      intfConfig.circuitId = newId
      _deleteDrIntfConfigIfAllAttributeHaveDefaults( mode )

def noIntfCircuitId( mode, args ):
   if configConflict( mode ):
      return
   intfConfig = _drIntfConfig( mode )
   if intfConfig:
      intfConfig.circuitId = intfConfig.name
      _deleteDrIntfConfigIfAllAttributeHaveDefaults( mode )

class IpDhcpRelayInformationOptionCircuitIdCmd( CliCommand.CliCommandClass ):
   syntax = 'ip dhcp relay information option circuit-id ID'
   noOrDefaultSyntax = 'ip dhcp relay information option circuit-id ...'

   data = {
      'ip' : ipMatcherForConfigIf,
      'dhcp' : dhcpMatcherForConfig,
      'relay' : matcherRelay,
      'information' : 'Option 82 (Information Option)',
      'option' : 'DHCP option',
      'circuit-id' : 'Relay Agent (Option 82) Circuit ID for this interface',
      'ID' : circuitIdValueMatcher,
   }

   handler = setIntfCircuitId
   noOrDefaultHandler = noIntfCircuitId

modelet.addCommandClass( IpDhcpRelayInformationOptionCircuitIdCmd )

dhcpHelp = 'DHCP server, client and relay interface configuration'
ipv4DhcpRelayHelpDesc = 'IPv4 DHCP relay configuration'
ipv4DhcpRelayDisabledHelpDesc = 'Disable DHCP relay configurations'

def setDhcpRelayDisabledCmd( mode, args ):
   no = CliCommand.isNoOrDefaultCmd( args )
   if configConflict( mode ):
      return

   intfConfig = _getOrCreateDrIntfConfig( mode )
   intfConfig.disabledV4 = False if no else True
   _deleteDrIntfConfigIfAllAttributeHaveDefaults( mode )

#-------------------------------------------------------------------------------
# "[no|default] dhcp relay ipv4 disabled" 
# interface command
#-------------------------------------------------------------------------------
class IpDhcpRelayDisabledCmd( CliCommand.CliCommandClass ):
   syntax = 'dhcp relay ipv4 disabled'
   noOrDefaultSyntax = syntax

   data = {
      'dhcp' : dhcpHelp,   
      'relay' : matcherRelay,
      'ipv4' : ipv4DhcpRelayHelpDesc,
      'disabled' : ipv4DhcpRelayDisabledHelpDesc,
      }

   handler = setDhcpRelayDisabledCmd
   noOrDefaultHandler = handler

if DhcpRelayToggleLib.toggleDhcpRelayGlobalModeServerCommandEnabled():
   modelet.addCommandClass( IpDhcpRelayDisabledCmd )

#-------------------------------------------------------------------------------
# The DhcpRelayIntf class is used to remove the DhcpRelay IntfConfig object when an
# interface is deleted. The Intf class will create a new instance of DhcpRelayIntf
# and call destroy when the interface is deleted
#-------------------------------------------------------------------------------
class DhcpRelayIntf( IntfCli.IntfDependentBase ):
   #----------------------------------------------------------------------------
   # Destroys the dhcprelay IntfConfig object for this interface if it exists.
   #----------------------------------------------------------------------------
   def setDefault( self ):
      del drConfig().intfConfig[ self.intf_.name ]

#-------------------------------------------------------------------------------
# Have the Cli Agent mount all needed state from sysdb
#-------------------------------------------------------------------------------
def Plugin( entityManager ):
   # Global variable undefined so far
   # pylint: disable-msg=W0601
   global ipConfig
   global dhcpRelayConfigDir
   global dhcpRelayCounterConfigDir
   global dhcpFwdConfig
   global dhcpRelayStatusDir
   global aclConfig
   global aclCpConfig
   global aclStatus
   global aclCheckpoint
   global mlagDhcpRelayStatusDir

   # pkgdeps: rpmwith %{_libdir}/SysdbMountProfiles/DhcpRelay
   ipConfig = LazyMount.mount( entityManager, "ip/config", "Ip::Config", "r" )
   dhcpRelayConfigDir = ConfigMount.mount( entityManager,
                                           "ip/helper/dhcprelay/config",
                                           "Ip::Helper::DhcpRelay::Config", "w" )
   dhcpRelayCounterConfigDir = LazyMount.mount( entityManager,
                                    "ip/helper/dhcprelay/counterConfig",
                                    "Ip::Helper::DhcpRelay::CounterConfig", "w" )
   # check the config for the incompatible dhcp-forwarder
   dhcpFwdConfig = LazyMount.mount( entityManager, "ip/dhcp/relay/config",
                                    "Ip::Dhcp::Relay::Config", "r" )
   dhcpRelayStatusDir = LazyMount.mount( entityManager, "ip/helper/dhcprelay/status",
                                         "Ip::Helper::DhcpRelay::Status", "r" )
   mlagDhcpRelayStatusDir = LazyMount.mount( entityManager,
                                             "ip/helper/dhcprelay/mlagStatus",
                                             "MlagDhcpRelay::Status", "r" )
   IntfCli.Intf.registerDependentClass( DhcpRelayIntf )
   aclConfig = ConfigMount.mount( entityManager, "acl/config/cli",
                                  "Acl::Input::Config", "w" )
   aclCpConfig = ConfigMount.mount( entityManager, "acl/cpconfig/cli",
                                  "Acl::Input::CpConfig", "w" )
   aclStatus = LazyMount.mount( entityManager, "acl/status/all",
                                "Acl::Status", "r" )
   aclCheckpoint = LazyMount.mount( entityManager, "acl/checkpoint",
                                   "Acl::CheckpointStatus", "w" )   
