#!/usr/bin/env python
# Copyright (c) 2019 Arista Networks, Inc.  All rights reserved.
# Arista Networks, Inc. Confidential and Proprietary.

from __future__ import absolute_import, division, print_function

import Arnet
import AclCliLib, AclLib
import BasicCliModes
import BasicCli
import CliCommand
import CliMatcher
from CliMode.DhcpRelay import DhcpRelayGlobalBaseMode
import CliParser
import CliPlugin.AclCli as AclCli
import CliPlugin.DhcpSnoopingCli as DhcpSnoopingCli
import CliPlugin.Dhcpv6SnoopingCli as Dhcpv6SnoopingCli
import CliPlugin.Ip6AddrMatcher as Ip6AddrMatcher
import CliPlugin.VlanCli as VlanCli
from CliPlugin.AclCli import (
      accessGroupKwMatcher,
      inKwMatcher,
)
from CliPlugin.DhcpRelayHelperCli import (
      configConflict, drConfig, vrfIpSrcIntfSrcIp,
      vrfHostnameSrcIntfSrcIp,
)
from CliPlugin.VrfCli import DEFAULT_VRF
from CliToken.Dhcp import dhcpMatcherForConfig
from CliToken.Ip import (
      ipMatcherForConfig,
)
from CliToken.Ipv6 import (
      ipv6MatcherForConfig,
)
import ConfigMount, LazyMount
import HostnameCli
import socket
import Tac
from eunuchs.in_h import IPPROTO_UDP
import Toggles.DhcpRelayToggleLib as DhcpRelayToggleLib

matcherRelay = CliMatcher.KeywordMatcher( 'relay',
      helpdesc='DHCP Relay' )
matcherNone = CliMatcher.KeywordMatcher( 'none',
      helpdesc='Do not include type in the Circuit ID sub-option' )
matcherSnooping = CliMatcher.KeywordMatcher( 'snooping',
      helpdesc='DHCPv6 Snooping configuration' )
matcherType = CliMatcher.KeywordMatcher( 'type',
      helpdesc='Type encoded in the Circuit ID sub-option' )
matcherVlan = CliMatcher.KeywordMatcher( 'vlan',
      helpdesc='VLAN' )
matcherVrf = CliMatcher.KeywordMatcher( 'vrf',
      helpdesc='Configure the VRF in which to apply the access control list' )
nodeSnooping = CliCommand.guardedKeyword( 
      'snooping', helpdesc='DHCPv6 Snooping configuration', 
      guard=Dhcpv6SnoopingCli.dhcp6SnoopingSupportedGuard )
addressFamilyExprForConfig = CliMatcher.EnumMatcher( {
   'ip' : ipMatcherForConfig.helpdesc_,
   'ipv6' : ipv6MatcherForConfig.helpdesc_,
} )

aclConfig = None
aclCpConfig = None
dhcpSnoopingConfig = None
dhcpRelayConfig = None
nativeRelayConfig = None
dhcp6SnoopingConfig = None

#--------------------------------------------------------------------------------
# [ no | default ] ( ip | ipv6 ) dhcp relay always-on
#--------------------------------------------------------------------------------
def setAlwaysOn( mode, args ):
   no = CliCommand.isNoOrDefaultCmd( args )
   if configConflict( mode ):
      return
   drConfig().alwaysOn = not no

class DhcpRelayAlwaysOnCmd( CliCommand.CliCommandClass ):
   syntax = '( ip | ipv6 ) dhcp relay always-on'
   noOrDefaultSyntax = syntax
   data = {
      'ip' : ipMatcherForConfig,
      'ipv6' : ipv6MatcherForConfig,
      'dhcp' : dhcpMatcherForConfig,
      'relay' : matcherRelay,
      'always-on' : 'Enable Agent Always-On Option',
   }

   handler = setAlwaysOn
   noOrDefaultHandler = handler
   
BasicCliModes.GlobalConfigMode.addCommandClass( DhcpRelayAlwaysOnCmd )

#--------------------------------------------------------------------------------
# [ no | default ] ( ip | ipv6 ) dhcp relay qos dscp DSCP
#--------------------------------------------------------------------------------
def setDscp( mode, args ):
   dscpValue = args[ 'DSCP' ]
   if 'ip' in args:
      drConfig().dscpV4 = dscpValue
   else:
      drConfig().dscpV6 = dscpValue

def noDscp( mode, args ):
   if 'ip' in args:
      drConfig().dscpV4 = drConfig().dscpValueDefault
   else:
      drConfig().dscpV6 = drConfig().dscpValueDefault

class DhcpRelayQosDscpCmd( CliCommand.CliCommandClass ):
   syntax = '( ip | ipv6 ) dhcp relay qos dscp DSCP'
   noOrDefaultSyntax = '( ip | ipv6 ) dhcp relay qos dscp ...'
   data = {
      'ip' : ipMatcherForConfig,
      'ipv6' : ipv6MatcherForConfig,
      'dhcp' : dhcpMatcherForConfig,
      'relay' : matcherRelay,
      'qos' : 'Configure QoS parameters',
      'dscp' : 'Set DSCP value in IP header',
      'DSCP' : CliMatcher.IntegerMatcher( 
         0, 63, helpdesc='DSCP value' ),
   }

   handler = setDscp
   noOrDefaultHandler = noDscp

BasicCliModes.GlobalConfigMode.addCommandClass( DhcpRelayQosDscpCmd )

#--------------------------------------------------------------------------------
# [ no | default ] ip dhcp relay access-group ACLNAME [ vrf VRFNAME ] [ in ]
#--------------------------------------------------------------------------------
def setAcl( mode, args ):
   no = CliCommand.isNoOrDefaultCmd( args )
   aclName = args.get( 'ACL' )
   vrfName = args.get( 'VRF' )
   ipv4 = 'ip' in args
   aclType = 'ip' if ipv4  else 'ipv6'
   proto = 'bootps' if ipv4 else 'dhcpv6-server' 
   if no:
      AclCliLib.noServiceAcl( mode, 'dhcpRelay', aclConfig, aclCpConfig, 
                              aclName, aclType, vrfName )
   else:
      AclCliLib.setServiceAcl( mode, 'dhcpRelay', IPPROTO_UDP,
                               aclConfig, aclCpConfig, aclName, aclType, vrfName,
                               port=[ AclLib.getServByName( IPPROTO_UDP, proto ) ] )

class IpDhcpRelayAccessGroupAclnameCmd( CliCommand.CliCommandClass ):
   syntax = '( ip | ipv6 ) dhcp relay access-group ACL [ VRF ] [ in ]'
   noOrDefaultSyntax = '( ip | ipv6 ) dhcp relay access-group [ ACL ] [ VRF ] [ in ]'
   data = {
      'ip' : ipMatcherForConfig,
      'ipv6' : ipv6MatcherForConfig,
      'dhcp' : dhcpMatcherForConfig,
      'relay' : matcherRelay,
      'access-group' : accessGroupKwMatcher,
      'ACL' : AclCli.ipAclNameMatcher,
      'VRF' : AclCli.vrfKwAndNameExprFactory,
      'in' : inKwMatcher,
   }

   handler = setAcl
   noOrDefaultHandler = handler
   
BasicCliModes.GlobalConfigMode.addCommandClass( IpDhcpRelayAccessGroupAclnameCmd )

#--------------------------------------------------------------------------------
# [ no | default ] ip dhcp relay all-subnets default
#--------------------------------------------------------------------------------
def setSmartRelay( mode, args ):
   no = CliCommand.isNoOrDefaultCmd( args )
   if configConflict( mode ):
      return
   drConfig().smartRelayGlobal = not no

class IpDhcpRelayAllSubnetsDefaultCmd( CliCommand.CliCommandClass ):
   syntax = 'ip dhcp relay all-subnets default'
   noOrDefaultSyntax = syntax
   data = {
      'ip' : ipMatcherForConfig,
      'dhcp' : dhcpMatcherForConfig,
      'relay' : matcherRelay,
      'all-subnets' : 'DHCP Relay Forwarding on all the subnets',
      'default' : 'Enable DHCP Relay Forwarding across all interfaces',
   }

   handler = setSmartRelay
   noOrDefaultHandler = handler

BasicCliModes.GlobalConfigMode.addCommandClass( IpDhcpRelayAllSubnetsDefaultCmd )

#--------------------------------------------------------------------------------
# [ no | default ] ip dhcp smart-relay global
#--------------------------------------------------------------------------------
class IpDhcpSmartRelayGlobalCmd( CliCommand.CliCommandClass ):
   syntax = 'ip dhcp smart-relay global'
   noOrDefaultSyntax = syntax
   data = {
      'ip' : ipMatcherForConfig,
      'dhcp' : dhcpMatcherForConfig,
      'smart-relay' : CliCommand.Node( matcher=CliMatcher.KeywordMatcher( 
         'smart-relay',
         helpdesc='DHCP Smart Relay Agent Forwarding' ),
         deprecatedByCmd='ip dhcp relay all-subnets default' ),
      'global' : 'Enable DHCP Smart Relay Agent Forwarding across all interfaces',
   }

   handler = setSmartRelay
   noOrDefaultHandler = handler
   
BasicCliModes.GlobalConfigMode.addCommandClass( IpDhcpSmartRelayGlobalCmd )

#--------------------------------------------------------------------------------
# [ no | default ] ip dhcp relay information option
#--------------------------------------------------------------------------------
def setIntfDhcpRelayInfo( mode, args ):
   no = CliCommand.isNoOrDefaultCmd( args )
   if configConflict( mode ):
      return
   drConfig().circuitIdOptOn = not no

class IpDhcpRelayInformationOptionCmd( CliCommand.CliCommandClass ):
   syntax = 'ip dhcp relay information option'
   noOrDefaultSyntax = syntax
   data = {
      'ip' : ipMatcherForConfig,
      'dhcp' : dhcpMatcherForConfig,
      'relay' : matcherRelay,
      'information' : 'Option 82 (Information Option)',
      'option' : 'Enable Option 82',
   }

   handler = setIntfDhcpRelayInfo
   noOrDefaultHandler = handler
   
BasicCliModes.GlobalConfigMode.addCommandClass( IpDhcpRelayInformationOptionCmd )

#--------------------------------------------------------------------------------
# [ no | default ] ip dhcp snooping
#--------------------------------------------------------------------------------
def setDhcpSnooping( mode, args ):
   no = CliCommand.isNoOrDefaultCmd( args )
   dhcpSnoopingConfig.enabled = not no
   
class IpDhcpSnoopingCmd( CliCommand.CliCommandClass ):
   syntax = 'ip dhcp snooping'
   noOrDefaultSyntax = syntax
   data = {
      'ip' : ipMatcherForConfig,
      'dhcp' : dhcpMatcherForConfig,
      'snooping' : CliCommand.guardedKeyword(
         'snooping', helpdesc='DHCP Snooping configuration',
         guard=DhcpSnoopingCli.dhcpSnoopingSupportedGuard ),
   }

   handler = setDhcpSnooping
   noOrDefaultHandler = handler
   
BasicCliModes.GlobalConfigMode.addCommandClass( IpDhcpSnoopingCmd )

#--------------------------------------------------------------------------------
# ip dhcp snooping information option circuit-id type ( none | NONE ) 
#         format ( %h:%p | %p:%v )
#--------------------------------------------------------------------------------
def setInfoCircuitId( mode, args ):
   circuitIdType = args.get( 'TYPE' )
   formatString = args.get( '%h:%p' ) 
   if formatString is None:
      formatString = args[ '%p:%v' ]
   if circuitIdType is None:
      dhcpSnoopingConfig.circuitIdTypeValid = False
   else:
      dhcpSnoopingConfig.circuitIdTypeValid = True
      dhcpSnoopingConfig.circuitIdType = int( circuitIdType )

   dhcpSnoopingConfig.circuitIdFormatString = formatString

def resetInfoCircuitId( mode, args ):
   dhcpSnoopingConfig.circuitIdType = dhcpSnoopingConfig.circuitIdTypeDefault
   dhcpSnoopingConfig.circuitIdFormatString = \
      dhcpSnoopingConfig.circuitIdFormatStringDefault
   dhcpSnoopingConfig.circuitIdTypeValid = \
      dhcpSnoopingConfig.circuitIdTypeValidDefault

class IpDhcpSnoopingInformationCmd( CliCommand.CliCommandClass ):
   syntax = ( 'ip dhcp snooping information option circuit-id type ( none | TYPE ) '
              'format ( %h:%p | %p:%v )' )
   noOrDefaultSyntax = 'ip dhcp snooping information option circuit-id ...'
   data = {
      'ip' : ipMatcherForConfig,
      'dhcp' : dhcpMatcherForConfig,
      'snooping' : CliCommand.guardedKeyword(
         'snooping', helpdesc='DHCP Snooping configuration',
         guard=DhcpSnoopingCli.dhcpSnoopingSupportedGuard ),
      'information' : 'Option-82 information',
      'format' : 'Specify Circuit ID format',
      'option' : 'Configure Information Option',
      'circuit-id' : 'Circuit ID subopton in Option-82',
      'type' : matcherType,
      'none' : matcherNone,
      'TYPE' : CliMatcher.IntegerMatcher(
         0, 255, helpdesc='Specify the type in the Circuit ID sub-option' ),
      '%h:%p' : 'Hostname and interface name',
      '%p:%v' : 'Interface name and VLAN ID',
   }

   handler = setInfoCircuitId
   noOrDefaultHandler = resetInfoCircuitId

BasicCliModes.GlobalConfigMode.addCommandClass( IpDhcpSnoopingInformationCmd )

#--------------------------------------------------------------------------------
# [ no | default ] ip dhcp snooping information option
#--------------------------------------------------------------------------------
def setInfoOption( mode, args ):
   no = CliCommand.isNoOrDefaultCmd( args )
   dhcpSnoopingConfig.informationOption = not no
   
class IpDhcpSnoopingInformationOptionCmd( CliCommand.CliCommandClass ):
   syntax = 'ip dhcp snooping information option'
   noOrDefaultSyntax = syntax
   data = {
      'ip' : ipMatcherForConfig,
      'dhcp' : dhcpMatcherForConfig,
      'snooping' : CliCommand.guardedKeyword(
         'snooping', helpdesc='DHCP Snooping configuration',
         guard=DhcpSnoopingCli.dhcpSnoopingSupportedGuard ),
      'information' : 'Option-82 information',
      'option' : 'Configure Information Option',
   }

   handler = setInfoOption
   noOrDefaultHandler = handler
   
BasicCliModes.GlobalConfigMode.addCommandClass( IpDhcpSnoopingInformationOptionCmd )

#--------------------------------------------------------------------------------
# [ no | default ] ip dhcp snooping bridging
#--------------------------------------------------------------------------------
def setBridging( mode, args ):
   no = CliCommand.isNoOrDefaultCmd( args )
   dhcpSnoopingConfig.bridging = not no
   
class IpDhcpSnoopingBridgingCmd( CliCommand.CliCommandClass ):
   syntax = 'ip dhcp snooping bridging'
   noOrDefaultSyntax = syntax
   data = {
      'ip' : ipMatcherForConfig,
      'dhcp' : dhcpMatcherForConfig,
      'snooping' : CliCommand.guardedKeyword(
         'snooping', helpdesc='DHCP Snooping configuration',
         guard=DhcpSnoopingCli.dhcpSnoopingSupportedGuard ),
      'bridging' : 'Enable DHCP Snooping with bridging',
   }

   handler = setBridging
   noOrDefaultHandler = handler

if DhcpRelayToggleLib.toggleDhcpSnoopingBridgingV4Enabled():
   BasicCliModes.GlobalConfigMode.addCommandClass( IpDhcpSnoopingBridgingCmd )

#--------------------------------------------------------------------------------
# [ no | default ] ip dhcp snooping vlan VLANSET
#--------------------------------------------------------------------------------
class IpDhcpSnoopingVlanVlansetCmd( CliCommand.CliCommandClass ):
   syntax = 'ip dhcp snooping vlan VLANSET'
   noOrDefaultSyntax = 'ip dhcp snooping vlan [ VLANSET ]'
   data = {
      'ip' : ipMatcherForConfig,
      'dhcp' : dhcpMatcherForConfig,
      'snooping' : CliCommand.guardedKeyword(
         'snooping', helpdesc='DHCP Snooping configuration',
         guard=DhcpSnoopingCli.dhcpSnoopingSupportedGuard ),
      'vlan' : matcherVlan,
      'VLANSET' : VlanCli.vlanSetMatcher
   }

   @staticmethod
   def handler( mode, args ):
      for vlan in args[ 'VLANSET' ].ids:
         dhcpSnoopingConfig.vlan[ vlan ] = True
   
   @staticmethod
   def noOrDefaultHandler( mode, args ):
      vlans = args.get( 'VLANSET' )
      if vlans:
         for vlan in vlans.ids:
            del dhcpSnoopingConfig.vlan[ vlan ]
      else:
         dhcpSnoopingConfig.vlan.clear()
   
BasicCliModes.GlobalConfigMode.addCommandClass( IpDhcpSnoopingVlanVlansetCmd )

#--------------------------------------------------------------------------------
# [ no | default ] ipv6 dhcp relay option link-layer address
#--------------------------------------------------------------------------------
def setDhcpRelayv6LinkLayerAddrOpt( mode, args ):
   no = CliCommand.isNoOrDefaultCmd( args )
   if configConflict( mode ):
      return
   drConfig().clientLinkLayerAddrOptOn = not no

class Ipv6DhcpRelayOptionLinkLayerCmd( CliCommand.CliCommandClass ):
   syntax = 'ipv6 dhcp relay option link-layer address'
   noOrDefaultSyntax = syntax
   data = {
      'ipv6' : ipv6MatcherForConfig,
      'dhcp' : dhcpMatcherForConfig,
      'relay' : matcherRelay,
      'option' : 'DHCP option',
      'link-layer' : 'Option 79 (Link Layer Address Option)',
      'address' : 'Address keyword',
   }

   handler = setDhcpRelayv6LinkLayerAddrOpt
   noOrDefaultHandler = handler
   
BasicCliModes.GlobalConfigMode.addCommandClass( Ipv6DhcpRelayOptionLinkLayerCmd )

#--------------------------------------------------------------------------------
# "[no|default] ipv6 dhcp relay option remote-id format ( %m:%i | %m:%p )"
#-------------------------------------------------------------------------------
def setDhcpRelayv6RemoteIdOptFormat( mode, args ):
   no = CliCommand.isNoOrDefaultCmd( args )
   if configConflict( mode ):
      return
   if no:
      drConfig().remoteIdEncodingFormat = drConfig().remoteIdEncodingFormatDefault
      return
   if "%m:%i" in args:
      drConfig().remoteIdEncodingFormat = '%m:%i'
   elif "%m:%p" in args:
      drConfig().remoteIdEncodingFormat = '%m:%p'

class Ipv6DhcpRelayRemoteIdFormatCmd( CliCommand.CliCommandClass ):
   syntax = 'ipv6 dhcp relay option remote-id format ( %m:%i | %m:%p )'
   noOrDefaultSyntax = 'ipv6 dhcp relay option remote-id format ...'
   data = {
      'ipv6' : ipv6MatcherForConfig,
      'dhcp' : dhcpMatcherForConfig,
      'relay' : matcherRelay,
      'option' : 'DHCP option',
      'remote-id' : 'RemoteID option 37',
      'format': 'Encoding format',
      '%m:%i' : 'MAC address and interface ID',
      '%m:%p' :  'MAC address and interface name',
   }
   handler = setDhcpRelayv6RemoteIdOptFormat
   noOrDefaultHandler = handler

BasicCliModes.GlobalConfigMode.addCommandClass( Ipv6DhcpRelayRemoteIdFormatCmd )
#--------------------------------------------------------------------------------
# [ no | default ] ipv6 dhcp snooping
#--------------------------------------------------------------------------------
def setDhcp6Snooping( mode, args ):
   no = CliCommand.isNoOrDefaultCmd( args )
   dhcp6SnoopingConfig.enabled = not no

class Ipv6DhcpSnoopingCmd( CliCommand.CliCommandClass ):
   syntax = 'ipv6 dhcp snooping'
   noOrDefaultSyntax = syntax
   data = {
      'ipv6' : ipv6MatcherForConfig,
      'dhcp' : dhcpMatcherForConfig,
      'snooping' : nodeSnooping,
   }

   handler = setDhcp6Snooping
   noOrDefaultHandler = handler
   
BasicCliModes.GlobalConfigMode.addCommandClass( Ipv6DhcpSnoopingCmd )

#--------------------------------------------------------------------------------
# [ no | default ] ipv6 dhcp snooping remote-id option
#--------------------------------------------------------------------------------
def setInfoOptionV6( mode, args ):
   no = CliCommand.isNoOrDefaultCmd( args )
   dhcp6SnoopingConfig.remoteIdOption = not no
   
class Ipv6DhcpSnoopingRemoteIdOptionCmd( CliCommand.CliCommandClass ):
   syntax = 'ipv6 dhcp snooping remote-id option'
   noOrDefaultSyntax = syntax
   data = {
      'ipv6' : ipv6MatcherForConfig,
      'dhcp' : dhcpMatcherForConfig,
      'snooping' : nodeSnooping,
      'remote-id' : 'Option-37 information',
      'option' : 'Configure Remote-ID option',
   }

   handler = setInfoOptionV6
   noOrDefaultHandler = handler
   
BasicCliModes.GlobalConfigMode.addCommandClass( Ipv6DhcpSnoopingRemoteIdOptionCmd )

#--------------------------------------------------------------------------------
# [ no | default ] ipv6 dhcp snooping vlan VLANSET
#--------------------------------------------------------------------------------
class Ipv6DhcpSnoopingVlanVlansetCmd( CliCommand.CliCommandClass ):
   syntax = 'ipv6 dhcp snooping vlan VLANSET'
   noOrDefaultSyntax = 'ipv6 dhcp snooping vlan [ VLANSET ]'
   data = {
      'ipv6' : ipv6MatcherForConfig,
      'dhcp' : dhcpMatcherForConfig,
      'snooping' : nodeSnooping,
      'vlan' : matcherVlan,
      'VLANSET' : VlanCli.vlanSetMatcher,
   }

   @staticmethod
   def handler( mode, args ):
      for vlan in args[ 'VLANSET' ].ids:
         dhcp6SnoopingConfig.vlan[ vlan ] = True

   @staticmethod
   def noOrDefaultHandler( mode, args ):
      vlans = args.get( 'VLANSET' )
      if vlans:
         for vlan in vlans.ids:
            del dhcp6SnoopingConfig.vlan[ vlan ]
      else:
         dhcp6SnoopingConfig.vlan.clear()
 
BasicCliModes.GlobalConfigMode.addCommandClass( Ipv6DhcpSnoopingVlanVlansetCmd )

#--------------------------------------------------------------------------------
# [ no | default } ipv6 dhcp relay all-subnets default
#--------------------------------------------------------------------------------
def setSmartRelayV6( mode, args ):
   no = CliCommand.isNoOrDefaultCmd( args )
   if configConflict( mode ):
      return
   drConfig().allSubnetsV6Global = not no

class Ipv6DhcpRelayAllSubnetsDefaultCmd( CliCommand.CliCommandClass ):
   syntax = 'ipv6 dhcp relay all-subnets default'
   noOrDefaultSyntax = syntax
   data = {
      'ipv6' : ipv6MatcherForConfig,
      'dhcp' : dhcpMatcherForConfig,
      'relay' : matcherRelay,
      'all-subnets' : 'DHCPv6 relay forwarding on all the subnets',
      'default' : 'Enable DHCPv6 all subnet relaying across all interfaces',
   }

   handler = setSmartRelayV6
   noOrDefaultHandler = handler

BasicCliModes.GlobalConfigMode.addCommandClass( Ipv6DhcpRelayAllSubnetsDefaultCmd )

#--------------------------------------------------------------------------------
# dhcp relay
# in mode: config
#--------------------------------------------------------------------------------
class DhcpRelayGlobalMode( DhcpRelayGlobalBaseMode, BasicCli.ConfigModeBase ):
   name = "DHCP Relay"
   modeParseTree = CliParser.ModeParseTree()
   
   def __init__( self, parent, session ):
      DhcpRelayGlobalBaseMode.__init__( self )
      BasicCli.ConfigModeBase.__init__( self, parent, session )

def setDhcpRelayGlobalDefaultValues( mode, args ):
   # set everything to the default state
   config = drConfig()
   config.dhcpRelayGlobalMode = False
   config.dhcpRelayGlobalModeConfig = None

class EnterDhcpRelayGlobalModeCmd( CliCommand.CliCommandClass ):
   syntax = '''dhcp relay'''
   noOrDefaultSyntax = syntax

   data = {
      'dhcp': 'DHCP configuration',
      'relay': 'DHCP relay configuration',
      }

   @staticmethod
   def handler( mode, args ):
      drConfig().dhcpRelayGlobalMode = True
      childMode = mode.childMode( DhcpRelayGlobalMode )
      mode.session_.gotoChildMode( childMode )
   
   noOrDefaultHandler = setDhcpRelayGlobalDefaultValues

if DhcpRelayToggleLib.toggleDhcpRelayGlobalDhcpRelayModeEnabled():
   BasicCli.GlobalConfigMode.addCommandClass( EnterDhcpRelayGlobalModeCmd )

#--------------------------------------------------------------------------------
# server ( IP_GENADDR | HOSTNAME )
# in mode: dhcp relay
# Equivalent to helper-address in interface mode
#--------------------------------------------------------------------------------
def _deleteAllGlobalConfigsIfAllAttributesHaveDefaults():
   # verify if we still have global dhcp relay commands
   config = drConfig()
   drModeConfig = config.dhcpRelayGlobalModeConfig
   if ( not drModeConfig.serverIpGlobal and 
        not drModeConfig.serverIp6Global and
        not drModeConfig.serverHostnameGlobal and
        not drModeConfig.tunnelReqDisable ):
      config.dhcpRelayGlobalModeConfig = None

def setDhcpRelayGlobalServerCmd( mode, args ):
   if configConflict( mode ):
      return

   ipAddr = args.get( 'IP_ADDR' )
   ipv4 = isinstance( ipAddr, Tac.Type( 'Arnet::IpAddr' ) )
   ipv6 = isinstance( ipAddr, Tac.Type( 'Arnet::Ip6Addr' ) )
   key = args[ 'key' ]
   config = drConfig()
   no = CliCommand.isNoOrDefaultCmd( args )

   # remove
   if no and config.dhcpRelayGlobalModeConfig:
      drModeConfig = config.dhcpRelayGlobalModeConfig
      if ipv4:
         del drModeConfig.serverIpGlobal[ key ]
      elif ipv6:
         del drModeConfig.serverIp6Global[ key ]
      else:
         del drModeConfig.serverHostnameGlobal[ key ]
      _deleteAllGlobalConfigsIfAllAttributesHaveDefaults()

   # add
   else:
      if not config.dhcpRelayGlobalModeConfig:
         config.dhcpRelayGlobalModeConfig = ( 'DhcpRelayGlobalModeConfig', )
      
      drModeConfig = config.dhcpRelayGlobalModeConfig
      if ipv4:
         drModeConfig.serverIpGlobal[ key ] = True
      elif ipv6:
         # We should match setIntfDhcp6Relay implementation
         # in order to have a config like intfConfig. 
         linkIp = Arnet.IpAddress( 0, addrFamily=socket.AF_INET6 )
         drModeConfig.serverIp6Global[ key ] = linkIp
      else:
         drModeConfig.serverHostnameGlobal[ key ] = True

class DhcpRelayGlobalServerCmd( CliCommand.CliCommandClass ):
   syntax = '''server HOST_OR_ADDR'''
   noOrDefaultSyntax = syntax 
   
   # Using IpAddrOrHostnameMatcher, since having two separate matchers
   # (HostnameCli.HostnameMatcher and IpAddrMatcher.IpAddrMatcher) causes
   # both nodes to match simultaneously for inputs like 1.0.0.0
   data = {
      'server' : 'Specify DHCP Server',
      'HOST_OR_ADDR' : HostnameCli.IpAddrOrHostnameMatcher( 
                           helpdesc='IPv4 address or Hostname of DHCP server' ),
      }

   # Check if we want to add ipv6 configuration
   if DhcpRelayToggleLib.toggleDhcpRelayGlobalModeServerCommandIPv6Enabled():
      syntax = '''server ( HOST_OR_ADDR | IPV6_ADDR )'''
      noOrDefaultSyntax = syntax
      data[ 'IPV6_ADDR' ] = Ip6AddrMatcher.Ip6AddrMatcher( 
                                   helpdesc='IPv6 address of DHCP server' )
   
   @staticmethod
   def adapter( mode, args , argsList ):
      # default value to create key
      vrfName = DEFAULT_VRF
      ipAddr = '0.0.0.0'
      srcIntf = ''
      srcIp = '0.0.0.0'
      
      # set ipAddr or hostname
      ipAddrOrHostname = args.get( 'HOST_OR_ADDR' )
      ipAddr, hostname = None, None
      if ipAddrOrHostname:
         try:
            ipAddr = Arnet.IpAddress( ipAddrOrHostname )
         except ValueError:
            hostname = ipAddrOrHostname
      if hostname:
         args[ 'HOSTNAME' ] = hostname
         key = vrfHostnameSrcIntfSrcIp( vrfName, hostname, srcIntf, srcIp )
      else:
         args[ 'IP_ADDR' ] = ipAddr or args.pop( 'IPV6_ADDR' )
         key = vrfIpSrcIntfSrcIp( vrfName, args[ 'IP_ADDR' ], srcIntf, srcIp )
      # save the key to enter the right collection
      args[ 'key' ] = key

   handler = setDhcpRelayGlobalServerCmd
   noOrDefaultHandler = handler

if DhcpRelayToggleLib.toggleDhcpRelayGlobalModeServerCommandEnabled():
   DhcpRelayGlobalMode.addCommandClass( DhcpRelayGlobalServerCmd )

#--------------------------------------------------------------------------------
# (config)# dhcp relay
# (config-dhcp-relay)# [ no | default ] tunnel requests disabled
#--------------------------------------------------------------------------------
def setTunnelRequestsDisabled( mode, args ):
   config = drConfig()
   no = CliCommand.isNoOrDefaultCmd( args )
   if not config.dhcpRelayGlobalModeConfig:
      config.dhcpRelayGlobalModeConfig = ( 'DhcpRelayGlobalModeConfig', )
   drModeConfig = config.dhcpRelayGlobalModeConfig
   drModeConfig.tunnelReqDisable = not no
   if no and config.dhcpRelayGlobalModeConfig:
      _deleteAllGlobalConfigsIfAllAttributesHaveDefaults()

class DhcpRelayGlobalTunnelRequestsDisabledCmd( CliCommand.CliCommandClass ):
   syntax = 'tunnel requests disabled'
   noOrDefaultSyntax = syntax
   data = {
      'tunnel' : 'Configure tunnel setting for DHCP relay',
      'requests' : 'DHCP requests',
      'disabled' : 'Disable processing DHCP request received from tunnel'
   }

   handler = setTunnelRequestsDisabled
   noOrDefaultHandler = handler

if DhcpRelayToggleLib.toggleDhcpRelayGlobalDhcpRelayModeEnabled() and \
      DhcpRelayToggleLib.toggleDhcpRelayTunnelReqDisableEnabled():
   DhcpRelayGlobalMode.addCommandClass( DhcpRelayGlobalTunnelRequestsDisabledCmd )

def Plugin( entityManager ):
   global dhcp6SnoopingConfig
   global dhcpSnoopingConfig
   global dhcpRelayConfig
   global nativeRelayConfig
   global aclConfig
   global aclCpConfig

   dhcp6SnoopingConfig = ConfigMount.mount( entityManager,
                                            "bridging/dhcpsnooping/dhcp6Config",
                                            "Bridging::DhcpSnooping::Dhcp6Config",
                                            "w" )
   dhcpSnoopingConfig = ConfigMount.mount( entityManager,
                                           "bridging/dhcpsnooping/config",
                                           "Bridging::DhcpSnooping::Config", "w" )
   dhcpRelayConfig = ConfigMount.mount( entityManager, "ip/dhcp/relay/config",
                                        "Ip::Dhcp::Relay::Config", "w" )
   nativeRelayConfig = LazyMount.mount( entityManager, "ip/helper/dhcprelay/config",
                                        "Ip::Helper::DhcpRelay::Config", "r" )
   aclConfig = ConfigMount.mount( entityManager, "acl/config/cli",
                                  "Acl::Input::Config", "w" )
   aclCpConfig = ConfigMount.mount( entityManager, "acl/cpconfig/cli",
                                    "Acl::Input::CpConfig", "w" )
