#!/usr/bin/env python
# Copyright (c) 2019 Arista Networks, Inc.  All rights reserved.
# Arista Networks, Inc. Confidential and Proprietary.

from __future__ import absolute_import, division, print_function

import BasicCliModes
import CliCommand
import CliMatcher
import LazyMount
import CliPlugin.AclCli as AclCli
import CliPlugin.IntfCli as IntfCli
from CliPlugin.DhcpRelayHelperCli import (
   drCounterConfig, drStatus,
)
from CliPlugin.DhcpSnoopingCli import dhcpSnoopingSupportedGuard
from CliPlugin.Dhcpv6SnoopingCli import dhcp6SnoopingSupportedGuard
from CliPlugin.VirtualIntfRule import VirtualIntfMatcher
from CliPlugin.VxlanCli import VxlanIntf
from CliToken.Dhcp import dhcpMatcherForConfig
from CliToken.Ip import ipMatcherForClear
from CliToken.Ipv6 import ipv6MatcherForClear
from CliToken.Lanz import matcherAccessList
from CommonGuards import ssoStandbyGuard

dhcp6SnoopingCounterConfig = None
aclCheckpoint = None
aclStatus = None
dhcpSnoopingCounterConfig = None
dhcp6SnoopingCounterConfig = None

nodeClear = CliCommand.guardedKeyword( 'clear', helpdesc='Reset functions', 
      guard=ssoStandbyGuard )
addressFamilyExprForClear = CliMatcher.EnumMatcher( {
   'ip' : ipMatcherForClear.helpdesc_,
   'ipv6' : ipv6MatcherForClear.helpdesc_,
} )
nodeSnoopingGuarded = CliCommand.guardedKeyword( 
   'snooping', helpdesc='DHCP Snooping information',
   guard=dhcpSnoopingSupportedGuard )
nodeSnooping6Guarded = CliCommand.guardedKeyword( 
   'snooping', helpdesc='DHCPv6 Snooping information',
   guard=dhcp6SnoopingSupportedGuard )
matcherCounters = CliMatcher.KeywordMatcher( 'counters', helpdesc='counters' )
matcherDebug = CliMatcher.KeywordMatcher( 'debug',
      helpdesc='Internal counters for debugging' )

#--------------------------------------------------------------------------------
# clear ( ip | ipv6 ) dhcp relay counters INTF
#--------------------------------------------------------------------------------
def resetDhcpRelayCounters( mode, args ):
   intf = args.get( 'INTF' )
   if intf is None:
      # pylint: disable-msg=E1103
      # Pylint generates the following very strange error here:
      # Instance of '_Proxy' has no 'resetAllCountersTrigger' member
      # (but some types could not be inferred)
      drCounterConfig().resetAllCountersTrigger = \
            ( drCounterConfig().resetAllCountersTrigger + 1 ) % 256
   else:
      intfId = intf.name
      # Only act on the clear counter command when there is an intfConfig
      # or egressIntf associated with it. If the counter intfConfig doesn't exist,
      # then create one. Counter intfConfig is never deleted.
      counterIntfs = drStatus().counterIntfStatus
      if intfId in counterIntfs:
         intfCounterConfig = drCounterConfig().intfConfig.get( intfId )
         if intfCounterConfig is None:
            intfCounterConfig = drCounterConfig().intfConfig.newMember( intfId )
         intfCounterConfig.resetCountersTrigger = \
            ( intfCounterConfig.resetCountersTrigger + 1 ) % 256
         return
      # do nothing until the actual intfConfig is created

vxlanIntfMatcher = VirtualIntfMatcher( 'Vxlan', 1, 1,
      value=lambda mode, intf: VxlanIntf( intf, mode ),
      helpdesc="VXLAN Tunnel Interface" )
routedOrVxlanIntfMatcher = IntfCli.Intf.matcherWithRoutingProtoSupport
routedOrVxlanIntfMatcher |= vxlanIntfMatcher

class ClearDhcpRelayCountersCmd( CliCommand.CliCommandClass ):
   syntax = 'clear ADDR_FAMILY dhcp relay counters [ interface INTF ]'
   data = {
      'clear' : nodeClear,
      'ADDR_FAMILY' : addressFamilyExprForClear,
      'dhcp' : dhcpMatcherForConfig,
      'relay' : 'DHCP Relay',
      'counters' : matcherCounters,
      'interface' : 'Specify an interface as a target for the command',
      'INTF' : routedOrVxlanIntfMatcher
   }

   handler = resetDhcpRelayCounters

BasicCliModes.EnableMode.addCommandClass( ClearDhcpRelayCountersCmd )

#--------------------------------------------------------------------------------
# clear ip dhcp relay counters access-list
#--------------------------------------------------------------------------------
def clearAclCounters( mode, args ):
   aclType = args[ 'ADDR_FAMILY' ]
   AclCli.clearServiceAclCounters( mode, aclStatus, aclCheckpoint, aclType )

class ClearIpDhcpRelayCountersAccessListCmd( CliCommand.CliCommandClass ):
   syntax = 'clear ADDR_FAMILY dhcp relay counters access-list'
   data = {
      'clear' : nodeClear,
      'ADDR_FAMILY' : addressFamilyExprForClear,
      'dhcp' : dhcpMatcherForConfig,
      'relay' : 'DHCP Relay',
      'counters' : matcherCounters,
      'access-list' : matcherAccessList,
   }

   # Moving this function into this file leads to a failure in CliTest004.py
   handler = clearAclCounters

BasicCliModes.EnableMode.addCommandClass( ClearIpDhcpRelayCountersAccessListCmd )

#--------------------------------------------------------------------------------
# clear ip dhcp snooping counters
#--------------------------------------------------------------------------------
def clearDhcpSnoopingCounters( mode, args ):
   dhcpSnoopingCounterConfig.resetCountersTrigger = \
      ( dhcpSnoopingCounterConfig.resetCountersTrigger + 1 ) % 256

class ClearIpDhcpSnoopingCountersCmd( CliCommand.CliCommandClass ):
   syntax = 'clear ip dhcp snooping counters'
   data = {
      'clear' : nodeClear,
      'ip' : ipMatcherForClear,
      'dhcp' : dhcpMatcherForConfig,
      'snooping' : nodeSnoopingGuarded,
      'counters' : matcherCounters,
   }

   handler = clearDhcpSnoopingCounters

BasicCliModes.EnableMode.addCommandClass( ClearIpDhcpSnoopingCountersCmd )

#--------------------------------------------------------------------------------
# clear ip dhcp snooping counters debug
#--------------------------------------------------------------------------------
def clearDhcpSnoopingDebugCounters( mode, args ):
   dhcpSnoopingCounterConfig.resetDebugCountersTrigger = \
      ( dhcpSnoopingCounterConfig.resetDebugCountersTrigger + 1 ) % 256

class ClearIpDhcpSnoopingCountersDebugCmd( CliCommand.CliCommandClass ):
   syntax = 'clear ip dhcp snooping counters debug'
   data = {
      'clear' : nodeClear,
      'ip' : ipMatcherForClear,
      'dhcp' : dhcpMatcherForConfig,
      'snooping' : nodeSnoopingGuarded,
      'counters' : matcherCounters,
      'debug' : matcherDebug,
   }

   handler = clearDhcpSnoopingDebugCounters

BasicCliModes.EnableMode.addCommandClass( ClearIpDhcpSnoopingCountersDebugCmd )

#--------------------------------------------------------------------------------
# clear ipv6 dhcp snooping counters
#--------------------------------------------------------------------------------
def clearDhcp6SnoopingCounters( mode, args ):
   dhcp6SnoopingCounterConfig.resetCountersTrigger = \
      ( dhcp6SnoopingCounterConfig.resetCountersTrigger + 1 ) % 256

class ClearIpv6DhcpSnoopingCountersCmd( CliCommand.CliCommandClass ):
   syntax = 'clear ipv6 dhcp snooping counters'
   data = {
      'clear' : nodeClear,
      'ipv6' : ipv6MatcherForClear,
      'dhcp' : dhcpMatcherForConfig,
      'snooping' : nodeSnooping6Guarded,
      'counters' : matcherCounters,
   }

   handler = clearDhcp6SnoopingCounters

BasicCliModes.EnableMode.addCommandClass( ClearIpv6DhcpSnoopingCountersCmd )

#--------------------------------------------------------------------------------
# clear ipv6 dhcp snooping counters debug
#--------------------------------------------------------------------------------
def clearDhcp6SnoopingDebugCounters( mode, args ):
   dhcp6SnoopingCounterConfig.resetDebugCountersTrigger = \
      ( dhcp6SnoopingCounterConfig.resetDebugCountersTrigger + 1 ) % 256

class ClearIpv6DhcpSnoopingCountersDebugCmd( CliCommand.CliCommandClass ):
   syntax = 'clear ipv6 dhcp snooping counters debug'
   data = {
      'clear' : nodeClear,
      'ipv6' : ipv6MatcherForClear,
      'dhcp' : dhcpMatcherForConfig,
      'snooping' : nodeSnooping6Guarded,
      'counters' : matcherCounters,
      'debug' : matcherDebug,
   }

   handler = clearDhcp6SnoopingDebugCounters

BasicCliModes.EnableMode.addCommandClass( ClearIpv6DhcpSnoopingCountersDebugCmd )

def Plugin( entityManager ):
   global aclStatus
   global aclCheckpoint
   global dhcpSnoopingCounterConfig
   global dhcp6SnoopingCounterConfig

   dhcp6SnoopingCounterConfig = \
         LazyMount.mount( entityManager, "bridging/dhcpsnooping/dhcp6CounterConfig",
               "Bridging::DhcpSnooping::CounterConfig", "w" )
   aclStatus = LazyMount.mount( entityManager, "acl/status/all",
                                "Acl::Status", "r" )
   aclCheckpoint = LazyMount.mount( entityManager, "acl/checkpoint",
                                    "Acl::CheckpointStatus", "w" )
   dhcpSnoopingCounterConfig = LazyMount.mount(
      entityManager, "bridging/dhcpsnooping/counterConfig",
      "Bridging::DhcpSnooping::CounterConfig", "w" )
   dhcp6SnoopingCounterConfig = LazyMount.mount(
      entityManager, "bridging/dhcpsnooping/dhcp6CounterConfig",
      "Bridging::DhcpSnooping::CounterConfig", "w" )
