# Copyright (c) 2019 Arista Networks, Inc.  All rights reserved.
# Arista Networks, Inc. Confidential and Proprietary.

from CliModel import Model, Submodel, Dict, Int, Float
from IntfModel import Interface
from TableOutput import createTable, Format
from Arnet import sortIntf
from Ark import utcTimeRelativeToNowStr

class Counters( Model ):
   received = Int( help="Received packets" )
   forwarded = Int( help="Forwarded packets" )
   dropped = Int( help="Dropped packets" )

class GlobalCounters( Model ):
   allRequests = Submodel( valueType=Counters,
                          help="Global counter for requests" )
   allResponses = Submodel( valueType=Counters,
                           help="Global counter for responses" )
   lastResetTime = Float( help="Last reset time for global counters" )

class InterfaceCounter( Model ):
   requests = Submodel( valueType=Counters,
                       help="Interface counter requests" )
   replies = Submodel( valueType=Counters,
                     help="Interface counter for replies" )
   lastResetTime = Float( help="Last reset time for interface counter" )

class DhcpRelayCounterModel( Model ):
   globalCounters = Submodel( valueType=GlobalCounters,
                             help="DHCP relay global counters" )
   interfaceCounters = Dict( keyType=Interface, valueType=InterfaceCounter,
                             help="A mapping of interface to its counters",
                             optional=True )

   def render( self ):
      f1 = Format( justify="right" )
      f1.padLimitIs( True )

      globalCounterReq = self.globalCounters.allRequests
      globalCounterResp = self.globalCounters.allResponses
      outputHeader = [ "Globals", \
            ( "DHCP Packets", "ch", ( "Rcvd", "Fwdd", "Drop" ) ),
            "Last Cleared" ]
      outputTable = createTable( outputHeader )
      globalOutput = [ [ "All Req", globalCounterReq.received,
                                    globalCounterReq.forwarded,
                                    globalCounterReq.dropped,
                                    utcTimeRelativeToNowStr(
                                       self.globalCounters.lastResetTime ) ],
                       [ "All Resp", globalCounterResp.received,
                                     globalCounterResp.forwarded,
                                     globalCounterResp.dropped,
                                     "" ], ]
      for row in globalOutput:
         outputTable.newRow( *row )
      outputTable.formatColumns( f1, f1, f1, f1, f1 )
      print "\n" + outputTable.output()

      if not self.interfaceCounters:
         return

      def prepareIntfCounterOutput( intfname, intfCounterVal ):
         output = [ intfname,
                    intfCounterVal.requests.received,
                    intfCounterVal.requests.forwarded,
                    intfCounterVal.requests.dropped,
                    intfCounterVal.replies.received,
                    intfCounterVal.replies.forwarded,
                    intfCounterVal.replies.dropped,
                    utcTimeRelativeToNowStr(
                       intfCounterVal.lastResetTime ) ]
         return output

      interfaceHeaders = [ "Interface",
            ( "DHCP Request Packets", "ch", ( "Rcvd", "Fwdd", "Drop" ) ),
            ( "DHCP Reply Packets", "ch", ( "Rcvd", "Fwdd", "Drop" ) ),
            "Last Cleared" ]
      outputTable = createTable( interfaceHeaders )
      interfaceOutput = []
      for intfname in sortIntf( self.interfaceCounters ):
         intfCounterVal = self.interfaceCounters[ intfname ]
         interfaceOutput.append(
               prepareIntfCounterOutput( intfname, intfCounterVal ) )
      for row in interfaceOutput:
         outputTable.newRow( *row )
      outputTable.formatColumns( f1, f1, f1, f1, f1, f1, f1, f1 )
      print outputTable.output()
