# Copyright (c) 2019 Arista Networks, Inc.  All rights reserved.
# Arista Networks, Inc. Confidential and Proprietary.

from Arnet import sortIntf
from CliModel import (
                       Bool,
                       Dict,
                       List,
                       Model,
                       Str,
                       Submodel
                     )
from IntfModel import Interface
from ArnetModel import IpGenericAddress, Ip6Address
from IpLibConsts import DEFAULT_VRF

def printState( condition ):
   return "enabled" if condition else "disabled"

class Helper( Model ):
   serverAddr = IpGenericAddress( help="Helper address", optional=True )
   serverHostname = Str( help="Helper hostname", optional=True )
   vrf = Str( help="Server VRF" )
   srcAddr = IpGenericAddress( help="Source address", optional=True )
   srcIntf = Interface( help="Source interface", optional=True )
   linkAddr = Ip6Address( help="Link address", optional=True )

   def render( self ):
      toPrint = []
      if self.serverAddr:
         toPrint.append( str( self.serverAddr ) )
      else:
         toPrint.append( self.serverHostname )

      if self.vrf != DEFAULT_VRF:
         toPrint += [ 'vrf', self.vrf ]
      
      if self.linkAddr:
         toPrint += [ 'link address', str( self.linkAddr ) ]
      
      print " ".join( toPrint )
      
def printHelper( helper, helperCount ):
   tab = " " * len( "  DHCPvX servers:" )
   if helperCount != 0:
      print tab,
   helper.render()
   return helperCount + 1

def getAllHelpersAsLists( helpers ):
   helperListIpv4 = []
   helperListIpv6 = []
   for helper in sorted( helpers, key=compareHelper ):
      ipAddr = helper.serverAddr
      # get DHCPv4 servers
      if ( ( ipAddr and ipAddr.af == 'ipv4' ) or helper.serverHostname ):
         helperListIpv4.append( helper )
      else:
         # get DHCPv6 servers
         helperListIpv6.append( helper )

   return ( helperListIpv4, helperListIpv6 )

# We sort the output based on serverAddress / serverHostname
# respectively, so the order of CLI output does not depend on
# iteration order of a HashTable.
def compareHelper( helper ):
   # Order the DHCP servers by VRF, then by address or
   # hostname, displaying DHCP servers with an address before
   # ones with a hostname.
   return ( helper.vrf, helper.serverHostname, helper.serverAddr )

class ConfiguredInterface( Model ):
   circuitId = Str( help="Circuit ID enabled interface", default='' )
   smartRelay = Bool( help="Smart relay enabled interface",
                      default=False )
   allSubnetsV6 = Bool( help="All subnet relaying enabled interface",
                        default=False )
   helpers = List( valueType=Helper,
                   help="Helper addresses/hostnames for an interface",
                   optional=True )
   usingGlobalModeServersV4 = Bool( help="Using DHCPv4 Relay mode configurations",
                                    optional=True )
   disabledV4 = Bool( help="DHCPv4 Relay is disabled", default=False )
   def render( self ):
      if self.circuitId:
         print "  Option 82 Circuit ID:", self.circuitId
      print "  DHCP Smart Relay is", printState( self.smartRelay)
      print "  DHCPv6 all subnet relaying is", printState( self.allSubnetsV6 )
      
      if self.usingGlobalModeServersV4 and not self.disabledV4:
         print "  Using default DHCPv4 servers"

      if self.helpers:
         helperListIpv4, helperListIpv6 = getAllHelpersAsLists( self.helpers )
         if ( helperListIpv4 and 
              not ( self.usingGlobalModeServersV4 or self.disabledV4 ) ):
            helperCount = 0
            print "  DHCPv4 servers:",
            for helper in helperListIpv4:
               helperCount = printHelper( helper, helperCount )

         if helperListIpv6:
            helperCount = 0 
            print "  DHCPv6 servers:",
            for helper in helperListIpv6:
               helperCount = printHelper( helper, helperCount )
   
      if self.disabledV4:
         print "  DHCPv4 Relay is disabled"

# Model to display dhcp relay global mode configurations
class DefaultServersShowModel( Model ):
   helpersV4 = List( valueType=Helper,
                   help="DHCP Relay mode default IPv4 addresses/hostnames",
                   optional=True )

   def render( self ):
      # servers
      if self.helpersV4:
         print "Default L3 interface DHCP servers:"
         # IPV4 addresses
         if self.helpersV4:
            helperCount = 0
            print "  DHCPv4 servers:",
            for helper in self.helpersV4:
               helperCount = printHelper( helper, helperCount )

class IpDhcpRelay( Model ):
   activeState = Bool( help="DHCP Relay is active" )
   alwaysOn = Bool( help="DHCP relay alwaysOn state", default=False )
   option82 = Bool( help="DHCP Relay Option 82 is enabled", default=False )
   linkLayerAddrOpt = \
         Bool( help="DHCPv6 Relay Link-layer Address (Option 79) is enabled", \
         default=False )
   remoteIdEncodingFormat = \
         Str( help="DHCPv6 Relay Remote ID (Option 37) encoding format", \
         default='%m:%i' )
   smartRelay = Bool( help="DHCP Smart Relay is enabled", default=False )
   allSubnetsV6 = Bool( help="DHCPv6 all subnet relaying is enabled", default=False )
   tunnelReqDisable = Bool( help="DHCP tunnel requests are disabled", default=False )
   configuredInterfaces = Dict( keyType=Interface, valueType=ConfiguredInterface,
                           help="A mapping of interface to its DHCP configuration" )
   defaultServers = Submodel( valueType=DefaultServersShowModel,
                              help="DHCP Relay mode default addresses/hostnames",
                              optional=True )
   def render( self ):
      print "DHCP Relay is", "active" if self.activeState else "not active"
      if self.activeState:
         if self.alwaysOn:
            print "DHCP Relay always-on mode enabled"
         print "DHCP Relay Option 82 is", printState( self.option82 )
         print "DHCPv6 Relay Link-layer Address Option (79) is", \
               printState( self.linkLayerAddrOpt )
         fmt = self.remoteIdEncodingFormat
         if fmt in ( '%m:%i', '%m:%p' ):
            fmt = 'ID' if fmt == '%m:%i' else 'name'
            print "DHCPv6 Relay Remote ID (Option 37) encoding format:" \
                  " MAC address:interface", fmt
         print "DHCP Smart Relay is", printState( self.smartRelay )
         print "DHCPv6 all subnet relaying is", printState( self.allSubnetsV6 )
         if self.tunnelReqDisable:
            print "DHCP tunnel requests are disabled"
      
      # print dhcp relay mode servers
      if self.defaultServers:
         self.defaultServers.render()

      for intfName in sortIntf( self.configuredInterfaces ):
         print "Interface:", intfName
         self.configuredInterfaces[ intfName ].render()
