# Copyright (c) 2010 Arista Networks, Inc.  All rights reserved.
# Arista Networks, Inc. Confidential and Proprietary.

from __future__ import absolute_import, division, print_function

import Tac, Debug
import CliMatcher, CliGlobal, BasicCliModes, CliCommand

#------------------------------------------------------------------------------------
# This module implements the CLI for configuring global debug message settings.
# It provides the following commands:
#
# - [ no ] debug <categories> <message>
# - show debugging
#
#------------------------------------------------------------------------------------

matcherDebug = CliMatcher.KeywordMatcher( 'debug',
   helpdesc='Configure debug messages' )
matcherUndebug = CliMatcher.KeywordMatcher( 'undebug',
   helpdesc='Disable debug messages' )
matcherNo = CliMatcher.KeywordMatcher( 'no',
   helpdesc='Disable the following' )

gv = CliGlobal.CliGlobal( dict( debugConfig=None, categoryHelpDescs=None,
   messageHelpDescs=None, messagesFromCategory=None, categoriesFromParent=None ) )

#------------------------------------------------------------------------------------
# ( ( [ no ] debug ) | undebug )...
# This builds the debug category hierarchy dynamically
#------------------------------------------------------------------------------------
 
def setMessageType( category, messageName, enable ):
   message = Tac.nonConst( category.messageType[ messageName ] )
   message.enabled = enable
   category.messageType.addMember( message )

def setMessageTypePath( fullPath, enable ):
   resolvedPath = Debug.walkMessagePath( gv.debugConfig, fullPath )
   setMessageType( resolvedPath.category, resolvedPath.messageName, enable )

def disableCategoryMessages( category, categoryStack=None ):
   for message in category.messageType:
      setMessageType( category, message, False )

def disableCategoryPath( mode, args ):
   category = Debug.walkCategoryPath( gv.debugConfig, args[ 'FULLPATH' ] )
   Debug.walkDebugCliTree( category, disableCategoryMessages )

def ruleStackPusher( item, ruleStack ):
   ruleStack.append( item )

def fullPathAdapter( mode, args, argsList ):
   """adds a FULLPATH field in args, which is a list of matching keywords
      (categories and an optional message) in the order they appear
   """
   startIndex = 2 if 'no' in args else 1 # exclude 'no', 'debug', and 'undebug'
   args[ 'FULLPATH' ]  = [ i[ 1 ] for i in argsList[ startIndex : ] ]

def getDebugCategories( mode, context ):
   """retrieve all valid child categories based on the last matching category"""
   lastCategory = '' if not context.sharedResult else \
      context.sharedResult[ 'CATEGORIES' ][ -1 ]

   return gv.categoriesFromParent[ lastCategory ]

def getMessagesForCategory( mode, context ):
   """retrieve all valid messages for the last matching category"""
   lastCategory = context.sharedResult[ 'CATEGORIES' ][ -1 ]
   return gv.messagesFromCategory[ lastCategory ]

#---------------------------------------------------------------
# ( ( no debug ) | undebug ) { CATEGORIES }
#---------------------------------------------------------------
class NoDebugCategoryCmd( CliCommand.CliCommandClass ):
   syntax = '( ( no debug ) | undebug ) { CATEGORIES }'
   data = {
      'no' : matcherNo,
      'debug' : matcherDebug,
      'undebug' : matcherUndebug,
      'CATEGORIES' : CliCommand.Node( CliMatcher.DynamicKeywordMatcher(
                                                      getDebugCategories,
                                                      passContext=True ),
                                      storeSharedResult=True ),
   }
   adapter = fullPathAdapter
   handler = disableCategoryPath

BasicCliModes.EnableMode.addCommandClass( NoDebugCategoryCmd )

#-----------------------------------------------------------------
# ( ( [ no ] debug ) | undebug ) { CATEGORIES } MESSAGE
#-----------------------------------------------------------------
class DebugOrNoMessageCmd( CliCommand.CliCommandClass ):
   syntax = '( ( [ no ] debug ) | undebug ) { CATEGORIES } MESSAGE'
   data = {
      'no' : matcherNo,
      'debug' : matcherDebug,
      'undebug' : matcherUndebug,
      'CATEGORIES' : CliCommand.Node( CliMatcher.DynamicKeywordMatcher(
                                                      getDebugCategories,
                                                      passContext=True ),
                                      storeSharedResult=True ),
      'MESSAGE' : CliCommand.Node( CliMatcher.DynamicKeywordMatcher(
                                                      getMessagesForCategory,
                                                      passContext=True ) ),
   }
   adapter = fullPathAdapter

   @staticmethod
   def handler( mode, args ):
      enable = ( 'undebug' not in args and 'no' not in args )
      setMessageTypePath( args[ 'FULLPATH' ], enable )
   
BasicCliModes.EnableMode.addCommandClass( DebugOrNoMessageCmd )

def addRulesForCategory( category, ruleStack ):
   # map the current category to its helpdesc if it doesn't already exist,
   # which avoids conflicting helpdescs
   if category.name not in gv.categoryHelpDescs:
      gv.categoryHelpDescs[ category.name ] = category.helpDesc

   # track all child categories for this current category
   if category.name not in gv.categoriesFromParent:
      gv.categoriesFromParent[ category.name ] = {}

   # add this category to its parent category
   parent = '' if len( ruleStack ) == 1 else ruleStack[ len( ruleStack ) - 2 ].name
   gv.categoriesFromParent[ parent ][ category.name ] = \
      gv.categoryHelpDescs[ category.name ]

   # add the associated messages for the current category
   if category.name not in gv.messagesFromCategory:
      gv.messagesFromCategory[ category.name ] = {}

   # add commands for each message in the current category
   for messageName, message in category.messageType.items():
      # map the message to its helpdesc if it doesn't exist already
      if messageName not in gv.messageHelpDescs:
         gv.messageHelpDescs[ messageName ] = message.helpDesc
      gv.messagesFromCategory[ category.name ][ messageName ] = \
         gv.messageHelpDescs[ messageName ]

#------------------------------------------------------------------------------------
# no debug - Disable all debugging messages
#------------------------------------------------------------------------------------

def disableAllCategories( mode, args ):
   Debug.walkDebugCliTree( gv.debugConfig, disableCategoryMessages )

#------------------------------------------------------------------------------------
# show debugging
#------------------------------------------------------------------------------------

class CategoryPrintState( object ):
   def __init__( self, category ):
      self.category = category
      self.printed = False

showDebugIndent = '  '

def printCategoryNames( categoryStack ):
   indent = 0
   for categoryState in categoryStack:
      if not categoryState.printed:
         categoryState.printed = True
         category = categoryState.category
         print( '{}{} - {}:'.format( showDebugIndent * indent, category.name,
                                     category.helpDesc ) )
      indent += 1

def showDebugCategoryStackPusher( category, categoryStack ):
   categoryStack.append( CategoryPrintState( category ) )

def showCategory( category, categoryStack ):
   printedCategoryNames = False

   indentLevel = len( categoryStack )
   indent = showDebugIndent * indentLevel

   for message in category.messageType.values():
      if message.enabled:
         if not printedCategoryNames:
            printCategoryNames( categoryStack )
            printedCategoryNames = True
         print( '{}{} - {} debugging is on'.format( indent, message.name,
                                                   message.helpDesc ) )

def showDebug( mode, args ):
   """Show all enabled debug settings"""
   Debug.walkDebugCliTree( gv.debugConfig, showCategory, 
                           categoryStackPusher=showDebugCategoryStackPusher )

def Plugin( entityManager ):
   # pylint: disable-msg=W0603
   mg = entityManager.mountGroup()
   gv.debugConfig = mg.mount( 'debug/config', 'Debug::Config', 'w' )

   gv.categoryHelpDescs = {} # map a category to its helpdesc
   gv.messageHelpDescs = {} # map a message to its helpdesc
   gv.categoriesFromParent = {} # map a category to its child categories
   gv.categoriesFromParent[ '' ] = {} # root categories
   gv.messagesFromCategory = {} # maps a category to its messages

   def _finishMounts():
      Debug.walkDebugCliTree( gv.debugConfig, addRulesForCategory,
                              categoryStackPusher=ruleStackPusher )
   mg.close( _finishMounts )
