# Copyright (c) 2009, 2010, 2011 Arista Networks, Inc.  All rights reserved.
# Arista Networks, Inc. Confidential and Proprietary.

import time

import Arnet
import BasicCli
import CliCommand
import CliMatcher
import CliParser
import CliPlugin.IntfCli as IntfCli
import CliPlugin.EthIntfCli as EthIntfCli
import CliPlugin.TechSupportCli
import ConfigMount
import Intf.IntfRange
import DcbxLib
from DcbxTypes import ApplicationPriorityEntryKey
from DcbxTypes import applicationPriorityEntry
from DcbxTypes import applicationTable
import LazyMount
import ShowCommand
import Tac
import Tracing

t0 = Tracing.trace0

dcbxConfig = None
pfcConfig = None
dcbxStatus = None
lldpConfig = None

matcherDcbx = CliMatcher.KeywordMatcher( 'dcbx',
      helpdesc='Configure DCBX' )
matcherEts = CliMatcher.KeywordMatcher( 'ets',
      helpdesc='Configure the CEE DCBX priority group' )
matcherTrafficClass = CliMatcher.KeywordMatcher( 'traffic-class',
      helpdesc='Configure a Qos map' )
matcherApplication = CliMatcher.KeywordMatcher( 'application',
      helpdesc='Configure the IEEE DCBX application priority table' )

class DcbxIntf( IntfCli.IntfDependentBase ):
   def setDefault( self ):
      del dcbxConfig.portEnabled[ self.intf_.name ]

# Straight from LldConfigCli.py
class DcbxModelet( CliParser.Modelet ):
   modeletParseTree = CliParser.ModeletParseTree()
   def __init__( self, mode ):
      # This is a bit hacky, but there's no better way to recognize physical Ethernet
      # interfaces than to look at their name.
      CliParser.Modelet.__init__( self )

   @staticmethod
   def shouldAddModeletRule( mode ):
      return ( mode.intf.name.startswith( 'Ethernet' ) and 
               not mode.intf.isSubIntf() )

IntfCli.IntfConfigMode.addModelet( DcbxModelet )
#--------------------------------------------------------------------------------
# [ no | default ] dcbx mode MODE
#--------------------------------------------------------------------------------
class DcbxModeCmd( CliCommand.CliCommandClass ):
   syntax = 'dcbx mode MODE'
   noOrDefaultSyntax = 'dcbx mode ...'
   data = {
      'dcbx' : matcherDcbx,
      'mode' : 'Select DCBX mode',
      'MODE' : CliMatcher.EnumMatcher( {
         'none' : 'Disable DCBX',
         'ieee' : 'Select IEEE DCBX mode',
         'cee' : 'Select CEE DCBX mode',
      } )
   }

   @staticmethod
   def handler( mode, args ):
      newDcbxMode = args.get( 'MODE', 'none' )
      if newDcbxMode == 'none':
         if mode.intf.name in dcbxConfig.portEnabled:
            del dcbxConfig.portEnabled[ mode.intf.name ]
            del dcbxConfig.portDcbxEnabled[ mode.intf.name ]
      else:
         dcbxConfig.portDcbxEnabled[ mode.intf.name ] = True
         if newDcbxMode == 'ieee':
            dcbxConfig.portEnabled[ mode.intf.name ] = "modeIeee"
         else:
            _initCeeETSConfigInfo()
            dcbxConfig.portEnabled[ mode.intf.name ] = "modeCee"

   noOrDefaultHandler = handler

DcbxModelet.addCommandClass( DcbxModeCmd )

# dcbx ets qos map cos <0-7> traffic-class <0-7>
# dcbx ets traffic-class <0-7> bandwidth <1-100> 
# no dcbx ets
# no dcbx ets qos map cos <0-7> traffic-class <0-7>
# no dcbx ets traffic-class <0-7> bandwidth <1-100> 

def _initCeeETSConfigInfo():
   if not dcbxConfig.ceeETSConfigInfo:
      Tac.newInstance( "Dcbx::CeeETSConfigInfo", "CeeETSConfigInfo" )
      dcbxConfig.ceeETSConfigInfo = ("CeeETSConfigInfo",)
      for c in range(0, 8):
         dcbxConfig.ceeETSConfigInfo.cosToTrafficClass[ c ] = 0
      for t in range(0, 8):
         dcbxConfig.ceeETSConfigInfo.trafficClassBandwidth[ t ] = 0

def _getPGEnabled():
   for cos in range( 0, 8 ):
      # If at least one TC has config then mark enable as True
      if ( dcbxConfig.ceeETSConfigInfo.cosToTrafficClass[ cos ] or
           dcbxConfig.ceeETSConfigInfo.trafficClassBandwidth[ cos ] ):
         return True

   return False

def _setEtsQMap( mode, cos, trafficClass ):
   _initCeeETSConfigInfo()
   dcbxConfig.ceeETSConfigInfo.cosToTrafficClass[ cos ] = trafficClass
   dcbxConfig.ceeETSConfigInfo.enable = _getPGEnabled()

def _setTrafficBandwidth( mode, trafficClass, bandwidth ):
   dcbxConfig.ceeETSConfigInfo.trafficClassBandwidth[ trafficClass ] = bandwidth
   dcbxConfig.ceeETSConfigInfo.enable = _getPGEnabled()

#--------------------------------------------------------------------------------
# [ no | default ] dcbx ets qos map cos COS traffic-class TRAFFIC_CLASS
#--------------------------------------------------------------------------------
class EtsQMapCmd( CliCommand.CliCommandClass ):
   syntax = 'dcbx ets qos map cos COS traffic-class TRAFFIC_CLASS'
   noOrDefaultSyntax = syntax
   data = {
      'dcbx' : matcherDcbx,
      'ets' : matcherEts,
      'qos' : 'Configure a Qos map',
      'map' : 'Configure a Qos map',
      'cos' : 'Configure a Qos map',
      'COS' : CliMatcher.IntegerMatcher( 0, 7, helpdesc='Cos' ),
      'traffic-class' : matcherTrafficClass,
      'TRAFFIC_CLASS' : CliMatcher.IntegerMatcher( 0, 7,
         helpdesc='Internal class of traffic' ),
   }

   @staticmethod
   def handler( mode, args ):
      _setEtsQMap( mode, args[ 'COS' ], args[ 'TRAFFIC_CLASS' ] )

   @staticmethod
   def noOrDefaultHandler( mode, args ):
      if not dcbxConfig.ceeETSConfigInfo:
         return
      _setEtsQMap( mode, args[ 'COS' ], 0 )

BasicCli.GlobalConfigMode.addCommandClass( EtsQMapCmd )

#--------------------------------------------------------------------------------
# [ no | default ] dcbx ets traffic-class TRAFFIC_CLASS bandwidth BANDWIDTH
#--------------------------------------------------------------------------------
class EtsTrafficBandwidthCmd( CliCommand.CliCommandClass ):
   syntax = 'dcbx ets traffic-class TRAFFIC_CLASS bandwidth BANDWIDTH'
   noOrDefaultSyntax = syntax
   data = {
      'dcbx' : matcherDcbx,
      'ets' : matcherEts,
      'traffic-class' : matcherTrafficClass,
      'TRAFFIC_CLASS' : CliMatcher.IntegerMatcher( 0, 7,
         helpdesc='Internal class of traffic' ),
      'bandwidth' : 'Configure bandwidth',
      'BANDWIDTH' : CliMatcher.IntegerMatcher( 0, 100,
         helpdesc='Bandwidth % - default: 0' ),
   }

   @staticmethod
   def handler( mode, args ):
      trafficClass = args[ 'TRAFFIC_CLASS' ]
      bandwidth = args[ 'BANDWIDTH' ]
      _initCeeETSConfigInfo()
      s1 = sum( dcbxConfig.ceeETSConfigInfo.trafficClassBandwidth[ cos ]
                for cos in range( 0, trafficClass ) )
      s2 = sum( dcbxConfig.ceeETSConfigInfo.trafficClassBandwidth[ cos ]
                for cos in range( trafficClass+1, 8 ) )
      total = s1 + s2 + bandwidth
      if total > 100:
         mode.addError( 'Total bandwidth allocation for all Traffc Classes '
                        'cannot exceed 100%' )
         return
      _setTrafficBandwidth( mode, trafficClass, bandwidth )

   @staticmethod
   def noOrDefaultHandler( mode, args ):
      if not dcbxConfig.ceeETSConfigInfo:
         return
      _setTrafficBandwidth( mode, args[ 'TRAFFIC_CLASS' ], 0 )

BasicCli.GlobalConfigMode.addCommandClass( EtsTrafficBandwidthCmd )

#--------------------------------------------------------------------------------
# [ no | default ] dcbx application
#              ( ( ether ETHER_PORT ) | ( PROTOCOL PORT ) | GROUP ) priority PRIORITY
#--------------------------------------------------------------------------------
class ApplicationPriorityCmd( CliCommand.CliCommandClass ):
   syntax = ( 'dcbx application '
              '( ( ether ETHER_PORT ) | ( PROTOCOL PORT ) | GROUP ) '
              'priority PRIORITY' )
   noOrDefaultSyntax = ( 'dcbx application '
                         '( ( ether ETHER_PORT ) | ( PROTOCOL PORT ) | GROUP ) '
                         'priority ...' )
   data = {
      'dcbx' : matcherDcbx,
      'application' : matcherApplication,
      'ether' : 'Configure an EtherType rule',
      'ETHER_PORT' : CliMatcher.IntegerMatcher( 1536, 65535, helpdesc='EtherType' ),
      'PROTOCOL' : CliMatcher.EnumMatcher( {
         'tcp-sctp' : 'Configure a TCP/SCTP port number rule',
         'tcp-sctp-udp' : 'Configure a TCP/SCTP/UDP port number rule',
         'udp' : 'Configure a UDP port number rule',
      } ),
      'PORT' : CliMatcher.IntegerMatcher( 1, 65535,
         helpdesc='Number of the port to use' ),
      'GROUP' : CliMatcher.EnumMatcher( {
         'iscsi' : 'Configure the iSCSI rule',
      } ),
      'priority' : 'Configure application priority',
      'PRIORITY' : CliMatcher.IntegerMatcher( 0, 7, helpdesc='Priority' ),
   }

   @staticmethod
   def handler( mode, args ):
      no = CliCommand.isNoOrDefaultCmd( args )
      priority = args.get( 'PRIORITY', -1 )
      if 'GROUP' in args:
         applications = applicationTable[ args[ 'GROUP' ] ]
      elif 'ether' in args:
         applications = ( ( 'ether', args[ 'ETHER_PORT' ] ), )
      else:
         appMap =  { 'tcp-sctp': 'tcpSctp', 'udp': 'udp',
                     'tcp-sctp-udp': 'tcpSctpUdp' }
         applications = ( ( appMap[ args[ 'PROTOCOL' ] ], args[ 'PORT' ] ), )

      if not no and len( dcbxConfig.applicationPriorityEntry ) + \
            len( applications ) > 168:
         mode.addError(
               "You cannot configure more than 168 application priority "
               "table entries." )
         return

      for sel, protocolId in applications:
         if no:
            key = ApplicationPriorityEntryKey( sel, protocolId )
            # If the user specified a priority in the no command, only delete the
            # entry if that priority is actually the one specified in the table.
            if key in dcbxConfig.applicationPriorityEntry and (
                  priority < 0 or
                  dcbxConfig.applicationPriorityEntry[ key ].priority == priority ):
               del dcbxConfig.applicationPriorityEntry[ key ]
         else:
            entry = applicationPriorityEntry( priority, sel, protocolId )
            dcbxConfig.addApplicationPriorityEntry( entry )

   noOrDefaultHandler = handler

BasicCli.GlobalConfigMode.addCommandClass( ApplicationPriorityCmd )

#--------------------------------------------------------------------------------
# ( no | default ) dcbx application
#--------------------------------------------------------------------------------
class NoDcbxApplicationCmd( CliCommand.CliCommandClass ):
   noOrDefaultSyntax = 'dcbx application'
   data = {
      'dcbx' : matcherDcbx,
      'application' : matcherApplication,
   }

   @staticmethod
   def noOrDefaultHandler( mode, args ):
      """Clear the application priority table."""
      dcbxConfig.applicationPriorityEntry.clear()

BasicCli.GlobalConfigMode.addCommandClass( NoDcbxApplicationCmd )

#--------------------------------------------------------------------------------
# show dcbx [ INTF ] [ TYPE ]
#--------------------------------------------------------------------------------
def printLines( lines, prefix='  ' ):
   dash = "-"
   for line in lines:
      print prefix + dash, line
      dash = " "

def bitsSet( n ):
   bits = 0
   while n > 0:
      bits += n & 1
      n >>= 1
   return bits

def printPriorityFlowControlConfigInfo( pfcci, portName ):
   printLines( DcbxLib.priorityFlowControlConfigInfoStr( pfcci ) )
   pfcPortConfig = pfcConfig.portConfig.get( portName, None )
   if pfcPortConfig and pfcPortConfig.enabled and \
         bitsSet( pfcPortConfig.priorities ) > pfcci.pfcCapability:
      print "    WARNING: PFC is configured locally on more priorities " \
            "than the peer can support"
   if ( pfcPortConfig and (
            ( pfcPortConfig.enabled and 
                  pfcci.pfcEnable != pfcPortConfig.priorities ) or
            ( not pfcPortConfig.enabled and pfcci.pfcEnable ) ) ) or \
         ( not pfcPortConfig and pfcci.pfcEnable ):
      print "    WARNING: peer PFC configuration does not match " \
            "the local PFC configuration"

def printApplicationPriorityConfigInfo( apci, _portName ):
   printLines( DcbxLib.applicationPriorityConfigInfoStr( apci ) )
   # TODO: print a warning if our config doesn't match what we received

def showDcbxPort( intf, typ ):
   print "%s:" % intf
   if typ is None or typ == 'status':
      if intf in dcbxConfig.portEnabled and dcbxConfig.portEnabled[ intf ]:
         print "  IEEE DCBX is enabled",
         if intf in lldpConfig.portConfig:
            lldpAdminStatus = lldpConfig.portConfig[ intf ].adminStatus
            if lldpAdminStatus == 'txOnly':
               print "and only partially active because " \
                     "LLDP is enabled for transmit only"
            elif lldpAdminStatus == 'rxOnly':
               print "and only partially active because " \
                     "LLDP is enabled for receive only"
            elif lldpAdminStatus == 'txAndRx':
               print "and active"
            else:
               print "but not active because LLDP is disabled"
         else:
            print "but not active because LLDP is disabled"
   if intf not in dcbxStatus.peerPortStatus:
      print "  No IEEE DCBX TLVs were received"
      return
   pps = dcbxStatus.peerPortStatus[ intf ]

   timestampInUtcTime = pps.update + Tac.utcNow() - Tac.now()
   print "  Last LLDPDU received on", time.ctime( timestampInUtcTime )

   types = [
         ( 'priority-flow-control-configuration',
           'priorityFlowControlConfig',
           'priority flow control configuration' ),
         ( 'application-priority-configuration',
           'applicationPriorityConfig',
           'application priority configuration' )
           ]
   for cliType, codeType, humanType in types:
      # pylint: disable-msg=eval-used
      if typ is None or typ == cliType:
         received = eval( "pps.%sTlvReceived" % codeType )
         malformed = eval( "pps.%sTlvMalformed" % codeType )
         info = eval( "pps.%sInfo" % codeType )
         if received == 0:
            print "  No %s TLV received" % humanType
         elif malformed > 0:
            print "  ERROR: %d out of %d %s TLVs " \
                  "received were malformed" % ( malformed, received, humanType )
         elif received > 1:
            print "  ERROR: %d %s TLVs received" % ( received, humanType )
         else:
            function = eval( "print%s%sInfo" % (
                     codeType[ 0 ].upper(), codeType[ 1 : ] ) )
            function( info, intf )

class DcbxCmd( ShowCommand.ShowCliCommandClass ):
   syntax = 'show dcbx [ INTF ] [ TYPE ]'
   data = {
      'dcbx' : 'Show IEEE DCBX information',
      'INTF' : Intf.IntfRange.IntfRangeMatcher(
         explicitIntfTypes=( EthIntfCli.EthPhyAutoIntfType, ) ),
      'TYPE' : CliMatcher.EnumMatcher( {
         'application-priority-configuration' : ( 'Show IEEE DCBX peer application '
                                                  'priority configuration' ),
         'priority-flow-control-configuration' : ( 'Show IEEE DCBX peer priority '
                                                   'flow control configuration' ),
         'status' : 'Show DCBX status',
      } )
   }

   @staticmethod
   def handler( mode, args ):
      if 'INTF' in args:
         # Interfaces were specified explicitly.
         intfNames = args[ 'INTF' ].intfNames()
      else:
         candidates = set()
         for intf in dcbxStatus.peerPortStatus:
            candidates.add( intf )
         for intf in dcbxConfig.portEnabled:
            if dcbxConfig.portEnabled[ intf ]:
               candidates.add( intf )
         intfNames = Arnet.sortIntf( candidates )

      for intfName in intfNames:
         showDcbxPort( intfName, args.get( 'TYPE' ) )

BasicCli.addShowCommandClass( DcbxCmd )

#-----------------------------------------------------
# Register show dcbx command into "show tech-support".
#-----------------------------------------------------
# Timestamps are made up to maintain historical order within show tech-support
CliPlugin.TechSupportCli.registerShowTechSupportCmdCallback( 
                                '2010-01-01 00:08:30',
                                lambda : [ 'show dcbx' ] )

def Plugin( entityManager ):
   global dcbxConfig
   global pfcConfig
   global dcbxStatus
   global lldpConfig
   dcbxConfig = ConfigMount.mount(
      entityManager, "dcb/dcbx/config", "Dcbx::Config", "w" )
   pfcConfig = LazyMount.mount(
      entityManager, "dcb/pfc/config", "Pfc::Config", "r" )
   dcbxStatus = LazyMount.mount(
      entityManager, "dcb/dcbx/status", "Dcbx::Status", "r" )
   lldpConfig = LazyMount.mount(
      entityManager, "l2discovery/lldp/config", "Lldp::Config", "r" )
   IntfCli.Intf.registerDependentClass( DcbxIntf )
