# Copyright (c) 2020 Arista Networks, Inc.  All rights reserved.
# Arista Networks, Inc. Confidential and Proprietary.

import ArPyUtils
from ArnetModel import IpGenericAddress, IpGenericPrefix
from CliModel import Model, Int, List, Dict, Submodel

class NeighborModel( Model ):
   metric = Int( help='Link cost' )
   interfaceAddresses = List( valueType=IpGenericAddress, optional=True,
                              help='Local interface IP addresses' )
   neighborAddresses = List( valueType=IpGenericPrefix, optional=True,
                             help='Remote interface IP addresses' )

   def renderData( self, addrFamily ):
      print '  Metric:', self.metric
      if self.interfaceAddresses:
         print '', addrFamily, 'interface addresses:'
         # pylint: disable=not-an-iterable
         for addr in self.interfaceAddresses:
            # pylint: enable=not-an-iterable
            print ' ', addr
      if self.neighborAddresses:
         print '', addrFamily, 'neighbor addresses:'
         # pylint: disable=not-an-iterable
         for addr in self.neighborAddresses:
            # pylint: enable=not-an-iterable
            print ' ', addr

class ReachabilityModel( Model ):
   metric = Int( help='Link cost' )

class RouterModel( Model ):
   p2ps = Dict( keyType=str, valueType=NeighborModel, optional=True,
                help='Neighbor information per P2P neighbor' )
   lans = Dict( keyType=str, valueType=NeighborModel, optional=True,
                help='Neighbor information per LAN neighbor' )
   reachability = Dict( keyType=IpGenericPrefix, valueType=ReachabilityModel,
                        optional=True, help='Reachability information per prefix' )

   def renderData( self, addrFamily ):
      linkModels = [ self.p2ps, self.lans ]
      linkTypes = [ 'P2P', 'LAN' ]
      for ( linkType, linkModel ) in zip( linkTypes, linkModels ):
         if linkModel:
            print 'Network type:', linkType
            for neighborId, neighbor in linkModel.items():
               print '  Neighbor:', neighborId
               neighbor.renderData( addrFamily )
      if self.reachability:
         print 'Reachability:'
         for prefix in ArPyUtils.naturalsorted( self.reachability ):
            # pylint: disable=unsubscriptable-object
            print '  Prefix:', prefix, 'Metric:', self.reachability[ prefix ].metric
            # pylint: enable=unsubscriptable-object

class CommonProtocolModel( Model ):
   ipv4 = Dict( keyType=str, valueType=RouterModel, optional=True,
                help='IPv4 topology information per router ID' )
   ipv6 = Dict( keyType=str, valueType=RouterModel, optional=True,
                help='IPv6 topology information per router ID' )

   def renderData( self, protocol, identifier ):
      for routers, af in zip( [ self.ipv4, self.ipv6 ], [ 'IPv4', 'IPv6' ] ):
         if routers:
            print '\nSource:', protocol, identifier, af, 'database'
            for routerId in ArPyUtils.naturalsorted( routers ):
               print 'Router-ID:', routerId
               routers[ routerId ].renderData( af )

class IsisModel( Model ):
   level1 = Submodel( valueType=CommonProtocolModel, optional=True,
                      help='IS-IS Level-1 topology information' )
   level2 = Submodel( valueType=CommonProtocolModel, optional=True,
                      help='IS-IS Level-2 topology information' )

   def render( self ):
      if self.level1 is not None:
         self.level1.renderData( 'IS-IS', 'Level-1' )

      if self.level2 is not None:
         self.level2.renderData( 'IS-IS', 'Level-2' )

class VrfModel( Model ):
   isis = Submodel( valueType=IsisModel, optional=True,
                    help='IGP topology information for IS-IS' )

   def render( self ):
      if self.isis is not None:
         self.isis.render()

class TopologyModel( Model ):
   vrfs = Dict( keyType=str, valueType=VrfModel,
                help='IGP topology database per VRF' )

   def render( self ):
      for vrf in ArPyUtils.naturalsorted( self.vrfs ):
         print 'VRF:', vrf
         self.vrfs[ vrf ].render()
