# Copyright (c) 2020 Arista Networks, Inc.  All rights reserved.
# Arista Networks, Inc. Confidential and Proprietary.

import itertools
import BasicCli
import CliGlobal
import CliToken
import RoutingIsisCli
import ShowCommand
import SmashLazyMount
from CliPlugin.VrfCli import VrfExprFactory
from CspfShowTopologyModel import (
   NeighborModel,
   ReachabilityModel,
   RouterModel,
   VrfModel,
   IsisModel,
   CommonProtocolModel,
   TopologyModel )
from IpLibConsts import DEFAULT_VRF

gv = CliGlobal.CliGlobal(
   dict( entityManager=None, readerInfo=None, topoDbMounts={} ) )

def mountTopoDb( protocol, identifier, af, vrf ):
   if protocol == 'isis':
      idPath = identifier[ -1 ]

   if af == 'ipv4':
      afPath = 'ip'
   elif af == 'ipv6':
      afPath = 'ip6'

   topoDbPath = 'routing/%s/topodb/%s/%s/%s' % ( protocol, afPath, idPath, vrf )
   topoDb = gv.topoDbMounts.get( topoDbPath )
   if topoDb is None:
      topoDb = SmashLazyMount.mount( gv.entityManager, topoDbPath,
                                     'Routing::Topology::TopoDb', gv.readerInfo )
      gv.topoDbMounts[ topoDbPath ] = topoDb

   return topoDb

def filterSources( args ):
   # Only show IS-IS info in default VRF
   protocols = [ 'isis' ]
   vrfs = [ DEFAULT_VRF ]

   if 'LEVEL' in args:
      levels = [ args[ 'LEVEL' ] ]
   else:
      levels = [ 'level-1', 'level-2' ]

   if 'ADDR_FAMILY' in args:
      afs = [ args[ 'ADDR_FAMILY' ] ]
   else:
      afs = [ 'ipv4', 'ipv6' ]

   return itertools.product( protocols, levels, afs, vrfs )

def populateProtocolModel( protoIdModel, topoDb, af ):
   if af == 'ipv4':
      afModel = protoIdModel.ipv4
   elif af == 'ipv6':
      afModel = protoIdModel.ipv6

   for routerId, router in topoDb.router.items():
      rtrInfoModel = RouterModel()
      afModel[ routerId.stringValue ] = rtrInfoModel

      for ngbKey in router.neighbor.values():
         ngb = topoDb.neighbor.get( ngbKey )
         if ngb is None:
            continue
         ngbModel = NeighborModel()
         ngbModel.metric = ngb.metric

         if ngbKey.neighborType == 'neighborTypeRouter':
            neighborId = ngbKey.neighborId.routerId.stringValue
            linksModel = rtrInfoModel.p2ps
         elif ngbKey.neighborType == 'neighborTypeNetwork':
            neighborId = '%s.%x' % ( ngbKey.networkId.drRouterId,
                                    ngbKey.networkId.drNetworkId )
            linksModel = rtrInfoModel.lans
         linksModel[ neighborId ] = ngbModel

         for addr in ngb.interfaceAddr.values():
            ngbModel.interfaceAddresses.append( addr.stringValue )

         for addr in ngb.neighborAddr.values():
            ngbModel.neighborAddresses.append( addr.stringValue )

         # Exclude optional empty collections from output model
         if not ngbModel.interfaceAddresses:
            ngbModel.interfaceAddresses = None
         if not ngbModel.neighborAddresses:
            ngbModel.neighborAddresses = None

      reachCollection = topoDb.reachability.get( routerId )
      if reachCollection:
         for _, reach in reachCollection.prefix.items():
            reachModel = ReachabilityModel()
            reachModel.metric = reach.metric
            # pylint: disable=unsupported-assignment-operation
            rtrInfoModel.reachability[ reach.ipPrefix ] = reachModel
            # pylint: enable=unsupported-assignment-operation

      # Exclude optional empty collections from output model
      if not rtrInfoModel.p2ps:
         rtrInfoModel.p2ps = None
      if not rtrInfoModel.lans:
         rtrInfoModel.lans = None
      if not rtrInfoModel.reachability:
         rtrInfoModel.reachability = None

def populateLevelModel( levelModel, topoDb, af ):
   if levelModel is None:
      protoIdModel = CommonProtocolModel()
      populateProtocolModel( protoIdModel, topoDb, af )
      # Only return the new model if something was read from TopoDb
      if protoIdModel.ipv4 or protoIdModel.ipv6:
         return protoIdModel
      else:
         return None
   else:
      populateProtocolModel( levelModel, topoDb, af )
      return levelModel

def populateIsisModel( vrfModel, identifier, af, vrf ):
   topoDb = mountTopoDb( 'isis', identifier, af, vrf )
   if vrfModel.isis is None:
      vrfModel.isis = IsisModel()

   if identifier == 'level-1':
      vrfModel.isis.level1 = populateLevelModel( vrfModel.isis.level1, topoDb, af )
   elif identifier == 'level-2':
      vrfModel.isis.level2 = populateLevelModel( vrfModel.isis.level2, topoDb, af )

def populateTopologyModel( mode, args ):
   sources = filterSources( args )
   topoModel = TopologyModel()
   for protocol, identifier, af, vrf in sources:
      vrfModel = topoModel.vrfs.get( vrf )
      if vrfModel is None:
         vrfModel = VrfModel()
         topoModel.vrfs[ vrf ] = vrfModel

      if protocol == 'isis':
         populateIsisModel( vrfModel, identifier, af, vrf )

   return topoModel

# --------------------------------------------------------------------------------
# show igp topology
# --------------------------------------------------------------------------------
class TopologyCmd( ShowCommand.ShowCliCommandClass ):
   syntax = 'show router igp topology [ isis [ LEVEL ] ] [ ADDR_FAMILY ] [ VRF ]'
   data = {
      'router' : CliToken.Router.routerMatcherForConfig,
      'igp' : 'Interior gateway protocols information',
      'topology' : 'Exported topology database',
      'isis' : 'IS-IS only',
      'LEVEL' : RoutingIsisCli.levelMatcherForShow,
      'ADDR_FAMILY' : RoutingIsisCli.afMatcher,
      'VRF' : VrfExprFactory( helpdesc='VRF name',
                              inclDefaultVrf=True,
                              inclAllVrf=True ),
   }
   handler = populateTopologyModel
   cliModel = TopologyModel

BasicCli.addShowCommandClass( TopologyCmd )

def Plugin( entityManager ):
   gv.entityManager = entityManager
   gv.readerInfo = SmashLazyMount.mountInfo( 'reader' )
