# Copyright (c) 2018 Arista Networks, Inc.  All rights reserved.
# Arista Networks, Inc. Confidential and Proprietary.

from CliModel import Model
from CliModel import ( Str,
                       Bool,
                       Enum,
                       Int,
                       Float,
                       Dict,
                       Submodel,
                       List )
from ArnetModel import IpGenericAddress
from IntfModel import Interface
import Ark
import Tac
from itertools import izip_longest
import TableOutput
from CliPlugin.IsisCliModels import bw_best_value_units
import MultiRangeRule

# pylint: disable-msg=R1702

entryFormatStr = "%-15s %-10s %-50s %-15s"
entryFormatStrIpPathId = "%-15s %-11s"
entryFormatStrWithoutIp = "%-50s %-15s"

# Sample output
# rtr1#show traffic-engineering cspf path
#
# Destination     Path ID    Constraint                                    Path
# 20.0.0.1        0          exclude Ethernet1                             1.1.1.1
#                                                                          1.1.1.2
#                                                                          2.2.2.2
#                                                                          3.3.3.2
#
#                 1          bandwidth 18.75 Mbps                          1.1.1.1
#                            setup priority 4                              1.1.1.2
#                            share bandwidth with path 10.0.1.1 ID 8       2.2.2.2
#                            share bandwidth with path 10.1.1.1 ID 5       3.3.3.2
#
#                 2          exclude node 1.0.0.1                          20.0.2.2
#                            exclude link with address 1.0.1.3             20.1.2.2
#                                                                          20.2.2.2
#
# rtr1#show traffic-engineering cspf path detail
#
# Destination: 20.0.0.1
#   Path ID: 1
#    Path Constraint: exclude Ethernet1
#       Request Sequence number: 1
#       Response Sequence number: 1
#       Number of times path updated: 2
#       Reoptimize: Always
#       Last updated: 00:01:58
#       Path:
#          1.1.1.1
#          1.1.1.2
#          2.2.2.2
#          3.3.3.2
#   Path ID: 2
#    Path Constraint: exclude Ethernet
#       Request Sequence number: 2
#       Response Sequence number: 2
#       Number of times path updated: 3
#       Reoptimize: On request
#       Last updated: 00:00:38
#       Path:
#          1.1.1.1
#          1.1.1.2
#          2.2.2.2
#          3.3.3.2
#   Path ID: 3
#    Path Constraint: exclude node 1.0.0.1
#       Request Sequence number: 2
#       Response Sequence number: 2
#       Number of times path updated: 3
#       Reoptimize: On request
#       Last updated: 00:00:38 ago
#       Path:
#          1.1.1.1
#          1.1.1.2
#          2.2.2.2
#          3.3.3.2
PATH_STATUS_MAP = Ark.ReversibleDict( { 'pathFound' : 'Path found',
                                        'pathNotFound' : 'Path not found',
                                        'cspfPending' : 'CSPF pending' } )
SPACE_23 = " " * 23
MAX_WIDTH = 85

def wrapText( string, width=MAX_WIDTH, startWidth=20, subsequent_indent=SPACE_23 ):
   output = ""
   constraintInfo = string.split( ',' )
   currWidth = startWidth
   for token in constraintInfo:
      if currWidth + len( token ) < width:
         output += token + ","
      else:
         currWidth = len( subsequent_indent )
         output += "\n" + subsequent_indent + token + ","
      currWidth += len( token )
   return output[ : -1 ]

class SrlgIdToNameMapHelper( Model ):
   # _srlgIdToNameMap has srlgName for corresponding srlgId's. Name mapping not
   # needed in capi model so it is just a hidden attribute.
   _srlgIdToNameMap = Dict( keyType=int, valueType=str, valueOptional=True,
                            help="Ordered list of srlgId specifying srlgName",
                            optional=True )

   def printSrlgMap( self ):
      output = ""

      def toStr( groupId, groupName ):
         text = str( groupId )
         return " %s (%s)," % ( groupName, text ) if groupName else " " + text + ','

      for groupId in self.srlgIds:
         groupName = self._srlgIdToNameMap.get( groupId )
         output += toStr( groupId, groupName )
      return output[ : -1 ]

class IntfSrlg( SrlgIdToNameMapHelper ):
   interface = Interface( help="Interface name" )
   srlgIds = List( valueType=int, optional=True,
                   help="List of SRLG IDs of the interface" )

class CspfPathHelperModel( SrlgIdToNameMapHelper ):
   destination = IpGenericAddress( help="Destination address" )
   pathId = Int( help="Path index" )
   srlgIds = List( valueType=int, optional=True,
                   help="List of SRLG IDs for the path" )

class CspfIncludeHopModel( Model ):
   hop = IpGenericAddress( help="Interface address or TE Router ID of the hop" )
   loose = Bool( help="The include hop constraint is loose" )

class CspfPathConstraintModel( Model ):
   excludeIntf = Interface( help="Interface excluded from this path",
                            optional=True )
   excludeNode = Str( help="Node excluded from this path", optional=True )
   excludeAdminGroup = Int( help="Admin Group excluded", optional=True )
   includeAllAdminGroup = Int( help="Admin Group included", optional=True )
   includeAnyAdminGroup = Int( help="Any Admin Group included", optional=True )
   excludeSrlgOfIntf = Submodel( valueType=IntfSrlg, optional=True,
                                 help="SRLG excluded of interface" )
   excludeLinksWithAddress = List( valueType=IpGenericAddress, optional=True,
                                   help="Addresses of links excluded" )
   bandwidth = Int( help="Bandwidth reserved for this path in bits per second",
                    optional=True )
   bwSetupPriority = Int( help="Priority of bandwidth reservation request",
                          optional=True )
   sharedBwGroupId = Int( help="Shared bandwidth group ID", optional=True )
   # _sharedBwPaths is only used to show the paths sharing bandwidth
   # instead of showing the shared bandwidth group ID. It is not
   # required in the CAPI model.
   _sharedBwPaths = List( help="Keys of paths with the same sharedBwGroupId",
                          valueType=CspfPathHelperModel, optional=True )
   excludeSrlgOfPaths = List( help="List of all paths whose SRLGs to be excluded",
                              valueType=CspfPathHelperModel, optional=True )
   includeHops = List( help="Nodes or links included in this path",
                       valueType=CspfIncludeHopModel, optional=True )
   explicitPath = List( help="Complete list of hops making up this path",
                        valueType=IpGenericAddress, optional=True )

   def getRangeStr( self, bitMask ):
      """
      Convert an integer representing a bitmask to a string representing
      the range of bits set. Used to render Admin Group constraints.
      Eg:
      4 -> 0100 -> '2'
      117 -> 0111 0101 -> '0,2,4-6'
      """
      setBits = []
      for i in range( bitMask.bit_length() ):
         if bitMask & 1:
            setBits.append( i )
         bitMask >>= 1
      return MultiRangeRule.multiRangeToCanonicalString( setBits )

   def getConstraintList( self ):
      # pylint: disable-msg=no-member
      constraintList = []
      if self.excludeIntf:
         constraintList.append( "exclude %s" % self.excludeIntf.stringValue )
      # pylint: enable-msg=no-member
      elif self.excludeNode:
         constraintList.append( "exclude node %s" % self.excludeNode )
      if self.excludeSrlgOfIntf:
         if self.excludeSrlgOfIntf.srlgIds:
            # printSrlgMap add an extra space at the begining of output.
            output = wrapText( "exclude SRLG of %s:%s" % (
                               self.excludeSrlgOfIntf.interface.stringValue,
                               self.excludeSrlgOfIntf.printSrlgMap() ) )
            constraintList.append( output )
         else:
            constraintList.append( "exclude SRLG of %s" %
                                   self.excludeSrlgOfIntf.interface.stringValue )
      if self.excludeSrlgOfPaths:
         for p in self.excludeSrlgOfPaths:
            if p.srlgIds:
               # printSrlgMap add an extra space at the begining of output.
               output = wrapText( "exclude SRLG of path %s ID %d:%s" % (
                                  p.destination, p.pathId, p.printSrlgMap() ) )
               constraintList.append( output )
            else:
               constraintList.append( "exclude SRLG of path %s ID %d" %
                                      ( p.destination, p.pathId ) )
      if self.includeAllAdminGroup:
         constraintList.append( "include all Admin Group %s" %
                                self.getRangeStr( self.includeAllAdminGroup ) )
      if self.includeAnyAdminGroup:
         constraintList.append( "include any Admin Group %s" %
                                self.getRangeStr( self.includeAnyAdminGroup ) )
      if self.excludeAdminGroup:
         constraintList.append( "exclude Admin Group %s" %
                                self.getRangeStr( self.excludeAdminGroup ) )
      if self.excludeLinksWithAddress:
         for excludeAddr in sorted( self.excludeLinksWithAddress,
                                    key=lambda ip: ip.sortKey ):
            constraintList.append( "exclude link with address %s" % excludeAddr )
      if self.bandwidth:
         bwValueUnits = bw_best_value_units( self.bandwidth )
         constraintList.append( "bandwidth %0.2f %s" % ( bwValueUnits[ 0 ],
                                                         bwValueUnits[ 1 ] ) )
         constraintList.append( "setup priority %d" % self.bwSetupPriority )
      for p in sorted( self._sharedBwPaths,
                       key=lambda p: ( p.destination.sortKey, p.pathId ) ):
         constraintList.append( "share bandwidth with path %s ID %d" %
                                ( p.destination, p.pathId ) )
      if self.includeHops:
         for includeHop in self.includeHops:
            constraintList.append( "include hop %s (%s)" %
                                  ( includeHop.hop,
                                    'loose' if includeHop.loose else 'strict' ) )
      if self.explicitPath:
         constraintList.append( "explicit path" )
         for explicitHop in self.explicitPath:
            constraintList.append( "   %s" % explicitHop )
      return constraintList

class CspfPathHopModel( Model ):
   ipAddr = IpGenericAddress( help="Interface IP address" )
   teRouterId = IpGenericAddress( help="TE Router ID" )
   includeIps = List( valueType=IpGenericAddress, optional=True,
                       help="Non-ingress IP addresses in includeHop constraint "
                       "corresponding to teRouterId" )

class CspfPathEntryModel( Model ):
   constraint = Submodel( valueType=CspfPathConstraintModel,
                          help="Path Constraint" )
   hops = List( valueType=CspfPathHopModel,
                help="Hops specifying the path to the destination" )
   status = Enum( values=PATH_STATUS_MAP,
                  help="Status of path computation" )

   class Details( Model ):
      refreshReqSeq = Int( help="Request Sequence number" )
      refreshRespSeq = Int( help="Response Sequence number", optional=True )
      changeCount = Int( help="Number of times path updated", optional=True )
      lastUpdatedTime = Float( help="UTC timestamp of the last path update",
                               optional=True )
      autoUpdate = Bool( help="Whether or not reoptimize the path automatically" )

   details = Submodel( valueType=Details, help="Detailed path information",
                       optional=True )

   def getPathEntryStr( self, _detailsPresent ):
      pathStr = ""
      if _detailsPresent:
         if len( self.constraint.getConstraintList() ):
            first = True
            for constaint_ in self.constraint.getConstraintList():
               if first:
                  pathStr += "   Path Constraint: %s\n" % constaint_
               else:
                  pathStr += "                    %s\n" % constaint_
               first = False
         else:
            pathStr += "   Path Constraint: None\n"

         pathStr += "      Request Sequence number: %d\n" % \
                                 self.details.refreshReqSeq
         if self.details.refreshRespSeq:
            pathStr += "      Response Sequence number: %d\n" % \
                                    self.details.refreshRespSeq
         if self.details.changeCount:
            pathStr += "      Number of times path updated: %d\n" % \
                                    self.details.changeCount
         if self.details.lastUpdatedTime:
            pathStr += "      Last updated: %s\n" % \
                                   Ark.timestampToStr( self.details.lastUpdatedTime )
         pathStr += "      Reoptimize: %s\n" % ( "Always"
                                 if self.details.autoUpdate else "On request" )
         if len( self.hops ) == 0:
            pathStr += "      Path: %s\n" % ( PATH_STATUS_MAP[ self.status ] )
         else:
            includeIpsPresent = False
            headings = ( ( 'Ingress IP', 'l' ), ( 'TE router ID', 'l' ),
                         ( 'Include IP', 'l' ) )
            pathTable = TableOutput.createTable( headings )
            for hop in self.hops:
               includeIpStr = ", ".join( str( i ) for i in hop.includeIps or [] )
               if includeIpStr:
                  includeIpsPresent = True
               pathTable.newRow( str( hop.ipAddr ), str( hop.teRouterId ),
                                 includeIpStr )
            pathEntries = pathTable.output().split( '\n' )
            if not includeIpsPresent:
               # Strip off the `Include IP` heading and its horizontal border
               pathEntries[ 0 ] = pathEntries[ 0 ].rsplit( 'Include IP' )[ 0 ]
               pathEntries[ 1 ] = pathEntries[ 1 ].rsplit( ' -', 1 )[ 0 ]
            pathStr += "      Path: %s\n" % pathEntries[ 0 ].rstrip()
            for pathEntry in pathEntries[ 1 : ]:
               pathStr += "            %s\n" % pathEntry
      else:
         constraintList = self.constraint.getConstraintList()
         if len( constraintList ) == 0:
            constraintList.append( 'None' )
         if len( self.hops ):
            first = True
            for constraint_, hop in izip_longest( constraintList, self.hops,
                                                  fillvalue='' ):
               if first:
                  pathStr += entryFormatStrWithoutIp % \
                                 ( constraint_, hop.ipAddr if hop else '' )
               else:
                  pathStr += entryFormatStr % \
                                 ( '', '', constraint_, hop.ipAddr if hop else '' )
               pathStr += "\n"
               first = False
         else:
            first = True
            for constraint_ in constraintList:
               if first:
                  pathStr += entryFormatStrWithoutIp % ( constraint_,
                                                PATH_STATUS_MAP[ self.status ] )
               else:
                  pathStr += entryFormatStr % ( '', '', constraint_, '' )
               pathStr += "\n"
               first = False
      return pathStr

class CspfPathDestIpModel( Model ):
   _dstAddr = IpGenericAddress( help="Address for destination" )
   paths = Dict( keyType=int, valueType=CspfPathEntryModel,
                 help="A mapping of an id representing a constraint to"
                      " the corresponding Cspf path" )
   _detailsPresent = Bool( default=False,
                           help="Private attribute to indicate that the"
                                " details submodel is present" )

   def renderEntry( self, _detailsPresent ):
      dstStr = ""
      if not _detailsPresent:
         dstAddr = self._dstAddr
         first = True
         for key in sorted( self.paths ):
            dstStr += entryFormatStrIpPathId % ( dstAddr, str( key ) )
            dstStr += self.paths[ key ].getPathEntryStr( _detailsPresent )
            dstStr += "\n"
            if first:
               first = False
               dstAddr = ''
      else:
         dstStr += "Destination: %s\n" % self._dstAddr
         for key in sorted( self.paths ):
            dstStr += "  Path ID: %d\n" % key
            dstStr += self.paths[ key ].getPathEntryStr( _detailsPresent )
      print dstStr,

class CspfPathAfModel( Model ):
   pathIps = Dict( keyType=IpGenericAddress, valueType=CspfPathDestIpModel,
                   help="A mapping of a destination IP to"
                        " all Cspf paths for that destination IP" )
   _detailsPresent = Bool( default=False,
                           help="Private attribute to indicate that the"
                                " details submodel is present" )

   def render( self ):
      if not self._detailsPresent:
         print entryFormatStr % ( 'Destination', 'Path ID', 'Constraint', 'Path' )
      for key in sorted( self.pathIps ):
         self.pathIps[ key ].renderEntry( self._detailsPresent )

class CspfPathVrfModel( Model ):
   v4Info = Submodel( valueType=CspfPathAfModel, optional=True,
                      help="Cspf path information for IPv4 address family" )
   v6Info = Submodel( valueType=CspfPathAfModel, optional=True,
                      help="Cspf path information for IPv6 address family" )

   def render( self ):
      for af in [ self.v4Info, self.v6Info ]:
         if af:
            af.render()

class CspfPathModel( Model ):
   __revision__ = 2
   vrfs = Dict( keyType=str, valueType=CspfPathVrfModel,
                help="A mapping between Vrf and information of all Cspf paths"
                     " in that vrf" )

   def render( self ):
      for key in sorted( self.vrfs ):
         self.vrfs[ key ].render()

   def degrade( self, dictRepr, revision ):
      if revision == 1:
         # Changes from revision 1 -> revsion 2
         # - Renames includeAdminGroup attribute to includeAllAdminGroup
         # - Adds a new attribute includeAnyAdminGroup
         # - Replaces ipAddrs dict attribute with hops list
         for cspfPathVrfs in dictRepr[ 'vrfs' ].values():
            for afInfo in [ 'v4Info', 'v6Info' ]:
               if afInfo in cspfPathVrfs:
                  for pathIp in cspfPathVrfs[ afInfo ][ 'pathIps' ].values():
                     for path in pathIp[ 'paths' ].values():
                        path[ 'ipAddrs' ] = {}
                        for idx, hop in enumerate( path[ 'hops' ] ):
                           path[ 'ipAddrs' ][ idx ] = hop[ 'ipAddr' ]
                        del path[ 'hops' ]
                        constraint = path[ 'constraint' ]
                        agValue = constraint.pop( 'includeAllAdminGroup', None )
                        constraint[ 'includeAdminGroup' ] = agValue
                        constraint.pop( 'includeAnyAdminGroup', None )
      return dictRepr

tilfaEntryFormatStr = "%s%-15s %-30s %-15s"
tilfaEntryFormatStrIp = "%s%-15s "
tilfaEntryFormatStrWithoutIp = "%-30s %-15s"
space = " "

class SystemIdHostnameModel( Model ):
   sysId = Str( help='System identifier' )
   hostname = Str( help='Hostname', optional=True )

class TilfaPathConstraintModel( Model ):
   excludeIntf = Interface( help="Interface excluded from this path",
                            optional=True )
   excludeNode = Str( help="Node excluded from this path", optional=True )
   excludeSrlgOfIntf = Submodel( valueType=IntfSrlg, optional=True,
                                  help="SRLG exclude of interface" )
   excludeSrlgMode = Enum( values=( 'loose', 'strict' ), optional=True,
                           help="Type of SRLG exclusion" )

   def getConstraintList( self ):
      constraintList = []
      if self.excludeIntf:
         # pylint: disable=no-member
         constraintList.append( "exclude %s" % self.excludeIntf.stringValue )
      elif self.excludeNode:
         constraintList.append( "exclude node %s" % self.excludeNode )
      if self.excludeSrlgOfIntf:
         if self.excludeSrlgOfIntf.srlgIds:
            # printSrlgMap add an extra space at the begining of output.
            output = wrapText( "exclude SRLG of %s:%s" % (
                               self.excludeSrlgOfIntf.interface.stringValue,
                               self.excludeSrlgOfIntf.printSrlgMap() ) )
            constraintList.append( output )
            constraintList.append( "SRLG %s" % self.excludeSrlgMode )
         else:
            constraintList.append( "exclude SRLG of %s" %
                                   self.excludeSrlgOfIntf.interface.stringValue )
            constraintList.append( "SRLG %s" % self.excludeSrlgMode )

      return constraintList

class TilfaPathDetails( Model ):
   refreshReqSeq = Int( help="Request Sequence number" )
   refreshRespSeq = Int( help="Response Sequence number", optional=True )
   changeCount = Int( help="Number of times path updated", optional=True )
   lastUpdatedTime = Float( help="UTC timestamp of the last path update",
                               optional=True )

class TilfaPathEntryModel( Model ):
   _pathId = Int( help="Path ID" )
   constraint = Submodel( valueType=TilfaPathConstraintModel,
                          help="Path Constraint" )
   sysIds = List( valueType=SystemIdHostnameModel,
                  help="Ordered list of system identifiers or hostname specifying "
                  "the path to the destination" )
   status = Enum( values=PATH_STATUS_MAP.keys(),
                  help="Status of path computation" )
   details = Submodel( valueType=TilfaPathDetails, help="Detailed path information",
                       optional=True )

   def renderDetails( self, indent ):
      detailStr = ""
      detailStr += "%sRequest sequence number: %d\n" % ( indent,
                     self.details.refreshReqSeq )
      if self.details.refreshRespSeq is not None:
         detailStr += "%sResponse sequence number: %d\n" % ( indent,
                        self.details.refreshRespSeq )
      if self.details.changeCount is not None:
         detailStr += "%sNumber of times path updated: %d\n" % ( indent,
                        self.details.changeCount )
      if self.details.lastUpdatedTime:
         detailStr += "%sLast updated: %s\n" % ( indent,
                        Ark.timestampToStr( self.details.lastUpdatedTime ) )
      detailStr += "%sID: 0x%x\n" % ( indent, self._pathId )
      return detailStr

   def renderPathDetails( self, indent ):
      pathStr = ""
      lenSysIds = len( self.sysIds )
      if lenSysIds == 1:
         suffix = ' [PQ-node]'
         sysidOrHostname = self.sysIds[ 0 ].hostname or \
                           self.sysIds[ 0 ].sysId
         pathStr += "%s%s%s\n" % ( indent, sysidOrHostname, suffix )
      else:
         suffixP = ' [P-node]'
         suffixQ = ' [Q-node]'
         for key in xrange( lenSysIds ):
            if key == 0:
               suffix = suffixP
            elif key == ( lenSysIds - 1 ):
               suffix = suffixQ
            else:
               suffix = ''
            sysidOrHostname = self.sysIds[ key ].hostname or \
                              self.sysIds[ key ].sysId
            pathStr += "%s%s%s\n" % ( indent, sysidOrHostname, suffix )
      return pathStr

   def summaryEntryGenerator( self ):
      constraints = self.constraint.getConstraintList()
      path = []
      if not self.sysIds:
         path = [ PATH_STATUS_MAP[ self.status ] ]
      else:
         path = [ s.hostname or s.sysId for s in self.sysIds ]
      for constraint, pathSysId in izip_longest( constraints, path, fillvalue="" ):
         yield constraint, pathSysId

   def getPathEntryStrDetail( self ):
      pathEntryStr = ""
      lenSysIds = len( self.sysIds )
      constraintList = self.constraint.getConstraintList()
      first = True
      for constraint in constraintList:
         if first:
            pathEntryStr += ( "%sPath constraint: %s\n" %
                              ( space * 6, constraint ) )
            first = False
         else:
            pathEntryStr += ( "%s%s\n" % ( space * 23, constraint ) )
      pathEntryStr += self.renderDetails( space * 9 )
      if lenSysIds == 0:
         pathEntryStr += "%sPath: %s\n" % ( space * 9,
                                            PATH_STATUS_MAP[ self.status ] )
      else:
         pathEntryStr += "%sPath:\n" % ( space * 9 )
         pathEntryStr += self.renderPathDetails( space * 12 )
      return pathEntryStr

class TilfaPathDestModel( Model ):
   _dstId = Str( help="Destination System identifier" )
   hostname = Str( help="Hostname", optional=True )
   pathIds = Dict( keyType=long, valueType=TilfaPathEntryModel,
                 help="A mapping of an ID representing a constraint to the"
                 " corresponding TI-LFA path" )

   def summaryEntryGenerator( self ):
      dest = ( self.hostname or self._dstId )
      firstPrintDone = False
      for pathId in sorted( self.pathIds ):
         pathEntry = self.pathIds[ pathId ]
         for constraint, path in pathEntry.summaryEntryGenerator():
            yield "" if firstPrintDone else dest, constraint, path
            firstPrintDone = True
         # Print empty line between paths
         yield "", "", ""

   def renderDetailEntry( self ):
      dstStr = ""
      dstOrHostname = ( self.hostname or self._dstId )
      dstStr = "%sDestination: %s\n" % ( space * 3, dstOrHostname )
      for key in sorted( self.pathIds ):
         dstStr += self.pathIds[ key ].getPathEntryStrDetail()
      print dstStr

class TilfaPathTopoIdModel( Model ):
   destinations = Dict( keyType=str, valueType=TilfaPathDestModel,
                        help="A mapping of path systemIds to all TI-LFA "
                        "paths for that destination" )
   _detailsPresent = Bool( default=False,
                           help="Private attribute to indicate that the detail"
                           " submodel is present" )

   def renderSummaryOutput( self ):
      headings = ( ( 'Destination', 'l' ), ( 'Constraint', 'l' ), ( 'Path', 'l' ) )
      table = TableOutput.createTable( headings )
      for dstEntry in self.destinations.itervalues():
         for dest, constraint, path in dstEntry.summaryEntryGenerator():
            table.newRow( dest, constraint, path )
      print table.output()

   def render( self ):
      if not self._detailsPresent:
         self.renderSummaryOutput()
         return
      for key in sorted( self.destinations ):
         self.destinations[ key ].renderDetailEntry()

class TilfaPathAfModel( Model ):
   topologies = Dict( keyType=int, valueType=TilfaPathTopoIdModel,
                      help="A mapping of topology ID to all TI-LFA paths" )

   def render( self ):
      for topoId, topoModel in self.topologies.iteritems():
         if topoModel.destinations:
            print "%sTopo-id: Level-%d" % ( space * 3, topoId )
            topoModel.render()

class TilfaPathVrfModel( Model ):
   v4Info = Submodel( valueType=TilfaPathAfModel, optional=True,
                       help="TI-LFA path information for IPv4 address family" )
   v6Info = Submodel( valueType=TilfaPathAfModel, optional=True,
                      help="TI-LFA path information for IPv6 address family" )

   def render( self ):
      if self.v4Info and self.v4Info.topologies:
         print "TI-LFA paths for IPv4 address family"
         self.v4Info.render()
      if self.v6Info and self.v6Info.topologies:
         print "TI-LFA paths for IPv6 address family"
         self.v6Info.render()

class TilfaPathModel( Model ):
   vrfs = Dict( keyType=str, valueType=TilfaPathVrfModel,
                help="A mapping between a vrf and its TI-LFA paths" )

   def render( self ):
      for key in sorted( self.vrfs ):
         self.vrfs[ key ].render()
