# Copyright (c) 2018 Arista Networks, Inc.  All rights reserved.
# Arista Networks, Inc. Confidential and Proprietary.

#-------------------------------------------------------------------------------
# This module implements Cspf show commands
#-------------------------------------------------------------------------------

from collections import defaultdict
import Tac
import Tracing
import SharedMem
import Smash
import BasicCli
import CliMatcher
import CliCommand
import ShowCommand
from CliPlugin.RoutingIsisCli import getHostName
from CliPlugin.IsisCliModels import getRouterId
from CliPlugin.IpAddrMatcher import IpAddrMatcher
from CliPlugin.TeCli import getSrlgIdToNameMap, teConfiguration
from .CspfShowPathModel import CspfPathModel
from .CspfShowPathModel import CspfPathVrfModel
from .CspfShowPathModel import CspfPathAfModel
from .CspfShowPathModel import CspfPathDestIpModel
from .CspfShowPathModel import CspfPathEntryModel
from .CspfShowPathModel import CspfPathConstraintModel
from .CspfShowPathModel import IntfSrlg
from .CspfShowPathModel import TilfaPathEntryModel, TilfaPathDestModel
from .CspfShowPathModel import TilfaPathAfModel, TilfaPathVrfModel
from .CspfShowPathModel import TilfaPathModel, SystemIdHostnameModel
from .CspfShowPathModel import TilfaPathTopoIdModel, TilfaPathDetails
from .CspfShowPathModel import TilfaPathConstraintModel
from .CspfShowPathModel import CspfPathHelperModel
from .CspfShowPathModel import CspfPathHopModel
from .CspfShowPathModel import CspfIncludeHopModel

__defaultTraceHandle__ = Tracing.Handle( 'CspfCli' )
t0 = Tracing.trace0

matcherDetail = CliMatcher.KeywordMatcher( 'detail',
                helpdesc='Show detailed path information' )
matcherTrafficEngineering = CliMatcher.KeywordMatcher( 'traffic-engineering',
                            helpdesc='Traffic Engineering related information' )
entityManager = None
clientDir = None
AddressFamily = Tac.Type( "Arnet::AddressFamily" )
TopoId = Tac.Type( "Cspf::TopoId" )

def getIntfSrlgList( intfName ):
   srlgIdList = []
   config = teConfiguration()
   if intfName not in config.intfConfig:
      return srlgIdList
   for groupId in config.intfConfig[ intfName ].srlgIdList:
      srlgIdList.append( groupId )
   for groupName in config.intfConfig[ intfName ].srlgNameList:
      groupIdEntry = config.srlgMap.get( groupName )
      if groupIdEntry:
         srlgIdList.append( groupIdEntry.srlgId )
   return srlgIdList

def getClientConfigStatus( clientName, vrf, af ):
   shmemEm = SharedMem.entityManager( sysdbEm=entityManager )
   smi = Smash.mountInfo( 'reader' )
   configPath = "te/cspf/config/vrf/" + vrf + "/" + af + "/" + clientName
   statusPath = "te/cspf/status/vrf/" + vrf + "/" + af + "/" + clientName
   cspfConfig = shmemEm.doMount( configPath, 'Cspf::Config', smi )
   cspfStatus = shmemEm.doMount( statusPath, 'Cspf::Status', smi )
   return ( cspfConfig, cspfStatus )

def populateExcludeSrlgOfPaths( pathEntries, srlgGroupOrder, constraintModel,
                                srlgIdToNameMap, pathStatus, detail ):
   """
   pathEntries - A list of tuples ( pathKey, srlgGroupOrder )
   srlgGroupOder - SrlgGroupOrder for the pathConfig for which excludeSrlgOfPaths
                   will be populated in constraintModel
   constraintModel - It hold constraint entries in CAPI model
   srlgIdToNameMap - A map from SRLG ids to SRLG group name
   detail - If set we will also populate SRLG in excludeSrlgOfPaths
   """
   for ( pathKey, srlgGpOrder ) in pathEntries:
      if srlgGroupOrder <= srlgGpOrder:
         continue
      pModel = CspfPathHelperModel( destination=pathKey.dstIp,
                                    pathId=pathKey.id,
                                    srlgIds=None )
      if detail:
         # SRLG value will be shown only in detail output
         pModel._srlgIdToNameMap = srlgIdToNameMap
         pStatus = pathStatus.get( pathKey )
         if pStatus and pStatus.pathSrlg:
            srlgs = set()
            for value in pStatus.pathSrlg.values():
               srlgs.update( value.srlg.values() )
            if srlgs:
               pModel.srlgIds = sorted( srlgs )
      constraintModel.excludeSrlgOfPaths.append( pModel )
   # excludeSrlgOfPaths is an optional attribute, if there is no entries
   # in it, we will not show it in CAPI output
   if not constraintModel.excludeSrlgOfPaths:
      constraintModel.excludeSrlgOfPaths = None

def getSharedBwGroupAndPathBySrlgGroupIdMap():
   sharedBwGroupMap = defaultdict( lambda: defaultdict( list ) )
   pathBySrlgGroupMap = defaultdict( lambda: defaultdict( dict ) )
   for ( clientName, client ) in clientDir.entityPtr.iteritems():
      if clientName not in [ "rsvpFrr", "rsvpLer" ]:
         continue
      pathBySrlgGroupForClient = pathBySrlgGroupMap[ clientName ]
      for vrf in client.vrf:
         vrfBwGroupMap = sharedBwGroupMap[ vrf ]
         pathBySrlgGroupForVrf = pathBySrlgGroupForClient[ vrf ]
         for af in client.addressFamily:
            ( config, _ ) = getClientConfigStatus( clientName, vrf, af )
            pathBySrlgGroup = defaultdict( list )
            for p, pathConfig in config.pathConfig.iteritems():
               if pathConfig.srlgGroupId != 0:
                  pathBySrlgGroup[ pathConfig.srlgGroupId ].extend(
                     [ ( p, pathConfig.srlgGroupOrder ) ] )
               sharedBwGroupId = pathConfig.constraint.sharedBwGroupId
               if not sharedBwGroupId:
                  continue
               pathModel = CspfPathHelperModel( destination=p.dstIp, pathId=p.id,
                                                srlgIds=None )
               vrfBwGroupMap[ sharedBwGroupId ].extend( [ pathModel ] )
            pathBySrlgGroupForVrf[ af ] = pathBySrlgGroup
   return ( sharedBwGroupMap, pathBySrlgGroupMap )

def showClientCspfPathInfo( mode, args ):
   ipAddr = args.get( "IP_ADDR" )
   detail = "detail" in args
   pathModel = CspfPathModel()
   srlgIdToNameMap = getSrlgIdToNameMap()
   sharedBwGroupMap, pathBySrlgGroupMap = getSharedBwGroupAndPathBySrlgGroupIdMap()

   # Create vrf model for each vrf for the client
   for ( clientName, client ) in clientDir.entityPtr.iteritems():
      # We are only interested in rsvpFrr and rsvpLer client
      if clientName not in [ "rsvpFrr", "rsvpLer" ]:
         continue
      getClientCspfPathInfo( client, clientName, ipAddr, detail, pathModel,
                             srlgIdToNameMap, sharedBwGroupMap,
                             pathBySrlgGroupMap[ clientName ] )
   return pathModel


def getClientCspfPathInfo( client, clientName, ipAddr, detail, pathModel,
                           srlgIdToNameMap, sharedBwGroupMap, pathBySrlgGroupMap ):
   for vrf in client.vrf:
      if vrf not in pathModel.vrfs:
         pathModel.vrfs[ vrf ] = CspfPathVrfModel()
      vrfModel = pathModel.vrfs[ vrf ]
      vrfSharedBwMap = sharedBwGroupMap.get( vrf, {} )
      pathBySrlgGroupForVrf = pathBySrlgGroupMap.get( vrf, {} )
      # Create af model for each af for the client
      for af in client.addressFamily:
         if af == AddressFamily.ipv4:
            if not vrfModel.v4Info:
               vrfModel.v4Info = CspfPathAfModel()
            afModel = vrfModel.v4Info
         elif af == AddressFamily.ipv6:
            if not vrfModel.v6Info:
               vrfModel.v6Info = CspfPathAfModel()
            afModel = vrfModel.v6Info
         else:
            assert False, "Client should not have ipunknown as addressFamily"
         pathBySrlgGroup = pathBySrlgGroupForVrf.get( af, {} )
         if detail:
            afModel._detailsPresent = True
         ( config, status ) = getClientConfigStatus( clientName, vrf, af )
         # Create ip model and entry model for each entry
         for p, pathConfig in config.pathConfig.iteritems():
            if ipAddr and str( ipAddr ) != str( p.dstIp ):
               continue
            if p.dstIp not in afModel.pathIps:
               ipModel = CspfPathDestIpModel()
               ipModel._dstAddr = p.dstIp
               afModel.pathIps[ p.dstIp ] = ipModel
            else:
               ipModel = afModel.pathIps[ p.dstIp ]
            if p.id not in ipModel.paths:
               entryModel = CspfPathEntryModel()
               constraintModel = CspfPathConstraintModel()
               excludeAdminGroup = pathConfig.constraint.excludeAdminGroup
               includeAllAdminGroup = pathConfig.constraint.includeAllAdminGroup
               includeAnyAdminGroup = pathConfig.constraint.includeAnyAdminGroup
               excludeSrlgOfIntf = IntfSrlg()
               excludeSrlgOfIntf.srlgIds = None
               bandwidth = pathConfig.constraint.bandwidth
               bwSetupPriority = pathConfig.constraint.bwSetupPriority
               sharedBwGroupId = pathConfig.constraint.sharedBwGroupId
               if pathConfig.constraint.excludeIntf:
                  constraintModel.excludeIntf = pathConfig.constraint.excludeIntf
                  if pathConfig.constraint.excludeSrlg:
                     excludeSrlgOfIntf.interface = \
                                 pathConfig.constraint.excludeIntf
               elif pathConfig.constraint.excludeNode:
                  constraintModel.excludeNode = \
                        pathConfig.constraint.excludeNode.node.stringValue
                  if pathConfig.constraint.excludeSrlg:
                     excludeSrlgOfIntf.interface = \
                              pathConfig.constraint.excludeNode.nexthopIntf
               if excludeAdminGroup:
                  constraintModel.excludeAdminGroup = excludeAdminGroup
               if includeAllAdminGroup:
                  constraintModel.includeAllAdminGroup = includeAllAdminGroup
               if includeAnyAdminGroup:
                  constraintModel.includeAnyAdminGroup = includeAnyAdminGroup
               for excludeAddr in pathConfig.constraint.excludeLinkWithIp.values():
                  constraintModel.excludeLinksWithAddress.append( excludeAddr )
               if not constraintModel.excludeLinksWithAddress:
                  constraintModel.excludeLinksWithAddress = None
               if bandwidth:
                  # Convert bandwidth to bits per second for show command
                  constraintModel.bandwidth = bandwidth * 8
                  # bwSetupPriority and sharedBwGroupId take effect only when
                  # bandwidth constraint is present
                  constraintModel.bwSetupPriority = bwSetupPriority
                  if sharedBwGroupId:
                     constraintModel.sharedBwGroupId = sharedBwGroupId
                     constraintModel._sharedBwPaths = (
                           vrfSharedBwMap[ sharedBwGroupId ] )
                     # srlgIds is an optional attribute.
                     pathKeyModel = CspfPathHelperModel( destination=p.dstIp,
                                                         pathId=p.id, srlgIds=None )
                     # Exclude the current path itself from shared paths
                     constraintModel._sharedBwPaths.remove( pathKeyModel )
               if pathConfig.constraint.includeHopAllExplicit:
                  for explicitHop in pathConfig.constraint.includeHop.values():
                     constraintModel.explicitPath.append( explicitHop.hop )
               else:
                  for includeHop in pathConfig.constraint.includeHop.values():
                     includeHopModel = CspfIncludeHopModel( hop=includeHop.hop,
                                                            loose=includeHop.loose )
                     constraintModel.includeHops.append( includeHopModel )
               if not constraintModel.includeHops:
                  constraintModel.includeHops = None
               if not constraintModel.explicitPath:
                  constraintModel.explicitPath = None
               pathStatus = status.pathStatus.get( p )
               if pathStatus:
                  if len( pathStatus.path ):
                     includeIpMap = defaultdict( list )
                     for includeIp in pathStatus.includeIp.values():
                        # Only incude entries from pathStatus.includeIp in which
                        # incudeIP address is different from both teRouterId or the
                        # ingress IP address a that hopIndex
                        iip = includeIp.includeIp
                        hopIndex = includeIp.hopIndex
                        if ( iip != pathStatus.path[ hopIndex ] and
                             iip != pathStatus.pathTeRouterId[ hopIndex ] ):
                           ( includeIpMap[ includeIp.hopIndex ]
                             .append( includeIp.includeIp ) )
                     for idx, hopIpAddr in pathStatus.path.items():
                        hopTeRouterId = pathStatus.pathTeRouterId[ idx ]
                        hopIncludeIps = ( includeIpMap[ idx ] if idx in includeIpMap
                                          else None )
                        hopModel = CspfPathHopModel( ipAddr=hopIpAddr,
                                                     teRouterId=hopTeRouterId,
                                                     includeIps=hopIncludeIps )
                        entryModel.hops.append( hopModel )
                     entryModel.status = 'pathFound'
                  else:
                     entryModel.status = 'pathNotFound'
               else:
                  entryModel.status = 'cspfPending'
               if detail:
                  if excludeSrlgOfIntf.interface:
                     srlgIds = getIntfSrlgList(
                        excludeSrlgOfIntf.interface.stringValue )
                     if len( srlgIds ):
                        excludeSrlgOfIntf.srlgIds = srlgIds
                        excludeSrlgOfIntf._srlgIdToNameMap = srlgIdToNameMap
                  details = CspfPathEntryModel.Details()
                  details.refreshReqSeq = pathConfig.refreshReqSeq
                  details.autoUpdate = pathConfig.autoUpdate
                  if pathStatus:
                     details.refreshRespSeq = pathStatus.refreshRespSeq
                     details.changeCount = pathStatus.changeCount
                     details.lastUpdatedTime = pathStatus.lastUpdatedTime
                  entryModel.details = details
               if pathConfig.constraint.excludeSrlg and excludeSrlgOfIntf.interface:
                  constraintModel.excludeSrlgOfIntf = excludeSrlgOfIntf

               # 0 is not a valid srlgGroupId value.
               if pathConfig.srlgGroupId != 0:
                  pathEntries = pathBySrlgGroup[ pathConfig.srlgGroupId ]
                  populateExcludeSrlgOfPaths( pathEntries, pathConfig.srlgGroupOrder,
                                              constraintModel, srlgIdToNameMap,
                                              status.pathStatus, detail )
               else:
                  # If srlgGroupId is 0 - srlgGroup is not set
                  # In this case, excludeSrlgOfPaths attribute will not be shown
                  # in CAPI model ( It is an optional attribute )
                  constraintModel.excludeSrlgOfPaths = None
               entryModel.constraint = constraintModel
               ipModel.paths[ p.id ] = entryModel

def showClientTilfaPathInfo( mode, args ):
   pathModel = TilfaPathModel()
   reqDstId = str( args.get( "DST_ID", "" ) )
   detail = "detail" in args
   # Only show tilfa related path information in this show cmd.
   clientName = "tilfa"
   client = clientDir.entityPtr.get( clientName )
   srlgIdToNameMap = getSrlgIdToNameMap()
   if not ( client and clientName in clientDir.entryState ):
      return pathModel
   for vrf in client.vrf:
      vrfModel = TilfaPathVrfModel()
      pathModel.vrfs[ vrf ] = vrfModel
      for af in client.addressFamily:
         afModel = TilfaPathAfModel()
         if af == AddressFamily.ipv4:
            vrfModel.v4Info = afModel
         elif af == AddressFamily.ipv6:
            vrfModel.v6Info = afModel

         topoIdModelL1 = TilfaPathTopoIdModel()
         topoIdModelL2 = TilfaPathTopoIdModel()
         destConstraintDictL1 = {}
         destConstraintDictL2 = {}

         ( config, status ) = getClientConfigStatus( clientName, vrf, af )
         for key, pathConfig in config.pathConfig.iteritems():
            dstId = key.dst.stringValue
            hostname = None
            if key.dst.type == 'vtxIsisSystemId':
               hostname = getHostName( getRouterId( dstId ) )
            if reqDstId and reqDstId not in ( dstId, hostname ):
               continue
            if pathConfig.constraint.topoId == TopoId.isisL1:
               topoIdModel = topoIdModelL1
               destConstraintDict = destConstraintDictL1
            else:
               topoIdModel = topoIdModelL2
               destConstraintDict = destConstraintDictL2
            if detail:
               topoIdModel._detailsPresent = True
            afModel.topologies[ pathConfig.constraint.topoId.protoId ] = topoIdModel
            if dstId not in topoIdModel.destinations:
               sysIdModel = TilfaPathDestModel()
               sysIdModel._dstId = dstId
               sysIdModel.hostname = hostname
               topoIdModel.destinations[ sysIdModel._dstId ] = sysIdModel
            else:
               sysIdModel = topoIdModel.destinations[ dstId ]
            destConstraintTuple = None
            if pathConfig.constraint.excludeIntf:
               destConstraintTuple = ( dstId, pathConfig.constraint.excludeIntf )
            elif pathConfig.constraint.excludeNode:
               destConstraintTuple = ( dstId, pathConfig.constraint.excludeNode )

            if key.id not in sysIdModel.pathIds and \
               destConstraintTuple not in destConstraintDict:
               entryModel = TilfaPathEntryModel()
               constraintModel = TilfaPathConstraintModel()
               excludeSrlgOfIntf = IntfSrlg()
               excludeSrlgOfIntf.srlgIds = None
               if pathConfig.constraint.excludeIntf:
                  constraintModel.excludeIntf = pathConfig.constraint.excludeIntf
                  if pathConfig.constraint.excludeSrlg:
                     excludeSrlgOfIntf.interface = pathConfig.constraint.excludeIntf
               elif pathConfig.constraint.excludeNode:
                  constraintModel.excludeNode = \
                                 pathConfig.constraint.excludeNode.node.stringValue
                  if pathConfig.constraint.excludeSrlg:
                     excludeSrlgOfIntf.interface = \
                              pathConfig.constraint.excludeNode.nexthopIntf
               if pathConfig.constraint.excludeSrlg:
                  constraintModel.excludeSrlgOfIntf = excludeSrlgOfIntf
                  constraintModel.excludeSrlgMode = \
                        'strict' if pathConfig.constraint.srlgStrictMode else 'loose'

               entryModel.constraint = constraintModel
               if detail and excludeSrlgOfIntf.interface:
                  srlgIds = getIntfSrlgList(
                     excludeSrlgOfIntf.interface.stringValue )
                  if len( srlgIds ):
                     excludeSrlgOfIntf.srlgIds = srlgIds
                     excludeSrlgOfIntf._srlgIdToNameMap = srlgIdToNameMap

               if not detail:
                  destConstraintDict[ destConstraintTuple ] = True
               pathStatus = status.pathStatus.get( key )
               lfaIdx = Tac.Type( "Cspf::Lfa::PqIdx" )
               statusStr = 'pathNotFound'
               if pathStatus:
                  if ( pathStatus.repairPathIdx != lfaIdx.invalid and
                     status.pqInfo.get( pathStatus.repairPathIdx ) ):
                     pqInfoNode = status.pqInfo.get( pathStatus.repairPathIdx )
                     # Added an extra check when pqInfo is again evaluated.
                     if pqInfoNode:
                        for sysid in pqInfoNode.path.itervalues():
                           sysidHostnameModel = SystemIdHostnameModel()
                           sysidHostnameModel.sysId = sysid
                           sysidHostnameModel.hostname = getHostName( getRouterId(
                              sysid ) )
                           entryModel.sysIds.append( sysidHostnameModel )
                        statusStr = 'pathFound'
               else:
                  statusStr = 'cspfPending'
               entryModel.status = statusStr
               if detail:
                  details = TilfaPathDetails()
                  details.refreshReqSeq = pathConfig.refreshReqSeq
                  if pathStatus:
                     details.refreshRespSeq = pathStatus.refreshRespSeq
                     details.changeCount = pathStatus.changeCount
                     details.lastUpdatedTime = pathStatus.lastUpdatedTime
                  entryModel.details = details
               entryModel._pathId = long( key.id )
               sysIdModel.pathIds[ long( key.id ) ] = entryModel

   return pathModel

#--------------------------------------------------------------------------------
# show isis ti-lfa path [ { ( DST_ID | detail ) } ]
#--------------------------------------------------------------------------------
dstIdMatcher = CliMatcher.PatternMatcher(
      pattern='^(?![dD]([eE]([tT]([aA]([iI]([lL])?)?)?)?)?$)[A-Za-z0-9,_:'
              '\\-\\./#%+]{1,255}',
      helpdesc='Destination system identifier, hostname or prefix',
      helpname='IDENTIFIER' )

class IsisTiLfaPathCmd( ShowCommand.ShowCliCommandClass ):
   syntax = 'show isis ti-lfa path [ { ( DST_ID | detail ) } ]'
   data = {
      'isis' : 'IS-IS commands',
      'ti-lfa' : 'TI-LFA related path information',
      'path' : 'TI-LFA path to destination',
      'DST_ID' : CliCommand.Node( matcher=dstIdMatcher, maxMatches=1 ),
      'detail' : CliCommand.Node( matcher=matcherDetail, maxMatches=1 ),
   }
   handler = showClientTilfaPathInfo
   cliModel = TilfaPathModel

BasicCli.addShowCommandClass( IsisTiLfaPathCmd )

#--------------------------------------------------------------------------------
# show traffic-engineering cspf path [ IP_ADDR ] [ detail ]
#--------------------------------------------------------------------------------
class TrafficEngineeringCspfPathCmd( ShowCommand.ShowCliCommandClass ):
   syntax = 'show traffic-engineering cspf path [ IP_ADDR ] [ detail ]'
   data = {
      'traffic-engineering' : matcherTrafficEngineering,
      'cspf' : 'Constrained Shortest Path related information',
      'path' : 'CSPF path to destination',
      'IP_ADDR' : IpAddrMatcher( helpdesc='CSPF path to this destination'
                                          'IP address' ),
      'detail' : matcherDetail,
   }
   handler = showClientCspfPathInfo
   cliModel = CspfPathModel

BasicCli.addShowCommandClass( TrafficEngineeringCspfPathCmd )

def Plugin( entMan ):
   global entityManager
   global clientDir
   entityManager = entMan
   mg = entityManager.mountGroup()
   clientDir = mg.mount( "te/cspf/client", "Tac::Dir", "ri" )
   mg.close( None )
