# Copyright (c) 2019 Arista Networks, Inc.  All rights reserved.
# Arista Networks, Inc. Confidential and Proprietary.

from CliModel import Model
from CliModel import ( Int,
                       Str,
                       List,
                       Dict,
                       Submodel )
from ArnetModel import Ip4Address, Ip6Address
from CliPlugin.IsisCliModels import bw_best_value_units

SPACES_10 = " " * 10
SPACES_12 = " " * 12
SPACES_14 = " " * 14

class ReservablePriorityBandwidth( Model ):
   priority0 = Int( help="Reservable Bandwidth (bps) for priority level 0" )
   priority1 = Int( help="Reservable Bandwidth (bps) for priority level 1" )
   priority2 = Int( help="Reservable Bandwidth (bps) for priority level 2" )
   priority3 = Int( help="Reservable Bandwidth (bps) for priority level 3" )
   priority4 = Int( help="Reservable Bandwidth (bps) for priority level 4" )
   priority5 = Int( help="Reservable Bandwidth (bps) for priority level 5" )
   priority6 = Int( help="Reservable Bandwidth (bps) for priority level 6" )
   priority7 = Int( help="Reservable Bandwidth (bps) for priority level 7" )

   def render( self ):
      print SPACES_12 + "Unreserved BW:"
      NUM_BW_CLASSES = 8
      priorityHeader = SPACES_14 + "TE class %d: %0.2f %s\tTE class %d: %0.2f %s"
      linesWithTwoClasses = int( NUM_BW_CLASSES / 2 )
      for i in range( linesWithTwoClasses ):
         bw0 = getattr( self, "priority" + str( 2 * i ) )
         bw1 = getattr( self, "priority" + str( 2 * i + 1 ) )
         best_value_units_bw0 = bw_best_value_units( bw0 )
         best_value_units_bw1 = bw_best_value_units( bw1 )
         print ( priorityHeader ) % ( 2 * i, best_value_units_bw0[ 0 ],
                 best_value_units_bw0[ 1 ], 2 * i + 1, best_value_units_bw1[ 0 ],
                 best_value_units_bw1[ 1 ] )

class InterfaceAddress( Model ):
   ipv4Address = Ip4Address( help="IPv4 address", optional=True )
   ipv6Address = Ip6Address( help="IPv6 address", optional=True )

   def render( self ):
      spaces14 = " " * 14
      intfHeader = spaces14 + "%s"
      for addr in [ self.ipv4Address, self.ipv6Address ]:
         if addr is not None:
            print intfHeader % ( str( addr ) )

class SharedRiskLinkGroup( Model ):
   groupId = Int( help="Shared Risk Link Group Identifier" )
   groupName = Str( help="Shared Risk Link Group Name", optional=True )

   def render( self ):
      if self.groupName:
         srlgGroupHeader = SPACES_14 + "Group: %s (%s)"
         print srlgGroupHeader % ( self.groupName, str( self.groupId ) )
      else:
         srlgGroupHeader = SPACES_14 + "Group: %s"
         print srlgGroupHeader % ( str( self.groupId ) )

class NgbTeInfoModel( Model ):
   neighborId = Str( help="Neighbor Protocol ID" )
   administrativeGroup = Int( optional=True,
                              help="Administrative Group of the Link" )
   metric = Int( optional=True, help="TE Link cost" )
   interfaceAddresses = List( valueType=InterfaceAddress,
                             help=" List of local Interface Addresses"
                                   " forming the adjacency" )
   neighborAddresses = List( valueType=InterfaceAddress, optional=True,
                            help="List of remote Neighbor Addresses"
                                  " forming the adjacency" )
   maxLinkBw = Int( optional=True,
                    help="Maximum bandwidth (bps) that can be used"
                    "on Directed Link" )
   maxReservableBw = Int( optional=True, help="Maximum bandwidth (bps) that can be"
                          "reserved on Directed Link" )
   unreservedBw = Submodel( valueType=ReservablePriorityBandwidth, optional=True,
                            help="Maximum bandwidth reservable for a priority" )
   sharedRiskLinkGroups = List( valueType=SharedRiskLinkGroup, optional=True,
                                help="List of Shared Risk Link Group" )

   def renderEntry( self, addrFamily ):
      ngbIdHeader = SPACES_10 + "Neighbor: %s"
      adminColorHeader = SPACES_12 + "Administrative group (Color): 0x%x"
      teMetricHeader = SPACES_12 + "TE metric: %d"
      intfAddrsHeader = SPACES_12 + "%s Interface Addresses:"
      ngbAddrsHeader = SPACES_12 + "%s Neighbor Addresses:"
      maxLinkBwHeader = SPACES_12 + "Maximum link BW: %0.2f %s"
      maxReservableBwHeader = SPACES_12 + "Maximum reservable link BW: %0.2f %s"
      srlgHeader = SPACES_12 + "Shared Risk Link Groups:"

      print ngbIdHeader % ( self.neighborId )
      if self.administrativeGroup is not None:
         print adminColorHeader % ( self.administrativeGroup )
      if self.metric is not None:
         print teMetricHeader % ( self.metric )
      print intfAddrsHeader % ( addrFamily )
      for addr in self.interfaceAddresses:
         addr.render()
      if self.neighborAddresses:
         print ngbAddrsHeader % ( addrFamily )
         for addr in self.neighborAddresses:
            addr.render()
      if self.maxLinkBw is not None:
         bestValueUnit = bw_best_value_units( self.maxLinkBw )
         print maxLinkBwHeader % ( bestValueUnit[ 0 ], bestValueUnit[ 1 ] )
      if self.maxReservableBw is not None:
         bestValueUnit = bw_best_value_units( self.maxReservableBw )
         print maxReservableBwHeader % ( bestValueUnit[ 0 ], bestValueUnit[ 1 ] )
      if self.unreservedBw:
         self.unreservedBw.render()
      if self.sharedRiskLinkGroups:
         print srlgHeader
         for sharedRiskLinkGroup in self.sharedRiskLinkGroups:
            sharedRiskLinkGroup.render()

class TeLinksModel( Model ):
   links = List( valueType=NgbTeInfoModel,
                        help="TE information of each neighboring link" )

   def renderData( self, addrFamily ):
      for ngbEntry in self.links:
         ngbEntry.renderEntry( addrFamily )

class RouterInfoModel( Model ):
   numLinks = Int( help="Number of links" )
   p2p = Submodel( valueType=TeLinksModel, optional=True,
                        help="TE DB information for P2P links" )
   lan = Submodel( valueType=TeLinksModel, optional=True,
                        help="TE DB information for LAN links" )

   def renderData( self, addrFamily ):
      linkModels = [ self.p2p, self.lan ]
      linkTypes = [ "P2P", "LAN" ]
      numLinksHeader = "      Number of Links: %s"
      linkTypeHeader = "        Network type: %s"
      print numLinksHeader % ( self.numLinks )
      for i in range( len( linkModels ) ):
         if linkModels[ i ] is not None:
            print linkTypeHeader % ( linkTypes[ i ] )
            linkModels[ i ].renderData( addrFamily )

class TeSourceProtocolDbModel( Model ):
   routers = Dict( keyType=str, valueType=RouterInfoModel,
                   help="Protcol specific router information" )

   def renderData( self, addrFamily, igpName ):
      isisSystemIdHeader = "    IS-IS System-ID: %s"
      ospfRouterIdHeader = "    OSPFv2 Router-ID: %s"
      for protocolId in self.routers:
         if igpName == 'isis':
            print isisSystemIdHeader % ( protocolId )
         else:
            assert igpName == 'ospf'
            print ospfRouterIdHeader % ( protocolId )
         self.routers[ protocolId ].renderData( addrFamily )

class IgpModel( Model ):
   levels = Dict( keyType=int, valueType=TeSourceProtocolDbModel,
                  help="A mapping of the IS-IS level to the corresponding IS-IS "
                       "TE database" )
   areas = Dict( keyType=Ip4Address, valueType=TeSourceProtocolDbModel,
                 help="A mapping of the OSPF area ID to the area's TE "
                      "database information" )

   def renderData( self, addrFamily, instanceId ):
      srcDbList = [ "IS-IS Level-1", "IS-IS Level-2",
                    "OSPFv2" ]
      srcDbHeader = "  Source: %s %s Topology Database"
      ospfHeader = 'Instance ID {} Area-ID {}'
      for level in sorted( self.levels ):
         print srcDbHeader % ( srcDbList[ level - 1 ], addrFamily )
         self.levels[ level ].renderData( addrFamily, 'isis' )

      for area in sorted( self.areas ):
         print srcDbHeader % ( srcDbList[ 2 ],
                               ospfHeader.format( instanceId, area ) )
         self.areas[ area ].renderData( addrFamily, 'ospf' )

class IgpInstanceInfoModel( Model ):
   instances = Dict( keyType=int, valueType=IgpModel,
                      help="A mapping of the IGP instance ID to the IGP "
                           "database" )

   def renderData( self, addrFamily ):
      for instanceId, igpModel in self.instances.iteritems():
         igpModel.renderData( addrFamily, instanceId )

class TeRouterDbModel( Model ):
   igps = Dict( keyType=str, valueType=IgpInstanceInfoModel,
                help="A mapping of the IGP name to the IGP "
                     "information" )

   def renderData( self, addrFamily ):
      for igpInstance in self.igps.values():
         igpInstance.renderData( addrFamily )

class TeAfDbModel( Model ):
   teRouterIds = Dict( keyType=str, valueType=TeRouterDbModel,
                       help="TE DB information per TE Router ID" )

   def renderData( self, addrFamily ):
      teRouterHeader = "TE Router-ID: %s"
      for teRouterId in sorted( self.teRouterIds ):
         print teRouterHeader % ( teRouterId )
         self.teRouterIds[ teRouterId ].renderData( addrFamily )

class TeVrfDbModel( Model ):
   ipv4 = Submodel( valueType=TeAfDbModel, optional=True,
                    help="TE DB information for IPv4 address family" )
   ipv6 = Submodel( valueType=TeAfDbModel, optional=True,
                    help="TE DB information for IPv6 address family" )

   def render( self ):
      afModelList = [ self.ipv4, self.ipv6 ]
      afList = [ "IPv4", "IPv6" ]
      for i in range( len( afModelList ) ):
         if afModelList[ i ] is not None:
            afModelList[ i ].renderData( afList[ i ] )

class TeDbModel( Model ):
   __revision__ = 2
   vrfs = Dict( keyType=str, valueType=TeVrfDbModel,
                help="TE DB information per vrf" )

   def render( self ):
      for key in sorted( self.vrfs ):
         self.vrfs[ key ].render()

   def degrade( self, dictRepr, revision ):
      # Revision 1 only supports IS-IS output for a single instance ( Id 0 )
      if revision == 1:
         for teVrfDbModel in dictRepr[ 'vrfs' ].values():
            for teAfDbModel in teVrfDbModel.values():
               for teRtrId, teRtrDbModel in teAfDbModel[ 'teRouterIds' ].iteritems():
                  isisDict = teRtrDbModel.get( 'igps', {} ).get( 'isis', {} ).\
                           get( 'instances', {} ).get( '0', {} ).get( 'levels', {} )
                  isisL1 = isisDict.get( '1', {} )
                  isisL2 = isisDict.get( '2', {} )
                  teAfDbModel[ 'teRouterIds' ][ teRtrId ] = {}
                  if isisL1:
                     teAfDbModel[ 'teRouterIds' ][ teRtrId ][ 'isisLevel1' ] = isisL1
                  if isisL2:
                     teAfDbModel[ 'teRouterIds' ][ teRtrId ][ 'isisLevel2' ] = isisL2
      return dictRepr
