# Copyright (c) 2018 Arista Networks, Inc.  All rights reserved.
# Arista Networks, Inc. Confidential and Proprietary.

import struct
import socket
import threading
import Ark
import Tac
import BasicCli
import CliMatcher
import ShowCommand
import SmashLazyMount
import SharedMem
import LazyMount
import Toggles.gatedToggleLib
from CliPlugin.CspfShowDbModel import ( TeDbModel,
                                        ReservablePriorityBandwidth,
                                        InterfaceAddress,
                                        SharedRiskLinkGroup,
                                        NgbTeInfoModel,
                                        TeLinksModel,
                                        TeSourceProtocolDbModel,
                                        RouterInfoModel,
                                        TeRouterDbModel,
                                        TeAfDbModel,
                                        TeVrfDbModel,
                                        IgpInstanceInfoModel,
                                        IgpModel,
                                    )
from CliPlugin.TeCli import getSrlgIdToNameMap
from IpLibConsts import DEFAULT_VRF

matcherTrafficEngineering = CliMatcher.KeywordMatcher( 'traffic-engineering',
                            helpdesc='Traffic Engineering related information' )
AddressFamily = Tac.Type( "Arnet::AddressFamily" )
isisIpL1TopoDb = None
isisIpL2TopoDb = None
isisIp6L1TopoDb = None
isisIp6L2TopoDb = None
topoDbExport = None
shmemEm = None
readerInfo = SmashLazyMount.mountInfo( 'reader' )
mountLock = threading.RLock()

def splitSmashPath( topoDbPath ):
   tSplit = topoDbPath.fullName.split( '/' )
   if 'isis' in topoDbPath.fullName:
      proto = 'isis'
      igpId = int( tSplit[ -2 ] )
      addrFamily = tSplit[ -3 ]
      instId = 0
   else:
      assert 'ospf' in topoDbPath.fullName
      proto = 'ospf'
      # Convert area ID in integer to an IP address
      igpId = socket.inet_ntoa( struct.pack( '!I', int( tSplit[ -2 ] ) ) )
      addrFamily = tSplit[ -4 ]
      instId = int( tSplit[ -3 ] )
   return proto, addrFamily, instId, igpId

def setSourceDbModel( proto, instId, igpId, teRouter ):
   sourceDbModel = TeSourceProtocolDbModel()
   igpInstModel = teRouter.igps[ proto ].instances[ instId ]
   if proto == 'isis':
      igpInstModel.levels[ igpId ] = sourceDbModel
   else:
      igpInstModel.areas[ igpId ] = sourceDbModel
   return sourceDbModel

def getSourceDbModel( proto, instId, igpId, teRouter ):
   igpInstModel = teRouter.igps[ proto ].instances[ instId ]
   if proto == 'isis':
      sourceDbModel = igpInstModel.levels[ igpId ]
   else:
      sourceDbModel = igpInstModel.areas[ igpId ]
   return sourceDbModel

def setLanNeighborId( proto, ngbKey ):
   neighborId = ngbKey.networkId.drRouterId.stringValue
   if proto == 'isis':
      neighborId = neighborId + "." + \
            str( format( ngbKey.networkId.drNetworkId, 'x' ) )
   return neighborId

@Ark.synchronized( mountLock )
def showTeDatabase( mode, args ):
   # We only show IS-IS IPv4 L1 and L2 DBs in default vrf
   topoDbList = [ isisIpL1TopoDb, isisIpL2TopoDb, isisIp6L1TopoDb, isisIp6L2TopoDb ]

   if Toggles.gatedToggleLib.toggleOspfTopoDbExportEnabled():
      ospfSmashPath = []
      for key in topoDbExport:
         smashPath = key.replace( '-', '/' )
         ospfSmashPath.append( smashPath )
         topoDbList.append( shmemEm.doMount( smashPath,
                                             'Routing::Topology::TopoDb',
                                              readerInfo ) )

   # List of TE Router IDs_TopoDbs seen so far
   # teTopoDbSeen = [ TE1_IPv4L1, TE1_IPv4L2, TE2_IPv4L2, ..]
   teTopoDbSeen = []
   nonTeRouterId = '0.0.0.0'
   teDatabase = TeDbModel()
   vrfModel = TeVrfDbModel()
   teDatabase.vrfs[ DEFAULT_VRF ] = vrfModel
   # SRLG Id to name Mapping.
   srlgIdToNameMap = getSrlgIdToNameMap()
   for topoDb in topoDbList:
      proto, addrFamily, instId, igpId = splitSmashPath( topoDb )

      for routerId, router in topoDb.router.items():
         if addrFamily == 'ip':
            if vrfModel.ipv4 is None:
               vrfModel.ipv4 = TeAfDbModel()
            af = AddressFamily.ipv4
            afModel = vrfModel.ipv4
         elif addrFamily == 'ip6':
            if vrfModel.ipv6 is None:
               vrfModel.ipv6 = TeAfDbModel()
            af = AddressFamily.ipv6
            afModel = vrfModel.ipv6
         # Check if the TE-router ID has already been seen
         existingTeRouter = afModel.teRouterIds.get( router.teRouterId.stringValue,
                                                     None )
         if existingTeRouter != None:
            teRouter = existingTeRouter
         # Omit the inclusion of the exported non-TE nodes
         # A non-TE node has 0.0.0.0 as its router ID
         elif router.teRouterId.stringValue == nonTeRouterId:
            continue
         else:
            teRouter = TeRouterDbModel()
            afModel.teRouterIds[ router.teRouterId.stringValue ] = teRouter
         # TeSourceProtocolDbModel() is created when :
         # 1. A TE router ID has not been seen before
         # 2. A TE router ID has been seen but belongs to a different topology
         # The model is not created when :
         # 1. Two differnt routers in the same topology have the same TE router ID
         teTopoDb = router.teRouterId.stringValue + topoDb.fullName
         if teTopoDb not in teTopoDbSeen:
            if proto not in teRouter.igps:
               igpInstanceInfo = IgpInstanceInfoModel()
               igpInstanceInfo.instances[ instId ] = IgpModel()
               teRouter.igps[ proto ] = igpInstanceInfo

            sourceDbModel = setSourceDbModel( proto, instId, igpId, teRouter )
            teTopoDbSeen.append( teTopoDb )

         else:
            sourceDbModel = getSourceDbModel( proto, instId, igpId, teRouter )

         rtrInfoModel = RouterInfoModel()
         rtrInfoModel.numLinks = 0
         sourceDbModel.routers[ routerId.stringValue ] = rtrInfoModel
         for ngbKey in router.neighbor.values():
            ngb = topoDb.neighbor.get( ngbKey )
            if ngb is None:
               continue
            # Omit the inclusion of non-TE links
            # A non-TE link is one which does not have TE enabled on its interface.
            # Thus its corresponding local interface address sub-TLV remains empty.
            if not ngb.interfaceAddr:
               continue
            ngbTeInfo = NgbTeInfoModel()
            rtrInfoModel.numLinks += 1
            # Create a TE links model for either P2P or LAN
            # If it is a P2P link
            if ngbKey.neighborType == 'neighborTypeRouter':
               if rtrInfoModel.p2p != None:
                  rtrInfoModel.p2p.links.append( ngbTeInfo )
               else:
                  linkType = TeLinksModel()
                  linkType.links.append( ngbTeInfo )
                  rtrInfoModel.p2p = linkType
               ngbTeInfo.neighborId = ngbKey.neighborId.routerId.stringValue
            # If it is a LAN link
            elif ngbKey.neighborType == 'neighborTypeNetwork':
               if rtrInfoModel.lan != None:
                  rtrInfoModel.lan.links.append( ngbTeInfo )
               else:
                  linkType = TeLinksModel()
                  linkType.links.append( ngbTeInfo )
                  rtrInfoModel.lan = linkType
               ngbTeInfo.neighborId = setLanNeighborId( proto, ngbKey )

            # Fill in the neighbor TE information
            if ngb.adminGroupPresent:
               ngbTeInfo.administrativeGroup = ngb.adminGroup
            if ngb.teMetricPresent:
               ngbTeInfo.metric = ngb.teMetric
            for addr in ngb.interfaceAddr.values():
               intfAddr = InterfaceAddress()
               if af == AddressFamily.ipv4:
                  intfAddr.ipv4Address = addr.stringValue
               elif af == 'ipv6':
                  intfAddr.ipv6Address = addr.stringValue
               ngbTeInfo.interfaceAddresses.append( intfAddr )
            for addr in ngb.neighborAddr.values():
               intfAddr = InterfaceAddress()
               if af == AddressFamily.ipv4:
                  intfAddr.ipv4Address = addr.stringValue
               elif af == AddressFamily.ipv6:
                  intfAddr.ipv6Address = addr.stringValue
               ngbTeInfo.neighborAddresses.append( intfAddr )
            # BW values in Smash are in Bytes per second and are stored as float.
            # Bytes per second are converted to bits per second for show commands.
            # This consversion from float to int would not cause an overflow issue
            # since Python will implictly convert the value to long and long type
            # can have unlimited length.
            if ngb.maxBwPresent:
               ngbTeInfo.maxLinkBw = int( ngb.maxBw * 8 )
            if ngb.maxReservableBwPresent:
               ngbTeInfo.maxReservableBw = int( ngb.maxReservableBw * 8 )
            if ngb.unreservedBwPresent:
               NUM_BW_CLASSES = 8
               unreservedBw = ReservablePriorityBandwidth()
               ngbTeInfo.unreservedBw = unreservedBw
               for i in range( NUM_BW_CLASSES ):
                  val = ngb.unreservedBw.values()[ i ]
                  if val is not None:
                     setattr( unreservedBw, "priority" + str( i ),
                              int( ngb.unreservedBw.values()[ i ] * 8 ) )
            if ngb.srlgCount != 0:
               for srlgId in ngb.srlg.values():
                  srlg = SharedRiskLinkGroup()
                  srlg.groupId = srlgId
                  srlg.groupName = srlgIdToNameMap.get( srlgId, None )
                  ngbTeInfo.sharedRiskLinkGroups.append( srlg )

   if Toggles.gatedToggleLib.toggleOspfTopoDbExportEnabled():
      for smashPath in ospfSmashPath:
         shmemEm.doUnmount( smashPath )
      ospfSmashPath = []

   return teDatabase

#--------------------------------------------------------------------------------
# show traffic-engineering database
#--------------------------------------------------------------------------------
class TrafficEngineeringDatabaseCmd( ShowCommand.ShowCliCommandClass ):
   syntax = 'show traffic-engineering database'
   data = {
      'traffic-engineering' : matcherTrafficEngineering,
      'database' : 'Show TE database information',
   }
   handler = showTeDatabase
   cliModel = TeDbModel

BasicCli.addShowCommandClass( TrafficEngineeringDatabaseCmd )

def Plugin( entityManager ):
   global isisIpL1TopoDb
   global isisIpL2TopoDb
   global isisIp6L1TopoDb
   global isisIp6L2TopoDb
   global topoDbExport
   global shmemEm
   global readerInfo

   # Change the path to include default vrf in the path
   isisIpL1TopoDb = SmashLazyMount.mount( entityManager,
                                          'routing/isis/topodb/ip/1/' + DEFAULT_VRF,
                                          'Routing::Topology::TopoDb', readerInfo )
   isisIpL2TopoDb = SmashLazyMount.mount( entityManager,
                                          'routing/isis/topodb/ip/2/' + DEFAULT_VRF,
                                          'Routing::Topology::TopoDb', readerInfo )
   isisIp6L1TopoDb = SmashLazyMount.mount( entityManager,
                                          'routing/isis/topodb/ip6/1/' + DEFAULT_VRF,
                                          'Routing::Topology::TopoDb', readerInfo )
   isisIp6L2TopoDb = SmashLazyMount.mount( entityManager,
                                          'routing/isis/topodb/ip6/2/' + DEFAULT_VRF,
                                          'Routing::Topology::TopoDb', readerInfo )

   if Toggles.gatedToggleLib.toggleOspfTopoDbExportEnabled():
      topoDbExport = LazyMount.mount( entityManager,
                                      'routing/topoDbExport', 'Tac::Dir',
                                      'ri' )
      shmemEm = SharedMem.entityManager( sysdbEm=entityManager )
