# Copyright (c) 2019 Arista Networks, Inc.  All rights reserved.
# Arista Networks, Inc. Confidential and Proprietary.

from __future__ import absolute_import, division, print_function

import functools

import BasicCli
import CliCommand
import CliMatcher
import CliPlugin.ControllerCli as ControllerCli
import CliPlugin.ControllerdbLib as ControllerdbLib
import CliPlugin.FileReplicationModels as FileReplicationModels
from CliPlugin.ControllerdbModel import (
      ClusterStatus,
      ClusterStatusDebug,
      CvxService,
      CvxAllServices,
      CvxStatus,
      CvxAllConnections,
      ServiceFileReplicationStatus,
      CvxConnection,
      CvxMounts,
      CvxAllMounts,
      GroupMountStates, 
      PeerStatus,
      PeerStatusDebug,
      ServiceMountStates,
      SslProfileStatus
)
from ControllerModel import (
      HeartbeatStatus,
      MountState,
      ServiceStatus,
      utcTimestamp
)
import ShowCommand
import LazyMount
import Tac

REG_STATE = Tac.Type( "ControllerCluster::PeerRegistrationStatus::State" )
ELEC_STATE = Tac.Type( "ControllerCluster::ElectionState" )

#--------------------------------------------------------------------------------
# show cvx [ debug ]
#--------------------------------------------------------------------------------
class CvxCmd( ShowCommand.ShowCliCommandClass ):
   syntax = 'show cvx [ debug ]'
   data = {
      'cvx': ControllerCli.showCvxKwMatcher,
      'debug':
         CliCommand.Node(
            CliMatcher.KeywordMatcher(
               'debug', 'show controller debug information' ),
            hidden=True ),
   }

   @staticmethod
   def handler( mode, args ):
      debug = ( 'debug' in args )

      sslProfileStatus = None
      clusterStatusModel = None
      clusterStatus = ControllerdbLib.clusterStatus()
      peerStatusDict = dict()
      for peerName in clusterStatus.peerStatus:
         peerStatus = clusterStatus.peerStatus.get( peerName )
         if not peerStatus:
            continue
         peerSystemId = None
         registrationErrorStatus = None
         if peerStatus.registrationStatus.state == REG_STATE.registrationComplete:
            peerSystemId = peerStatus.registrationStatus.systemId.stringValue
         elif peerStatus.registrationStatus.state == REG_STATE.registrationError:
            registrationErrorStatus = peerStatus.registrationStatus.errorCodeDesc

         if debug:
            peerStatusDict[ peerStatus.name ] = PeerStatusDebug( 
               peerName=peerStatus.name,
               peerIp=peerStatus.resolvedConnectionConfig.ip,
               sshKeyCopyStatus=peerStatus.sshKeyCopyStatus.capitalize(), 
               registrationState=peerStatus.registrationStatus.stateDesc(),
               peerSystemId=peerSystemId, 
               registrationErrorStatus=registrationErrorStatus,
               serviceVersionStatus=peerStatus.serviceVersionStatusDesc() )
         else:
            peerStatusDict[ peerStatus.name ] = PeerStatus( 
               peerName=peerStatus.name,
               peerIp=peerStatus.resolvedConnectionConfig.ip,
               registrationState=peerStatus.registrationStatus.stateDesc(),
               peerSystemId=peerSystemId, 
               registrationErrorStatus=registrationErrorStatus,
               serviceVersionStatus=peerStatus.serviceVersionStatusDesc() )

      electionSts = clusterStatus.electionStatus

      leader = None
      if electionSts.leader.isEmpty():
         leader = ""
      elif electionSts.state != ELEC_STATE.leader:
         leader = electionSts.leader.stringValue

      # find ip address for this leader
      for peerName in clusterStatus.peerStatus:
         peerStatus = clusterStatus.peerStatus.get( peerName )
         if ( peerStatus and peerStatus.registrationStatus and
              peerStatus.registrationStatus.systemId.stringValue == leader ):
            leaderName = peerStatus.resolvedConnectionConfig.ip
            break
      else:
         leaderName = leader

      if peerStatusDict:
         leaderChangeTs = utcTimestamp( electionSts.lastLeaderChangeTs )
         if debug:
            clusterStatusModel = ClusterStatusDebug( 
               clusterName=clusterStatus.clusterName,
               role=clusterStatus.role.capitalize(),
               currLeader=leaderName,
               currState=electionSts.state.capitalize(),
               currTerm=str( electionSts.term ),
               peerTimeout=ControllerCli.clusterConfig().peerTimeout,
               lastLeaderChangeTerm=str( electionSts.lastLeaderChangeTerm ),
               lastLeaderChangeTs=leaderChangeTs,
               peerStatus=peerStatusDict )
         else:
            clusterStatusModel = ClusterStatus( 
               clusterName=clusterStatus.clusterName,
               role=clusterStatus.role.capitalize(),
               currLeader=leaderName,
               peerTimeout=ControllerCli.clusterConfig().peerTimeout,
               lastLeaderChangeTs=leaderChangeTs,
               peerStatus=peerStatusDict )

      if ControllerdbLib.oobStatus().sslProfileStatus:
         disableReason = None
         if not ControllerdbLib.oobStatus().sslProfileStatus.enabled:
            disableReason = \
                  ControllerdbLib.oobStatus().sslProfileStatus.disableReason
         sslProfileStatus = SslProfileStatus(
            sslProfileName=\
                  ControllerdbLib.oobStatus().sslProfileStatus.sslProfileName,
            enabled=ControllerdbLib.oobStatus().sslProfileStatus.enabled,
            disableReason=disableReason )
      return CvxStatus(
            clusterStatus=clusterStatusModel,
            enabled=ControllerCli.controllerdbStatus.enabled,
            clusterMode=bool( ControllerCli.clusterConfig().peerConfig ),
            controllerUUID=clusterStatus.controllerUUID,
            heartbeatInterval=
               ControllerCli.oobConfig().heartbeatConfig.punchInterval,
            heartbeatTimeout=ControllerCli.oobConfig().heartbeatConfig.timeout,
            sslProfileStatus=sslProfileStatus,
            connectionPreserved=ControllerCli.oobConfig().connectionPreserve )

   cliModel = CvxStatus

BasicCli.addShowCommandClass( CvxCmd )

#--------------------------------------------------------------------------------
# show cvx connections [ all | inactive | client ( SYSTEMID | HOSTNAME ) ]
#--------------------------------------------------------------------------------
def getCvxConnectionModel( mode, name, clientStatus, brief=None ):
   def _cvxConnection( mode, name, clientStatus ):
      if not clientStatus:
         # clientStatus does not present for inactive peers
         for systemId, peer in ControllerCli.inactive().iteritems():
            if peer.peerAddr == name or peer.connectionConfig.hostname == name:
               hbStat = HeartbeatStatus(
                  lastHeartbeatSent=utcTimestamp( 0.0 ),
                  lastHeartbeatReceived=utcTimestamp( 0.0 ) )
               return CvxConnection( switchId=str( systemId ),
                                     hostname=peer.connectionConfig.hostname,
                                     oobState='disconnecting',
                                     connectionTime=utcTimestamp( None ),
                                     oobConnectionSecured=False,
                                     ibConnectionSecured=False,
                                     heartbeatStatus=hbStat,
                                     oobConnectionActive=False )
         return None
      hbs = clientStatus.heartbeatStatus
      hbStat = HeartbeatStatus(
         lastHeartbeatSent=utcTimestamp( hbs.lastHeartbeatSent ),
         lastHeartbeatReceived=utcTimestamp( hbs.lastHeartbeatReceived ) )
      if not hbStat:
         mode.addError( "No heartbeat status for switch %s" % name )
         return None

      connectionConfig = clientStatus.connectionConfig

      # Include hostname and status if they are known
      hostname = ''
      oobConnSecured = False
      ibConnSecured = False
      if connectionConfig:
         # May not be known yet when switches are connecting
         hostname = connectionConfig.hostname
      if ControllerCli.oobConfig().sslProfileConfig.sslProfileName:
         oobConnSecured = True
         # Set in-band connection status to secured only when SSL is configured
         # and negotiated version >= 3
         if clientStatus.negotiatedVersion.value >= 3:
            ibConnSecured = True

      peer = ControllerdbLib.established().peer.get( clientStatus.systemId )
      connectionTime = peer.connectedTime if peer else None

      return CvxConnection(
         switchId=str( clientStatus.systemId ),
         hostname=hostname,
         oobState=clientStatus.state.replace( 'oobState', '' ).lower(),
         connectionTime=utcTimestamp( connectionTime ),
         heartbeatStatus=hbStat,
         oobConnectionSecured=oobConnSecured,
         ibConnectionSecured=ibConnSecured,
         oobConnectionActive=True )

   conn = _cvxConnection( mode, name, clientStatus )
   return conn

def getCvxConnectionBySystemId( systemId ):
   systemId = Tac.Value( 'Controller::SystemId', systemId )
   peer = ControllerdbLib.established().peer.get( systemId )
   if peer:
      client = ControllerdbLib.oobStatus().clientStatus.get( peer.peerAddr )
      return ( peer.peerAddr, client )
   else:
      for name in ControllerdbLib.oobStatus().clientStatus:
         client = ControllerdbLib.oobStatus().clientStatus.get( name )
         if client and client.systemId == systemId:
            return ( name, client )
   inactivePeer = ControllerCli.inactive().get( systemId )
   if inactivePeer:
      return ( inactivePeer.peerAddr, None )
   return ( None, None )

def getCvxConnectionByHostname( hostname ):
   for name in ControllerdbLib.oobStatus().clientStatus:
      client = ControllerdbLib.oobStatus().clientStatus.get( name )
      if client and client.connectionConfig.hostname == hostname:
         return ( name, client )

   for systemId in ControllerCli.inactive():
      inactivePeer = ControllerCli.inactive().get( systemId )
      if inactivePeer.connectionConfig.hostname == hostname:
         return ( hostname, None )
   return ( None, None )

def getCvxConnectionModelBySystemIdOrHostname( mode, systemId=None, hostname=None ):
   if systemId:
      name, client = getCvxConnectionBySystemId( systemId )
   elif hostname:
      name, client = getCvxConnectionByHostname( hostname )

   if name:
      return getCvxConnectionModel( mode, name, client )

   mode.addError( "Unknown switch: %s" % client )
   return None

connectionsNode = CliCommand.guardedKeyword(
      'connections', 'Show all connected switches', ControllerdbLib.controllerGuard )

class CvxConnectionsClientCmd( ShowCommand.ShowCliCommandClass ):
   syntax = 'show cvx connections client ( SYSTEMID | HOSTNAME )'
   data = {
      'cvx' : ControllerCli.showCvxKwMatcher,
      'connections': connectionsNode,
      'client': ControllerCli.clientMatcher,
      'SYSTEMID': ControllerCli.systemIdMatcher,
      'HOSTNAME': ControllerCli.hostnameMatcher,
   }

   @staticmethod
   def handler( mode, args ):
      systemId = args.get( 'SYSTEMID' )
      hostname = args.get( 'HOSTNAME' )
      return getCvxConnectionModelBySystemIdOrHostname( mode, systemId, hostname )

   cliModel = CvxConnection

BasicCli.addShowCommandClass( CvxConnectionsClientCmd )

#--------------------------------------------------------------------------------
# show cvx connections [ all | inactive | client ( SYSTEMID | HOSTNAME ) ] brief
#--------------------------------------------------------------------------------
briefMatcher = CliMatcher.KeywordMatcher( 'brief', 'Show brief status only' )

class CvxConnectionsClientBriefCmd( ShowCommand.ShowCliCommandClass ):
   syntax = 'show cvx connections client ( SYSTEMID | HOSTNAME ) brief'
   data = {
      'cvx' : ControllerCli.showCvxKwMatcher,
      'connections': connectionsNode,
      'client': ControllerCli.clientMatcher,
      'SYSTEMID': ControllerCli.systemIdMatcher,
      'HOSTNAME': ControllerCli.hostnameMatcher,
      'brief': briefMatcher,
   }

   @staticmethod
   def handler( mode, args ):
      systemId = args.get( 'SYSTEMID' )
      hostname = args.get( 'HOSTNAME' )
      conn = getCvxConnectionModelBySystemIdOrHostname( mode, systemId, hostname )
      if conn:
         return CvxAllConnections( connections=[ conn ], brief=True )
      else:
         return None

   cliModel = CvxAllConnections

BasicCli.addShowCommandClass( CvxConnectionsClientBriefCmd )

#--------------------------------------------------------------------------------
# show cvx connections [ all | inactive ] [ brief ]
#--------------------------------------------------------------------------------
class CvxConnectionsCmd( ShowCommand.ShowCliCommandClass ):
   syntax = 'show cvx connections [ all | inactive ] [ brief ]'
   data = {
      'cvx' : ControllerCli.showCvxKwMatcher,
      'connections': connectionsNode,
      'all':
         CliCommand.Node(
            CliMatcher.KeywordMatcher(
               'all', 'show all connected switches' ),
            hidden=True ),
      'inactive': 'show all inactive switches',
      'brief': briefMatcher,
   }

   @staticmethod
   def handler( mode, args ):
      inactiveOnly = ( 'inactive' in args )
      brief = ( 'brief' in args ) or inactiveOnly or False
      connections = []
      if not inactiveOnly:
         for name in ControllerdbLib.oobStatus().clientStatus:
            client = ControllerdbLib.oobStatus().clientStatus.get( name )
            if not client:
               continue
            conn = getCvxConnectionModel( mode, name, client, brief=brief )
            # getCvxConnection might return None if it finds a problem.
            # It should have already added an error to 'mode', so it's
            # safe to just return.
            if not conn:
               return None
            connections.append( conn )
      for systemId in ControllerCli.inactive():
         inactivePeer = ControllerCli.inactive().get( systemId )
         conn = getCvxConnectionModel( mode, inactivePeer.peerAddr, None,
               brief=brief )
         if not conn:
            return None
         connections.append( conn )

      return CvxAllConnections( connections=connections, brief=brief )

   cliModel = CvxAllConnections

BasicCli.addShowCommandClass( CvxConnectionsCmd )

#--------------------------------------------------------------------------------
# show cvx file-replication
#--------------------------------------------------------------------------------
nodeFileReplication = CliCommand.guardedKeyword(
         'file-replication', 'File replication status of all cvx services',
         ControllerdbLib.controllerGuard )

def isLeader():
   electionSts = ControllerdbLib.clusterStatus().electionStatus
   return electionSts.state == ELEC_STATE.leader

def fileReplicationStatusNames( mode ):
   return ControllerCli.fileRepStatusDir.keys()

def getCvxFileReplicationName( mode, name ):
   if name not in ControllerCli.fileRepStatusDir:
      mode.addError( 'Invalid service name' )
      return None
   if not isLeader():
      return FileReplicationModels.Status()

   status = ControllerCli.fileRepStatusDir[ name ]
   requesters = {}
   for reqName, reqStatus in status.requester.items():
      files = {}
      for fileName, fileStatus in reqStatus.file.items():
         files[ fileName ] = FileReplicationModels.FileStatus( 
            status=fileStatus.status, lastSyncTime=fileStatus.lastSynchronized )
      requesters[ reqName ] = FileReplicationModels.RequesterStatus( files=files )

   return FileReplicationModels.Status( requesters=requesters )

class CvxFileReplicationCmd( ShowCommand.ShowCliCommandClass ):
   syntax = 'show cvx file-replication'
   data = {
      'cvx': ControllerCli.showCvxKwMatcher,
      'file-replication': nodeFileReplication,
   }

   @staticmethod
   def handler( mode, args ):
      services = {}
      if isLeader():
         for name in fileReplicationStatusNames( mode ):
            services[ name ] = getCvxFileReplicationName( mode, name )
      return ServiceFileReplicationStatus( services=services )

   cliModel = ServiceFileReplicationStatus

BasicCli.addShowCommandClass( CvxFileReplicationCmd )

#--------------------------------------------------------------------------------
# show cvx file-replication NAME
#--------------------------------------------------------------------------------
class CvxFileReplicationNameCmd( ShowCommand.ShowCliCommandClass ):
   syntax = 'show cvx file-replication NAME'
   data = {
      'cvx' : ControllerCli.showCvxKwMatcher,
      'file-replication' : nodeFileReplication,
      'NAME' : CliMatcher.DynamicNameMatcher(
         fileReplicationStatusNames, 'Service name' ),
   }

   @staticmethod
   def handler( mode, args ):
      return getCvxFileReplicationName( mode, args[ 'NAME' ] )

   cliModel = FileReplicationModels.Status

BasicCli.addShowCommandClass( CvxFileReplicationNameCmd )

#--------------------------------------------------------------------------------
# show cvx mounts
#--------------------------------------------------------------------------------
nodeMounts = CliCommand.Node(
      CliMatcher.KeywordMatcher( 'mounts', 'cvx mounts' ),
      guard=ControllerdbLib.controllerGuard,
      hidden=True )

def showCvxMountWithActivityLock( func ):
   @functools.wraps( func )
   def _withActivityLock( *args, **kwargs ):
      # Force mount before grabbing activity lock. Otherwise 
      # it will result in a deadlock as mounts also tries to 
      # grab activity lock.
      LazyMount.force( ControllerCli.controllerMountConfig )
      LazyMount.force( ControllerCli.controllerMountStatus )
      LazyMount.force( ControllerCli.proxyControllerMountConfig )
      LazyMount.force( ControllerCli.proxyControllerMountStatus )
      
      with Tac.ActivityLockHolder():
         return func( *args, **kwargs )

   return _withActivityLock

@showCvxMountWithActivityLock
def getCvxMountsSystemId( mode, systemId ):
   systemId = Tac.Value( 'Controller::SystemId', systemId )
   if not ControllerdbLib.established() or \
         systemId not in ControllerdbLib.established().peer:
      mode.addError( "Unknown system ID: '%s'" % systemId )
      return None

   # Enumerate mount path types for each service (from config)
   mountTypes = {}
   systemWhitelist = {}
   versionWhitelist = {}
   if ControllerCli.controllerMountConfig:
      for ( _, service ) in ( ControllerCli.controllerMountConfig.items() + 
                              ControllerCli.proxyControllerMountConfig.items() ) :
         serviceName = service.serviceName
         if not serviceName:
            continue
         groups = []
         # for this service, enumerate all mount groups
         if service and service.mountGroupConfig:
            groupsMountConfig = service.mountGroupConfig
            systemWhitelist[ serviceName ] = {}
            versionWhitelist[ serviceName ] = {}
            for ( groupName, group ) in groupsMountConfig.items():
               for path in group.mountConfig:
                  mountTypes[ path ] = group.mountConfig[ path ].taccType
                  
               systemWhitelist[ serviceName ][ groupName ] = []
               versionWhitelist[ serviceName ][ groupName ] = []
               if group.systemFilter:
                  for systemName in group.systemFilter.includedSystem:
                     switchName = \
                           ControllerdbLib.switchIdCache.getHost( systemName ) or \
                           systemName
                     systemWhitelist[ serviceName ][ groupName ].append( switchName )
               if group.versionFilter:
                  for version in group.versionFilter.includedServiceVersion:
                     versionWhitelist[ serviceName ][ groupName ].append( version )

   # Enumerate controller mount states for each service
   controllerMountStates = []
   if ControllerCli.controllerMountStatus:
      for ( _, service ) in ( ControllerCli.controllerMountStatus.items() +
                              ControllerCli.proxyControllerMountStatus.items() ):
         serviceName = service.serviceName
         if not serviceName:
            continue
         groups = []

         # navigate to systemStatus/[systemId]/serviceMountGroupStatus
         if ( service and \
              service.systemStatus and \
              ( systemId.stringValue in service.systemStatus ) ):
            groupsMountStatus = service.\
                                systemStatus[ systemId.stringValue ].\
                                serviceMountGroupStatus
            for ( groupName, group ) in groupsMountStatus.items():
               pathStates = []
               for path in group.mountStatus:
                  pathStatus = group.mountStatus[ path ]
                  mountType = "(not configured)"
                  if path in mountTypes:
                     mountType = mountTypes[ path ]

                  pathStates.append(
                        MountState( path=path,
                                    type=mountType,
                                    state=pathStatus.mountState ) )

               try:
                  whitelistedSystems = systemWhitelist[ serviceName ][ groupName ]
                  whitelistedVersions = versionWhitelist[ serviceName ][ groupName ]
               except KeyError:
                  continue

               groups.append( GroupMountStates( group=group.name,
                                                state=group.mountState,
                                                pathStates=pathStates,
                                                systemWhitelist=whitelistedSystems,
                                                versionWhitelist=whitelistedVersions
                                                ) )

         if groups:
            serviceMountState = ServiceMountStates( service=serviceName,
                                                    mountStates=groups )
            controllerMountStates.append( serviceMountState )
   else:
      mode.addError( "No controller mount information available." )

   switchName = ControllerdbLib.switchIdCache.getHost( systemId.stringValue )
   return CvxMounts( switchId=str( systemId ),
                     hostname=switchName,
                     mounts=controllerMountStates )

class CvxMountsCmd( ShowCommand.ShowCliCommandClass ):
   syntax = 'show cvx mounts'
   data = {
      'cvx' : ControllerCli.showCvxKwMatcher,
      'mounts' : nodeMounts,
   }

   @staticmethod
   @showCvxMountWithActivityLock
   def handler( mode, args ):
      connections = []
      for systemId in ControllerdbLib.established().peer:
         conn = getCvxMountsSystemId( mode, str( systemId ) ) 

         # getCvxMountsSystemId might return None if it finds a problem.
         # It should have already added an error to 'mode', so its safe to
         # just return.
         if not conn:
            return None

         connections.append( conn )
      return CvxAllMounts( connections=connections )

   cliModel = CvxAllMounts

BasicCli.addShowCommandClass( CvxMountsCmd )

#--------------------------------------------------------------------------------
# show cvx mounts ( HOSTNAME | SYSTEMID )
#--------------------------------------------------------------------------------
@showCvxMountWithActivityLock
def getCvxMountsHostname( mode, hostname ):
   systemId = ControllerdbLib.switchIdCache.getSwitch( hostname )
   if systemId and systemId != hostname:
      switchId = Tac.Value( 'Controller::SystemId' )
      switchId.stringValue = systemId
      return getCvxMountsSystemId( mode, str( switchId ) )

   mode.addError( "Unknown hostname: '%s'" % hostname )
   return None

class CvxMountsHostnameOrSystemIdCmd( ShowCommand.ShowCliCommandClass ):
   syntax = 'show cvx mounts ( HOSTNAME | SYSTEMID )'
   data = {
      'cvx': ControllerCli.showCvxKwMatcher,
      'mounts': nodeMounts,
      'HOSTNAME': ControllerCli.hostnameMatcher,
      'SYSTEMID': ControllerCli.systemIdMatcher,
   }

   @staticmethod
   def handler( mode, args ):
      hostname = args.get( 'HOSTNAME' )
      if hostname:
         return getCvxMountsHostname( mode, hostname )
      else:
         macAddr = args.get( 'SYSTEMID' )
         return getCvxMountsSystemId( mode, macAddr )

   cliModel = CvxMounts

BasicCli.addShowCommandClass( CvxMountsHostnameOrSystemIdCmd )

#--------------------------------------------------------------------------------
# show cvx service
#--------------------------------------------------------------------------------
nodeService = CliCommand.guardedKeyword( 'service', 'Services on the controller',
      ControllerdbLib.controllerGuard )

def allServiceNames( mode ):
   return [ k for k in ControllerCli.serviceConfig.service.keys() if k != "CVX" ]

def getCvxServiceName( mode, name ):   
   # We do case-insensitive comparison here
   # Convert the input parameter 'name' to lowercase
   # Covert service names to lowercase and do comparison
   serviceNames = allServiceNames( mode )
   serviceNames = [ serviceName for serviceName in serviceNames
                    if serviceName.lower() == name.lower() ]
   if not serviceNames:
      mode.addError( "No service named '%s'" % name )
      return None

   name = serviceNames[ 0 ]
   allStatuses = {}
   for switchId in ControllerdbLib.established().peer:
      try:
         peer = ControllerdbLib.established().peer[ switchId ]
         statusDir = ControllerdbLib.oobStatus().clientStatus[ peer.peerAddr ]\
               .serviceStatusDir
         peerServiceStatus = statusDir.service[ name ]
      except KeyError:
         continue
      vs = peerServiceStatus.versionStatus
      switchName = ControllerdbLib.switchIdCache.getHost( switchId.stringValue ) or \
            str( switchId )
      allStatuses[ switchName ] = ServiceStatus(
            enabled=peerServiceStatus.enabled,
            versionNegotiationComplete=vs.versionNegotiationComplete,
            versionCompatible=vs.versionCompatible,
            negotiatedVersion=vs.negotiatedVersion.value )

   conf = ControllerCli.serviceConfig.service[ name ]
   return CvxService( name=name, status=conf.enabled,
                      versions=conf.versionConfig.version.keys(),
                      switches=allStatuses )

class CvxServiceCmd( ShowCommand.ShowCliCommandClass ):
   syntax = 'show cvx service'
   data = {
      'cvx' : ControllerCli.showCvxKwMatcher,
      'service' : nodeService,
   }

   @staticmethod
   def handler( mode, args ):
      services = []
      for name in allServiceNames( mode ):
         services.append( getCvxServiceName( mode, name ) )
      return CvxAllServices( services=services )

   cliModel = CvxAllServices

BasicCli.addShowCommandClass( CvxServiceCmd )

#--------------------------------------------------------------------------------
# show cvx service NAME
#--------------------------------------------------------------------------------
class CvxServiceNameCmd( ShowCommand.ShowCliCommandClass ):
   syntax = 'show cvx service NAME'
   data = {
      'cvx' : ControllerCli.showCvxKwMatcher,
      'service' : nodeService,
      'NAME' : CliMatcher.DynamicNameMatcher( allServiceNames, 'Service name' ),
   }

   @staticmethod
   def handler( mode, args ):
      return getCvxServiceName( mode, args[ 'NAME' ] )

   cliModel = CvxService

BasicCli.addShowCommandClass( CvxServiceNameCmd )
