# Copyright (c) 2014 Arista Networks, Inc.  All rights reserved.
# Arista Networks, Inc. Confidential and Proprietary.

import Ark
import ConfigMount
import ControllerdbEntityManager
import Plugins
import Tac
import Tracing
import os
import time

__defaultTraceHandle__ = Tracing.Handle( 'ControllerCli' )
t8 = Tracing.trace8

Constants = Tac.Value( "Controller::Constants" )

clusterStatusDir = None
config = None
status = None
em = None
controllerdbMgr = None

def controllerNotReady():
   if controllerdbMgr.controllerEnabled_:
      return None
   return "controller not ready"

def controllerOnSwitchGuard( mode, token ):
   if config.runOnPhysicalSwitch:
      return None
   platform = Ark.getPlatform()
   if platform not in ( 'veos', 'ceoslab' ):
      return "CVX cannot be enabled on a physical switch"
   return None

def controllerGuard( mode, token ):
   return controllerNotReady()

def registerNotifiee( notifiee ):
   t8( "registering notifiee", notifiee )
   controllerdbMgr.controllerNotifieeIs( notifiee )

# When Controllerdb is shutdown from Cli, attrlog socket to Controllerdb will
# get closed from under us. That will cause error messages to be printed on
# the console and syslog. We disable those error messages here.
def attrlogErrorIs( enable ):
   """
   'enable == True' enables errors on console and syslog, 'enable == False'
   disables it.
   """
   outSm = Tac.Type( "Tac::NboAttrLog::Out" )
   outSm.muteSocketError( "tbl://controllerdb/+n", not enable, not enable )
   
class ClusterStatusDirReactor( Tac.Notifiee ):
   notifierTypeName = "ControllerCluster::ClusterStatusDir"

   def __init__( self, mgr ):
      t8( "initializing cluster status reactor" )
      Tac.Notifiee.__init__( self, clusterStatusDir )
      self.controllerLeaderReactor_ = {}
      self.mgr_ = mgr

      for clusterName in clusterStatusDir.status:
         self.handleStatus( clusterName )

   @Tac.handler( "status" )
   def handleStatus( self, clusterName ):
      if clusterName in clusterStatusDir.status:
         self.controllerLeaderReactor_[ clusterName ] = \
               ControllerLeaderReactor( clusterName, self.mgr_ )
      else:
         del self.controllerLeaderReactor_[ clusterName ]

class ControllerLeaderReactor( Tac.Notifiee ):
   notifierTypeName = "ControllerCluster::ClusterStatus"

   def __init__( self, clusterName, mgr ):
      t8( "initializing controller leader reactor" )
      self.mgr = mgr
      Tac.Notifiee.__init__( self, clusterStatusDir.status[ clusterName ] )
      self.handleLeader()

   @Tac.handler( "isStandaloneOrLeader" )
   def handleLeader( self ):
      if( self.notifier().enabled and 
            self.notifier().isStandaloneOrLeader ):
         self.mgr.notifyControllerNotifiees( True ) 

class ControllerdbStatusReactor( Tac.Notifiee ):
   notifierTypeName = "Controllerdb::Status"

   def __init__( self ):
      t8( "initializing controllerdbstatus reactor" )
      Tac.Notifiee.__init__( self, status )
      self.handleEnabled()

   @Tac.handler( "enabled" )
   def handleEnabled( self ):
      controllerdbMgr.handleEnabled( self.notifier_.enabled )

class ControllerdbMgr( object ):
   def __init__( self ):
      t8( "initializing controllerdbmgr" )
      self.controllerdbEm_ = None
      self.controllerdbRoot_ = None
      self.controllerdbStatusReactor_ = None
      self.clusterStatusDirReactor_ = None
      self.controllerEnabled_ = False
      self.controllerNotifiees_ = []
      self.firstTimeConnection_ = True

   def controllerdbStatusReactorIs( self, ):
      t8( "instantiating controllerdb status reactor" )
      self.controllerdbStatusReactor_ = ControllerdbStatusReactor()

   def controllerLeaderReactorIs( self, ):
      t8( "instantiating controllerdb status reactor" )
      self.clusterStatusDirReactor_ = ClusterStatusDirReactor( self )

   def controllerNotifieeIs( self, func ):
      t8( "registering notifiee", func )
      assert self.firstTimeConnection_
      self.controllerNotifiees_.append( func )

   def notifyControllerNotifiees( self, enabled ):
      if not enabled:
         return
      assert isinstance( self.controllerNotifiees_, list )
      for notifiee in self.controllerNotifiees_:
         if not self.controllerdbEm_:
            t8( "controllerdbEm is None, not notifying controller notifiees" )
            return

         t8( "notifying", notifiee )
         notifiee( self.controllerdbEm_ )

   def handleMountFailure( self, mountUrl=None ):
      t8( "Ignoring mount failure" )
      if not self.controllerEnabled_:
         attrlogErrorIs( True )

   def handleEnabled( self, enabled ):
      t8( "handleEnabled enabled", enabled )
      if enabled:
         def _finish():
            t8( "mounts suceeded - disabling controller guard" )
            self.notifyControllerNotifiees( status.enabled )
            self.controllerEnabled_ = status.enabled
            attrlogErrorIs( not status.enabled )

         if em.local():
            self.controllerdbEm_ = ControllerdbEntityManager.Local( em.sysname() )
            _finish()
         else:
            if em.redundancyStatus().mode != 'active':
               # This is only for ptests that run CVX on a physical switch
               # Without having this check, ConfigAgent tries to mount
               # Controllerdb indefinitely. See BUG251295.
               return
            if self.firstTimeConnection_:
               t8( "Attempting to mount controllerdb" )
               self.firstTimeConnection_ = False
               controllerSockname = ( os.environ.get( "CONTROLLERDBSOCKNAME" )
                                      or Constants.controllerdbDefaultSockname )
               self.controllerdbEm_ = ControllerdbEntityManager.Controllerdb(
                     em.sysname(), controllerdbSockname_=controllerSockname,
                     dieOnDisconnect=False, mountRoot=False )
               mg = self.controllerdbEm_.mountGroup(
                     mountFailureCallback=self.handleMountFailure,
                     persistent=True )
               self.controllerdbRoot_ = mg.mount( "", "Tac::Dir", "rt" )
               mg.close( _finish )
            else:
               t8( "Not a first time connection - disabling controller guard" )
               self.controllerEnabled_ = True
      else:
         t8( "enabling controller guard" )
         self.controllerEnabled_ = False

#
# Switch to hostname and IP cache.
#
class SwitchIdCache( object ):
   """
   CVX infra mounts Switches under a 'switchId' path. Switch ID is unique per
   switch. To allow a more user friendly switch specification we allow
   hostname or IP of the switch to be used. This cache build a forward and
   reverse map for all mounted switches from system generated ID to its
   hostname or IP.

   The cache is invalid after 60 seconds and will be refreshed on next access.
   """

   RETENTION = 60 # Cache expiry in seconds
 
   def __init__( self ):
      self.switchIdToIp = {}
      self.switchIdToHostname = {}
      self.ipToSwitchId = {}
      self.hostnameToSwitchId = {}
      self.cacheExpiration = 0
      self.switchNames = []

   def scan( self ):
      self.switchIdToIp = {}
      self.switchIdToHostname = {}
      self.ipToSwitchId = {}
      self.hostnameToSwitchId = {}

      connected = established()
      for p in connected.peer:
         connConfig = connected.peer[ p ].connectionConfig
         if not connConfig:
            continue
         systemId = p.stringValue

         # There are no clever and efficient ways of doing this, or for
         # inverting the dictionary. Linear scan and populate works.
         self.switchIdToHostname[ systemId ] = connConfig.hostname
         self.hostnameToSwitchId[ connConfig.hostname ] = systemId

         self.switchIdToIp[ systemId ] = connConfig.ip
         self.ipToSwitchId[ connConfig.ip ] = systemId

      self.cacheExpiration = time.time() + self.RETENTION

   def getSwitch( self, switchArg ):
      """
      Given a switchId, return switchId if a switchId mapping exists, return
      False if switchArg is not None and no mapping exists, if switchArg
      evaluates to False return None.
      """
      if not switchArg:
         return None

      if time.time() > self.cacheExpiration:
         self.scan()

      switchId = self.hostnameToSwitchId.get( switchArg )
      if not switchId:
         switchId = self.ipToSwitchId.get( switchArg )
      if not switchId:
         switchId = switchArg if switchArg in self.switchIdToIp else False

      return switchId

   def getHost( self, switchId ):
      """
      Return hostname given switchId maps to, None if there is no entry
      """
      if time.time() > self.cacheExpiration:
         self.scan()

      return self.switchIdToHostname.get( switchId )

   def getIp( self, switchId ):
      """
      Return IP given switchId maps to, None if there is no entry
      """
      if time.time() > self.cacheExpiration:
         self.scan()

      return self.switchIdToIp.get( switchId )

   def switchIds( self ):
      if time.time() > self.cacheExpiration:
         self.scan()
      return self.switchIdToIp.keys()

   def allNames( self, mode=None ):
      """
      Return a list of all switchIds, hostnames and IP addresses for switches
      under VCS control.
      """
      if time.time() > self.cacheExpiration or not self.switchNames:
         self.scan()
         self.switchNames = ( self.switchIdToIp.keys() +
                              self.hostnameToSwitchId.keys() +
                              self.ipToSwitchId.keys() )
      return self.switchNames

# Global cache object for switch id to hostname or IP or reverse cache
switchIdCache = SwitchIdCache()

def clusterStatus():
   # The default status is created in DefaultConfigPlugin. It should always exist
   return clusterStatusDir.status[ Constants.clusterDefaultName ]

def oobStatus():
   return clusterStatus().oobStatus

def established():
   return oobStatus().established if oobStatus() else None

@Plugins.plugin( provides=( "ControllerdbMgr", ) )
def Plugin( entityManager ):
   global em
   global controllerdbMgr
   global status
   global clusterStatusDir
   global config

   em = entityManager
   controllerdbMgr = ControllerdbMgr()

   def doMountsComplete():
      controllerdbMgr.controllerdbStatusReactorIs()
      controllerdbMgr.controllerLeaderReactorIs()

   mg = em.mountGroup()
   clusterStatusDir = mg.mount( "controller/cluster/statusDir",
         "ControllerCluster::ClusterStatusDir", "r" )
   config = ConfigMount.mount( entityManager, "controller/config",
                                           "Controllerdb::Config", "w" )
   status = mg.mount( "controller/status", "Controllerdb::Status", "r" )
   mg.close( doMountsComplete )

