# Copyright (c) 2019 Arista Networks, Inc.  All rights reserved.
# Arista Networks, Inc. Confidential and Proprietary.

from __future__ import absolute_import, division, print_function

import CliCommand
import CliMatcher
import ControllerModel
import CliPlugin.ControllerCli as ControllerCli
import CliPlugin.IntfCli as IntfCli
import CliPlugin.Ssl as Ssl
from IpLibConsts import DEFAULT_VRF

matcherPeer = CliMatcher.KeywordMatcher( 'peer',
      helpdesc='Configure connection to peer CVX server' )

#--------------------------------------------------------------------------------
# [ no | default ] connection state preserve
#--------------------------------------------------------------------------------
class ConnectionStatePreserveCmd( CliCommand.CliCommandClass ):
   syntax = 'connection state preserve'
   noOrDefaultSyntax = syntax
   data = {
      'connection': 'Client connections properties',
      'state': 'Client connection state',
      'preserve': 'Preserve state of clients upon disconnection',
   }

   @staticmethod
   def handler( mode, args ):
      ControllerCli.oobConfig().connectionPreserve = True

   @staticmethod
   def noOrDefaultHandler( mode, args ):
      ControllerCli.oobConfig().connectionPreserve = False

ControllerCli.CvxConfigMode.addCommandClass( ConnectionStatePreserveCmd )

#--------------------------------------------------------------------------------
# [ no | default ] heartbeat-interval INTERVAL
#--------------------------------------------------------------------------------
class HeartbeatIntervalIntervalCmd( CliCommand.CliCommandClass ):
   syntax = 'heartbeat-interval INTERVAL'
   noOrDefaultSyntax = 'heartbeat-interval ...'
   data = {
      'heartbeat-interval': 'Time between CVX heartbeat messages',
      'INTERVAL':
         CliMatcher.IntegerMatcher(
            ControllerCli.heartbeatIntervalType.min,
            ControllerCli.heartbeatIntervalType.max,
            helpdesc='Seconds' ),
   }

   @staticmethod
   def handler( mode, args ):
      interval = args.get( 'INTERVAL' )
      if not interval:
         interval = ControllerCli.heartbeatIntervalType.defaultVal

      ControllerCli.oobConfig().heartbeatConfig.punchInterval = interval

   noOrDefaultHandler = handler

ControllerCli.CvxConfigMode.addCommandClass( HeartbeatIntervalIntervalCmd )

#--------------------------------------------------------------------------------
# [ no | default ] heartbeat-timeout TIMEOUT
#--------------------------------------------------------------------------------
class HeartbeatTimeoutTimeoutCmd( CliCommand.CliCommandClass ):
   syntax = 'heartbeat-timeout TIMEOUT'
   noOrDefaultSyntax = 'heartbeat-timeout ...'
   data = {
      'heartbeat-timeout':
         'Elapsed time since last heartbeat before breaking connection with a '
         'client',
      'TIMEOUT':
         CliMatcher.IntegerMatcher(
            ControllerCli.heartbeatTimeoutType.min,
            ControllerCli.heartbeatTimeoutType.max,
            helpdesc='Seconds' ),
   }

   @staticmethod
   def handler( mode, args ):
      timeout = args.get( 'TIMEOUT' )
      if not timeout:
         timeout = ControllerCli.heartbeatTimeoutType.defaultVal

      ControllerCli.oobConfig().heartbeatConfig.timeout = timeout

   noOrDefaultHandler = handler

ControllerCli.CvxConfigMode.addCommandClass( HeartbeatTimeoutTimeoutCmd )

#--------------------------------------------------------------------------------
# [ no | peer ] peer host HOSTNAME
#--------------------------------------------------------------------------------
def isMyIp( ipAddr ):
   ip = ipAddr.stringValue
   if ip == '0.0.0.0':
      return False
   for ipIntfConfig in ControllerCli.ipConfig.ipIntfConfig.itervalues():
      if ipIntfConfig.vrf != DEFAULT_VRF:
         # CVX is currently only supported in default vrf
         continue
      if ( ip == ipIntfConfig.addrWithMask.address or 
           ip == ipIntfConfig.virtualAddrWithMask.address ):
         return True
      for ipAddrMask in ipIntfConfig.secondaryWithMask:
         if ip == ipAddrMask.address:
            return True
   return False

class PeerHostHostnameCmd( CliCommand.CliCommandClass ):
   syntax = 'peer host HOSTNAME'
   noOrDefaultSyntax = 'peer host [ HOSTNAME ]'
   data = {
      'peer': matcherPeer,
      'host': 'Hostname or IP Address of CVX server',
      'HOSTNAME': ControllerCli.hostnameMatcher,
   }

   @staticmethod
   def handler( mode, args ):
      hostname = args[ 'HOSTNAME' ]
      ip, name = ControllerModel.ipOrHostname( hostname )
      if not name and ControllerModel.invalidHostIp( ip ):
         mode.addError( "Invalid peer IP address" )
         return

      if isMyIp( ip ):
         mode.addWarning( "Self address %s is configured for peer host" % ip )

      clusterCfg = ControllerCli.clusterConfig()
      peerConfig = clusterCfg.peerConfig.get( hostname )
      if not peerConfig:
         peerConfig = clusterCfg.newPeerConfig( hostname, )
         peerConfig.connectionConfig = ( "%sConnectionConfig" % hostname, )
         connectionConfig = peerConfig.connectionConfig
         connectionConfig.ip = ip
         connectionConfig.hostname = name
         connectionConfig.port = ControllerCli.constants.clusterMgmtDefaultPort

   @staticmethod
   def noOrDefaultHandler( mode, args ):
      hostname = args.get( 'HOSTNAME' )
      if hostname:
         del ControllerCli.clusterConfig().peerConfig[ hostname ]
      else:
         ControllerCli.clusterConfig().peerConfig.clear()

ControllerCli.CvxConfigMode.addCommandClass( PeerHostHostnameCmd )

#--------------------------------------------------------------------------------
# [ no | default ] peer timeout TIMEOUT
#--------------------------------------------------------------------------------
class PeerTimeoutTimeoutCmd( CliCommand.CliCommandClass ):
   syntax = 'peer timeout TIMEOUT'
   noOrDefaultSyntax = 'peer timeout ...'
   data = {
      'peer': matcherPeer,
      'timeout': 'Heartbeat timeout for intra cluster communication',
      'TIMEOUT':
         CliMatcher.IntegerMatcher(
            ControllerCli.peerTimeoutType.min,
            ControllerCli.peerTimeoutType.max,
            helpdesc='Seconds' ),
   }

   @staticmethod
   def handler( mode, args ):
      timeout = args.get( 'TIMEOUT' )
      if not timeout:
         timeout = ControllerCli.peerTimeoutType.defaultVal

      ControllerCli.clusterConfig().peerTimeout = timeout

   noOrDefaultHandler = handler

ControllerCli.CvxConfigMode.addCommandClass( PeerTimeoutTimeoutCmd )

#--------------------------------------------------------------------------------
# [ no | default ] port PORT
#--------------------------------------------------------------------------------
class PortPortCmd( CliCommand.CliCommandClass ):
   # BUG74338: This command currently has no effect
   syntax = 'port PORT'
   noOrDefaultSyntax = 'port ...'
   data = {
      'port': 'CVX connection port',
      'PORT':
         CliMatcher.IntegerMatcher(
            1, 65535, helpdesc='Number of the port to use', helpname='PORT' ),
   }
   hidden = True

   @staticmethod
   def handler( mode, args ):
      ControllerCli.clusterConfig().port = \
            args.get( 'PORT', ControllerCli.constants.controllerdbDefaultPort )

   noOrDefaultHandler = handler

ControllerCli.CvxConfigMode.addCommandClass( PortPortCmd )

#--------------------------------------------------------------------------------
# [ no | default ] shutdown
#--------------------------------------------------------------------------------
def shutdown( mode, enabled ):
   # Run any registered shutdown callbacks.
   ControllerCli.runCvxShutdownCallbacks( mode, enabled )

   ControllerCli.clusterConfig().enabled = enabled
   ControllerCli.controllerdbConfig.enabled = enabled
   ControllerCli.serviceConfig.service[ "CVX" ].enabled = enabled

class ShutdownCmd( CliCommand.CliCommandClass ):
   syntax = 'shutdown'
   noOrDefaultSyntax = syntax
   data = {
      'shutdown': 'Disable CVX server',
   }

   @staticmethod
   def handler( mode, args ):
      shutdown( mode, False )

   defaultHandler = handler

   @staticmethod
   def noHandler( mode, args ):
      shutdown( mode, True )

ControllerCli.CvxConfigMode.addCommandClass( ShutdownCmd )

#--------------------------------------------------------------------------------
# [ no | default ] source-interface INTF
#--------------------------------------------------------------------------------
class SourceInterfaceIntfCmd( CliCommand.CliCommandClass ):
   syntax = 'source-interface INTF'
   noOrDefaultSyntax = 'source-interface ...'
   data = {
      'source-interface': 'Interface providing the IP source address of CVX packets',
      'INTF': IntfCli.Intf.matcherWithIpSupport,
   }

   @staticmethod
   def handler( mode, args ):
      intf = args[ 'INTF' ]
      ControllerCli.clusterConfig().sourceIntf = intf.name

   @staticmethod
   def noOrDefaultHandler( mode, args ):
      ControllerCli.clusterConfig().sourceIntf = ''

ControllerCli.CvxConfigMode.addCommandClass( SourceInterfaceIntfCmd )

#--------------------------------------------------------------------------------
# [ no | default ] ssl profile ( PROFILE_NAME [ strict ] | strict )
#--------------------------------------------------------------------------------
class SslProfileSslprofilenameCmd( CliCommand.CliCommandClass ):
   syntax = 'ssl profile PROFILE_NAME [ strict ]'
   noOrDefaultSyntax = 'ssl profile [ strict ]'
   data = {
      'ssl': Ssl.sslMatcher,
      'profile': Ssl.profileMatcher,
      'PROFILE_NAME':
         CliMatcher.DynamicNameMatcher(
            lambda mode: ControllerCli.mgmtSecConfig.profileConfig,
            'SSL profile name' ),
      'strict':
         CliCommand.Node(
            CliMatcher.KeywordMatcher(
               'strict',
               'strict option is deprecated and does not affect the config' ),
            # strict is deprecated, so it is not set anywhere
            hidden=True ),
   }

   @staticmethod
   def handler( mode, args ):
      sslProfileName = args[ 'PROFILE_NAME' ]
      ControllerCli.oobConfig().sslProfileConfig.sslProfileName = sslProfileName

   @staticmethod
   def noOrDefaultHandler( mode, args ):
      if not args.get( 'strict' ):
         ControllerCli.oobConfig().sslProfileConfig.sslProfileName = ''

ControllerCli.CvxConfigMode.addCommandClass( SslProfileSslprofilenameCmd )

