# Copyright (c) 2019 Arista Networks, Inc.  All rights reserved.
# Arista Networks, Inc. Confidential and Proprietary.

from __future__ import absolute_import, division, print_function

import BasicCli
import CliMatcher
from ControllerModel import (
      ServiceStatus,
      HeartbeatStatus,
      MountState,
      utcTimestamp,
)
import ShowCommand
import Tac
import CliPlugin.ConfigMgmtMode as ConfigMgmtMode
import CliPlugin.ControllerClient as ControllerClient
from CliPlugin.ControllerClientModel import (
      MgmtCvxService,
      MgmtCvxStatus,
      MgmtCvxClusterStatus,
      MgmtCvxMounts,
      MgmtCvxAllServices,
      ControllerStatus,
      SslProfileStatus,
)

#--------------------------------------------------------------------------------
# show management cvx
#--------------------------------------------------------------------------------
class ManagementCvxCmd( ShowCommand.ShowCliCommandClass ):
   syntax = 'show management cvx'
   data = {
      'management': ConfigMgmtMode.managementShowKwMatcher,
      'cvx': ControllerClient.cvxKwMatcherShowMgmt,
   }

   @staticmethod
   def _showManagementCvxCmd( clientConfig, clientStatus ):
      controllerStatuses = {}
      for controllerName in clientStatus.controllerStatus:
         ctrlStatus = clientStatus.controllerStatus.get( controllerName )
         if not ctrlStatus:
            continue
         hbStatus = HeartbeatStatus(
            lastHeartbeatSent=utcTimestamp(
               ctrlStatus.heartbeatStatus.lastHeartbeatSent ),
            lastHeartbeatReceived=utcTimestamp(
               ctrlStatus.heartbeatStatus.lastHeartbeatReceived ) )

         oobConnSecured = False
         ibConnSecured = False
         if ctrlStatus.connectionStatus.state == \
               Tac.Type( "Controller::OobState").oobStateEstablished:
            if clientConfig.sslProfileConfig.sslProfileName:
               oobConnSecured = True
               # Set in-band connection status to secured only when SSL is configured
               # and negotiated version >= 3
               if ctrlStatus.negotiatedVersion.value >= 3:
                  ibConnSecured = True

         peer = clientStatus.established.peer.get( ctrlStatus.systemId )
         connectionTime = peer.connectedTime if peer else None

         oobConnState=ctrlStatus.connectionStatus.state.replace(
            'oobState', '' ).lower()
         if oobConnState == 'shuttingdown':
            oobConnState = 'shutdown'
         controllerStatuses[ controllerName ] = ControllerStatus(
            oobConnectionStatus=oobConnState,
            leader=ctrlStatus.leader,
            lastLeaderChangeTs=utcTimestamp( ctrlStatus.lastLeaderChangeTs ),
            connectionTime=utcTimestamp( connectionTime ),
            versionCompatibility=ctrlStatus.versionCompatibility,
            negotiatedVersion=ctrlStatus.negotiatedVersion.value,
            heartbeatStatus=hbStatus,
            controllerUUID=ctrlStatus.controllerUUID,
            oobConnectionSecured=oobConnSecured,
            ibConnectionSecured=ibConnSecured,
            duplicatedConnection=ctrlStatus.duplicatedConnection )

      sslProfileStatus = None
      if clientStatus.sslProfileStatus:
         profileStatus = clientStatus.sslProfileStatus
         disableReason = None
         if not profileStatus.enabled:
            disableReason = profileStatus.disableReason
         sslProfileStatus = SslProfileStatus(
                              sslProfileName=profileStatus.sslProfileName,
                              enabled=profileStatus.enabled,
                              disableReason=disableReason )

      sourceIntfAddr = None
      intf = ControllerClient.ipStatus.ipIntfStatus.get(
            clientConfig.switchHost.sourceIntf )
      if intf:
         sourceIntfAddr = intf.activeAddrWithMask.address

      return MgmtCvxClusterStatus(
         enabled=clientStatus.enabled,
         cvxClusterName=clientConfig.clusterName,
         controllerStatuses=controllerStatuses,
         heartbeatInterval=clientConfig.heartbeatConfig.punchInterval,
         heartbeatTimeout=clientConfig.heartbeatConfig.timeout,
         sslProfileStatus=sslProfileStatus,
         sourceIntf=clientConfig.switchHost.sourceIntf,
         sourceIntfAddr=sourceIntfAddr,
         vrfName=clientConfig.switchHost.vrfName )


   @staticmethod
   def handler( mode, args ):
      clientConfig = ControllerClient.clientConfig
      clientStatus = ControllerClient.clientStatus
      mgmtCvxStatus = MgmtCvxStatus(
            clusterStatus = ManagementCvxCmd._showManagementCvxCmd( 
         clientConfig, clientStatus ) )
      secondaryClientConfig = ControllerClient.gv.secondaryClientConfig
      secondaryClientStatus = ControllerClient.gv.secondaryClientStatus
      if secondaryClientConfig and secondaryClientConfig.enabled and \
            secondaryClientStatus:
         mgmtCvxStatus.secondaryClusterStatus =\
               ManagementCvxCmd._showManagementCvxCmd ( 
            ControllerClient.gv.secondaryClientConfig,
            ControllerClient.gv.secondaryClientStatus )
      else:
         mgmtCvxStatus.secondaryClusterStatus = None
      return mgmtCvxStatus

   cliModel = MgmtCvxStatus

BasicCli.addShowCommandClass( ManagementCvxCmd )

#--------------------------------------------------------------------------------
# show management cvx mounts
#--------------------------------------------------------------------------------
class ManagementCvxMountsCmd( ShowCommand.ShowCliCommandClass ):
   syntax = 'show management cvx mounts'
   data = {
      'management': ConfigMgmtMode.managementShowKwMatcher,
      'cvx': ControllerClient.cvxKwMatcherShowMgmt,
      'mounts': 'Show CVX Mounts',
   }

   @staticmethod
   def handler( mode, args ):
      mountTypes = {}
      pubMounts = []
      def commandHelper( sysdbConfig, sysdbStatus, secondary=False ):
         if sysdbConfig.publishedMountConfig:
            for controllerdbPath in sysdbConfig.publishedMountConfig:
               try:
                  mountTypes[ controllerdbPath ] = \
                        sysdbConfig.publishedMountConfig[ controllerdbPath ]\
                        .mountDesc.taccType
               except KeyError:
                  continue

         if sysdbStatus.publishedMountStatus:
            for controllerdbPath in sysdbStatus.publishedMountStatus:
               try:
                  pubMountStatus = \
                        sysdbStatus.publishedMountStatus[ controllerdbPath ]
               except KeyError:
                  continue
               
               mountType = "(not configured)"
               if controllerdbPath in mountTypes:
                  mountType = mountTypes[ controllerdbPath ]

               suffix = ""
               if secondary:
                  suffix = "*"
               
               pubMounts.append( MountState(
                  path=controllerdbPath + suffix,
                  type=mountType,
                  state=pubMountStatus ) )
      commandHelper( ControllerClient.sysdbConfig,
            ControllerClient.sysdbStatus )
      if ControllerClient.gv.secondarySysdbConfig\
            and ControllerClient.gv.secondarySysdbStatus:
         commandHelper( ControllerClient.gv.secondarySysdbConfig,
               ControllerClient.gv.secondarySysdbStatus, True )

      return MgmtCvxMounts( mountStates=pubMounts )

   cliModel = MgmtCvxMounts

BasicCli.addShowCommandClass( ManagementCvxMountsCmd )

#--------------------------------------------------------------------------------
# show management cvx service
#--------------------------------------------------------------------------------
matcherService = CliMatcher.KeywordMatcher( 'service',
      helpdesc='Show local services' )

def leaderServiceStatus():
   clientStatus = ControllerClient.clientStatus
   for controllerName in clientStatus.controllerStatus:
      controllerStatus = clientStatus.controllerStatus.get( controllerName )
      if controllerStatus and controllerStatus.leader:
         # In standalone mode, the leader is set.
         return controllerStatus.serviceStatusDir
   return None

def allServiceNames( mode ):
   serviceStatusDir = leaderServiceStatus()
   if serviceStatusDir:
      return [ k for k in serviceStatusDir.service.keys() if k != "CVX" ]
   return []

def showMgmtCvxServiceName( mode, name ):
   serviceStatusDir = leaderServiceStatus()
   serviceStatus = None
   if serviceStatusDir:
      serviceStatus = serviceStatusDir.service.get( name )
   if not serviceStatus:
      mode.addError( "No service named '%s'" % name )
      return None

   vs = serviceStatus.versionStatus
   status = ServiceStatus(
      enabled=serviceStatus.enabled,
      versionNegotiationComplete=vs.versionNegotiationComplete,
      versionCompatible=vs.versionCompatible,
      negotiatedVersion=vs.negotiatedVersion.value )
   
   return MgmtCvxService( name=name, status=status )

class ManagementCvxServiceCmd( ShowCommand.ShowCliCommandClass ):
   syntax = 'show management cvx service'
   data = {
      'management': ConfigMgmtMode.managementShowKwMatcher,
      'cvx': ControllerClient.cvxKwMatcherShowMgmt,
      'service': matcherService,
   }

   @staticmethod
   def handler( mode, args ):
      serviceStatusDir = leaderServiceStatus()
      if not serviceStatusDir:
         return MgmtCvxAllServices( services=[] )

      allServices = []
      for name in sorted( allServiceNames( mode ) ):
         allServices.append( showMgmtCvxServiceName( mode, name ) )
      return MgmtCvxAllServices( services=allServices )

   cliModel = MgmtCvxAllServices

BasicCli.addShowCommandClass( ManagementCvxServiceCmd )

#--------------------------------------------------------------------------------
# show management cvx service NAME
#--------------------------------------------------------------------------------
class ManagementCvxServiceNameCmd( ShowCommand.ShowCliCommandClass ):
   syntax = 'show management cvx service NAME'
   data = {
      'management': ConfigMgmtMode.managementShowKwMatcher,
      'cvx': ControllerClient.cvxKwMatcherShowMgmt,
      'service': matcherService,
      'NAME':
         CliMatcher.DynamicNameMatcher( allServiceNames, 'Service name' ),
   }

   @staticmethod
   def handler( mode, args ):
      name = args[ 'NAME' ]
      return showMgmtCvxServiceName( mode, name )

   cliModel = MgmtCvxService

BasicCli.addShowCommandClass( ManagementCvxServiceNameCmd )
