   # Copyright (c) 2019 Arista Networks, Inc.  All rights reserved.
# Arista Networks, Inc. Confidential and Proprietary.

from __future__ import absolute_import, division, print_function

import CliCommand
import CliMatcher
import ControllerModel
import HostnameCli
import Tac
import CliPlugin.ControllerClient as ControllerClient
import CliPlugin.IntfCli as IntfCli
import CliPlugin.VrfCli as VrfCli
import CliPlugin.Ssl as Ssl

defaultSysname = ""
defaultSourceIntf = Tac.Value( "Arnet::IntfId" )

#--------------------------------------------------------------------------------
# [ no | default ] heartbeat-interval INTERVAL
#--------------------------------------------------------------------------------
class HeartbeatIntervalCmd( CliCommand.CliCommandClass ):
   syntax = 'heartbeat-interval INTERVAL'
   noOrDefaultSyntax = 'heartbeat-interval ...'
   data = {
      'heartbeat-interval': 'Time between CVX heartbeat messages',
      'INTERVAL': CliMatcher.IntegerMatcher(
         ControllerClient.heartbeatIntervalType.min,
         ControllerClient.heartbeatIntervalType.max, helpdesc='Seconds' ),
   }

   @staticmethod
   def _setHeartBeatPunchInterval( mode, args, clientConfig ):
      interval = args.get( 'INTERVAL' )
      if not interval:
         interval = ControllerClient.heartbeatIntervalType.defaultVal
      else:
         interval = Tac.Value( "Controller::Heartbeat::Interval", interval )
      clientConfig.heartbeatConfig.punchInterval = interval


   @staticmethod
   def handler( mode, args ):
      return HeartbeatIntervalCmd._setHeartBeatPunchInterval( 
            mode, args, ControllerClient.clientConfig )

   noOrDefaultHandler = handler

ControllerClient.CvxMgmtConfigMode.addCommandClass( HeartbeatIntervalCmd )

#--------------------------------------------------------------------------------
# [ no | default ] heartbeat-timeout TIMEOUT
#--------------------------------------------------------------------------------
class HeartbeatTimeoutCmd( CliCommand.CliCommandClass ):
   syntax = 'heartbeat-timeout TIMEOUT'
   noOrDefaultSyntax = 'heartbeat-timeout ...'
   data = {
      'heartbeat-timeout':
         'Elapsed time since last heartbeat before breaking CVX connection',
      'TIMEOUT':
         CliMatcher.IntegerMatcher(
            ControllerClient.heartbeatTimeoutType.min,
            ControllerClient.heartbeatTimeoutType.max, helpdesc='Seconds' ),
   }

   @staticmethod
   def _setHeartBeatTimeout( mode, args, clientConfig ):
      timeout = args.get( 'TIMEOUT' )
      if not timeout or timeout == "default":
         timeout = ControllerClient.heartbeatTimeoutType.defaultVal
      else:
         timeout = Tac.Value( "Controller::Heartbeat::Timeout", timeout )
      clientConfig.heartbeatConfig.timeout = timeout

   @staticmethod
   def handler( mode, args ):
      return HeartbeatTimeoutCmd._setHeartBeatTimeout(
            mode, args, ControllerClient.clientConfig )

   noOrDefaultHandler = handler

ControllerClient.CvxMgmtConfigMode.addCommandClass( HeartbeatTimeoutCmd )

#--------------------------------------------------------------------------------
# [ no | default ] server host HOSTNAME [ sysname SYSNAME ]
#--------------------------------------------------------------------------------
sysnameNode = CliCommand.Node( CliMatcher.KeywordMatcher(
                                 'sysname', helpdesc='Controller Sysname' ),
                               hidden=True )
class ServerHostnameCmd( CliCommand.CliCommandClass ):
   syntax = 'server host HOSTNAME [ sysname SYSNAME ]'
   noOrDefaultSyntax = 'server host [ HOSTNAME ]'
   data = {
      'server': 'Configure connection to CVX server',
      'host': 'Hostname or IP Address of CVX server',
      'HOSTNAME': HostnameCli.IpAddrOrHostnameMatcher(
         helpname='WORD', helpdesc='Hostname or A.B.C.D' ),
      'sysname': sysnameNode,
      'SYSNAME': CliMatcher.StringMatcher(
         helpdesc='Controller Sysname', helpname='WORD' ),
   }

   @staticmethod
   def _setServerHostname( mode, args, clientConfig ):
      hostname = args[ 'HOSTNAME' ]
      sysname = args.get( 'SYSNAME' )

      ip, name = ControllerModel.ipOrHostname( hostname )
      if not name and ControllerModel.invalidHostIp( ip ):
         mode.addError( "Invalid server IP address" )
         return

      controllerConfig = clientConfig.controllerConfig.get( hostname )
      if not controllerConfig:
         controllerConfig = clientConfig.newControllerConfig( hostname )
         controllerConfig.connectionConfig = ( "connection-%s" % hostname, )
         connectionConfig = controllerConfig.connectionConfig
         connectionConfig.ip = ip
         connectionConfig.hostname = name
         connectionConfig.sslProfileConfig = clientConfig.sslProfileConfig
         connectionConfig.port = ControllerClient.constants.controllerOobPort
      controllerConfig.connectionConfig.sysname = sysname or defaultSysname

   @staticmethod
   def handler( mode, args ):
      return ServerHostnameCmd._setServerHostname( mode, args,
            ControllerClient.clientConfig )

   @staticmethod
   def _setNoServerHostname( mode, args, clientConfig ):
      hostname = args.get( 'HOSTNAME' )
      if not hostname:
         clientConfig.controllerConfig.clear()
      elif hostname in clientConfig.controllerConfig:
         del clientConfig.controllerConfig[ hostname ]

   @staticmethod
   def noOrDefaultHandler( mode, args ):
      return ServerHostnameCmd._setNoServerHostname( mode, args,
            ControllerClient.clientConfig )


ControllerClient.CvxMgmtConfigMode.addCommandClass( ServerHostnameCmd )

#--------------------------------------------------------------------------------
# [ no | default ] shutdown
#--------------------------------------------------------------------------------
def setClientConfigEnabled( enabled ):
   ControllerClient.clientConfig.enabled = enabled

class ShutdownCmd( CliCommand.CliCommandClass ):
   syntax = 'shutdown'
   noOrDefaultSyntax = syntax
   data = {
      'shutdown': 'Disable the controller connection',
   }

   @staticmethod
   def handler( mode, args ):
      setClientConfigEnabled( False )

   defaultHandler = handler

   @staticmethod
   def noHandler ( mode, args ):
      setClientConfigEnabled( True )

ControllerClient.CvxMgmtConfigMode.addCommandClass( ShutdownCmd )

#--------------------------------------------------------------------------------
# [ no | default ] source-interface INTF ...
#--------------------------------------------------------------------------------
class SourceInterfaceCmd( CliCommand.CliCommandClass ):
   syntax = 'source-interface INTF ...'
   noOrDefaultSyntax = 'source-interface ...'
   data = {
      'source-interface': 'Interface providing the IP source address of CVX packets',
      'INTF': IntfCli.Intf.matcherWithIpSupport,
   }

   @staticmethod
   def _setSourceIntf( mode, args, clientConfig ):
      intf = args.get( 'INTF' )
      if intf:
         if intf.name not in ControllerClient.ipConfig.ipIntfConfig:
            mode.addWarning( "Interface IP address not configured" )
         clientConfig.switchHost.sourceIntf = intf.name
      else:
         clientConfig.switchHost.sourceIntf = \
               ControllerClient.defaultSourceIntf


   @staticmethod
   def handler( mode, args ):
      return SourceInterfaceCmd._setSourceIntf( mode, args,
            ControllerClient.clientConfig )

   noOrDefaultHandler = handler
   
ControllerClient.CvxMgmtConfigMode.addCommandClass( SourceInterfaceCmd )

#--------------------------------------------------------------------------------
# [ no | default ] ssl profile SSLPROFILENAME
#--------------------------------------------------------------------------------
class SslProfileNameCmd( CliCommand.CliCommandClass ):
   syntax = 'ssl profile SSLPROFILENAME'
   noOrDefaultSyntax = 'ssl profile ...'
   data = {
      'ssl': Ssl.sslMatcher,
      'profile': Ssl.profileMatcher,
      'SSLPROFILENAME':
         CliMatcher.DynamicNameMatcher(
            lambda mode: ControllerClient.mgmtSecConfig.profileConfig,
            helpdesc='SSL profile name' ),
   }

   @staticmethod
   def _setSslProfileName( mode, args, clientConfig ):
      sslProfileName = args.get( 'SSLPROFILENAME', '' )
      clientConfig.sslProfileConfig.sslProfileName = sslProfileName

   @staticmethod
   def handler( mode, args ):
      return SslProfileNameCmd._setSslProfileName( mode, args,
            ControllerClient.clientConfig )

   noOrDefaultHandler = handler

ControllerClient.CvxMgmtConfigMode.addCommandClass( SslProfileNameCmd )

#--------------------------------------------------------------------------------
# [ no | default ] vrf VRF
#--------------------------------------------------------------------------------
class VrfNameCmd( CliCommand.CliCommandClass ):
   syntax = 'vrf VRF'
   noOrDefaultSyntax = 'vrf ...'
   data = {
      'vrf': 'VRF name',
      'VRF': VrfCli.VrfNameExprFactory(),
   }

   @staticmethod
   def _setVrfName( mode, args, clientConfig ):
      vrfName = args.get( 'VRF', VrfCli.DEFAULT_VRF )
      clientConfig.switchHost.vrfName = vrfName

   @staticmethod
   def handler( mode, args ):
      return VrfNameCmd._setVrfName( mode, args, ControllerClient.clientConfig )

   noOrDefaultHandler = handler

ControllerClient.CvxMgmtConfigMode.addCommandClass( VrfNameCmd )
