# Copyright (c) 2018-2019 Arista Networks, Inc.  All rights reserved.
# Arista Networks, Inc. Confidential and Proprietary.
from __future__ import absolute_import, division, print_function

import collections
import itertools
import re

import BasicCli
import CliCommand
import CliMatcher
from CliMode.ContainerTracer import CtMode, CtClusterMode
import CliParser
from CliPlugin import ContainerTracerModel as model
from CliPlugin import ControllerdbLib
from CliPlugin import ControllerCli
import ConfigMount
import ContainerTracerLib
import LazyMount
import Plugins
import ReversibleSecretCli
import ShowCommand
import Tac
import Tracing


__defaultTraceHandle__ = Tracing.Handle( 'ContainerTracerCli' )
debug = Tracing.trace8

AGENT_NAME = 'ContainerTracer'


#-------------------------------------------------------------------------------
# ContainerTracer Plugin
#-------------------------------------------------------------------------------
MI = collections.namedtuple( 'MountInfo', 'name path tacModel mode')


class _Mounts( object ):
   _statusMounts = (
      # pkgdeps: library ContainerTracer
      MI( 'status', 'containertracer/status', 'ContainerTracer::Status', 'w' ),
      # pkgdeps: library Controllerdb
      MI( 'controllerConfig',
          'controller/config',
          'Controllerdb::Config',
          'r'
      ),
   )

   _configMounts = (
      # pkgdeps: library ContainerTracer
      MI( 'config', 'containertracer/config', 'ContainerTracer::Config', 'w' ),
      # pkgdeps: library Controller
      MI( 'serviceConfigDir',
          'controller/service/config',
          'Controller::ServiceConfigDir',
          'w'
      ),
   )

   _ctlrMounts = (
       # pkgdeps: library NetworkTopology
       MI( 'topologyStatus',
            'topology/version3/global/status' ,
            'NetworkTopologyAggregatorV3::Status',
            'r'
       ),
   )

   def __init__( self ):
      self.cdbMountsComplete = False

   def _mountHelper(self, entityManager, mount, mountDetails ):
      for m in mountDetails:
         setattr(self, m.name, mount( entityManager, m.path, m.tacModel, m.mode ) )

   def doMounts( self, entityManager ):
      self._mountHelper( entityManager, LazyMount.mount, self._statusMounts )
      self._mountHelper( entityManager, ConfigMount.mount, self._configMounts )

   def doControllerMounts( self, entityManager ):
      debug( "doControllerMounts" )
      self._mountHelper( entityManager, LazyMount.mount, self._ctlrMounts )
      self.cdbMountsComplete = True

   def clearMounts( self ):
      for m in itertools.chain( self._statusMounts, self._configMounts ):
         delattr( self, m.name )

   def clearControllerMounts( self ):
      for m in self._ctlrMounts:
         delattr( self, m.name )
      self.cdbMountsComplete = False


mounts = _Mounts()

@Plugins.plugin( requires=( 'ControllerdbMgr', ) )
def Plugin( entityManager ):
   debug( "Registering controller mount callback" )
   mounts.doMounts( entityManager )
   ControllerdbLib.registerNotifiee( mounts.doControllerMounts )

#-------------------------------------------------------------------------------
# Container Tracer Cli
#-------------------------------------------------------------------------------

clusterNameMatcher = CliMatcher.DynamicNameMatcher(
   lambda mode: mounts.config.cluster,
   'Cluster name', helpname='CLUSTERNAME' )

class ContainerTracerConfigMode( CtMode, BasicCli.ConfigModeBase ):
   name = 'cvx-container-tracer'
   modeParseTree = CliParser.ModeParseTree()

   def __init__( self, parent, session ):
      CtMode.__init__( self )
      BasicCli.ConfigModeBase.__init__( self, parent, session )

   def setShutdown( self, enabled=False ):
      if enabled and not mounts.controllerConfig.enabled:
         self.addError(
            "Container Tracer cannot be enabled until the CVX is enabled"
         )
         return

      mounts.config.enabled = enabled
      mounts.serviceConfigDir.service[ AGENT_NAME ].enabled = enabled

#--------------------------------------------------------------------------------
# (config-cvx)# [ no | default ] service container tracer
#--------------------------------------------------------------------------------
class ServiceContainerTracerCmd( CliCommand.CliCommandClass ):
   syntax = 'service container tracer'
   noOrDefaultSyntax = syntax
   data = {
      'service' : ControllerCli.serviceKwMatcher,
      'container' : 'Configure Container services',
      'tracer' : 'Configure ContainerTracer service',
   }

   @staticmethod
   def handler( mode, args ):
      mode.session_.gotoChildMode( mode.childMode( ContainerTracerConfigMode ) )

   @staticmethod
   def noOrDefaultHandler( mode, args=None ):
      debug( "Cleaning up the container tracer config" )
      mounts.config.reset()

      mounts.config.enabled = False
      mounts.serviceConfigDir.service[ AGENT_NAME ].enabled = False

ControllerCli.CvxConfigMode.addCommandClass( ServiceContainerTracerCmd )

# handle the no cvx case
ControllerCli.addNoCvxCallback( ServiceContainerTracerCmd.noOrDefaultHandler )

#-------------------------------------------------------------------------------
# (config-cvx-container-tracer)# [ no|default ] shutdown
#-------------------------------------------------------------------------------
class ShutdownCmd( CliCommand.CliCommandClass ):
   syntax = 'shutdown'
   noOrDefaultSyntax = syntax
   data = {
      'shutdown' : 'Shutdown Container Tracer Service',
   }

   @staticmethod
   def handler( mode, args ):
      mode.setShutdown()

   @staticmethod
   def noHandler( mode, args ):
      mode.setShutdown( True )

   defaultHandler = handler

ContainerTracerConfigMode.addCommandClass( ShutdownCmd )

#-------------------------------------------------------------------------------
# Cluster Mode
#-------------------------------------------------------------------------------
# pkgdeps: library ContainerTracer
tacAuthType = Tac.Type( 'ContainerTracer::AuthType' )
configDefaults = Tac.Type( 'ContainerTracer::ClusterConfigDefaults' )

class ContainerTracerClusterConfigMode( CtClusterMode, BasicCli.ConfigModeBase ):
   name = 'cvx-container-tracer-cluster'
   modeParseTree = CliParser.ModeParseTree()

   def __init__( self, parent, session, clusterName ):
      CtClusterMode.__init__( self, clusterName )
      BasicCli.ConfigModeBase.__init__( self, parent, session )
      if clusterName in mounts.config.cluster:
         self.cluster = mounts.config.cluster[ clusterName ]
      else:
         self.cluster = mounts.config.cluster.newMember( clusterName )
      self.clusterName = clusterName

   def setAuthSecret( self, authType=tacAuthType.tokenAuth, authSecret='' ):
      if authSecret:
         self.cluster.authType = tacAuthType.tokenAuth
      else:
         self.cluster.authType = tacAuthType.unknownAuth
      self.cluster.authSecret = authSecret

   def setUrl( self, url='' ):
      # urlparse is very forgiving and parse results can be unexpected for
      # malformed URLs, so use a more restrictive RE to validate
      authUrlRe = (
         r'^https?://' # scheme
         r'(?:(?:[A-Z0-9](?:[A-Z0-9-]{0,61}[A-Z0-9])?\.)+[A-Z]{2,6}\.?|' # hostname
         r'\d{1,3}\.\d{1,3}\.\d{1,3}\.\d{1,3})' # or IP
         r'(?::\d+)?' # optional port
         r'(?:/?|[/?]\S+)$' # path
      )
      authRe = re.compile( authUrlRe, re.IGNORECASE )
      if url and not authRe.match( url ):
         self.addError( 'Invalid URL %s' % url )
         return
      self.cluster.url = url

   def setApiVersion(self, version=None):
      version = version or configDefaults.apiVersion
      if version != configDefaults.apiVersion:
         self.addError(
            'Version %s not supported. Only %s is supported' %
            ( version, configDefaults.apiVersion )
         )
      self.cluster.apiVersion = version

#--------------------------------------------------------------------------------
# [ no | default ] cluster NAME
#--------------------------------------------------------------------------------
class ClusterClusternameCmd( CliCommand.CliCommandClass ):
   syntax = 'cluster NEWNAME'
   noOrDefaultSyntax = 'cluster CLUSTERNAME'
   data = {
      'cluster' : 'Configure cluster information for ContainerTracer',
      'NEWNAME' : CliMatcher.QuotedStringMatcher(
         helpdesc='Cluster name',
         helpname='name',
         pattern=CliParser.excludePipeTokensPattern ),
      'CLUSTERNAME' : clusterNameMatcher,
   }

   @staticmethod
   def handler( mode, args ):
      mode.session_.gotoChildMode(
         mode.childMode( ContainerTracerClusterConfigMode,
                         clusterName=args[ 'NEWNAME' ] )
      )

   @staticmethod
   def noOrDefaultHandler( mode, args ):
      clusterName = args[ 'CLUSTERNAME' ]
      debug( "Deleting cluster %s" % clusterName )
      del mounts.config.cluster[ clusterName ]

ContainerTracerConfigMode.addCommandClass( ClusterClusternameCmd )

#--------------------------------------------------------------------------------
# (config-cvx-container-tracer-cluster-<clusterName>)# url URL
#--------------------------------------------------------------------------------
class ClusterUrlCmd( CliCommand.CliCommandClass ):
   syntax = 'url URL'
   noOrDefaultSyntax = 'url ...'
   data = {
      'url' : 'URL for Kubernetes Cluster API',
      'URL' : CliMatcher.PatternMatcher( pattern='.*',
                                        helpname='URL',
                                        helpdesc='URL for Kubernetes Cluster API.' )
   }

   @staticmethod
   def handler( mode, args ):
      mode.setUrl( args.get( 'URL', '' ) )

   noOrDefaultHandler = handler

ContainerTracerClusterConfigMode.addCommandClass( ClusterUrlCmd )

#--------------------------------------------------------------------------------
# (config-cvx-container-tracer-cluster-<clusterName>)# version VERSION
#--------------------------------------------------------------------------------
class ClusterVersionCmd( CliCommand.CliCommandClass ):
   syntax = 'version VERSION'
   noOrDefaultSyntax = 'version ...'
   data = {
      'version' : 'The API version for the Kubernetes API',
      'VERSION' : CliMatcher.QuotedStringMatcher(
          helpname='version',
          helpdesc='The version for the Service Account user' )
   }

   @staticmethod
   def handler( mode, args ):
      mode.setApiVersion( args.get( 'VERSION' ) )

   noOrDefaultHandler = handler

ContainerTracerClusterConfigMode.addCommandClass( ClusterVersionCmd )

#-------------------------------------------------------------------------------
# (config-cvx-container-tracer-cluster-<clusterName>)# auth token TOKEN
#-------------------------------------------------------------------------------

class ClusterAuthCommand( CliCommand.CliCommandClass ):
   syntax = 'auth token TOKEN'
   noOrDefaultSyntax = 'auth token ...'
   data = {
      'auth' : 'Cluster authentication configuration',
      'token' : 'Use a token to authenticate the Service Account user',
      'TOKEN' : ReversibleSecretCli.reversibleSecretCliExpression( 'TOKEN' )
   }

   @staticmethod
   def handler( mode, args ):
      mode.setAuthSecret( authSecret=args.get( 'TOKEN', '' ) )

   noOrDefaultHandler = handler

ContainerTracerClusterConfigMode.addCommandClass( ClusterAuthCommand )

#-------------------------------------------------------------------------------
# Basic tokens for show commands
#-------------------------------------------------------------------------------

class _ServiceContainerTracerExpr( CliCommand.CliExpression ):
   expression = 'service container tracer'
   data = {
      'service' : ControllerCli.serviceAfterShowKwMatcher,
      'container' : 'Show information about container services',
      'tracer' : 'Show information about the ContainerTracer service'
   }

class _ClusterNameExpr( CliCommand.CliExpression ):
   expression = 'cluster CLUSTERNAME'
   data = {
      'cluster' : 'Show cluster information',
      'CLUSTERNAME' : clusterNameMatcher,
   }

#-------------------------------------------------------------------------------
# show service container tracer clusters
#-------------------------------------------------------------------------------

class ShowClustersCmd( ShowCommand.ShowCliCommandClass ):
   syntax = 'show SCT clusters'
   data = {
      'SCT' : _ServiceContainerTracerExpr,
      'clusters' : 'Show cluster information',
   }

   cliModel = model.ClustersModel

   @staticmethod
   def handler( mode, args ):
      clusters = model.ClustersModel( enabled=mounts.config.enabled )
      for _, cc in mounts.config.cluster.items():
         clusters.clusters[ cc.name ] = model.ClusterConfigModel(
            name=cc.name,
            url=cc.url,
            authType=cc.authType,
            authSecret=cc.authSecret,
            apiVersion=cc.apiVersion
         )
      return clusters

BasicCli.addShowCommandClass( ShowClustersCmd )

#-------------------------------------------------------------------------------
# show service container tracer cluster <cluster-name> nodes <filterName>]
#-------------------------------------------------------------------------------
def findNodeInTopology( mode, nodeName, interface=None, placeholder=True ):
   """Find node in the topology.

   When interface is provided the node returned is the neighbor of nodeName.
   """
   result = None
   if placeholder:
      result = model.NodeModel(
         name=nodeName,
         switch='Unknown',
         switchInterface='Unknown',
         hostInterface='Unknown'
      )

   # search for host
   hostsWithName = mounts.topologyStatus.hostsByHostname.get( nodeName )
   if not hostsWithName:
      msg = 'Unable to find node "%s" in topology' % nodeName
      mode.addWarning( msg )
      return result
   elif len( hostsWithName.host ) > 1:
      mode.addWarning(
         'Multiple matches for node %s found in topology' % nodeName
      )
      debug(
         'Unable to find exact match for %s. Candidates: %s' %
         ( nodeName, hostsWithName.host.keys() )
      )
      return result
   host = next( hostsWithName.host.itervalues() )

   try:
      fromPort = None
      edge = None
      toPort = None

      # use the interface passed or default to first one for the host
      fromPort = host.port.get( interface or next( host.port.iterkeys() ) )

      # fallback to logical port
      if fromPort is None:
         if interface in host.logicalPort:
            fromPort = next(
               host.logicalPort.get( interface ).memberPort.itervalues()
            )

      if fromPort is None:
         mode.addWarning(
            'Unable to find interface %s on %s' % ( interface, nodeName )
         )
         return result
      edge = mounts.topologyStatus.edge.get( fromPort )
      toPort = next( edge.toPort.iterkeys() )
      neighbor = toPort.host()

      if interface:
         # swap the order since the search is really for the neighbor
         host, fromPort, neighbor, toPort = neighbor, toPort, host, fromPort

      intfName = toPort.portGroup.name if toPort.portGroup else toPort.name

      return model.NodeModel(
         name=host.hostname,
         switch=neighbor.hostname,
         switchInterface=intfName,
         hostInterface=fromPort.name
      )

   except ( StopIteration, AttributeError ) as e:
      debug(
         'Unable to find valid edge %s. fromPort: %s edge: %s toPort: %s' %
         ( e, fromPort, edge, toPort )
      )

   return result

def getClusterClient( clusterName ):
   if clusterName in mounts.config.cluster:
      c = mounts.config.cluster[ clusterName ]
      return ContainerTracerLib.getClient( c )

def showClusterNodes( mode, clusterName, nodeName=None ):
   nodes = model.NodesModel()

   client = getClusterClient( clusterName )
   if client is None:
      mode.addError( 'Cluster %s is not configured' % clusterName )
      return nodes

   try:
      for n in client.getNodes( nodeName ):
         nodes.nodes[ n ] = findNodeInTopology( mode, n )
   except ContainerTracerLib.ClientError as e:
      mode.addError( e.message )

   return nodes

class ShowClusterNodesCmd( ShowCommand.ShowCliCommandClass ):
   syntax = 'show SCT CN nodes [ NODENAME ]'
   data = {
      'SCT' : _ServiceContainerTracerExpr,
      'CN' : _ClusterNameExpr,
      'nodes' : 'Show cluster nodes with network topoology',
      'NODENAME' : CliMatcher.PatternMatcher(
         pattern='[-._a-zA-Z0-9]+',
         helpdesc='Cluster node name',
         helpname='name' ),
   }

   cliModel = model.NodesModel

   @staticmethod
   def handler( mode, args ):
      return showClusterNodes( mode, args[ 'CLUSTERNAME' ], args.get( 'NODENAME' ) )

BasicCli.addShowCommandClass( ShowClusterNodesCmd )


#-------------------------------------------------------------------------------
# show service container tracer cluster <cluster-name> pods [<filterName>]
#-------------------------------------------------------------------------------

def showClusterPods( mode, clusterName, podName='', nodeName='', nodes=None ):
   if nodes is None:
      nodes = model.NodesModel( podDetails=True )

   client = getClusterClient( clusterName )
   if client is None:
      mode.addError( 'Cluster %s is not configured' % clusterName )
      return nodes

   try:
      for pi in client.getPods( podName, nodeName ):
         if pi.nodeName not in nodes.nodes:
            nodes.nodes[ pi.nodeName ] = findNodeInTopology( mode, pi.nodeName )

         node = nodes.nodes[ pi.nodeName ]
         node.pods[ pi.name ] = model.PodModel( name=pi.name, phase=pi.phase )
   except ContainerTracerLib.ClientError as e:
      mode.addError( e.message )
   return nodes

class ShowClusterPodsCmd( ShowCommand.ShowCliCommandClass ):
   syntax = 'show SCT CN pods [ PODNAME ]'
   data = {
      'SCT' : _ServiceContainerTracerExpr,
      'CN' : _ClusterNameExpr,
      'pods' : 'Show cluster pods with network topoology',
      'PODNAME' : CliMatcher.PatternMatcher(
         pattern='[-._a-zA-Z0-9]+',
         helpdesc='Cluster pod name',
         helpname='name' ),
   }

   cliModel = model.NodesModel

   @staticmethod
   def handler( mode, args ):
      return showClusterPods( mode, args[ 'CLUSTERNAME' ], args.get( 'PODNAME' ) )

BasicCli.addShowCommandClass( ShowClusterPodsCmd )

#-------------------------------------------------------------------------------
# show container-tracer switch <host> interface <intf>
#-------------------------------------------------------------------------------
def showPodsForSwitchInterface( mode, switchName, interfaceName ):
   nodes = model.NodesModel( podDetails=True )

   node = findNodeInTopology( mode, switchName, interfaceName, False )

   if not node:
      return nodes

   nodes.nodes[ node.name ] = node

   # Search all clusters for nodeName
   for c in mounts.config.cluster:
      nodes = showClusterPods( mode, c, nodeName=node.name, nodes=nodes )
      if node.pods:
         break  # a node cannot be a member of more than one cluster
   return nodes

class ShowPodsForSwitchInterfaceCmd( ShowCommand.ShowCliCommandClass ):
   syntax = 'show SCT switch SWITCHNAME interface INTERFACENAME'
   data = {
      'SCT' : _ServiceContainerTracerExpr,
      'switch' : 'Show pods scheduled to a host connected to a switch',
      'SWITCHNAME' : CliMatcher.QuotedStringMatcher(
          helpdesc='switch host name',
          helpname='switch' ),
      'interface' : 'Show pods scheduled to a host connected on an interface',
      'INTERFACENAME' : CliMatcher.QuotedStringMatcher(
          helpdesc='switch interface name',
          helpname='interface' ),
   }

   cliModel = model.NodesModel

   @staticmethod
   def handler( mode, args ):
      return showPodsForSwitchInterface(
         mode,
         args[ 'SWITCHNAME' ],
         args[ 'INTERFACENAME' ] )

BasicCli.addShowCommandClass( ShowPodsForSwitchInterfaceCmd )
