#!/usr/bin/env python
# Copyright (c) 2016 Arista Networks, Inc.  All rights reserved.
# Arista Networks, Inc. Confidential and Proprietary.

import Tac
import LazyMount
import sys
import re
import os
import BasicCli
import simplejson as json
import CliMatcher
import CliPlugin.TechSupportCli
from ContainerMgrModels import ContainerMgrImages, ContainerMgrImage, \
      ContainerMgrContainers, ContainerMgrContainer, ContainerMgrRegistries, \
      ContainerMgrRegistry, ContainerMgrInfo, ContainerMgrBackup, \
      ContainerMgrBackupFiles, ContainerMgrPort, ContainerMgrLogs
from ContainerMgrCliLib import dockerRemoteAPICmd, getContainerArgs
from ContainerMgrCliLib import removeNonAscii
from ContainerMgrCliLib import timestampToStr
from ContainerMgrCliLib import authConfigFile
from ContainerMgrConfigCli import imageNameMatcher, containerNameMatcher, \
      registryNameMatcher
import ShowCommand

containerMgrConfig = None
containerConfig = None

matcherContainerManager = CliMatcher.KeywordMatcher( 'container-manager',
      helpdesc='Display container-manager configuration' )

def isContainerMgrRunning():
   # Check if daemon is running. Will prevent surprises on namespace duts
   # too.
   if containerMgrConfig.daemonEnable:
      if os.access( "/var/run/docker.pid", os.F_OK ) and os.access(
      "/var/run/docker.sock", os.F_OK ):
         return True
   return False

def runDockerRemoteAPI( cmd ):
   if not isContainerMgrRunning():
      return ''

   try:
      output = Tac.run( cmd, stdout=Tac.CAPTURE,
                        stderr=sys.stderr, asRoot=True )
   except Tac.SystemCommandError:
      return ''

   return output

#--------------------------------------------------------------------------------
# show container-manager images [ IMAGE_NAME ]
#--------------------------------------------------------------------------------
class ContainerManagerImagesCmd( ShowCommand.ShowCliCommandClass ):
   syntax = 'show container-manager images [ IMAGE_NAME ]'
   data = {
      'container-manager' : matcherContainerManager,
      'images' : 'Display all images information',
      'IMAGE_NAME' : imageNameMatcher,
   }
   cliModel = ContainerMgrImages

   @staticmethod
   def handler( mode, args ):
      imageName = args.get( 'IMAGE_NAME' )
      allImages = {}
      if not isContainerMgrRunning():
         mode.addWarning( "ContainerMgr daemon is not running" )
         return ContainerMgrImages( containerMgrImages={} )

      if not imageName:
         cmd = [ dockerRemoteAPICmd + 'images/json' ]
      else:
         cmd = [ dockerRemoteAPICmd + 'images/' + imageName + '/json' ]

      cmd = [ 'curl', '-sS' ] + cmd
      try:
         output = json.loads( runDockerRemoteAPI( cmd ) )
      except json.decoder.JSONDecodeError:
         return ContainerMgrImages( containerMgrImages={} )

      if 'message' in output:
         # With new docker we get message in output
         return ContainerMgrImages( containerMgrImages={} )

      # When we query for a specific image it returns dict.
      # Making it list.
      if type( output ) is not list:
         output = [ output ]

      for image in output:
         model = ContainerMgrImage()
         model.imageId = image[ 'Id' ]
         model.timeOfCreation = timestampToStr( image[ 'Created' ] )
         model.imageSize = image[ 'Size' ]
         if image[ 'RepoTags' ]:
            img = image[ 'RepoTags' ][ 0 ]
         else:
            img = image[ 'RepoDigests' ][ 0 ].split( '@' )[ 0 ] + ':<none>'
         allImages[ img ] = model
      return ContainerMgrImages( containerMgrImages=allImages )

BasicCli.addShowCommandClass( ContainerManagerImagesCmd )

#--------------------------------------------------------------------------------
# show container-manager log CONTAINER_NAME
#--------------------------------------------------------------------------------
class ContainerManagerLogContainernameCmd( ShowCommand.ShowCliCommandClass ):
   syntax = 'show container-manager log CONTAINER_NAME'
   data = {
      'container-manager' : matcherContainerManager,
      'log' : 'Display log of a container',
      'CONTAINER_NAME' : containerNameMatcher,
   }
   cliModel = ContainerMgrLogs

   @staticmethod
   def handler( mode, args ):
      containerName = args[ 'CONTAINER_NAME' ]
      if not isContainerMgrRunning():
         mode.addWarning( "ContainerMgr daemon is not running" )
         return ContainerMgrLogs()

      cmd = [ dockerRemoteAPICmd + 'containers/' + containerName + \
            '/logs?stderr=1&stdout=1' ]
      cmd = [ 'curl', '-sS' ]  + cmd
      output = runDockerRemoteAPI( cmd )
      output = removeNonAscii( unicode( output ) )
      model = ContainerMgrLogs()
      model.containerMgrLogs = output

      return model

BasicCli.addShowCommandClass( ContainerManagerLogContainernameCmd )

#--------------------------------------------------------------------------------
# show container-manager containers [ CONTAINER_NAME ] [ brief ]
#--------------------------------------------------------------------------------
class ContainerManagerContainersCmd( ShowCommand.ShowCliCommandClass ):
   syntax = 'show container-manager containers [ CONTAINER_NAME ] [ brief ]'
   data = {
      'container-manager' : matcherContainerManager,
      'containers' : 'Display all containers information',
      'CONTAINER_NAME' : containerNameMatcher,
      'brief' : 'Display tabular output of summary information',
   }
   cliModel = ContainerMgrContainers

   @staticmethod
   def handler( mode, args ):
      brief = 'brief' in args
      containerName = args.get( 'CONTAINER_NAME' )
      allContainers = {}
      if not isContainerMgrRunning():
         mode.addWarning( "ContainerMgr daemon is not running" )
         return ContainerMgrContainers( containerMgrContainers={} )

      # To handle show container-manager containers brief
      # We aren't handling if container name is 'brief'.
      if containerName == 'brief':
         brief = True
         containerName = None
      if not containerName:
         cmd = [ dockerRemoteAPICmd + 'containers/json?all=1' ]
      else:
         cmd = [ dockerRemoteAPICmd + 'containers/' + containerName + '/json' ]

      cmd = [ 'curl', '-sS' ] + cmd
      try:
         output = json.loads( runDockerRemoteAPI( cmd ) )
      except json.decoder.JSONDecodeError:
         return ContainerMgrContainers( containerMgrContainers={} )

      if 'message' in output:
         # With new docker we get message in output
         return ContainerMgrContainers( containerMgrContainers={} )

      # When we query for a specific container it returns dict.
      # Making it list.
      if type( output ) is not list:
         output = [ output ]
      for container in output:
         model = ContainerMgrContainer()
         ports = []
         if not containerName:
            containerKey = container[ 'Names' ][ 0 ][ 1: ]
            model.imageName = container[ 'Image' ]
            model.command = container[ 'Command' ]
            model.state = container[ 'State' ]
            model.imageId = container[ 'ImageID' ]
            containerPorts = container[ 'Ports' ]
            for port in containerPorts:
               p = ContainerMgrPort()
               p.ip = port[ 'IP' ]
               p.privatePort = port[ 'PrivatePort' ]
               p.publicPort = port[ 'PublicPort' ]
               p.portType = port[ 'Type' ]
               ports.append( p )
         else:
            containerKey = container[ 'Name' ][ 1: ]
            model.imageName = container[ 'Config' ][ 'Image' ]
            model.command = container[ 'Path' ] + ' ' + \
                            getContainerArgs( container[ 'Args' ] )
            model.state = container[ 'State' ][ 'Status' ]
            model.imageId = container[ 'Image' ]
            containerPorts = container[ 'NetworkSettings' ][ 'Ports' ]
            if containerPorts:
               for port in containerPorts:
                  for hostPort in containerPorts[ port ]:
                     p = ContainerMgrPort()
                     p.ip = hostPort[ 'HostIp' ]
                     p.privatePort = int( port.split( '/' )[ 0 ] )
                     p.publicPort = int( hostPort[ 'HostPort' ] )
                     p.portType = port.split( '/' )[ 1 ]
                     ports.append( p )
         model.ports = ports
         model.containerId = container[ 'Id' ]
         model.timeOfCreation = timestampToStr( container[ 'Created' ] )
         model.onBoot = False
         if containerKey in containerConfig.container:
            model.onBoot = containerConfig.container[ containerKey ].onBoot
         # pylint: disable=W0212
         model._brief = True if brief else False
         allContainers[ containerKey ] = model
      return ContainerMgrContainers( containerMgrContainers=allContainers )

BasicCli.addShowCommandClass( ContainerManagerContainersCmd )

#--------------------------------------------------------------------------------
# show container-manager info
#--------------------------------------------------------------------------------
class ContainerManagerInfoCmd( ShowCommand.ShowCliCommandClass ):
   syntax = 'show container-manager info'
   data = {
      'container-manager' : matcherContainerManager,
      'info' : 'Show container-manager information',
   }
   cliModel = ContainerMgrInfo

   @staticmethod
   def handler( mode, args ):
      if not isContainerMgrRunning():
         mode.addWarning( "ContainerMgr daemon is not running" )
         return ContainerMgrInfo()

      cmd = [ dockerRemoteAPICmd + 'info' ]
      cmd = [ 'curl', '-sS' ] + cmd
      try:
         output = json.loads( runDockerRemoteAPI( cmd ) )
      except json.decoder.JSONDecodeError:
         mode.addWarning( "ContainerMgr info not available" )
         return ContainerMgrInfo()

      if 'message' in output:
         # With new docker we get message in output
         return ContainerMgrInfo()

      model = ContainerMgrInfo()
      model.containerNum = output[ 'Containers' ]
      model.runningContainerNum = output[ 'ContainersRunning' ]
      model.pausedContainerNum = output[ 'ContainersPaused' ]
      model.stoppedContainerNum = output[ 'ContainersStopped' ]
      model.imagesNum = output[ 'Images' ]
      model.storageDriver = output[ 'Driver' ]
      if output[ 'DriverStatus' ][ 0 ][ 0 ] == 'Backing Filesystem':
         model.backingFilesystem = output[ 'DriverStatus' ][ 0 ][ 1 ]
      model.loggingDriver = output[ 'LoggingDriver' ]
      model.cgroupDriver = output[ 'CgroupDriver' ]
      model.volumeType = output[ 'Plugins' ][ 'Volume' ]
      model.networkType = output[ 'Plugins' ][ 'Network' ]
      model.containerMgrEngineId = output[ 'ID' ]
      model.hostName = output[ 'Name' ]
      model.containerMgrRootDir = output[ 'DockerRootDir' ]
      model.cpuNum = output[ 'NCPU' ]
      model.memory = output[ 'MemTotal' ]
      return model

BasicCli.addShowCommandClass( ContainerManagerInfoCmd )

#--------------------------------------------------------------------------------
# show container-manager registry [ REGISTRY_NAME ]
#--------------------------------------------------------------------------------
class ContainerManagerRegistryCmd( ShowCommand.ShowCliCommandClass ):
   syntax = 'show container-manager registry [ REGISTRY_NAME ]'
   data = {
      'container-manager' : matcherContainerManager,
      'registry' : 'Display all the configured registries',
      'REGISTRY_NAME' : registryNameMatcher,
   }
   cliModel = ContainerMgrRegistries

   @staticmethod
   def handler( mode, args ):
      registryName = args.get( 'REGISTRY_NAME' )
      allRegistries = {}
      if not isContainerMgrRunning():
         mode.addWarning( "ContainerMgr daemon is not running" )
         return ContainerMgrRegistries( containerMgrRegistries={} )

      if not len( containerMgrConfig.registryConfig ):
         return ContainerMgrRegistries( containerMgrRegistries={} )

      data = Tac.run( [ 'cat', authConfigFile ], stdout=Tac.CAPTURE,
                      stderr=Tac.CAPTURE, asRoot=True, ignoreReturnCode=True )
      if 'No such file or directory' in data:
         loggedRegisteries = []
      else:
         auth = json.loads( data )[ 'auths' ]
         loggedRegisteries = auth.keys()

      insecureRegistries = []
      dockerdArgs = Tac.run( [ 'pgrep', '-l', '-f', 'dockerd' ], stdout=Tac.CAPTURE,
                             asRoot=True )
      insecurePattern = r'--insecure-registry=(\S+)'
      for x in re.finditer( insecurePattern, dockerdArgs ):
         insecureRegistries.append( x.groups()[ 0 ] )

      for registry in containerMgrConfig.registryConfig.keys():
         reg = containerMgrConfig.registryConfig[ registry ]
         if not reg.serverName:
            continue
         if registryName and registryName != registry:
            continue
         model = ContainerMgrRegistry()
         model.serverName = reg.serverName
         model.username = reg.userName
         model.insecure = reg.insecure
         if reg.insecure:
            if reg.serverName in insecureRegistries:
               model.status = "Success"
            else:
               model.status = "Failed"
         else:
            # In the new docker version /root/.docker/config.json has remove https://
            if '://' in reg.serverName:
               serverName = reg.serverName.split( '://' )[ 1 ]
            if serverName in loggedRegisteries:
               model.status = "Success"
            else:
               model.status = "Failed"
         allRegistries[ registry ] = model
      return ContainerMgrRegistries( containerMgrRegistries=allRegistries )

BasicCli.addShowCommandClass( ContainerManagerRegistryCmd )

#--------------------------------------------------------------------------------
# show container-manager backup
#--------------------------------------------------------------------------------
class ContainerManagerBackupCmd( ShowCommand.ShowCliCommandClass ):
   syntax = 'show container-manager backup'
   data = {
      'container-manager' : matcherContainerManager,
      'backup' : 'Display all the backup files for container-manager',
   }
   cliModel = ContainerMgrBackup

   @staticmethod
   def handler( mode, args ):
      backup = {}
      persistPaths = [ containerMgrConfig.defaultPersistentPath ]
      if containerMgrConfig.persistentPath != \
            containerMgrConfig.defaultPersistentPath:
         persistPaths += [ containerMgrConfig.persistentPath ]

      for path in persistPaths:
         persistPath = path + '.containermgr'
         if not os.path.exists( persistPath ):
            continue
         result = ContainerMgrBackupFiles()
         result.backupFiles = os.listdir( persistPath )
         backup[ path ] = result
      return ContainerMgrBackup( backup=backup )

BasicCli.addShowCommandClass( ContainerManagerBackupCmd )

#-----------------------------------------------------------------------------------
# Register ContainerMgr Show CLIs into "show Tech-Support"
#------------------------------------------------------------------------------------
def _containerMgrShowTechCmds():
   Cmds = [ "show container-manager images",
            "show container-manager containers",
            "show container-manager registry",
            "show container-manager info",
            "show container-manager log",
            "show container-manager backup" ]
   if containerMgrConfig.daemonEnable:
      return Cmds
   return []


CliPlugin.TechSupportCli.registerShowTechSupportCmdCallback(
   '2016-11-13 00:00:40',
   _containerMgrShowTechCmds )

def Plugin( entityManager ):
   global containerMgrConfig, containerConfig
   containerMgrConfig = LazyMount.mount( entityManager, 'containerMgr/config',
                                         'ContainerMgr::ContainerMgrConfig', 'r' )
   containerConfig = LazyMount.mount( entityManager,
                                      'containerMgr/container/config',
                                      'ContainerMgr::ContainerConfig', 'r' )
