#!/usr/bin/env python
# Copyright (c) 2016 Arista Networks, Inc.  All rights reserved.
# Arista Networks, Inc. Confidential and Proprietary.

from CliModel import Model, Str, Dict, Enum, Int, Bool, List
from ArnetModel import IpGenericAddress
from ContainerMgrCliLib import unixTimestampToStr, containerState
import TableOutput
from ContainerMgrCliLib import bytesInStr, portsToString

class ContainerMgrImage( Model ):
   imageId = Str( help='Image id in sha256 format' )
   timeOfCreation = Int( help='Time of image creation' )
   imageSize = Int( help='Size of the image' )

class ContainerMgrPort( Model ):
   ip = IpGenericAddress( help='IP address of the port' )
   privatePort = Int( help='Private port number' )
   publicPort = Int( help='Public port number' )
   portType = Enum( values=( "tcp", "udp" ), help='Port type' )

class ContainerMgrContainer( Model ):
   containerId = Str( help='Container id' )
   imageName = Str( help='Image name with tag' )
   imageId = Str( help='Image id in sha256 format' )
   command = Str( help='Command for the container' )
   timeOfCreation = Int( help='Time of container creation' )
   state = Enum( values=containerState, help='State of the container' )
   onBoot = Bool( help='Container will persist across reboots' )
   ports = List( valueType=ContainerMgrPort, help='List of all the ports '
                 'for this container' )
   _brief = Bool( help='Briefed output required, private' )

class ContainerMgrRegistry( Model ):
   serverName = Str( help='Registry server name' )
   username = Str( help='Username for registry' )
   insecure = Bool( help='Insecure registry' )
   status = Enum( values=( "Success", "Failed" ), help='Registry status' )

class ContainerMgrImages( Model ):
   containerMgrImages = Dict( valueType=ContainerMgrImage,
         help="Container Manager images keyed by imageName:tag" )
   def render( self ):
      if not len( self.containerMgrImages ):
         return

      def createTable( tableHeadings ):
         table = TableOutput.createTable( tableHeadings )
         f1 = TableOutput.Format( justify="left", minWidth=10 )
         f2 = TableOutput.Format( justify="left", minWidth=10 )
         f3 = TableOutput.Format( justify="left", minWidth=12 )
         f4 = TableOutput.Format( justify="left", minWidth=13 )
         f5 = TableOutput.Format( justify="left", minWidth=7 )
         table.formatColumns( f1, f2, f3, f4, f5 )
         return table

      tableHeadings = ( "Name", "Tag", "Id", "Created", "Size" )
      table = createTable( tableHeadings )
      for image in self.containerMgrImages.keys():
         imageInfo = self.containerMgrImages[ image ]
         table.newRow( image.split( ':' )[ 0 ], image.split( ':' )[ 1 ],
                       imageInfo.imageId[ 7 : 19 ],
                       unixTimestampToStr( imageInfo.timeOfCreation ),
                       bytesInStr( imageInfo.imageSize ) )
      print table.output()

class ContainerMgrContainers( Model ):
   containerMgrContainers = Dict( valueType=ContainerMgrContainer,
                                  help="Container Manager containers keyed "
                                       "by containerName" )
   def render( self ):
      if not len( self.containerMgrContainers ):
         return

      def createTable( tableHeadings ):
         table = TableOutput.createTable( tableHeadings )
         f1 = TableOutput.Format( justify="left", minWidth=15 )
         f2 = TableOutput.Format( justify="left", minWidth=22 )
         f3 = TableOutput.Format( justify="left", minWidth=6 )
         f4 = TableOutput.Format( justify="left", minWidth=7 )
         f5 = TableOutput.Format( justify="left", minWidth=22 )
         table.formatColumns( f1, f2, f3, f4, f5 )
         return table
      firstKey = self.containerMgrContainers.keys()[ 0 ]
      # pylint: disable=W0212
      brief = self.containerMgrContainers[ firstKey ]._brief
      if brief:
         tableHeadings = ( "Container", "Image", "Onboot", "State", "Command" )
         table = createTable( tableHeadings )
         for container in self.containerMgrContainers.keys():
            containerInfo = self.containerMgrContainers[ container ]
            table.newRow( container, containerInfo.imageName,
                          containerInfo.onBoot, containerInfo.state,
                          containerInfo.command )
         print table.output()
      else:
         for container in self.containerMgrContainers.keys():
            containerInfo = self.containerMgrContainers[ container ]
            print 'Container Name: %s' % container
            print 'Container Id: %s' % containerInfo.containerId
            print 'Image Name: %s' % containerInfo.imageName
            print 'Image Id: %s' % containerInfo.imageId
            print 'Onboot: %s' % containerInfo.onBoot
            print 'Command: %s' % containerInfo.command
            print 'Created: %s'\
                              % unixTimestampToStr( containerInfo.timeOfCreation )
            print 'Ports: %s' % portsToString( containerInfo.ports )
            print 'State: %s' % containerInfo.state
            print

class ContainerMgrLogs( Model ):
   containerMgrLogs = Str( help='Log of a container', default='' )
   def render( self ):
      if not self.containerMgrLogs:
         return
      
      for line in str(self.containerMgrLogs).splitlines():
         print line

class ContainerMgrRegistries( Model ):
   containerMgrRegistries = Dict( valueType=ContainerMgrRegistry,
                                  help="Container Manager registry keyed "
                                       "by registryName" )
   def render( self ):
      if not len( self.containerMgrRegistries ):
         return

      def createTable( tableHeadings ):
         table = TableOutput.createTable( tableHeadings )
         f1 = TableOutput.Format( justify="left", minWidth=12 )
         f2 = TableOutput.Format( justify="left", minWidth=20 )
         f3 = TableOutput.Format( justify="left", minWidth=12 )
         f4 = TableOutput.Format( justify="left", minWidth=6 )
         f5 = TableOutput.Format( justify="left", minWidth=10 )
         table.formatColumns( f1, f2, f3, f4, f5 )
         return table
      tableHeadings = ( "Registry Name", "Server", "Username",
                        "Secure", "Status")
      table = createTable( tableHeadings )
      for registry in self.containerMgrRegistries.keys():
         reg = self.containerMgrRegistries[ registry ]
         table.newRow( registry, reg.serverName, reg.username,
                       not reg.insecure, reg.status )
      print table.output()

class ContainerMgrInfo( Model ):
   containerNum = Int( help='Total number of Containers', default=0 )
   runningContainerNum = Int( help='Total number of running Containers', default=0 )
   pausedContainerNum = Int( help='Total number of paused Containers', default=0 )
   stoppedContainerNum = Int( help='Total number of stopped Containers', default=0 )
   imagesNum = Int( help='Total number of images', default=0 )
   storageDriver = Str( help='storage Driver', default='' )
   backingFilesystem = Str( help='backing Filesystem', default='' )
   loggingDriver = Str( help='logging Driver', default='' )
   cgroupDriver = Str( help='cgroup Driver', default='' )
   volumeType = List( valueType=str, help='Volume types', default=[] )
   networkType = List( valueType=str, help='Network types', default=[] )
   containerMgrEngineId = Str( help='Container-manager engine ID', default='' )
   hostName = Str( help='hostname', default='' )
   containerMgrRootDir = Str( help='Container-manager root dir', default='' )
   cpuNum = Int( help='Total number of CPUs', default=0 )
   memory = Int( help='Total Memory', default=0 )

   def render( self ):
      # Either all the attributes will be set or none of them.
      # So, just checking for one of them.
      if not self.storageDriver:
         return

      print 'Total Number of Containers: %s' % self.containerNum
      print 'Total Number of paused Containers: %s' % self.pausedContainerNum
      print 'Total Number of stopped Containers: %s' % self.stoppedContainerNum
      print 'Total Number of Images: %s' % self.imagesNum
      print 'Storage Driver: %s' % self.storageDriver
      print '   Backing Filesystem: %s' % self.backingFilesystem
      print 'Logging Driver: %s' % self.loggingDriver
      print 'Cgroup Driver: %s' % self.cgroupDriver
      print 'Plugins:'
      print '   Volume: %s' % ", ".join( self.volumeType )
      print '   Network: %s' % ", ".join( self.networkType )
      print 'ID: %s' % self.containerMgrEngineId
      print 'ContainerMgr Root Dir: %s' % self.containerMgrRootDir
      print 'CPUs: %s' % self.cpuNum
      print 'Total Memory: %s' % bytesInStr( self.memory )
      print

class ContainerMgrBackupFiles( Model ):
   backupFiles = List( valueType=str, help='Backup files' )

class ContainerMgrBackup( Model ):
   backup = Dict( valueType=ContainerMgrBackupFiles,
                  help="Container manager backup keyed by persistent directory" )
   def render( self ):
      if not len( self.backup ):
         return
      def createTable( tableHeadings ):
         table = TableOutput.createTable( tableHeadings )
         f1 = TableOutput.Format( justify="left", minWidth=22 )
         f2 = TableOutput.Format( justify="left", minWidth=22 )
         table.formatColumns( f1, f2 )
         return table
      tableHeadings = ( "Files", "Directory" )
      table = createTable( tableHeadings )
      for d in self.backup:
         directory = self.backup[ d ]
         for f in directory.backupFiles:
            table.newRow( f.split( '.' )[ 0 ], d )
      print table.output()
